/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.tserver.compactions;

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.core.dataImpl.KeyExtent;
import org.apache.accumulo.core.metadata.schema.ExternalCompactionId;
import org.apache.accumulo.core.spi.compaction.CompactionExecutorId;
import org.apache.accumulo.core.spi.compaction.CompactionKind;
import org.apache.accumulo.core.spi.compaction.CompactionServiceId;
import org.apache.accumulo.core.spi.compaction.CompactionServices;
import org.apache.accumulo.core.tabletserver.thrift.TCompactionQueueSummary;
import org.apache.accumulo.core.util.Retry;
import org.apache.accumulo.core.util.compaction.CompactionExecutorIdImpl;
import org.apache.accumulo.core.util.compaction.CompactionServicesConfig;
import org.apache.accumulo.core.util.threads.Threads;
import org.apache.accumulo.server.ServerContext;
import org.apache.accumulo.tserver.compactions.Compactable;
import org.apache.accumulo.tserver.compactions.CompactionExecutor;
import org.apache.accumulo.tserver.compactions.CompactionService;
import org.apache.accumulo.tserver.compactions.ExternalCompactionExecutor;
import org.apache.accumulo.tserver.compactions.ExternalCompactionJob;
import org.apache.accumulo.tserver.metrics.CompactionExecutorsMetrics;
import org.apache.accumulo.tserver.tablet.Tablet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompactionManager {
    private static final Logger log = LoggerFactory.getLogger(CompactionManager.class);
    private Iterable<Compactable> compactables;
    private volatile Map<CompactionServiceId, CompactionService> services;
    private LinkedBlockingQueue<Compactable> compactablesToCheck = new LinkedBlockingQueue();
    private long maxTimeBetweenChecks;
    private ServerContext context;
    private CompactionServicesConfig currentCfg;
    private long lastConfigCheckTime = System.nanoTime();
    private CompactionExecutorsMetrics ceMetrics;
    private String lastDeprecationWarning = "";
    private Map<CompactionExecutorId, ExternalCompactionExecutor> externalExecutors;
    private Map<ExternalCompactionId, ExtCompInfo> runningExternalCompactions;

    private void warnAboutDeprecation(String warning) {
        if (!warning.equals(this.lastDeprecationWarning)) {
            log.warn(warning);
            this.lastDeprecationWarning = warning;
        }
    }

    private void mainLoop() {
        long lastCheckAllTime = System.nanoTime();
        long increment = Math.max(1L, this.maxTimeBetweenChecks / 10L);
        Retry.RetryFactory retryFactory = Retry.builder().infiniteRetries().retryAfter(increment, TimeUnit.MILLISECONDS).incrementBy(increment, TimeUnit.MILLISECONDS).maxWait(this.maxTimeBetweenChecks, TimeUnit.MILLISECONDS).backOffFactor(1.07).logInterval(1L, TimeUnit.MINUTES).createFactory();
        Retry retry = retryFactory.createRetry();
        Compactable last = null;
        while (true) {
            try {
                while (true) {
                    long passed;
                    if ((passed = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - lastCheckAllTime)) >= this.maxTimeBetweenChecks) {
                        HashSet<ExternalCompactionId> runningEcids = new HashSet<ExternalCompactionId>(this.runningExternalCompactions.keySet());
                        Iterator<Compactable> iterator = this.compactables.iterator();
                        while (iterator.hasNext()) {
                            Compactable compactable;
                            last = compactable = iterator.next();
                            this.submitCompaction(compactable);
                            compactable.getExternalCompactionIds(runningEcids::remove);
                        }
                        lastCheckAllTime = System.nanoTime();
                        this.runningExternalCompactions.keySet().removeAll(runningEcids);
                    } else {
                        Compactable compactable = this.compactablesToCheck.poll(this.maxTimeBetweenChecks - passed, TimeUnit.MILLISECONDS);
                        if (compactable != null) {
                            last = compactable;
                            this.submitCompaction(compactable);
                        }
                    }
                    last = null;
                    if (retry.hasRetried()) {
                        retry = retryFactory.createRetry();
                    }
                    this.checkForConfigChanges(false);
                }
            }
            catch (Exception e) {
                KeyExtent extent = last == null ? null : last.getExtent();
                log.warn("Failed to compact {} ", (Object)extent, (Object)e);
                retry.useRetry();
                try {
                    retry.waitForNextAttempt(log, "compaction initiation loop");
                    continue;
                }
                catch (InterruptedException e1) {
                    log.debug("Retry interrupted", (Throwable)e1);
                    continue;
                }
            }
            break;
        }
    }

    private void submitCompaction(Compactable compactable) {
        for (CompactionKind ctype : CompactionKind.values()) {
            CompactionServiceId csid = compactable.getConfiguredService(ctype);
            CompactionService service = this.services.get(csid);
            if (service == null) {
                this.checkForConfigChanges(true);
                service = this.services.get(csid);
                if (service == null) {
                    log.error("Tablet {} returned non-existent compaction service {} for compaction type {}.  Check the table compaction dispatcher configuration. Attempting to fall back to {} service.", new Object[]{compactable.getExtent(), csid, ctype, CompactionServicesConfig.DEFAULT_SERVICE});
                    service = this.services.get(CompactionServicesConfig.DEFAULT_SERVICE);
                }
            }
            if (service == null) continue;
            service.submitCompaction(ctype, compactable, this.compactablesToCheck::add);
        }
    }

    public CompactionManager(Iterable<Compactable> compactables, ServerContext context, CompactionExecutorsMetrics ceMetrics) {
        this.compactables = compactables;
        this.currentCfg = new CompactionServicesConfig(context.getConfiguration(), this::warnAboutDeprecation);
        this.context = context;
        this.ceMetrics = ceMetrics;
        this.externalExecutors = new ConcurrentHashMap<CompactionExecutorId, ExternalCompactionExecutor>();
        this.runningExternalCompactions = new ConcurrentHashMap<ExternalCompactionId, ExtCompInfo>();
        HashMap tmpServices = new HashMap();
        this.currentCfg.getPlanners().forEach((serviceName, plannerClassName) -> {
            try {
                tmpServices.put(CompactionServiceId.of((String)serviceName), new CompactionService((String)serviceName, (String)plannerClassName, this.currentCfg.getRateLimit(serviceName), this.currentCfg.getOptions().getOrDefault(serviceName, Map.of()), context, ceMetrics, this::getExternalExecutor));
            }
            catch (RuntimeException e) {
                log.error("Failed to create compaction service {} with planner:{} options:{}", new Object[]{serviceName, plannerClassName, this.currentCfg.getOptions().getOrDefault(serviceName, Map.of()), e});
            }
        });
        this.services = Map.copyOf(tmpServices);
        this.maxTimeBetweenChecks = context.getConfiguration().getTimeInMillis(Property.TSERV_MAJC_DELAY);
        ceMetrics.setExternalMetricsSupplier(this::getExternalMetrics);
    }

    public void compactableChanged(Compactable compactable) {
        this.compactablesToCheck.add(compactable);
    }

    private synchronized void checkForConfigChanges(boolean force) {
        try {
            long secondsSinceLastCheck = TimeUnit.NANOSECONDS.toSeconds(System.nanoTime() - this.lastConfigCheckTime);
            if (!force && secondsSinceLastCheck < 1L) {
                return;
            }
            this.lastConfigCheckTime = System.nanoTime();
            CompactionServicesConfig tmpCfg = new CompactionServicesConfig(this.context.getConfiguration(), this::warnAboutDeprecation);
            if (!this.currentCfg.equals((Object)tmpCfg)) {
                HashMap tmpServices = new HashMap();
                tmpCfg.getPlanners().forEach((serviceName, plannerClassName) -> {
                    try {
                        CompactionServiceId csid = CompactionServiceId.of((String)serviceName);
                        CompactionService service = this.services.get(csid);
                        if (service == null) {
                            tmpServices.put(csid, new CompactionService((String)serviceName, (String)plannerClassName, tmpCfg.getRateLimit(serviceName), tmpCfg.getOptions().getOrDefault(serviceName, Map.of()), this.context, this.ceMetrics, this::getExternalExecutor));
                        } else {
                            service.configurationChanged((String)plannerClassName, tmpCfg.getRateLimit(serviceName), tmpCfg.getOptions().getOrDefault(serviceName, Map.of()));
                            tmpServices.put(csid, service);
                        }
                    }
                    catch (RuntimeException e) {
                        throw new RuntimeException("Failed to create or update compaction service " + serviceName + " with planner:" + plannerClassName + " options:" + tmpCfg.getOptions().getOrDefault(serviceName, Map.of()), e);
                    }
                });
                Sets.SetView deletedServices = Sets.difference(this.currentCfg.getPlanners().keySet(), tmpCfg.getPlanners().keySet());
                for (String serviceName2 : deletedServices) {
                    this.services.get(CompactionServiceId.of((String)serviceName2)).stop();
                }
                this.services = Map.copyOf(tmpServices);
                HashSet activeExternalExecs = new HashSet();
                this.services.values().forEach(cs -> cs.getExternalExecutorsInUse(activeExternalExecs::add));
                this.externalExecutors.keySet().retainAll(activeExternalExecs);
            }
        }
        catch (RuntimeException e) {
            log.error("Failed to reconfigure compaction services ", (Throwable)e);
        }
    }

    public void start() {
        log.debug("Started compaction manager");
        Threads.createThread((String)"Compaction Manager", () -> this.mainLoop()).start();
    }

    public CompactionServices getServices() {
        final Set<CompactionServiceId> serviceIds = this.services.keySet();
        return new CompactionServices(){

            public Set<CompactionServiceId> getIds() {
                return serviceIds;
            }
        };
    }

    public boolean isCompactionQueued(KeyExtent extent, Set<CompactionServiceId> servicesUsed) {
        return servicesUsed.stream().map(this.services::get).filter(Objects::nonNull).anyMatch(compactionService -> compactionService.isCompactionQueued(extent));
    }

    public int getCompactionsRunning() {
        return this.services.values().stream().mapToInt(cs -> cs.getCompactionsRunning(CompactionExecutor.CType.INTERNAL)).sum() + this.runningExternalCompactions.size();
    }

    public int getCompactionsQueued() {
        return this.services.values().stream().mapToInt(cs -> cs.getCompactionsQueued(CompactionExecutor.CType.INTERNAL)).sum() + this.externalExecutors.values().stream().mapToInt(ee -> ee.getCompactionsQueued(CompactionExecutor.CType.EXTERNAL)).sum();
    }

    public ExternalCompactionJob reserveExternalCompaction(String queueName, long priority, String compactorId, ExternalCompactionId externalCompactionId) {
        log.debug("Attempting to reserve external compaction, queue:{} priority:{} compactor:{}", new Object[]{queueName, priority, compactorId});
        ExternalCompactionExecutor extCE = this.getExternalExecutor(queueName);
        ExternalCompactionJob ecJob = extCE.reserveExternalCompaction(priority, compactorId, externalCompactionId);
        if (ecJob != null) {
            this.runningExternalCompactions.put(ecJob.getExternalCompactionId(), new ExtCompInfo(ecJob.getExtent(), extCE.getId()));
            log.debug("Reserved external compaction {}", (Object)ecJob.getExternalCompactionId());
        }
        return ecJob;
    }

    ExternalCompactionExecutor getExternalExecutor(CompactionExecutorId ceid) {
        return this.externalExecutors.computeIfAbsent(ceid, id -> new ExternalCompactionExecutor((CompactionExecutorId)id));
    }

    ExternalCompactionExecutor getExternalExecutor(String queueName) {
        return this.getExternalExecutor(CompactionExecutorIdImpl.externalId((String)queueName));
    }

    public void registerExternalCompaction(ExternalCompactionId ecid, KeyExtent extent, CompactionExecutorId ceid) {
        this.runningExternalCompactions.put(ecid, new ExtCompInfo(extent, ceid));
    }

    public void commitExternalCompaction(ExternalCompactionId extCompactionId, KeyExtent extentCompacted, Map<KeyExtent, Tablet> currentTablets, long fileSize, long entries) {
        ExtCompInfo ecInfo = this.runningExternalCompactions.get(extCompactionId);
        if (ecInfo != null) {
            Preconditions.checkState((boolean)ecInfo.extent.equals((Object)extentCompacted), (String)"Unexpected extent seen on compaction commit %s %s", (Object)ecInfo.extent, (Object)extentCompacted);
            Tablet tablet = currentTablets.get(ecInfo.extent);
            if (tablet != null) {
                tablet.asCompactable().commitExternalCompaction(extCompactionId, fileSize, entries);
                this.compactablesToCheck.add(tablet.asCompactable());
            }
            this.runningExternalCompactions.remove(extCompactionId);
        }
    }

    public void externalCompactionFailed(ExternalCompactionId ecid, KeyExtent extentCompacted, Map<KeyExtent, Tablet> currentTablets) {
        ExtCompInfo ecInfo = this.runningExternalCompactions.get(ecid);
        if (ecInfo != null) {
            Preconditions.checkState((boolean)ecInfo.extent.equals((Object)extentCompacted), (String)"Unexpected extent seen on compaction commit %s %s", (Object)ecInfo.extent, (Object)extentCompacted);
            Tablet tablet = currentTablets.get(ecInfo.extent);
            if (tablet != null) {
                tablet.asCompactable().externalCompactionFailed(ecid);
                this.compactablesToCheck.add(tablet.asCompactable());
            }
            this.runningExternalCompactions.remove(ecid);
        }
    }

    public List<TCompactionQueueSummary> getCompactionQueueSummaries() {
        return this.externalExecutors.values().stream().flatMap(ece -> ece.summarize()).collect(Collectors.toList());
    }

    public Collection<ExtCompMetric> getExternalMetrics() {
        HashMap metrics = new HashMap();
        this.externalExecutors.forEach((eeid, ece) -> {
            ExtCompMetric ecm = new ExtCompMetric();
            ecm.ceid = eeid;
            ecm.queued = ece.getCompactionsQueued(CompactionExecutor.CType.EXTERNAL);
            metrics.put(eeid, ecm);
        });
        this.runningExternalCompactions.values().forEach(eci -> {
            ExtCompMetric ecm = metrics.computeIfAbsent(eci.executor, id -> {
                ExtCompMetric newEcm = new ExtCompMetric();
                newEcm.ceid = id;
                return newEcm;
            });
            ++ecm.running;
        });
        return metrics.values();
    }

    public void compactableClosed(KeyExtent extent, Set<CompactionServiceId> servicesUsed, Set<ExternalCompactionId> ecids) {
        this.runningExternalCompactions.keySet().removeAll(ecids);
        servicesUsed.stream().map(this.services::get).filter(Objects::nonNull).forEach(compService -> compService.compactableClosed(extent));
    }

    public static class ExtCompMetric {
        public CompactionExecutorId ceid;
        public int running;
        public int queued;
    }

    static class ExtCompInfo {
        final KeyExtent extent;
        final CompactionExecutorId executor;

        public ExtCompInfo(KeyExtent extent, CompactionExecutorId executor) {
            this.extent = extent;
            this.executor = executor;
        }
    }
}

