/* Xlunzip - Test tool for the lzip_decompress linux module
   Copyright (C) 2016-2024 Antonio Diaz Diaz.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
/*
   Exit status: 0 for a normal exit, 1 for environmental problems
   (file not found, invalid command-line options, I/O errors, etc), 2 to
   indicate a corrupt or invalid input file, 3 for an internal consistency
   error (e.g., bug) which caused xlunzip to panic.
*/

#define _FILE_OFFSET_BITS 64

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>		/* SSIZE_MAX */
#include <signal.h>
#include <stdbool.h>
#include <stdint.h>		/* SIZE_MAX */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <utime.h>
#include <sys/stat.h>
#if defined __MSVCRT__ || defined __OS2__ || defined __DJGPP__
#include <io.h>
#if defined __MSVCRT__
#define fchmod(x,y) 0
#define fchown(x,y,z) 0
#define SIGHUP SIGTERM
#define S_ISSOCK(x) 0
#ifndef S_IRGRP
#define S_IRGRP 0
#define S_IWGRP 0
#define S_IROTH 0
#define S_IWOTH 0
#endif
#endif
#if defined __DJGPP__
#define S_ISSOCK(x) 0
#define S_ISVTX 0
#endif
#endif

#include "carg_parser.h"
#include "linux_lzip.h"
#include "linux_lunzip.h"
#include "lzip.h"

#ifndef O_BINARY
#define O_BINARY 0
#endif

#if CHAR_BIT != 8
#error "Environments where CHAR_BIT != 8 are not supported."
#endif

#if ( defined  SIZE_MAX &&  SIZE_MAX < ULONG_MAX ) || \
    ( defined SSIZE_MAX && SSIZE_MAX <  LONG_MAX )
#error "Environments where 'size_t' is narrower than 'long' are not supported."
#endif

static int verbosity = 0;
static void cleanup_and_fail( const int retval );
static void show_error( const char * const msg, const int errcode,
                        const bool help );

static const char * const program_name = "xlunzip";
static const char * const program_year = "2024";
static const char * invocation_name = "xlunzip";	/* default value */

static const struct { const char * from; const char * to; } known_extensions[] = {
  { ".lz",  ""     },
  { ".tlz", ".tar" },
  { 0,      0      } };

/* Variables used in signal handler context.
   They are not declared volatile because the handler never returns. */
static char * output_filename = 0;
static int outfd = -1;
static bool delete_output_on_interrupt = false;


static void show_help( void )
  {
  printf( "Xlunzip is a test tool for the lzip decompression code of my lzip patch for\n"
          "linux. Xlunzip is similar to lunzip, but it uses the lzip_decompress linux\n"
          "module as a backend. Xlunzip tests the module for stream, buffer-to-buffer,\n"
          "and mixed decompression modes, including in-place decompression (using the\n"
          "same buffer for input and output). You can use xlunzip to check that the\n"
          "module produces correct results when decompressing single member files,\n"
          "multimember files, or the concatenation of two or more compressed files.\n"
          "Xlunzip can be used with unzcrash to test the robustness of the module to\n"
          "the decompression of corrupted data.\n"
          "\nThe distributed index feature of the lzip format allows xlunzip to\n"
          "decompress concatenated files in place. This can't be guaranteed to work\n"
          "with formats like gzip or bzip2 because they can't detect whether a high\n"
          "compression ratio in the first members of the multimember data is being\n"
          "masked by a low compression ratio in the last members.\n"
          "\nUsage: %s [options] [files]\n", invocation_name );
  printf( "\nOptions:\n"
          "  -h, --help                 display this help and exit\n"
          "  -V, --version              output version information and exit\n"
          "  -c, --stdout               write to standard output, keep input files\n"
          "  -d, --decompress           decompress (this is the default)\n"
          "  -f, --force                overwrite existing output files\n"
          "  -I, --in-place             decompress or test using only one buffer\n"
          "  -k, --keep                 keep (don't delete) input files\n"
          "  -o, --output=<file>        write to <file>, keep input files\n"
          "  -q, --quiet                suppress all messages\n"
          "  -t, --test                 test compressed file integrity\n"
          "  -v, --verbose              be verbose (a 2nd -v gives more)\n"
          "These options are ignored when --in-place is in effect:\n"
          "      --insize[=<size>]      pre-allocate and fill inbuf [default 16 KiB]\n"
          "      --outsize[=<size>]     pre-allocate outbuf [default 512 MiB]\n"
          "      --nofill               do not pass a fill function; requires --insize\n"
          "      --noflush              do not pass a flush function; requires --outsize\n"
          "\nIf no file names are given, or if a file is '-', xlunzip decompresses\n"
          "from standard input to standard output.\n"
          "Numbers may be followed by a multiplier: k = kB = 10^3 = 1000,\n"
          "Ki = KiB = 2^10 = 1024, M = 10^6, Mi = 2^20, G = 10^9, Gi = 2^30, etc...\n"
          "\nTo extract all the files from archive 'foo.tar.lz', use the commands\n"
          "'tar -xf foo.tar.lz' or 'xlunzip -cd foo.tar.lz | tar -xf -'.\n"
          "\nExit status: 0 for a normal exit, 1 for environmental problems\n"
          "(file not found, invalid command-line options, I/O errors, etc), 2 to\n"
          "indicate a corrupt or invalid input file, 3 for an internal consistency\n"
          "error (e.g., bug) which caused xlunzip to panic.\n"
          "\nReport bugs to lzip-bug@nongnu.org\n"
          "Xlunzip home page: http://www.nongnu.org/lzip/xlunzip.html\n" );
  }


static void show_version( void )
  {
  printf( "%s %s\n", program_name, PROGVERSION );
  printf( "Copyright (C) %s Antonio Diaz Diaz.\n", program_year );
  printf( "License GPLv2+: GNU GPL version 2 or later <http://gnu.org/licenses/gpl.html>\n"
          "This is free software: you are free to change and redistribute it.\n"
          "There is NO WARRANTY, to the extent permitted by law.\n" );
  }


/* assure at least a minimum size for buffer 'buf' */
static void * resize_buffer( void * buf, const unsigned min_size )
  {
  if( buf ) buf = realloc( buf, min_size );
  else buf = malloc( min_size );
  if( !buf ) { show_error( mem_msg, 0, false ); cleanup_and_fail( 1 ); }
  return buf;
  }


static void Pp_init( struct Pretty_print * const pp,
                     const char * const filenames[], const int num_filenames )
  {
  pp->name = 0;
  pp->padded_name = 0;
  pp->stdin_name = "(stdin)";
  pp->longest_name = 0;
  pp->first_post = false;

  if( verbosity <= 0 ) return;
  const unsigned stdin_name_len = strlen( pp->stdin_name );
  int i;
  for( i = 0; i < num_filenames; ++i )
    {
    const char * const s = filenames[i];
    const unsigned len = (strcmp( s, "-" ) == 0) ? stdin_name_len : strlen( s );
    if( pp->longest_name < len ) pp->longest_name = len;
    }
  if( pp->longest_name == 0 ) pp->longest_name = stdin_name_len;
  }

static void Pp_set_name( struct Pretty_print * const pp,
                         const char * const filename )
  {
  unsigned name_len, padded_name_len, i = 0;

  if( filename && filename[0] && strcmp( filename, "-" ) != 0 )
    pp->name = filename;
  else pp->name = pp->stdin_name;
  name_len = strlen( pp->name );
  padded_name_len = max( name_len, pp->longest_name ) + 4;
  pp->padded_name = resize_buffer( pp->padded_name, padded_name_len + 1 );
  while( i < 2 ) pp->padded_name[i++] = ' ';
  while( i < name_len + 2 ) { pp->padded_name[i] = pp->name[i-2]; ++i; }
  pp->padded_name[i++] = ':';
  while( i < padded_name_len ) pp->padded_name[i++] = ' ';
  pp->padded_name[i] = 0;
  pp->first_post = true;
  }

static void Pp_show_msg( struct Pretty_print * const pp, const char * const msg )
  {
  if( verbosity < 0 ) return;
  if( pp->first_post )
    {
    pp->first_post = false;
    fputs( pp->padded_name, stderr );
    if( !msg ) fflush( stderr );
    }
  if( msg ) fprintf( stderr, "%s\n", msg );
  }


/* separate numbers of 5 or more digits in groups of 3 digits using '_' */
static const char * format_num3( unsigned long long num )
  {
  enum { buffers = 8, bufsize = 4 * sizeof num, n = 10 };
  const char * const si_prefix = "kMGTPEZYRQ";
  const char * const binary_prefix = "KMGTPEZYRQ";
  static char buffer[buffers][bufsize];	/* circle of static buffers for printf */
  static int current = 0;
  int i;
  char * const buf = buffer[current++]; current %= buffers;
  char * p = buf + bufsize - 1;		/* fill the buffer backwards */
  *p = 0;	/* terminator */
  if( num > 1024 )
    {
    char prefix = 0;			/* try binary first, then si */
    for( i = 0; i < n && num != 0 && num % 1024 == 0; ++i )
      { num /= 1024; prefix = binary_prefix[i]; }
    if( prefix ) *(--p) = 'i';
    else
      for( i = 0; i < n && num != 0 && num % 1000 == 0; ++i )
        { num /= 1000; prefix = si_prefix[i]; }
    if( prefix ) *(--p) = prefix;
    }
  const bool split = num >= 10000;

  for( i = 0; ; )
    {
    *(--p) = num % 10 + '0'; num /= 10; if( num == 0 ) break;
    if( split && ++i >= 3 ) { i = 0; *(--p) = '_'; }
    }
  return p;
  }


void show_option_error( const char * const arg, const char * const msg,
                        const char * const option_name )
  {
  if( verbosity >= 0 )
    fprintf( stderr, "%s: '%s': %s option '%s'.\n",
             program_name, arg, msg, option_name );
  }


/* Recognized formats: <num>k, <num>Ki, <num>[MGTPEZYRQ][i] */
static unsigned long getnum( const char * const arg,
                             const char * const option_name,
                             const unsigned long llimit,
                             const unsigned long ulimit )
  {
  char * tail;
  errno = 0;
  unsigned long result = strtoul( arg, &tail, 0 );
  if( tail == arg )
    { show_option_error( arg, "Bad or missing numerical argument in",
                         option_name ); exit( 1 ); }

  if( !errno && tail[0] )
    {
    const unsigned factor = ( tail[1] == 'i' ) ? 1024 : 1000;
    int exponent = 0;				/* 0 = bad multiplier */
    int i;
    switch( tail[0] )
      {
      case 'Q': exponent = 10; break;
      case 'R': exponent = 9; break;
      case 'Y': exponent = 8; break;
      case 'Z': exponent = 7; break;
      case 'E': exponent = 6; break;
      case 'P': exponent = 5; break;
      case 'T': exponent = 4; break;
      case 'G': exponent = 3; break;
      case 'M': exponent = 2; break;
      case 'K': if( factor == 1024 ) exponent = 1; break;
      case 'k': if( factor == 1000 ) exponent = 1; break;
      }
    if( exponent <= 0 )
      { show_option_error( arg, "Bad multiplier in numerical argument of",
                           option_name ); exit( 1 ); }
    for( i = 0; i < exponent; ++i )
      {
      if( ulimit / factor >= result ) result *= factor;
      else { errno = ERANGE; break; }
      }
    }
  if( !errno && ( result < llimit || result > ulimit ) ) errno = ERANGE;
  if( errno )
    {
    if( verbosity >= 0 )
      fprintf( stderr, "%s: '%s': Value out of limits [%s,%s] in "
               "option '%s'.\n", program_name, arg, format_num3( llimit ),
               format_num3( ulimit ), option_name );
    exit( 1 );
    }
  return result;
  }


static int extension_index( const char * const name )
  {
  int eindex;
  for( eindex = 0; known_extensions[eindex].from; ++eindex )
    {
    const char * const ext = known_extensions[eindex].from;
    const unsigned name_len = strlen( name );
    const unsigned ext_len = strlen( ext );
    if( name_len > ext_len &&
        strncmp( name + name_len - ext_len, ext, ext_len ) == 0 )
      return eindex;
    }
  return -1;
  }


static void set_d_outname( const char * const name, const int eindex )
  {
  const unsigned name_len = strlen( name );
  if( eindex >= 0 )
    {
    const char * const from = known_extensions[eindex].from;
    const unsigned from_len = strlen( from );
    if( name_len > from_len )
      {
      output_filename = resize_buffer( output_filename, name_len +
                                       strlen( known_extensions[eindex].to ) + 1 );
      strcpy( output_filename, name );
      strcpy( output_filename + name_len - from_len, known_extensions[eindex].to );
      return;
      }
    }
  output_filename = resize_buffer( output_filename, name_len + 4 + 1 );
  strcpy( output_filename, name );
  strcat( output_filename, ".out" );
  if( verbosity >= 1 )
    fprintf( stderr, "%s: %s: Can't guess original name -- using '%s'\n",
             program_name, name, output_filename );
  }


static int open_instream( const char * const name, struct stat * const in_statsp,
                          const bool one_to_one )
  {
  int infd = open( name, O_RDONLY | O_BINARY );
  if( infd < 0 )
    show_file_error( name, "Can't open input file", errno );
  else
    {
    const int i = fstat( infd, in_statsp );
    const mode_t mode = in_statsp->st_mode;
    const bool can_read = ( i == 0 &&
                            ( S_ISBLK( mode ) || S_ISCHR( mode ) ||
                              S_ISFIFO( mode ) || S_ISSOCK( mode ) ) );
    if( i != 0 || ( !S_ISREG( mode ) && ( !can_read || one_to_one ) ) )
      {
      if( verbosity >= 0 )
        fprintf( stderr, "%s: %s: Input file is not a regular file%s.\n",
                 program_name, name, ( can_read && one_to_one ) ?
                 ",\n  and neither '-c' nor '-o' were specified" : "" );
      close( infd );
      infd = -1;
      }
    }
  return infd;
  }


static bool open_outstream( const bool force, const bool protect )
  {
  const mode_t usr_rw = S_IRUSR | S_IWUSR;
  const mode_t all_rw = usr_rw | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH;
  const mode_t outfd_mode = protect ? usr_rw : all_rw;
  int flags = O_APPEND | O_CREAT | O_RDWR | O_BINARY;
  if( force ) flags |= O_TRUNC; else flags |= O_EXCL;

  outfd = open( output_filename, flags, outfd_mode );
  if( outfd >= 0 ) delete_output_on_interrupt = true;
  else if( errno == EEXIST )
    show_file_error( output_filename,
                     "Output file already exists, skipping.", 0 );
  else
    show_file_error( output_filename, "Can't create output file", errno );
  return outfd >= 0;
  }


static void set_signals( void (*action)(int) )
  {
  signal( SIGHUP, action );
  signal( SIGINT, action );
  signal( SIGTERM, action );
  }


static void cleanup_and_fail( const int retval )
  {
  set_signals( SIG_IGN );			/* ignore signals */
  if( delete_output_on_interrupt )
    {
    delete_output_on_interrupt = false;
    show_file_error( output_filename, "Deleting output file, if it exists.", 0 );
    if( outfd >= 0 ) { close( outfd ); outfd = -1; }
    if( remove( output_filename ) != 0 && errno != ENOENT )
      show_error( "warning: deletion of output file failed", errno, false );
    }
  exit( retval );
  }


static void signal_handler( int sig )
  {
  if( sig ) {}				/* keep compiler happy */
  show_error( "Control-C or similar caught, quitting.", 0, false );
  cleanup_and_fail( 1 );
  }


static bool check_tty_in( const char * const input_filename, const int infd,
                          const bool testing, int * const retval )
  {
  if( isatty( infd ) )				/* for example /dev/tty */
    { show_file_error( input_filename,
                       "I won't read compressed data from a terminal.", 0 );
      close( infd ); set_retval( retval, 2 );
      if( !testing ) cleanup_and_fail( *retval );
      return false; }
  return true;
  }


/* Set permissions, owner, and times. */
static void close_and_set_permissions( const struct stat * const in_statsp )
  {
  bool warning = false;
  if( in_statsp )
    {
    const mode_t mode = in_statsp->st_mode;
    /* fchown in many cases returns with EPERM, which can be safely ignored. */
    if( fchown( outfd, in_statsp->st_uid, in_statsp->st_gid ) == 0 )
      { if( fchmod( outfd, mode ) != 0 ) warning = true; }
    else
      if( errno != EPERM ||
          fchmod( outfd, mode & ~( S_ISUID | S_ISGID | S_ISVTX ) ) != 0 )
        warning = true;
    }
  if( close( outfd ) != 0 )
    { show_file_error( output_filename, "Error closing output file", errno );
      cleanup_and_fail( 1 ); }
  outfd = -1;
  delete_output_on_interrupt = false;
  if( in_statsp )
    {
    struct utimbuf t;
    t.actime = in_statsp->st_atime;
    t.modtime = in_statsp->st_mtime;
    if( utime( output_filename, &t ) != 0 ) warning = true;
    }
  if( warning && verbosity >= 1 )
    show_file_error( output_filename,
                     "warning: can't change output file attributes", errno );
  }


int convert_retval( const int retval )
  {
  switch( retval )
    {
    case 0:                return 0;
    case LZIP_OOM_INBUF:
    case LZIP_OOM_OUTBUF:
    case LZIP_WRITE_ERROR: return 1;
    case LZIP_HEADER1_EOF:
    case LZIP_HEADER2_EOF:
    case LZIP_BAD_MAGIC1:
    case LZIP_BAD_MAGIC2:
    case LZIP_BAD_VERSION:
    case LZIP_BAD_DICT_SIZE:
    case LZIP_BAD_DATA:
    case LZIP_DATA_EOF:
    case LZIP_BAD_CRC:     return 2;
    default:               return 3;
    }
  }


static int global_infd = -1;		/* needed by the fill function */

static long fill( void * buf, unsigned long size )
  {
  unsigned long sz = 0;
  errno = 0;
  while( sz < size )
    {
    const int n = read( global_infd, (uint8_t *)buf + sz, min( 1UL << 20, size - sz ) );
    if( n > 0 ) sz += n;
    else if( n == 0 ) break;				/* EOF */
    else if( errno != EINTR ) break;
    errno = 0;
    }
  return sz;
  }

long flush( void * buf, unsigned long size )
  {
  unsigned long sz = ( outfd >= 0 ) ? 0 : size;
  errno = 0;
  while( sz < size )
    {
    const int n = write( outfd, (uint8_t *)buf + sz, min( 1UL << 20, size - sz ) );
    if( n > 0 ) sz += n;
    else if( n < 0 && errno != EINTR ) break;
    errno = 0;
    }
  return sz;
  }

static const char * global_name;	/* copy of filename for 'error' */
static void error(char *x) { show_file_error( global_name, x, 0 ); }


static int decompress( const int infd, struct Pretty_print * const pp,
                       const long cl_insize, const long cl_outsize,
                       const bool nofill, const bool noflush, const bool testing )
  {
  long in_len = cl_insize;
  uint8_t * const inbuf = (in_len > 0) ? malloc( in_len ) : 0;
  long out_size = cl_outsize;
  uint8_t * const outbuf = (out_size > 0) ? malloc( out_size ) : 0;
  long in_pos, out_pos;
  int retval;
  if( ( in_len > 0 && !inbuf ) || ( out_size > 0 && !outbuf ) )
    { show_error( mem_msg, 0, false ); return 1; }

  global_infd = infd;
  if( inbuf )
    {
    const long len = fill( inbuf, in_len );
    if( len < in_len )
      { if( errno ) { show_file_error( pp->name, "Read error", errno );
                      global_infd = -1; return 1; }
        in_len = len; }
    }
  global_name = pp->name;
  retval = convert_retval( __lunzip( inbuf, in_len, nofill ? 0 : fill,
                                     noflush ? 0 : flush, outbuf, out_size,
                                     &in_pos, &out_pos, error ) );
  global_infd = -1;
  if( retval ) return retval;
  if( outbuf && noflush )
    {
    const long len = flush( outbuf, out_pos );
    if( len < out_pos )
      { show_file_error( pp->name, "Write error", errno ); return 1; }
    }
  show_results( pp, in_pos, out_pos, testing );
  return 0;
  }


void show_results( struct Pretty_print * const pp, const long in_pos,
                   const long out_pos, const bool testing )
  {
  if( verbosity >= 1 ) Pp_show_msg( pp, 0 );
  if( verbosity >= 2 )
    {
    if( out_pos <= 0 || in_pos <= 0 )
      fputs( "no data compressed. ", stderr );
    else
      fprintf( stderr, "%6.3f:1, %5.2f%% ratio, %5.2f%% saved. ",
               (double)out_pos / in_pos,
               ( 100.0 * in_pos ) / out_pos,
               100.0 - ( ( 100.0 * in_pos ) / out_pos ) );
    if( verbosity >= 3 )
      fprintf( stderr, "%9lu out, %8lu in. ", out_pos, in_pos );
    }
  if( verbosity >= 1 ) fputs( testing ? "ok\n" : "done\n", stderr );
  }


static void show_error( const char * const msg, const int errcode,
                        const bool help )
  {
  if( verbosity < 0 ) return;
  if( msg && msg[0] )
    fprintf( stderr, "%s: %s%s%s\n", program_name, msg,
             ( errcode > 0 ) ? ": " : "",
             ( errcode > 0 ) ? strerror( errcode ) : "" );
  if( help )
    fprintf( stderr, "Try '%s --help' for more information.\n",
             invocation_name );
  }


void show_file_error( const char * const filename, const char * const msg,
                      const int errcode )
  {
  if( verbosity >= 0 )
    fprintf( stderr, "%s: %s: %s%s%s\n", program_name, filename, msg,
             ( errcode > 0 ) ? ": " : "",
             ( errcode > 0 ) ? strerror( errcode ) : "" );
  }


static void internal_error( const char * const msg )
  {
  if( verbosity >= 0 )
    fprintf( stderr, "%s: internal error: %s\n", program_name, msg );
  exit( 3 );
  }


int main( const int argc, const char * const argv[] )
  {
  const char * default_output_filename = "";
  long cl_insize = 0;
  long cl_outsize = 0;
  int i;
  bool force = false;
  bool in_place = false;
  bool keep_input_files = false;
  bool nofill = false;
  bool noflush = false;
  bool testing = false;
  bool to_stdout = false;
  if( argc > 0 ) invocation_name = argv[0];

  enum { opt_insize = 256, opt_outsize, opt_nofill, opt_noflush };
  const struct ap_Option options[] =
    {
    { 'c', "stdout",          ap_no  },
    { 'd', "decompress",      ap_no  },
    { 'f', "force",           ap_no  },
    { 'h', "help",            ap_no  },
    { 'I', "in-place",        ap_no  },
    { 'k', "keep",            ap_no  },
    { 'n', "threads",         ap_yes },
    { 'o', "output",          ap_yes },
    { 'q', "quiet",           ap_no  },
    { 't', "test",            ap_no  },
    { 'v', "verbose",         ap_no  },
    { 'V', "version",         ap_no  },
    { opt_insize,  "insize",  ap_maybe  },
    { opt_outsize, "outsize", ap_maybe  },
    { opt_nofill,  "nofill",  ap_no  },
    { opt_noflush, "noflush", ap_no  },
    {  0, 0,                  ap_no  } };

  /* static because valgrind complains and memory management in C sucks */
  static struct Arg_parser parser;
  if( !ap_init( &parser, argc, argv, options, 0 ) )
    { show_error( mem_msg, 0, false ); return 1; }
  if( ap_error( &parser ) )				/* bad option */
    { show_error( ap_error( &parser ), 0, true ); return 1; }

  int argind = 0;
  for( ; argind < ap_arguments( &parser ); ++argind )
    {
    const int code = ap_code( &parser, argind );
    if( !code ) break;					/* no more options */
    const char * const pn = ap_parsed_name( &parser, argind );
    const char * const arg = ap_argument( &parser, argind );
    switch( code )
      {
      case 'c': to_stdout = true; break;
      case 'd': testing = false; break;
      case 'f': force = true; break;
      case 'h': show_help(); return 0;
      case 'I': in_place = true; break;
      case 'k': keep_input_files = true; break;
      case 'n': break;
      case 'o': if( strcmp( arg, "-" ) == 0 ) to_stdout = true;
                else { default_output_filename = arg; } break;
      case 'q': verbosity = -1; break;
      case 't': testing = true; break;
      case 'v': if( verbosity < 4 ) ++verbosity; break;
      case 'V': show_version(); return 0;
      case opt_insize:
        cl_insize = arg[0] ? getnum( arg, pn, 1, LONG_MAX ) : 16384; break;
      case opt_outsize: cl_outsize = arg[0] ?
        getnum( arg, pn, min_dictionary_size, LONG_MAX ) : max_dictionary_size;
        break;
      case opt_nofill: nofill = true; break;
      case opt_noflush: noflush = true; break;
      default: internal_error( "uncaught option." );
      }
    } /* end process options */

#if defined __MSVCRT__ || defined __OS2__ || defined __DJGPP__
  setmode( STDIN_FILENO, O_BINARY );
  setmode( STDOUT_FILENO, O_BINARY );
#endif

  static const char ** filenames = 0;
  int num_filenames = max( 1, ap_arguments( &parser ) - argind );
  filenames = resize_buffer( filenames, num_filenames * sizeof filenames[0] );
  filenames[0] = "-";

  bool filenames_given = false;
  for( i = 0; argind + i < ap_arguments( &parser ); ++i )
    {
    filenames[i] = ap_argument( &parser, argind + i );
    if( strcmp( filenames[i], "-" ) != 0 ) filenames_given = true;
    }

  if( testing ) to_stdout = false;			/* apply overrides */
  if( testing || to_stdout ) default_output_filename = "";

  output_filename = resize_buffer( output_filename, 1 );
  output_filename[0] = 0;
  if( to_stdout && !testing ) outfd = STDOUT_FILENO;
  else outfd = -1;

  const bool to_file = !to_stdout && !testing && default_output_filename[0];
  if( !to_stdout && !testing && ( filenames_given || to_file ) )
    set_signals( signal_handler );

  static struct Pretty_print pp;
  Pp_init( &pp, filenames, num_filenames );

  int failed_tests = 0;
  int retval = 0;
  const bool one_to_one = !to_stdout && !testing && !to_file;
  bool stdin_used = false;
  struct stat in_stats;
  for( i = 0; i < num_filenames; ++i )
    {
    const char * input_filename = "";
    int infd;

    Pp_set_name( &pp, filenames[i] );
    if( strcmp( filenames[i], "-" ) == 0 )
      {
      if( stdin_used ) continue; else stdin_used = true;
      infd = STDIN_FILENO;
      if( !check_tty_in( pp.name, infd, testing, &retval ) ) continue;
      if( one_to_one ) { outfd = STDOUT_FILENO; output_filename[0] = 0; }
      }
    else
      {
      input_filename = filenames[i];
      infd = open_instream( input_filename, &in_stats, one_to_one );
      if( infd < 0 ) { set_retval( &retval, 1 ); continue; }
      if( !check_tty_in( pp.name, infd, testing, &retval ) ) continue;
      if( one_to_one )			/* open outfd after checking infd */
        {
        set_d_outname( input_filename, extension_index( input_filename ) );
        if( !open_outstream( force, true ) )
          { close( infd ); set_retval( &retval, 1 ); continue; }
        }
      }

    if( to_file && outfd < 0 )		/* open outfd after checking infd */
      {
      output_filename = resize_buffer( output_filename,
                        strlen( default_output_filename ) + 1 );
      strcpy( output_filename, default_output_filename );
      if( !open_outstream( force, false ) ) return 1;
      }

    const struct stat * const in_statsp =
      ( input_filename[0] && one_to_one ) ? &in_stats : 0;
    int tmp;
    if( in_place )
      tmp = decompress_in_place( infd, &pp, testing );
    else
      tmp = decompress( infd, &pp, cl_insize, cl_outsize, nofill, noflush, testing );
    if( close( infd ) != 0 )
      { show_file_error( pp.name, "Error closing input file", errno );
        set_retval( &tmp, 1 ); }
    set_retval( &retval, tmp );
    if( tmp )
      { if( !testing ) cleanup_and_fail( retval );
        else ++failed_tests; }

    if( delete_output_on_interrupt && one_to_one )
      close_and_set_permissions( in_statsp );
    if( input_filename[0] && !keep_input_files && one_to_one )
      remove( input_filename );
    }
  if( delete_output_on_interrupt )					/* -o */
    close_and_set_permissions( ( retval == 0 && !stdin_used &&
      filenames_given && num_filenames == 1 ) ? &in_stats : 0 );
  else if( outfd >= 0 && close( outfd ) != 0 )				/* -c */
    {
    show_error( "Error closing stdout", errno, false );
    set_retval( &retval, 1 );
    }
  if( failed_tests > 0 && verbosity >= 1 && num_filenames > 1 )
    fprintf( stderr, "%s: warning: %d %s failed the test.\n",
             program_name, failed_tests,
             ( failed_tests == 1 ) ? "file" : "files" );
  free( output_filename );
  free( filenames );
  ap_free( &parser );
  return retval;
  }
