/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */

#ifndef __COMMON_H__
#define __COMMON_H__

#ifndef PACKAGE_NAME
#error config.h not included
#endif

#include <stdio.h>

#include <xine.h>
#include <xine/xineutils.h>
#include <xine/sorted_array.h>

typedef struct gGui_st gGui_t;

#include "xitk.h"

typedef struct xui_osd_s xui_osd_t;

typedef struct xui_keyedit_s xui_keyedit_t;
typedef struct xui_vwin_st xui_vwin_t;
typedef struct xui_actrl_st xui_actrl_t;
typedef struct xui_vctrl_st xui_vctrl_t;
typedef struct xui_event_sender_st xui_event_sender_t;
typedef struct xui_help_s xui_help_t;
typedef struct xui_mrlb_st xui_mrlb_t;
typedef struct xui_panel_st xui_panel_t;
typedef struct xui_mmkedit_s xui_mmkedit_t;
typedef struct xui_playlist_st xui_playlist_t;
typedef struct xui_setup_st xui_setup_t;
typedef struct xui_viewlog_s xui_viewlog_t;
typedef struct filebrowser_s filebrowser_t;
typedef struct xui_tvset_s xui_tvset_t;
typedef struct xui_tvout_s xui_tvout_t;
typedef struct xui_skdloader_s xui_skdloader_t;
typedef struct xui_sinfo_s xui_sinfo_t;
typedef struct xui_lirc_s xui_lirc_t;
typedef struct xui_network_remote_s xui_network_remote_t;

typedef struct kbinding_s kbinding_t;
typedef struct oxine_s oxine_t;
typedef struct xui_session_s xui_session_t;

#include "i18n.h"
#include "post.h"

#include "libcommon.h"

#ifdef HAVE_ORBIT
#include "../corba/xine-server.h"
#endif

/*
 * config related constants
 */
#define CONFIG_LEVEL_BEG  0 /* => beginner */
#define CONFIG_LEVEL_ADV 10 /* advanced user */
#define CONFIG_LEVEL_EXP 20 /* expert */
#define CONFIG_LEVEL_MAS 30 /* motku */
#define CONFIG_LEVEL_DEB 40 /* debugger (only available in debug mode) */

#define CONFIG_NO_DESC NULL
#define CONFIG_NO_HELP NULL
#define CONFIG_NO_CB   NULL
#define CONFIG_NO_DATA NULL

typedef enum action_id_e action_id_t;

/* mixer control method */
typedef enum {
  SOUND_CARD_MIXER = 0,
  SOFTWARE_MIXER,
  LAST_MIXER
} xui_mixer_type_t;

/* Playlist loop modes */
typedef enum {
  PLAYLIST_LOOP_NO_LOOP = 0, /* no loop (default) */
  PLAYLIST_LOOP_LOOP,        /* loop the whole playlist */
  PLAYLIST_LOOP_REPEAT,      /* loop the current mrl */
  PLAYLIST_LOOP_SHUFFLE,     /* random selection in playlist */
  PLAYLIST_LOOP_SHUF_PLUS,   /* random selection in playlist, never ending */
  PLAYLIST_LOOP_MODES_NUM
} playlist_loop_t;

#define PLAYLIST_CONTROL_STOP         0x00000001 /* Don't start next entry in playlist */
#define PLAYLIST_CONTROL_STOP_PERSIST 0x00000002 /* Don't automatically reset PLAYLIST_CONTROL_STOP */
#define PLAYLIST_CONTROL_IGNORE       0x00000004 /* Ignore some playlist adding action */

#define SAFE_FREE(x) do { free (x); (x) = NULL; } while (0)

/* Our default location for skin downloads */
#define SKIN_SERVER_URL "http://xine.sourceforge.net/skins/skins.slx"

#define fontname       "-*-helvetica-medium-r-*-*-12-*-*-*-*-*-*-*"
#define boldfontname   "-*-helvetica-bold-r-*-*-11-*-*-*-*-*-*-*"
#define hboldfontname  "-*-helvetica-bold-r-*-*-12-*-*-*-*-*-*-*"
#define lfontname      "-*-helvetica-bold-r-*-*-11-*-*-*-*-*-*-*"
#define tabsfontname   "-*-helvetica-bold-r-*-*-12-*-*-*-*-*-*-*"
#define br_fontname    "-misc-fixed-medium-r-normal-*-12-*-*-*-*-*-*-*"
#define btnfontname    "-*-helvetica-bold-r-*-*-12-*-*-*-*-*-*-*"

/* easy pseudo widgets for *_main calls. see the .h files for suport. */
#define XUI_W_TOGGLE NULL
#define XUI_W_ON ((xitk_widget_t *)1)
#define XUI_W_OFF ((xitk_widget_t *)2)
#define XUI_W_INIT ((xitk_widget_t *)3)
#define XUI_W_DEINIT ((xitk_widget_t *)4)

typedef struct skins_locations_s skins_locations_t;

typedef enum {
  XUI_EXTS_ANY = 0,
  XUI_EXTS_MEDIA,
  XUI_EXTS_AUDIO,
  XUI_EXTS_VIDEO,
  XUI_EXTS_IMAGE,
  XUI_EXTS_SUBTITLE,
  XUI_EXTS_PLAYLIST,
  XUI_EXTS_MANIFEST,
  XUI_EXTS_LAST
} xui_exts_t;

typedef struct alt_mrl_s alt_mrl_t;

typedef struct {
  const char *ident, *mrl, *sub;
  alt_mrl_t *alt[2]; /** << first, current */
  int        start; /** << seconds */
  int        end;   /** << seconds, -1 == till the end of stream */
  int        av_offset, spu_offset; /** << milliseconds */
  uint32_t   played; /** << | 1 (played, used with shuffle loop mode), | 2 (played at least 1 time or probed) */
  int        length; /** << milliseconds, or -1 if not known */
  enum mediamark_type_e {
    MMK_TYPE_FILE = 0,
    MMK_TYPE_NET
  }          type;
  enum mediamark_from_e {
    MMK_FROM_USER = 0,
    MMK_FROM_PLAYLIST,
    MMK_FROM_DIR
  }          from;
} mediamark_t;

struct gGui_st {
  struct gGui_st       *nextprev[3];

  char                 *cfg_file;

  xine_t               *xine;
  xine_video_port_t    *vo_port, *vo_none;
  xine_audio_port_t    *ao_port, *ao_none;
  xine_stream_t        *stream, *spu_stream;

  int                   verbosity;

  int                   broadcast_port;

  struct {
    /* Seek through xine_play () may be slow. We defer it to a separate thread. */
    pthread_mutex_t     mutex;
    int                 running; /** << 0 = no, 1 = yes, 2 = shutdown. */
    int                 newpos;
    int                 timestep;
    int                 pos_time_length[3];
  }                     seek;

  xine_event_queue_t   *event_queue;

  /* A set of videos that can play instead of audio visualzation on user request.
   * 0..(num_mrls - 1) are added by the command line -N pption.
   * num_mrls is the default or the one named in the current skin,
   * which is used when num_mrls == 0. */
  struct {
    xine_stream_t      *stream;
    xine_event_queue_t *event_queue;
    int                 running;
    int                 current;
    int                 enabled; /* 0, 1:vpost, 2:vanim */

    int                 num_mrls;
    char              **mrls;

    int                 post_plugin_num;
    int                 post_changed;
    post_element_t      post_output_element;
  } visual_anim;

  post_info_t           post_audio, post_video;

  xui_osd_t            *osd;

  int                   experience_level;

  int                   logo_mode;
  int                   logo_has_changed;
  int                   display_logo;
  const char           *logo_mrl;

  /* stuff like ACTID_x */
  int                   action_num_arg;

  char                 *autoscan_plugin;

  xitk_t               *xitk;

  /* basic X11 stuff */
  xitk_image_t         *icon;

  struct {
    int                 diff, n;
  }                     be_brake;

  xitk_skin_config_t   *skin_config;
  char                 *skin_server_url;

  int                   cursor_visible;
  int                   cursor_grabbed;

  int                   is_display_mrl; /** << show 0 (ident), 1 (mrl) */
  int                   runtime_mode;   /** << show 0 (elapsed time), 1 (remaining time) */

  /* Current mediamark */
  mediamark_t           mmk;
  /* Recursive mutex, protecting .mmk and .playlist. */
  pthread_mutex_t       mmk_mutex;

  /* playlist */
  struct {
    mediamark_t       **mmk;

    int                 max;        /** << number of entries that fit .mmk */
    int                 num;        /** << number of entries set in .mmk */
    int                 cur;        /** << current entry in playlist */
    int                 ref_append; /** << append mrl reference entry to this position */
    playlist_loop_t     loop;       /** << see PLAYLIST_LOOP_* */
    int                 control;    /** << see PLAYLIST_CONTROL_* */
    uint32_t            gen;        /** << up 1 for each change, except .cur */
    int                 played;     /** << number of entries played (0...num) */
    int                 probed;     /** << number of entries played at least 1 time or probed (0...num) */
    uint32_t            probe_mode; /** << | 1 (probe files), | 2 (probe network mrls) */
    uint32_t            probe_gen;  /** << avoid useless reprobe */

    xine_sarray_t      *exts[XUI_EXTS_LAST];
  } playlist;


  int                   on_quit;
  int                   running;
  int                   ignore_next;

  int                   stdctl_enable;

  struct {
    xui_mixer_type_t    type_volume, type_mute;
    int                 mute[LAST_MIXER];
    int                 level[LAST_MIXER]; /** << -1 if not supported */
  } mixer;

  struct {
    const char         *msg[4];
    int                 flags;
  } transform;

  int                   layer_above;

  int                   network;
  int                   network_port;
  xui_network_remote_t *network_remote;

  int                   use_root_window;

  int                   ssaver_enabled;
  int                   ssaver_timeout;

  int                   skip_by_chapter;

  const char           *snapshot_location;

  char                 *keymap_file;
  kbinding_t           *kbindings;
  int                   shortcut_style;    /** << 0 (windows), 1 (emacs). */
  int                   kbindings_enabled; /** << | 1 (enabled), | 2 (with implicit aliases). */

  /* event handling */
  int                   event_reject;
  int                   event_pending;
  pthread_mutex_t       event_mutex;
  pthread_cond_t        event_safe;

  char                  curdir[XITK_PATH_MAX];

#define XUI_FLAG_splash         0x0001
#define XUI_FLAG_no_gui         0x0002
#define XUI_FLAG_no_mouse       0x0004
#define XUI_FLAG_play_anyway    0x0008
#define XUI_FLAG_warn_too_slow  0x0010
#define XUI_FLAG_eventer_sticky 0x0020
#define XUI_FLAG_smart_mode     0x0040
#define XUI_FLAG_sinfo_update   0x0080
#define XUI_FLAG_auto_vo_vis    0x0100
#define XUI_FLAG_auto_panel_vis 0x0200
#define XUI_FLAG_deint          0x0400
#define XUI_FLAG_post_v         0x0800
#define XUI_FLAG_post_a         0x1000
#define XUI_FLAG_auto_subtitle  0x2000
#define XUI_FLAG_save_screen    0x4000
#define XUI_FLAG_wait_key_up    0x8000
#define XUI_FLAG_start_play     0x00010000
#define XUI_FLAG_start_plist    0x00020000
#define XUI_FLAG_start_vis      0x00040000
#define XUI_FLAG_start_vo_vis   0x00080000
#define XUI_FLAG_start_fullscr  0x00100000
#define XUI_FLAG_start_xinerama 0x00200000
#define XUI_FLAG_start_deint    0x00400000
#define XUI_FLAG_start_setup    0x00800000
#define XUI_FLAG_start_lirc     0x01000000
#define XUI_FLAG_start_quit     0x20000000
#define XUI_FLAG_trust_skin     0x40000000
  unsigned int          flags;

  unsigned int          gfx_saturation;

  FILE                 *report;

  void                (*nongui_error_msg) (gGui_t *gui, const char *text);

  FILE                 *orig_stdout; /* original stdout at startup        */
                                          /* before an evtl. later redirection */

  xitk_window_t        *splash_win;
  xui_panel_t          *panel;
  xui_vwin_t           *vwin;
  xui_setup_t          *setup;
  xui_mrlb_t           *mrlb;
  xui_actrl_t          *actrl;
  xui_vctrl_t          *vctrl;
  xui_event_sender_t   *eventer;
  xui_skdloader_t      *skdloader;
  xui_help_t           *help;
  xui_viewlog_t        *viewlog;
  xui_keyedit_t        *keyedit;
  xui_playlist_t       *plwin;
  xui_mmkedit_t        *mmkedit;
  xui_sinfo_t          *streaminfo;
  xui_tvset_t          *tvset;
  xui_tvout_t          *tvout;
  xui_lirc_t           *lirc;
  filebrowser_t        *pl_fb[2]; /** << load, save */

  oxine_t              *oxine;
  xui_session_t        *session;

  /* actions.c */
  struct {
    xine_stream_t      *stream;
    int                 start_pos;
    int                 start_time_in_secs;
    int                 update_mmk;
    int                 running;
  } play_data;
  filebrowser_t        *load_stream;
  filebrowser_t        *load_sub;
  int                   last_playback_speed;
  int                   last_step_time;

  struct {
    /** when multithreding, xitk_lock shall be enough to keep return valid. */
    int               (*get_names)(gGui_t *gui, const char **names, int max);
    xine_sarray_t      *a;
    skins_locations_t  *s[2]; /* default, current */
    int                 i[5]; /* default, current, changing, num, user_skin_selected */
  } skins;

  struct {
    int                 fd;
    pthread_t           thread;
    FILE               *fbk;
  } stdctl;

  /* NOTE:
   * 1. a failing xine_open () will wait up to 1 second for the stream event queues
   *    to process the error messages. Some old libxine may even wait infinitely.
   *    better defer event messages during such calls.
   * 2. drop exactly repeated messages.
   * 3. have at most 4 unanswered messages open.
   * 4. have at most 1 unanswered "more info" open. */
  struct {
    pthread_mutex_t     mutex;
    struct timeval      until;
    int                 level;
#define XUI_MAX_SHOWN_MSGS 4
    uint8_t             helipad[XUI_MAX_SHOWN_MSGS];
    xitk_register_key_t msg_windows[XUI_MAX_SHOWN_MSGS + 1]; /** << + more_info_win */
    unsigned int        msg_index;
    int                 used;
    unsigned int        flags[XUI_MAX_SHOWN_MSGS + 1]; /** << + last_flags */
    char                msg[XUI_MAX_SHOWN_MSGS + 1][8192]; /** << + last_msg */
  } no_messages;
};

void set_window_type_start(gGui_t *gui, xitk_window_t *xwin);
#define set_window_type_start(gui, xwin) \
  xitk_window_set_wm_window_type (xwin, video_window_is_visible ((gui)->vwin) < 2 ? WINDOW_TYPE_NORMAL : WINDOW_TYPE_NONE)

void gui_load_window_pos (gGui_t *gui, const char *name, int *x, int *y);
void gui_save_window_pos (gGui_t *gui, const char *name, xitk_register_key_t key);

#ifdef _XITK_H_
typedef struct gui_new_window_s {
  gGui_t *gui;         /** << in */
  const char *title;   /** << in */
  const char *id;      /** << in: window pos register name */
  const char *skin;    /** << in: NULL if noskin only */
  const char *wfskin;  /** << in: window focus image (optional) */
  void (*adjust) (struct gui_new_window_s *nw); /** << in: func to call before opening window or NULL */
  xitk_rect_t wr;      /** << in: default pos/noskin size or 0, out: actual pos/size */
  xitk_window_t *xwin; /** << out */
  xitk_image_t *bg;    /** << out */
  xitk_widget_list_t *wl; /** << out */
  xitk_register_key_t key; /** << in for gui_window_delete () */
} gui_new_window_t;
/** return -3 (skin error), -2 (window error), -1 (wrong args), 0 (noskin OK), 1 (skinned OK). */
int gui_window_new (gui_new_window_t *nw);
void gui_window_delete (gui_new_window_t *nw);
#endif

void gui_playlist_init (gGui_t *gui);
int gui_playlist_trylock (gGui_t *gui);
#define gui_playlist_trylock(_gui) pthread_mutex_trylock (&(_gui)->mmk_mutex)
void gui_playlist_lock (gGui_t *gui);
#define gui_playlist_lock(_gui) pthread_mutex_lock (&(_gui)->mmk_mutex)
void gui_playlist_unlock (gGui_t *gui);
#define gui_playlist_unlock(_gui) pthread_mutex_unlock (&(_gui)->mmk_mutex)
void gui_playlist_deinit (gGui_t *gui);
void gui_playlist_probe_start (gGui_t *gui);

#ifdef HAVE_XML_PARSER_REENTRANT
# define xml_parser_init_R(X,D,L,M) X = xml_parser_init_r ((D), (L), (M))
# define xml_parser_build_tree_R(X,T) xml_parser_build_tree_r ((X), (T))
# define xml_parser_finalize_R(X) xml_parser_finalize_r ((X))
#else
# define xml_parser_init_R(X,D,L,M) xml_parser_init ((D), (L), (M))
# define xml_parser_build_tree_R(X,T) xml_parser_build_tree ((T))
# define xml_parser_finalize_R(X) do {} while (0)
#endif

#endif
