/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

//----------------------------------------------------------------------------
//
//  File:       Options.h
//
//  Notes:      This file contains the definition of a generic class for command line options
//              parsing.
//
//----------------------------------------------------------------------------

#ifndef _OPTIONS_H
#define _OPTIONS_H

#include <map>
#include <vector>

/*
Options is a generic class for parsing command line arguments. The command line is expected to
be of the form: progname.exe -opt1 val1 -opt2 val2... where opt1 is the option name, and val1
is the value for this option.
The option names are specified in advance, and the ones specified are the only valid ones:
any other option name is considered an error. The values of the parameters are optional, but it
must be specified in advance whether each parameter has a value.
Any other parameter in the command line, that doesn't start with '-' is ignored, and would not
cause errors.
*/
class Options
{
private:
	//internal function object for comparing strings in map (case insensitive)
	struct StrLess
	{
		bool operator()(const char* left, const char* right) const;
	};

public:
	enum Error {
		ERROR_NONE = 0, ERROR_ARGC_ARGV, ERROR_OPTION_ALREADY_SPECIFIED,
		ERROR_INVALID_OPTION, ERROR_NO_OPTION_PARAM, ERROR_TOO_MANY_PARAMS,
		ERROR_NOT_ENOUGH_PARAMS};

	static const char DEFAULT_SWITCH_CHAR;

	//One node in the format list of the options.
	struct FormatNode
	{
		const char* option; //Name of option
		bool bHasParam; //Indication whether this option has a parameter
		bool bShouldRead; //Indication whether this option should read value from stdin
	};

	//class that represents the entire format of the options.
	class Format
	{
	public:
		//ctor to build Format from a list of FormatNode's
		Format(const FormatNode rawFormat[], int rawFormatSize,
			std::size_t numMandatoryParams = 0,
			char switchChar = Options::DEFAULT_SWITCH_CHAR);
		//default ctor
		Format();

		//Add one option to the format, with option name and parameter indication.
		void addOption(const char* option, bool bHasParam, bool shouldRead);

		//returns true iff a certain option exists in the format.
		bool optionExists(const char *option) const;

		//returns true iff a certain option has a parameter indicated by the
		//format. Sets error to ERROR_INVALID_OPTION if option is invalid.
		//clears error on success.
		bool optionHasParam(const char* option) const;

		//returns true if a certain option should read value from stdin
		//Sets error to ERROR_INVALID_OPTION if option is invalid.
		//clears error on success.
		bool optionShouldRead(const char* option) const;

		//set the number of mandatory parameters
		void setNumMandatoryParams(std::size_t num) {m_numMandatoryParams = num;}

		//returns the number of mandatory parameters
		std::size_t numMandatoryParams() const {return m_numMandatoryParams;}

		//set the switch char (the character that indicates an option)
		void setSwitchChar(char switchChar) {m_switchChar = switchChar;}

		//returns the switch char (the character that indicates an option)
		char switchChar() const {return m_switchChar;}

	private:
		typedef std::map<const char*, bool, StrLess> FormatMap;
		//Map that holds the association between option name and parameter indication.
		FormatMap m_formatMap;
		//Map that holds the association between option name and read input indication.
		FormatMap m_inputMap;
		std::size_t m_numMandatoryParams;
		char m_switchChar;
	};

	//ctor to build Options by parsing a command line, with a given Format object.
	Options(int argc, const char* argv[], const Format &format);

	//ctor to build Options by parsing a command line, with a given array of FormatNode's
	Options(int argc, const char* argv[], const FormatNode rawFormat[],
		int rawFormatSize, std::size_t numMandatoryParams = 0,
		char switchChar = DEFAULT_SWITCH_CHAR);

	//default ctor. parse() should be used after using it.
	Options();

	//Reset this Options object (clears error and isParsed status)
	void reset();

	//return the value of a given option.
	//Returns NULL if option doesn't exist, "" if option has no value.
	//If option doesn't exist, sets error to ERROR_INVALID_OPTION.
	//Clears error on success.
	const char* getOption(const char* option) const;

	//returns true iff a certain option exists in the Options object.
	bool optionExists(const char* option) const;

	//returns true iff parse was called - either directly or by a ctor.
	//Will return false if parsing wasn't done or if it failed.
	bool isParsed() const {return m_parsed;}

	//parse a command line according to Format object, and store results in this
	//Options object.
	//Returns true on success, false otherwise. On failure, using the error()
	//member function tells you the specific error. clears error on success.
	bool parse(int argc, const char* argv[], const Format &format);

	//parse a command line according to array of FormatNode's, and store results
	//in this Options object.
	//Returns true on success, false otherwise. On failure, using the error()
	//member function tells you the specific error. clears error on success.
	bool parse(int argc, const char* argv[], const FormatNode rawFormat[],
		int rawFormatSize, std::size_t numMandatoryParams = 0,
		char switchChar = DEFAULT_SWITCH_CHAR);

	//Returns the number of mandatory parameters in this options object
	std::size_t numMandatoryParams() const;

	//Returns the mandatory parameter of the specified index, or NULL
	//if the index is illegal. In this case, error() will return
	//ERROR_NOT_ENOUGH_PARAMS. clears error on success.
	const char* getMandatoryParam(unsigned int index) const;

	//Returns the error code of the last operation
	Error error() const {return m_error;}

	//Returns the string representation of the error code of the last operation.
	const char* errorStr() const;

	//Returns the string representation of a specified error code
	static const char* errorToStr(Error err);

private:
	bool setError(Error err) const;

	typedef std::map<const char*, const char*, StrLess> OptionsMap;

	OptionsMap m_optionsMap; //map that associates option names with their values.
	bool m_parsed; //was parse() called
	typedef std::vector<const char*> MandatoryParamsVector;
	MandatoryParamsVector m_MandatoryParams;
	mutable Error m_error;

};

#endif //#ifndef _OPTIONS_H
