use core:lang;
use core:asm;
use lang:bs:macro;

class CppDecl on Compiler {
	SStr name;
	Visibility? visibility;

	init(SStr name) {
		init { name = name; }
	}

	// Create this thing.
	void create(NameSet inside) : abstract;
}

class CppFnDecl extends CppDecl {
	STypeName result;
	Formal[] formals;

	// If 'null', then this is just a declaration.
	FnBody? body;

	init(STypeName result, SStr name, Formal[] formals, FnBody? body) {
		init(name) {
			result = result;
			formals = formals;
			body = body;
		}

		// We match "body" a bit too narrowly in the grammar.
		if (body) {
			body.stmt.pos.end++;
		}
	}

	// Create this function.
	void create(NameSet inside) {
		// Don't try to create declarations.
		unless (body)
			return;

		CppPlainFunction fn(cppScope(inside), this, inside, body);
		if (visibility)
			fn.visibility = visibility;
		inside.add(fn);
	}
}

// Operator declaration. Performs some checking and transformation of the name and parameter types.
CppFnDecl operatorDecl(STypeName result, SStr name, Formal[] formals, FnBody? body) {
	CppFnDecl(result, name, formals, body);
}

// Function body, contains a SStmtList and a "step" variable that indicates the presence of
// "NO_STEP" in the function declaration.
class FnBody {
	SStmtList stmt;
	Bool step;

	init(SStmtList stmt) {
		init { stmt = stmt; step = true; }
	}

	init(SStmtList stmt, Bool step) {
		init { stmt = stmt; step = step; }
	}
}

// A formal parameter.
value Formal {
	STypeName type;
	SStr name;

	init(STypeName type, SStr name) {
		init { type = type; name = name; }
	}
}

/**
 * A C++ function.
 */
class CppFunction extends Function {
	Scope scope;

	init(Scope scope, SrcPos pos, Value result, Str name, Value[] params) {
		init(pos, result, name, params) {
			scope = scope;
		}
	}

	Value findResult() {
		result;
	}

	Bool hasResultParam() {
		false;
	}
}

// Does this function add a parameter for the return value?
Bool hasResultParam(Function fn) {
	if (fn as CppPlainFunction) {
		return fn.hasResultParam();
	}
	false;
}

// Should this parameter be returned by a hidden parameter?
Bool returnByParam(Value type) {
	if (type.isAsmType())
		return false;

	if (type = type.type as PtrType)
		return false;

	return type.isValue();
}

/**
 * A plain member function (i.e. not a ctor or dtor).
 *
 * Note: If we're returning a "complex" type, we will add a hidden parameter right after the "this" parameter.
 */
class CppPlainFunction extends CppFunction {
	// Names of all formal parameters.
	private SStr[] formals;
	private Bool[] paramConst;
	private FnBody body;
	private Bool resultParam;

	init(Scope scope, CppFnDecl decl, NameSet inside, FnBody body) {
		Value[] fTypes;
		Bool[] fConst;
		SStr[] fNames;

		// Do we need a this-pointer?
		if (inside as Type) {
			fTypes << Value(thisPtr(inside));
			fConst << false;
			fNames << SStr("this", decl.name.pos);

			// We also need a hidden 'offset' ptr, that allows us to reconstruct a Ptr<> object.
			fTypes << Value(named{Nat});
			fConst << false;
			fNames << SStr("offset", decl.name.pos);
		}

		// Do we need a result parameter?
		Value result = decl.result.transform(scope);
		Bool resultParam = returnByParam(result);
		if (resultParam) {
			fTypes << wrapPtr(result).asRef();
			fConst << false;
			fNames << SStr("ret", decl.name.pos);
			result = Value();
		}

		for (f in decl.formals) {
			var type = f.type.transform(scope);
			// Everything is a reference in this C++ implementation.
			fTypes << type.asRef();
			fConst << type.const;
			fNames << f.name;
		}

		init(scope, decl.name.pos, result, decl.name.v, fTypes) {
			formals = fNames;
			paramConst = fConst;
			body = body;
			resultParam = resultParam;
		}

		setCode(LazyCode(&this.code));
	}

	// Find the result type.
	Value findResult() {
		if (hasResultParam) {
			if (isMember()) {
				return unwrapPtr(params[2]);
			} else {
				return unwrapPtr(params[0]);
			}
		} else {
			return result;
		}
	}

	// Did we use a result parameter?
	Bool hasResultParam() {
		resultParam;
	}

	// Generate code on demand.
	private CodeGen code() {
		FnRoot root(body.stmt.pos, this);
		root.step = body.step;

		Nat skip = 0;
		// Don't add 'this' and 'offset'.
		if (isMember())
			skip += 2;
		// Don't add 'return'.
		if (hasResultParam)
			skip += 1;

		// Add parameters.
		for (id, v in formals) {
			if (id >= skip)
				root.addParam(v.v, params[id], paramConst[id], v.pos);
		}

		body.stmt.transform(root);

		CodeGen gen(runOn, isMember, result);
		gen.l << prolog();

		if (root.code(gen)) {
			// Nothing returned. Perhaps we need to warn about that!
			if (result != Value())
				throw SyntaxError(pos, "Function returns a value, but return statements are missing.");

			// Generate a regular return.
			gen.l << fnRet();
		}

		// print("Generated code for ${name}:\n${gen.l}");

		gen;
	}
}

/**
 * Root block in a function. Knows a bit about the function to make "return" etc. work properly.
 */
class FnRoot extends StmtBlock {
	// Owner.
	CppFunction owner;

	// Return value of the function.
	Value result;

	// Result pointer, if applicable.
	Var resultPtr;

	// The 'this' parameter, if we're a member function. This is stored as a regular pointer and
	// handled as a special case to be compatible with the calling convention of the rest of
	// Storm. This is actually not too bad, since &this is not accepted by e.g. GCC on X86-64.
	Var thisPtr;

	// The offset of 'thisPtr', so that we can re-construct the this-pointer if we need to.
	Var thisOffset;

	// Parameters. Also added as usual to the block.
	private LocalVar[] params;

	init(SrcPos pos, CppFunction fn) {
		init(pos, fn.scope) {
			result = fn.findResult;
			owner = fn;
		}
	}

	// Add a parameter.
	void addParam(Str name, Value type, Bool const, SrcPos pos) {
		if (t = type.type) {
			LocalVar var(name, t, pos, const, false);
			add(var);
			params << var;
		}
	}

	// Custom code generation.
	Bool code(CodeGen gen) : override {
		// We don't need to create a child block. We use the root block right away!

		// this-ptr.
		if (owner.isMember()) {
			thisPtr = gen.l.createParam(ptrDesc);
			Value memberOf = owner.params[0];
			if (type = memberOf.type)
				gen.l.varInfo(thisPtr, Listing:VarInfo("this", type, true, owner.pos));

			// The offset parameter.
			thisOffset = gen.l.createParam(intDesc);
		}

		// result ptr.
		if (owner.hasResultParam) {
			Var formal = gen.l.createParam(ptrDesc);
			CppVar var = allocType(gen, safeType(wrapPtr(result), owner.pos));
			var.adjust(gen, ptrA);

			gen.l << mov(ptrB, formal);
			gen.l << mov(ptrRel(ptrA, Offset()), ptrRel(ptrB, Offset()));
			gen.l << mov(intRel(ptrA, Offset(sPtr)), intRel(ptrB, Offset(sPtr)));

			var.created(gen);
			resultPtr = var.v;
			if (type = result.type)
				gen.l.varInfo(resultPtr, Listing:VarInfo("↲", type, true, owner.pos));
		}

		for (param in params) {
			Var formal = gen.l.createParam(ptrDesc);
			CppVar var = allocType(gen, param.type);
			var.adjust(gen, ptrA);

			if (copy = param.type.copyCtor) {
				gen.l << fnParam(ptrDesc, ptrA);
				gen.l << fnParam(ptrDesc, formal);
				gen.l << fnCall(copy.ref, true);
			} else {
				Size sz = param.type.size;
				if (sz != Size()) {
					gen.l << mov(ptrB, formal);
					gen.l << mov(xRel(sz, ptrA), xRel(sz, ptrB));
				}
			}

			var.created(gen);
			param.var = var;
		}

		initVariables(gen);
		blockCode(gen);
	}

	// Check if we're a member function.
	Type? memberOf() {
		if (owner.isMember)
			return owner.params[0].type;
		null;
	}
}
