use ui;
use layout;
use graphics;
use core:geometry;
use core:io;
use progvis:net;
use lang:bs:macro;

/**
 * Main window.
 */
class MainWin extends Frame {
	private MainPainter painter;

	// Currently open file(s).
	private Url[] currentFiles;

	// Current behavior.
	private Behavior behavior;

	// Host to connect to when going online.
	private Str onlineUrl;

	// Current client connection, if any.
	private Client? client;

	// The 'online' menu so that we can enable/disable items as we please.
	// The first one is the "sign in" item.
	private PopupMenu onlineMenu;

	// Menu element for tracking reads/writes.
	private Menu:Check trackMemory;

	// Settings.
	private Settings settings;

	init() {
		init("Progvis", Size(800, 800)) {
			settings = Settings:load();
			onlineUrl = "storm-lang.org";
			trackMemory("Track reads/writes");
		}

		applySettings();

		PopupMenu fileMenu;
		fileMenu
			<< Menu:Text(mnemonic("_Open file..."), ctrlChord(Key:o), &this.onOpen())
			<< Menu:Text(mnemonic("_Reload program"), KeyChord(Key:r, Modifiers:ctrl + Modifiers:shift), &this.onReload())
			<< Menu:Text(mnemonic("Open in _editor"), ctrlChord(Key:e), &this.onOpenEditor())
			<< Menu:Separator()
			<< Menu:Text(mnemonic("_Settings..."), &this.onSettings());

		PopupMenu runMenu;
		runMenu
			<< Menu:Text(mnemonic("_Restart program"), ctrlChord(Key:r), &this.onRestart())
			<< Menu:Text(mnemonic("_Spawn new thread"), ctrlChord(Key:n), &this.onSpawnThread())
			<< trackMemory
			<< Menu:Separator()
			<< Menu:Text(mnemonic("S_top all threads"), ctrlChord(Key:q), &this.onStopAll())
			<< Menu:Text(mnemonic("Start _all threads (slow)"), ctrlChord(Key:a), &this.onRunSlow())
			<< Menu:Text(mnemonic("Start all threads (_fast)"), ctrlChord(Key:s), &this.onRunFast());

		trackMemory.checked = true;
		trackMemory.onClick = &this.onTrackMemory(Bool);
		painter.trackMemory(true);

		onlineMenu
			<< Menu:Text(mnemonic("_Connect"), &this.connect())
			<< Menu:Text(mnemonic("_Disconnect"), &this.disconnect())
			<< Menu:Text(mnemonic("_Status..."), &this.onOnlineStatus())
			<< Menu:Text(mnemonic("_Problems..."), ctrlChord(Key:p), &this.onOnlineProblems())
			<< Menu:Text("Submit problem...", &this.onOnlineSubmit())
			<< Menu:Text("Sign out", &this.logout());

		for (Nat i = 1; i < onlineMenu.count; i++)
			onlineMenu[i].enabled = false;

		PopupMenu helpMenu;
		helpMenu << Menu:Text(mnemonic("_About..."), &this.onAbout());

		MenuBar m;
		m
			<< Menu:Submenu(mnemonic("_File"), fileMenu)
			<< Menu:Submenu(mnemonic("_Run"), runMenu)
			<< Menu:Submenu(mnemonic("_Online"), onlineMenu)
			<< Menu:Submenu(mnemonic("_Help"), helpMenu);

		menu = m;

		painter(painter);
		create();
	}

	// Mostly for testing.
	void open(Url file) {
		open([file]);
	}

	// Called when the frame is closed.
	void close() : override {
		behavior.onDispose();
		painter.cleanup();
		super:close();
	}

	void applySettings() {
		painter.setZoom(settings.zoom);
		painter.repaint();
	}

	// For debug mode.
	assign onlineHost(Str to) {
		onlineUrl = to;
	}

	// Connect in the background.
	public void connect() {
		if (client.empty)
			(spawn bgConnect()).detach();
	}

	// Function executed in the background to connect.
	private void bgConnect() {
		onlineMenu[0].enabled = false;

		try {
			Client c = Client:connect(onlineUrl, settings.onlineId);
			this.client = c;

			for (Nat i = 1; i < onlineMenu.count; i++)
				onlineMenu[i].enabled = true;

			// This is only for testing.
			if (onlineUrl == "localhost")
				onOnlineProblems();
		} catch (SignInRedirect redirect) {
			showMessage(this, "Sign in", "You need to sign in before using the online features. Click OK to continue.");
			core:sys:open(redirect.to);
			onlineMenu[0].enabled = true;
		} catch (ServerError error) {
			showMessage(this, "Error", "The server returned an error: ${error.message}");
			onlineMenu[0].enabled = true;
		}
	}

	// For testing, to ensure that we can terminate the server cleanly.
	public void disconnect() {
		if (c = client) {
			c.close();
		}
		client = null;

		for (Nat i = 1; i < onlineMenu.count; i++)
			onlineMenu[i].enabled = false;
		onlineMenu[0].enabled = true;

		// Reset the behavior so that we don't try to use the client anymore.
		updateBehavior(DefaultBehavior(this));
	}

	private void logout() {
		if (c = client) {
			c.query(LogoutRequest());
			disconnect();
		}
	}

	Bool onMouseMove(Point pt) : override {
		painter.mouseMoved(pt);
		true;
	}

	void onMouseEnter() : override {}

	void onMouseLeave() : override {
		painter.mouseLeave();
	}

	Bool onClick(Bool down, Point pt, MouseButton button) {
		painter.mouseClicked(pt, down, button);
		true;
	}

	// Used to transition between behaviors. Typically set through "open", but some behaviors need
	// to switch during a simulation.
	public void updateBehavior(Behavior b) {
		behavior.onDispose();
		behavior = b;
		painter.updateBehavior(b);
		onNewBehavior();
		repaint();
	}

	// Called when a new behavior has been set.
	private void onNewBehavior() {
		if (!behavior.allowTrackMemory()) {
			trackMemory.enabled = false;
			trackMemory.checked = true;
			painter.trackMemory(true);
		} else {
			trackMemory.enabled = true;
		}
	}

	private void onOpen() {
		FileTypes ft("Source code");
		for (k in Program:supportedFiles)
			ft.add("${k}-source", [k]);

		FilePicker picker = FilePicker:open(ft).okLabel("Open").multiselect();
		if (!picker.show(this))
			return;

		open(picker.results);
	}

	public void onReload() {
		if (currentFiles.empty) {
			showMessage(this, "Error", "You need to open a file before you can reload it.");
		} else if (msg = behavior.allowReload()) {
			showMessage(this, "Error", msg);
		} else {
			try {
				painter.open(currentFiles, behavior);
			} catch (core:lang:CodeError error) {
				CompileErrorDialog dlg(error.messageText, error.pos);
				dlg.show(this);
			} catch (Exception error) {
				// Print the stack trace in the terminal to make it easier to debug.
				print("Error:\n${error}");
				showMessage(this, "Error opening code", "Unable to open the selected files:\n${error.message}");
			}
		}
	}

	public void onOpenEditor() {
		if (currentFiles.empty()) {
			showMessage(this, "No files open", "You have no files open currently.");
		} else {
			try {
				settings.open(currentFiles[0]);
			} catch (Exception e) {
				showMessage(this, "Filed to launch editor", "Failed to launch the editor: ${e}. Is your configuration correct?");
			}
		}
	}

	private void onOnlineStatus() {
		if (c = client) {
			StatusDlg(c).show(this);
		}
	}

	private void onOnlineProblems() {
		if (c = client) {
			problems:ProblemsDlg dlg(c, settings);
			dlg.show(this);
			if (action = dlg.action as problems:SolveProblem) {
				open(action, RecordBehavior(this, c, action));
			} else if (action = dlg.action as problems:ShowProblem) {
				open(action, RevisitBehavior(this, action.title));
			} else if (action = dlg.action as problems:ShowSolution) {
				open(action, ReplayBehavior(this, action.solution));
			}
		}
	}

	private void onOnlineSubmit() {
		if (msg = behavior.allowSubmit) {
			showMessage(this, "Unable to submit", msg);
			return;
		}

		if (c = client) {
			if (currentFiles.count != 1 | !painter.hasProgram) {
				showMessage(this, "Open a source file first", "To create a new problem, start by opening the file you want to upload.");
				return;
			}

			Url file = currentFiles[0];

			// Grab the source from the program, in case it changed on disk.
			if (text = painter.sourceFor(file)) {
				UploadDlg(c, text, file.ext).show(this);
			} else {
				showMessage(this, "Error", "Failed to find the source for this file.");
			}
		}
	}

	// Get source for the current file.
	public Str currentSource() {
		if (currentFiles.count != 1 | !painter.hasProgram)
			throw InternalError("Failed to acquire source for the current file.");
		if (text = painter.sourceFor(currentFiles[0]))
			return text;
		throw InternalError("Failed to acquire source for the current file.");
	}

	private void onSettings() {
		SettingsDlg(settings, &this.applySettings).show(this);
	}

	public void clearClient() {
		client = null;
	}

	public void onRestart() {
		painter.restart();
	}

	private void onSpawnThread() {
		painter.spawnThread();
	}

	private void onTrackMemory(Bool b) {
		painter.trackMemory(b);
		painter.repaint();
	}

	private void onStopAll() {
		painter.stopThreads();
	}

	private void onRunSlow() {
		painter.resumeThreads(1 s);
	}

	private void onRunFast() {
		painter.resumeThreads(500 ms);
	}

	private void onAbout() {
		var license = named{PROGVIS};
		var version = named{PROGVIS_VERSION};
		showLicenseDialog(this, ProgramInfo("Progvis", "Filip Strömbäck", version.version, license));
	}

	// Apply an action as if the user clicked something. The "step" received here is the same format
	// that the Behavior class receives in its "onUserAction".
	public void applyAction(Str step) {
		painter.applyAction(step);
	}

	// Open one or more files, shows nice messages on error.
	private void open(Url[] files) {
		open(files, join(files, ", ", (x) => x.name).toS, DefaultBehavior(this));
	}

	// Open a problem from the server.
	private void open(problems:Action problem, Behavior behavior) {
		open([problem.source], "Solving: " + problem.title, behavior);
	}

	private void open(Url[] files, Str title, Behavior behavior) {
		currentFiles = files;
		this.text = "Progvis - " + title;

		// Set it to default first, so that in case of failure we will behave as "normal".
		this.behavior.onDispose();
		this.behavior = DefaultBehavior(this);
		onNewBehavior();

		try {
			painter.cleanup();
			painter.open(files, behavior);
			this.behavior = behavior;
			onNewBehavior();
		} catch (core:lang:CodeError error) {
			CompileErrorDialog dlg(error.messageText, error.pos);
			dlg.show(this);
		} catch (Exception error) {
			// Print the stack trace in the terminal to make it easier to debug.
			print("Error:\n${error}");
			showMessage(this, "Error opening code", "Unable to open the selected files:\n${error.message}");
		}
	}

	// Called when an error is triggered from a program.
	private void onError(Nat threadId, Exception e) {
		showMessage(this, "Thread ${threadId} crashed", "Thread ${threadId} crashed with the following message:\n${e.message}");
	}

	// Called when a concurrency error is encountered.
	private void onConcurrencyError(Str[] messages) {
		StrBuf msg;
		msg << "Concurrency issues were encountered:\n";
		for (m in messages)
			msg << m << "\n";
		showMessage(this, "Concurrency issues", msg.toS);
	}
}


void main() on Compiler {
	named{progvis}.compile();

	MainWin win;
	win.waitForClose();
}

void debug() on Compiler {
	named{progvis}.compile();

	if (url = named{progvis}.url) {
		url = url / ".." / "progvis_demo";
		MainWin win;
		// win.open(url / "assert.c");
		// win.open(url / "atomics.c");
		// win.open(url / "demo2.bs");
		// win.open(url / "array.bs");
		// win.open(url / "cpp.cpp");
		// win.open(url / "cpp2.cpp");
		// win.open(url / "cpp3.cpp");
		// win.open(url / "members.cpp");
		// win.open(url / "special_functions.cpp");
		// win.open(url / "return_complex.cpp");
		// win.open(url / "arrays.cpp");
		// win.open(url / "cpp_error.cpp");
		// win.open(url / "str.cpp");
		// win.open(url / "const_example.cpp");
		// win.open(url / "pintos_debug.cpp");
		// win.open(url / "pintos_live.c");
		// win.open(url / "thread_example.c");
		// win.open(url / "printf.c");
		// win.open(url / "globals.cpp");
		win.open(url / "restore_globals.c");
		// win.open(url / "bank" / "bank.c");
		// win.open(url / "bank" / "bank_good.c");
		// win.open(url / "buffer" / "buffer.c");
		// win.open(url / "simple_buffer.c");
		// win.open(url / "cond.c");
		win.waitForClose();
	}
}

// Debug the online capabilities with a local server.
void debugServer() {
	named{progvis}.compile();

	spawn simpleServer();

	MainWin win;
	win.onlineHost = "localhost";
	sleep(100 ms);
	win.connect();
	win.waitForClose();
	win.disconnect();

	// Wait for the other UThread to realize that we have disconnected.
	sleep(500 ms);
}

// void testCpp() on Compiler {
// 	if (url = named{progvis_demo}.url) {
// 		if (p = Program:load(url / "cpp.cpp")) {
// 			p.run(p.main);
// 		} else {
// 			print("Failed to run the program!");
// 		}
// 	}
// }
