/***************************************************************************
 *   Copyright (C) 2000-2008 by Johan Maes                                 *
 *   on4qz@telenet.be                                                      *
 *   http://users.telenet.be/on4qz                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
/*!
	The dispatcher is the central system that routes all messages from the different threads.
It also starts, stops and synchronizes the threads.

*/
#include "dispatcher.h"
#include "global.h"
#include <QSettings>
#include <QSplashScreen>
#include <QMessageBox>
#include <QDialog>
#include "mainwindow.h"
#include "gallerywidget.h"
#include <sys/ioctl.h> 
//#include <unistd.h>
//#include <fcntl.h>

#include "configparams.h"
#include "configdialog.h"
#include "rxwidget.h"
#include "txwidget.h"
#include "editor/editor.h"

#include "ui_soundcontrol.h"
#include "supportfunctions.h"
#include "soundio.h"
#include "rxfunctions.h"
#include "txfunctions.h"
#include "ui_calibrationform.h"
#include "modes/modebase.h"
#include "ftp.h"
#include "rigcontrol.h"




/*!
creates dispatcher instance
 */

dispatcher::dispatcher()
{
	// start reading the configuration file
	serialP=0;
  logTimerIndex=startTimer(5000);
  rxRestart=FALSE;
  prTimerIndex=-1;
  confDiag=NULL;
}

/*!
delete dispatcher instance
 */

dispatcher::~dispatcher()
{

	delete txMW;
	delete rxMW;
	delete galMW;
	confDiag->writeSettings();

  writeSettings();
  prTimerIndex=-1;
  delete txFuncPtr;
  delete rxFuncPtr;
  delete confDiag;
  delete sndIO;
}

void dispatcher::init()
{

  rigController=new rigControl;
  confDiag=new configDialog(); // must be the first one to be started
	sndIO=new soundIO();
	readSettings();
// start the different windows
	sndIO->setSamplingrate(samplingrate);
  rxFuncPtr=new rxFunction();
  txFuncPtr=new txFunction();
  editorActive=FALSE;
}

void dispatcher::readSettings()
{
	QSettings qSettings;
	qSettings.beginGroup ("SOUND");
	sndIO->soundRoutingInput=  (soundIO::edataSrc)qSettings.value("soundRoutingInput",  0 ).toInt();
	sndIO->soundRoutingOutput= (soundIO::edataDst)qSettings.value("soundRoutingOutput", 0 ).toInt();
	sndIO->waveLen= qSettings.value("waveLen", 0 ).toInt();
	tpNumber=qSettings.value("tpNumber", 0 ).toInt();
	addNoise=qSettings.value("addnoise", FALSE ).toBool();
	noiseLevel=qSettings.value("noiselevel", -15).toDouble();
	qSettings.endGroup();
  logfile->readSettings(qSettings);
}

void dispatcher::writeSettings()
{
	QSettings qSettings;
	qSettings.beginGroup ("SOUND");
	qSettings.setValue ( "soundRoutingInput", sndIO->soundRoutingInput );
	qSettings.setValue ( "soundRoutingOutput",sndIO->soundRoutingOutput );
	qSettings.setValue ( "waveLen",sndIO->waveLen );
	qSettings.setValue ( "tpnumber",tpNumber );
	qSettings.setValue ( "addnoise",addNoise);
	qSettings.setValue ( "noiselevel",noiseLevel);
	qSettings.endGroup();
  logfile->writeSettings(qSettings);
}

/*!
	All communication between the threads are passed via this eventhandler.
*/

void dispatcher::customEvent( QEvent * e )
{
	int exit;
	dispatchEventType type;
	type=(dispatchEventType)e->type();
	addToLog(((baseEvent*)e)->description,DBDISPAT);
	switch(type)
		{
			case closeWindows:
			exit=QMessageBox::information(mw->centralWidget(), tr("Quit..."),
                                    tr("Do your really want to quit QSSTV?"),
                                    QMessageBox::Ok, QMessageBox::Cancel);
  			if(exit==QMessageBox::Ok)
            {
              stopRXTX();
              galMW->writeSettings();
              txMW->writeSettings();
              rxMW->writeSettings();
							QApplication::quit();
						}
			break;
			case callEditor:
				if(editorActive) break;
				editorActive=TRUE;
				ed=new editor();
				ed->show();
				iv=((callEditorEvent*)e)->getImageViewer();
        addToLog (QString(" callEditorEvent imageViewPtr: %1")
            .arg(QString::number((ulong)iv,16)),DBDISPAT);
        addToLog(QString("editor: filename %1").arg(((callEditorEvent*)e)->getFilename()),DBDISPAT);
				ed->openFile(((callEditorEvent*)e)->getFilename());
			break;

      case templatesChanged:
        txMW->setupTemplatesComboBox();
      break;
			case editorFinished:
				if(!editorActive) break;
				if(((editorFinishedEvent*)e)->isOK())
					{
            addToLog (QString(" editorFinishedEvent imageViewPtr: %1")
              .arg(QString::number((ulong)iv,16)),DBDISPAT);
            addToLog ("dummy",DBDISPAT);
            iv->reload();
					}
				editorActive=FALSE;
				delete ed;
			break;
			case soundcardIdle:
        rxViewer->updateImage();
        if(rxMW->autoSave)
          {
           // debug joma addToLog("dispatcher: soundcardIdle savingRxImage",DBDISPAT);
           //debug joma  saveRxImage();
          }
        addToLog("dispatcher: soundcardIdle",DBDISPAT);
				stopRXTX();
			break;
			case displayFFT:
					addToLog("dispatcher: displayFFT",DBDISPAT);
					rxMW->fftFrame()->realFFT(((displayFFTEvent*)e)->data());
					rxMW->fftFrame()->repaint();
			break;
      case displaySync:
        // addToLog("dispatcher: displaySync",DBDISPAT);
          uint s,v;
          ((displaySyncEvent*)e)->getInfo(s,v);
          rxMW->syncWidget()->setLeftValue(s);
          rxMW->syncWidget()->setRightValue(v);
         // rxMW->syncWidget()->repaint();
      break;



      case progressTX:
        txTimeCounter=0;
        prTimerIndex=startTimer(((progressTXEvent*)e)->getInfo()*10); // tim in seconds -> times 1000 for msec,divide by 100 for progress

      break;
      case startImageRX:
        addToLog("dispatcher: clearing RxImage",DBDISPAT);
        rxViewer->clear();
        ((startImageRXEvent*)e)->setDone();
      break;

      case verticalRetrace:
        addToLog("dispatcher: verticalRetrace",DBDISPAT);
        if(rxMW->autoSave)
          {
            addToLog("dispatcher: verticalRetrace savingRxImage",DBDISPAT);
            saveRxImage();
          }
        rxFuncPtr->retraceVertical();
      break;
      case syncLost:
        addToLog("dispatcher: syncLost",DBDISPAT);
        rxFuncPtr->syncLost();
        if(rxMW->autoSave)
          {
            addToLog("dispatcher:endImage savingRxImage",DBDISPAT);
            saveRxImage();
          }
        break;
        case endImageRX:
        if(rxMW->autoSave)
          {
            addToLog("dispatcher:endImage savingRxImage",DBDISPAT);
            saveRxImage();
          }
      break;

      case endImageTX:
        addToLog("dispatcher: endTXImage",DBDISPAT);
        sndIO->stop();
        while(sndIO->getStatus()!=soundIO::SND_IDLE)
          {
            qApp->processEvents();
          }
        rxRestart=TRUE;
      break;

			case createMode:
				{
          addToLog("dispatcher: createMode",DBDISPAT);
				}
			break;
			case lineDisplay:
				{
//					addToLog("dispatcher: update image",DBDISPAT);
            rxViewer->updateImage();
				}
			break;
			case statusMsg:
				{
          statusBarPtr->showMessage(((statusMsgEvent*)e)->getStr());
				}
      break;
      case changeRXFilter:
        {
          addToLog(QString("dispatcher: change RX filter: %1").arg(((filterRXChangedEvent *)e)->index()),DBALL);
          rxFuncPtr->setFilter(((filterRXChangedEvent *)e)->index());
        }
      break;
      case startRepeater:
      case startAutoRepeater:
        {
          sendImage();
        }
      break;
      case stopRxTx:
        stopRXTX();
      break;
			default:
        qDebug() << "dispatcher: unknown event: " << type;
        addToLog(QString("dispatcher: unknown event: %1").arg(type),DBALL);
			break;
		}
  ((baseEvent*)e)->setDone();
  addToLog("done set",DBDISPAT);
	e->accept();
}

void dispatcher::timerEvent(QTimerEvent *event)
 {
  if(event->timerId()==prTimerIndex)
    {
      txMW->setProgress(++txTimeCounter);
      if(txTimeCounter>=100)
       {
         if(prTimerIndex>=0)
           {
             killTimer(prTimerIndex);
             prTimerIndex=-1;
             txMW->setProgress(0);
           }
       }
      txMW->setProgress(txTimeCounter);
    }
  else if(event->timerId()==logTimerIndex)
    {
//      addToLog(QString("dumping dispatcher status"),DBALL);
//      if( rxFuncPtr) rxFuncPtr->logStatus();
//      if( txFuncPtr)txFuncPtr->logStatus();
//      if( sndIO)sndIO->logStatus();
    }
}

void dispatcher::slotSoundControl()
{
	QDialog dlg;
	Ui::soundControlDlg sndCtrl;
	sndCtrl.setupUi(&dlg);
	enum soundIO::edataSrc inp=sndIO->soundRoutingInput;
	enum soundIO::edataDst outp=sndIO->soundRoutingOutput;
	int wl=sndIO->waveLen;

	if(inp==soundIO::SNDIN) sndCtrl.sndIn->setChecked(TRUE);
	else if (inp==soundIO::SNDFILEIN) sndCtrl.sndFileIn->setChecked(TRUE);
	else sndCtrl.sndInToFile->setChecked(TRUE);

	if(outp==soundIO::SNDOUT) sndCtrl.sndOut->setChecked(TRUE);
	else sndCtrl.sndToSCRec->setChecked(TRUE);

	setValue(wl, sndCtrl.mbSpinBox);
	if(dlg.exec())
		{
			if (sndCtrl.sndIn->isChecked()) inp=soundIO::SNDIN;
			else if(sndCtrl.sndFileIn->isChecked()) inp=soundIO::SNDFILEIN;
			else inp=soundIO::SNDINTOFILE;

			if (sndCtrl.sndOut->isChecked()) outp=soundIO::SNDOUT;
			else outp=soundIO::SNDFILEOUT;;
			getValue(sndIO->waveLen,sndCtrl.mbSpinBox);
		}
	else
		{
			return;
		}
	if((inp!=sndIO->soundRoutingInput)||(outp!=sndIO->soundRoutingOutput))
		{
			stopRXTX();
			sndIO->soundRoutingInput=inp;
			sndIO->soundRoutingOutput=outp;
		}
}

void dispatcher::stopTX()
{
  if(prTimerIndex>=0)
  {
    killTimer(prTimerIndex);
    prTimerIndex=-1;
    txMW->setProgress(0);
  }
	activatePTT(FALSE);
  if(txFuncPtr==NULL) return;
  if(txFuncPtr->isRunning())
		{
      txFuncPtr->abort();
      txFuncPtr->wait();
      addToLog("dispatcher: TX stopped",DBDISPAT);
		}
}

void dispatcher::stopRX()
{
  if(rxFuncPtr==NULL) return;
  if(rxFuncPtr->isRunning())
		{
      rxFuncPtr->abort();
      addToLog("dispatcher:RX abort called",DBDISPAT);
      rxFuncPtr->wait();
      addToLog("dispatcher: RX stopped",DBDISPAT);
		}
}

void dispatcher::stopRXTX()
{
  addToLog("dispatcher: stopping RXTX",DBDISPAT);
	stopTX();
	stopRX();
	sndIO->abort();
	sndIO->wait();
  addToLog("dispatcher: RXTX stopped",DBDISPAT);
  if(rxRestart)
  {
    addToLog("dispatcher: trying to restart RX",DBDISPAT);
    rxRestart=FALSE;
    startRX();
  }
}


void dispatcher::activatePTT(bool b)
{
  int modemlines;
  if(enableSerialPTT)
  {
    if (pttSerialPort.isEmpty()) return;
    if(serialP==0)
      {
        serialP=::open(pttSerialPort.toLatin1().data(),O_RDONLY);
        if (serialP<=0)
          {
            QMessageBox::warning(txMW,"Serial Port Error",
            QString("Unable to open serial port %1\ncheck Options->Configuration\n"
            "make sure that you have read/write permission\nIf you do not have a serial port,\n"
            "then disable -Serial PTT- option in the configuration").arg(pttSerialPort) ,
            QMessageBox::Ok,0 );
            return;
          }
      }
    if(serialP>0)
      {
        if(b)
          {
          ioctl(serialP,TIOCMGET,&modemlines);
          modemlines |= TIOCM_DTR;
          modemlines |= TIOCM_RTS;
          ioctl(serialP,TIOCMSET,&modemlines);
          //ioctl(serial,TIOCMBIS,&t);
          }
        else
          {
          ioctl(serialP,TIOCMGET,&modemlines);
          modemlines &= ~TIOCM_DTR;
          modemlines &= ~TIOCM_RTS;
          ioctl(serialP,TIOCMSET,&modemlines);
          //	ioctl(serial,TIOCMBIC,&t);
          }
      }
  }
  else rigController->setPTT(b); // does nothing if rigController is disabled
  if(b)
    {
      addToLog("dispatcher: PTT activated",DBDISPAT);
    }
  else
   {
      addToLog("dispatcher: PTT deactivated",DBDISPAT);
   }
}


void dispatcher::startRX()
{
	QString errorString;
  if (rxFuncPtr->isRunning())
  {
    addToLog("dispatcher: restarting RX",DBDISPAT);
    rxFuncPtr->restart();
    return;
  }
	stopRXTX();
  addToLog("dispatcher: starting RX",DBDISPAT);
	rxMW->fftFrame()->init(RXSTRIPE,samplingrate);
	if(!sndIO->setRecord(&errorString,FALSE))
	{
		QMessageBox::critical(mw,"Sound Error",errorString);
		return ;
	}
	sndIO->start();
  rxFuncPtr->start();
}
void dispatcher::restartRX()
{
  rxRestart=TRUE;
  stopRXTX();
}

void dispatcher::resyncRX()
{
  if (!rxFuncPtr->isRunning()) return;
  addToLog("dispatcher: resync RX",DBDISPAT);
  rxFuncPtr->restart();
}


void dispatcher::startTX()
{
	QString errorString;
  if (txFuncPtr->isRunning()) return;
	stopRXTX();
  addToLog("dispatcher: starting TX",DBDISPAT);
	if(!sndIO->setPlayBack(&errorString,FALSE)) return ;
  activatePTT(TRUE);
	sndIO->start();
  txFuncPtr->start();
}


void dispatcher::startCalibration()
{
  #define NUMSAMPLES (1024*2000)
  //#define NUMSAMPLES (1024*20) // debug joma
	QTime time;
	QString temp;
  double rxFreq=0,txFreq=0;
	calibResult=-1;
  int timeValue=0;
  int i;
	int count=0;
	int totalCount=0;
	QString errorString;
	time.start();
  if (rxFuncPtr->isRunning()) return;
	stopRXTX();
  addToLog("dispatcher: starting calibration",DBDISPAT);
	if(!sndIO->setRecord(&errorString,TRUE)) return ;
	sndIO->start();
	QDialog *qd=new QDialog(0);
	Ui::CalibrationForm *cf=new Ui::CalibrationForm();
	cf->setupUi(qd);
	connect(qd,SIGNAL(finished(int)),SLOT(slotCalibFinished(int)));
  cf->rxProgressBar->setMinimum(0);
	cf->rxProgressBar->setMaximum(NUMSAMPLES);
  cf->txProgressBar->setMinimum(0);
	cf->txProgressBar->setMaximum(NUMSAMPLES);
	cf->txProgressBar->setValue(0);
	qd->show();
	for(i=0;i<5;)
		{
			count=sndIO->clkRxCheck();
      addToLog(QString("count=%1 %2").arg(count).arg(i),DBDISPAT);
			i++;
		}
	i=0;
	time.restart();
	totalCount=0;
	while((calibResult<0) && (totalCount<NUMSAMPLES))
		{
			totalCount+=sndIO->clkRxCheck();
			timeValue=time.elapsed();
			cf->rxProgressBar->setValue(totalCount);
			QApplication::processEvents();
		}
	if (calibResult<0)
		{
			rxFreq=((DSPFLOAT)(totalCount))/((DSPFLOAT)timeValue/1000.);
      addToLog(QString("result freq=%1, time=%2 total=%3").arg(rxFreq).arg(timeValue).arg(totalCount),DBDISPAT);

		}
	temp= QString("RX %1").arg(rxFreq,8,'f',3);
	cf->rxLabel->setText(temp);
	stopRXTX();
//	addToLog("dispatcher: starting TX",DBDISPAT);
	if(!sndIO->setPlayBack(&errorString,TRUE)) return ;
	sndIO->start();
	for(i=0;i<10;)
		{
			count=sndIO->clkTxCheck();
//		addToLog(QString("count=%1 %2").arg(count).arg(i),DBDISPAT);
			i++;
		}
	i=0;
	time.restart();
	totalCount=0;
	while((calibResult!=0) && (totalCount<NUMSAMPLES))
		{
			totalCount+=sndIO->clkTxCheck();
			timeValue=time.elapsed();
			cf->txProgressBar->setValue(totalCount);
			QApplication::processEvents();
		}
//	sndIO->clkTxCheckStop();
	if (calibResult<0)
		{
			txFreq=((DSPFLOAT)(totalCount))/((DSPFLOAT)timeValue/1000.);
 // 		addToLog(QString("result freq=%1, total=%2").arg(txFreq).arg(timeValue),DBDISPAT);
			temp= QString("TX %1").arg(txFreq,8,'f',3);
			cf->txLabel->setText(temp);
		}
	stopRXTX();
//	addToLog("dispatcher: stopping TX Check",DBDISPAT);
	if (calibResult>=0) return;
	while(calibResult==-1) QApplication::processEvents();
	if(calibResult==1)
		{
			//set config
			rxClock=rxFreq;
			txClock=txFreq;
		}
}



void dispatcher::slotCalibFinished(int result)
{
	calibResult=result;
}

void dispatcher::saveRxImage()
{
  if (rxFuncPtr->getModeString()->isEmpty()) return;
  if(!rxFuncPtr->saveOK()) return;
  QString s,tmp;
  QDateTime dt(QDateTime::currentDateTime().toUTC()); //this is compatible with QT 4.6
  dt.setTimeSpec(Qt::UTC);
  int intdate=dt.date().year()*10000+dt.date().month()*100+dt.date().day();
  int inttime=dt.time().hour()*10000+100*dt.time().minute()+dt.time().second();
  s.sprintf("%s_%d_%d",rxFuncPtr->getModeString()->toLatin1().data(),intdate,inttime);
  tmp=rxImagesPath+"/"+s+"."+defaultImageFormat;
  addToLog("dispatcher: saveRxImage() ",DBDISPAT);
  rxViewer->save(tmp,defaultImageFormat.toUpper());
  galMW->putRxImage(tmp);
  txMW->setPreviewWidget(tmp);
  if(enableFTP)
    {
      if(defaultImageFormat!=ftpDefaultImageFormat)
        {
          tmp=rxImagesPath+s+"."+ftpDefaultImageFormat;
          rxViewer->save(tmp,ftpDefaultImageFormat.toUpper());
          ftpIntf->uploadFile(tmp,ftpFilename+'.'+ftpDefaultImageFormat,TRUE);
        }
      else
        {
          ftpIntf->uploadFile(tmp,ftpFilename+'.'+ftpDefaultImageFormat,FALSE);
        }
    }
}

void dispatcher::setTXImage(QString fn)
{
  if(txFuncPtr->isRunning()) return;
  txMW->setImage(fn);
}

void dispatcher::setTXImage(QImage *im)
{
  if(txFuncPtr->isRunning()) return;
  txMW->setImage(im);
}

void dispatcher::sendImage()
{
  txFuncPtr->setTXState(txFunction::TXSSTVIMAGE);
  startTX();
 }

void dispatcher::setupTX(esstvMode m)
{
   txFuncPtr->create(m,txClock);
}

void dispatcher::setupRX(esstvMode m)
{
   rxFuncPtr->create(m,rxClock);
}

void dispatcher::sendTone(double duration,double freq)
{
  if(txFuncPtr->isRunning()) return;
  txFuncPtr->setToneParam(duration,freq);
  txFuncPtr->setTXState(txFunction::TXSENDTONE);
  addToLog("sendTone",DBDISPAT);
  startTX();
 }

void dispatcher::sendSweepTone(double duration,double lowerFreq,double upperFreq)
{
  if(txFuncPtr->isRunning()) return;
  txFuncPtr->setToneParam(duration,lowerFreq,upperFreq);
  txFuncPtr->setTXState(txFunction::TXSENDTONE);
  addToLog("sendSweepTone",DBDISPAT);
  startTX();
}

bool dispatcher::rxBusy()
{
  return rxFuncPtr->isReadingImage();
}


bool dispatcher::txBusy()
{
  return txFuncPtr->isRunning();
}





