/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "UnaryExpression.h"

#include <GTLCore/LLVMBackend/CodeGenerator_p.h>
#include <GTLCore/Type.h>
#include <GTLCore/LLVMBackend/ExpressionResult_p.h>
#include <GTLCore/Debug.h>
#include <GTLCore/LLVMBackend/ExpressionGenerationContext_p.h>
#include "GenerationVisitor.h"

using namespace GTLCore;
using namespace AST;

//--------- UnaryExpression ---------//

UnaryExpression::~UnaryExpression()
{
  delete m_rhs;
}

void UnaryExpression::markAsReturnExpression()
{
  m_rhs->markAsReturnExpression();
}

//--------- MinusUnaryExpression ---------//

LLVMBackend::ExpressionResult MinusUnaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc  ) const
{
  return _gc.codeGenerator()->createMinusExpression( _egc.currentBasicBlock(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type()  );
}

ExpressionResultSP MinusUnaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateMinusExpression( rightHandSide()->generateValue(_generationVisitor), annotation());
}

//--------- MinusMinusUnaryExpression ---------//

LLVMBackend::ExpressionResult MinusMinusUnaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc  ) const
{
  GTL_ASSERT( m_rhs );
  llvm::Value* ptr = m_rhs->pointer( _gc, _egc);
  _gc.codeGenerator()->createDecrementExpression( _gc, _egc.currentBasicBlock(), ptr );
  return rightHandSide()->generateValue( _gc, _egc);
}

ExpressionResultSP MinusMinusUnaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  GTL_ABORT("Unimplemented");
  return 0;
}

//--------- PlusPlusUnaryExpression ---------//

LLVMBackend::ExpressionResult PlusPlusUnaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc  ) const
{
  GTL_ASSERT( m_rhs );
  llvm::Value* ptr = m_rhs->pointer( _gc, _egc);
  _gc.codeGenerator()->createIncrementExpression( _egc.currentBasicBlock(), ptr );
  return rightHandSide()->generateValue( _gc, _egc);
}

ExpressionResultSP PlusPlusUnaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  GTL_ABORT("Unimplemented");
  return 0;
}

//--------- TildeUnaryExpression ---------//

const Type* NotUnaryExpression::type() const
{
  return Type::Boolean;
}

LLVMBackend::ExpressionResult NotUnaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc  ) const
{
  return _gc.codeGenerator()->createNotExpression( _egc.currentBasicBlock(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

ExpressionResultSP NotUnaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateNotExpression( rightHandSide()->generateValue(_generationVisitor), annotation());
}

//--------- TildeUnaryExpression ---------//

LLVMBackend::ExpressionResult TildeUnaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc  ) const
{
  return _gc.codeGenerator()->createTildeExpression( _egc.currentBasicBlock(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

ExpressionResultSP TildeUnaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateTildExpression( rightHandSide()->generateValue(_generationVisitor), annotation());
}
