/*  BEGIN software license
 *
 *  msXpertSuite - mass spectrometry software suite
 *  -----------------------------------------------
 *  Copyright(C) 2009, 2017 Filippo Rusconi
 *
 *  http://www.msxpertsuite.org
 *
 *  This file is part of the msXpertSuite project.
 *
 *  The msXpertSuite project is the successor of the massXpert project. This
 *  project now includes various independent modules:
 *  
 *  - massXpert, model polymer chemistries and simulate mass spectrometric data;
 *  - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * END software license
 */



/////////////////////// Qt includes
#include <QDebug>
#include <QRegularExpressionMatch>
#include <QScriptEngine>
#include <QScriptString>
#include <QScriptClassPropertyIterator>


/////////////////////// Local includes
#include <globals/globals.hpp>

#include <libmass/Trace.hpp>
#include <libmass/DataPointJs.hpp>
#include <libmass/TraceJs.hpp>
#include <libmass/TraceJsPrototype.hpp>
#include <libmass/TraceJsPropertyIterator.hpp>


#include <stdlib.h>


Q_DECLARE_METATYPE(msXpSlibmass::TraceJs*);
Q_DECLARE_METATYPE(msXpSlibmass::DataPoint*);


namespace msXpSlibmass
{


	TraceJs::TraceJs(QScriptEngine *engine)
		: 
			QObject(engine), QScriptClass(engine)
	{
		qScriptRegisterMetaType<Trace>(engine, toScriptValue, fromScriptValue);

		title = engine->toStringHandle(QLatin1String("title"));
		length = engine->toStringHandle(QLatin1String("length"));

		proto = engine->newQObject(new TraceJsPrototype(this),
				QScriptEngine::QtOwnership,
				QScriptEngine::SkipMethodsInEnumeration
				| QScriptEngine::ExcludeSuperClassMethods
				| QScriptEngine::ExcludeSuperClassProperties);

		QScriptValue global = engine->globalObject();

		proto.setPrototype(global.property("Object").property("prototype"));

		ctor = engine->newFunction(construct, proto);
		ctor.setData(engine->toScriptValue(this));
	}


	TraceJs::~TraceJs()
	{
	}


	QString
		TraceJs::name() const
		{
			return QLatin1String("Trace");
		}


	QScriptValue 
		TraceJs::prototype() const
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__;
			return proto;
		}


	QScriptValue 
		TraceJs::constructor()
		{
			return ctor;
		}


	QScriptValue 
		TraceJs::newInstance()
		{
			Trace trace;

			return newInstance(trace);
		}


	QScriptValue 
		TraceJs::newInstance(const QString &title)
		{
			Trace trace(title);

			return newInstance(trace);
		}


	QScriptValue 
		TraceJs::newInstance(const Trace &other)
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
				//<< "Right before getting QScriptValue as newVariant(QVariant::fromValue(Trace))";

			QScriptValue data = engine()->newVariant(QVariant::fromValue(other));

			// 'this', below, is because the JS object (data) needs to be linked to
			// the corresponding JS class that must match the class of the object.
			// 'data' is an object of Trace class and thus needs to be
			// associated to the TraceJs class (that is, this).

			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
				//<< "Right before returning QScriptValue as newObject(this, data);";

			return engine()->newObject(this, data);
		}


	QScriptValue::PropertyFlags TraceJs::propertyFlags(const QScriptValue &/*object*/,
			const QScriptString &name, uint /*id*/)
	{
		if (name == title) 
			return QScriptValue::Undeletable;

		return QScriptValue::Undeletable;
	}


	QScriptClass::QueryFlags 
		TraceJs::queryProperty(const QScriptValue &object,
				const QScriptString &name, QueryFlags flags, uint *id)
		{
			Trace *trace = qscriptvalue_cast<Trace *>(object.data());

			if (trace == Q_NULLPTR)
				return 0;

			if(name == title) 
			{
				return flags;
			}

			if(name == length) 
			{
				return flags;
			}

			bool isArrayIndex;

			qint32 index = name.toArrayIndex(&isArrayIndex);

			if(!isArrayIndex)
				return 0;

			*id = index;

			if((flags & HandlesReadAccess) && (index >= trace->size()))
				flags &= ~HandlesReadAccess;

			return flags;
		}


	QScriptValue 
		TraceJs::property(const QScriptValue &object,
				const QScriptString &name, uint id)
		{
			Trace *trace = qscriptvalue_cast<Trace *>(object.data());

			if(!trace)
				return QScriptValue();

			if(name == title)
			{
				return QScriptValue(trace->m_title);
			}
			else if(name == length) 
			{
				return trace->size();
			}
			else
			{
				qint32 index = id;

				if((index < 0) || (index >= trace->size()))
					return QScriptValue();

				//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ 
					//<< "The index is:" << index
					//<< "and the variant:" << QVariant::fromValue(*(trace->at(index)));

				QScriptValue data = engine()->newVariant(QVariant::fromValue(trace->at(index)));

				// We need to allocate the proper JS proxy for the object class: the
				// class of the native object is DataPoint, so we need to allocate a
				// DataPointJs instance to feed to newObject() below. Only if we do this,
				// will we have a properly behaving DataPoint JS object when accessing
				// the individual members of the mass spectrum.

				DataPointJs *dataPointJs = new DataPointJs(engine());

				return engine()->newObject(dataPointJs, data);
			}

			return QScriptValue();
		}


	void 
		TraceJs::setProperty(QScriptValue &object, const QScriptString &name,
				uint id, const QScriptValue &value)
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()";

			Trace *trace = qscriptvalue_cast<Trace *>(object.data());

			if(name == title)
			{
				trace->m_title = qscriptvalue_cast<QString>(value);
			}

			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()"
				//<< "Right before returning void";
			return ;
		}


	QScriptClassPropertyIterator *
		TraceJs::newIterator(const QScriptValue &object)
		{
			return new TraceJsPropertyIterator(object);
		}



	QScriptValue
		TraceJs::construct(QScriptContext *ctx, QScriptEngine *)
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" ;

			TraceJs *msJs = qscriptvalue_cast<TraceJs*>(ctx->callee().data());

			if (!msJs)
				return QScriptValue();

			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()"
				//<< "Found the TraceJs class.";

			int argCount = ctx->argumentCount();

			if(argCount == 1)
			{
				QScriptValue arg = ctx->argument(0);

				if(arg.instanceOf(ctx->callee()))
				{
					//qDebug() << __FILE__ << __LINE__ << __FUNCTION__<< "()"
						//<< "The argument passed to the constructor is also a Trace."
						//<< "Right before returning newInstance(Trace-C++ casted from QScritValue)";

					return msJs->newInstance(qscriptvalue_cast<Trace>(arg));
				}

				if(arg.isString())
				{
					// The string is the title of the mass spectrum.
					return msJs->newInstance(Trace(qscriptvalue_cast<QString>(arg)));
				}
			}

			// By default return an empty mass spectrum.
			return msJs->newInstance();
		}


	QScriptValue 
		TraceJs::toScriptValue(QScriptEngine *eng, const Trace &trace)
		{
			QScriptValue ctor = eng->globalObject().property("Trace");

			TraceJs *msJs = qscriptvalue_cast<TraceJs*>(ctor.data());

			if (!msJs)
				return eng->newVariant(QVariant::fromValue(trace));

			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
				//<< "Right before newInstance(Trace-C++)";

			return msJs->newInstance(trace);
		}


	void 
		TraceJs::fromScriptValue(const QScriptValue &obj, Trace &trace)
		{
			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
				//<< "Right before trace = qvariant_cast<Trace>(obj.data().toVariant());"
				//<< "to fill-in Trace" << &trace;

			QVariant variant = obj.data().toVariant();

			qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
			<< "Variant type:" << variant.userType();

			trace = qvariant_cast<Trace>(variant);

			//qDebug() << __FILE__ << __LINE__ << __FUNCTION__ << "()" 
				//<< "Right after trace = qvariant_cast<Trace>(obj.data().toVariant());";
		}


} // namespace msXpSlibmass
