#region License
/*
Microsoft Public License (Ms-PL)
XnaTouch - Copyright © 2009 The XnaTouch Team

All rights reserved.

This license governs use of the accompanying software. If you use the software, you accept this license. If you do not
accept the license, do not use the software.

1. Definitions
The terms "reproduce," "reproduction," "derivative works," and "distribution" have the same meaning here as under 
U.S. copyright law.

A "contribution" is the original software, or any additions or changes to the software.
A "contributor" is any person that distributes its contribution under this license.
"Licensed patents" are a contributor's patent claims that read directly on its contribution.

2. Grant of Rights
(A) Copyright Grant- Subject to the terms of this license, including the license conditions and limitations in section 3, 
each contributor grants you a non-exclusive, worldwide, royalty-free copyright license to reproduce its contribution, prepare derivative works of its contribution, and distribute its contribution or any derivative works that you create.
(B) Patent Grant- Subject to the terms of this license, including the license conditions and limitations in section 3, 
each contributor grants you a non-exclusive, worldwide, royalty-free license under its licensed patents to make, have made, use, sell, offer for sale, import, and/or otherwise dispose of its contribution in the software or derivative works of the contribution in the software.

3. Conditions and Limitations
(A) No Trademark License- This license does not grant you rights to use any contributors' name, logo, or trademarks.
(B) If you bring a patent claim against any contributor over patents that you claim are infringed by the software, 
your patent license from such contributor to the software ends automatically.
(C) If you distribute any portion of the software, you must retain all copyright, patent, trademark, and attribution 
notices that are present in the software.
(D) If you distribute any portion of the software in source code form, you may do so only under this license by including 
a complete copy of this license with your distribution. If you distribute any portion of the software in compiled or object 
code form, you may only do so under a license that complies with this license.
(E) The software is licensed "as-is." You bear the risk of using it. The contributors give no express warranties, guarantees
or conditions. You may have additional consumer rights under your local laws which this license cannot change. To the extent
permitted under your local laws, the contributors exclude the implied warranties of merchantability, fitness for a particular
purpose and non-infringement.
*/
#endregion License

#region Using Statements
using System;
using System.Drawing;
using System.Collections.Generic;

using MonoMac.CoreAnimation;
using MonoMac.Foundation;
using MonoMac.ObjCRuntime;
using MonoMac.OpenGL;
using MonoMac.AppKit;

using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Input.Touch;

#endregion Using Statements

namespace Microsoft.Xna.Framework
{
	public class GameWindow : MonoMacGameView
	{
		//private readonly Rectangle clientBounds;
		private Rectangle clientBounds;
		private Game _game;
        private MacGamePlatform _platform;

		private GameTime _updateGameTime;
		private GameTime _drawGameTime;
		private DateTime _lastUpdate;
		private DateTime _now;
		private NSTrackingArea _trackingArea;
		private bool _needsToResetElapsedTime = false;
		private bool _isFirstTime = true;
		private TimeSpan _extraElapsedTime;
		#region UIVIew Methods		
		public GameWindow(Game game, RectangleF frame) : base (frame)
		{
            if (game == null)
                throw new ArgumentNullException("game");
            _game = game;
            _platform = (MacGamePlatform)_game.Services.GetService(typeof(MacGamePlatform));

			//LayerRetainsBacking = false; 
			//LayerColorFormat	= EAGLColorFormat.RGBA8;
			this.AutoresizingMask = MonoMac.AppKit.NSViewResizingMask.HeightSizable
					| MonoMac.AppKit.NSViewResizingMask.MaxXMargin 
					| MonoMac.AppKit.NSViewResizingMask.MinYMargin
					| MonoMac.AppKit.NSViewResizingMask.WidthSizable;
			
			RectangleF rect = NSScreen.MainScreen.Frame;
			
			clientBounds = new Rectangle (0,0,(int)rect.Width,(int)rect.Height);

			// Enable multi-touch
			//MultipleTouchEnabled = true;

			// Initialize GameTime
			_updateGameTime = new GameTime ();
			_drawGameTime = new GameTime (); 

			// Initialize _lastUpdate
			_lastUpdate = DateTime.Now;
		}

		public GameWindow(Game game, RectangleF frame, NSOpenGLContext context) :
            this(game, frame)
		{
		}

		[Export("initWithFrame:")]
		public GameWindow () : base (NSScreen.MainScreen.Frame)
		{
			this.AutoresizingMask = MonoMac.AppKit.NSViewResizingMask.HeightSizable
					| MonoMac.AppKit.NSViewResizingMask.MaxXMargin 
					| MonoMac.AppKit.NSViewResizingMask.MinYMargin
					| MonoMac.AppKit.NSViewResizingMask.WidthSizable;

			RectangleF rect = NSScreen.MainScreen.Frame;
			clientBounds = new Rectangle (0,0,(int)rect.Width,(int)rect.Height);

			// Enable multi-touch
			//MultipleTouchEnabled = true;

			// Initialize GameTime
			//_updateGameTime = new GameTime ();
			//_drawGameTime = new GameTime ();

			// Initialize _lastUpdate
			//_lastUpdate = DateTime.Now;
		}

		~GameWindow ()
		{
			//
		}

		#endregion

		public void StartRunLoop(double updateRate)
		{
			Run(updateRate);
			// Initialize GameTime
			_updateGameTime = new GameTime ();
			_drawGameTime = new GameTime ();
			_now = _lastUpdate = DateTime.Now;
		}

		public void ResetElapsedTime ()
		{
			_needsToResetElapsedTime = true;
		}
		#region MonoMacGameView Methods

		protected override void OnClosed (EventArgs e)
		{
			base.OnClosed (e);
		}

		protected override void OnDisposed (EventArgs e)
		{
			base.OnDisposed (e);
		}

		protected override void OnLoad (EventArgs e)
		{
			base.OnLoad (e);
		}

		protected override void OnRenderFrame (FrameEventArgs e)
		{
			base.OnRenderFrame (e);

            // FIXME: Since Game.Exit may be called during an Update loop (and
            //        in fact that is quite likely to happen), this code is now
            //        littered with checks to _platform.IsRunning.  It would be
            //        nice if there weren't quite so many.  The move to a
            //        Game.Tick-centric architecture may eliminate this problem
            //        automatically.
			if (_game != null) {

				_now = DateTime.Now;
				if (_isFirstTime) {
					// Initialize GameTime
					_updateGameTime = new GameTime ();
					_drawGameTime = new GameTime ();
					_lastUpdate = DateTime.Now;
					_isFirstTime = false;
				}

				if (_needsToResetElapsedTime) {
					_drawGameTime.ResetElapsedTime();
					_needsToResetElapsedTime = false;
				}

				// Try to catch up with frames
				_drawGameTime.Update (_now - _lastUpdate);
				TimeSpan catchup = _drawGameTime.ElapsedGameTime;
				if (catchup > _game.TargetElapsedTime) {
					while (catchup > _game.TargetElapsedTime) {
						//Console.WriteLine("Catching up " + (catchup - _game.TargetElapsedTime));
						catchup -= _game.TargetElapsedTime;
						_drawGameTime.ElapsedGameTime = _game.TargetElapsedTime;
                        if (_platform.IsRunning)
    						_game.DoUpdate (_drawGameTime);
						_extraElapsedTime += catchup;
					}
					if (_extraElapsedTime > _game.TargetElapsedTime) {
						//Console.WriteLine("FastForward " + _extraElapsedTime);
                        if (_platform.IsRunning)
    						_game.DoUpdate (_drawGameTime);
						_extraElapsedTime = TimeSpan.Zero;
					}
				}
				else {
                    if (_platform.IsRunning)
    					_game.DoUpdate (_drawGameTime);
				}

				//Console.WriteLine("Render " + _drawGameTime.ElapsedGameTime);
//				_game.DoUpdate(_drawGameTime);
                if (_platform.IsRunning)
    				_game.DoDraw (_drawGameTime);
				_lastUpdate = _now;
			}

		}
//		protected override void OnUpdateFrame (FrameEventArgs e)
//		{
//			base.OnUpdateFrame (e);
//
//			if (_game != null) {
//				Console.WriteLine("Update");
//				if (_isFirstTime) {
//					// Initialize GameTime
//					_updateGameTime = new GameTime ();
//					_drawGameTime = new GameTime ();
//					_lastUpdate = DateTime.Now;
//					_now = DateTime.Now;
//				}
//				else {
//					_now = DateTime.Now;
//					_updateGameTime.Update (_now - _lastUpdate);
//					if (_needsToResetElapsedTime) {
//						_updateGameTime.ResetElapsedTime();
//						_drawGameTime.ResetElapsedTime();
//						_needsToResetElapsedTime = false;
//					}
//				}
//				_isFirstTime = false;
//				_game.DoUpdate (_updateGameTime);
//			}
//		}
		protected override void OnResize (EventArgs e)
		{
            var manager = (GraphicsDeviceManager)_game.Services.GetService(typeof(IGraphicsDeviceManager));
            if (_game.Initialized)
            {
    			manager.OnDeviceResetting(EventArgs.Empty);
    			
    			Microsoft.Xna.Framework.Graphics.Viewport _vp =
    			new Microsoft.Xna.Framework.Graphics.Viewport();
    				
    			_vp.X = (int)Bounds.X;
    			_vp.Y = (int)Bounds.Y;
    			_vp.Width = (int)Bounds.Width;
    			_vp.Height = (int)Bounds.Height;

    			_game.GraphicsDevice.Viewport = _vp;
            }
			
			clientBounds = new Rectangle((int)Bounds.X,(int)Bounds.Y,(int)Bounds.Width,(int)Bounds.Height);
			
			base.OnResize(e);
			OnClientSizeChanged(e);

            if (_game.Initialized)
    			manager.OnDeviceReset(EventArgs.Empty);
		}
		
		protected virtual void OnClientSizeChanged (EventArgs e)
		{
			var h = ClientSizeChanged;
			if (h != null)
				h (this, e);
		}
		
		protected override void OnTitleChanged (EventArgs e)
		{
			base.OnTitleChanged (e);
		}

		protected override void OnUnload (EventArgs e)
		{
			base.OnUnload (e);
		}

		protected override void OnVisibleChanged (EventArgs e)
		{			
			base.OnVisibleChanged (e);	
		}

		protected override void OnWindowStateChanged (EventArgs e)
		{		
			base.OnWindowStateChanged (e);	
		}

		#endregion

		#region UIVIew Methods

		/* TODO private readonly Dictionary<IntPtr, TouchLocation> previousTouches = new Dictionary<IntPtr, TouchLocation>();

		private void FillTouchCollection(NSSet touches)
		{
			UITouch []touchesArray = touches.ToArray<UITouch>();

			TouchPanel.Collection.Clear();
			TouchPanel.Collection.Capacity = touchesArray.Length;

			for (int i=0; i<touchesArray.Length;i++)
			{
				TouchLocationState state;				
				UITouch touch = touchesArray[i];
				switch (touch.Phase)
				{
					case UITouchPhase.Began	:	
						state = TouchLocationState.Pressed;
						break;
					case UITouchPhase.Cancelled	:
					case UITouchPhase.Ended	:
						state = TouchLocationState.Released;
						break;
					default :
						state = TouchLocationState.Moved;
						break;					
				}

				TouchLocation tlocation;
				TouchLocation previousTouch;
				if (state != TouchLocationState.Pressed && previousTouches.TryGetValue (touch.Handle, out previousTouch))
				{
					Vector2 position = new Vector2 (touch.LocationInView (touch.View));
					Vector2 translatedPosition = position;

					switch (CurrentOrientation)
					{
						case DisplayOrientation.Portrait :
						{																		
							break;
						}

						case DisplayOrientation.LandscapeRight :
						{				
							translatedPosition = new Vector2( ClientBounds.Height - position.Y, position.X );							
							break;
						}

						case DisplayOrientation.LandscapeLeft :
						{							
							translatedPosition = new Vector2( position.Y, ClientBounds.Width - position.X );							
							break;
						}

						case DisplayOrientation.PortraitUpsideDown :
						{				
							translatedPosition = new Vector2( ClientBounds.Width - position.X, ClientBounds.Height - position.Y );							
							break;
						}
					}
					tlocation = new TouchLocation(touch.Handle.ToInt32(), state, translatedPosition, 1.0f, previousTouch.State, previousTouch.Position, previousTouch.Pressure);
				}
				else
				{
					Vector2 position = new Vector2 (touch.LocationInView (touch.View));
					Vector2 translatedPosition = position;

					switch (CurrentOrientation)
					{
						case DisplayOrientation.Portrait :
						{																		
							break;
						}

						case DisplayOrientation.LandscapeRight :
						{				
							translatedPosition = new Vector2( ClientBounds.Height - position.Y, position.X );							
							break;
						}

						case DisplayOrientation.LandscapeLeft :
						{							
							translatedPosition = new Vector2( position.Y, ClientBounds.Width - position.X );							
							break;
						}

						case DisplayOrientation.PortraitUpsideDown :
						{				
							translatedPosition = new Vector2( ClientBounds.Width - position.X, ClientBounds.Height - position.Y );							
							break;
						}
					}
					tlocation = new TouchLocation(touch.Handle.ToInt32(), state, translatedPosition, 1.0f);
				}

				TouchPanel.Collection.Add (tlocation);

				if (state != TouchLocationState.Released)
					previousTouches[touch.Handle] = tlocation;
				else
					previousTouches.Remove(touch.Handle);
			}
		}

		public override void TouchesBegan (NSSet touches, UIEvent evt)
		{
			base.TouchesBegan (touches, evt);

			FillTouchCollection(touches);

			GamePad.Instance.TouchesBegan(touches,evt);	
		}

		public override void TouchesEnded (NSSet touches, UIEvent evt)
		{
			base.TouchesEnded (touches, evt);

			FillTouchCollection(touches);	

			GamePad.Instance.TouchesEnded(touches,evt);								
		}

		public override void TouchesMoved (NSSet touches, UIEvent evt)
		{
			base.TouchesMoved (touches, evt);

			FillTouchCollection(touches);

			GamePad.Instance.TouchesMoved(touches,evt);
		}

		public override void TouchesCancelled (NSSet touches, UIEvent evt)
		{
			base.TouchesCancelled (touches, evt);

			FillTouchCollection(touches);

			GamePad.Instance.TouchesCancelled(touches,evt);
		} */

		#endregion

		public string ScreenDeviceName {
			get {
				throw new System.NotImplementedException ();
			}
		}

		public Rectangle ClientBounds {
			get {
				return clientBounds;
			}
		}

		public bool AllowUserResizing {
			get { return (Window.StyleMask & NSWindowStyle.Resizable) == NSWindowStyle.Resizable; }
			set
            {
                if (value)
                    Window.StyleMask |= NSWindowStyle.Resizable;
                else
                    Window.StyleMask &= ~NSWindowStyle.Resizable;
            }
		}	

		private DisplayOrientation _currentOrientation;

		public DisplayOrientation CurrentOrientation { 
			get {
				return _currentOrientation;
			}
			internal set {
				if (value != _currentOrientation) {
					_currentOrientation = value;
					if (OrientationChanged != null) {
						OrientationChanged (this, EventArgs.Empty);
					}
				}
			}
		}

		public event EventHandler<EventArgs> ClientSizeChanged;
		public event EventHandler<EventArgs> OrientationChanged;
		public event EventHandler<EventArgs> ScreenDeviceNameChanged;
		
		// make sure we get mouse move events.
		public override bool AcceptsFirstResponder ()
		{
			return true;
		}

		public override bool BecomeFirstResponder ()
		{
			return true;
		}
		public override void CursorUpdate (NSEvent theEvent)
		{
			base.CursorUpdate (theEvent);
		}

		public override void ViewWillMoveToWindow (NSWindow newWindow)
		{
			//Console.WriteLine("View will move to window");
			if (_trackingArea != null) RemoveTrackingArea(_trackingArea);
			_trackingArea = new NSTrackingArea(Frame,
			                      	NSTrackingAreaOptions.MouseMoved | 
			                        NSTrackingAreaOptions.MouseEnteredAndExited |
			                        NSTrackingAreaOptions.EnabledDuringMouseDrag |
			                        NSTrackingAreaOptions.ActiveWhenFirstResponder |
			                        NSTrackingAreaOptions.InVisibleRect |
				NSTrackingAreaOptions.CursorUpdate,
			                      this, new NSDictionary());
			AddTrackingArea(_trackingArea);

		}

		// These variables are to handle our custom cursor for when IsMouseVisible is false.
		// Hiding and unhiding the cursor was such a pain that I decided to let the system
		// take care of this with Cursor Rectangles
		NSImage cursorImage = null;	// Will be set to our custom image
		NSCursor cursor = null;		// Our custom cursor
		public override void ResetCursorRects ()
		{

			// If we do not have a cursor then we create an image size 1 x 1
			// and then create our custom cursor with clear colors
			if (cursor == null) {
				cursorImage = new NSImage(new SizeF(1,1));
				cursor = new NSCursor(cursorImage, NSColor.Clear, NSColor.Clear, new PointF(0,0));
			}

			// if the cursor is not to be visible then we us our custom cursor.
			if (!_game.IsMouseVisible)
				AddCursorRect(Frame, cursor);
			else
				AddCursorRect(Frame, NSCursor.CurrentSystemCursor);

		}

		public override void DiscardCursorRects ()
		{
			base.DiscardCursorRects ();
			//Console.WriteLine("DiscardCursorRects");
		}
		private void UpdateKeyboardState ()
		{
			_keyStates.Clear ();
			_keyStates.AddRange (_flags);
			_keyStates.AddRange (_keys);
			var kbs = new KeyboardState (_keyStates.ToArray ());
			Keyboard.State = kbs;
		}
		
		// This method should only be called when necessary like when the Guide is displayed
		internal void ClearKeyCacheState() {
			_keys.Clear();	
		}
		
		List<Keys> _keys = new List<Keys> ();
		List<Keys> _keyStates = new List<Keys> ();

		public override void KeyDown (NSEvent theEvent)
		{
			Keys kk = KeyUtil.GetKeys (theEvent); 

			if (!_keys.Contains (kk))
				_keys.Add (kk);

			UpdateKeyboardState ();
		}

		public override void KeyUp (NSEvent theEvent)
		{
			Keys kk = KeyUtil.GetKeys (theEvent); 

			_keys.Remove (kk);

			UpdateKeyboardState ();
		}

		List<Keys> _flags = new List<Keys> ();

		public override void FlagsChanged (NSEvent theEvent)
		{

			_flags.Clear ();
			var modInt = (uint)theEvent.ModifierFlags & 0xFFFF0000;
			var modifier = ((NSEventModifierMask)Enum.ToObject (typeof(NSEventModifierMask), modInt));

			switch (modifier) {
			//case NSEventModifierMask.AlphaShiftKeyMask:
			// return Keys.None;
			case NSEventModifierMask.AlternateKeyMask:
				_flags.Add (Keys.LeftAlt);
				_flags.Add (Keys.RightAlt);
				break;

			case NSEventModifierMask.CommandKeyMask:
				_flags.Add (Keys.LeftWindows);
				_flags.Add (Keys.RightWindows);
				break;
			case NSEventModifierMask.ControlKeyMask:
				_flags.Add (Keys.LeftControl);
				_flags.Add (Keys.RightControl);
				break;
			case NSEventModifierMask.HelpKeyMask:
				_flags.Add (Keys.Help);
				break;
			case NSEventModifierMask.ShiftKeyMask:
				_flags.Add (Keys.RightShift);
				_flags.Add (Keys.LeftShift);
				break;
			}

			UpdateKeyboardState ();
		}

		public override void MouseDown (NSEvent theEvent)
		{
			PointF loc = theEvent.LocationInWindow;
			SetMousePosition (loc);
			switch (theEvent.Type) {
			case NSEventType.LeftMouseDown:
				Mouse.LeftButton = ButtonState.Pressed;
				break;
			}
		}

		public override void MouseUp (NSEvent theEvent)
		{
			PointF loc = theEvent.LocationInWindow;
			SetMousePosition (loc);
			switch (theEvent.Type) {

			case NSEventType.LeftMouseUp:
				Mouse.LeftButton = ButtonState.Released;
				break;
			}
		}
		
		public override void MouseDragged (NSEvent theEvent)
		{
			PointF loc = theEvent.LocationInWindow;
			SetMousePosition (loc);
		}
		
		public override void RightMouseDown (NSEvent theEvent)
		{
			PointF loc = theEvent.LocationInWindow;
			SetMousePosition (loc);
			switch (theEvent.Type) {
			case NSEventType.RightMouseDown:
				Mouse.RightButton = ButtonState.Pressed;
				break;
			}
		}
		
		public override void RightMouseUp (NSEvent theEvent)
		{
			PointF loc = theEvent.LocationInWindow;
			SetMousePosition (loc);
			switch (theEvent.Type) {
			case NSEventType.RightMouseUp:
				Mouse.RightButton = ButtonState.Released;
				break;
			}
		}
		
		public override void RightMouseDragged (NSEvent theEvent)
		{
			PointF loc = theEvent.LocationInWindow;
			SetMousePosition (loc);
		}
		
		public override void OtherMouseDown (NSEvent theEvent)
		{
			PointF loc = theEvent.LocationInWindow;
			SetMousePosition (loc);
			switch (theEvent.Type) {
			case NSEventType.OtherMouseDown:
				Mouse.MiddleButton = ButtonState.Pressed;
				break;
			}
		}
		
		public override void OtherMouseUp (NSEvent theEvent)
		{
			PointF loc = theEvent.LocationInWindow;
			SetMousePosition (loc);
			switch (theEvent.Type) {
			case NSEventType.OtherMouseUp:
				Mouse.MiddleButton = ButtonState.Released;
				break;
			}
		}
		
		public override void OtherMouseDragged (NSEvent theEvent)
		{
			PointF loc = theEvent.LocationInWindow;
			SetMousePosition (loc);
		}
		
		public override void ScrollWheel (NSEvent theEvent)
		{
			PointF loc = theEvent.LocationInWindow;
			SetMousePosition(loc);
			
			switch (theEvent.Type) {
				case NSEventType.ScrollWheel:
					if (theEvent.DeltaY > 0) {
						Mouse.ScrollWheelValue += (theEvent.DeltaY*0.1f+0.09f)*1200;
					} else {
						Mouse.ScrollWheelValue += (theEvent.DeltaY*0.1f-0.09f)*1200;
					}
				break;
			}	
		}

		public override void MouseMoved (NSEvent theEvent)
		{
			PointF loc = theEvent.LocationInWindow;
			SetMousePosition (loc);

			switch (theEvent.Type) {
				case NSEventType.MouseMoved:
				//Mouse.Moved = true;
				break;
			}			
		}

		private void SetMousePosition (PointF location)
		{
			Mouse.SetPosition ((int)location.X, (int)(ClientBounds.Height - location.Y));

		}

	}
}

