/* 
  Copyright (C) 2008 Kai Hertel

	This file is part of mmpong.

	mmpong is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	mmpong is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with mmpong.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "addrtable.h"
#include "hashmap.h"
#include "linklist.h"
#include "sizeof.h"

#define FAST_ADDRTABLE (( __UNSIGNED_SIZE__ == __VOID_PTR_SIZE__) && (__UNSIGNED_SIZE__ == __IN_ADDR_T_SIZE__))

static struct hashmap *addrtable= NULL;
static struct linklist *addrlist= NULL, *addrtail= NULL;
static pthread_mutex_t atabsync;

static unsigned addrtable_map(const unsigned, const void *);
static int addrtable_cmp(const void *, const void *);
#if !FAST_ADDRTABLE
static int prep_addrtail(const struct sockaddr_in);
#endif
static void free_wrapper(void *);



int addrtable_init(max_connect, mutexattr)
const int max_connect;
const pthread_mutexattr_t *mutexattr;
{
	if (addrtable) return (-1);
	// prepare address lookup table for flood protection
	unsigned mapsize= LOOKUP_TABLE_SIZE;
	if (max_connect >0)
		mapsize= max_connect /2;
	addrtable= hashmap_create(mapsize, NULL, NULL);
	if (addrtable) {
		if (pthread_mutex_init(&atabsync, mutexattr)) {
			fprintf(stderr, "Error creating synchronization primitive, dropping flood protection.\n");
			hashmap_destroy(addrtable);
			addrtable= NULL;
		}
	}
	return (addrtable == NULL) * (-1);
}



void addrtable_destroy(hard)
const short hard;
{
	if (pthread_mutex_lock(&atabsync))
		fprintf(stderr, "Error syncing on mutex.\n");
	if (addrtable) {
		hashmap_destroy(addrtable);
		addrtable= NULL;
	}
	if (addrlist) {
		linklist_destroy(addrlist, free_wrapper);
		addrlist= addrtail= NULL;
	}
	if (pthread_mutex_unlock(&atabsync))
		fprintf(stderr, "Error releasing mutex.\n");

	if (hard && pthread_mutex_destroy(&atabsync))
		fprintf(stderr, "Error destroying mutex object.\n");
}



int addrtable_is_active(void) 	// new compilers can `inline` this
{
	return (addrtable != NULL);
}



#if FAST_ADDRTABLE

// fast and elegant
static unsigned addrtable_map(size, key)
const unsigned size;
const void *key;
{
	return ((unsigned)key) %size;
}



static int addrtable_cmp(ikey, skey)
const void *ikey, *skey;
{
	return ((unsigned)ikey) != ((unsigned)skey);
}

#else

// slower, but generic
static unsigned addrtable_map(size, key)
const unsigned size;
const void *key;
{
/*	struct sockaddr_in sample;
	memcpy(&sample.sin_addr.s_addr, key, sizeof(sample.sin_addr.s_addr));
	return ((unsigned) sample.sin_addr.s_addr)%size; */
	return *((unsigned *)key) %size;
}



static int addrtable_cmp(ikey, skey)
const void *ikey, *skey;
{
	struct sockaddr_in sample;
	return memcmp(ikey, skey, sizeof(sample.sin_addr.s_addr));
}



static int prep_addrtail(peer)
const struct sockaddr_in peer;
{
	if (!addrtail) {
		if (!addrlist) addrlist= calloc(1, sizeof(struct linklist));
		addrtail= addrlist;
		if (addrtail) 	// forward to the end
			while (addrtail->next) addrtail= addrtail->next;
	}
	if (!addrtail) return (-1);
	if (!addrtail->data)
		addrtail->data= malloc(sizeof(peer.sin_addr.s_addr));
	if (!addrtail->data) return (-1);
	memcpy(addrtail->data, &peer.sin_addr.s_addr, sizeof(peer.sin_addr.s_addr));
	return 0;
}

#endif



// just to ensure libc independence
static void free_wrapper(data)
void *data;
{
	if (data) free(data);
}



int addrtable_remove(peer)
const struct sockaddr_in peer;
{
	// guard
	if (!addrtable_is_active())
		return (-1);
	if (pthread_mutex_lock(&atabsync)) {
		fprintf(stderr, "Error syncing on mutex.\n");
		return (-2);
	}
	if (!addrtable_is_active()) {
		if (pthread_mutex_unlock(&atabsync))
			fprintf(stderr, "Error releasing mutex.\n");
		return (-1);
	}

	int retval= (-1);
#if FAST_ADDRTABLE
	// fast, but highly dependent on the architecture/protocol used
	retval= hashmap_remove(addrtable, (void *)(peer.sin_addr.s_addr), addrtable_map, addrtable_cmp);
#else
	// independent of arch/proto
	retval= hashmap_remove(addrtable, &peer.sin_addr.s_addr, addrtable_map, addrtable_cmp);
	if (!retval) { 	// key has been added
		struct linklist *this, *prev= NULL;
		for (this= addrlist; this; prev= this, this= this->next) {
			if (!this->data) continue;
			if (!addrtable_cmp(this->data, &peer.sin_addr.s_addr)) {
//				if (prev == this) prev= NULL;
				if (prev) prev->next= this->next;
				else addrlist= prev= this->next;
				if (this->data) free(this->data);
				free(this);
//				this= prev;
				break;
			}
		}
	}
#endif
	if (pthread_mutex_unlock(&atabsync))
		fprintf(stderr, "Error releasing mutex.\n");
	return retval;
}



// return values:
// (>=0) == success (return value is operation-specific)
// ( -1) == table uninitialized
// ( -2) == error acquiring mutex
// ( -3) == error adding list element
int addrtable_atomic(peer, operation, preset)
const struct sockaddr_in peer;
int (*operation)(unsigned *);
unsigned (*preset)(const struct sockaddr_in);
{
	// guard
	if (!addrtable_is_active())
		return (-1);
	if (pthread_mutex_lock(&atabsync)) {
		fprintf(stderr, "Error syncing on mutex.\n");
		return (-2);
	}
	int retval= (-1);
	if (!addrtable_is_active())
		goto leave;

	unsigned *value= NULL;
	retval= 0;
#if FAST_ADDRTABLE
	// fast, but highly dependent on the architecture/protocol used
	unsigned preval= 0;
	value= (unsigned *)hashmap_lookup(addrtable, (void *)(peer.sin_addr.s_addr), addrtable_map, addrtable_cmp);
#else
	// independent of arch/proto
	void *prevalgeneric= NULL;
	if (!prep_addrtail(peer))
		value= (unsigned *)hashmap_lookup(addrtable, addrtail->data, addrtable_map, addrtable_cmp);
#endif 

	if (value == NULL) {
#if FAST_ADDRTABLE
		// fast, but highly dependent on the architecture/protocol used
		if (preset) preval= preset(peer);
		value= &preval;
#else
		// independent of arch/proto
		if (preset) prevalgeneric= ((void *)NULL) + preset(peer);
		value= (unsigned *)&prevalgeneric;
#endif
	}

	int passthru= operation(value);
	if (passthru < 0) passthru= 0;

#if FAST_ADDRTABLE
	if (value == &preval)
		retval= hashmap_insert(addrtable, (void *)(peer.sin_addr.s_addr), (void *)preval, addrtable_map, addrtable_cmp);
#else
	if (value == (unsigned *)&prevalgeneric) {
		retval= (-3);
		if (prep_addrtail(peer)) {
			fprintf(stderr, "Error dealing with a linked list.\n");
			goto leave;
		}
		retval= hashmap_insert(addrtable, addrtail->data, prevalgeneric, addrtable_map, addrtable_cmp);
		if (!retval) { 	// key has been added
			addrtail->next= calloc(1, sizeof(struct linklist));
			addrtail= addrtail->next;
			if (!addrtail) retval= (-3);
		}
	}
#endif
	if (!retval)
		retval= passthru;

leave:
	if (pthread_mutex_unlock(&atabsync))
		fprintf(stderr, "Error releasing mutex.\n");
	return retval;
}

