/*
  Copyright (C) 2008 André Gaul, Jan Friederich, Steffen Basting, Kai Hertel

        This file is part of mmpong.

        mmpong is free software: you can redistribute it and/or modify
        it under the terms of the GNU General Public License as published by
        the Free Software Foundation, either version 3 of the License, or
        (at your option) any later version.

        mmpong is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
        GNU General Public License for more details.

        You should have received a copy of the GNU General Public License
        along with mmpong.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifdef WIN32
# ifndef _WIN32_WINNT
#  define _WIN32_WINNT 0x0501 //assume windows xp or above
# endif
# include <winsock2.h>
# include <ws2tcpip.h>
#else
# include <sys/socket.h>
# include <arpa/inet.h>
# include <netdb.h>
# include <netinet/tcp.h>
# include <netinet/ip.h>
# include <fcntl.h>
#endif

#include <stdexcept>
#include <iostream>
#include <string>
#include <cstring>
#include <cerrno>
#include "netgame.h"

using namespace std;

#ifdef WIN32
# define CLOSE closesocket
#else
# define CLOSE close
#endif

void timeval_combine(const int lfact, struct timeval *lval, const int rfact, const struct timeval *rval)
{
	// combine linearly
	lval->tv_sec= lval->tv_sec * lfact + rval->tv_sec * rfact;
	lval->tv_usec= lval->tv_usec * lfact + rval->tv_usec * rfact;
	// normalize
	if (lval->tv_usec < 0) {
		lval->tv_sec+= lval->tv_usec / (1000L * 1000L) -1;
		lval->tv_usec-= (lval->tv_usec / (1000L * 1000L) -1) * (1000L * 1000L);
	}
	if (lval->tv_usec >= 1000L * 1000L) {
		lval->tv_sec+= lval->tv_usec / (1000L * 1000L);
		lval->tv_usec-= (lval->tv_usec / (1000L * 1000L)) * (1000L * 1000L);
	}
}



NetGame::NetGame(const string &server, const string &port, bool _blocking) {
	blocking = _blocking;
	recvbuf.len = recvbuf.pos = sendbuf.len = sendbuf.pos = 0;
	recvbuf.sz = sendbuf.sz = sizeof(struct netmessage);

	struct addrinfo criteria;
	memset(&criteria, 0, sizeof(criteria));
	criteria.ai_family= AF_INET;
	criteria.ai_socktype= SOCK_STREAM;
	criteria.ai_protocol= IPPROTO_TCP;
	criteria.ai_flags=
# ifndef WIN32
		// only in vista, so we are disabling it for win32 completely
		AI_ADDRCONFIG |
# endif
		AI_CANONNAME; // | AI_IDN | AI_CANONIDN

	struct addrinfo *peers;
	int stat;
	if ((stat= getaddrinfo(server.c_str(), port.c_str(), &criteria, &peers))) 
		throw runtime_error(string("NetGame: could not get host address (") + gai_strerror(stat) + ")");
	
	sock=-1;
	for (struct addrinfo *peer=peers; peer!=NULL; peer=peer->ai_next) {
		cout << "NetGame: Connecting to " << peer->ai_canonname;

#ifdef WIN32
		// what the hell did these guys smoke?? they introduced inet_ntop in vista,
		// BUT it's name is InetNtop. so we are using good old inet_ntoa
		const char *addrstr = inet_ntoa(((struct sockaddr_in *)(peer->ai_addr))->sin_addr);
		if (addrstr)
			cout << " (" << addrstr << ":" << ntohs(((struct sockaddr_in *)(peer->ai_addr))->sin_port) << ")" << endl;
#else
		char addrstr[INET_ADDRSTRLEN];
		if (inet_ntop(peer->ai_addr->sa_family, &((struct sockaddr_in *)(peer->ai_addr))->sin_addr, addrstr, INET_ADDRSTRLEN))
			cout << " (" << addrstr << ":" << ntohs(((struct sockaddr_in *)(peer->ai_addr))->sin_port) << ")" << endl;
#endif

		sock= socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
		if (sock <0 )
			throw runtime_error(string("NetGame: could not create socket (") + strerror(errno) + ")");

		if (connect(sock, peer->ai_addr, peer->ai_addrlen) == -1) {
			cerr << "NetGame: Connect failed (" << strerror(errno)  << ")" << endl;
			CLOSE(sock);
			errno=0;
			sock=-1;
			continue;
		}
		break;
	}

	if (sock < 0)
		throw runtime_error("NetGame: Could not connect to server");

	freeaddrinfo(peers);

# ifdef WIN32
	BOOL yes=TRUE;
	if ( setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, (char*)&yes, sizeof(yes)) == (-1) ||
			setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, (char *)&yes, sizeof(yes)) == (-1) )
		cerr << "NetGame: Socket options have not been set properly" << endl;
# else
	int yes= 1, rcvlow= 2;
	int maxseg= sizeof(struct netmessage);
	uint8_t tos= IPTOS_LOWDELAY;
	if ( setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, &yes, sizeof(yes)) == (-1) ||
			// we expect to receive short integers
			setsockopt(sock, SOL_SOCKET, SO_RCVLOWAT, &rcvlow, sizeof(rcvlow)) == (-1) ||
			setsockopt(sock, IPPROTO_IP, IP_TOS, &tos, sizeof(tos)) == (-1) ||
			// ideally, we don't send or receive more data than a netmessage contains
			setsockopt(sock, IPPROTO_TCP, TCP_MAXSEG, &maxseg, sizeof(maxseg)) == (-1) ||
			setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, &yes, sizeof(yes)) == (-1)
			)
		cerr << "NetGame: Socket options have not been set properly" << endl;
# endif
	cout << "NetGame: Connected." << endl;

	struct netmessage msg;
	int netcode = 0;

	// make sure to retrieve a full game state from the server when joining the game
	if ( (netcode= netmessage_recv(sock, &msg, sizeof(msg), &recvbuf)) == NETMSG_SUCCESS ) {
		switch (msg.hdr.id) {

			case NETMSG_STAT:
				gameplay_apply_state(&msg, &game, &team);
				break;

			case NETMSG_KICK:
				cout << "NetGame: Got kicked! Do you see what happens, Larry: " << msg.payload.data << endl;
				throw runtime_error(string("NetGame: Got kicked! Do you see what happens, Larry: ") + msg.payload.data);
				
				break;

			default:
				cerr << "NetGame: Received unknown message, id == " << msg.hdr.id << " ('" << (char)msg.hdr.id << "')" << endl;
				break;
		}
	} else 
		throw runtime_error("NetGame: Cannot determine initial game state");

# ifdef DEBUG
	cout << 
		"[Packet | Initial] Ball pos == ( " << game.ball.pos[0] << ", " << game.ball.pos[1] << 
		" ), dir == (	" << game.ball.dir[0] << ", " << game.ball.dir[1] << 
		" ), paddles == ( " << game.pad[0].mean << ", " <<  game.pad[1].mean << 
		", team == " <<	team << endl;
# endif

	if (!blocking) {
# ifdef WIN32
		unsigned long nonblk=1;
		if (ioctlsocket(sock, FIONBIO, &nonblk))
			cerr << "NetGame: Socket I/O is blocking" << endl;
# else
		if (fcntl(sock, F_SETFL, O_NONBLOCK) == (-1))
			cerr << "NetGame: Socket I/O is blocking" << endl;
# endif
	}
}


NetGame::~NetGame() {
	CLOSE(sock);
}


void NetGame::update(const struct timeval *time_wait) {
	struct netmessage msg;
	int netcode = NETMSG_SUCCESS;

	bool recv=true;

	fd_set listen;
	FD_ZERO(&listen);
#ifdef WIN32
	FD_SET((SOCKET)sock, &listen);
#else
	FD_SET(sock, &listen);
#endif
	struct timeval time, time_start, time_diff;
	gettimeofday(&time_start, NULL);
	memcpy(&time, time_wait, sizeof(struct timeval));
	
	//handle incoming messages
	while ( recv && (blocking || (select(sock +1, &listen, NULL, NULL, &time) > 0) ) ) {
		gettimeofday(&time_diff, NULL);
		timeval_combine(1, &time_diff, -1, &time_start);
		gettimeofday(&time_start, NULL);
		// check for time 'overflow'
		if (time.tv_sec < time_diff.tv_sec || 
				(time.tv_sec == time_diff.tv_sec && 
				 time.tv_usec < time_diff.tv_usec) )
			// no more time left :( 
		{
			time.tv_sec = 0;
			time.tv_usec = 0;
		}
		else
			timeval_combine(1, &time, -1, &time_diff);

		netcode = netmessage_recv(sock, &msg, sizeof(msg), &recvbuf);
		if (netcode != NETMSG_SUCCESS)
			continue;
		switch (msg.hdr.id) {

			case NETMSG_STAT:
			case NETMSG_UPDT:
				gameplay_apply_state(&msg, &game, &team);
				break;

			case NETMSG_KICK:
				cout << "NetGame: Got kicked! Do you see what happens, Larry: " << msg.payload.data << endl;
				throw runtime_error(string("NetGame: Got kicked! Do you see what happens, Larry: ") + msg.payload.data);
				break;

			default:
				cerr <<  "NetGame: Received unknown message, id == " << msg.hdr.id << "('" << (char)msg.hdr.id << "')" << endl;
				break;
		}
		if (blocking)
			recv=false;
	}

	if ((netcode != NETMSG_SUCCESS) && (netcode != NETMSG_FAIL_DELIVER) && (netcode != NETMSG_PARTIAL)) {
		cerr << "NetGame: " << "Socket failure while reading (" << netcode << ")" << endl;
		throw runtime_error("NetGame: Socket failure while reading");
	}
}


void NetGame::sendpos(float pos) {
	uint16_t sendpos = (uint16_t)(pos * PONG_RANGE_SPREAD);
	int netcode = netmessage_send(sock, NETMSG_POS, &sendpos, sizeof(sendpos), &sendbuf);
	if ( (netcode!= NETMSG_SUCCESS) && (netcode != NETMSG_FAIL_DELIVER) && (netcode != NETMSG_PARTIAL)) {
		cerr << "NetGame: " << "Socket failure while sending (" << netcode << ")" << endl;
		throw runtime_error("NetGame: Socket failure while sending");
	}
}


void NetGame::sendpos_flush() {
	int netcode = netmessage_buffer_flush(sock, &sendbuf);
	if ( (netcode!= NETMSG_SUCCESS) && (netcode != NETMSG_FAIL_DELIVER) && (netcode != NETMSG_PARTIAL)) {
		cerr << "NetGame: " << "Socket failure while sending (" << netcode << ")" << endl;
		throw runtime_error("NetGame: Socket failure while sending");
	}
}
