/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "MvQFeatureIconItem.h"

#include <QApplication>
#include <QDebug>
#include <QGraphicsSceneMouseEvent>
#include <QImage>
#include <QMouseEvent>
#include <QPainter>
#include <QStyle>

#include "ObjectList.h"
#include "MvQMethods.h"
#include "MvQPlotView.h"
#include "MvQFeatureFactory.h"
#include "MvQFeatureType.h"
#include "MvQPalette.h"
#include "MgQLayoutItem.h"

#include <assert.h>

//#define MVQFEATUREICON_DEBUG_


//========================================
//
// MvQFeatureIconItem
//
//========================================

MvQFeatureIconItem::MvQFeatureIconItem(MvQFeatureType* feature, MvQPlotView* view, QGraphicsItem* parent) :
    MvQFeatureItem(feature, view, parent)
{
    Q_ASSERT(feature_);
    keepAspectRatio_ = true;

    // The pixmap is always twice as large as the itemrect so that we can allow for the
    // scaling in the plot window (it can go up to 200% at the moment)
//    int pixSize = 24;
//    int factor = 2;
//    pix_ = feature_->pixmap(pixSize*factor, pixSize*factor);
//    itemRect_ = QRectF(-pix_.width() / (2*factor), -pix_.height() / (2*factor), pix_.width()/factor, pix_.height()/factor);
//    bRect_ = itemRect_;
}

MvQFeatureIconItem::MvQFeatureIconItem(const MvQFeatureIconItem& o) :
    MvQFeatureItem(o)
{
    pix_ = o.pix_;
}

MvQFeatureItem* MvQFeatureIconItem::clone() const
{
    return new MvQFeatureIconItem(*this);
}

void MvQFeatureIconItem::paint(QPainter* painter, const QStyleOptionGraphicsItem* /*o*/, QWidget* /*w*/)
{
    painter->setRenderHint(QPainter::SmoothPixmapTransform,true);
    painter->drawPixmap(itemRect_, pix_, pix_.rect());
#ifdef MVQ_FEATURE_DEV_MODE_
    if (outOfView_) {
        auto pen = QPen(devModeColor_);
        painter->setBrush(Qt::NoBrush);
        painter->drawRect(itemRect_.adjusted(1,1,-1,-1));
    }
#endif
}

void MvQFeatureIconItem::makePixmap()
{
    if (!shapeCol_.isValid()) {
        pix_ = feature_->pixmap(2*itemRect_.width(), 2*itemRect_.height());
    } else {
        pix_ = feature_->pixmap(2*itemRect_.width(), 2*itemRect_.height(), shapeCol_);
    }
}

void MvQFeatureIconItem::resize(QRectF targetRectInParent)
{    
#ifdef MVQFEATUREICON_DEBUG_
    qDebug() << "MvQFeatureIconItem::resize";
    qDebug() << " pos:" << pos() << "itemRect:" << itemRect_ << "targetRectInParent:" << targetRectInParent;
#endif
    prepareGeometryChange();

    // the input rect is in parent coordinates. It defines the desired
    // boundingRect size
    QRectF targetRect = mapFromParent(targetRectInParent).boundingRect();

    // reduce the targetRect with the halo. We want to fit the itemRect into it!!!
    targetRect.adjust(halo(), halo(), -halo(), -halo());

    auto w = targetRect.width();
    auto h = targetRect.height();
    itemRect_ = QRectF(-w/2., -h/2., w, h);
#ifdef MVQFEATUREICON_DEBUG_
    qDebug() << " -> itemRect:" << itemRect_ << "targetRect:" << targetRect;
#endif
    bRect_ = itemRect_;
    setPos(targetRectInParent.center());
    makePixmap();
    adjustBRect();
    adjustGeoCoord();
    adjustSizeInReq();
}

void MvQFeatureIconItem::resizeTo(const MvQFeatureGeometry& geom)
{
    prepareGeometryChange();
    setPos(geom.pos_);
    itemRect_ = geom.itemRect_;
    bRect_ = itemRect_;
    makePixmap();
    adjustBRect();
    adjustGeoCoord();
    adjustSizeInReq();
}

void MvQFeatureIconItem::adjustSizeInReq()
{
    req_("SIZE") = itemRect_.width();
    broadcastRequestChanged();
}

void MvQFeatureIconItem::updateRect(double width, double height)
{
    prepareGeometryChange();
    itemRect_ = QRectF(-width/2.,-height/2., width, height);
    bRect_ = itemRect_;
}

void MvQFeatureIconItem::adjustBRect()
{
   if (fabs(halo() > 1E-5)) {
       prepareGeometryChange();
       bRect_ = itemRect_.adjusted(-halo(), -halo(), halo(), halo());
   }
}

double MvQFeatureIconItem::halo() const
{
    auto w = boxPen_.width();
    return (w <=1)?0:1+w/2;
}

//========================================
//
// MvQFeatureStandardIconItem
//
//========================================

MvQFeatureStandardIconItem::MvQFeatureStandardIconItem(MvQFeatureType* feature, MvQPlotView* view, QGraphicsItem* parent) :
    MvQFeatureIconItem(feature, view, parent)
{
    if (feature_->name() == "high") {
        req_ = MvRequest("FEATUREHIGH");
    } else if (feature_->name() == "low") {
        req_ = MvRequest("FEATURELOW");
    } else {
        req_ = MvRequest("FEATUREICON");
    }
    initStyle();
}

MvQFeatureStandardIconItem::MvQFeatureStandardIconItem(const MvQFeatureStandardIconItem& o) :
    MvQFeatureIconItem(o)
{
    initStyle();
}

MvQFeatureItem* MvQFeatureStandardIconItem::clone() const
{
    return new MvQFeatureStandardIconItem(*this);
}

bool MvQFeatureStandardIconItem::getRequestParameters()
{
    if (!req_)
        return false;

    req_ = ObjectList::ExpandRequest(req_, EXPAND_DEFAULTS);

    std::string val;
    shapeCol_ = QColor();
    if (req_.getValue("COLOUR", val, true) && !val.empty()) {
        shapeCol_ = MvQPalette::magics(val);
    }

    // size
    req_.getValue("SIZE", val);
    int iv = 0;
    try {
        iv = std::stoi(val);
    } catch (...) {
        iv = 0;
    }
    if (iv > 0 && iv != static_cast<int>(itemRect_.width())) {
        updateRect(iv, iv);
    }

    return true;
}

//========================================
//
// MvQFeatureWmoSymbolItem
//
//========================================

MvQFeatureWmoSymbolItem::MvQFeatureWmoSymbolItem(MvQFeatureType* feature, MvQPlotView* view, QGraphicsItem* parent) :
    MvQFeatureIconItem(feature, view, parent)
{
    req_ = MvRequest("FEATUREWMOSYMBOL");
    initStyle();
}

MvQFeatureWmoSymbolItem::MvQFeatureWmoSymbolItem(const MvQFeatureWmoSymbolItem& o) :
    MvQFeatureIconItem(o)
{
    initStyle();
}

MvQFeatureItem* MvQFeatureWmoSymbolItem::clone() const
{
    return new MvQFeatureWmoSymbolItem(*this);
}

void MvQFeatureWmoSymbolItem::paint(QPainter* painter, const QStyleOptionGraphicsItem* /*o*/, QWidget* /*w*/)
{
    if (boxBrush_.style() != Qt::NoBrush) {
        painter->fillRect(itemRect_, boxBrush_);
    }
    painter->setRenderHint(QPainter::SmoothPixmapTransform,true);
    painter->drawPixmap(itemRect_, pix_, pix_.rect());

    if (boxPen_.style() != Qt::NoPen) {
        painter->setPen(boxPen_);
        painter->drawRect(itemRect_);
    }

#ifdef MVQ_FEATURE_DEV_MODE_
    if (outOfView_) {
        auto pen = boxPen_;
        if (boxPen_.style() == Qt::NoPen) {
            pen = QPen(devModeColor_);
        } else {
            pen.setColor(devModeColor_);
        }
        painter->setBrush(Qt::NoBrush);
        painter->drawRect(itemRect_);
    }
#endif
}

bool MvQFeatureWmoSymbolItem::getRequestParameters()
{
    if (!req_)
        return false;

    req_ = ObjectList::ExpandRequest(req_, EXPAND_DEFAULTS);

    std::string val;
    shapeCol_ = QColor();
    if (req_.getValue("SHAPE_COLOUR", val, true) && !val.empty()) {
        shapeCol_ = MvQPalette::magics(val);
    }

    // Set line parameters
    req_.getValue("LINE", val);
    if (val == "OFF") {
        boxPen_ = QPen(Qt::NoPen);
    } else {
        boxPen_ = MvQ::makePen((const char*)req_("LINE_STYLE"),
                              (int)req_("LINE_THICKNESS"),
                              (const char*)req_("LINE_COLOUR"),
                              Qt::SquareCap, Qt::MiterJoin);
    }

    // Set fill parameters
    req_.getValue("FILL", val);
    if (val == "OFF") {
        boxBrush_ = QBrush(Qt::NoBrush);
    } else {
        boxBrush_ = MvQ::makeBrush("SOLID",
                                (const char*)req_("FILL_COLOUR"));
    }


    // size
    req_.getValue("SIZE", val);
    int iv = 0;
    try {
        iv = std::stoi(val);
    } catch (...) {
        iv = 0;
    }
    if (iv > 0 && iv != static_cast<int>(itemRect_.width())) {
        updateRect(iv, iv);
    }

    return true;
}

static MvQFeatureMaker<MvQFeatureStandardIconItem> icMaker("icon");
static MvQFeatureMaker<MvQFeatureWmoSymbolItem> wmoMaker("wmo");
