/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include <X11/IntrinsicP.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/CoreP.h>
#include <X11/cursorfont.h>
#include "Symbols.h"
#include "SymbolsP.h"



#ifndef ABS
#define ABS(a)           ((a)>=0?(a):-(a))
#endif
#ifndef MIN
#define MIN(a,b)         ((a)>(b)?(b):(a))
#endif

static void    create_gcs();
static void    calc_new_size();

/*
  Widget class methods
*/

static void    Initialize();
static void    Redisplay();
static void    Resize();
static void    Destroy();
static void    Activate();
static Boolean SetValues();

static char defaultTranslations[] = "\
<BtnDown>:activate()\n\
~s ~m ~a <Key>Return:PrimitiveParentActivate()\n\
<Key>osfActivate:PrimitiveParentActivate()\n\
<Key>osfCancel:PrimitiveParentCancel()\n\
<Key>osfSelect:activate()\n\
<Key>osfHelp:Help()\n\
~s ~m ~a <Key>space:activate()\n\
<Key>osfHelp:Help()";


static XtActionsRec actionsList[] = {
	{ "activate", (XtActionProc)Activate, },
};


static XtResource resources[] = {
	{XmNactivateCallback,XtCCallback,XtRCallback,sizeof(caddr_t),
	XtOffset (SymbolsWidget, symbols.activate),XtRCallback,(XtPointer)NULL},
	{ XmNfontList, XmCFontList, XmRFontList, sizeof (XmFontList),
	XtOffset (SymbolsWidget, symbols.fontlist), XmRString, "fixed"},
	{ XmNlabelString, XmCString, XmRString, sizeof (String),
	XtOffset (SymbolsWidget, symbols.no_colours), XmRString, "None"},
};

/*---------------------------------------------------------------*/
/* Static initialisation of the class record                     */
/*---------------------------------------------------------------*/

SymbolsClassRec  symbolsClassRec = {
	{
	(WidgetClass) &xmPrimitiveClassRec,  /* superclass            */
	"Symbols",                            /* class_name            */
	sizeof(SymbolsRec),                   /* widget_size           */
	NULL,                                /* class_initialize      */
	NULL,                                /* class_part_initialize */
	FALSE,                               /* class_inited          */
	Initialize,                          /* initialize            */
	NULL,                                /* initialize_hook       */
	XtInheritRealize,                    /* realize               */
	actionsList,                         /* actions               */
	XtNumber(actionsList),               /* num_actions           */
	resources,                           /* resources             */
	XtNumber(resources),                 /* num_resources         */
	NULLQUARK,                           /* xrm_class             */
	TRUE,                                /* compress_motion       */
	TRUE,                                /* compress_exposure     */
	TRUE,                                /* compress_enterleave   */
	TRUE,                                /* visible_interest      */
	Destroy,                             /* destroy               */
	Resize,                              /* resize                */
	Redisplay,                           /* expose                */
	SetValues,                           /* set_values            */
	NULL,                                /* set_values_hook       */
	XtInheritSetValuesAlmost,            /* set_values_almost     */
	NULL,                                /* get_values_hook       */
	NULL,                                /* accept_focus          */
	XtVersion,                           /* version               */
	NULL,                                /* callback private      */
	defaultTranslations,                 /* tm_table              */
	NULL,                                /* query_geometry        */
	NULL,                                /* display_accelerator   */
	NULL,                                /* extension             */
	},
	{
	(XtWidgetProc)_XtInherit,             /* border_highlight      */
	(XtWidgetProc)_XtInherit,             /* border_unhighligh     */
	XtInheritTranslations,                /* translations          */
	(XtActionProc)_XtInherit,             /* arm_and_activate      */
	NULL,                                 /* syn_resources         */
	0,                                    /* num_syn_resources     */
	NULL,                                 /* extension             */
	},
	{
	0,                                    /* ignore                */
	},
};

WidgetClass symbolsWidgetClass = (WidgetClass) &symbolsClassRec;

/*---------------------------------------------------------------*/
/* Create the two GCs needed                                     */
/*---------------------------------------------------------------*/


/*--------------------------------------------------------------*/
/* Initialize: Create the GCs                                   */
/*--------------------------------------------------------------*/

static void Initialize (SymbolsWidget request, SymbolsWidget new)
{
	XGCValues values;
	XtGCMask  valueMask;

	Display *dpy   = XtDisplay(new);
	int     screen = DefaultScreen(dpy);
	Window  root   = RootWindow(dpy,screen);

	/* Check the size of the widget */

	if (request->core.width == 0)
		new->core.width = 5;

	if (request->core.height == 0)
		new->core.height = 5;
	/* Create the GCs */

	new->symbols.max     = 10;
	new->symbols.count   = 0;
	new->symbols.pixmaps = NULL;
	new->symbols.widths  = NULL;
	new->symbols.heights = NULL;
	new->symbols.width   = NULL;
	new->symbols.height  = NULL;
	new->symbols.pixmaps = NULL;

	valueMask = GCBackground;
	values.background   = new->core.background_pixel;
	new->symbols.fontGC = XtGetGC((Widget)new,valueMask,&values);

}


/*--------------------------------------------------------------*/
/* Destroy the widget: release all memory alocated              */
/*--------------------------------------------------------------*/

static void Destroy (SymbolsWidget w)
{
	XtReleaseGC((Widget)w,w->symbols.fontGC);
	XtFree((XtPointer)w->symbols.pixmaps);
	XtFree((XtPointer)w->symbols.widths);
	XtFree((XtPointer)w->symbols.heights);
}


static void Resize (SymbolsWidget w)
{
}

static void draw(SymbolsWidget w)
{
	int i;
	int frame  = w->primitive.highlight_thickness 
	+ w->primitive.shadow_thickness;
	int height = w->core.height - 2*frame;
	int width  = w->core.width  - 2*frame ;
	int x      = frame;
	int y      = frame;


	if(w->symbols.count)
	{
		width /= w->symbols.count;
		x++;

		if(w->symbols.width < width)
			width = w->symbols.width;

		if(w->symbols.height < height)
			y += (height - w->symbols.height) / 2;

		width++;

		for(i=0;i<w->symbols.count;i++)
		{
			if(w->symbols.pixmaps[i])
			{
			
				XCopyArea(XtDisplay (w), 
					w->symbols.pixmaps[i],
					XtWindow(w),
					w->symbols.fontGC,
					0,0,width,height,x,y);
				x += width;
			}

		}
	}
	else {
		XmString s = XmStringCreateSimple(w->symbols.no_colours);
		XRectangle r;

		r.x      = x;
		r.y      = y;
		r.width  = width;
		r.height = height;

		XClearArea(XtDisplay(w),XtWindow(w), x,y,width,height,False);
		y += (height - XmStringHeight(w->symbols.fontlist,s))/2;

		XmStringDraw (XtDisplay(w), XtWindow(w),
		    w->symbols.fontlist, s,
		    w->symbols.fontGC,
		    x,y,width,
		    XmALIGNMENT_CENTER,XmSTRING_DIRECTION_L_TO_R,&r);
		XmStringFree(s);
	}
}

static void Redisplay(SymbolsWidget w, XEvent *event, Region region)
{
	draw(w);

	_XmDrawShadows(XtDisplay(w), XtWindow(w),
	    w->primitive.bottom_shadow_GC,
	    w->primitive.top_shadow_GC,
	    (int) w->primitive.highlight_thickness,
	    (int) w->primitive.highlight_thickness,
	    (int) (w->core.width - 
	    (2*w->primitive.highlight_thickness)),
	    (int) (w->core.height - 
	    (2*w->primitive.highlight_thickness)),
	    (int) w->primitive.shadow_thickness,
	    XmSHADOW_OUT);
}

/*------------------------------------------------------------------*/
/* SetValues : redraw only for font or color changes                */
/*------------------------------------------------------------------*/

static Boolean SetValues (SymbolsWidget current, SymbolsWidget request, 
SymbolsWidget new)
{
	Boolean    redraw = FALSE;
	redraw = True;
	return (redraw);
}


static void Activate(Widget widget,XEvent *event,char **args,int nargs)
{
	XmAnyCallbackStruct cb;
	cb.reason = XmCR_ACTIVATE;
	cb.event  = event;
	XtCallCallbacks (widget,XmNactivateCallback,(XtPointer)&cb);
}

/*-----------------------------------------------------------------------*/
/* Create a new SymbolsWidget                                              */
/*-----------------------------------------------------------------------*/

Widget CreateSymbolsWidget(Widget parent,char *name,Arg *al,int ac)
{
	return XtCreateWidget(name,symbolsWidgetClass,parent,al,ac);
}

void SymbolsSetPixmaps(Widget w,int c,Pixmap *pix)
{
	SymbolsWidget cw = (SymbolsWidget)w;
	int i;

	if(c == cw->symbols.count)
	{
		for(i=0;i<c;i++)
			if(cw->symbols.pixmaps[i] != pix[i])
				break;

		if(i == c) return;
	}

	XtFree((XtPointer)cw->symbols.pixmaps);
	XtFree((XtPointer)cw->symbols.widths);
	XtFree((XtPointer)cw->symbols.heights);
	cw->symbols.width = cw->symbols.height = 0;

	cw->symbols.pixmaps = (Pixmap*)XtCalloc(c,sizeof(Pixmap));
	cw->symbols.widths  = (unsigned int*)XtCalloc(c,sizeof(unsigned int));
	cw->symbols.heights = (unsigned int*)XtCalloc(c,sizeof(unsigned int));

	cw->symbols.count = c;
	for(i=0;i<c;i++)
		if(pix[i] != None && pix[i] != XmUNSPECIFIED_PIXMAP)
		{

			Display *dpy   = XtDisplay(w);
			int     screen = DefaultScreen(dpy);
			Window  root   = RootWindow(dpy,screen);
			int         x,y;
			unsigned int width;
			unsigned int height;
			unsigned int dummy;
			unsigned int depth;

			XGetGeometry(dpy,pix[i],&root,&x,&y,&width,&height,&dummy,&depth);
			cw->symbols.widths[i] = width;
			cw->symbols.heights[i] = height;

			if(height > cw->symbols.height)
				cw->symbols.height = height;

			if(width > cw->symbols.width)
				cw->symbols.width = width;
			cw->symbols.pixmaps[i] = pix[i];

		}

	if(XtIsRealized(w) && XtIsManaged(w))
		XClearArea(XtDisplay(w),XtWindow(w),0,0,0,0,True);
}
