/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2008  Michael Bell <michael.bell@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include "data_sync_common.h"
#include "libsyncml/syncml_internals.h"
#include "libsyncml/sml_error_internals.h"

#include "data_sync_devinf.h"

SmlBool smlDataSyncIsTimestamp(const char *anchor, SmlBool timestampDefault)
{
	smlTrace(TRACE_ENTRY, "%s(%s, %d)", __func__, VA_STRING(anchor), timestampDefault);
	if (anchor == NULL || strlen(anchor) < 1)
	{
		/* there is no anchor, so the default is sent */
		smlTrace(TRACE_EXIT, "%s - default", __func__);
		return timestampDefault;
	} else {
		/* last is set in the anchor database */
		if (strstr(anchor, "Z")) {
			/* anchor is a timestamp */
			smlTrace(TRACE_EXIT, "%s - TRUE", __func__);
			return TRUE;
		} else {
			/* last is a number */
			smlTrace(TRACE_EXIT, "%s - FALSE", __func__);
			return FALSE;
		}
	}
}

char *smlDataSyncGetNextAnchor(
		SmlDataSyncDatastore *datastore,
		const char *last,
		SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%s)", __func__, VA_STRING(last));
	CHECK_ERROR_REF
	SmlBool use_timestamp = TRUE;
	char *next = NULL;

	use_timestamp = smlDataSyncIsTimestamp(last,
				datastore->dsObject->useTimestampAnchor);
	smlTrace(TRACE_INTERNAL, "%s: use timestamp is %d", __func__, use_timestamp);

	if (use_timestamp)
	{
		/* It is necessary to be sure that the remote device supports UTC. */
		smlTrace(TRACE_INTERNAL,
			"%s: session %p, localtime %d, remoteDevInf %p",
			__func__,
			datastore->dsObject->session,
			datastore->dsObject->onlyLocaltime,
			datastore->dsObject->remoteDevInf);
		if (datastore->dsObject->session &&
		    !datastore->dsObject->onlyLocaltime &&
		    !datastore->dsObject->remoteDevInf &&
		    !smlDataSyncManageDevInf(datastore->dsObject, FALSE, error))
		{
			goto error;
		}
		next = smlTryMalloc0(sizeof(char)*17, error);
		if (!next)
			goto error;
		time_t htime = time(NULL);
		if (datastore->dsObject->onlyLocaltime) {
			smlTrace(TRACE_INTERNAL, "%s: use localtime", __func__);
			strftime(next, 17, "%Y%m%dT%H%M%SZ", localtime(&htime));
		} else {
			smlTrace(TRACE_INTERNAL, "%s: use UTC", __func__);
			strftime(next, 17, "%Y%m%dT%H%M%SZ", gmtime(&htime));
		}
	} else {
		if (last == NULL)
		{
			next = g_strdup("1");
		} else {
			unsigned long count = strtoul(last, NULL, 10);
			count++;
			next = g_strdup_printf("%lu", count);
		}
	}
	smlTrace(TRACE_EXIT, "%s(%s)", __func__, VA_STRING(next));
	return next;
error:
	smlTrace(TRACE_EXIT_ERROR, "%s - %s", __func__, smlErrorPrint(error));
	return NULL;
}
