/* pathEncoding.t.cc
 */
#include "osl/pathEncoding.h"
#include "osl/stl/hash_set.h"
#include "osl/oslConfig.h"
#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <iostream>
using namespace osl;

class PathEncodingTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(PathEncodingTest);
  CPPUNIT_TEST(testPass);
  CPPUNIT_TEST(testDag);
  CPPUNIT_TEST(testPop);
  CPPUNIT_TEST(testUniq);
  CPPUNIT_TEST(testTurn);
  CPPUNIT_TEST_SUITE_END();
public:
  void testPass();
  void testUniq();
  void testDag();
  void testPop();
  void testTurn();
};

CPPUNIT_TEST_SUITE_REGISTRATION(PathEncodingTest);

void PathEncodingTest::testPass()
{
  PathEncoding b(BLACK);
  PathEncoding b2 = b;
  b2.pushMove(Move::PASS(BLACK));
  CPPUNIT_ASSERT(b != b2);

  PathEncoding w(WHITE);
  PathEncoding w2 = w;
  w2.pushMove(Move::PASS(WHITE));
  CPPUNIT_ASSERT(w != w2);
}

void PathEncodingTest::testTurn()
{
  PathEncoding b(BLACK);
  CPPUNIT_ASSERT_EQUAL(BLACK, b.turn());

  PathEncoding w(WHITE);
  CPPUNIT_ASSERT_EQUAL(WHITE, w.turn());
}

void PathEncodingTest::testDag()
{
  const Move m76fu = Move(Square(7,7), Square(7,6), PAWN, 
			     PTYPE_EMPTY, false, BLACK);
  const Move m26fu = Move(Square(2,7), Square(2,6), PAWN, 
			     PTYPE_EMPTY, false, BLACK);
  const Move m34fu = Move(Square(3,3), Square(3,4), PAWN, 
			     PTYPE_EMPTY, false, WHITE);
  PathEncoding path1, path2;
  CPPUNIT_ASSERT_EQUAL(BLACK, path1.turn());

  PathEncoding path76fu(path1, m76fu);
  CPPUNIT_ASSERT(Path_Encoding_Table.get(0, m76fu));
  CPPUNIT_ASSERT(path1 != path76fu);
  path1.pushMove(m76fu);
  CPPUNIT_ASSERT_EQUAL(WHITE, path1.turn());
  CPPUNIT_ASSERT_EQUAL(WHITE, path76fu.turn());
  CPPUNIT_ASSERT_EQUAL(path1, path76fu);

  CPPUNIT_ASSERT(Path_Encoding_Table.get(1, m34fu));
  path1.pushMove(m34fu);
  CPPUNIT_ASSERT_EQUAL(BLACK, path1.turn());

  CPPUNIT_ASSERT(Path_Encoding_Table.get(2, m26fu));
  path1.pushMove(m26fu);
  CPPUNIT_ASSERT_EQUAL(WHITE, path1.turn());
  CPPUNIT_ASSERT_EQUAL(3, path1.getDepth());
  
  path2.pushMove(m26fu);
  CPPUNIT_ASSERT(path1 != path2);
  path2.pushMove(m34fu);
  CPPUNIT_ASSERT(path1 != path2);
  path2.pushMove(m76fu);
  CPPUNIT_ASSERT_EQUAL(WHITE, path1.turn());
  CPPUNIT_ASSERT(path1 != path2);
  CPPUNIT_ASSERT_EQUAL(path1.getDepth(), path2.getDepth());
}

void PathEncodingTest::testPop()
{
  const Move m76fu = Move(Square(7,7), Square(7,6), PAWN, 
			     PTYPE_EMPTY, false, BLACK);
  const Move m34fu = Move(Square(3,3), Square(3,4), PAWN, 
			     PTYPE_EMPTY, false, WHITE);
  PathEncoding path1, path2;
  CPPUNIT_ASSERT(path1 == path2);
  CPPUNIT_ASSERT_EQUAL(BLACK, path1.turn());

  path1.pushMove(m76fu);
  CPPUNIT_ASSERT_EQUAL(WHITE, path1.turn());
  CPPUNIT_ASSERT_EQUAL(1, path1.getDepth());
  CPPUNIT_ASSERT(path1 != path2);

  path1.popMove(m76fu);
  CPPUNIT_ASSERT_EQUAL(BLACK, path1.turn());
  CPPUNIT_ASSERT(path1 == path2);
  CPPUNIT_ASSERT_EQUAL(0, path1.getDepth());
  
  path1.pushMove(m76fu);
  CPPUNIT_ASSERT_EQUAL(WHITE, path1.turn());

  path2.pushMove(m76fu);
  CPPUNIT_ASSERT_EQUAL(WHITE, path2.turn());
  CPPUNIT_ASSERT(path1 == path2);

  path1.pushMove(m34fu);
  CPPUNIT_ASSERT(path1 != path2);
  CPPUNIT_ASSERT_EQUAL(WHITE, path2.turn());

  path1.popMove(m34fu);
  CPPUNIT_ASSERT(path1 == path2);
  CPPUNIT_ASSERT_EQUAL(WHITE, path2.turn());
}

void PathEncodingTest::testUniq()
{
  typedef hash_set<unsigned int> set_t;
  set_t highs, lows;
  size_t deja_high = 0, deja_low = 0;
  for (size_t i=0; i<PathEncodingTable::MaxEncodingLength; ++i)
  {
    if (OslConfig::inUnitTest() < 2 && (i % 7))
      continue;
    if (i % 16==0)
      std::cerr << '.';
    for (size_t j=0; j<Square::SIZE; ++j)
    {
      for (int k=0; k<PTYPE_SIZE; ++k)
      {
	const unsigned long long value
	  = Path_Encoding_Table.get(i, Square::nth(j), (Ptype)(k+PTYPE_MIN));
	const unsigned int h = value >> 32;
	const unsigned int l = (unsigned int)value;
	CPPUNIT_ASSERT(l);
	CPPUNIT_ASSERT(h);
	CPPUNIT_ASSERT((l % 2) == 0);
	const bool uh = highs.insert(h).second;
	const bool ul = lows.insert(l).second;
	if (! uh)
	{
	  ++deja_high;
	}
	if (! ul)
	{
	  ++deja_low;
	}
	CPPUNIT_ASSERT(uh || ul);
      }
    }
  }
  if (OslConfig::verbose())
    std::cerr << " dl " << deja_low << " dh " << deja_high << "\n";
  CPPUNIT_ASSERT(deja_low + deja_high< 387); // 374
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
