/* pool_allocator.h
 */
#ifndef OSL_POOL_ALLOCATOR_H
#define OSL_POOL_ALLOCATOR_H

#ifdef USE_TBB_SCALABLE_ALLOCATOR
#  include <tbb/scalable_allocator.h>
#endif
#ifdef USE_BOOST_POOL_ALLOCATOR
#  include "osl/misc/lightMutex.h"
#  include <boost/pool/pool_alloc.hpp>
#  include <boost/mpl/if.hpp>
#else
#  include <memory>
#endif

namespace osl
{
  /**
   * stl コンテナのallocatorを取り替えたものを提供する.
   *
   * 選択肢
   * - std::allocator を使って tcmalloc と組合せる (default)
   * - intel thread building blocks の scalable_allocator を使う
   * - boost の pool_allocator を使う
   */
  namespace stl
  {
    extern const int pool_allocator_type;
    struct ConsistencyCheck
    {
      ConsistencyCheck(int);
    };
#ifdef USE_TBB_SCALABLE_ALLOCATOR
    static ConsistencyCheck consistency_check(-1);
    template <class T>
    struct pool_allocator : tbb::scalable_allocator<T>
    {
      pool_allocator() {}
      template <class T2>
      pool_allocator(const tbb::scalable_allocator<T2>& src) : tbb::scalable_allocator<T>(src) {}
    };
#elif USE_BOOST_POOL_ALLOCATOR
    static ConsistencyCheck consistency_check(2);
    template <class T>
    struct fast_pool_allocator 
      : boost::mpl::if_c<(sizeof(T) <= 128), 
      boost::fast_pool_allocator
      <T,
       boost::default_user_allocator_new_delete,
       osl::LightMutex,
       128*65536/sizeof(T)+1
      >, 
      std::allocator<T> >::type
    {
      fast_pool_allocator() {}
      template <class T2, class T3, class T4, unsigned int U>
      fast_pool_allocator(const boost::fast_pool_allocator<T2,T3,T4,U>& src) {}
      template <class T2>
      fast_pool_allocator(const std::allocator<T2>& src) {}
    };
    template <class T>
    struct pool_allocator : std::allocator<T>
    {
      pool_allocator() {}
      template <class T2>
      pool_allocator(const std::allocator<T2>&) {}
    };
#else
    static ConsistencyCheck consistency_check(0);
    template <class T>
    struct pool_allocator : std::allocator<T>
    {
      pool_allocator() {}
      template <class T2>
      pool_allocator(const std::allocator<T2>&) {}
    };
#endif
  } // namespace stl
} // namespace osl

#endif /* OSL_POOL_ALLOCATOR_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
