/* This file is part of the KDE project
   Copyright (C) 2001 Christoph Cullmann <cullmann@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#ifndef DOCUMENTMANAGER_H
#define DOCUMENTMANAGER_H

#include <kate_export.h>

#include <QtCore/QObject>
#include <kurl.h>

namespace KTextEditor
{
  class Document;
}

namespace Kate
{
  /**
   * \brief Interface for the document manager.
   *
   * This interface provides access to Kate's document manager. The document
   * manager manages all documents. Use document() get a given document,
   * activeDocument() to retrieve the active document. Check with isOpen()
   * whether an URL is opened and use findDocument() to get it. To get the
   * number of managed documents use documents().
   *
   * Open new documents with openUrl() and close a document with closeDocument()
   * or closeAllDocuments(). Several signals are provided, documentChanged() is
   * emitted whenever the document's content changed, documentCreated() when a
   * new document was created and documentDeleted() when a document was closed.
   *
   * To access the document manager use the global accessor function
   * documentManager() or Application::documentManager(). You should never have
   * to create an instance of this class yourself.
   *
   * \author Christoph Cullmann \<cullmann@kde.org\>
   */
  class KATEINTERFACES_EXPORT DocumentManager : public QObject
  {
      friend class PrivateDocumentManager;

      Q_OBJECT

    public:
      /**
       * Construtor.
       *
       * The constructor is internally used by the Kate application, so it is
       * of no interest for plugin developers. Plugin developers should use the
       * global accessor pluginManager() instead.
       *
       * \param documentManager internal usage
       *
       * \internal
       */
      DocumentManager ( void *documentManager  );
      /**
       * Virtual destructor.
       */
      virtual ~DocumentManager ();

    public:
      /**
       * Get a list of all documents.
       * @return all documents
       */
      const QList<KTextEditor::Document*> &documents () const;

      /**
       * Get the document with the URL \p url.
       * if multiple documents match the searched url, return the first found one...
       * \param url the document's URL
       * \return the document with the given \p url or NULL, if no such document
       *         is in the document manager's internal list.
       */
      KTextEditor::Document *findUrl (const KUrl &url) const;

      /**
       * Open the document \p url with the given \p encoding.
       * if the url is empty, a new empty document will be created
       * \param url the document's url
       * \param encoding the preferred encoding. If encoding is QString() the
       *        encoding will be guessed or the default encoding will be used.
       * \return a pointer to the created document
       */
      KTextEditor::Document *openUrl (const KUrl &url, const QString &encoding = QString());

      /**
       * Close the given \p document.
       * \param document the document to be closed
       * \return \e true on success, otherwise \e false
       */
      bool closeDocument (KTextEditor::Document* document);

      /**
       * Close a list of documents. If any of them are modified, show a "save modified" dialog.
       * \param documents list of documents to be closed
       * \return \e true on success, otherwise \e false
       */
      bool closeDocumentList (QList<KTextEditor::Document*> documents);

      //
      // SIGNALS !!!
      //
#ifndef Q_MOC_RUN
#undef signals
#define signals public
#endif
    signals:
#ifndef Q_MOC_RUN
#undef signals
#define signals protected
#endif

      /**
       * This signal is emitted when the \p document was created.
       */
      void documentCreated (KTextEditor::Document *document);

      /**
       * This signal is emitted before a \p document which should be closed is deleted
       * The document is still accessible and usable, but it will be deleted
       * after this signal was send.
       */
      void documentWillBeDeleted (KTextEditor::Document *document);

      /**
       * This signal is emitted when the \p document has been deleted.
       *
       *  Warning !!! DO NOT ACCESS THE DATA REFERENCED BY THE POINTER, IT IS ALREADY INVALID
       *  Use the pointer only to remove mappings in hash or maps
       */
      void documentDeleted (KTextEditor::Document *document);

      /**
       * will be loading a bunch of documents, you can step back for a while
       */
      void aboutToLoadDocuments();

      /**
       * bunch of documents have been loaded, you can come back
       */
      void documentsLoaded(const QList<KTextEditor::Document *> &);

      /**
       * signal which documents are going to be deleted soon
       *
       * note that the batch can be interupted in the middle and only some
       * of the documents may be actually deleted. See @documentsDeleted signal.
       */
      void aboutToDeleteDocuments(const QList<KTextEditor::Document *> &);

      /**
       * the batch closing signal for @aboutToDeleteDocuments
       * @documents the documents that weren't deleted after all
       */
      void documentsDeleted(const QList<KTextEditor::Document *> &documents);

    private:
      class PrivateDocumentManager *d;
  };

  /**
   * Global accessor to the document manager object.
   * \return document manager object
   */
  KATEINTERFACES_EXPORT DocumentManager *documentManager ();

}

#endif // DOCUMENTMANAGER_H

// kate: space-indent on; indent-width 2; replace-tabs on;
