/*
 * KAKASI (Kanji Kana Simple inversion program)
 * $Id: level.h,v 1.1 2002-12-06 13:45:56 baba Exp $
 * Copyright (C) 2002 Hajime BABA  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either versions 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with KAKASI, see the file COPYING.  If not, write to the Free
 * Software Foundation Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifndef _LEVEL_H
#define _LEVEL_H


/*
 * Distribution table of educational kanji (Kyouiku Kanji: 1006 characters).
 * http://www.mext.go.jp/b_menu/shuppan/sonota/990301b/990301d.htm
 */

/*
 * Kanji for the first grade of elementary school (80 characters)
 */
static unsigned char level1_table[80][3] = {
  "\260\354", "\261\246", "\261\253", "\261\337", "\262\246",  /*   5 */
  "\262\273", "\262\274", "\262\320", "\262\326", "\263\255",  /*  10 */
  "\263\330", "\265\244", "\265\331", "\266\314", "\266\342",  /*  15 */
  "\266\345", "\266\365", "\267\356", "\270\244", "\270\253",  /*  20 */
  "\270\336", "\270\375", "\271\273", "\272\270", "\273\260",  /*  25 */
  "\273\263", "\273\315", "\273\322", "\273\345", "\273\372",  /*  30 */
  "\274\252", "\274\267", "\274\326", "\274\352", "\275\275",  /*  35 */
  "\275\320", "\275\367", "\276\256", "\276\345", "\277\271",  /*  40 */
  "\277\315", "\277\345", "\300\265", "\300\270", "\300\304",  /*  45 */
  "\300\320", "\300\326", "\300\350", "\300\351", "\300\356",  /*  50 */
  "\301\341", "\301\360", "\302\255", "\302\274", "\302\347",  /*  55 */
  "\303\313", "\303\335", "\303\346", "\303\356", "\304\256",  /*  60 */
  "\305\267", "\305\304", "\305\332", "\306\363", "\306\374",  /*  65 */
  "\306\376", "\307\257", "\307\362", "\310\254", "\311\264",  /*  70 */
  "\312\270", "\313\334", "\314\276", "\314\332", "\314\334",  /*  75 */
  "\315\274", "\316\251", "\316\317", "\316\323", "\317\273",  /*  80 */
};

/* Block index on the Japanese character code (JIS X 0208) */
static int level1_index[94] = {
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,  /* 10 */
    0,   0,   0,   0,   0,   0,   1,   4,   9,  11,  /* 20 */
   11,  13,  17,  18,  22,  23,  24,  30,  34,  37,  /* 30 */
   39,  42,  50,  52,  55,  59,  60,  63,  66,  68,  /* 40 */
   69,  70,  71,  72,  75,  76,  79,  80,  80,  80,  /* 50 */
   80,  80,  80,  80,  80,  80,  80,  80,  80,  80,  /* 60 */
   80,  80,  80,  80,  80,  80,  80,  80,  80,  80,  /* 70 */
   80,  80,  80,  80,  80,  80,  80,  80,  80,  80,  /* 80 */
   80,  80,  80,  80,  80,  80,  80,  80,  80,  80,  /* 90 */
   80,  80,  80,  80, 
};


/*
 * Kanji for the second grade of elementary school (160 characters)
 */
static unsigned char level2_table[160][3] = {
  "\260\372", "\261\251", "\261\300", "\261\340", "\261\363",  /*   5 */
  "\262\253", "\262\277", "\262\306", "\262\310", "\262\312",  /*  10 */
  "\262\316", "\262\350", "\262\361", "\262\363", "\263\244",  /*  15 */
  "\263\250", "\263\260", "\263\321", "\263\332", "\263\350",  /*  20 */
  "\264\326", "\264\335", "\264\344", "\264\351", "\265\242",  /*  25 */
  "\265\245", "\265\255", "\265\335", "\265\355", "\265\373",  /*  30 */
  "\265\376", "\266\257", "\266\265", "\266\341", "\267\273",  /*  35 */
  "\267\301", "\267\327", "\270\265", "\270\266", "\270\300",  /*  40 */
  "\270\305", "\270\315", "\270\341", "\270\345", "\270\354",  /*  45 */
  "\270\362", "\270\367", "\270\370", "\271\251", "\271\255",  /*  50 */
  "\271\315", "\271\324", "\271\342", "\271\347", "\271\361",  /*  55 */
  "\271\365", "\272\243", "\272\315", "\272\331", "\272\356",  /*  60 */
  "\273\273", "\273\320", "\273\324", "\273\327", "\273\337",  /*  65 */
  "\273\346", "\273\373", "\273\376", "\274\253", "\274\274",  /*  70 */
  "\274\322", "\274\345", "\274\363", "\275\251", "\275\265",  /*  75 */
  "\275\325", "\275\361", "\276\257", "\276\354", "\277\247",  /*  80 */
  "\277\251", "\277\264", "\277\267", "\277\306", "\277\336",  /*  85 */
  "\277\364", "\300\261", "\300\262", "\300\274", "\300\276",  /*  90 */
  "\300\332", "\300\343", "\300\376", "\301\245", "\301\260",  /*  95 */
  "\301\310", "\301\366", "\302\277", "\302\300", "\302\316",  /* 100 */
  "\302\346", "\303\253", "\303\316", "\303\317", "\303\323",  /* 105 */
  "\303\343", "\303\353", "\304\253", "\304\271", "\304\273",  /* 110 */
  "\304\276", "\304\314", "\304\357", "\305\271", "\305\300",  /* 115 */
  "\305\305", "\305\337", "\305\341", "\305\354", "\305\366",  /* 120 */
  "\305\372", "\306\254", "\306\261", "\306\273", "\306\311",  /* 125 */
  "\306\342", "\306\356", "\306\371", "\307\317", "\307\343",  /* 130 */
  "\307\344", "\307\376", "\310\276", "\310\326", "\311\343",  /* 135 */
  "\311\367", "\312\254", "\312\271", "\312\306", "\312\342",  /* 140 */
  "\312\354", "\312\375", "\313\314", "\313\345", "\313\350",  /* 145 */
  "\313\374", "\314\300", "\314\304", "\314\323", "\314\347",  /* 150 */
  "\314\353", "\314\356", "\314\360", "\315\247", "\315\313",  /* 155 */
  "\315\321", "\315\350", "\315\375", "\316\244", "\317\303",  /* 160 */
};

static int level2_index[94] = {
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,  /* 10 */
    0,   0,   0,   0,   0,   0,   1,   5,  14,  20,  /* 20 */
   24,  31,  34,  37,  48,  56,  60,  68,  73,  77,  /* 30 */
   79,  86,  93,  97, 101, 107, 113, 121, 128, 132,  /* 40 */
  134, 136, 142, 146, 153, 158, 159, 160, 160, 160,  /* 50 */
  160, 160, 160, 160, 160, 160, 160, 160, 160, 160,  /* 60 */
  160, 160, 160, 160, 160, 160, 160, 160, 160, 160,  /* 70 */
  160, 160, 160, 160, 160, 160, 160, 160, 160, 160,  /* 80 */
  160, 160, 160, 160, 160, 160, 160, 160, 160, 160,  /* 90 */
  160, 160, 160, 160, 
};


/*
 * Kanji for the third grade of elementary school (200 characters)
 */
static unsigned char level3_table[200][3] = {
  "\260\255", "\260\302", "\260\305", "\260\321", "\260\325",  /*   5 */
  "\260\345", "\260\351", "\260\367", "\260\373", "\261\241",  /*  10 */
  "\261\277", "\261\313", "\261\330", "\261\373", "\262\243",  /*  15 */
  "\262\260", "\262\271", "\262\275", "\262\331", "\263\246",  /*  20 */
  "\263\253", "\263\254", "\264\250", "\264\266", "\264\301",  /*  25 */
  "\264\333", "\264\337", "\264\374", "\265\257", "\265\322",  /*  30 */
  "\265\334", "\265\336", "\265\345", "\265\346", "\265\351",  /*  35 */
  "\265\356", "\266\266", "\266\310", "\266\311", "\266\312",  /*  40 */
  "\266\344", "\266\350", "\266\354", "\266\361", "\267\257",  /*  45 */
  "\267\270", "\267\332", "\267\350", "\267\354", "\270\246",  /*  50 */
  "\270\251", "\270\313", "\270\320", "\270\376", "\271\254",  /*  55 */
  "\271\301", "\271\346", "\272\254", "\272\327", "\272\344",  /*  60 */
  "\273\256", "\273\305", "\273\310", "\273\317", "\273\330",  /*  65 */
  "\273\340", "\273\355", "\273\365", "\273\366", "\273\375",  /*  70 */
  "\274\241", "\274\260", "\274\302", "\274\314", "\274\324",  /*  75 */
  "\274\347", "\274\350", "\274\351", "\274\362", "\274\365",  /*  80 */
  "\275\243", "\275\246", "\275\252", "\275\254", "\275\270",  /*  85 */
  "\275\273", "\275\305", "\275\311", "\275\352", "\275\353",  /*  90 */
  "\275\365", "\276\241", "\276\246", "\276\274", "\276\303",  /*  95 */
  "\276\317", "\276\350", "\277\242", "\277\274", "\277\275",  /* 100 */
  "\277\277", "\277\300", "\277\310", "\277\312", "\300\244",  /* 105 */
  "\300\260", "\300\316", "\301\264", "\301\333", "\301\352",  /* 110 */
  "\301\367", "\302\251", "\302\256", "\302\262", "\302\276",  /* 115 */
  "\302\307", "\302\320", "\302\324", "\302\345", "\302\350",  /* 120 */
  "\302\352", "\303\272", "\303\273", "\303\314", "\303\345",  /* 125 */
  "\303\354", "\303\355", "\303\372", "\304\242", "\304\264",  /* 130 */
  "\304\311", "\304\352", "\304\355", "\305\253", "\305\264",  /* 135 */
  "\305\276", "\305\320", "\305\324", "\305\331", "\305\347",  /* 140 */
  "\305\352", "\305\362", "\305\371", "\306\246", "\306\260",  /* 145 */
  "\306\270", "\307\300", "\307\310", "\307\333", "\307\334",  /* 150 */
  "\310\242", "\310\252", "\310\257", "\310\277", "\310\304",  /* 155 */
  "\310\341", "\310\351", "\310\376", "\311\241", "\311\256",  /* 160 */
  "\311\271", "\311\275", "\311\302", "\311\303", "\311\312",  /* 165 */
  "\311\351", "\311\364", "\311\376", "\312\241", "\312\252",  /* 170 */
  "\312\277", "\312\326", "\312\331", "\312\374", "\314\243",  /* 175 */
  "\314\277", "\314\314", "\314\344", "\314\362", "\314\364",  /* 180 */
  "\314\375", "\315\255", "\315\263", "\315\267", "\315\275",  /* 185 */
  "\315\315", "\315\316", "\315\323", "\315\325", "\315\333",  /* 190 */
  "\315\356", "\316\256", "\316\271", "\316\276", "\316\320",  /* 195 */
  "\316\351", "\316\363", "\316\375", "\317\251", "\317\302",  /* 200 */
};

static int level3_index[94] = {
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,  /* 10 */
    0,   0,   0,   0,   0,   0,   9,  14,  19,  22,  /* 20 */
   28,  36,  44,  49,  54,  57,  60,  70,  80,  91,  /* 30 */
   97, 104, 107, 111, 121, 128, 133, 143, 146, 150,  /* 40 */
  158, 168, 174, 174, 181, 191, 198, 200, 200, 200,  /* 50 */
  200, 200, 200, 200, 200, 200, 200, 200, 200, 200,  /* 60 */
  200, 200, 200, 200, 200, 200, 200, 200, 200, 200,  /* 70 */
  200, 200, 200, 200, 200, 200, 200, 200, 200, 200,  /* 80 */
  200, 200, 200, 200, 200, 200, 200, 200, 200, 200,  /* 90 */
  200, 200, 200, 200, 
};


/*
 * Kanji for the fourth grade of elementary school (200 characters)
 */
static unsigned char level4_table[200][3] = {
  "\260\246", "\260\306", "\260\312", "\260\314", "\260\317",  /*   5 */
  "\260\337", "\260\341", "\260\365", "\261\311", "\261\321",  /*  10 */
  "\261\366", "\262\257", "\262\303", "\262\314", "\262\335",  /*  15 */
  "\262\337", "\262\352", "\262\376", "\263\243", "\263\262",  /*  20 */
  "\263\271", "\263\306", "\263\320", "\264\260", "\264\261",  /*  25 */
  "\264\311", "\264\321", "\264\330", "\264\352", "\264\356",  /*  30 */
  "\264\357", "\264\365", "\264\372", "\265\241", "\265\250",  /*  35 */
  "\265\252", "\265\304", "\265\337", "\265\341", "\265\343",  /*  40 */
  "\265\353", "\265\363", "\265\371", "\266\245", "\266\246",  /*  45 */
  "\266\250", "\266\300", "\266\313", "\267\261", "\267\263",  /*  50 */
  "\267\264", "\267\277", "\267\302", "\267\312", "\267\335",  /*  55 */
  "\267\347", "\267\353", "\267\362", "\267\372", "\270\263",  /*  60 */
  "\270\307", "\270\365", "\270\371", "\271\245", "\271\257",  /*  65 */
  "\271\322", "\271\360", "\272\271", "\272\307", "\272\332",  /*  70 */
  "\272\340", "\272\362", "\272\376", "\273\241", "\273\245",  /*  75 */
  "\273\246", "\273\262", "\273\266", "\273\272", "\273\304",  /*  80 */
  "\273\312", "\273\313", "\273\316", "\273\341", "\273\356",  /*  85 */
  "\273\371", "\274\243", "\274\255", "\274\272", "\274\332",  /*  90 */
  "\274\357", "\274\376", "\275\313", "\275\347", "\275\351",  /*  95 */
  "\276\247", "\276\276", "\276\306", "\276\310", "\276\312",  /* 100 */
  "\276\320", "\276\335", "\276\336", "\277\256", "\277\303",  /* 105 */
  "\300\256", "\300\266", "\300\305", "\300\312", "\300\321",  /* 110 */
  "\300\336", "\300\341", "\300\342", "\300\357", "\300\365",  /* 115 */
  "\301\252", "\301\263", "\301\322", "\301\343", "\301\350",  /* 120 */
  "\302\246", "\302\253", "\302\263", "\302\264", "\302\271",  /* 125 */
  "\302\323", "\302\342", "\303\243", "\303\261", "\303\326",  /* 130 */
  "\303\347", "\303\371", "\303\373", "\304\262", "\304\343",  /* 135 */
  "\304\344", "\304\354", "\305\252", "\305\265", "\305\301",  /* 140 */
  "\305\314", "\305\330", "\305\364", "\306\257", "\306\262",  /* 145 */
  "\306\300", "\306\303", "\306\307", "\307\256", "\307\260",  /* 150 */
  "\307\324", "\307\337", "\307\356", "\310\323", "\310\361",  /* 155 */
  "\310\364", "\311\254", "\311\270", "\311\274", "\311\324",  /* 160 */
  "\311\325", "\311\327", "\311\334", "\311\373", "\312\264",  /* 165 */
  "\312\274", "\312\314", "\312\321", "\312\325", "\312\330",  /* 170 */
  "\312\361", "\313\241", "\313\276", "\313\322", "\313\366",  /* 175 */
  "\313\376", "\314\244", "\314\256", "\314\261", "\314\265",  /* 180 */
  "\314\363", "\315\246", "\315\327", "\315\334", "\315\341",  /* 185 */
  "\315\370", "\316\246", "\316\301", "\316\311", "\316\314",  /* 190 */
  "\316\330", "\316\340", "\316\341", "\316\343", "\316\344",  /* 195 */
  "\316\362", "\317\242", "\317\253", "\317\267", "\317\277",  /* 200 */
};

static int level4_index[94] = {
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,  /* 10 */
    0,   0,   0,   0,   0,   0,   8,  11,  18,  23,  /* 20 */
   33,  43,  48,  59,  63,  67,  73,  86,  92,  95,  /* 30 */
  103, 105, 115, 120, 127, 133, 137, 143, 148, 153,  /* 40 */
  156, 164, 171, 176, 181, 186, 196, 200, 200, 200,  /* 50 */
  200, 200, 200, 200, 200, 200, 200, 200, 200, 200,  /* 60 */
  200, 200, 200, 200, 200, 200, 200, 200, 200, 200,  /* 70 */
  200, 200, 200, 200, 200, 200, 200, 200, 200, 200,  /* 80 */
  200, 200, 200, 200, 200, 200, 200, 200, 200, 200,  /* 90 */
  200, 200, 200, 200, 
};


/*
 * Kanji for the fifth grade of elementary school (185 characters)
 */
static unsigned char level5_table[185][3] = {
  "\260\265", "\260\327", "\260\334", "\260\370", "\261\304",  /*   5 */
  "\261\312", "\261\322", "\261\325", "\261\327", "\261\351",  /*  10 */
  "\261\375", "\261\376", "\262\270", "\262\276", "\262\301",  /*  15 */
  "\262\304", "\262\317", "\262\341", "\262\354", "\262\362",  /*  20 */
  "\262\367", "\263\312", "\263\316", "\263\333", "\264\251",  /*  25 */
  "\264\264", "\264\267", "\264\343", "\264\360", "\264\363",  /*  30 */
  "\265\254", "\265\273", "\265\301", "\265\325", "\265\327",  /*  35 */
  "\265\354", "\265\357", "\265\366", "\266\255", "\266\275",  /*  40 */
  "\266\321", "\266\330", "\266\347", "\267\262", "\267\320",  /*  45 */
  "\267\351", "\267\357", "\267\364", "\270\241", "\270\261",  /*  50 */
  "\270\272", "\270\275", "\270\302", "\270\304", "\270\316",  /*  55 */
  "\270\356", "\270\372", "\270\374", "\271\275", "\271\314",  /*  60 */
  "\271\326", "\271\333", "\272\256", "\272\272", "\272\306",  /*  65 */
  "\272\312", "\272\316", "\272\322", "\272\335", "\272\337",  /*  70 */
  "\272\341", "\272\342", "\272\371", "\273\250", "\273\277",  /*  75 */
  "\273\300", "\273\325", "\273\326", "\273\331", "\273\336",  /*  80 */
  "\273\361", "\273\364", "\273\367", "\274\250", "\274\261",  /*  85 */
  "\274\301", "\274\313", "\274\325", "\274\370", "\275\244",  /*  90 */
  "\275\321", "\275\322", "\275\340", "\275\370", "\276\265",  /*  95 */
  "\276\267", "\276\332", "\276\357", "\276\360", "\276\362",  /* 100 */
  "\276\365", "\277\245", "\277\246", "\300\251", "\300\252",  /* 105 */
  "\300\255", "\300\257", "\300\272", "\300\275", "\300\307",  /* 110 */
  "\300\323", "\300\325", "\300\334", "\300\337", "\300\344",  /* 115 */
  "\300\345", "\301\254", "\301\304", "\301\307", "\301\355",  /* 120 */
  "\301\374", "\301\375", "\302\244", "\302\247", "\302\254",  /* 125 */
  "\302\260", "\302\273", "\302\326", "\302\337", "\302\340",  /* 130 */
  "\303\304", "\303\307", "\303\333", "\304\245", "\304\363",  /* 135 */
  "\304\370", "\305\250", "\305\254", "\305\375", "\306\263",  /* 140 */
  "\306\274", "\306\301", "\306\310", "\307\244", "\307\263",  /* 145 */
  "\307\275", "\307\313", "\310\275", "\310\307", "\310\310",  /* 150 */
  "\310\346", "\310\356", "\310\363", "\310\367", "\311\266",  /* 155 */
  "\311\276", "\311\317", "\311\330", "\311\331", "\311\333",  /* 160 */
  "\311\360", "\311\374", "\312\243", "\312\251", "\312\324",  /* 165 */
  "\312\333", "\312\335", "\312\350", "\312\363", "\313\255",  /* 170 */
  "\313\275", "\313\307", "\313\311", "\314\263", "\314\264",  /* 175 */
  "\314\302", "\314\312", "\315\242", "\315\276", "\315\302",  /* 180 */
  "\315\306", "\316\250", "\316\254", "\316\261", "\316\316",  /* 185 */
};

static int level5_index[94] = {
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,  /* 10 */
    0,   0,   0,   0,   0,   0,   4,  12,  21,  24,  /* 20 */
   30,  38,  43,  48,  58,  62,  73,  83,  89,  94,  /* 30 */
  101, 103, 116, 122, 130, 133, 136, 139, 143, 147,  /* 40 */
  154, 162, 169, 173, 177, 181, 185, 185, 185, 185,  /* 50 */
  185, 185, 185, 185, 185, 185, 185, 185, 185, 185,  /* 60 */
  185, 185, 185, 185, 185, 185, 185, 185, 185, 185,  /* 70 */
  185, 185, 185, 185, 185, 185, 185, 185, 185, 185,  /* 80 */
  185, 185, 185, 185, 185, 185, 185, 185, 185, 185,  /* 90 */
  185, 185, 185, 185, 
};


/*
 * Kanji for the sixth grade of elementary school (181 characters)
 */
static unsigned char level6_table[181][3] = {
  "\260\333", "\260\344", "\260\350", "\261\247", "\261\307",  /*   5 */
  "\261\344", "\261\350", "\262\346", "\263\245", "\263\310",  /*  10 */
  "\263\325", "\263\327", "\263\344", "\263\364", "\264\254",  /*  15 */
  "\264\263", "\264\307", "\264\312", "\264\355", "\264\370",  /*  20 */
  "\264\371", "\265\256", "\265\277", "\265\333", "\266\241",  /*  25 */
  "\266\273", "\266\277", "\266\320", "\266\332", "\267\311",  /*  30 */
  "\267\317", "\267\331", "\267\340", "\267\343", "\267\352",  /*  35 */
  "\267\373", "\270\242", "\270\250", "\270\267", "\270\273",  /*  40 */
  "\270\306", "\270\312", "\270\355", "\271\241", "\271\247",  /*  45 */
  "\271\304", "\271\310", "\271\335", "\271\337", "\271\357",  /*  50 */
  "\271\362", "\271\374", "\272\244", "\272\275", "\272\302",  /*  55 */
  "\272\321", "\272\333", "\272\366", "\272\375", "\273\275",  /*  60 */
  "\273\321", "\273\344", "\273\352", "\273\353", "\273\354",  /*  65 */
  "\273\357", "\274\247", "\274\315", "\274\316", "\274\334",  /*  70 */
  "\274\343", "\274\371", "\274\375", "\275\241", "\275\242",  /*  75 */
  "\275\260", "\275\276", "\275\304", "\275\314", "\275\317",  /*  80 */
  "\275\343", "\275\350", "\275\360", "\275\364", "\275\374",  /*  85 */
  "\275\375", "\276\255", "\276\343", "\276\353", "\276\370",  /*  90 */
  "\277\313", "\277\316", "\277\342", "\277\344", "\300\243",  /*  95 */
  "\300\271", "\300\273", "\300\277", "\300\353", "\300\354",  /* 100 */
  "\300\364", "\300\366", "\300\367", "\301\261", "\301\317",  /* 105 */
  "\301\325", "\301\330", "\301\340", "\301\353", "\301\365",  /* 110 */
  "\302\241", "\302\242", "\302\270", "\302\272", "\302\360",  /* 115 */
  "\303\264", "\303\265", "\303\302", "\303\310", "\303\312",  /* 120 */
  "\303\315", "\303\350", "\303\351", "\303\370", "\304\243",  /* 125 */
  "\304\254", "\304\272", "\304\302", "\304\313", "\305\270",  /* 130 */
  "\305\336", "\305\374", "\306\244", "\306\317", "\306\361",  /* 135 */
  "\306\375", "\307\247", "\307\274", "\307\276", "\307\311",  /* 140 */
  "\307\320", "\307\322", "\307\330", "\307\331", "\310\311",  /* 145 */
  "\310\325", "\310\335", "\310\343", "\310\353", "\312\242",  /* 150 */
  "\312\263", "\312\302", "\312\304", "\312\305", "\312\322",  /* 155 */
  "\312\344", "\312\353", "\312\365", "\313\254", "\313\264",  /* 160 */
  "\313\272", "\313\300", "\313\347", "\313\353", "\314\251",  /* 165 */
  "\314\301", "\314\317", "\314\365", "\315\245", "\315\271",  /* 170 */
  "\315\304", "\315\337", "\315\342", "\315\360", "\315\361",  /* 175 */
  "\315\367", "\316\242", "\316\247", "\316\327", "\317\257",  /* 180 */
  "\317\300", 
};

static int level6_index[94] = {
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,  /* 10 */
    0,   0,   0,   0,   0,   0,   3,   7,   8,  14,  /* 20 */
   21,  24,  29,  36,  43,  52,  59,  66,  73,  86,  /* 30 */
   90,  94, 103, 110, 115, 124, 129, 132, 136, 144,  /* 40 */
  149, 149, 158, 164, 168, 176, 179, 181, 181, 181,  /* 50 */
  181, 181, 181, 181, 181, 181, 181, 181, 181, 181,  /* 60 */
  181, 181, 181, 181, 181, 181, 181, 181, 181, 181,  /* 70 */
  181, 181, 181, 181, 181, 181, 181, 181, 181, 181,  /* 80 */
  181, 181, 181, 181, 181, 181, 181, 181, 181, 181,  /* 90 */
  181, 181, 181, 181, 
};


/*
 * Kanji for common use (Jyouyou Kanji: 1945 characters)
 * http://kanji.zinbun.kyoto-u.ac.jp/~yasuoka/kanjibukuro/japan-joyo.html
 *
 * Except for the educational kanji, 939 characters are included
 * in the following table.
 */
static unsigned char levelj_table[939][3] = {
  "\260\241", "\260\245", "\260\256", "\260\267", "\260\315",  /*   5 */
  "\260\316", "\260\322", "\260\323", "\260\326", "\260\331",  /*  10 */
  "\260\335", "\260\336", "\260\343", "\260\346", "\260\355",  /*  15 */
  "\260\357", "\260\360", "\260\362", "\260\371", "\261\242",  /*  20 */
  "\261\243", "\261\244", "\261\262", "\261\272", "\261\306",  /*  25 */
  "\261\323", "\261\324", "\261\326", "\261\331", "\261\332",  /*  30 */
  "\261\333", "\261\334", "\261\343", "\261\347", "\261\352",  /*  35 */
  "\261\354", "\261\356", "\261\357", "\261\364", "\261\370",  /*  40 */
  "\261\372", "\261\374", "\262\241", "\262\244", "\262\245",  /*  45 */
  "\262\247", "\262\255", "\262\261", "\262\265", "\262\267",  /*  50 */
  "\262\272", "\262\302", "\262\307", "\262\311", "\262\313",  /*  55 */
  "\262\315", "\262\322", "\262\324", "\262\325", "\262\332",  /*  60 */
  "\262\333", "\262\343", "\262\355", "\262\356", "\262\360",  /*  65 */
  "\262\364", "\262\365", "\262\370", "\262\371", "\262\373",  /*  70 */
  "\262\374", "\262\375", "\263\247", "\263\257", "\263\264",  /*  75 */
  "\263\265", "\263\266", "\263\272", "\263\300", "\263\305",  /*  80 */
  "\263\313", "\263\314", "\263\315", "\263\317", "\263\323",  /*  85 */
  "\263\324", "\263\326", "\263\331", "\263\335", "\263\343",  /*  90 */
  "\263\345", "\263\347", "\263\351", "\263\352", "\263\354",  /*  95 */
  "\263\355", "\263\356", "\264\242", "\264\245", "\264\247",  /* 100 */
  "\264\252", "\264\253", "\264\255", "\264\256", "\264\262",  /* 105 */
  "\264\265", "\264\270", "\264\271", "\264\272", "\264\275",  /* 110 */
  "\264\276", "\264\277", "\264\300", "\264\304", "\264\305",  /* 115 */
  "\264\306", "\264\313", "\264\314", "\264\316", "\264\317",  /* 120 */
  "\264\323", "\264\324", "\264\325", "\264\327", "\264\331",  /* 125 */
  "\264\336", "\264\350", "\264\353", "\264\361", "\264\364",  /* 130 */
  "\264\366", "\264\367", "\264\373", "\264\375", "\264\376",  /* 135 */
  "\265\247", "\265\260", "\265\261", "\265\262", "\265\263",  /* 140 */
  "\265\264", "\265\266", "\265\267", "\265\271", "\265\272",  /* 145 */
  "\265\274", "\265\275", "\265\276", "\265\306", "\265\310",  /* 150 */
  "\265\312", "\265\315", "\265\321", "\265\323", "\265\324",  /* 155 */
  "\265\326", "\265\332", "\265\340", "\265\347", "\265\352",  /* 160 */
  "\265\360", "\265\361", "\265\362", "\265\365", "\265\367",  /* 165 */
  "\265\375", "\266\247", "\266\253", "\266\256", "\266\262",  /* 170 */
  "\266\263", "\266\264", "\266\267", "\266\270", "\266\271",  /* 175 */
  "\266\272", "\266\274", "\266\301", "\266\303", "\266\304",  /* 180 */
  "\266\305", "\266\307", "\266\324", "\266\327", "\266\333",  /* 185 */
  "\266\335", "\266\337", "\266\340", "\266\343", "\266\356",  /* 190 */
  "\266\362", "\266\363", "\266\366", "\266\370", "\266\371",  /* 195 */
  "\266\376", "\267\241", "\267\244", "\267\253", "\267\254",  /* 200 */
  "\267\256", "\267\260", "\267\271", "\267\272", "\267\274",  /* 205 */
  "\267\300", "\267\303", "\267\304", "\267\306", "\267\307",  /* 210 */
  "\267\310", "\267\314", "\267\321", "\267\324", "\267\326",  /* 215 */
  "\267\334", "\267\336", "\267\337", "\267\342", "\267\346",  /* 220 */
  "\267\360", "\267\363", "\267\365", "\267\367", "\267\370",  /* 225 */
  "\267\371", "\267\374", "\270\245", "\270\252", "\270\254",  /* 230 */
  "\270\255", "\270\256", "\270\257", "\270\262", "\270\270",  /* 235 */
  "\270\271", "\270\274", "\270\311", "\270\314", "\270\317",  /* 240 */
  "\270\330", "\270\333", "\270\334", "\270\335", "\270\337",  /* 245 */
  "\270\342", "\270\344", "\270\346", "\270\347", "\270\353",  /* 250 */
  "\270\364", "\271\243", "\271\246", "\271\252", "\271\261",  /* 255 */
  "\271\262", "\271\263", "\271\264", "\271\265", "\271\266",  /* 260 */
  "\271\271", "\271\276", "\271\277", "\271\302", "\271\303",  /* 265 */
  "\271\305", "\271\306", "\271\312", "\271\313", "\271\316",  /* 270 */
  "\271\323", "\271\325", "\271\327", "\271\330", "\271\331",  /* 275 */
  "\271\332", "\271\340", "\271\341", "\271\344", "\271\351",  /* 280 */
  "\271\353", "\271\356", "\271\363", "\271\366", "\271\370",  /* 285 */
  "\271\376", "\272\246", "\272\247", "\272\250", "\272\251",  /* 290 */
  "\272\253", "\272\260", "\272\262", "\272\264", "\272\266",  /* 295 */
  "\272\276", "\272\277", "\272\304", "\272\305", "\272\313",  /* 300 */
  "\272\314", "\272\317", "\272\320", "\272\325", "\272\330",  /* 305 */
  "\272\334", "\272\336", "\272\351", "\272\352", "\272\357",  /* 310 */
  "\272\361", "\272\367", "\272\370", "\273\243", "\273\244",  /* 315 */
  "\273\261", "\273\264", "\273\267", "\273\303", "\273\307",  /* 320 */
  "\273\311", "\273\314", "\273\334", "\273\335", "\273\343",  /* 325 */
  "\273\347", "\273\350", "\273\351", "\273\360", "\273\362",  /* 330 */
  "\273\363", "\273\370", "\273\374", "\274\242", "\274\245",  /* 335 */
  "\274\264", "\274\271", "\274\276", "\274\277", "\274\300",  /* 340 */
  "\274\307", "\274\317", "\274\320", "\274\321", "\274\327",  /* 345 */
  "\274\330", "\274\331", "\274\333", "\274\337", "\274\340",  /* 350 */
  "\274\341", "\274\344", "\274\353", "\274\354", "\274\355",  /* 355 */
  "\274\356", "\274\361", "\274\364", "\274\367", "\274\373",  /* 360 */
  "\274\374", "\275\245", "\275\250", "\275\255", "\275\256",  /* 365 */
  "\275\261", "\275\267", "\275\271", "\275\274", "\275\300",  /* 370 */
  "\275\301", "\275\302", "\275\303", "\275\306", "\275\307",  /* 375 */
  "\275\312", "\275\315", "\275\316", "\275\323", "\275\326",  /* 380 */
  "\275\332", "\275\333", "\275\334", "\275\336", "\275\341",  /* 385 */
  "\275\342", "\275\344", "\275\345", "\275\356", "\275\357",  /* 390 */
  "\275\366", "\275\371", "\275\376", "\276\242", "\276\243",  /* 395 */
  "\276\244", "\276\251", "\276\254", "\276\260", "\276\262",  /* 400 */
  "\276\264", "\276\266", "\276\270", "\276\272", "\276\275",  /* 405 */
  "\276\302", "\276\304", "\276\307", "\276\311", "\276\313",  /* 410 */
  "\276\314", "\276\315", "\276\316", "\276\321", "\276\322",  /* 415 */
  "\276\323", "\276\327", "\276\331", "\276\333", "\276\334",  /* 420 */
  "\276\342", "\276\346", "\276\351", "\276\352", "\276\355",  /* 425 */
  "\276\356", "\276\364", "\276\366", "\276\371", "\276\372",  /* 430 */
  "\276\373", "\276\374", "\276\376", "\277\243", "\277\250",  /* 435 */
  "\277\253", "\277\255", "\277\257", "\277\260", "\277\261",  /* 440 */
  "\277\262", "\277\263", "\277\265", "\277\266", "\277\273",  /* 445 */
  "\277\302", "\277\305", "\277\307", "\277\311", "\277\314",  /* 450 */
  "\277\317", "\277\322", "\277\323", "\277\324", "\277\327",  /* 455 */
  "\277\330", "\277\335", "\277\341", "\277\343", "\277\346",  /* 460 */
  "\277\347", "\277\350", "\277\352", "\277\353", "\277\354",  /* 465 */
  "\277\356", "\277\357", "\277\361", "\277\362", "\277\365",  /* 470 */
  "\277\370", "\277\371", "\300\241", "\300\245", "\300\246",  /* 475 */
  "\300\247", "\300\253", "\300\254", "\300\267", "\300\300",  /* 480 */
  "\300\301", "\300\302", "\300\306", "\300\311", "\300\313",  /* 485 */
  "\300\315", "\300\317", "\300\322", "\300\327", "\300\333",  /* 490 */
  "\300\335", "\300\340", "\300\347", "\300\352", "\300\360",  /* 495 */
  "\300\362", "\300\370", "\300\373", "\301\241", "\301\246",  /* 500 */
  "\301\251", "\301\253", "\301\255", "\301\257", "\301\262",  /* 505 */
  "\301\265", "\301\266", "\301\272", "\301\274", "\301\302",  /* 510 */
  "\301\303", "\301\305", "\301\306", "\301\312", "\301\313",  /* 515 */
  "\301\316", "\301\320", "\301\323", "\301\324", "\301\334",  /* 520 */
  "\301\335", "\301\336", "\301\342", "\301\345", "\301\347",  /* 525 */
  "\301\361", "\301\362", "\301\364", "\301\370", "\301\372",  /* 530 */
  "\301\373", "\301\376", "\302\243", "\302\245", "\302\250",  /* 535 */
  "\302\257", "\302\261", "\302\304", "\302\305", "\302\306",  /* 540 */
  "\302\314", "\302\321", "\302\325", "\302\330", "\302\331",  /* 545 */
  "\302\332", "\302\333", "\302\336", "\302\341", "\302\354",  /* 550 */
  "\302\356", "\302\362", "\302\363", "\302\364", "\302\365",  /* 555 */
  "\302\367", "\302\371", "\302\372", "\303\242", "\303\245",  /* 560 */
  "\303\246", "\303\252", "\303\260", "\303\262", "\303\270",  /* 565 */
  "\303\274", "\303\300", "\303\303", "\303\305", "\303\306",  /* 570 */
  "\303\321", "\303\324", "\303\325", "\303\327", "\303\331",  /* 575 */
  "\303\334", "\303\337", "\303\340", "\303\341", "\303\342",  /* 580 */
  "\303\344", "\303\352", "\303\357", "\303\362", "\303\363",  /* 585 */
  "\304\244", "\304\246", "\304\247", "\304\250", "\304\251",  /* 590 */
  "\304\257", "\304\260", "\304\261", "\304\266", "\304\267",  /* 595 */
  "\304\274", "\304\277", "\304\300", "\304\301", "\304\303",  /* 600 */
  "\304\304", "\304\305", "\304\306", "\304\315", "\304\322",  /* 605 */
  "\304\332", "\304\340", "\304\342", "\304\345", "\304\347",  /* 610 */
  "\304\350", "\304\351", "\304\353", "\304\356", "\304\361",  /* 615 */
  "\304\371", "\304\372", "\304\373", "\304\376", "\305\241",  /* 620 */
  "\305\245", "\305\246", "\305\251", "\305\257", "\305\260",  /* 625 */
  "\305\261", "\305\263", "\305\272", "\305\302", "\305\307",  /* 630 */
  "\305\311", "\305\315", "\305\317", "\305\323", "\305\333",  /* 635 */
  "\305\334", "\305\335", "\305\340", "\305\342", "\305\343",  /* 640 */
  "\305\351", "\305\353", "\305\355", "\305\357", "\305\360",  /* 645 */
  "\305\367", "\305\373", "\305\376", "\306\245", "\306\247",  /* 650 */
  "\306\250", "\306\251", "\306\253", "\306\255", "\306\256",  /* 655 */
  "\306\266", "\306\271", "\306\275", "\306\277", "\306\304",  /* 660 */
  "\306\306", "\306\314", "\306\315", "\306\326", "\306\332",  /* 665 */
  "\306\336", "\306\337", "\306\354", "\306\360", "\306\364",  /* 670 */
  "\306\365", "\307\241", "\307\242", "\307\245", "\307\246",  /* 675 */
  "\307\253", "\307\255", "\307\264", "\307\272", "\307\273",  /* 680 */
  "\307\304", "\307\306", "\307\314", "\307\321", "\307\323",  /* 685 */
  "\307\325", "\307\332", "\307\335", "\307\336", "\307\345",  /* 690 */
  "\307\346", "\307\354", "\307\357", "\307\361", "\307\365",  /* 695 */
  "\307\366", "\307\367", "\307\371", "\307\372", "\307\373",  /* 700 */
  "\310\251", "\310\255", "\310\261", "\310\262", "\310\263",  /* 705 */
  "\310\264", "\310\266", "\310\274", "\310\301", "\310\302",  /* 710 */
  "\310\312", "\310\313", "\310\314", "\310\315", "\310\316",  /* 715 */
  "\310\317", "\310\321", "\310\322", "\310\327", "\310\332",  /* 720 */
  "\310\334", "\310\336", "\310\340", "\310\342", "\310\344",  /* 725 */
  "\310\347", "\310\350", "\310\352", "\310\355", "\310\357",  /* 730 */
  "\310\362", "\310\370", "\310\371", "\311\244", "\311\261",  /* 735 */
  "\311\272", "\311\301", "\311\304", "\311\315", "\311\320",  /* 740 */
  "\311\321", "\311\322", "\311\323", "\311\335", "\311\336",  /* 745 */
  "\311\337", "\311\341", "\311\342", "\311\344", "\311\345",  /* 750 */
  "\311\346", "\311\350", "\311\352", "\311\353", "\311\355",  /* 755 */
  "\311\356", "\311\361", "\311\365", "\311\372", "\311\375",  /* 760 */
  "\312\244", "\312\247", "\312\250", "\312\256", "\312\257",  /* 765 */
  "\312\260", "\312\266", "\312\267", "\312\272", "\312\273",  /* 770 */
  "\312\275", "\312\276", "\312\300", "\312\301", "\312\311",  /* 775 */
  "\312\312", "\312\320", "\312\327", "\312\336", "\312\341",  /* 780 */
  "\312\346", "\312\347", "\312\351", "\312\355", "\312\357",  /* 785 */
  "\312\360", "\312\364", "\312\366", "\312\370", "\312\372",  /* 790 */
  "\313\242", "\313\244", "\313\245", "\313\246", "\313\247",  /* 795 */
  "\313\253", "\313\256", "\313\260", "\313\263", "\313\265",  /* 800 */
  "\313\266", "\313\267", "\313\270", "\313\271", "\313\273",  /* 805 */
  "\313\274", "\313\277", "\313\301", "\313\302", "\313\303",  /* 810 */
  "\313\304", "\313\305", "\313\315", "\313\317", "\313\320",  /* 815 */
  "\313\321", "\313\327", "\313\331", "\313\333", "\313\335",  /* 820 */
  "\313\336", "\313\337", "\313\340", "\313\341", "\313\342",  /* 825 */
  "\313\343", "\313\344", "\313\354", "\313\364", "\313\365",  /* 830 */
  "\313\372", "\313\375", "\314\241", "\314\245", "\314\250",  /* 835 */
  "\314\257", "\314\262", "\314\267", "\314\270", "\314\273",  /* 840 */
  "\314\274", "\314\303", "\314\307", "\314\310", "\314\320",  /* 845 */
  "\314\321", "\314\324", "\314\325", "\314\326", "\314\327",  /* 850 */
  "\314\333", "\314\341", "\314\346", "\314\350", "\314\361",  /* 855 */
  "\314\366", "\314\370", "\314\373", "\314\376", "\315\241",  /* 860 */
  "\315\243", "\315\251", "\315\252", "\315\253", "\315\261",  /* 865 */
  "\315\265", "\315\266", "\315\272", "\315\273", "\315\277",  /* 870 */
  "\315\300", "\315\307", "\315\310", "\315\311", "\315\312",  /* 875 */
  "\315\317", "\315\322", "\315\330", "\315\331", "\315\336",  /* 880 */
  "\315\343", "\315\345", "\315\347", "\315\352", "\315\353",  /* 885 */
  "\315\355", "\315\357", "\315\363", "\315\364", "\315\371",  /* 890 */
  "\315\372", "\316\241", "\316\245", "\316\262", "\316\263",  /* 895 */
  "\316\264", "\316\265", "\316\270", "\316\272", "\316\273",  /* 900 */
  "\316\275", "\316\300", "\316\303", "\316\304", "\316\305",  /* 905 */
  "\316\310", "\316\315", "\316\321", "\316\322", "\316\331",  /* 910 */
  "\316\335", "\316\336", "\316\337", "\316\345", "\316\353",  /* 915 */
  "\316\354", "\316\355", "\316\356", "\316\357", "\316\360",  /* 920 */
  "\316\361", "\316\364", "\316\365", "\316\366", "\316\367",  /* 925 */
  "\316\370", "\317\243", "\317\247", "\317\252", "\317\255",  /* 930 */
  "\317\260", "\317\262", "\317\263", "\317\272", "\317\305",  /* 935 */
  "\317\307", "\317\310", "\317\321", "\317\323", 
};

static int levelj_index[94] = {
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,  /* 10 */
    0,   0,   0,   0,   0,   0,  19,  42,  72,  97,  /* 20 */
  135, 166, 196, 227, 251, 286, 313, 333, 361, 393,  /* 30 */
  433, 472, 498, 532, 558, 585, 619, 648, 671, 700,  /* 40 */
  733, 760, 790, 832, 859, 891, 926, 939, 939, 939,  /* 50 */
  939, 939, 939, 939, 939, 939, 939, 939, 939, 939,  /* 60 */
  939, 939, 939, 939, 939, 939, 939, 939, 939, 939,  /* 70 */
  939, 939, 939, 939, 939, 939, 939, 939, 939, 939,  /* 80 */
  939, 939, 939, 939, 939, 939, 939, 939, 939, 939,  /* 90 */
  939, 939, 939, 939, 
};


/*
 * Kanji for personal name (Jinmei-you Kanji: 285 characters)
 * http://law.e-gov.go.jp/htmldata/S22/S22F00501000094.html
 * http://kanji.zinbun.kyoto-u.ac.jp/~yasuoka/kanjibukuro/japan-jimmei.html
 */
static unsigned char leveln_table[285][3] = {
  "\260\244", "\260\252", "\260\253", "\260\257", "\260\260",  /*   5 */
  "\260\264", "\260\274", "\260\275", "\260\276", "\260\311",  /*  10 */
  "\260\313", "\260\324", "\260\347", "\260\352", "\260\353",  /*  15 */
  "\260\364", "\260\375", "\261\254", "\261\257", "\261\264",  /*  20 */
  "\261\303", "\261\315", "\261\360", "\261\361", "\261\367",  /*  25 */
  "\262\242", "\262\300", "\262\305", "\262\330", "\262\342",  /*  30 */
  "\263\241", "\263\256", "\263\276", "\263\360", "\263\362",  /*  35 */
  "\263\371", "\263\375", "\264\246", "\264\320", "\264\340",  /*  40 */
  "\264\354", "\264\362", "\265\243", "\265\251", "\265\265",  /*  45 */
  "\265\303", "\265\307", "\265\314", "\265\374", "\266\251",  /*  50 */
  "\266\254", "\266\306", "\266\315", "\266\323", "\266\325",  /*  55 */
  "\266\326", "\266\334", "\266\336", "\266\352", "\266\353",  /*  60 */
  "\266\360", "\267\247", "\267\252", "\267\266", "\267\275",  /*  65 */
  "\267\305", "\267\313", "\267\375", "\270\276", "\270\325",  /*  70 */
  "\270\327", "\270\340", "\270\343", "\270\350", "\270\352",  /*  75 */
  "\270\361", "\270\366", "\271\250", "\271\260", "\271\267",  /*  80 */
  "\271\270", "\271\300", "\271\311", "\271\343", "\272\267",  /*  85 */
  "\272\273", "\272\274", "\272\300", "\272\310", "\272\323",  /*  90 */
  "\272\343", "\272\363", "\272\373", "\273\251", "\273\270",  /*  95 */
  "\274\244", "\274\254", "\274\256", "\274\257", "\274\305",  /* 100 */
  "\274\323", "\275\247", "\275\324", "\275\327", "\275\330",  /* 105 */
  "\275\331", "\275\337", "\275\346", "\275\354", "\275\355",  /* 110 */
  "\275\372", "\276\261", "\276\271", "\276\273", "\276\277",  /* 115 */
  "\276\324", "\276\326", "\276\347", "\276\367", "\277\270",  /* 120 */
  "\277\272", "\277\301", "\277\334", "\277\351", "\277\360",  /* 125 */
  "\277\363", "\277\367", "\300\331", "\301\326", "\301\332",  /* 130 */
  "\301\356", "\301\357", "\301\363", "\302\301", "\302\343",  /* 135 */
  "\302\344", "\302\353", "\302\357", "\302\366", "\302\376",  /* 140 */
  "\303\244", "\303\247", "\303\266", "\303\311", "\303\322",  /* 145 */
  "\303\366", "\304\252", "\304\263", "\304\307", "\304\320",  /* 150 */
  "\304\325", "\304\330", "\304\335", "\304\341", "\304\360",  /* 155 */
  "\304\365", "\304\367", "\305\316", "\306\243", "\306\264",  /* 160 */
  "\306\267", "\306\322", "\306\323", "\306\327", "\306\330",  /* 165 */
  "\306\340", "\306\341", "\306\344", "\306\350", "\306\357",  /* 170 */
  "\306\372", "\307\265", "\307\267", "\307\303", "\307\353",  /* 175 */
  "\310\245", "\310\267", "\310\273", "\310\345", "\310\354",  /* 180 */
  "\310\375", "\311\242", "\311\247", "\311\262", "\311\267",  /* 185 */
  "\311\313", "\311\347", "\311\366", "\311\371", "\312\313",  /* 190 */
  "\312\343", "\312\345", "\312\376", "\313\250", "\313\261",  /* 195 */
  "\313\262", "\313\323", "\313\352", "\313\357", "\313\362",  /* 200 */
  "\313\373", "\314\246", "\314\255", "\314\272", "\314\322",  /* 205 */
  "\314\351", "\314\352", "\314\355", "\314\357", "\314\367",  /* 210 */
  "\315\244", "\315\250", "\315\256", "\315\257", "\315\264",  /* 215 */
  "\315\270", "\315\314", "\315\324", "\315\326", "\315\332",  /* 220 */
  "\315\362", "\315\365", "\315\366", "\315\373", "\315\374",  /* 225 */
  "\315\376", "\316\260", "\316\274", "\316\277", "\316\306",  /* 230 */
  "\316\307", "\316\312", "\316\313", "\316\326", "\316\333",  /* 235 */
  "\316\334", "\316\342", "\316\346", "\316\347", "\316\350",  /* 240 */
  "\317\241", "\317\244", "\317\275", "\317\301", "\317\313",  /* 245 */
  "\320\322", "\322\246", "\324\367", "\326\305", "\327\302",  /* 250 */
  "\332\345", "\332\347", "\332\357", "\332\360", "\332\366",  /* 255 */
  "\333\331", "\334\277", "\335\334", "\336\253", "\336\255",  /* 260 */
  "\336\346", "\336\373", "\337\272", "\337\371", "\340\242",  /* 265 */
  "\340\366", "\342\253", "\342\310", "\343\371", "\345\272",  /* 270 */
  "\345\305", "\346\306", "\346\373", "\347\375", "\350\275",  /* 275 */
  "\350\301", "\353\316", "\353\331", "\354\342", "\355\354",  /* 280 */
  "\360\363", "\361\245", "\363\325", "\364\245", "\364\246",  /* 285 */
};

static int leveln_index[94] = {
    0,   0,   0,   0,   0,   0,   0,   0,   0,   0,  /* 10 */
    0,   0,   0,   0,   0,   0,  17,  25,  30,  37,  /* 20 */
   42,  49,  61,  68,  77,  84,  93,  95, 101, 111,  /* 30 */
  119, 127, 128, 133, 140, 146, 157, 158, 171, 175,  /* 40 */
  181, 189, 193, 201, 210, 226, 240, 245, 246, 246,  /* 50 */
  247, 247, 248, 248, 249, 250, 250, 250, 255, 256,  /* 60 */
  257, 258, 262, 264, 266, 266, 268, 269, 269, 271,  /* 70 */
  273, 274, 276, 276, 276, 278, 279, 280, 280, 280,  /* 80 */
  281, 282, 282, 283, 285, 285, 285, 285, 285, 285,  /* 90 */
  285, 285, 285, 285, 
};



#endif /* _LEVEL_H */
