/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_CL)


#include "iclshell.h"


#include "icontrolmodule.h"
#include "ierror.h"
#include "iimagefactory.h"
#include "ioutputchannel.h"
#include "isystem.h"
#include "iviewmodule.h"
#include "ivtk.h"

#include "iclcommandinterpreter.h"
#include "iclinputchannel.h"
#include "iclshell.h"

#include <vtkCommand.h>
#include <vtkRenderWindow.h>
#include <vtkRenderWindowInteractor.h>


//#define USE_UI_WINDOW

#ifdef USE_UI_WINDOW
#include <vtkActor2D.h>
#include <vtkImageMapper.h>
#include <vtkRenderer.h>
#include <vtkRenderWindow.h>
#endif


namespace iclShell_Private
{
	class OutputChannel : public iOutputChannel
	{

	public:

		OutputChannel(iShell *s) : iOutputChannel(s)
		{
		}

	protected:

		virtual void DisplayBody(iConsole::MessageType type, const iString &message, const char *file, int line)
		{
			iString text = this->FormPlainTextMessage(type,message,file,line);

			switch(type)
			{
			case iConsole::_Info:
			case iConsole::_Notification:
				{
					cout << text.ToCharPointer();
					break;
				}
			case iConsole::_Warning:
			case iConsole::_LowError:
			case iConsole::_HighError:
			case iConsole::_FatalError:
				{
					cerr << text.ToCharPointer();
					break;
				}
			}
		}
	};


	class TimerObserver : public vtkCommand
	{

	public:

		static TimerObserver* New(iclCommandInterpreter *interpreter)
		{
			return new TimerObserver(interpreter);
		}

		virtual void Execute(vtkObject *obj, unsigned long id, void *)
		{
			if(mWorking) return;
			mWorking = true;

			vtkRenderWindowInteractor *rwi = iRequiredCast<vtkRenderWindowInteractor>(INFO,obj);
			rwi->SetLastEventPosition(rwi->GetEventPosition());

			if(mInterpreter->IsReady())
			{
				rwi->TerminateApp();
			}

			mWorking = false;
		}

	private:

		TimerObserver(iclCommandInterpreter *interpreter)
		{
			mInterpreter = interpreter; IERROR_ASSERT(mInterpreter);
			mWorking = false;

#ifndef IVTK_PRE52
			this->PassiveObserverOn();
#endif
		}

		bool mWorking;
		iclCommandInterpreter *mInterpreter;
	};
};


using namespace iParameter;
using namespace iclShell_Private;


iclShell::iclShell(int argc, char **argv) : iShell("cl",argc,argv)
{
	mInputChannel = new iclInputChannel; IERROR_ASSERT(mInputChannel);
	mInterpreter = 0; // must be constructed when the shell is complete.
	mTimerId = 0;
	mDriver = 0;
}


iclShell::~iclShell()
{
}


void iclShell::DefineSpecificCommandLineOptions()
{
}


void iclShell::PrepareForConstruction()
{
	cout << "IFrIT is being initialized...\r";
	cout.flush();
}


void iclShell::ConstructorBody()
{
	mInterpreter = new iclCommandInterpreter(this); IERROR_ASSERT(mInterpreter);
	iOutputChannel::SetInstance(new OutputChannel(this));

	if(this->GetControlModule()->GetViewModule(0)->GetRenderWindow()->IsA("vtkCocoaRenderWindow") != 0)
	{
		cerr << "Command-line shell does not work with Cocoa rendering system." << endl;
		exit(-1);
	}
}

  
void iclShell::DestructorBody()
{
	mInterpreter->Stop();
	mInterpreter->Delete();
}


void iclShell::PrepareToLoadStateFile()
{
}


void iclShell::Start()
{
	int i;

	for(i=0; i<this->GetControlModule()->GetNumberOfViewModules(); i++)
	{
		this->GetControlModule()->GetViewModule(i)->UpdateWindowNumber();
	}

	this->GetControlModule()->Render(RenderOption::All);  // render all windows

	for(i=0; i<this->GetControlModule()->GetNumberOfViewModules(); i++)
	{
		this->GetControlModule()->GetViewModule(i)->UpdateWindowNumber();
	}
	this->GetControlModule()->Render(RenderOption::All);  // render all windows

	mInterpreter->Start();
}


void iclShell::Exit()
{
	if(mDriver != 0)
	{
#ifdef IVTK_PRE52
		mDriver->DestroyTimer();
#else
		mDriver->DestroyTimer(mTimerId);
#endif
		mDriver->TerminateApp();
		mDriver = 0;
	}
	else
	{
		exit(0);
	}
}


#ifdef USE_UI_WINDOW
void iclShell::RunBody()
{
	const iImage *image = iImageFactory::FindIcon("genie2.png"); IERROR_ASSERT(image);

	vtkImageMapper *m = vtkImageMapper::New(); IERROR_ASSERT(m);
	m->SetColorWindow(255.0);
	m->SetColorLevel(127.5);
	m->RenderToRectangleOff();
	m->UseCustomExtentsOff();
	m->SetInput(image->DataObject());

	vtkActor2D *a = vtkActor2D::New(); IERROR_ASSERT(a);
	a->GetPositionCoordinate()->SetCoordinateSystemToNormalizedViewport();
	a->SetPosition(0.0,0.0);
	a->GetPosition2Coordinate()->SetCoordinateSystemToNormalizedViewport();
	a->SetPosition2(1.0,1.0);
	a->PickableOff();
	a->SetMapper(m);
	m->Delete();

	vtkRenderer *ren = vtkRenderer::New(); IERROR_ASSERT(ren);
	ren->SetInteractive(0);
	ren->AddActor2D(a);
	a->Delete();

	vtkRenderWindow *win = vtkRenderWindow::New();
	win->SetSize(image->Width(),image->Height());
	win->SetPosition(0,0);
	//win->SetBorders(0);
	win->SetWindowName("IFrIT");
	win->AddRenderer(ren);
	ren->Delete();

	mDriver = vtkRenderWindowInteractor::New(); IERROR_ASSERT(mDriver);
	mDriver->SetInteractorStyle(0);
	mDriver->SetEnableRender(0);
	mDriver->SetRenderWindow(win);
	win->Delete();

	mDriver->Initialize();

	//win->SetPosition(-100,-100);
	win->SetWindowName("IFrIT");

	TimerObserver *obs = TimerObserver::New(mInterpreter);

	mTimerId = mDriver->CreateRepeatingTimer(30);
	mDriver->AddObserver(vtkCommand::TimerEvent,obs);

	do
	{
		mDriver->Start();
		mInterpreter->CommitExec();
	}
	while(mDriver != 0);

	obs->Delete();
}
#else
void iclShell::RunBody()
{
	mDriver = 0;
	TimerObserver *obs = TimerObserver::New(mInterpreter);

	do
	{
		if(mDriver != this->GetControlModule()->GetViewModule(0)->GetInteractor())
		{
			mDriver = this->GetControlModule()->GetViewModule(0)->GetInteractor();

#ifdef IVTK_PRE52
			mTimerId = mDriver->CreateTimer(VTKI_TIMER_FIRST);
#else
			mTimerId = mDriver->CreateRepeatingTimer(30);
#endif
			mDriver->AddObserver(vtkCommand::TimerEvent,obs);
		}
		mDriver->Start();
		mInterpreter->CommitExec();
	}
	while(mDriver != 0);

	obs->Delete();
}
#endif


bool iclShell::AnalyseOneExtendedCommandLineOption(const struct Option&)
{
	return false;
}


//
//  State saving
//
bool iclShell::LoadShellStateFromFileBody(iFile &)
{
	//
	//  Nothing to do here
	//
	return true;
}


bool iclShell::SaveShellStateToFileBody(iFile &) const
{
	//
	//  Nothing to do here
	//
	return true;
}

#endif
