/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file urlHandler.c
 * \brief Common url handling functions
 */

#include <ffgtk.h>

/**
 * \brief url handler write callback for CURL
 * \param pPtr data pointer
 * \param nSize data size
 * \param nMembers number of data members
 * \param pStream url handler pointer
 */
static size_t urlWrite( void *pPtr, size_t nSize, size_t nMembers, void *pStream ) {
	struct sUrlHandler *psHandler = ( struct sUrlHandler * ) pStream;
	int nRealSize = nSize * nMembers;

	psHandler -> pnData = g_realloc( psHandler -> pnData, psHandler -> nSize + nRealSize + 1);
	memcpy( &psHandler -> pnData[ psHandler -> nSize ], pPtr, nRealSize );
	psHandler -> nSize += nRealSize;
	psHandler -> pnData[ psHandler -> nSize ] = '\0';

	return nMembers;
}

/**
 * \brief Get url
 * \param psHandler url handler pointer
 * \param psProfile profile structure pointer
 * \return error code
 */
gint readUrl( struct sUrlHandler *psHandler, struct sProfile *psProfile ) {
	static GStaticMutex sUrlMutex = G_STATIC_MUTEX_INIT;
	CURLcode nCode = CURLE_FAILED_INIT;

	if ( psHandler != NULL && psHandler -> psHandle ) {
		if ( psHandler -> pnData != NULL ) {
			g_free( psHandler -> pnData );
		}
		psHandler -> pnData = NULL;
		g_static_mutex_lock( &sUrlMutex );
		nCode = curl_easy_perform( psHandler -> psHandle );
		g_static_mutex_unlock( &sUrlMutex );
	} else {
		if ( psHandler == NULL ) {
			Debug( KERN_DEBUG, "ERROR: psHandler NULL\n" );
		} else {
			Debug( KERN_DEBUG, "ERROR: psHandler -> psHandle NULL\n" );
		}
	}

	return nCode;
}

/**
 * \brief Set POST data to url handler structure
 * \param psHandler url handler structure
 * \param pnString post string
 */
void setPostData( struct sUrlHandler *psHandler, char *pnString, ... ) {
	GString *psPost = g_string_new( "" );
	va_list arg;

	if ( psHandler -> pnPost != NULL ) {
		g_free( psHandler -> pnPost );
		psHandler -> pnPost = NULL;
	}

	va_start( arg, pnString );
	g_string_vprintf( psPost, pnString, arg );
	psHandler -> pnPost = g_string_free( psPost, FALSE );

	curl_easy_setopt( psHandler -> psHandle, CURLOPT_POSTFIELDS, psHandler -> pnPost );
	va_end( arg );
}

/**
 * \brief Create url handler
 * \param pnHost host address
 * \param nPort port address
 * \return new url handler pointer or NULL on error
 */
struct sUrlHandler *urlHandler( const char *pnHost, int nPort ) {
	struct sUrlHandler *psHandler = g_malloc( sizeof( struct sUrlHandler ) );

	if ( psHandler != NULL ) {
		memset( psHandler, 0, sizeof( struct sUrlHandler ) );
		psHandler -> pnData = NULL;
		psHandler -> nSize = 0;
		psHandler -> psHandle = curl_easy_init();
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_WRITEFUNCTION, urlWrite );
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_WRITEDATA, psHandler );
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_PORT, nPort );
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_URL, pnHost );
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_HTTPGET, 1L );
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_USERAGENT, "ffgtk" );
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_DNS_CACHE_TIMEOUT, 2L );
		curl_easy_setopt( psHandler -> psHandle, CURLOPT_FOLLOWLOCATION, 1L );

		// Debug only options
		//curl_easy_setopt( psHandler -> psHandle, CURLOPT_VERBOSE, 1L );
		//curl_easy_setopt( psHandler -> psHandle, CURLOPT_HEADER, 1L );

		if ( strncmp( pnHost, "https", 5 ) == 0 ) {
			curl_easy_setopt( psHandler -> psHandle, CURLOPT_SSL_VERIFYPEER, FALSE );
			curl_easy_setopt( psHandler -> psHandle, CURLOPT_SSL_VERIFYHOST, 1 );
		}
	}

	return psHandler;
}

/**
 * \brief Free url handler structure
 * \param psHandler url handler pointer
 * \return error code (0)
 */
gint freeHandler( struct sUrlHandler *psHandler ) {
	if ( psHandler != NULL ) {
		if ( psHandler -> pnData != NULL ) {
			g_free( psHandler -> pnData );
		}
		psHandler -> pnData = NULL;
		psHandler -> nSize = 0;
		if ( psHandler -> pnPost != NULL ) {
			g_free( psHandler -> pnPost );
		}
		psHandler -> pnPost = NULL;
		curl_easy_cleanup( psHandler -> psHandle );
		g_free( psHandler );
	}

	return 0;
}
