#!/usr/bin/env python3

import configparser

from fenrirscreenreader.core.i18n import _


class command:
    def __init__(self):
        pass

    def initialize(self, environment):
        self.env = environment

    def shutdown(self):
        pass

    def get_description(self):
        return "Show temporary changes not yet saved to file"

    def run(self):
        self.env["runtime"]["OutputManager"].present_text(
            "Checking for unsaved changes...", interrupt=True
        )

        try:
            # Read the current config file
            settings_file = self.env["runtime"][
                "SettingsManager"
            ].settings_file
            file_config = configparser.ConfigParser(interpolation=None)
            file_config.read(settings_file)

            # Compare with runtime settings
            runtime_settings = self.env["runtime"]["SettingsManager"].settings

            changes = []

            # Check speech settings specifically
            speech_sections = [
                "speech",
                "sound",
                "keyboard",
                "screen",
                "general",
            ]

            for section in speech_sections:
                if section in runtime_settings and section in file_config:
                    for option in runtime_settings[section]:
                        runtime_value = runtime_settings[section][option]
                        try:
                            file_value = file_config[section][option]
                        except Exception as e:
                            file_value = ""

                        if runtime_value != file_value:
                            changes.append(
                                f"{section}.{option}: {file_value} → {runtime_value}"
                            )

            if changes:
                self.env["runtime"]["OutputManager"].present_text(
                    f"found {len(changes)} unsaved changes:", interrupt=True
                )
                for change in changes[:5]:  # Limit to first 5 changes
                    self.env["runtime"]["OutputManager"].present_text(
                        change, interrupt=True
                    )

                if len(changes) > 5:
                    self.env["runtime"]["OutputManager"].present_text(
                        f"... and {len(changes) - 5} more changes",
                        interrupt=True,
                    )

                self.env["runtime"]["OutputManager"].present_text(
                    "Use save command to make these changes permanent",
                    interrupt=True,
                )
                self.env["runtime"]["OutputManager"].play_sound("Accept")
            else:
                self.env["runtime"]["OutputManager"].present_text(
                    "No unsaved changes found", interrupt=True
                )
                self.env["runtime"]["OutputManager"].play_sound("Cancel")

        except Exception as e:
            self.env["runtime"]["OutputManager"].present_text(
                f"Error checking for changes: {str(e)}", interrupt=True
            )
            self.env["runtime"]["OutputManager"].play_sound("Error")

    def set_callback(self, callback):
        pass
