/*******************************************************************************

  Intel Data Center Bridging (DCB) Software
  Copyright(c) 2007-2009 Intel Corporation.

  This program is free software; you can redistribute it and/or modify it
  under the terms and conditions of the GNU General Public License,
  version 2, as published by the Free Software Foundation.

  This program is distributed in the hope it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

  The full GNU General Public License is included in this distribution in
  the file called "COPYING".

  Contact Information:
  e1000-eedc Mailing List <e1000-eedc@lists.sourceforge.net>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

*******************************************************************************/

#include "dcb_osdep.h"
#include "ports.h"
#include "l2_packet.h"
#include "tlv.h"
#include "states.h"
#include "mibdata.h"
#include "dcb_protocol.h"
#include "messages.h"

static u32 check_feature_not_present(char *device_name, u32 subtype,
				u32 EventFlag, u32 feature);

void rxInitializeLLDP(struct port *port) {

	port->rx.rcvFrame = FALSE;
	port->rx.badFrame = FALSE;
	port->rx.tooManyNghbrs = FALSE;
	port->rx.rxInfoAge = FALSE;
	if (port->rx.framein) {
		free(port->rx.framein);
		port->rx.framein = NULL;
	}
	port->rx.sizein = 0;

	mibDeleteObjects(port);
	return;
}

void rxReceiveFrame(void *ctx, unsigned int ifindex, const u8 *buf, size_t len)
{

	struct port * port;
	u8  frame_error = 0;
	struct l2_ethhdr *hdr;
	struct l2_ethhdr example_hdr,*ex;

	port = (struct port *)ctx;

	if (port->rx.framein)
		free(port->rx.framein);

	port->rx.framein = (u8 *)malloc(len);
	if (port->rx.framein == NULL) {
		printf("ERROR - could not allocate memory for rx'ed frame\n");
		return;
	}
	memcpy(port->rx.framein, buf, len);

	port->rx.sizein = (u16)len;
	ex = &example_hdr;
	memcpy(ex->h_dest, multi_cast_source, ETH_ALEN);
	ex->h_proto = htons(ETH_P_LLDP);
	hdr = (struct l2_ethhdr *)port->rx.framein;

	if ((memcmp(hdr->h_dest,ex->h_dest, ETH_ALEN) != 0)) {
		printf("ERROR LLDP multicast address error in incoming frame. "
			"Dropping frame.\n");
		frame_error++;
		free(port->rx.framein);
		port->rx.framein = NULL;
		port->rx.sizein = 0;
		return;
	}

	if (hdr->h_proto != example_hdr.h_proto) {
		printf("ERROR Ethertype not LLDP ethertype but ethertype "
			"'%x' in incoming frame.\n", htons(hdr->h_proto));
		frame_error++;
		free(port->rx.framein);
		port->rx.framein = NULL;
		port->rx.sizein = 0;
		return;
	}

	if (!frame_error) {
		port->stats.statsFramesInTotal++;
		port->rx.rcvFrame = 1;
	}
	run_rx_sm(port, FALSE);
}

void rxProcessFrame(struct port * port)
{
	u16 tlv_cnt = 0;
	u8  tlv_type = 0;
	u16 tlv_length = 0;
	u16 tlv_offset = 0;
	u16 *tlv_head_ptr = NULL;
	u8  frame_error = 0;
	u8  oui[DCB_OUI_LEN] = INIT_DCB_OUI;
	boolean_t msap_compare_1 = FALSE;
	boolean_t msap_compare_2 = FALSE;
	boolean_t good_neighbor  = FALSE;
	boolean_t tlv_stored     = FALSE;

	assert(port->rx.framein && port->rx.sizein);
	port->lldpdu = 0;
	port->dcbdu = 0;
	port->rx.dupTlvs = 0;

	port->rx.dcbx_st = 0;
	port->rx.manifest = (rxmanifest *)malloc(sizeof(rxmanifest));
	if (port->rx.manifest == NULL) {
		printf("ERROR - could not allocate memory for receive "
			"manifest\n");
		return;
	}
	memset(port->rx.manifest,0, sizeof(rxmanifest));

	tlv_offset = sizeof(struct l2_ethhdr);  /* Points to 1st TLV */

	do {
		tlv_cnt++;
		if (tlv_offset > port->rx.sizein) {
			printf("ERROR: Frame overrun!\n");
			frame_error++;
			goto out;
		}

		tlv_head_ptr = (u16 *)&port->rx.framein[tlv_offset];
		tlv_length = htons(*tlv_head_ptr) & 0x01FF;
		tlv_type = (u8)(htons(*tlv_head_ptr) >> 9);

		if (tlv_cnt <= 3) {
			if (tlv_cnt != tlv_type) {
				printf("ERROR:TLV missing or TLVs out "
					"of order!\n");
				frame_error++;
				goto out;
			}
		}

		if (tlv_cnt > 3) {
			if ((tlv_type == 1) || (tlv_type == 2) ||
				(tlv_type == 3)) {
				printf("ERROR: Extra Type 1 Type2, or "
					"Type 3 TLV!\n");
				frame_error++;
				goto out;
			}
		}

		if ((tlv_type == TIME_TO_LIVE_TLV) && (tlv_length != 2)) {
			printf("ERROR:TTL TLV validation error! \n");
			frame_error++;
			goto out;
		}

		u16 tmp_offset = tlv_offset + tlv_length;
		if (tmp_offset > port->rx.sizein) {
			printf("ERROR: Frame overflow error: offset=%d, "
				"rx.size=%d \n", tmp_offset, port->rx.sizein);
			frame_error++;
			goto out;
		}

		u8 *info = (u8 *)&port->rx.framein[tlv_offset +
					sizeof(*tlv_head_ptr)];

		struct unpacked_tlv *tlv = create_tlv();

		if (!tlv) {
			printf("ERROR: Failed to malloc space for "
				"incoming TLV. \n");
			goto out;
		}
	
		if ((tlv_length == 0) && (tlv->type != TYPE_0)) {
				printf("ERROR: tlv_length == 0\n");
				free_unpkd_tlv(tlv);
				goto out;
		}
		tlv->type = tlv_type;
		tlv->length = tlv_length;
		tlv->info = (u8 *)malloc(tlv_length);
		if (tlv->info) {
			memset(tlv->info,0, tlv_length);
			memcpy(tlv->info, info, tlv_length);
		} else {
			printf("ERROR: Failed to malloc space for incoming "
				"TLV info \n");
			free_unpkd_tlv(tlv);
			goto out;
		}

		/* Validate the TLV */
		tlv_offset += sizeof(*tlv_head_ptr) + tlv_length;
		/* Get MSAP info */
		if (tlv->type == TYPE_1) { /* chassis ID */
			if (port->lldpdu & RCVD_LLDP_TLV_TYPE1) {
				printf("Received multiple Chassis ID TLVs in this "
					"LLDPDU\n");
				frame_error++;
				free_unpkd_tlv(tlv);
				goto out;
			} else {
				port->lldpdu |= RCVD_LLDP_TLV_TYPE1;
				port->rx.manifest->chassis = tlv;
				tlv_stored = TRUE;
			}

			if (port->msap.msap1 == NULL) {
				port->msap.length1 = tlv->length;
				port->msap.msap1 = (u8 *)malloc(tlv->length);
				if (!(port->msap.msap1)) {
					printf("ERROR: Failed to malloc space "
						"for msap1 \n");
					goto out;
				}
				memcpy(port->msap.msap1, tlv->info,
					tlv->length);
			} else {
				if (tlv->length == port->msap.length1) {
					if ((memcmp(tlv->info,port->msap.msap1,
						tlv->length) == 0))
						msap_compare_1 = TRUE;
				}
			}
		}
		if (tlv->type == TYPE_2) { /* port ID */
			if (port->lldpdu & RCVD_LLDP_TLV_TYPE2) {
				printf("Received multiple Port ID TLVs in this "
					"LLDPDU\n");
				frame_error++;
				free_unpkd_tlv(tlv);
				goto out;
			} else {
				port->lldpdu |= RCVD_LLDP_TLV_TYPE2;
				port->rx.manifest->portid = tlv;
				tlv_stored = TRUE;
			}

			if (port->msap.msap2 == NULL) {
				port->msap.length2 = tlv->length;
				port->msap.msap2 = (u8 *)malloc(tlv->length);
				if (!(port->msap.msap2)) {
					printf("ERROR: Failed to malloc space "
						"for msap2 \n");
					goto out;
				}
				memcpy(port->msap.msap2, tlv->info, tlv->length);
			} else {
				if (tlv->length == port->msap.length2) {
					if ((memcmp(tlv->info,port->msap.msap2,
						tlv->length) == 0))
						msap_compare_2 = TRUE;
				}
				if ((msap_compare_1 == TRUE) &&
					(msap_compare_2 == TRUE)) {
					msap_compare_1 = FALSE;
					msap_compare_2 = FALSE;
					good_neighbor = TRUE;
				} else {
					/* New neighbor */
					port->rx.tooManyNghbrs = TRUE;
					printf("** TOO_MANY_NGHBRS\n");
				}
			}
		}
		if (tlv->type == TYPE_3) { /* time to live */
			if (port->lldpdu & RCVD_LLDP_TLV_TYPE3) {
				printf("Received multiple TTL TLVs in this "
					"LLDPDU\n");
				frame_error++;
				free_unpkd_tlv(tlv);
				goto out;
			} else {
				port->lldpdu |= RCVD_LLDP_TLV_TYPE3;
				port->rx.manifest->ttl = tlv;
				tlv_stored = TRUE;
			}
			if ((port->rx.tooManyNghbrs == TRUE) &&
				(good_neighbor == FALSE)) {
				printf("** set tooManyNghbrsTimer\n");
				port->timers.tooManyNghbrsTimer =
					max(ntohs(*(u16 *)tlv->info), 
					port->timers.tooManyNghbrsTimer);
				msap_compare_1 = FALSE;
				msap_compare_2 = FALSE;
			} else {
				port->timers.rxTTL = ntohs(*(u16 *)tlv->info);
				good_neighbor = FALSE;
			}
		}
		if (tlv->type == TYPE_4) { /* port description */
			port->lldpdu |= RCVD_LLDP_TLV_TYPE4;
			port->rx.manifest->portdesc = tlv;
			tlv_stored = TRUE;
		}
		if (tlv->type == TYPE_5) { /* system name */
			port->lldpdu |= RCVD_LLDP_TLV_TYPE5;
			port->rx.manifest->sysname = tlv;
			tlv_stored = TRUE;
		}
		if (tlv->type == TYPE_6) { /* system description */
			port->lldpdu |= RCVD_LLDP_TLV_TYPE6;
			port->rx.manifest->sysdesc = tlv;
			tlv_stored = TRUE;
		}
		if (tlv->type == TYPE_7) { /* system capabilities */
			port->lldpdu |= RCVD_LLDP_TLV_TYPE7;
			port->rx.manifest->syscap = tlv;
			tlv_stored = TRUE;
		}
		if (tlv->type == TYPE_8) { /* mgmt address */
			port->lldpdu |= RCVD_LLDP_TLV_TYPE8;
			port->rx.manifest->mgmtadd = tlv;
			tlv_stored = TRUE;
		}

		if (tlv->type == TYPE_127) {
		if (tlv->length < (DCB_OUI_LEN + OUI_SUBTYPE_LEN)) {
				printf("Received invalid Type 127 TLV\n");
				frame_error++;
				free_unpkd_tlv(tlv);
				goto out;
			}
			if ((memcmp(tlv->info, &oui, DCB_OUI_LEN) != 0)) {
				assert(port->tlvs.cur_peer == NULL);
				/* not a DCBX TLV */
				tlv = free_unpkd_tlv(tlv);
				tlv_stored = TRUE;
				continue;
			}

			if (port->dcbx_st == dcbx_subtype2) {
				if ((tlv->info[DCB_OUI_LEN] == dcbx_subtype2)
					&& (port->lldpdu & RCVD_LLDP_DCBX2_TLV)){
					printf("Received duplicate DCBX TLVs in "
						"this LLDPDU\n");
					frame_error++;
					free_unpkd_tlv(tlv);
					goto out;
				}
			}
			if ((tlv->info[DCB_OUI_LEN] == dcbx_subtype1)
				&& (port->lldpdu & RCVD_LLDP_DCBX1_TLV)) {
				printf("Received duplicate DCBX TLVs in "
					"this LLDPDU\n");
				frame_error++;
				free_unpkd_tlv(tlv);
				goto out;
			}

			if ((port->dcbx_st == dcbx_subtype2) &&
				(tlv->info[DCB_OUI_LEN] == dcbx_subtype2)) {
				port->lldpdu |= RCVD_LLDP_DCBX2_TLV;
				port->rx.manifest->dcbx2 = tlv;
			} else if (tlv->info[DCB_OUI_LEN] == dcbx_subtype1) {
				port->lldpdu |= RCVD_LLDP_DCBX1_TLV;
				port->rx.manifest->dcbx1 = tlv;
			} else {
				/* not a DCBX subtype we support */
				tlv = free_unpkd_tlv(tlv);
				tlv_stored = TRUE;
				continue;
			}

			tlv_stored = TRUE;
		}
		if (tlv->type == TYPE_0) {
			free_unpkd_tlv(tlv);
			tlv_stored = TRUE;
			/*  unpack highest dcbx subtype first, to allow lower 
			 *  subtype to store tlvs if higher was not present
			 */
			if ((port->dcbx_st == dcbx_subtype2) &&
				 port->rx.manifest->dcbx2) {
				load_peer_tlvs(port, port->rx.manifest->dcbx2, 
					CURRENT_PEER);
				if (unpack_dcbx2_tlvs(port,
					port->rx.manifest->dcbx2) == FALSE) {
					printf("Error unpacking the DCBX2"
						"TLVs - Discarding LLDPDU\n");
					frame_error++;
					goto out;
				}
			} 
			if (port->rx.manifest->dcbx1) {
				if (!port->rx.manifest->dcbx2)
					load_peer_tlvs(port,
						port->rx.manifest->dcbx1, 
						CURRENT_PEER);
				if (unpack_dcbx1_tlvs(port,
					port->rx.manifest->dcbx1) == FALSE) {
					printf("Error unpacking the DCBX1"
						"TLVs - Discarding LLDPDU\n");
					frame_error++;
					goto out;
				}
			}
			if (port->tlvs.cur_peer) {
				process_dcbx_tlv(port, port->tlvs.cur_peer);
			}
		}

		if (!tlv_stored) {
			printf("\nrxProcessFrame: allocated TLV "
				   " was not stored! (%p)\n", tlv);
			assert(tlv_stored);
			tlv = free_unpkd_tlv(tlv);
		}
		tlv = NULL;
	} while(tlv_type != 0);

out:
	if (frame_error) {
		/* discard the frame because of errors. */
		port->stats.statsFramesDiscardedTotal++;
		port->stats.statsFramesInErrorsTotal++;
		port->rx.badFrame = TRUE;
		/* return; */
	}
	/* Maybe want to zero out the frame buffer on exit?
	 * bzero(port->rx.framein, port->rx.sizein);
	 */
	free(port->rx.framein);
	port->rx.framein = NULL;
	port->rx.sizein = 0;
	port->lldpdu = 0;
	port->dcbdu = 0;
	if (port->tlvs.cur_peer)
		port->tlvs.cur_peer =
			free_unpkd_tlv(port->tlvs.cur_peer);
	clear_manifest(port);

	return;
}

u8 mibDeleteObjects(struct port *port)
{
	control_protocol_attribs  peer_control;
	pg_attribs  peer_pg;
	pfc_attribs peer_pfc;
	app_attribs peer_app;
	llink_attribs peer_llink;
	u32 subtype = 0;
	u32 EventFlag = 0;

	/* Set any stored values for this TLV to !Present */
	if (get_peer_pg(port->ifname, &peer_pg) == dcb_success) {
		if (peer_pg.protocol.TLVPresent == TRUE) {
			peer_pg.protocol.TLVPresent = FALSE;
			put_peer_pg(port->ifname, &peer_pg);
			DCB_SET_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PG);
		}
	} else {
		return (u8)-1;
	}

	if (get_peer_pfc(port->ifname, &peer_pfc) == dcb_success) {
		if (peer_pfc.protocol.TLVPresent == TRUE) {
			peer_pfc.protocol.TLVPresent = FALSE;
			put_peer_pfc(port->ifname, &peer_pfc);
			DCB_SET_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PFC);
		}
	} else {
		return (u8)-1;
	}

	if (get_peer_app(port->ifname, subtype, &peer_app) ==
		dcb_success) {
		if (peer_app.protocol.TLVPresent == TRUE) {
			peer_app.protocol.TLVPresent = FALSE;
			peer_app.Length = 0;
			put_peer_app(port->ifname, subtype, &peer_app);
			DCB_SET_FLAGS(EventFlag, DCB_REMOTE_CHANGE_APPTLV);
		}
	} else {
		return (u8)-1;
	}

	if (get_peer_llink(port->ifname, subtype, &peer_llink) == dcb_success) {
		if (peer_llink.protocol.TLVPresent == TRUE) {
			peer_llink.protocol.TLVPresent = FALSE;
			put_peer_llink(port->ifname, subtype, &peer_llink);
			DCB_SET_FLAGS(EventFlag, DCB_REMOTE_CHANGE_LLINK);
		}
	} else {
		return (u8)-1;
	}

	if (get_peer_control(port->ifname, &peer_control) ==
		dcb_success) {
		peer_control.RxDCBTLVState = DCB_PEER_EXPIRED;
		put_peer_control(port->ifname, &peer_control);
	} else {
		return (u8)-1;
	}

	if (EventFlag != 0) {
		/* process for all subtypes */
		run_dcb_protocol(port->ifname, EventFlag, DCB_MAX_APPTLV+1);
		EventFlag = 0;
	}

	/* Clear history */
	if (port->tlvs.last_peer) {
		port->tlvs.last_peer = free_unpkd_tlv(port->tlvs.last_peer);
	}

	port->msap.length1 = 0;
	if (port->msap.msap1) {
		free(port->msap.msap1);
		port->msap.msap1 = NULL;
	}

	port->msap.length2 = 0;
	if (port->msap.msap2) {
		free(port->msap.msap2);
		port->msap.msap2 = NULL;
	}
	return 0;
}

/* for the specified remote feature, if the feature is not present in the
 * EventFlag parameter (indicating it was not received in the DCB TLV), then
 * check and update the peer data store object for the feature if it is
 * currently marked as being present.
 *
 * returns TRUE if the feature is not present now
 *              the peer data store feature object is set to 'not present'
 *         FALSE otherwise.
*/
static u32 check_feature_not_present(char *device_name, u32 subtype,
				u32 EventFlag, u32 feature)
{
	pg_attribs   peer_pg;
	pfc_attribs  peer_pfc;
	app_attribs  peer_app;
	llink_attribs  peer_llink;

	/* if (!DCB_TEST_FLAGS(EventFlag, feature, feature)) { */
	if (DCB_TEST_FLAGS(EventFlag, feature, feature))
		return FALSE;

	switch (feature) {
	case DCB_REMOTE_CHANGE_PG:
		if ((get_peer_pg(device_name, &peer_pg) == dcb_success)
			&& (peer_pg.protocol.TLVPresent == TRUE)) {
			peer_pg.protocol.TLVPresent = FALSE;
			put_peer_pg(device_name, &peer_pg);
		}
		break;
	case DCB_REMOTE_CHANGE_PFC:
		if ((get_peer_pfc(device_name, &peer_pfc) == dcb_success)
			 && (peer_pfc.protocol.TLVPresent == TRUE)) {
			peer_pfc.protocol.TLVPresent = FALSE;
			put_peer_pfc(device_name, &peer_pfc);
		}
		break;
	case DCB_REMOTE_CHANGE_APPTLV:
		if ((get_peer_app(device_name, subtype, &peer_app) ==
			dcb_success) &&
			(peer_app.protocol.TLVPresent == TRUE)) {
			peer_app.protocol.TLVPresent = FALSE;
			peer_app.Length = 0;
			put_peer_app(device_name, subtype, &peer_app);
		}
		break;
	case DCB_REMOTE_CHANGE_LLINK:
		if ((get_peer_llink(device_name, subtype, &peer_llink) ==
			dcb_success) && (peer_llink.protocol.TLVPresent ==
			TRUE)) {
			peer_llink.protocol.TLVPresent = FALSE;
			put_peer_llink(device_name, subtype, &peer_llink);
		}
		break;
	default:
		break;
	}

	return TRUE;
}


void  mibUpdateObjects(struct port *port)
{
	u32 EventFlag = 0;

	if (port->rx.manifest->dcbx_ctrl) {
		if (process_dcbx_ctrl_tlv(port) != TRUE) {
			/* Error Set error condition for all features
			 * on this port and trash DCB TLV */
		}
	} else {
		/* Error Set error condition for all features
		 * on this port and trash DCB TLV */
	}
	if (port->rx.manifest->dcbx_pg) {
		if (process_dcbx_pg_tlv(port) != TRUE) {
			 /* mark feature not present */
			if (check_feature_not_present(port->ifname, 0,
				EventFlag, DCB_REMOTE_CHANGE_PG)) {
				DCB_SET_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PG);
			}
		} else {
			DCB_SET_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PG);
		}
	} else {
		if (check_feature_not_present(port->ifname, 0,
			EventFlag, DCB_REMOTE_CHANGE_PG)) {
			DCB_SET_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PG);
		}
	}
	if (port->rx.manifest->dcbx_pfc) {
		if (process_dcbx_pfc_tlv(port) != TRUE) {
			/* mark feature not present */
			if (check_feature_not_present(port->ifname, 0,
				EventFlag, DCB_REMOTE_CHANGE_PFC)) {
				DCB_SET_FLAGS(EventFlag,DCB_REMOTE_CHANGE_PFC);
			}
		 } else {
			 DCB_SET_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PFC);
		 }
	} else {
		if (check_feature_not_present(port->ifname, 0,
			EventFlag, DCB_REMOTE_CHANGE_PFC)) {
			DCB_SET_FLAGS(EventFlag, DCB_REMOTE_CHANGE_PFC);
		}
	}
	if (port->rx.manifest->dcbx_app) {
		if (process_dcbx_app_tlv(port) != TRUE) {
			/* mark feature not present */
			if (check_feature_not_present(port->ifname, 0,
				EventFlag, DCB_REMOTE_CHANGE_APPTLV)) {
				DCB_SET_FLAGS(EventFlag,
					DCB_REMOTE_CHANGE_APPTLV);
			}
		} else {
			DCB_SET_FLAGS(EventFlag,
				DCB_REMOTE_CHANGE_APPTLV);
		}
	} else {
		if (check_feature_not_present(port->ifname, 0,
			EventFlag, DCB_REMOTE_CHANGE_APPTLV)) {
			DCB_SET_FLAGS(EventFlag, DCB_REMOTE_CHANGE_APPTLV);
		}
	}
	if (port->rx.manifest->dcbx_llink) {
		if (process_dcbx_llink_tlv(port) != TRUE) {
			/* mark feature not present */
			if (check_feature_not_present(port->ifname, 0,
				EventFlag, DCB_REMOTE_CHANGE_LLINK)) {
				DCB_SET_FLAGS(EventFlag, 
					DCB_REMOTE_CHANGE_LLINK);
			}
		} else {
			 DCB_SET_FLAGS(EventFlag, DCB_REMOTE_CHANGE_LLINK);
		}
	} else {
		if (check_feature_not_present(port->ifname, 0,
			EventFlag, DCB_REMOTE_CHANGE_LLINK)) {
			DCB_SET_FLAGS(EventFlag, DCB_REMOTE_CHANGE_LLINK);
		}
	}

	/* Run the feature & control protocol for all features and subtypes */
	run_dcb_protocol(port->ifname, EventFlag, DCB_MAX_APPTLV+1);
	EventFlag = 0;
	port->rxChanges = TRUE;
	return;
}

void run_rx_sm(struct port *port, int update_timers)
{
	set_rx_state(port);
	do {
		switch(port->rx.state) {
		case LLDP_WAIT_PORT_OPERATIONAL:
			process_wait_port_operational(port);
			break;
		case DELETE_AGED_INFO:
			process_delete_aged_info(port);
			break;
		case RX_LLDP_INITIALIZE:
			process_rx_lldp_initialize(port);
			break;
		case RX_WAIT_FOR_FRAME:
			process_wait_for_frame(port);
			break;
		case RX_FRAME:
			process_rx_frame(port);
			break;
		case DELETE_INFO:
			process_delete_info(port);
			break;
		case UPDATE_INFO:
			process_update_info(port);
			break;
		default:
			printf("ERROR: The RX State Machine is broken!\n");
			log_message(MSG_ERR_TX_SM_INVALID, "%s", port->ifname);
		}
	} while (set_rx_state(port) == TRUE);

	if (update_timers == TRUE)
		update_rx_timers(port);
}

boolean_t set_rx_state(struct port *port)
{
	if ((port->rx.rxInfoAge == FALSE) && (port->portEnabled == FALSE)) {
		printf("set_rx_state: portEnabled==FALSE \n");
		rx_change_state(port, LLDP_WAIT_PORT_OPERATIONAL);
	}

	switch(port->rx.state) {
	case LLDP_WAIT_PORT_OPERATIONAL:
		if (port->rx.rxInfoAge == TRUE) {
			rx_change_state(port, DELETE_AGED_INFO);
			return TRUE;
		}
		else if (port->portEnabled == TRUE) {
			rx_change_state(port, RX_LLDP_INITIALIZE);
			return TRUE;
		}
		return FALSE;
	case DELETE_AGED_INFO:
		rx_change_state(port, LLDP_WAIT_PORT_OPERATIONAL);
		return TRUE;
	case RX_LLDP_INITIALIZE:
		if ((port->adminStatus == enabledRxTx) ||
			(port->adminStatus == enabledRxOnly)) {
			rx_change_state(port, RX_WAIT_FOR_FRAME);
			return TRUE;
		}
		return FALSE;
	case RX_WAIT_FOR_FRAME:
		if (port->rx.rxInfoAge == TRUE) {
			rx_change_state(port, DELETE_INFO);
			return TRUE;
		} else if (port->rx.rcvFrame == TRUE) {
			rx_change_state(port, RX_FRAME);
			return TRUE;
		}
		return FALSE;
	case DELETE_INFO:
		rx_change_state(port, RX_WAIT_FOR_FRAME);
		return TRUE;
	case RX_FRAME:
		if (port->timers.rxTTL == 0) {
			rx_change_state(port, DELETE_INFO);
			return TRUE;
		} else if ((port->timers.rxTTL != 0) &&
			(port->rxChanges == TRUE)) {
			rx_change_state(port, UPDATE_INFO);
			return TRUE;
		}
		rx_change_state(port, RX_WAIT_FOR_FRAME);
		return TRUE;
	case UPDATE_INFO:
		rx_change_state(port, RX_WAIT_FOR_FRAME);
		return TRUE;
	default:
		printf("ERROR: The RX State Machine is broken!\n");
		log_message(MSG_ERR_RX_SM_INVALID, "%s", port->ifname);
		return FALSE;
	}
}

void process_wait_port_operational(struct port *port)
{
	/* Maybe could sleep for a sec. ? */
	return;
}

void process_delete_aged_info(struct port *port)
{
	port->rx.remoteChange = FALSE;
	mibDeleteObjects(port);
	port->rx.rxInfoAge = FALSE;
	port->rx.remoteChange = TRUE;
	return;
}

void process_rx_lldp_initialize(struct port *port)
{
	rxInitializeLLDP(port);
	port->rx.rcvFrame = FALSE;
	return;
}

void process_wait_for_frame(struct port *port)
{
	port->rx.badFrame  = FALSE;
	port->rx.rxInfoAge = FALSE;
	port->rx.remoteChange = FALSE;
	return;
}

void process_rx_frame(struct port *port)
{
	port->rxChanges = FALSE;
	port->rx.rcvFrame = FALSE;
	rxProcessFrame(port);
	return;
}

void process_delete_info(struct port *port)
{
	mibDeleteObjects(port);
	port->rx.remoteChange = TRUE;
	return;
}

void process_update_info(struct port *port)
{
	port->rx.remoteChange = TRUE;
	return;
}

void update_rx_timers(struct port *port)
{

	if (port->timers.rxTTL) {
		port->timers.rxTTL--;
		if (port->timers.rxTTL == 0) {
			port->rx.rxInfoAge = TRUE;
			if (port->timers.tooManyNghbrsTimer != 0) {
				printf("** clear tooManyNghbrsTimer\n");
				port->timers.tooManyNghbrsTimer = 0;
				port->rx.tooManyNghbrs = FALSE;
			}
		}
	}
	if (port->timers.tooManyNghbrsTimer) {
		port->timers.tooManyNghbrsTimer--;
		if (port->timers.tooManyNghbrsTimer == 0) {
			printf("** tooManyNghbrsTimer timeout\n");
			port->rx.tooManyNghbrs = FALSE;
		}
	}
}

void rx_change_state(struct port *port, u8 newstate) {
	switch(newstate) {
		case LLDP_WAIT_PORT_OPERATIONAL:
			break;
		case RX_LLDP_INITIALIZE:
			assert(port->rx.state ==
				LLDP_WAIT_PORT_OPERATIONAL);
			break;
		case DELETE_AGED_INFO:
			assert(port->rx.state ==
				LLDP_WAIT_PORT_OPERATIONAL);
			break;
		case RX_WAIT_FOR_FRAME:
			if (!(port->rx.state == RX_LLDP_INITIALIZE ||
				port->rx.state == DELETE_INFO ||
				port->rx.state == UPDATE_INFO ||
				port->rx.state == RX_FRAME)) {
				assert(port->rx.state !=
					RX_LLDP_INITIALIZE);
				assert(port->rx.state != DELETE_INFO);
				assert(port->rx.state != UPDATE_INFO);
				assert(port->rx.state != RX_FRAME);
			}
			break;
		case RX_FRAME:
			assert(port->rx.state == RX_WAIT_FOR_FRAME);
			break;
		case DELETE_INFO:
			if (!(port->rx.state == RX_WAIT_FOR_FRAME ||
				port->rx.state == RX_FRAME)) {
				assert(port->rx.state == RX_WAIT_FOR_FRAME);
				assert(port->rx.state == RX_FRAME);
			}
			break;
		case UPDATE_INFO:
			assert(port->rx.state == RX_FRAME);
			break;
		default:
			printf("ERROR: The RX State Machine is broken!\n");
			log_message(MSG_ERR_RX_SM_INVALID, "%s", port->ifname);
	}
	port->rx.state = newstate;
}


void load_peer_tlvs(struct port *port,struct unpacked_tlv *tlv,int type)
{
	struct unpacked_tlv *tmp = NULL;

	tmp = create_tlv();
	if (tmp) {
		tmp->type = tlv->type;
		tmp->length = tlv->length;
		tmp->info = (u8 *)malloc(tlv->length);
		if (tmp->info) {
			memset(tmp->info,0, tlv->length);
			memcpy(tmp->info,tlv->info,tlv->length);
		} else {
			printf("load_peer_tlvs: Failed to malloc info\n");
		}

		if (type == CURRENT_PEER) {
			assert(port->tlvs.cur_peer == NULL);
			port->tlvs.cur_peer = tmp;
		}
		else
		if (type == LAST_PEER) {
			assert(port->tlvs.last_peer == NULL);
			port->tlvs.last_peer = tmp;
		}
	}
	return;
}

boolean_t  unpack_dcbx1_tlvs(struct port *port, struct unpacked_tlv *tlv)
{
	/* unpack the tlvs and store in manifest */
	u8 *offset = NULL;   /* iterator */
	u16 current = 0, tl = 0;
	u16 end = 0;         /* End of data blob */
	struct unpacked_tlv     *dcbtlv;

	/* store highest dcbx subtype received */
	if (port->rx.dcbx_st < tlv->info[DCB_OUI_LEN]) {
		port->rx.dcbx_st = tlv->info[DCB_OUI_LEN];
	}
	/* OUI + subtype sizes equal the start of data blob */
	offset = (u8  *)&tlv->info[OUI_SUBTYPE_LEN + DCB_OUI_LEN];
	end = tlv->length  - (OUI_SUBTYPE_LEN + DCB_OUI_LEN);

	/* Process */
	do {
		dcbtlv = create_tlv();
		if (!dcbtlv) {
			printf("ERROR: Failed to malloc space for incoming "
				"DCB TLV. \n");
			return FALSE;
		}
		memcpy(&tl, offset, sizeof(tl));
		offset += sizeof(tl);
		dcbtlv->length = ntohs(tl) & 0x01FF;
		if (dcbtlv->length==0) {
			printf("ERROR: dcbtlv->length==0 \n");
			free_unpkd_tlv(dcbtlv);
			return FALSE;
		}
		dcbtlv->type   = (u8)(ntohs(tl) >> 9);
		dcbtlv->info = (u8 *)malloc(dcbtlv->length);
		if (dcbtlv->info) {
			memset(dcbtlv->info, 0, dcbtlv->length);
			memcpy(dcbtlv->info, offset, dcbtlv->length);
		} else {
			printf("ERROR: Failed to malloc space for incoming "
				"TLV info \n");
			free_unpkd_tlv(dcbtlv);
			return FALSE;
		}
		current += dcbtlv->length + sizeof(tl);
		offset += dcbtlv->length;
		switch(dcbtlv->type) {
		case DCB_CONTROL_TLV:
			if (!(port->dcbdu & RCVD_DCBX1_TLV_CTRL)) {
				port->dcbdu |= RCVD_DCBX1_TLV_CTRL;
				port->rx.manifest->dcbx_ctrl = dcbtlv;
			} else {
				printf("** ERROR: DUP Ctrl TLV1 \n");
				port->rx.dupTlvs |= DUP_DCBX_TLV_CTRL;
				free_unpkd_tlv(dcbtlv);
			}
			break;
		case DCB_PRIORITY_GROUPS_TLV:
			/* store if subtype 2 is not present */
			if (port->rx.dcbx_st == dcbx_subtype1) {
				if (port->rx.manifest->dcbx_pg == NULL) {
					port->dcbdu |= RCVD_DCBX_TLV_PG;
					port->rx.manifest->dcbx_pg = dcbtlv;
				} else {
					printf("** ERROR: DUP PG TLV1 \n");
					port->rx.dupTlvs |= DUP_DCBX_TLV_PG;
					free_unpkd_tlv(dcbtlv);
				}
			} else {
				free_unpkd_tlv(dcbtlv);
			}
			break;
		case DCB_PRIORITY_FLOW_CONTROL_TLV:
			/* store if subtype 2 is not present */
			if (port->rx.dcbx_st == dcbx_subtype1) {
				if (port->rx.manifest->dcbx_pfc == NULL) {
					port->dcbdu |= RCVD_DCBX_TLV_PFC;
					port->rx.manifest->dcbx_pfc = dcbtlv;
				} else {
					printf("** ERROR: DUP PFC TLV1 \n");
					port->rx.dupTlvs |= DUP_DCBX_TLV_PFC;
					free_unpkd_tlv(dcbtlv);
				}
			} else {
				free_unpkd_tlv(dcbtlv);
			}
			break;
		case DCB_APPLICATION_TLV:
			/* store if subtype 2 is not present */
			if ((port->rx.dcbx_st == dcbx_subtype1) &&
				(dcbtlv->info[DCBX_HDR_SUB_TYPE_OFFSET]
					== APP_FCOE_STYPE)) {
				if (port->rx.manifest->dcbx_app == NULL) {
					port->dcbdu |= RCVD_DCBX_TLV_APP;
					port->rx.manifest->dcbx_app = dcbtlv;
				} else {
					printf("** ERROR: DUP APP TLV1 \n");
					port->rx.dupTlvs |= DUP_DCBX_TLV_APP;
					free_unpkd_tlv(dcbtlv);
				}
			} else {
				free_unpkd_tlv(dcbtlv);
			}
		break;
		case DCB_LLINK_TLV:
			if (dcbtlv->info[DCBX_HDR_SUB_TYPE_OFFSET]
					== LLINK_FCOE_STYPE) {
				if (port->rx.manifest->dcbx_llink == NULL) {
					port->dcbdu |= RCVD_DCBX_TLV_LLINK;
					port->rx.manifest->dcbx_llink = dcbtlv;
				} else {
					printf("** ERROR: DUP LLINK TLV1 \n");
					port->rx.dupTlvs |= DUP_DCBX_TLV_LLINK;
					free_unpkd_tlv(dcbtlv);
				}
			} else {
				free_unpkd_tlv(dcbtlv);
			}
		break;
		default:
			free_unpkd_tlv(dcbtlv);
		break;
		}
		dcbtlv = NULL;
	} while(current < end);

	return TRUE;
}

boolean_t  unpack_dcbx2_tlvs(struct port *port, struct unpacked_tlv *tlv)
{
	/* unpack the tlvs and store in manifest */
	u8 *offset = NULL;   /* iterator */
	u16 current = 0, tl = 0;
	u16 end = 0;         /* End of data blob */
	struct unpacked_tlv     *dcbtlv;

	/* store highest dcbx subtype received */
	if (port->rx.dcbx_st < tlv->info[DCB_OUI_LEN]) {
		port->rx.dcbx_st = tlv->info[DCB_OUI_LEN];
	}
	/* OUI + subtype sizes equal the start of data blob */
	offset = (u8  *)&tlv->info[OUI_SUBTYPE_LEN + DCB_OUI_LEN];
	end = tlv->length  - (OUI_SUBTYPE_LEN + DCB_OUI_LEN);

	/* Process */
	do {
		dcbtlv = create_tlv();
		if (!dcbtlv) {
			printf("ERROR: Failed to malloc space for incoming "
				"DCB TLV. \n");
			return FALSE;
		}
		memcpy(&tl, offset, sizeof(tl));
		offset += sizeof(tl);
		dcbtlv->length = ntohs(tl) & 0x01FF;
		if (dcbtlv->length==0) {
			printf("ERROR: dcbtlv->length==0 \n");
			free_unpkd_tlv(dcbtlv);
			return FALSE;
		}
		dcbtlv->type   = (u8)(ntohs(tl) >> 9);
		dcbtlv->info = (u8 *)malloc(dcbtlv->length);
		if (dcbtlv->info) {
			memset(dcbtlv->info, 0, dcbtlv->length);
			memcpy(dcbtlv->info, offset, dcbtlv->length);
		} else {
			printf("ERROR: Failed to malloc space for incoming "
				"TLV info \n");
			free_unpkd_tlv(dcbtlv);
			return FALSE;
		}
		current += dcbtlv->length + sizeof(tl);
		offset += dcbtlv->length;
		switch(dcbtlv->type) {
		case DCB_CONTROL_TLV:
			if (port->rx.manifest->dcbx1 == NULL) {
				if (port->rx.manifest->dcbx_ctrl == NULL) {
					port->dcbdu |= RCVD_DCBX2_TLV_CTRL;
					port->rx.manifest->dcbx_ctrl = dcbtlv;
				} else if (port->dcbdu & RCVD_DCBX2_TLV_CTRL) {
					printf("** ERROR: DUP CTRL TLV2 \n");
					port->rx.dupTlvs |= DUP_DCBX_TLV_CTRL;
					free_unpkd_tlv(dcbtlv);
				}
			} else {
				free_unpkd_tlv(dcbtlv);
			}
			break;
		case DCB_PRIORITY_GROUPS_TLV2:
			if (port->rx.manifest->dcbx_pg == NULL) {
				port->dcbdu |= RCVD_DCBX_TLV_PG;
				port->rx.manifest->dcbx_pg = dcbtlv;
			} else {
				printf("** ERROR: DUP PG TLV2 \n");
				port->rx.dupTlvs |= DUP_DCBX_TLV_PG;
				free_unpkd_tlv(dcbtlv);
			}
			break;
		case DCB_PRIORITY_FLOW_CONTROL_TLV2:
			if (port->rx.manifest->dcbx_pfc == NULL) {
				port->dcbdu |= RCVD_DCBX_TLV_PFC;
				port->rx.manifest->dcbx_pfc = dcbtlv;
			} else {
				printf("** ERROR: DUP PFC TLV2 \n");
				port->rx.dupTlvs |= DUP_DCBX_TLV_PFC;
				free_unpkd_tlv(dcbtlv);
			}
			break;
		case DCB_APPLICATION_TLV2:
			if (dcbtlv->info[DCBX_HDR_SUB_TYPE_OFFSET]
					== APP_FCOE_STYPE) {
				if (port->rx.manifest->dcbx_app == NULL) {
					port->dcbdu |= RCVD_DCBX_TLV_APP;
					port->rx.manifest->dcbx_app = dcbtlv;
				} else {
					printf("** ERROR: DUP APP TLV2 \n");
					port->rx.dupTlvs |= DUP_DCBX_TLV_APP;
					free_unpkd_tlv(dcbtlv);
				}
			} else {
				free_unpkd_tlv(dcbtlv);
			}
			break;
		default:
			free_unpkd_tlv(dcbtlv);
			break;
		}
		dcbtlv = NULL;
	} while(current < end);

	return TRUE;
}

void process_dcbx_tlv(struct port *port,struct unpacked_tlv *tlv)
{
	if (port->rx.tooManyNghbrs){
		if(NULL != port->tlvs.last_peer) {
			if(NULL != port->tlvs.last_peer->info)
				free(port->tlvs.last_peer->info);
			free(port->tlvs.last_peer);
			port->tlvs.last_peer = NULL;
		}
		mibUpdateObjects(port);
	} else if (port->tlvs.last_peer == NULL) {
		/* First tlv. Create and load */
		load_peer_tlvs(port,tlv, LAST_PEER);
		mibUpdateObjects(port); /* Put 1st in peer mem */
	} else {
		if (tlv->length == port->tlvs.last_peer->length) {
			if (memcmp(tlv->info, port->tlvs.last_peer->info,
				tlv->length) != 0) {
				/* Strings are different unpack the tlvs and
				 * store in peer mem
				 * Load tlv in last_peer
				*/
				mibUpdateObjects(port);
				port->tlvs.last_peer->type = tlv->type;
				port->tlvs.last_peer->length = tlv->length;
				if (port->tlvs.last_peer->info) {
					memcpy(port->tlvs.last_peer->info,
					 tlv->info,tlv->length);
				}
			}
		} else {
			/* Lengths are different unpack the tlvs and store in
			 * peer memory. Load tlv in last_peer
			*/
			if(NULL != port->tlvs.last_peer) {
				if(NULL != port->tlvs.last_peer->info)
					free(port->tlvs.last_peer->info);
				free(port->tlvs.last_peer);
				port->tlvs.last_peer = NULL;
			}
			load_peer_tlvs(port,tlv, LAST_PEER);
			mibUpdateObjects(port);
		}
	}
	port->tlvs.cur_peer = free_unpkd_tlv(tlv);
	return;
}

boolean_t process_dcbx_ctrl_tlv(struct port *port)
{
	control_protocol_attribs  peer_control;

	if (port->rx.manifest->dcbx_ctrl->length != DCBX_CTRL_LEN) {
		printf("process_dcbx_ctrl_tlv: ERROR - len\n");
		return(FALSE);
	}

	memset(&peer_control, 0, sizeof(control_protocol_attribs));
	peer_control.Oper_version =	port->rx.manifest->dcbx_ctrl->info
		[DCBX_CTRL_OPER_VER_OFFSET];
	peer_control.Max_version = port->rx.manifest->dcbx_ctrl->info
		[DCBX_CTRL_MAX_VER_OFFSET];

	u32 tmp32 = 0;
	memcpy(&tmp32, &port->rx.manifest->dcbx_ctrl->info
		[DCBX_CTRL_SEQNO_OFFSET], sizeof(u32));
	peer_control.SeqNo = ntohl(tmp32);
	tmp32 = 0;
	memcpy(&tmp32,&port->rx.manifest->dcbx_ctrl->info
		[DCBX_CTRL_ACKNO_OFFSET], sizeof(u32));
	peer_control.AckNo = ntohl(tmp32);
	printf("*** Received a DCB_CONTROL_TLV: -- SeqNo=%d, AckNo=%d \n",
		peer_control.SeqNo, peer_control.AckNo);
	peer_control.RxDCBTLVState = DCB_PEER_PRESENT;

	if (port->rx.dupTlvs & DUP_DCBX_TLV_CTRL) {
		printf("** STORE: DUP CTRL TLV \n");
		peer_control.Error_Flag |= DUP_DCBX_TLV_CTRL;
	}
	if (port->rx.tooManyNghbrs) {
		printf("** STORE: TOO_MANY_NGHBRS\n");
		peer_control.Error_Flag |= TOO_MANY_NGHBRS;
	}

	put_peer_control(port->ifname, &peer_control);

	return(TRUE);
}

boolean_t 	process_dcbx_pg_tlv(struct port *port)
{
	pg_attribs   peer_pg;
	int i = 0;
	int j, k;
	u8 used[MAX_BANDWIDTH_GROUPS];

	if (port->rx.dcbx_st == dcbx_subtype2) {
		if (port->rx.manifest->dcbx_pg->length != DCBX2_PG_LEN) {
			printf("process_dcbx2_pg_tlv: ERROR - len\n");
			return(FALSE);
		}
	} else {
		if (port->rx.manifest->dcbx_pg->length != DCBX1_PG_LEN) {
			printf("process_dcbx1_pg_tlv: ERROR - len\n");
			return(FALSE);
		}
	}

	memset(&peer_pg, 0, sizeof(pg_attribs));
	peer_pg.protocol.Advertise = TRUE;
	peer_pg.protocol.Oper_version =	port->rx.manifest->dcbx_pg->info
		[DCBX_HDR_OPER_VERSION_OFFSET];
	peer_pg.protocol.Max_version = port->rx.manifest->dcbx_pg->info
		[DCBX_HDR_MAX_VERSION_OFFSET];
	if (port->rx.manifest->dcbx_pg->info[DCBX_HDR_EWE_OFFSET] & BIT7) {
		peer_pg.protocol.Enable = TRUE;
	} else {
		peer_pg.protocol.Enable = FALSE;
	}
	if (port->rx.manifest->dcbx_pg->info[DCBX_HDR_EWE_OFFSET] & BIT6) {
		peer_pg.protocol.Willing = TRUE;
	} else {
		peer_pg.protocol.Willing = FALSE;
	}
	if (port->rx.manifest->dcbx_pg->info[DCBX_HDR_EWE_OFFSET] & BIT5) {
		peer_pg.protocol.Error = TRUE;
	} else {
		peer_pg.protocol.Error = FALSE;
	}
	peer_pg.protocol.dcbx_st = port->rx.dcbx_st;
	peer_pg.protocol.TLVPresent = TRUE;

	if (port->rx.dupTlvs & DUP_DCBX_TLV_CTRL) {
		printf("** STORE: DUP CTRL TLV \n");
		peer_pg.protocol.Error_Flag |= DUP_DCBX_TLV_CTRL;
	}
	if (port->rx.dupTlvs & DUP_DCBX_TLV_PG) {
		printf("** STORE: DUP PG TLV \n");
		peer_pg.protocol.Error_Flag |= DUP_DCBX_TLV_PG;
	}

	if (port->rx.dcbx_st == dcbx_subtype2) {
		memset(used, FALSE, sizeof(used));
		for (j=0,k=0 ; k < MAX_BANDWIDTH_GROUPS; j++, k=k+2) {
			u8 tmpbyte = port->rx.manifest->dcbx_pg->info
				[DCBX2_PG_PGID_UP+j];
			peer_pg.tx.up[k+1].pgid = tmpbyte & 0xf;
			peer_pg.rx.up[k+1].pgid = tmpbyte & 0xf;
			peer_pg.tx.up[k].pgid = (tmpbyte >> 4) & 0xf;
			peer_pg.rx.up[k].pgid = (tmpbyte >> 4) & 0xf;
			if (peer_pg.tx.up[k+1].pgid == LINK_STRICT_PGID) {
				peer_pg.tx.up[k+1].strict_priority = dcb_link;
				peer_pg.rx.up[k+1].strict_priority = dcb_link;
			} else {
				used[peer_pg.tx.up[k+1].pgid] = TRUE;
			}
			if (peer_pg.tx.up[k].pgid == LINK_STRICT_PGID) {
				peer_pg.tx.up[k].strict_priority = dcb_link;
				peer_pg.rx.up[k].strict_priority = dcb_link;
			} else {
				used[peer_pg.tx.up[k].pgid] = TRUE;
			}
		}
		/* assign LINK_STRICT_PGID's to an unused pgid value */
		for (j = 0; j < MAX_BANDWIDTH_GROUPS; j++)
			if (!used[j])
				break;
		for (k = 0; k < MAX_BANDWIDTH_GROUPS; k++) {
			if (peer_pg.tx.up[k].pgid == LINK_STRICT_PGID) {
				peer_pg.tx.up[k].pgid = (u8)j;
				peer_pg.rx.up[k].pgid = (u8)j;
			}
		}

		for (i = 0; i < MAX_BANDWIDTH_GROUPS; i++) {
			peer_pg.tx.pg_percent[i] =
				port->rx.manifest->dcbx_pg->info
				[DCBX2_PG_PERCENT_OFFSET + i];
			peer_pg.rx.pg_percent[i] =
				port->rx.manifest->dcbx_pg->info
				[DCBX2_PG_PERCENT_OFFSET + i];
		}
		peer_pg.num_tcs = (u8)(port->rx.manifest->dcbx_pg->info
				[DCBX2_PG_NUM_TC_OFFSET]);
	} else {
		for (i = 0; i < MAX_BANDWIDTH_GROUPS; i++) {
			peer_pg.tx.pg_percent[i] =
				port->rx.manifest->dcbx_pg->info
				[DCBX1_PG_PERCENT_OFFSET + i];
			peer_pg.rx.pg_percent[i] =
				port->rx.manifest->dcbx_pg->info
				[DCBX1_PG_PERCENT_OFFSET + i];
		}
		for (i = 0; i < MAX_USER_PRIORITIES; i++) {
			u8 tmp_bwg_id = port->rx.manifest->dcbx_pg->info
				[DCBX1_PG_SETTINGS_OFFSET + 2*i +BYTE1_OFFSET];
			tmp_bwg_id = tmp_bwg_id >> 5;
			peer_pg.tx.up[i].pgid = tmp_bwg_id;
			peer_pg.rx.up[i].pgid = tmp_bwg_id;

			u8 tmp_strict_prio =
				port->rx.manifest->dcbx_pg->info
				[DCBX1_PG_SETTINGS_OFFSET + 2*i +BYTE1_OFFSET];
			tmp_strict_prio = tmp_strict_prio >> 3;
			tmp_strict_prio &= 0x3;
			peer_pg.tx.up[i].strict_priority =
				(dcb_strict_priority_type)tmp_strict_prio;
			peer_pg.rx.up[i].strict_priority =
				(dcb_strict_priority_type)tmp_strict_prio;

			peer_pg.tx.up[i].percent_of_pg_cap =
				port->rx.manifest->dcbx_pg->info
				[DCBX1_PG_SETTINGS_OFFSET + 2*i +BYTE2_OFFSET];
			peer_pg.rx.up[i].percent_of_pg_cap =
				port->rx.manifest->dcbx_pg->info
				[DCBX1_PG_SETTINGS_OFFSET + 2*i +BYTE2_OFFSET];
			tmp_bwg_id = tmp_strict_prio = 0;

			peer_pg.tx.up[i].tcmap = 0;
			peer_pg.rx.up[i].tcmap = 0;
		}
	}
	put_peer_pg(port->ifname, &peer_pg);

	return(TRUE);
}

boolean_t process_dcbx_pfc_tlv(struct port *port)
{
	pfc_attribs  peer_pfc;
	int i = 0;
	if (port->rx.dcbx_st == dcbx_subtype2) {
		if (port->rx.manifest->dcbx_pfc->length != DCBX2_PFC_LEN) {
			printf("process_dcbx2_pfc_tlv: ERROR - len\n");
			return(FALSE);
		}
	} else {
		if (port->rx.manifest->dcbx_pfc->length != DCBX1_PFC_LEN) {
			printf("process_dcbx1_pfc_tlv: ERROR - len\n");
			return(FALSE);
		}
	}

	memset(&peer_pfc, 0, sizeof(pfc_attribs));
	peer_pfc.protocol.Advertise = TRUE;
	peer_pfc.protocol.Oper_version = port->rx.manifest->dcbx_pfc->info
		[DCBX_HDR_OPER_VERSION_OFFSET];
	peer_pfc.protocol.Max_version =	port->rx.manifest->dcbx_pfc->info
		[DCBX_HDR_MAX_VERSION_OFFSET];
	if (port->rx.manifest->dcbx_pfc->info[DCBX_HDR_EWE_OFFSET] & BIT7) {
		peer_pfc.protocol.Enable = TRUE;
	} else {
		peer_pfc.protocol.Enable = FALSE;
	}
	if (port->rx.manifest->dcbx_pfc->info[DCBX_HDR_EWE_OFFSET] & BIT6) {
		peer_pfc.protocol.Willing = TRUE;
	} else {
		peer_pfc.protocol.Willing = FALSE;
	}
	if (port->rx.manifest->dcbx_pfc->info[DCBX_HDR_EWE_OFFSET] & BIT5) {
		peer_pfc.protocol.Error = TRUE;
	} else {
		peer_pfc.protocol.Error = FALSE;
	}
	peer_pfc.protocol.dcbx_st = port->rx.dcbx_st;
	peer_pfc.protocol.TLVPresent = TRUE;

	if (port->rx.dupTlvs & DUP_DCBX_TLV_CTRL) {
		printf("** STORE: DUP CTRL TLV \n");
		peer_pfc.protocol.Error_Flag |= DUP_DCBX_TLV_CTRL;
	}
	if (port->rx.dupTlvs & DUP_DCBX_TLV_PFC) {
		printf("** STORE: DUP PFC TLV \n");
		peer_pfc.protocol.Error_Flag |= DUP_DCBX_TLV_PFC;
	}

	u8 temp = 0;
	for (i = 0; i < MAX_USER_PRIORITIES; i++) {
		temp = port->rx.manifest->dcbx_pfc->info[DCBX_PFC_MAP_OFFSET];
		peer_pfc.admin[i] = (pfc_type)((temp >> i) & BIT0);
	}
	if (port->rx.dcbx_st == dcbx_subtype2) {
		peer_pfc.num_tcs = port->rx.manifest->dcbx_pfc->info
				[DCBX2_PFC__NUM_TC_OFFSET];
	}
	put_peer_pfc(port->ifname, &peer_pfc);

	return(TRUE);
}

boolean_t process_dcbx_app_tlv(struct port *port)
{
	app_attribs peer_app;
	u32         i=0, len=0;
	u8          sub_type=0, sel_field=0, *pBuf=NULL;
	u16         peer_proto=0;
	u8          oui[DCB_OUI_LEN]=INIT_DCB_OUI;
	u8          peer_oui[DCB_OUI_LEN];

	len = port->rx.manifest->dcbx_app->length;
	if (port->rx.dcbx_st == dcbx_subtype2) {
		if (len < DCBX2_APP_LEN) {
			printf("process_dcbx2_app_tlv: ERROR - len\n");
			return(FALSE);
		}
	} else {
		if (len < DCBX1_APP_LEN) {
			printf("process_dcbx1_app_tlv: ERROR - len\n");
			return(FALSE);
		}
	}

	memset(&peer_oui, 0, DCB_OUI_LEN);
	memset(&peer_app, 0, sizeof(app_attribs));
	pBuf = port->rx.manifest->dcbx_app->info;

	peer_app.protocol.Oper_version = pBuf[DCBX_HDR_OPER_VERSION_OFFSET];
	peer_app.protocol.Max_version =	pBuf[DCBX_HDR_MAX_VERSION_OFFSET];
	if (pBuf[DCBX_HDR_EWE_OFFSET] & BIT7) {
		peer_app.protocol.Enable = TRUE;
	} else {
		peer_app.protocol.Enable = FALSE;
	}
	if (pBuf[DCBX_HDR_EWE_OFFSET] & BIT6) {
		peer_app.protocol.Willing = TRUE;
	} else {
		peer_app.protocol.Willing = FALSE;
	}
	if (pBuf[DCBX_HDR_EWE_OFFSET] & BIT5) {
		peer_app.protocol.Error = TRUE;
	} else {
		peer_app.protocol.Error = FALSE;
	}
	peer_app.protocol.dcbx_st = port->rx.dcbx_st;

	if (port->rx.dupTlvs & DUP_DCBX_TLV_CTRL) {
		printf("** STORE: DUP CTRL TLV \n");
		peer_app.protocol.Error_Flag |= DUP_DCBX_TLV_CTRL;
	}
	if (port->rx.dupTlvs & DUP_DCBX_TLV_APP) {
		printf("** STORE: DUP APP TLV \n");
		peer_app.protocol.Error_Flag |= DUP_DCBX_TLV_APP;
	}

	if (port->rx.dcbx_st == dcbx_subtype2) {
		/* processs upper layer protocol IDs until we 
		 * match Selector Field, FCoE or FIP ID and OUI */
		len -= DCBX2_APP_DATA_OFFSET;
		pBuf = &pBuf[DCBX2_APP_DATA_OFFSET];
		while (len >= DCBX2_APP_SIZE) {
			sel_field = (u8)(pBuf[DCBX2_APP_BYTE1_OFFSET]
				& PROTO_ID_SF_TYPE);
			if (sel_field != PROTO_ID_L2_ETH_TYPE) {
				sel_field = 0;
				len -= DCBX2_APP_SIZE;
				pBuf = &pBuf[DCBX2_APP_SIZE];
				continue;
			}
			peer_proto = *((u16*)(&(pBuf[DCBX2_APP_PROTO_OFFSET])));
			if ((peer_proto != PROTO_ID_FCOE) && 
				(peer_proto != PROTO_ID_FIP)) {
				sel_field = 0;
				peer_proto = 0;
				len -= DCBX2_APP_SIZE;
				pBuf = &pBuf[DCBX2_APP_SIZE];
				continue;
			}
			peer_oui[0] = (u8)(pBuf[DCBX2_APP_BYTE1_OFFSET]
				& PROTO_ID_OUI_MASK);
			peer_oui[1] = pBuf[DCBX2_APP_LOW_OUI_OFFSET1];
			peer_oui[2] = pBuf[DCBX2_APP_LOW_OUI_OFFSET2];
			if (memcmp(peer_oui, oui, DCB_OUI_LEN) != 0) {
				sel_field = 0;
				peer_proto = 0;
				memset(&peer_oui, 0, DCB_OUI_LEN);
				len -= DCBX2_APP_SIZE;
				pBuf = &pBuf[DCBX2_APP_SIZE];
				continue;
			}
			peer_app.protocol.TLVPresent = TRUE;
			peer_app.Length = APP_FCOE_STYPE_LEN;
			memcpy (&(peer_app.AppData[0]), 
				&(pBuf[DCBX2_APP_UP_MAP_OFFSET]),
				peer_app.Length);
			put_peer_app(port->ifname, sub_type, &peer_app);
			return(TRUE);
		}
	} else {
		sub_type = pBuf[DCBX_HDR_SUB_TYPE_OFFSET];
		len = port->rx.manifest->dcbx_app->length -
			sizeof(struct  dcbx_tlv_header);
		peer_app.Length = len;
		if (DCB_MAX_TLV_LENGTH < len) {
			return FALSE;
		}
		for (i = 0; i < len; i++) {
			peer_app.AppData[i] = pBuf[DCBX1_APP_DATA_OFFSET + i];
		}
		peer_app.protocol.TLVPresent = TRUE;
		put_peer_app(port->ifname, sub_type, &peer_app);
		return(TRUE);
	}
	return(FALSE);
}

boolean_t process_dcbx_llink_tlv(struct port *port)
{
	llink_attribs   peer_llk;

	if (port->rx.manifest->dcbx_llink->length != DCBX_LLINK_LEN) {
		printf("process_dcbx_llink_tlv: ERROR - len\n");
		return(FALSE);
	}

	memset(&peer_llk, 0, sizeof(llink_attribs));
	peer_llk.protocol.Advertise = TRUE;
	peer_llk.protocol.Oper_version = port->rx.manifest->dcbx_llink->info
		[DCBX_HDR_OPER_VERSION_OFFSET];
	peer_llk.protocol.Max_version = port->rx.manifest->dcbx_llink->info
		[DCBX_HDR_MAX_VERSION_OFFSET];
	if (port->rx.manifest->dcbx_llink->info[DCBX_HDR_EWE_OFFSET] & BIT7) {
		peer_llk.protocol.Enable = TRUE;
	} else {
		peer_llk.protocol.Enable = FALSE;
	}
	if (port->rx.manifest->dcbx_llink->info[DCBX_HDR_EWE_OFFSET] & BIT6) {
		peer_llk.protocol.Willing = TRUE;
	} else {
		peer_llk.protocol.Willing = FALSE;
	}
	if (port->rx.manifest->dcbx_llink->info[DCBX_HDR_EWE_OFFSET] & BIT5) {
		peer_llk.protocol.Error = TRUE;
	} else {
		peer_llk.protocol.Error = FALSE;
	}
	peer_llk.protocol.dcbx_st = port->rx.dcbx_st;
	peer_llk.protocol.TLVPresent = TRUE;

	if (port->rx.dupTlvs & DUP_DCBX_TLV_CTRL) {
		printf("** STORE: DUP CTRL TLV \n");
		peer_llk.protocol.Error_Flag |= DUP_DCBX_TLV_CTRL;
	}
	if (port->rx.dupTlvs & DUP_DCBX_TLV_LLINK) {
		printf("** STORE: DUP LLINK TLV \n");
		peer_llk.protocol.Error_Flag |= DUP_DCBX_TLV_LLINK;
	}

	peer_llk.llink.llink_status = !!((port->rx.manifest->dcbx_llink->info
				[DCBX_LLINK_STATUS_OFFSET]) & BIT7);
	put_peer_llink(port->ifname, LLINK_FCOE_STYPE, &peer_llk);

	return(TRUE);
}

void clear_manifest(struct port *port) {
	if (port->rx.manifest->dcbx_llink)
		port->rx.manifest->dcbx_llink =
			free_unpkd_tlv(port->rx.manifest->dcbx_llink);
	if (port->rx.manifest->dcbx_app)
		port->rx.manifest->dcbx_app =
			free_unpkd_tlv(port->rx.manifest->dcbx_app);
	if (port->rx.manifest->dcbx_pfc)
		port->rx.manifest->dcbx_pfc =
			free_unpkd_tlv(port->rx.manifest->dcbx_pfc);
	if (port->rx.manifest->dcbx_pg)
		port->rx.manifest->dcbx_pg =
			free_unpkd_tlv(port->rx.manifest->dcbx_pg);
	if (port->rx.manifest->dcbx_ctrl)
		port->rx.manifest->dcbx_ctrl =
			free_unpkd_tlv(port->rx.manifest->dcbx_ctrl);
	if (port->rx.manifest->dcbx1)
		port->rx.manifest->dcbx1 =
			free_unpkd_tlv(port->rx.manifest->dcbx1);
	if (port->rx.manifest->dcbx2)
		port->rx.manifest->dcbx2 =
			free_unpkd_tlv(port->rx.manifest->dcbx2);
	if (port->rx.manifest->mgmtadd)
		port->rx.manifest->mgmtadd =
			free_unpkd_tlv(port->rx.manifest->mgmtadd);
	if (port->rx.manifest->syscap)
		port->rx.manifest->syscap =
			free_unpkd_tlv(port->rx.manifest->syscap);
	if (port->rx.manifest->sysdesc)
		port->rx.manifest->sysdesc =
			free_unpkd_tlv(port->rx.manifest->sysdesc);
	if (port->rx.manifest->sysname)
		port->rx.manifest->sysname =
			free_unpkd_tlv(port->rx.manifest->sysname);
	if (port->rx.manifest->portdesc)
		port->rx.manifest->portdesc =
			free_unpkd_tlv(port->rx.manifest->portdesc);
	if (port->rx.manifest->ttl)
		port->rx.manifest->ttl =
			free_unpkd_tlv(port->rx.manifest->ttl);
	if (port->rx.manifest->portid)
		port->rx.manifest->portid =
			free_unpkd_tlv(port->rx.manifest->portid);
	if (port->rx.manifest->chassis)
		port->rx.manifest->chassis =
			free_unpkd_tlv(port->rx.manifest->chassis);
	free(port->rx.manifest);
	port->rx.manifest = NULL;
}
