/*
  DB Mixer
  ========
  Description: 
    interface controller for an external mixer to work with the DBMix system.

	Copyright (c) 1999, 2000 Robert Michael S Dean

    DBMix Author: Robert Michael S Dean 
    exmixer Author: Simon Mark Werner 
	Version: 1.1


   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

 */


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <limits.h>
#include <fcntl.h>
#include <signal.h>
#include <gtk/gtk.h>
#include <sys/shm.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <xmms/plugin.h>
#include <dbchannel.h>
#include <dbaudiolib.h>
#include <dbdebug.h>
#include <dbsoundcard.h>
#include <pthread.h>

#include "dbmixer.h"

#ifdef EXT_MIXER 
  #include "exmixer.h" 
  #include "save_prefs.h" 
  extern int exmixer_enabled; 
#endif

gint mixer_flag = FALSE;
gint monitor_loop_flag = TRUE;

int standalone_flag = 0;

extern gint fade_time_base;

/******************************************************************/
/**                  Channel access functions                    **/
/******************************************************************/

void update_button_clicked(GtkWidget * w, gpointer data);
void reset_button_clicked (GtkWidget * w, gpointer data);
void record_session_clicked(GtkWidget * w, gpointer data);

void dbmixer_about(void);
void preferences_dialog(void);

int shmid, sysshmid;
dbfsd_data * sysdata;
local_channel * local_channels;
int * channel_indexes;
channel_widgets * widgets;

GtkWidget * window;
GtkWidget * menu;
GtkWidget * file_selector;
GtkLabel  * record_label;
gint        recording_flag;

extern int debug_level;



/* this function borrowed from GTK API reference on GtkDialog */
void db_message_box(gchar *message) 
{
   GtkWidget *dialog, *label, *okay_button;
   
   /* create the widgets */
   
   dialog = gtk_dialog_new();
   label = gtk_label_new (message);
   okay_button = gtk_button_new_with_label("OK");
   
   /* ensure that the dialog box is destroyed when the user clicks ok. */
   
   gtk_signal_connect_object (GTK_OBJECT (okay_button), "clicked",
                              GTK_SIGNAL_FUNC (gtk_widget_destroy), (gpointer)dialog);
   gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
                      okay_button);

   /* add the label, and show everything we've added to the dialog. */

   gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox),
                      label);
   gtk_widget_show_all (dialog);
}

void send_msg(local_channel * lch,long int  msg_type, float data)
{
	dbfsd_msg msg;

	msg.msg_type = msg_type;
	msg.data = data;

	if(msgsnd(lch->msg_q_id, &msg,sizeof(dbfsd_msg) - sizeof(long int),IPC_NOWAIT) == 0)
	{
		Debug("dbmixer: Message sent to %s.",lch->channel_name);
	}
	else
	{
	    Error("dbmixer: Message failure sending to %s.",lch->channel_name);
	}
}


int Cue_Enabled(local_channel * ch)
{
	if((ch->channel_flags & CUE_FLAG) == CUE_FLAG)
	{
		return TRUE;
	}
	else
	{
		return FALSE;
	}
}


int Channel_Init()
{
	/* Get system data */

	sysshmid = shmget((key_t) DB_SYSTEM_SM_KEY, sizeof(dbfsd_data), 0666);

	if(sysshmid == -1) 
	{
		Error("DBMixer ERROR: could not create shared memory for system data.");
		 
		return FAILURE; 
	}
   

	sysdata = shmat(sysshmid,(void *)0, 0);
   
	if((int)sysdata == -1)
	{
		Error("DBMixer ERROR: error attaching system data shared memory.");
      
		return FAILURE;
	}

	/* verify that there is a free channel into dbfsd */
	if(sysdata->free_channel_index == -1)
	{
/* 		ch = NULL; */

		if(shmdt(sysdata) == -1)
		{
			Error("DBAudio_Init: could not detach system data memory segment.");
			return FAILURE;
		}

		errno = ERROR_NO_FREE_CHANNELS;
		return FAILURE;
	}


	/* retrieve the channel memory id */
	shmid = shmget((key_t) DB_CHANNELS_SM_KEY, 
				   (sysdata->num_channels * sizeof(local_channel)), 0666);

	if(shmid == -1) 
	{
		Error("DBMixer ERROR: error creating shared memory.");
		return FAILURE;
	}
	else
    {
		Debug("DBMixer: shmid is: %d ",shmid);
    }
  
	/* attach the channel memory segment*/
	local_channels = (local_channel *) shmat(shmid,(void *)0, 0);

	if((int)local_channels == -1)
	{
		Error("DBMixer ERROR: error attaching shared memory.");

		return FAILURE;
	}


	if(sysdata->main_mixer_device[0] == '\0') strcpy(sysdata->main_mixer_device,MASTER_MIXER);

	if(sysdata->cue_mixer_device[0] == '\0') strcpy(sysdata->cue_mixer_device,CUE_MIXER);


	/*  get channel id */
/* 	ch = &(local_channels[sysdata->free_channel_index]); */

/* 	Debug("DBMixer: opening comm pipe... %s",ch->comm_filename); */

/* 	if((ch->client_comm_fd = open(ch->comm_filename, O_WRONLY)) == -1) */
/*     { */
/* 		perror("DBMixer: Error opening comm pipe."); */
/* 		return FAILURE; */
/*     } */


/* 	Debug("DBMixer: opening signal pipe..."); */

/* 	if((ch->client_sig_fd = open(ch->signal_filename, O_RDONLY)) == -1) */
/*     { */
/* 		perror("DBMixer: Error opening comm pipe."); */
/* 		return FAILURE; */
/*     } */

	Debug("DBMixer: dbaudio_init complete.");

	return SUCCESS;
}


int channel_exit()
{
	if(shmdt(local_channels) == -1)
	{
		Error("DBMixer: could not detach memory segment.");
		return FAILURE;
	}

	/* detach system data memory segment */
	if(shmdt(sysdata) == -1)
	{
		Error("DBMixer: could not detach system data memory segment.");
		return FAILURE;
	}

	if(standalone_flag)
	{
		Debug("calling gtk_main_quit()...");
		gtk_main_quit();
	}

#ifdef EXT_MIXER
	exmixer_stop();
#endif

	return SUCCESS;
}


void create_menu()
{
	GtkWidget * menu_item, * vbox;
	char str[256];

	menu = gtk_menu_new();

	/* create update option */
	sprintf(str,UPDATE_STR);
	menu_item = gtk_menu_item_new_with_label(str);
	gtk_menu_append(GTK_MENU(menu),menu_item);
	gtk_signal_connect_object (GTK_OBJECT (menu_item), "activate",

							   GTK_SIGNAL_FUNC (update_button_clicked), 
							   NULL);
	gtk_widget_show (menu_item);

	/* create reset option */
	sprintf(str,RESET_STR);
	menu_item = gtk_menu_item_new_with_label(str);
	gtk_menu_append(GTK_MENU(menu),menu_item);
	gtk_signal_connect_object (GTK_OBJECT (menu_item), "activate",
							   GTK_SIGNAL_FUNC (reset_button_clicked), 
							   NULL);
	gtk_widget_show (menu_item);

	/* add preferences string */
	sprintf(str,PREFERENCES_STR);
	menu_item = gtk_menu_item_new_with_label(str);
	gtk_menu_append(GTK_MENU(menu),menu_item);
	gtk_signal_connect_object (GTK_OBJECT (menu_item), "activate",
							   GTK_SIGNAL_FUNC (preferences_dialog), 
							   NULL);
	gtk_widget_show (menu_item);

	/* add record session option */
	record_label = (GtkLabel*) gtk_label_new(RECORD_WAV_STR);
	vbox = gtk_vbox_new(FALSE,0);
	menu_item = gtk_menu_item_new();
	
	gtk_box_pack_start(GTK_BOX(vbox),GTK_WIDGET(record_label),FALSE,FALSE,0);
	gtk_widget_show(GTK_WIDGET(record_label));
	gtk_widget_show(vbox);
	gtk_container_add(GTK_CONTAINER(menu_item),vbox);
	gtk_container_set_border_width(GTK_CONTAINER(menu_item),0);
	gtk_menu_append(GTK_MENU(menu),menu_item);
	gtk_signal_connect_object (GTK_OBJECT (menu_item), "activate",
							   GTK_SIGNAL_FUNC (record_session_clicked), 
							   NULL);
	gtk_widget_show(menu_item);
	recording_flag = 0;
	
	/* add about string */
	sprintf(str,ABOUT_STR);
	menu_item = gtk_menu_item_new_with_label(str);
	gtk_menu_append(GTK_MENU(menu),menu_item);
	gtk_signal_connect_object (GTK_OBJECT (menu_item), "activate",
							   GTK_SIGNAL_FUNC (dbmixer_about), 
							   NULL);
	gtk_widget_show (menu_item);


	sprintf(str,QUIT_STR);
	menu_item = gtk_menu_item_new_with_label(str);
	gtk_menu_append(GTK_MENU(menu),menu_item);
	gtk_signal_connect_object (GTK_OBJECT (menu_item), "activate",
							   GTK_SIGNAL_FUNC (cleanup), 
							   NULL);
	gtk_widget_show (menu_item);
}


/******************************************************************/
/**                    GTK Callback functions                    **/
/******************************************************************/

GtkWidget *channel_select,*mute_button, *cue_button, *gain_scale, *pitch_scale;


gint normalize_scale(gint i)
{
	if(i < 0) i = 0;
	if(i>100) i = 100;
	return i;
}


gint dbmixer_exit(GtkWidget * window, GdkEventAny * e, gpointer data)
{
	channel_exit();
/* 	gtk_main_quit(); */
	return SUCCESS;
}


/*makes a menu item. this function copied from example code found at www.gtk.org */
GtkWidget *make_menu_item(gchar *name, GtkSignalFunc callback,gpointer data )
{
	GtkWidget *item;
       
	item = gtk_menu_item_new_with_label (name);
	gtk_signal_connect (GTK_OBJECT (item), "activate",
						callback, data);
	gtk_widget_show (item);

	return(item);
}

void reset_button_clicked (GtkWidget * w, gpointer data)
{
	Debug("sending SIGHUP");
	kill(sysdata->pid,SIGHUP);
}


int update_recording(gpointer data)
{
	if (!GTK_WIDGET_VISIBLE(GTK_WIDGET(menu)))
	{
		if (sysdata->recording)
		{
			gtk_label_set_text(record_label,STOP_WAV_STR);
		}
		else
		{
			gtk_label_set_text(record_label,RECORD_WAV_STR);
		}
	}
	
	return TRUE;
}


void update_button_clicked(GtkWidget * w, gpointer data)
{
	update_recording(NULL);
	update_channels(NULL);
	update_soundcards(NULL);
	update_balance(NULL);
	update_sampler(NULL);
}

gint option_button_clicked(GtkWidget* widget, GdkEvent * event)
{
	
    if (event->type == GDK_BUTTON_PRESS) {
        GdkEventButton *bevent = (GdkEventButton *) event; 
        gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
                        bevent->button, bevent->time);
        /* Tell calling code that we have handled this event; the buck
         * stops here. */
        return TRUE;
    }
	
    /* Tell calling code that we have not handled this event; pass it on. */
    return FALSE;
}



void got_record_filename(GtkWidget *w, gint * data)
{
	gchar * filename;

	/* get filename */
	filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION(file_selector));

	/* copy filename to sysdata buffer */
	strcpy(sysdata->filename,filename);
	
	/* notify dbfsd to start recording */
    kill(sysdata->pid,SIGUSR1);
}


void record_session_clicked(GtkWidget * w, gpointer data)
{
	if (sysdata->recording)
	{
		Debug("stop recording");
		gtk_label_set_text(record_label,RECORD_WAV_STR);
		kill(sysdata->pid,SIGUSR2);
	}
	else
	{
		Debug("start recording");
		gtk_label_set_text(record_label,STOP_WAV_STR);

		/* get filename */

		/* Create the selector */   
		file_selector = gtk_file_selection_new(SAMPLE_FILE_STR);
		
		gtk_signal_connect(GTK_OBJECT (GTK_FILE_SELECTION(file_selector)->ok_button),
						   "clicked", GTK_SIGNAL_FUNC (got_record_filename), NULL);
		
		/* Ensure that the dialog box is destroyed when the user clicks a button. */
		
		gtk_signal_connect_object(GTK_OBJECT (GTK_FILE_SELECTION(file_selector)->ok_button),
								  "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
								  (gpointer) file_selector);
		
		gtk_signal_connect_object(GTK_OBJECT (GTK_FILE_SELECTION(file_selector)->cancel_button),
								  "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
								  (gpointer) file_selector);
		
		/* Display that dialog */
		gtk_widget_show (file_selector);
	}
}


/******************************************************************/
/**                     Main Window Creation                     **/
/******************************************************************/


void Window_Init()
{
	GtkWidget * tempwidget;
	GtkWidget * hbox;
	GtkWidget * main_box;
	GtkWidget * fader_box;
	GtkWidget * vbox;
	GtkWidget * channels_box;
	GtkWidget * clipbox;
	GtkWidget * soundcard_controls;
	GtkWidget * sample_editor_box;
	GtkWidget * option_button;
	GtkWidget * controls_box;
	GtkWidget * channels_v_box;
	GtkWidget * separator;
	int i;

	Debug("Indow_Init start.");

	channel_indexes = (int *) malloc(sysdata->num_channels * sizeof(int));

	widgets = (channel_widgets *) malloc(sysdata->num_channels * sizeof(channel_widgets));

	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_signal_connect (GTK_OBJECT (window), "destroy",
						GTK_SIGNAL_FUNC(dbmixer_exit),
						NULL);

	gtk_widget_realize(window);

	main_box = gtk_hbox_new(FALSE,0);
	gtk_widget_show(main_box);

	vbox = gtk_vbox_new(FALSE,0);
	gtk_widget_show(vbox);

	hbox = gtk_hbox_new(FALSE,0);
	gtk_widget_show(hbox);

	channels_box = gtk_hbox_new(FALSE,0);
	gtk_widget_show(channels_box);

	channels_v_box = gtk_vbox_new(FALSE,0);
	gtk_widget_show(channels_v_box);

	controls_box = gtk_vbox_new(FALSE,0);
	gtk_widget_show(controls_box);

	gtk_container_add(GTK_CONTAINER(vbox),hbox);

	gtk_window_set_title (GTK_WINDOW (window), DBMIXER_VERSION_STR);
/* 	gtk_window_set_default_size(window,500,400); */

	Debug("Creating channel controls...");
	for(i = 0; i < sysdata->num_channels; i++)
	{
		Debug("control %d... ",i);
		channel_indexes[i] = 0;

		tempwidget = create_channel_controls(i);
		gtk_widget_show(tempwidget);
		gtk_container_add(GTK_CONTAINER(channels_box),tempwidget);
		
		Debug("\bdone.");
	}

	Debug("Adding channel update timeout...");
	gtk_timeout_add(UPDATE_CHANNEL_TIMEOUT,update_channels,NULL);
	Debug("Adding sampler update timeout...");
	gtk_timeout_add(UPDATE_CHANNEL_TIMEOUT,update_sampler,NULL);
	Debug("Adding recording update timeout...");
	gtk_timeout_add(UPDATE_CHANNEL_TIMEOUT,update_recording,NULL);

	Debug("Creating clipping notifier...");
	clipbox = Create_Clipping_Light(window);
	gtk_widget_show(clipbox);
	gtk_box_pack_start(GTK_BOX(controls_box),clipbox,
					   TRUE,FALSE,0);	

	Debug("Creating clipping notifier timer...");
	{
		guint32 interval = CLIPPING_INTERVAL;
		gtk_timeout_add(interval,(GtkFunction)clipping_light_callback,NULL);
	}

	Debug("Creating master and cue volume controls...");

	/* add soundcard controls to controls_box */
	soundcard_controls = Create_Soundcard_Controls();
	gtk_widget_show(soundcard_controls);
	gtk_box_pack_start(GTK_BOX(controls_box),soundcard_controls,
					   TRUE,FALSE,0);

	/* add autofade speed control to controls_box */
	Create_Autofade_Scale(GTK_BOX(controls_box));

	/* create option button and associated menu  */
	create_menu(); 

	option_button = gtk_button_new_with_label(OPTION_STR);
    gtk_signal_connect_object (GTK_OBJECT (option_button), "event",
							   GTK_SIGNAL_FUNC (option_button_clicked), 
							   GTK_OBJECT (menu));	
	gtk_box_pack_start(GTK_BOX(controls_box),(GtkWidget*)option_button,
					   TRUE,FALSE,0);
	gtk_widget_show(option_button);

	/* create crossfader */
	fader_box = Create_Fader(window);

	/* create sample editor */
	sample_editor_box = create_sample_editor();

	Debug("packing main window...");

	gtk_box_pack_start(GTK_BOX(channels_v_box),channels_box,FALSE,FALSE,0);

	gtk_box_pack_start(GTK_BOX(hbox),channels_v_box,FALSE,FALSE,0);


	separator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox),separator,FALSE,FALSE,0);
	gtk_widget_show(separator);

	/* note: to make fader box to be larger than channels box, replace
             channels_v_box with vbox */
	gtk_box_pack_start(GTK_BOX(vbox),fader_box,FALSE,FALSE,10);

	gtk_box_pack_start(GTK_BOX(main_box),vbox,FALSE,FALSE,10);

	gtk_box_pack_start(GTK_BOX(hbox),sample_editor_box,FALSE,FALSE,10);
	gtk_widget_show(sample_editor_box);

	separator = gtk_vseparator_new();
	gtk_box_pack_start(GTK_BOX(main_box),separator,FALSE,FALSE,0);
	gtk_widget_show(separator);

	gtk_box_pack_start(GTK_BOX(main_box),controls_box,FALSE,FALSE,10);
	gtk_widget_show(controls_box);

	gtk_container_add(GTK_CONTAINER(window),main_box);
	gtk_widget_show(window);

	Debug("Window_Init end.");
}




void init(void)
{

#ifdef DBMIX_DEBUG
	debug_level = 1;
	printf("Debugging is on.\n");
#else
	debug_level = 0;
#endif

	fade_time_base = DEFAULT_FADE_TIME_BASE;

	/* do dbfsd init stuff */
	Debug("INIT: initializing shared memory...");
	if(Channel_Init() != SUCCESS)
	{
		Debug("INIT: Channel Init failed... Is dbfsd running?\nexiting...");
		channel_exit();
		return;
	}

	if(Soundcard_Mixer_Init() != SUCCESS)
	{
		Debug("INIT: could not initialize soundcard mixer... ");
	}
	else
	{
		Debug("INIT: soundcard mixer(s) has been initialized.\n");
	}	

	Debug("INIT: initailizing gui components...");
	/* create gui conponents */
	Window_Init();

/* 	update_button_clicked(NULL,NULL); */

#ifdef EXT_MIXER 
    /* Create exmixer and clear values */ 
    exmixer_clear(); 
    
    read_settings(); 
    
    /*                  */ 
    /* Start the exmixer */ 
    /*                  */ 
    
    /* Is the preferences file present? */ 
    if( exmixer_enabled ) 
    { 
       /* If no ~/.joystick file, must calibrate on the fly */ 
       set_calibration( !read_js_settings() ); 
        
       exmixer_start(); 
    } 
 
#endif /* EXT_MIXER */ 

	Debug("INIT: done.");
}



void cleanup(void)
{
	gtk_widget_destroy(window);
	window = NULL;

	/* tell monitor loop to shutdown */
	Debug("shutting down monitor flag.\n");
	monitor_loop_flag = 0;

	/* free shared memory */
	/* detach channel memory segment */
	if(shmdt(local_channels) == -1)
	{
		Error("DBMixer: could not detach channel memory segment.");
/* 		return FAILURE; */
	}

	/* detach system data memory segment */
	if(shmdt(sysdata) == -1)
	{
		Error("DBMixer: could not detach system data memory segment.");
/* 		return FAILURE; */
	}
}

