/* $Id: hdrl_combine.c,v 1.3 2013-10-16 11:31:02 cgarcia Exp $
 *
 * This file is part of the HDRL
 * Copyright (C) 2013 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-10-16 11:31:02 $
 * $Revision: 1.3 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
-----------------------------------------------------------------------------*/

#include "hdrl_combine.h"
#include "hdrl_iter.h"
#include "hdrl_utils.h"
#include "hdrl_sigclip.h"
#include <cpl.h>
#include <math.h>
#include <string.h>
#include <assert.h>

/*-----------------------------------------------------------------------------
                                   Static
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @defgroup hdrl_combine   COMBINE Module

  This module allows the combination of imagelists with error propagation.
  If input images size is large the user should use hdrl_imagelist_combine_it()
  for efficient RAM usage. Else can be used hdrl_imagelist_combine()
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* ---------------------------------------------------------------------------*/
/**
 * @brief combine imagelist with error propagation
 *
 * @param data     input data imagelist
 * @param errors   input imagelist containing errors of data
 * @param method   reduction method applied to the imagelists
 * @param out      pointer to storage for a pointer to the allocated
 *                 combined data image
 * @param out      pointer to storage for a pointer to the allocated
 *                 combined error image
 * @param contrib  pointer to storage for a pointer to the allocated
 *                 contribution map
 */
/* ---------------------------------------------------------------------------*/
cpl_error_code
hdrl_imagelist_combine(const cpl_imagelist * data,
                       const cpl_imagelist * errors,
                       hdrl_collapse_imagelist_to_image_t * method,
                       cpl_image ** out,
                       cpl_image ** err,
                       cpl_image ** contrib)
{
    cpl_ensure_code(data && errors, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out && err && contrib, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cpl_imagelist_get_size(data) > 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(cpl_imagelist_get_size(data) ==
                    cpl_imagelist_get_size(errors), CPL_ERROR_ILLEGAL_INPUT);

    hdrl_collapse_imagelist_to_image_call(method, data, errors,
                                        out, err, contrib);

    return cpl_error_get_code();
}


/* ---------------------------------------------------------------------------*/
/**
 * @brief combine imagelist provided by iterators with error propagation
 *
 * @param imgit   input iterator providing the images in an imagelist,
 *                the images may be slices of the full dataset
 * @param errit   input iterator providing the errorimages in an imagelist,
 *                the images may be slices of the full dataset
 * @param method  reduction method applied to the imagelist slices
 * @param oimgit  output iterator providing buffers to store the combined
 *                double image, buffers must have the same size as the images
 *                in the input imagelist iterator
 * @param oerrit  output iterator providing buffers to store the combined double
 *                error image, buffers must have the same size as images in the
 *                input imagelist iterator
 * @param ocontribit  output iterator providing buffers to store the combined
 *                    integer contribution maps, buffers must have the same
 *                    size as the images in the input imagelist
 *
 * @return cpl_error_code
 * @see hdrl_imagelist_combine
 *
 * This function should only be used if the input is already provided in form
 * of a suiteable drl iterator or the input data does not fit into memory. If
 * this is not the case hdrl_imagelist_combine should be used.
 *
 * The input iterators must have following properties:
 * HDRL_ITER_INPUT | HDRL_ITER_IMAGELIST
 *
 * The output iterators must have the following properties:
 * HDRL_ITER_OUTPUT | HDRL_ITER_IMAGE
 *
 * This combination variant cannot provide extra output the reduction method
 * may provide, if it is needed loop on the iterators manually and call
 * hdrl_imagelist_combine
 */
/* ---------------------------------------------------------------------------*/
cpl_error_code
hdrl_imagelist_combine_it(hdrl_iter_t * imgit,
                         hdrl_iter_t * errit,
                         hdrl_collapse_imagelist_to_image_t * method,
                         hdrl_iter_t * oimgit,
                         hdrl_iter_t * oerrit,
                         hdrl_iter_t * ocontribit)
{
    cpl_ensure_code(hdrl_iter_check(imgit, HDRL_ITER_INPUT | HDRL_ITER_IMAGELIST),
                    CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(hdrl_iter_check(errit, HDRL_ITER_INPUT | HDRL_ITER_IMAGELIST),
                    CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(hdrl_iter_check(oimgit, HDRL_ITER_OUTPUT | HDRL_ITER_IMAGE),
                    CPL_ERROR_ILLEGAL_OUTPUT);
    cpl_ensure_code(hdrl_iter_check(oerrit, HDRL_ITER_OUTPUT | HDRL_ITER_IMAGE),
                    CPL_ERROR_ILLEGAL_OUTPUT);
    cpl_ensure_code(hdrl_iter_check(ocontribit, HDRL_ITER_OUTPUT |
                                   HDRL_ITER_IMAGE), CPL_ERROR_ILLEGAL_OUTPUT);

    /* iteration cannot provide this information */
    hdrl_collapse_imagelist_to_image_disable_extra_out(method);
    for (cpl_imagelist * imgsl = hdrl_iter_next(imgit),
         * errsl = hdrl_iter_next(errit);
         imgsl != NULL && errsl != NULL;
         imgsl = hdrl_iter_next(imgit),
         errsl = hdrl_iter_next(errit)) {

        cpl_image * out, * err, * contrib;
        cpl_image * rout = hdrl_iter_next(oimgit);
        cpl_image * rerr = hdrl_iter_next(oerrit);
        cpl_image * rcontrib = hdrl_iter_next(ocontribit);
        hdrl_collapse_imagelist_to_image_call(method, imgsl, errsl,
                                            &out, &err, &contrib);
        if (cpl_image_get_size_x(out) != cpl_image_get_size_x(rout) ||
            cpl_image_get_size_y(out) != cpl_image_get_size_y(rout)) {
            cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                  "Output iterator buffer does not have "
                                  "expected size");
        }
        cpl_image_copy(rout, out, 1, 1);
        cpl_image_copy(rerr, err, 1, 1);
        cpl_image_copy(rcontrib, contrib, 1, 1);

        cpl_image_delete(out);
        cpl_image_delete(err);
        cpl_image_delete(contrib);
        cpl_imagelist_delete(imgsl);
        cpl_imagelist_delete(errsl);
        if (cpl_error_get_code())
            break;
    }

    return cpl_error_get_code();
}

/**@}*/
