/*
 *  Copyright 2011 Wolfgang Koller - http://www.gofg.at/
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

#include "fileapi.h"

#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QApplication>

//use static method create initialized map for hardcoded mimetype
FileAPI::MimeTypeMap FileAPI::mimeMap_(FileAPI::createMimeTypeMap());

FileAPI::FileAPI(Cordova *cordova) :
    CPlugin(cordova),
    m_persistentDir(QDir::homePath() + "/.local/share/cordova-ubuntu/persistent/" + QString(cordova->get_app_dir().toUtf8().toBase64())),
    lastRequestId(1) {
}

FileAPI::MimeTypeMap FileAPI::createMimeTypeMap() {
    MimeTypeMap map;
    map.insert(QString("txt"),QString("text/plain"));
    map.insert(QString("css"),QString("text/css"));
    map.insert(QString("js"),QString("text/javascript"));
    map.insert(QString("xml"),QString("text/xml"));
    map.insert(QString("html"),QString("text/html"));
    return map;
}

void FileAPI::requestFileSystem(int scId, int ecId, unsigned short p_type, unsigned long long p_size) {
    QDir dir;

    //FIXEME,what is quota value
    if (p_size >= 10000){
        this->callback(ecId, "FileException.cast(FileException.QUOTA_EXCEEDED_ERR)");
    }

    // Get correct system path
    if (p_type == 0) {
        dir = QDir::temp();
    }
    else {
        dir = QDir(m_persistentDir);
        QDir::root().mkpath(dir.absolutePath());
    }

    if (p_type == 0) {
        this->callback(scId, "FileSystem.cast('temporary', '" + dir.dirName() + "', '" + dir.absolutePath() + "')");
    } else if (p_type == 1){
        this->callback(scId, "FileSystem.cast('persistent', '" + dir.dirName() + "', '" + dir.absolutePath() + "')");
    } else {
        this->callback(ecId, "FileException.cast(FileException.SYNTAX_ERR)");
    }
}

void FileAPI::resolveLocalFileSystemURL(int scId, int ecId, QString p_url) {
    QUrl url = QUrl::fromUserInput(p_url);

    // Check if we have a valid URL
    if (!url.isValid() || p_url[0] == '/' || p_url[0] == '.') {
        this->callback(ecId, "FileException.cast(FileException.ENCODING_ERR)");
        return;
    }//FIXEME: invalid pass

    // Check for the correct scheme
    if (url.scheme() != "file") {
        this->callback(ecId, "FileException.cast(FileException.TYPE_MISMATCH_ERR)");
        return;
    }

    // Now get info about the path
    QFileInfo fileInfo(url.path());
    // Check if entry exists
    if (!fileInfo.exists()) {
        this->callback(ecId, "FileException.cast(FileException.NOT_FOUND_ERR)");
        return;
    }
    if (fileInfo.isDir()) {
        this->callback(scId, "DirectoryEntry.cast('" + fileInfo.fileName() + "', '" + QDir::cleanPath(fileInfo.absoluteFilePath()) + "')");
        return;
    }
    else {
        this->callback(scId, "FileEntry.cast('" + fileInfo.fileName() + "', '" + fileInfo.absoluteFilePath() + "')");
        return;
    }
}

void FileAPI::getFile(int scId, int ecId, QString p_path, QVariantMap p_options) {
//    qDebug() << Q_FUNC_INFO << QString(p_path);
    //NOTE: colon is not safe in url, it is not a valid path in Win and Mac, simple disable it here.
    if (p_path.contains(":")){
        this->callback(ecId, "FileException.cast(FileException.ENCODING_ERR)");
        return;
    }
    // Check if we have a valid URL
    QUrl url = QUrl::fromUserInput(p_path);
    if (!url.isValid()) {
        this->callback(ecId, "FileException.cast(FileException.ENCODING_ERR)");
        return;
    }
    // Check for the correct scheme
    if (url.scheme() != "file") {
        this->callback(ecId, "FileException.cast(FileException.TYPE_MISMATCH_ERR)");
        return;
    }
    bool create = p_options.value("create").toBool();
    bool exclusive = p_options.value("exclusive").toBool();

    QFile file(p_path);
    QFileInfo pathInfo(p_path);
    QString fileName(pathInfo.fileName());

    // if create is false and the path represents a directory, return error
    // if file does exist, and create is true and exclusive is true, return error
    // if file does not exist and create is false, return error
    // if file does not exist and create is true, create file and return File entry
    QFileInfo fileInfo(url.path());
    if ((!create) && fileInfo.isDir()) {
        this->callback(ecId, "FileException.cast(FileException.TYPE_MISMATCH_ERR)");
        return;
    }

    if (file.exists()) {
        if (create && exclusive) {
            this->callback(ecId, "FileException.cast(FileException.PATH_EXISTS_ERR)");
            return;
        }
    }
    else {
        if (!create) {
            this->callback(ecId, "FileException.cast(FileException.NOT_FOUND_ERR)");
            return;
        }

        // Create the file
        file.open(QIODevice::WriteOnly);
        file.close();

        // Check if creation was successfull
        if (!file.exists()) {
            this->callback(ecId, "FileException.cast(FileException.NO_MODIFICATION_ALLOWED_ERR)");
            return;
        }
    }

    // If we reach here, everything went well
    this->callback(scId, "FileEntry.cast('" + fileName + "', '" + QFileInfo(file).absoluteFilePath() + "')");
}

void FileAPI::getDirectory(int scId, int ecId, QString p_path, QVariantMap p_options) {
    //NOTE: colon is not safe in url, it is not a valid path in Win and Mac, simple disable it here.
    if (p_path.contains(":")){
        this->callback(ecId, "FileException.cast(FileException.ENCODING_ERR)");
    }
    // Check if we have a valid URL
    QUrl url = QUrl::fromUserInput(p_path);
    if (!url.isValid()) {
        this->callback(ecId, "FileException.cast(FileException.ENCODING_ERR)");
        return;
    }

    // Check for the correct scheme
    if (url.scheme() != "file") {
        this->callback(ecId, "FileException.cast(File:Exception.TYPE_MISMATCH_ERR)");
        return;
    }
    bool create = p_options.value("create").toBool();
    bool exclusive = p_options.value("exclusive").toBool();

    QDir dir(p_path);

    //  if create is false and the path represents a file, return error
    //  if directory does exist and create is true and exclusive is true, return error
    //  if directory does not exist and create is false and directory does not exist, return error
    //  if directory does not exist and create is true, create dir and return directory entry

    QFileInfo fileInfo(url.path());
    if ((!create) && fileInfo.isFile()) {
        this->callback(ecId, "FileException.cast(FileException.TYPE_MISMATCH_ERR)");
        return;
    }


    if (dir.exists()) {
        if (create && exclusive) {
            this->callback(ecId, "FileException.cast(FileException.PATH_EXISTS_ERR)");
            return;
        }
    }
    else {
        if (!create) {
            this->callback(ecId, "FileException.cast(FileException.NOT_FOUND_ERR)");
            return;
        }

        // Create the folder
        QString folderName = dir.dirName();
        dir.cdUp();
        dir.mkdir(folderName);
        dir.cd(folderName);

        // Check if creation was successfull
        if (!dir.exists()) {
            this->callback(ecId, "FileException.cast(FileException.NO_MODIFICATION_ALLOWED_ERR)");
            return;
        }
    }

    // If we reach here, everything went well
    this->callback(scId, "DirectoryEntry.cast('" + dir.dirName() + "', '" + dir.absolutePath() + "')");
}

void FileAPI::removeRecursively(int scId, int ecId, QString p_path) {
    QDir dir(p_path);
    if (FileAPI::rmDir(dir)) {
        this->cb(scId);
        return;
    }

    this->callback(ecId, "FileException.cast(FileException.NO_MODIFICATION_ALLOWED_ERR)");
}

void FileAPI::file(int scId, int ecId, QString p_path) {
    QFileInfo fileInfo(p_path);

    if (!fileInfo.exists()) {
        this->callback(ecId, "FileException.cast(FileException.NOT_FOUND_ERR)");
    } else {
        this->callback(scId, "File.cast('" + fileInfo.fileName() + "', '" + fileInfo.absoluteFilePath() + "', 'unknown/unknown', new Date(" + QString::number(fileInfo.lastModified().toMSecsSinceEpoch()) + "), " + QString::number(fileInfo.size()) + ")");
    }
}

void FileAPI::write(int scId, int ecId, QString p_path, unsigned long long p_position, QString _data, bool binary) {
    QFile file(p_path);
    //    if (!file.exists()) {
    // FIXME: always overwrite??
    file.open(QIODevice::WriteOnly);
    file.close();
        //    }

    if (!file.exists()) {
        this->callback(ecId, "FileException.cast(FileException.NOT_FOUND_ERR), 0, 0");
        return;
    }

    QFileInfo fileInfo(file);
    if (!file.open(QIODevice::ReadWrite)) {
        this->callback(ecId, "FileException.cast(FileException.NO_MODIFICATION_ALLOWED_ERR), 0, " + QString::number(fileInfo.size()));
        return;
    }

    if (!binary) {
        QTextStream textStream(&file);
        textStream.setCodec("UTF-8");
        textStream.setAutoDetectUnicode(true);

        if (!textStream.seek(p_position)) {
            file.close();
            fileInfo.refresh();

            this->callback(ecId, "FileException.cast(FileException.INVALID_MODIFICATION_ERR), 0, " + QString::number(fileInfo.size()));
            return;
        }

        textStream << _data;
        textStream.flush();
    } else {
        QByteArray p_data(_data.toUtf8());
        if (!file.seek(p_position)) {
            file.close();
            fileInfo.refresh();

            this->callback(ecId, "FileException.cast(FileException.INVALID_MODIFICATION_ERR), 0, " + QString::number(fileInfo.size()));
            return;
        }

        file.write(p_data.data(), p_data.length());
    }

    file.flush();
    file.close();
    fileInfo.refresh();
    if (p_position > 0) {
        if (!file.resize(p_position + _data.size())) {
            this->callback(ecId, "FileException.cast(FileException.NO_MODIFICATION_ALLOWED_ERR), " + QString::number(file.size()) + ", " + QString::number(file.size()));
            return;
        }
    }

    this->cb(scId, fileInfo.size(), fileInfo.size());
}

void FileAPI::truncate(int scId, int ecId, QString p_path, unsigned long long p_size) {
    QFile file(p_path);

    if (!file.exists()) {
        this->callback(ecId, "FileException.cast(FileException.NOT_FOUND_ERR), 0, 0");
        return;
    }

    // Try to resize (truncate) the file
    if (!file.resize(p_size)) {
        this->callback(ecId, "FileException.cast(FileException.NO_MODIFICATION_ALLOWED_ERR), " + QString::number(file.size()) + ", " + QString::number(file.size()));
        return;
    }

    this->cb(scId, p_size, p_size);
}

void FileAPI::getParent(int scId, int ecId, QString p_path) {
    QDir dir(p_path);

    //can't cdup more than app's root
    // Try to change into upper directory
    if (p_path != m_persistentDir){
        if (!dir.cdUp()) {
            this->callback(ecId, "FileException.cast(FileException.NOT_FOUND_ERR)");
            return;
        }

    }
    this->callback(scId, "DirectoryEntry.cast('" + dir.dirName() + "', '" + dir.absolutePath() + "')");
}

void FileAPI::remove(int scId, int ecId, QString p_path) {
    QFileInfo fileInfo(p_path);
    if (!fileInfo.exists() || (p_path == m_persistentDir)) {
        this->callback(ecId, "FileException.cast(FileException.NO_MODIFICATION_ALLOWED_ERR)");
        return;
    }

    // Check if we have a dir
    if (fileInfo.isDir()) {
        QDir dir(p_path);
        if (dir.rmdir(dir.absolutePath())) {
            this->cb(scId);
            return;
        }
    } else {
        QFile file(p_path);
        if (file.remove()) {
            this->cb(scId);
            return;
        }
    }

    this->callback(ecId, "FileException.cast(FileException.INVALID_MODIFICATION_ERR)");
}

void FileAPI::getMetadata(int scId, int ecId, QString p_path) {
    QFileInfo fileInfo(p_path);

    if (!fileInfo.exists()) {
        this->callback(ecId, "FileException.cast(FileException.NOT_FOUND_ERR)");
        return;
    }

    this->callback(scId, "Metadata.cast(new Date(" + QString::number(fileInfo.lastModified().toMSecsSinceEpoch()) + "))");
}

void FileAPI::readEntries(int scId, int ecId, QString p_path) {
    QDir dir(p_path);

    QString entriesList = "";

    if (dir.exists()) {
        // Iterate over entries and add them to response
        Q_FOREACH(const QFileInfo &fileInfo, dir.entryInfoList(QDir::Dirs | QDir::Files | QDir::NoDotAndDotDot)) {
            if (fileInfo.isDir()) {
                entriesList += "DirectoryEntry.cast('" + fileInfo.fileName() + "','" + fileInfo.absoluteFilePath() + "/'),";
            }
            else {
                entriesList += "FileEntry.cast('" + fileInfo.fileName() + "','" + fileInfo.absoluteFilePath() + "'),";
            }
        }

        // Remove trailing comma
        if (entriesList.size() > 0) entriesList.remove(entriesList.size()-1, 1);
        entriesList = "new Array(" + entriesList + ")";

        this->callback(scId, entriesList);
        return;
    }

    this->callback(ecId, "FileException.cast(FileException.NOT_FOUND_ERR)");
}

void FileAPI::readAsText(int scId, int ecId, QString p_path, bool sliced, int sliceStart, int sliceEnd) {
    QFile file(p_path);

    if (!file.exists()) {
        this->callback(ecId, "FileException.cast(FileException.NOT_FOUND_ERR)");
        return;
    }

    if (!file.open(QIODevice::ReadOnly)) {
        this->callback(ecId, "FileException.cast(FileException.NOT_READABLE_ERR)");
        return;
    }

    QByteArray content = file.readAll();

    if (sliceEnd == -1)
        sliceEnd = content.size();
    if (sliceEnd < 0) {
        sliceEnd++;
        sliceEnd = std::max(0, content.size() + sliceEnd);
    }
    if (sliceEnd > content.size())
        sliceEnd = content.size();

    if (sliceStart < 0)
        sliceStart = std::max(0, content.size() + sliceStart);
    if (sliceStart > content.size())
        sliceStart = content.size();

    if (sliceStart > sliceEnd)
        sliceEnd = sliceStart;

    if (sliced)
        content = content.mid(sliceStart, sliceEnd - sliceStart);

    this->cb(scId, content);
}

void FileAPI::readAsBinaryString(int scId, int ecId, QString p_path, bool sliced, int sliceStart, int sliceEnd) {
    QFile file(p_path);

    if (!file.exists()) {
        this->callback(ecId, "FileException.cast(FileException.NOT_FOUND_ERR)");
        return;
    }

    if (!file.open(QIODevice::ReadOnly)) {
        this->callback(ecId, "FileException.cast(FileException.NOT_READABLE_ERR)");
        return;
    }
    QString res;
    QByteArray content = file.readAll();

    if (sliceEnd == -1)
        sliceEnd = content.size();
    if (sliceEnd < 0) {
        sliceEnd++;
        sliceEnd = std::max(0, content.size() + sliceEnd);
    }
    if (sliceEnd > content.size())
        sliceEnd = content.size();

    if (sliceStart < 0)
        sliceStart = std::max(0, content.size() + sliceStart);
    if (sliceStart > content.size())
        sliceStart = content.size();

    if (sliceStart > sliceEnd)
        sliceEnd = sliceStart;

    if (sliced)
        content = content.mid(sliceStart, sliceEnd - sliceStart);
    res.reserve(content.length() * 6);
    for (uchar c:content) {
        res += "\\x";
        res += QString::number(c, 16).rightJustified(2, '0').toUpper();
    }
    this->callback(scId, "\"" + res + "\"");
}

void FileAPI::readAsDataURL(int scId, int ecId, QString p_path, bool sliced, int sliceStart, int sliceEnd) {
    QFile file(p_path);
    QFileInfo fileInfo(p_path);

    if (p_path.startsWith("content:")){
        this->callback(ecId, "FileException.cast(FileException.NOT_READABLE_ERR)");
        return;
    }

    if (!file.exists()) {
        this->callback(ecId, "FileException.cast(FileException.NOT_FOUND_ERR)");
        return;
    }
    // Try to open file for reading
    if (!file.open(QIODevice::ReadOnly)) {
        this->callback(ecId, "FileException.cast(FileException.NOT_READABLE_ERR)");
        return;
    }
    // Read the file content
    QByteArray content = file.readAll();
    QString contentType(mimeMap_[fileInfo.completeSuffix()]);

    if (sliceEnd == -1)
        sliceEnd = content.size();
    if (sliceEnd < 0) {
        sliceEnd++;
        sliceEnd = std::max(0, content.size() + sliceEnd);
    }
    if (sliceEnd > content.size())
        sliceEnd = content.size();

    if (sliceStart < 0)
        sliceStart = std::max(0, content.size() + sliceStart);
    if (sliceStart > content.size())
        sliceStart = content.size();

    if (sliceStart > sliceEnd)
        sliceEnd = sliceStart;

    if (sliced)
        content = content.mid(sliceStart, sliceEnd - sliceStart);

    this->cb(scId, QString("data:%1;base64,").arg(contentType) + content.toBase64());
}

/*
 * Helper function for recursively removing a directory
 */
bool FileAPI::rmDir(QDir p_dir) {
    if (p_dir == m_persistentDir) {//can't remove root dir
        return false;
    }
    bool result = true;
    if (p_dir.exists()) {
        // Iterate over entries and remove them
        Q_FOREACH(const QFileInfo &fileInfo, p_dir.entryInfoList(QDir::Dirs | QDir::Files | QDir::NoDotAndDotDot)) {
            if (fileInfo.isDir()) {
                result = rmDir(fileInfo.absoluteFilePath());
            }
            else {
                result = QFile::remove(fileInfo.absoluteFilePath());
            }

            if (!result) {
                return result;
            }
        }

        // Finally remove the current dir
        return p_dir.rmdir(p_dir.absolutePath());
    }
    return result;
}

bool FileAPI::copyFile(int scId, int ecId,const QString& sourceFile, const QString& destinationParentDir, const QString& newName)
{
    if (!QDir(destinationParentDir).exists()){
        this->callback(ecId, "FileException.cast(FileException.NOT_FOUND_ERR)");
        return false;
    }
    QFileInfo fileInfo(sourceFile);
    QString fileName = ((newName.isEmpty()) ? fileInfo.fileName() : newName);
    QString destinationFile(destinationParentDir + "/" + fileName);
    //NOTE: colon is not safe in url, it is not a valid path in Win and Mac, simple disable it here.
    if (!QUrl::fromUserInput(destinationFile).isValid() || destinationFile.contains(":")){
        this->callback(ecId, "FileException.cast(FileException.ENCODING_ERR)");
        return false;
    }

    if (QFile::copy(sourceFile, destinationFile)){
        this->callback(scId, "FileEntry.cast('" + fileName + "', '" + destinationFile + "')");
        return true;
    } else {
        this->callback(ecId, "FileException.cast(FileException.INVALID_MODIFICATION_ERR)");
        return false;
    }
}

void FileAPI::moveFile(int scId, int ecId,const QString& sourceFile, const QString& destinationParentDir, const QString& newName){

    QString fileName = ((newName.isEmpty()) ? QFileInfo(sourceFile).fileName() : newName);
    //if existing a file have the newName, remove it and rename sourceFile
    QString destinationFile(destinationParentDir + "/" + fileName);
    if (QFileInfo(destinationFile).exists() && (sourceFile != destinationFile)){
        if (!QFile::remove(destinationFile)){
            this->callback(ecId, "FileException.cast(FileException.INVALID_MODIFICATION_ERR)");
            return;
        }
        QFile::rename(sourceFile,destinationFile);
        this->callback(scId, "FileEntry.cast('" + fileName + "', '" + destinationFile + "')");
        return;
    }
    //try copy source file to dest file and remove it if success.
    if (copyFile(scId,ecId,sourceFile, destinationParentDir, newName)){
        remove(scId,ecId, sourceFile);
        this->cb(scId);
        return;
    } else {
        qDebug() << "unable to copy the file, ecID is callbacked by copyFile";
        return;
    }

}

void FileAPI::copyDir(int scId, int ecId,const QString& sourceFolder, const QString& destinationParentDir, const QString& newName)
{
    QDir sourceDir(sourceFolder);
    QString dirName = ((newName.isEmpty()) ? sourceDir.dirName() : newName);
    QString destFolder(destinationParentDir + "/" + dirName);

    //can't copy a dir on a file
    if (QFileInfo(destFolder).isFile()){
        this->callback(ecId, "FileException.cast(FileException.INVALID_MODIFICATION_ERR)");
        return;
    }
    QDir destDir(destFolder);
    //can't copy on or in itself
    if ((sourceFolder == destFolder) || (sourceFolder == destinationParentDir)){
        this->callback(ecId, "FileException.cast(FileException.INVALID_MODIFICATION_ERR)");
        return;
    }
    //create the dir
    if (!destDir.exists()){
        qDebug() << "mkdir" << destDir.mkdir(destFolder);;
    } else{
        this->callback(ecId, "FileException.cast(FileException.INVALID_MODIFICATION_ERR)");
        return;
    }

    //recursively copying begin
    if (copyFolder(sourceFolder, destFolder)){
        this->callback(scId, "DirectoryEntry.cast('" + dirName + "', '" + destFolder + "')");
        return;
    } else {
        this->callback(ecId, "FileException.cast(FileException.INVALID_MODIFICATION_ERR)");
        return;
    }
}

void FileAPI::moveDir(int scId, int ecId,const QString& sourceDir, const QString& destinationParentDir, const QString& newName){
    QString dirName = ((newName.isEmpty()) ? QDir(sourceDir).dirName() : newName);
    QString destFolder(destinationParentDir + "/" + dirName);
    QDir destDir(destFolder);

    //can't copy a dir on a file
    if (QFileInfo(destFolder).isFile()){
        this->callback(ecId, "FileException.cast(FileException.INVALID_MODIFICATION_ERR)");
        return;
    }
    //can't copy on or in itself
    if ((sourceDir == destFolder) || (sourceDir == destinationParentDir)){
        this->callback(ecId, "FileException.cast(FileException.INVALID_MODIFICATION_ERR)");
        return;
    }

    if (destDir.exists() && (destFolder != sourceDir)){
        if (QDir(destinationParentDir).rmdir(dirName)){
            qDebug() << "empty folder rmed";
        } else {
            this->callback(ecId, "FileException.cast(FileException.INVALID_MODIFICATION_ERR)");
            return;
        }
    }
    if (copyFolder(sourceDir, destFolder)){
        qDebug() << "rming src dir if src-> dest copy success";
        rmDir(sourceDir);
        this->callback(scId, "DirectoryEntry.cast('" + dirName + "', '" + destFolder + "')");
        return;
    } else {
        qDebug()<< "unable to copy dirs" <<Q_FUNC_INFO <<
                   ", "<<sourceDir << ", "<< destinationParentDir << ", "<<newName;
        return;
    }

}

//helper function to copy foler to new destination
bool FileAPI::copyFolder(const QString& sourceFolder, const QString& destFolder) {
    QDir sourceDir(sourceFolder);
    if (!sourceDir.exists())
        return false;
    QDir destDir(destFolder);
    if (!destDir.exists()){
        destDir.mkdir(destFolder);
    }
    QStringList files = sourceDir.entryList(QDir::Files);
    for (int i = 0; i< files.count(); i++)
    {
        QString srcName = sourceFolder + "/" + files[i];
        QString destName = destFolder + "/" + files[i];
        QFile::copy(srcName, destName);
    }
    files.clear();
    files = sourceDir.entryList(QDir::AllDirs | QDir::NoDotAndDotDot);
    for (int i = 0; i< files.count(); i++)
    {
        QString srcName = sourceFolder + "/" + files[i];
        QString destName = destFolder + "/" + files[i];
        copyFolder(srcName, destName);
    }
    return true;
}

void FileAPI::newTransferRequest(int scId, int ecId) {
    Q_UNUSED(ecId)

    this->cb(scId, ++lastRequestId);
}

void FileAPI::downloadFile(int scId, int ecId, int id, const QString& url) {
    QSharedPointer<FileTransferRequest> request(new FileTransferRequest(m_manager, scId, ecId, m_id2progress[id], id, this));

    m_id2request.insert(id, request);

    request->connect(request.data(), &FileTransferRequest::done, [&]() {
        auto it = m_id2request.find(id);
        while (it != m_id2request.end() && it.key() == id) {
            if (it.value().data() == request.data()) {
                m_id2request.erase(it);
                break;
            }
            it++;
        }
    });
    request->download(url);
}

void FileAPI::uploadFile(int scId, int ecId, int id, const QString& url, const QString& content, QString fileKey, QString fileName, QString mimeType, const QVariantMap &params, const QVariantMap &headers) {
    QSharedPointer<FileTransferRequest> request(new FileTransferRequest(m_manager, scId, ecId, m_id2progress[id], id, this));

    m_id2request.insert(id, request);

    request->connect(request.data(), &FileTransferRequest::done, [&]() {
        auto it = m_id2request.find(id);
        while (it != m_id2request.end() && it.key() == id) {
            if (it.value().data() == request.data()) {
                m_id2request.erase(it);
                break;
            }
            it++;
        }
    });
    request->upload(url, content, fileKey, fileName, mimeType, params, headers);
}

void FileAPI::transferRequestSetOnProgress(int scId, int ecId, int id) {
    Q_UNUSED(ecId);

    m_id2progress[id] = scId;
}

void FileAPI::abortRequests(int scId, int ecId, int id) {
    Q_UNUSED(scId)
    Q_UNUSED(ecId)

    auto it = m_id2request.find(id);
    while (it != m_id2request.end() && it.key() == id) {
        (*it)->abort();
        it++;
    }
}

void FileTransferRequest::download(const QString& _url) {
    QUrl url(_url);
    QNetworkRequest request;

    if (!url.isValid()) {
        m_plugin->cb(m_ecId, "invalidUrl");
        return;
    }

    request.setUrl(url);
    if (url.password().size() || url.userName().size()) {
        QString headerData = "Basic " + (url.userName() + ":" + url.password()).toLocal8Bit().toBase64();
        request.setRawHeader("Authorization", headerData.toLocal8Bit());
    }
    m_reply = QSharedPointer<QNetworkReply>(m_manager.get(request));

    m_reply->connect(m_reply.data(), &QNetworkReply::finished, [&]() {
        if (m_reply->error() != QNetworkReply::NoError)
            return;

         m_plugin->cb(m_scId, m_reply->readAll().toBase64());

         emit done();
    });
    m_reply->connect(m_reply.data(), SIGNAL(error(QNetworkReply::NetworkError)), this, SLOT(error(QNetworkReply::NetworkError)));
    m_reply->connect(m_reply.data(), SIGNAL(downloadProgress(qint64, qint64)), this, SLOT(progress(qint64, qint64)));
}

void FileTransferRequest::upload(const QString& _url, const QString& content, QString fileKey, QString fileName, QString mimeType, const QVariantMap &params, const QVariantMap &headers) {
    QUrl url(_url);
    QNetworkRequest request;

    if (!url.isValid()) {
        m_plugin->cb(m_ecId, "invalidUrl");
        return;
    }

    request.setUrl(url);
    if (url.password().size() || url.userName().size()) {
        QString headerData = "Basic " + (url.userName() + ":" + url.password()).toLocal8Bit().toBase64();
        request.setRawHeader("Authorization", headerData.toLocal8Bit());
    }

    for (const QString &key: headers.keys()) {
        const QString &value = headers.find(key)->toString();
        request.setRawHeader(key.toUtf8(), value.toUtf8());
    }

    QString boundary = QString("CORDOVA-QT-%1A").arg(qrand());
    while (content.contains(boundary)) {
        boundary += QString("B%1A").arg(qrand());
    }

    request.setHeader(QNetworkRequest::ContentTypeHeader, QString("multipart/form-data; boundary=") + boundary);

    fileKey.replace("\"", "");
    fileName.replace("\"", "");
    mimeType.replace("\"", "");
    QString part = "--" + boundary + "\r\n";

    part += "Content-Disposition: form-data; name=\"" + fileKey +"\"; filename=\"" + fileName + "\"\r\n";
    part += "Content-Type: " + mimeType + "\r\n\r\n";
    part += content + "\r\n";

    for (QString key: params.keys()) {
        part += "--" + boundary + "\r\n";
        part += "Content-Disposition: form-data; name=\"" +  key + "\";\r\n\r\n";
        part += params.find(key)->toString();
        part += "\r\n";
    }

    part += QString("--") + boundary + "--" + "\r\n";

    m_reply = QSharedPointer<QNetworkReply>(m_manager.post(request, QByteArray(part.toUtf8())));

    m_reply->connect(m_reply.data(), &QNetworkReply::finished, [&]() {
        if (m_reply->error() != QNetworkReply::NoError)
            return;
        int status = 200;
        QVariant statusCode = m_reply->attribute(QNetworkRequest::HttpStatusCodeAttribute);

        if (statusCode.isValid()) {
            status = statusCode.toInt();
        }
        m_plugin->callback(m_scId, QString("%1, atob('%2')").arg(status).arg(QString(m_reply->readAll().toBase64())));

        emit done();
    });
    m_reply->connect(m_reply.data(), SIGNAL(error(QNetworkReply::NetworkError)), this, SLOT(error(QNetworkReply::NetworkError)));
    m_reply->connect(m_reply.data(), SIGNAL(uploadProgress(qint64, qint64)), this, SLOT(progress(qint64, qint64)));
}

void FileTransferRequest::abort() {
    m_plugin->cb(m_ecId, "abort");
    emit done();
}

void FileTransferRequest::error(QNetworkReply::NetworkError code) {
    Q_UNUSED(code);

    int status = 404;
    QVariant statusCode = m_reply->attribute(QNetworkRequest::HttpStatusCodeAttribute);

    if (statusCode.isValid()) {
        status = statusCode.toInt();
    }
    m_plugin->cb(m_ecId, status, m_reply->readAll().toBase64());
    emit done();
}

void FileTransferRequest::progress(qint64 bytesReceived, qint64 bytesTotal) {
    if (bytesReceived && bytesTotal)
        m_plugin->callbackWithoutRemove(m_progressId, QString("%1, %2").arg(bytesReceived).arg(bytesTotal));
}
