/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

/**
   \file cdw_write_wizard.c
   \brief Window with basic write options, displayed when user selects writing action

   The window displays message "You are writing to X disc", allows user to
   select writing speed, and writing mode. 'More options' button is also
   available, so user can jump to separate window to set more advanced
   options without closing write wizard.

   If current task is writing ISO image to blank CD/DVD, then 'Verify' checkbox
   is also available. It is coupled with task->burn.verify (where task
   is argument passed to cdw_write_wizard()).

   Writing speed values and modes that can be selected in wizard window are
   calculated every time the window is displayed and are disc-specific.
   They are calculated by functions defined in this file using values stored
   in disc argument. Be sure to call run_command_cdrecord_atip(),
   run_command_cdrecord_msinfo() or run_command_dvd_rw_mediainfo() before
   calling cdw_write_wizard().

   Code in this file does not check writing preconditions - be sure to check
   burning preconditions before calling cdw_write_wizard().
*/

#define _GNU_SOURCE /* asprintf() */
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>

#include "cdw_config.h"
#include "cdw_config_window.h"
#include "gettext.h"
#include "cdw_string.h"
#include "cdw_widgets.h"
#include "cdw_ncurses.h"
#include "cdw_window.h"
#include "cdw_write_wizard.h"
#include "cdw_main_window.h"
#include "cdw_debug.h"
#include "cdw_ext_tools.h"
#include "cdw_form.h"
#include "cdw_growisofs.h"
#include "cdw_growisofs_options.h"
#include "cdw_cdrecord.h"
#include "cdw_cdrecord_options.h"
#include "cdw_mkisofs_options.h"
#include "cdw_mkisofsrc.h"
#include "cdw_xorriso.h"
#include "cdw_xorriso_options.h"
#include "cdw_xorrisorc.h"
#include "cdw_utils.h"
#include "cdw_ofs.h"


enum field_ids {
	f_header_l = 0,
	f_volume_id_l,
	f_volume_id_i,
	f_speed_l,
	f_speed_dd,
	f_session_l,
	f_session_dd,
	f_disc_l,
	f_disc_dd,
	f_verify_l,
	f_verify_i,
	f_dummy_l,
	f_dummy_i,
	f_eject_l,
	f_eject_cb,
	f_other_options_b,
	f_write_b,
	f_cancel_b
};


static cdw_rv_t cdw_write_wizard_init(cdw_task_t *task, cdw_disc_t *disc);
static cdw_rv_t cdw_write_wizard_build(cdw_task_t *task, cdw_disc_t *disc);
static void     cdw_write_wizard_destroy(void);
static int      cdw_write_wizard_driver(void);
static cdw_rv_t cdw_write_wizard_build_fields(cdw_task_t *task);
static cdw_rv_t cdw_write_wizard_validate_and_save(cdw_task_t *task, int *fi);


static int      cdw_write_wizard_options_window_show(cdw_id_t *pid, int *fi);
static cdw_rv_t cdw_write_wizard_options_window_sub(cdw_task_t *task, cdw_id_t *pid, int *fi);
static int      cdw_write_wizard_options_window_by_callback(cdw_form_t *cdw_form, void *dummy);
static int      cdw_write_wizard_options_window_by_flow(cdw_id_t *pid, int *fi);
static int      cdw_write_wizard_options_window_driver(cdw_task_t *task);
static void     cdw_write_wizard_options_window_destroy(void);


static cdw_rv_t cdw_write_wizard_options_save(cdw_task_t *task);
static cdw_rv_t cdw_write_wizard_options_validate(cdw_task_t *task, cdw_id_t *page_id, int *fi);
static cdw_rv_t cdw_write_wizard_options_validate_closed(cdw_task_t *task);


static CDW_DROPDOWN *cdw_write_wizard_make_speed_dropdown(WINDOW *window, int begin_y, int begin_x, int width);
static CDW_DROPDOWN *cdw_write_wizard_make_session_mode_dropdown(WINDOW *window, int begin_y, int begin_x, int width);
static CDW_DROPDOWN *cdw_write_wizard_make_disc_mode_dropdown(WINDOW *window, int begin_y, int begin_x, int width);


static void cdw_write_wizard_debug_at_exit(cdw_task_t *task);
static void cdw_write_wizard_debug_verify_invisible(cdw_task_t *task, cdw_disc_t *disc);

static bool cdw_write_wizard_show_verify_checkbox(cdw_task_t *task, cdw_disc_t *disc);
static bool cdw_write_wizard_show_dummy_checkbox(cdw_task_t *task, cdw_disc_t *disc);


/* Does not include guard. */
#define CDW_WRITE_WIZARD_N_FIELDS 18


static struct {
	/* used in code related to 'Verify' checkbox, we can do verification
	   only for first track of data on CD*/
	bool verify_checkbox_visible;

	/* only some disc types allow "dummy" option; use
	   cdw_growisofs_allow_dummy() or cdw_cdrecord_allow_dummy()
	   to check if the checkbox should be visible */
	bool dummy_checkbox_visible;

	/* you can't pass dao/tao/sao parameter to xorriso, so
	   there is no point in showing the dropdown in wizard */
	bool disc_mode_dd_visible;

	bool volume_id_visible;

	char *header; /* main message in wizard window */

	cdw_form_t *cdw_form;

	cdw_task_t *task;

	cdw_disc_t *disc;

	FIELD *fields[CDW_WRITE_WIZARD_N_FIELDS + 1];

	/* Text displayed at the bottom of wizard window. */
	char *window_tip;

} wizard;



static cdw_form_dropdown_maker_t dropdown_makers[] = {
	cdw_write_wizard_make_speed_dropdown,
	cdw_write_wizard_make_session_mode_dropdown,
	cdw_write_wizard_make_disc_mode_dropdown
};



/* *** some layout constraints *** */

#define first_col                  2  /* main message and some labels start in leftmost column */

#define window_n_cols             52
#define window_n_lines            20
#define subwindow_n_cols           (window_n_cols - 2)
#define subwindow_n_lines          (window_n_lines - 2)
#define wide_label_n_cols          (subwindow_n_cols - 2)
#define narrow_label_n_cols        (subwindow_n_cols - 15)
#define wide_dropdown_n_cols       (subwindow_n_cols - 6)
#define narrow_dropdown_n_cols     8



/**
   \brief Ncurses window showing summary of prepared write process

   A window that shows 'You are writing to X' message and mode and speed of
   writing. Mode and speed can be adjusted in the window. User can press
   'Configuration' button to see and change other options. Two other
   buttons available are 'Write' and 'Cancel'.

   \param task - variable describing current task
   \param disc - variable describing disc currently in drive

   \return CDW_OK if user selects 'Write' button
   \return CDW_CANCEL if user presses Escape key or selects 'Cancel' button
   \return CDW_GEN_ERROR on errors
*/
cdw_rv_t cdw_write_wizard(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_assert (task->id == CDW_TASK_BURN_FROM_FILES || task->id == CDW_TASK_BURN_FROM_IMAGE,
		    "ERROR: incorrect task id %lld\n", task->id);
	cdw_assert (disc->type_writable == CDW_TRUE, "ERROR: disc type is not writable\n");

	wizard.task = task;
	wizard.disc = disc;
	wizard.header = (char *) NULL;

	if (disc->type_writable != CDW_TRUE) {
		cdw_vdm ("ERROR: disc type is not writable\n");
		return CDW_ERROR;
	}

	cdw_rv_t crv = cdw_write_wizard_init(task, disc);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to init wizard\n");
		cdw_write_wizard_destroy();
		return CDW_ERROR;
	}
	crv = cdw_write_wizard_build(task, disc);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to build wizard\n");
		cdw_write_wizard_destroy();
		return CDW_ERROR;
	}

	wrefresh(wizard.cdw_form->subwindow);

	int key = cdw_write_wizard_driver();

	cdw_write_wizard_destroy();
	/* redraw parent */
	cdw_main_window_wrefresh();

	if (key == CDW_KEY_ENTER) {
#ifndef NDEBUG
		cdw_write_wizard_debug_at_exit(task);
#endif
		if (wizard.disc_mode_dd_visible) {
			cdw_assert (task->burn.disc_mode != CDW_DISC_MODE_INIT, "ERROR: write wizard didn't set disc writing mode\n");
		}
		cdw_assert (task->burn.session_mode != CDW_SESSION_MODE_INIT, "ERROR: write wizard didn't set session writing mode\n");
		return CDW_OK;
	} else {
		cdw_sdm ("INFO: not attempting writing, not displaying settings\n");
		return CDW_CANCEL;
	}
}





/**
   \brief Set initial values of some variables global in the file

   Function sets values of various fields of "wizard" variable.
*/
cdw_rv_t cdw_write_wizard_init(cdw_task_t *task, cdw_disc_t *disc)
{
	wizard.verify_checkbox_visible = cdw_write_wizard_show_verify_checkbox(task, disc);
	wizard.dummy_checkbox_visible = cdw_write_wizard_show_dummy_checkbox(task, disc);
	wizard.disc_mode_dd_visible = task->burn.tool.id == CDW_TOOL_XORRISO ? false : true;
	wizard.volume_id_visible = task->id == CDW_TASK_BURN_FROM_FILES ? true : false;

	wizard.cdw_form = cdw_form_new(CDW_WRITE_WIZARD_N_FIELDS);
	if (!wizard.cdw_form) {
		cdw_vdm ("ERROR: failed to create cdw form\n");
		return CDW_ERROR;
	}

	wizard.cdw_form->fields = wizard.fields;
	for (int i = 0; i < CDW_WRITE_WIZARD_N_FIELDS; i++ ) {
		wizard.cdw_form->fields[i] = (FIELD *) NULL;
	}

	wizard.window_tip = (char *) NULL;

	return CDW_OK;
}





int cdw_write_wizard_driver(void)
{
	int key = 'a';
	int fi = f_cancel_b; /* Initial focus on "Cancel" button. */
	cdw_form_driver_focus_on(wizard.cdw_form, fi);

	while (key != CDW_KEY_ESCAPE && key != 'q' && key != 'Q') {
		key = cdw_form_driver_new(wizard.cdw_form, &fi);

		/* In top-level driver of wizard we are mainly
		   interested in ENTER being pressed on either "Write"
		   or "Cancel" button.  If user presses ENTER on
		   "Cancel" button, form driver will return ESCAPE
		   key. */
		if (key == CDW_KEY_ENTER) {
			if (fi != f_write_b) {
				/* We are not on "Write" button, so do nothing, wait for next key. */
				continue;
			}

			/* Flush. */
			form_driver(wizard.cdw_form->form, REQ_VALIDATION);

			/* Check options in closed "Options" window. */
			cdw_rv_t crv = cdw_write_wizard_options_validate_closed(wizard.task);
			if (crv != CDW_OK) {
				return CDW_KEY_ESCAPE;
			}

			/* Check options in main wizard window. */
			crv = cdw_write_wizard_validate_and_save(wizard.task, &fi);
			if (crv == CDW_NO) {
				/* one of option strings in main
				   wizard window is invalid */

				/* 2TRANS: this is title of dialog window */
				cdw_buttons_dialog(_("Error"),
						   /* 2TRANS: this is message in dialog window */
						   _("One of option fields is incorrect or contains character that is not allowed. Please fix it."),
						   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

				cdw_form_driver_go_to_field(wizard.cdw_form, fi);
				/* Loop. */
			} else {
				/* Fields from Options window and from
				   main page of the wizard are
				   validated and saved into task
				   variable. */
				return CDW_KEY_ENTER;
			}
		} else if (key == CDW_KEY_ESCAPE) {
			/* Either "Cancel" button pressed (which is
			   configured to return Escape), or "Escape"
			   key anywhere else. The result is the
			   same. */
			return CDW_KEY_ESCAPE;
		} else {
			/* Some unspecified key. */
		}
	}

	return CDW_KEY_ESCAPE;
}





/* labels for _all_ possible session modes allowed for any optical disc */
cdw_id_clabel_t all_session_modes[] = {
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_SESSION_MODE_START_MULTI,      gettext_noop("Start new, appendable disc") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_SESSION_MODE_CREATE_SINGLE,    gettext_noop("Create non-appendable disc") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
        { CDW_SESSION_MODE_CONTINUE_MULTI,   gettext_noop("Append data, don't close disc") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_SESSION_MODE_WRITE_FINAL,      gettext_noop("Append data and close disc") },
	/* 2TRANS: label visible in dropdown menu - some error occurred */
	{ CDW_SESSION_MODE_ERROR,            gettext_noop("(ERROR)") },
	{ -1,                                (char *) NULL } };


/**
   \brief Create dropdown with values showing allowed session modes

   Function returns dropdown with items presenting session modes acceptable
   by current disc.
   Location of the dropdown in wizard window is specified by variables
   global in this file.

   The function creates all necessary dropdown labels and calls
   cdw_dropdown_finalize() on the dropdown, so result of this function
   is a ready to use dropdown.

   \param task - variable describing current task

   \return NULL on errors
   \return pointer to dropdown on success
*/
CDW_DROPDOWN *cdw_write_wizard_make_session_mode_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	cdw_id_clabel_t items[CDW_SESSION_MODE_N_MAX];
	int n_items = 0;
	for (n_items = 0; n_items < CDW_SESSION_MODE_N_MAX; ) {
		/* session_modes is non-sparse up until
		   element of value CDW_SESSION_MODE_INIT */

		cdw_id_t allowed_mode_id = wizard.task->burn.session_modes[n_items];
		if (allowed_mode_id == CDW_SESSION_MODE_INIT) {
			break;
		} else {
			items[n_items].id = allowed_mode_id;
			items[n_items].label = cdw_utils_id_label_table_get_label(all_session_modes, allowed_mode_id);
			n_items++;
		}
	}

	CDW_DROPDOWN *dd = cdw_dropdown_maker_wrapper(window, begin_y, begin_x, width, n_items, items);
	/* 0 - first mode (set by "task" module) is the default one */
	cdw_dropdown_set_current_item_by_ind(dd, 0);
	return dd;
}





/* _all_ possible types of writing mode allowed for any optical disc */
cdw_id_clabel_t all_disc_modes[] = {
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_DISC_MODE_UNSPECIFIED,    gettext_noop("Unspecified") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_DISC_MODE_TAO,            gettext_noop("TAO") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_DISC_MODE_DAO,            gettext_noop("DAO") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_DISC_MODE_SAO,            gettext_noop("SAO") },
#if 0 /* unsupported disc modes */
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_DISC_MODE_RAW,            gettext_noop("RAW") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_DISC_MODE_RAW96P,         gettext_noop("RAW96R") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_DISC_MODE_RAW96R,         gettext_noop("RAW96P") },
	/* 2TRANS: label visible in dropdown menu - mode of writing data to optical disc */
	{ CDW_DISC_MODE_RAW16,          gettext_noop("RAW16") },
#endif
	/* 2TRANS: label visible in dropdown menu - some error occurred */
	{ CDW_DISC_MODE_ERROR,          gettext_noop("(ERROR)") },

	{ -1,                           "ERROR: guard value in \"disc mode labels\" table" }
};





/**
   \brief Create dropdown with values showing allowed disc modes

   Function returns dropdown with items presenting allowed (and supported
   by cdw) modes of writing to current disc.
   Location of the dropdown in wizard window is specified by variables
   global in this file.

   The function creates all necessary dropdown labels and calls
   cdw_dropdown_finalize() on the dropdown, so result of this function
   is a ready to use dropdown.

   \param task - variable describing current task

   \return NULL on errors
   \return pointer to dropdown on success
*/
CDW_DROPDOWN *cdw_write_wizard_make_disc_mode_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	cdw_id_clabel_t items[CDW_DISC_MODE_N_MODES];
	int n_items = 0;
	for (n_items = 0; n_items < CDW_DISC_MODE_N_MODES; ) {
		/* disc_modes is non-sparse up until
		   element of value CDW_DISC_MODE_INIT */

		cdw_id_t allowed_mode_id = wizard.task->burn.disc_modes[n_items];
		if (allowed_mode_id == CDW_DISC_MODE_INIT) {
			break;
		} else {
			items[n_items].id = allowed_mode_id;
			items[n_items].label = cdw_utils_id_label_table_get_label(all_disc_modes, allowed_mode_id);
			n_items++;
		}
	}

	CDW_DROPDOWN *dd = cdw_dropdown_maker_wrapper(window, begin_y, begin_x, width, n_items, items);
	/* 0 - first mode (set by "task" module) is the default one */
	cdw_dropdown_set_current_item_by_ind(dd, 0);
	return dd;
}





/**
   \brief Create dropdown with values showing allowed write speeds

   Function returns dropdown with items presenting write speeds acceptable
   by current disc.
   Location of the dropdown in wizard window is specified by variables
   global in this file.

   The function creates all necessary dropdown labels and calls
   cdw_dropdown_finalize() on the dropdown, so result of this function
   is a ready to use dropdown.

   \param disc - variable describing current disc

   \return NULL on errors
   \return pointer to dropdown on success
*/
CDW_DROPDOWN *cdw_write_wizard_make_speed_dropdown(WINDOW *window, int begin_y, int begin_x, int width)
{
	CDW_DROPDOWN *dropdown = cdw_dropdown_new(window,
						  begin_y,
						  begin_x,
						  width,
						  wizard.disc->write_speeds.n_speeds,  /* n_items_max */
						  CDW_COLORS_DIALOG);

	if (!dropdown) {
		cdw_vdm ("ERROR: failed to create new dropdown\n");
		return (CDW_DROPDOWN *) NULL;
	}

	for (int i = 0; i < wizard.disc->write_speeds.n_speeds; i++) {
		char label[4 + 1];
		snprintf(label, 4 + 1, "%d", wizard.disc->write_speeds.speeds[i]);
		cdw_rv_t crv = cdw_dropdown_add_item(dropdown, wizard.disc->write_speeds.speeds[i], label);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to create dropdown label #%d of value %d\n", i, wizard.disc->write_speeds.speeds[i]);
			cdw_dropdown_delete(&dropdown);
			return (CDW_DROPDOWN *) NULL;
		}
	}

	cdw_rv_t crv = cdw_dropdown_finalize(dropdown);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to finalize dropdown\n");
		cdw_dropdown_delete(&dropdown);
		return (CDW_DROPDOWN *) NULL;
	} else {
		cdw_dropdown_set_current_item_by_id(dropdown, wizard.task->burn.speed_id);
		return dropdown;
	}
}





/**
   \brief Deallocate all resources allocated by write wizard
*/
void cdw_write_wizard_destroy(void)
{
	/* This order of these four function calls minimizes number of
	   problems reported by valgrind. */
	cdw_form_delete_form_objects(wizard.cdw_form);
	cdw_window_delete(&wizard.cdw_form->subwindow);
	cdw_window_delete(&wizard.cdw_form->window);
	cdw_form_delete(&wizard.cdw_form);

	cdw_string_delete(&wizard.header);

	return;
}





/**
   \brief Print to stderr values of \p task fields set in write wizard

   \param task - task variable, in which some fields were set in write wizard
*/
void cdw_write_wizard_debug_at_exit(cdw_task_t *task)
{
	/* use "task" variable to which wizard saves result at return,
	   not values from dropdowns/checkboxes */

	cdw_vdm ("INFO: disc mode dropdown is %svisible\n", wizard.disc_mode_dd_visible ? "" : "not ");
	cdw_vdm ("INFO: volume id field is %svisible\n", wizard.volume_id_visible ? "" : "not ");

	cdw_write_debug_print_options(&task->burn);
	cdw_task_debug_print_options(task, "at write wizard exit");

	return;
}





bool cdw_write_wizard_show_verify_checkbox(cdw_task_t *task, cdw_disc_t *disc)
{
	bool disc_blank = disc->state_empty == CDW_TRUE;
	bool disc_overwritable = disc->type == CDW_DVD_RWP
		|| disc->type == CDW_DVD_RW_RES;
	bool tool_available = cdw_ext_tools_is_digest_tool(task->calculate_digest.tool.id);

	bool is_iso9660_file = cdw_ofs_is_iso9660_file(task->image_file_fullpath);

	if ((disc_blank || disc_overwritable)
	    && task->id == CDW_TASK_BURN_FROM_IMAGE
	    && tool_available
	    && is_iso9660_file) { /* Right now cdw can read from optical disc only ISO9660 FS. */

		return true;
	} else {
		return false;
	}
}





/* for some combinations of burn tool and disc type
   dummy burning is just not possible */
bool cdw_write_wizard_show_dummy_checkbox(cdw_task_t *task, cdw_disc_t *disc)
{
	if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
		return cdw_cdrecord_allow_dummy(disc);
	} else if (task->burn.tool.id == CDW_TOOL_GROWISOFS) {
		return cdw_growisofs_allow_dummy(disc);
	} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
		return cdw_xorriso_allow_dummy(disc);
	} else {
		return false;
	}
}





/**
   \brief Print to stderr information why "verify" checkbox is not visible

   If some conditions are met, write wizard may display "verify write"
   checkbox. This function should be called if any of the conditions is
   not met to investigate these conditions and print to stderr information
   which of them was not met. This is a debug function

   \param task - variable describing current task
   \param disc - variable describing current disc
*/
void cdw_write_wizard_debug_verify_invisible(cdw_task_t *task, cdw_disc_t *disc)
{
	if (disc->state_empty != CDW_TRUE) {
		cdw_vdm ("INFO: \"verify\" checkbox is not visible because disc is not blank (according to cdio)\n");
	}
	if (disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_DVD) {
		if (! (disc->type == CDW_DVD_RWP || disc->type == CDW_DVD_RW_RES)) {
			cdw_vdm ("INFO: \"verify\" checkbox is not visible because disc is not DVD+RW or DVD-RW RES\n");
		}
	}
	if (task->id != CDW_TASK_BURN_FROM_IMAGE) {
		cdw_vdm ("INFO: \"verify\" checkbox is not visible because task is not CDW_TASK_BURN_FROM_IMAGE\n");
	}
	if (!cdw_ext_tools_digest_tool_available()) {
		cdw_vdm ("INFO: \"verify\" checkbox is not visible because there is no digest tool available\n");
	}
	if (!cdw_ext_tools_is_digest_tool(task->calculate_digest.tool.id)) {
		cdw_vdm ("INFO: \"verify\" checkbox is not visible because digest tool is not set (is %lld)\n", task->calculate_digest.tool.id);
	}
	if (!cdw_ofs_is_iso9660_file(task->image_file_fullpath)) {
		cdw_vdm ("INFO: \"verify\" checkbox is not visible because image file %s is not in supported format\n", task->image_file_fullpath);
	}

	return;
}





/**
   \brief Create all UI elements in wizard window

   \param task - variable describing current task
   \param disc - variable describing disc currently in drive

   \return CDW_OK on success
   \return CDW_ERROR on failure
*/
cdw_rv_t cdw_write_wizard_build(cdw_task_t *task, cdw_disc_t *disc)
{
	int begin_y = ((LINES - window_n_lines) / 2);
	int begin_x = (COLS - window_n_cols) / 2;
	wizard.cdw_form->window = cdw_window_new((WINDOW *) NULL,
						 window_n_lines, window_n_cols,
						 begin_y, begin_x,
						 CDW_COLORS_DIALOG,
						 /* 2TRANS: this is title of wizard window; 'write' as in
						    'writing to optical disc */
						 _("Write wizard (ISO9660)"),
						 /* 2TRANS: this is tip at the bottom of window - user can
						    switch between window elements using tab key */
						 _("Use 'Tab' key to move"));
	if (!wizard.cdw_form->window) {
		cdw_vdm ("ERROR: failed to create window\n");
		return CDW_ERROR;
	}

	wizard.cdw_form->subwindow = cdw_window_new(wizard.cdw_form->window,
						    subwindow_n_lines, subwindow_n_cols,
						    1, 1,
						    CDW_COLORS_DIALOG, (char *) NULL, (char *) NULL);

	if (!wizard.cdw_form->subwindow) {
		cdw_vdm ("ERROR: failed to create subwindow\n");
		return CDW_ERROR;
	}

	/* 2TRANS: this is message to user: current action will
	   be writing to %s, where %s is DVD or CD */
	int rv = asprintf(&(wizard.header), _("You are writing to %s disc"), disc->cdio->simple_type_label);
	if (rv == -1) {
		cdw_vdm ("ERROR: failed to create top message\n");
		return CDW_ERROR;
	}

	cdw_rv_t crv = cdw_write_wizard_build_fields(task);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to build fields\n");
		return CDW_ERROR;
	}

	wizard.cdw_form->form = cdw_ncurses_new_form(wizard.cdw_form->window,
						     wizard.cdw_form->subwindow,
						     wizard.cdw_form->fields);
	if (!wizard.cdw_form->form) {
		cdw_vdm ("ERROR: failed to create form\n");
		return CDW_ERROR;
	}
	set_current_field(wizard.cdw_form->form, wizard.cdw_form->fields[f_cancel_b]);

	if (!wizard.volume_id_visible) {
		field_opts_off(wizard.cdw_form->fields[f_volume_id_l], O_VISIBLE);
		field_opts_off(wizard.cdw_form->fields[f_volume_id_i], O_VISIBLE);
	}

	CDW_CHECKBOX *cb = cdw_form_get_checkbox(wizard.cdw_form, f_verify_i);
	if (wizard.verify_checkbox_visible) { /* this last one is added just in case */
		cdw_checkbox_set_state(cb, task->burn.verify);
	} else {
		cdw_checkbox_set_visibility(cb, false);
		/* invisible AND inactive */
		field_opts_off(wizard.cdw_form->fields[f_verify_i], O_VISIBLE);
		field_opts_off(wizard.cdw_form->fields[f_verify_l], O_VISIBLE);

#ifndef NDEBUG
		cdw_write_wizard_debug_verify_invisible(task, disc);
#endif
	}

	cb = cdw_form_get_checkbox(wizard.cdw_form, f_dummy_i);
	if (wizard.dummy_checkbox_visible) {
		cdw_checkbox_set_state(cb, task->burn.dummy);
	} else {
		cdw_checkbox_set_visibility(cb, false);
		/* invisible AND inactive */
		field_opts_off(wizard.cdw_form->fields[f_dummy_i], O_VISIBLE);
		field_opts_off(wizard.cdw_form->fields[f_dummy_l], O_VISIBLE);

#ifndef NDEBUG
		// TODO: to be implemented: cdw_write_wizard_debug_dummy_invisible(task, disc);
#endif
	}
	if (!wizard.disc_mode_dd_visible) {
		CDW_DROPDOWN *dd = cdw_form_get_dropdown(wizard.cdw_form, f_disc_dd);
		cdw_dropdown_make_invisible(dd);
		field_opts_off(wizard.cdw_form->fields[f_disc_l], O_VISIBLE);
		field_opts_off(wizard.cdw_form->fields[f_disc_dd], O_VISIBLE);
	}

	/* current items are set in functions creating respective
	   dropdowns; now just display them */
	cdw_form_redraw_widgets(wizard.cdw_form);

	cdw_form_add_return_chars(wizard.cdw_form, CDW_KEY_ENTER, CDW_KEY_ESCAPE, 'q', 'Q', 0);

	wrefresh(wizard.cdw_form->subwindow);
	wrefresh(wizard.cdw_form->window);

	return CDW_OK;
}





cdw_rv_t cdw_write_wizard_build_fields(cdw_task_t *task)
{
	/* 2TRANS: this is a button label of a button that opens configuration window */
	char *other_options = _("Other options");
	int other_options_n_cols = (int) strlen(other_options) + 2;

	cdw_form_descr_t descr[] = {
		/*     type                   begin_y   begin_x         n_cols                n_lines    field enum      data1                                 data2 */

		{ CDW_WIDGET_ID_STATIC_LABEL,       1,  first_col,      wide_label_n_cols,          1,   f_header_l,     wizard.header,                            0 },

		/* 2TRANS: this is label next to input field, where user can enter a label describing created ISO9660 volume */
		{ CDW_WIDGET_ID_DYNAMIC_LABEL,      2,  first_col,      wide_label_n_cols,          1,   f_volume_id_l,  _("Volume ID:"),                          0 },
		{ CDW_WIDGET_ID_SAFE_INPUT_LINE,    3,  first_col + 3,  CDW_ISO9660_VOLI_LEN,       1,   f_volume_id_i,  task->create_image.iso9660.volume_id,  CDW_ISO9660_VOLI_LEN },

		/* 2TRANS: this is a label in write wizard, after which a writing speed selection list will be displayed */
		{ CDW_WIDGET_ID_STATIC_LABEL,       4,  first_col,      wide_label_n_cols,          1,   f_speed_l,      _("Writing speed:"),                      0 },
		{ CDW_WIDGET_ID_DROPDOWN,           5,  first_col + 3,  narrow_dropdown_n_cols - 2, 1,   f_speed_dd,     dropdown_makers,                          0 },

		/* 2TRANS: this is a label in write wizard, after which a session writing mode selection list will be displayed */
		{ CDW_WIDGET_ID_STATIC_LABEL,       6,  first_col,      wide_label_n_cols,          1,   f_session_l,    _("Session writing mode:"),               0 },
		{ CDW_WIDGET_ID_DROPDOWN,           7,  first_col + 3,  wide_dropdown_n_cols - 2,   1,   f_session_dd,   dropdown_makers,                          1 },

		/* 2TRANS: this is a label in write wizard, after which a disc writing mode selection list will be displayed */
		{ CDW_WIDGET_ID_STATIC_LABEL,       8,  first_col,      wide_label_n_cols,          1,   f_disc_l,       _("Disc writing mode:"),                  0 },
		{ CDW_WIDGET_ID_DROPDOWN,           9,  first_col + 3,  wide_dropdown_n_cols - 2,   1,   f_disc_dd,      dropdown_makers,                          2 },

		/* 2TRANS: this is label next to checkbox; marked checkbox enables verification of correctness of writing to CD; this feature is experimental */
		{ CDW_WIDGET_ID_STATIC_LABEL,      11,  first_col + 4,  narrow_label_n_cols,        1,   f_verify_l,     _("Verify write (experimental)"),         0 },
		{ CDW_WIDGET_ID_CHECKBOX,          11,  first_col + 1,  1,                          1,   f_verify_i,     (void *) NULL,    task->burn.verify ? 1 : 0 },

		/* 2TRANS: this is checkbox label: keep no longer than original; "Dummy write" means attempting to write
		   to a disc and performing all normal operations (just for a try), but without actual burning */
		{ CDW_WIDGET_ID_STATIC_LABEL,      12,  first_col + 4,  narrow_label_n_cols,        1,   f_dummy_l,      _("Dummy write"),                         0 },
		{ CDW_WIDGET_ID_CHECKBOX,          12,  first_col + 1,  1,                          1,   f_dummy_i,      (void *) NULL,     task->burn.dummy ? 1 : 0 },

		/* 2TRANS: this is a checkbox label; "tray" is a tray of optical drive; "writing" is burning to optical disc */
		{ CDW_WIDGET_ID_STATIC_LABEL,      13,  first_col + 4,  narrow_label_n_cols,        1,   f_eject_l,      _("Eject tray after writing"),            0 },
		{ CDW_WIDGET_ID_CHECKBOX,          13,  first_col + 1,  1,                          1,   f_eject_cb,     (void *) NULL,     task->burn.eject ? 1 : 0 },

		/* 2TRANS: button label */
		{ CDW_WIDGET_ID_BUTTON,            14,  first_col,      other_options_n_cols,       1,   f_other_options_b,   other_options,       CDW_COLORS_DIALOG },

		/* 2TRANS: button label, it refers to writing to optical disc */
		{ CDW_WIDGET_ID_BUTTON,            16,   3,             2,                          1,   f_write_b,      _("Write"),               CDW_COLORS_DIALOG },
		/* 2TRANS: button label */
		{ CDW_WIDGET_ID_BUTTON,            16,  15,             2,                          1,   f_cancel_b,     _("Cancel"),              CDW_COLORS_DIALOG },

		/* guard */
		{ -1,                               0,   0,             0,                          0,   0,              (void *) NULL,                            0 }};

	wizard.cdw_form->n_fields = CDW_WRITE_WIZARD_N_FIELDS;
	wizard.cdw_form->fields = wizard.fields;

	/* the function adds guard at the end of fields */
	cdw_rv_t crv = cdw_form_description_to_fields(descr, wizard.cdw_form);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to convert form description to form\n");
		return CDW_ERROR;
	} else {
		cdw_form_set_widget_callback(wizard.cdw_form, f_other_options_b, cdw_write_wizard_options_window_by_callback);

		cdw_form_set_button_return_key(wizard.cdw_form, f_write_b, CDW_KEY_ENTER);
		cdw_form_set_button_return_key(wizard.cdw_form, f_cancel_b, CDW_KEY_ESCAPE);

		cdw_form_bind_input_and_label(wizard.cdw_form, f_volume_id_i, f_volume_id_l);
		return CDW_OK;
	}
}





/* some size and layout constants */
/* width and height of main options window can't be smaller
   than size of minimal supported terminal */

/** \brief Width of options window */
#define OPTIONS_WIDTH 80
/** \brief Height of options window */
#define OPTIONS_HEIGHT 24
/** \brief Width of right-hand area with tabs */
#define TABS_WIDTH 23
/** \brief Width of (most) labels in options window */
#define LABEL_WIDTH 30
/** \brief Width of some fields that should be as wide as possible: input
    fields storing file path, some dropdowns, "other options" fields,
    txt subwins, long labels */
#define WIDE_FIELD_WIDTH (OPTIONS_WIDTH - TABS_WIDTH - 4)
/** \brief Window column of first column of items in options page  */
#define FIRST_COL 1
/** \brief Window column of second column of items in options page  */
#define SECOND_COL (FIRST_COL + LABEL_WIDTH + 2)



enum {
	WRITE_CDRECORD = 0,
	WRITE_GROWISOFS,
	WRITE_XORRISO_BURN,
	WRITE_MKISOFS_ISO_MAIN,
	WRITE_XORRISO_ISO_MAIN,
	WRITE_MKISOFS_ISO_META,
	WRITE_XORRISO_ISO_META
};


/* this is first main component of options window: a big table
   aggregating data and functions that prepare the data; the data
   represents widgets displayed in pages of options window */
static cdw_options_page_t c_pages[] = {
	/* 2TRANS: label of a tab in configuration window with options related burning */
	{ false, CDW_CDRECORD_OPTIONS_N_FIELDS,          cdw_cdrecord_options_form,        gettext_noop("Write"),   (cdw_form_t *) NULL },
	/* 2TRANS: label of a tab in configuration window with options related burning */
	{ false, CDW_GROWISOFS_OPTIONS_N_FIELDS,         cdw_growisofs_options_form,       gettext_noop("Write"),   (cdw_form_t *) NULL },
	/* 2TRANS: label of a tab in configuration window with options related burning */
	{ false, CDW_XORRISO_BURN_OPTIONS_N_FIELDS,      cdw_xorriso_burn_options_form,    gettext_noop("Write"),   (cdw_form_t *) NULL },

	/* 2TRANS: label of a tab in configuration window with options related to ISO9660; probably you don't want to translate this */
	{ false, CDW_MKISOFS_ISO_MAIN_OPTIONS_N_FIELDS,  cdw_mkisofs_options_form,         gettext_noop("ISO9660"), (cdw_form_t *) NULL },
	/* 2TRANS: label of a tab in configuration window with options related to ISO9660; probably you don't want to translate this */
	{ false, CDW_XORRISO_ISO_MAIN_OPTIONS_N_FIELDS,  cdw_xorriso_iso_options_form,     gettext_noop("ISO9660"), (cdw_form_t *) NULL },

	/* 2TRANS: label of a tab in configuration window with options related to meta information of ISO9660 file system */
	{ false, CDW_MKISOFS_ISO_META_OPTIONS_N_FIELDS,  cdw_mkisofsrc_options_form,       gettext_noop("Meta data"), (cdw_form_t *) NULL },
	/* 2TRANS: label of a tab in configuration window with options related to meta information of ISO9660 file system */
	{ false, CDW_XORRISO_ISO_META_OPTIONS_N_FIELDS,  cdw_xorrisorc_options_form,       gettext_noop("Meta data"), (cdw_form_t *) NULL },
};


/* this is a pointer to second main component of options window:
   tabbed window, which enables switching between pages */
static cdw_tabs_window_t *c_twindow = (cdw_tabs_window_t *) NULL;





int cdw_write_wizard_options_window_by_callback(__attribute__((unused)) cdw_form_t *cdw_form, __attribute__((unused)) void *dummy)
{
	return cdw_write_wizard_options_window_show((cdw_id_t *) NULL, (int *) NULL);
}





int cdw_write_wizard_options_window_by_flow(cdw_id_t *pid, int *fi)
{
	return cdw_write_wizard_options_window_show(pid, fi);
}






int cdw_write_wizard_options_window_show(cdw_id_t *pid, int *fi)
{
	cdw_rv_t crv = cdw_write_wizard_options_window_sub(wizard.task, pid, fi);

	cdw_main_window_wrefresh();
	redrawwin(wizard.cdw_form->window);
	redrawwin(wizard.cdw_form->subwindow);
	wrefresh(wizard.cdw_form->window);
	wrefresh(wizard.cdw_form->subwindow);

	/* the function needs to return int representing a char */
	if (crv == CDW_OK) {
		return '1';
	} else {
		return '0';
	}
}





cdw_rv_t cdw_write_wizard_options_window_sub(cdw_task_t *task, cdw_id_t *pid, int *fi)
{
	/* 2TRANS: this is message at the bottom of options window;
	   cancel means: quit without saving; '%d' is an integer used to
	   create label of function key, e.g. "F10" */
	int rv = asprintf(&wizard.window_tip,
			  _("Press F%d / F%d key to save changes or ESC to cancel"),
			  CDW_CONFIG_UI_SnC_KEY_B, CDW_CONFIG_UI_SnC_KEY_A);
	if (!wizard.window_tip || rv == -1) {
		cdw_vdm ("ERROR: failed to create window tip\n");
		return CDW_ERROR;
	}
	int n_tabs = 0;
	if (task->id == CDW_TASK_BURN_FROM_FILES) {
		n_tabs = 3; /* (cdrecord|growisofs|xorriso) + (basic ISO9660 by mkisofs|xorriso) + (ISO9660 metadata)  */
	} else { /* CDW_TASK_BURN_FROM_IMAGE */
		n_tabs = 1; /* (cdrecord|growisofs|xorriso) */
	}
	/* 2TRANS: this is title of cdw options main window */
	c_twindow = cdw_tabs_window_init(n_tabs, _("Write options"), wizard.window_tip);
	cdw_tabs_window_set_geometry(c_twindow, OPTIONS_HEIGHT, OPTIONS_WIDTH, 4, 4, TABS_WIDTH);


	/* prepare tabbed window */

	cdw_id_t id = 0;
	int ind = 0;

	/* Tool-specific (cdrecord/growisofs/xorriso) options for
	   burning. These should be always present, whether we are
	   writing files on-the-fly, or burning existing image to
	   disc. */
	if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
		id = WRITE_CDRECORD;
		cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
	} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
		id = WRITE_XORRISO_BURN;
		cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
	} else {
		id = WRITE_GROWISOFS;
		cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
	}

	/* Options of tool creating ISO9660 file system on the fly.
	   The tool may be either mkisofs or xorriso, and there are
	   two pages for each of these tools (basic options + metadata
	   options). */
	if (task->id == CDW_TASK_BURN_FROM_FILES) {
		if (task->burn.tool.id == CDW_TOOL_CDRECORD || task->burn.tool.id == CDW_TOOL_GROWISOFS) {
			cdw_assert (task->create_image.tool.id == CDW_TOOL_MKISOFS, "ERROR: invalid ISO tool: %lld\n", task->create_image.tool.id);

			/* Basic mkisofs options for ISO9660 fs. */
			ind++;
			id = WRITE_MKISOFS_ISO_MAIN;
			cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);

			/* mkisofs options for ISO9660 fs metadata. */
			ind++;
			id = WRITE_MKISOFS_ISO_META;
			cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);

		} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
			cdw_assert (task->create_image.tool.id == CDW_TOOL_XORRISO, "ERROR: invalid ISO tool: %lld\n", task->create_image.tool.id);

			/* Basic xorriso options for ISO9660 fs. */
			ind++;
			id = WRITE_XORRISO_ISO_MAIN;
			cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);

			/* xorriso options for ISO9660 fs metadata. */
			ind++;
			id = WRITE_XORRISO_ISO_META;
			cdw_tabs_window_append_tab(c_twindow, id, KEY_F(ind + 2), c_pages[id].label);
		} else {
			cdw_assert (0, "ERROR: invalid burn tool: %lld\n", task->burn.tool.id);
		}
	}


	cdw_tabs_window_add_return_keys(c_twindow, KEY_F(CDW_CONFIG_UI_SnC_KEY_A), KEY_F(CDW_CONFIG_UI_SnC_KEY_B),
					CDW_KEY_ESCAPE, 'q', 'Q', 0);
	c_twindow->driver_reader = cdw_config_ui_tabs_window_form_reader;
	cdw_tabs_window_finalize(c_twindow);

	/* prepare forms in the tabbed window */
	for (int i = 0; i < n_tabs; i++) {
		id = c_twindow->tabs[i].id;

		c_pages[id].cdw_form = cdw_form_new(c_pages[id].n_fields);
		c_pages[id].cdw_form->form_id = id;
		c_pages[id].cdw_form->window = c_twindow->tabs[i].window;
		c_pages[id].cdw_form->subwindow = c_twindow->tabs[i].subwindow;

		c_pages[id].visible = true;

		cdw_rv_t crv;
		if (id == WRITE_MKISOFS_ISO_MAIN
		    || id == WRITE_MKISOFS_ISO_META
		    || id == WRITE_XORRISO_ISO_MAIN
		    || id == WRITE_XORRISO_ISO_META) {

			crv = c_pages[id].fields_builder(c_pages[id].cdw_form, &(task->create_image.iso9660), FIRST_COL, SECOND_COL, WIDE_FIELD_WIDTH, LABEL_WIDTH);
		} else {
			crv = c_pages[id].fields_builder(c_pages[id].cdw_form, task, FIRST_COL, SECOND_COL, WIDE_FIELD_WIDTH, LABEL_WIDTH);
		}

		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to build fields in page #%d\n", i);
			return CDW_ERROR;
		}

		c_pages[id].cdw_form->form = cdw_ncurses_new_form(c_pages[id].cdw_form->window,
								  c_pages[id].cdw_form->subwindow,
								  c_pages[id].cdw_form->fields);

		for (int j = 0; j < n_tabs; j++) {
			cdw_form_add_return_char(c_pages[id].cdw_form, KEY_F(j + CDW_CONFIG_UI_FX_START_KEY));
		}
		cdw_form_add_return_char(c_pages[id].cdw_form, KEY_F(CDW_CONFIG_UI_SnC_KEY_A));
		cdw_form_add_return_char(c_pages[id].cdw_form, KEY_F(CDW_CONFIG_UI_SnC_KEY_B));
		cdw_form_add_return_char(c_pages[id].cdw_form, 'q'); /* will work only for non-text fields */
	}
	c_twindow->user_data = (void *) c_pages;

	cdw_tabs_window_draw_tabs(c_twindow);
	cdw_config_ui_tabs_window_form_draw(c_twindow);

	if (pid && fi
	    && *pid != -1 && *fi != -1) {

		cdw_tabs_window_show_tab_by_id(c_twindow, *pid);
		cdw_form_driver_go_to_field(c_pages[*pid].cdw_form, *fi);
	}

	int key = cdw_write_wizard_options_window_driver(task);
	cdw_write_wizard_options_window_destroy();
	if (key == CDW_KEY_ENTER) {
		return CDW_OK;
	} else {
		return CDW_CANCEL;
	}

	return CDW_OK;
}





/**
   \brief Destructor function for write options window
*/
void cdw_write_wizard_options_window_destroy(void)
{
	if (wizard.window_tip) {
		free(wizard.window_tip);
		wizard.window_tip = (char *) NULL;
	}

	for (int i = 0; i < c_twindow->n_tabs; i++) {
		cdw_id_t id = c_twindow->tabs[i].id;
		if (c_pages[id].cdw_form) {
			cdw_form_delete_form_objects(c_pages[id].cdw_form);
		}
		if (c_pages[id].cdw_form) {
			cdw_form_delete(&(c_pages[id].cdw_form));
		}
	}

	cdw_tabs_window_delete(&c_twindow);

	return;
}





/**
   \brief Function reacting to keys pressed when options window is displayed

   Function catches all keys pressed when options window is displayed,
   and reacts to them either by switching pages, or by passing the keys to
   page driver(s).

   Keys configured as hotkeys for "tabbed window" tabs are used to switch
   between pages. F(CDW_CONFIG_SnC_KEY) key is interpreted as "save and close"
   key - values of all option fields are stored in \p config and validated.
   Function then returns CDW_OK if validation is successful, or displays
   incorrect field to user.
   ESCAPE key causes the function to return with CDW_CANCEL.

   \param config - variable in which function saves values of all option fields

   \return CDW_KEY_ESCAPE if user pressed ESCAPE key in options window
   \return CDW_KEY_ENTER if user pressed F10 and validation of \p tmp config was successful
*/
int cdw_write_wizard_options_window_driver(cdw_task_t *task)
{
	while (1) {
		int key = cdw_tabs_window_driver(c_twindow);
		if (key == CDW_KEY_ESCAPE || key == 'q' || key == 'Q') {
			break;
		} else if (key == KEY_F(CDW_CONFIG_UI_SnC_KEY_A)
			   || key == KEY_F(CDW_CONFIG_UI_SnC_KEY_B)) { /* SnC = Save and Close */

			/* flush */
			for (int i = 0; i < c_twindow->n_tabs; i++) {
				cdw_id_t id = c_twindow->tabs[i].id;
				form_driver(c_pages[id].cdw_form->form, REQ_VALIDATION);
			}
			cdw_id_t page_id = 0;
			int fi = 0;
			cdw_write_wizard_options_save(task);
			cdw_rv_t valid = cdw_write_wizard_options_validate(task, &page_id, &fi);
			if (valid == CDW_NO) { /* some option field is invalid */

				/* 2TRANS: this is title of dialog window */
				cdw_buttons_dialog(_("Error"),
						   /* 2TRANS: this is message in dialog window */
						   _("One of option fields is incorrect or contains character that is not allowed. Please fix it."),
						   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

				cdw_tabs_window_show_tab_by_id(c_twindow, page_id);
				cdw_form_driver_go_to_field(c_pages[page_id].cdw_form, fi);

				/* loop */
			} else {
				/* options from window with advanced
				   options are validated and saved into
				   task variable */
				return CDW_KEY_ENTER;
			}
		} else {
			/* loop */
		}
	} /* while (1) */

	return CDW_KEY_ESCAPE;
}





cdw_rv_t cdw_write_wizard_options_validate(cdw_task_t *task, cdw_id_t *pid, int *fi)
{
	cdw_rv_t crv = CDW_NO;

	if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
		*pid = WRITE_CDRECORD;
		crv = cdw_cdrecord_options_validate(task, fi);
		if (crv == CDW_NO) {
			return CDW_NO;
		}
	} else if (task->burn.tool.id == CDW_TOOL_GROWISOFS) {
		*pid = WRITE_GROWISOFS;
		crv = cdw_growisofs_options_validate(task, fi);
		if (crv == CDW_NO) {
			return CDW_NO;
		}
	} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
		*pid = WRITE_XORRISO_BURN;
		crv = cdw_xorriso_burn_options_validate(task, fi);
		if (crv == CDW_NO) {
			return CDW_NO;
		}
	} else {
		cdw_assert (0, "ERROR: invalid tool id for task \"burn\": %s (%lld)\n",
			    cdw_ext_tools_get_tool_name(task->burn.tool.id), task->burn.tool.id);
		return CDW_NO;
	}


	if (task->id != CDW_TASK_BURN_FROM_FILES) {
		/* Don't validate options for creating ISO9660 fs
		   on-the-fly. */

		return CDW_OK;
	}


	if (task->create_image.tool.id == CDW_TOOL_MKISOFS) {
		*pid = WRITE_MKISOFS_ISO_MAIN;
		crv = cdw_mkisofs_options_validate(&(task->create_image.iso9660), fi);
		if (crv == CDW_NO) {
			return CDW_NO;
		}

		*pid = WRITE_MKISOFS_ISO_META;
		crv = cdw_mkisofsrc_options_validate(&(task->create_image.iso9660), fi);
		if (crv == CDW_NO) {
			return CDW_NO;
		}

	} else if (task->create_image.tool.id == CDW_TOOL_XORRISO) {
		*pid = WRITE_XORRISO_ISO_MAIN;
		crv = cdw_xorriso_iso_options_validate(&(task->create_image.iso9660), fi);
		if (crv == CDW_NO) {
			return CDW_NO;
		}

		*pid = WRITE_XORRISO_ISO_META;
		crv = cdw_xorrisorc_options_validate(&(task->create_image.iso9660), fi);
		if (crv == CDW_NO) {
			return CDW_NO;
		}
	} else {
		cdw_assert (0, "ERROR: invalid tool id for task \"create image\": %s (%lld)\n",
			    cdw_ext_tools_get_tool_name(task->create_image.tool.id), task->create_image.tool.id);
	}

	return CDW_OK;
}





cdw_rv_t cdw_write_wizard_options_save(cdw_task_t *task)
{
	cdw_id_t pid = 0;
	cdw_form_t *cdw_form = (cdw_form_t *) NULL;

	if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
		pid = WRITE_CDRECORD;

		cdw_form = c_pages[pid].cdw_form;
		cdw_cdrecord_options_save(cdw_form, task);

	} else if (task->burn.tool.id == CDW_TOOL_GROWISOFS) {
		pid = WRITE_GROWISOFS;

		cdw_form = c_pages[pid].cdw_form;
		cdw_growisofs_options_save(cdw_form, task);

	} else if (task->burn.tool.id == CDW_TOOL_XORRISO) {
		pid = WRITE_XORRISO_BURN;

		cdw_form = c_pages[pid].cdw_form;
		cdw_xorriso_burn_options_save(cdw_form, task);

	} else {
		cdw_assert (0, "ERROR: invalid tool id for task \"burn\": %s (%lld)\n",
			    cdw_ext_tools_get_tool_name(task->burn.tool.id), task->burn.tool.id);
		return CDW_NO;
	}


	if (task->id != CDW_TASK_BURN_FROM_FILES) {
		/* Don't save options for creating ISO9660 fs
		   on-the-fly. */

		return CDW_OK;
	}


	if (task->create_image.tool.id == CDW_TOOL_MKISOFS) {
		pid = WRITE_MKISOFS_ISO_MAIN;

		cdw_form = c_pages[pid].cdw_form;
		cdw_mkisofs_options_save(cdw_form, &(task->create_image.iso9660));

		pid = WRITE_MKISOFS_ISO_META;

		cdw_form = c_pages[pid].cdw_form;
		cdw_mkisofsrc_options_save(cdw_form, &(task->create_image.iso9660));

	} else if (task->create_image.tool.id == CDW_TOOL_XORRISO) {
		pid = WRITE_XORRISO_ISO_MAIN;

		cdw_form = c_pages[pid].cdw_form;
		cdw_xorriso_iso_options_save(cdw_form, &(task->create_image.iso9660));

		pid = WRITE_XORRISO_ISO_META;

		cdw_form = c_pages[pid].cdw_form;
		cdw_xorrisorc_options_save(cdw_form, &(task->create_image.iso9660));

	} else {
		cdw_assert (0, "ERROR: invalid tool id for task \"create image\": %s (%lld)\n",
			    cdw_ext_tools_get_tool_name(task->create_image.tool.id), task->create_image.tool.id);
	}

	return CDW_OK;
}




/**
   \brief Check options for burning

   Check options used to burn a disc that *would* be displayed in
   options window. The function should be called when "Options" window
   has not been displayed, i.e. when user pressed "Write" button in
   wizard without visiting "Options" window.

   Some options in such non-visited window may be invalid, this
   function should detect this.

   First it tests contents of task variable, and if there is anything
   wrong with any of the options related to writing, it creates
   and display options window, with cursor placed in problematic
   field.

   User is given a chance to fix the options in the window. User may
   press Escape key in the window, the function will then return
   CDW_NO. The function also returns CDW_NO if it decides that for
   some reason they weren't fixed by user.

   \param task - task which needs to be validated

   \return CDW_OK if options for writing are valid
   \return CDW_NO if the options are invalid, or if user has pressed Escape key in options window
*/
cdw_rv_t cdw_write_wizard_options_validate_closed(cdw_task_t *task)
{
	cdw_id_t pid;
	int fi;
	cdw_rv_t valid = cdw_write_wizard_options_validate(task, &pid, &fi);
	if (valid == CDW_NO) { /* some option field is invalid */

		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("One of option fields is incorrect or contains character that is not allowed. Please fix it."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		int c = cdw_write_wizard_options_window_by_flow(&pid, &fi);
		if (c == '0') {
			return CDW_NO;
		} else {
			/* c == '1', invalid options have been fixed
			   and user pressed "Save and Close"
			   key. Validate again, just to be sure. */
			valid = cdw_write_wizard_options_validate(task, &pid, &fi);
			if (valid == CDW_NO) {
				/* 2TRANS: this is title of dialog window */
				cdw_buttons_dialog(_("Error"),
						   /* 2TRANS: this is message in dialog window */
						   _("The invalid option(s) in options window has not been fixed."),
						   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
				return CDW_NO;
			}
		}
	} else {
		/* Options are valid and user has pressed "save and close" key. */
	}

	return CDW_OK;
}





/**
   \brief Validate options from main wizard window, and save them

   Function validates values of all options in wizard window (but not
   in "other options" window), and stores the values in \p task.

   Since there are no variables in the window that could be really
   validated, the function jumps right to saving values.
   Since the result of validation can't be negative, the function
   always returns CDW_OK.

   \param task - task variable to save values to
   \param fi - unused, field index

   return CDW_OK
*/
cdw_rv_t cdw_write_wizard_validate_and_save(cdw_task_t *task, __attribute__((unused)) int *fi)
{
	/* validate */
	const char *s = field_buffer(wizard.cdw_form->fields[f_volume_id_i], 0);
	cdw_rv_t crv = cdw_string_security_parser(s, (char *) NULL);
	if (crv == CDW_NO) {
		*fi = f_volume_id_i;
		return CDW_NO;
	}

	/* save */
	s = cdw_form_get_string(wizard.cdw_form, f_volume_id_i);
	strncpy(task->create_image.iso9660.volume_id, s, CDW_ISO9660_VOLI_LEN);
	task->create_image.iso9660.volume_id[CDW_ISO9660_VOLI_LEN] = '\0';

	if (wizard.verify_checkbox_visible) {
		task->burn.verify = cdw_form_get_checkbox_state(wizard.cdw_form, f_verify_i);
	} else {
		task->burn.verify = false;
	}
	if (wizard.dummy_checkbox_visible) {
		task->burn.dummy = cdw_form_get_checkbox_state(wizard.cdw_form, f_dummy_i);
	} else {
		task->burn.dummy = false;
	}
	task->burn.eject = cdw_form_get_checkbox_state(wizard.cdw_form, f_eject_cb);

	CDW_DROPDOWN *dd = cdw_form_get_dropdown(wizard.cdw_form, f_speed_dd);
	task->burn.speed_id = cdw_dropdown_get_current_item_id(dd);

	dd = cdw_form_get_dropdown(wizard.cdw_form, f_session_dd);
	task->burn.session_mode = cdw_dropdown_get_current_item_id(dd);

	if (wizard.disc_mode_dd_visible) {
		dd = cdw_form_get_dropdown(wizard.cdw_form, f_disc_dd);
		task->burn.disc_mode = cdw_dropdown_get_current_item_id(dd);
	}

	return CDW_OK;
}
