(* :Name: TriangularInterpolate *)

(* :Title: TriangularInterpolate *)

(* :Author: Tom Wickham-Jones*)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 2.2 *)

(* :Summary:
	This package provides functions for triangular interpolation.
*)

(* :History:
	Created summer 1993 by Tom Wickham-Jones.

	This package is described in the book
	Mathematica Graphics: Techniques and Applications.
	Tom Wickham-Jones, TELOS/Springer-Verlag 1994.
*)

(*:Warnings:
	This package requires the triangulation package Delaunay.m
	
*)


BeginPackage[ "ExtendGraphics`TriangularInterpolate`",
				"ExtendGraphics`Delaunay`", 
				"ExtendGraphics`SurfaceGraphics3D`",
				"Utilities`FilterOptions`"]

TriangularInterpolate::usage =
	"TriangularInterpolate[ pts] constructs a TriangularInterpolation
	function which represents an approximate function that interpolates
	the data.  The data must have the form {x,y,z} and do not have to
	be regularly spaced."
	 
	 
TriangularInterpolating::usage =
	"TriangularInterpolating[ data] represents an approximate function
	 whose values are found by interpolation."

InterpolateSurface::usage =
	"InterpolateSurface[ SurfaceGraphics3D[ data]] returns a new
	 SurfaceGraphics3D object in which the data has been resampled
	 with regular points."


Begin[ "`Private`"]


TriangularInterpolate[ pnts_List /; 
			MatrixQ[ N[ pnts], NumberQ] && 
		       	Length[ First[ pnts]] === 3, opts___Rule]:=
	TriangularInterpolate[ N[pnts], Delaunay[ N[ pnts]], opts]			


TriangularInterpolate[ pnts_List /; 
			MatrixQ[ N[pnts], NumberQ] && 
		       	Length[ First[ pnts]] === 3,
		       {hull_, tri_}, opts___Rule] :=
    Block[{num, work, minx, maxx, miny, maxy, tris, ext},
    	num = Range[ Length[ tri]] ;
		tris = Map[ Part[ pnts, #]&, tri] ;
		work = Map[ Min[ Part[ Transpose[ #], 1]]&, tris] ;
		minx = SortFun[ work] ;
		work = Map[ Max[ Part[ Transpose[ #], 1]]&, tris] ;
		maxx = SortFun[ work] ;
		work = Map[ Min[ Part[ Transpose[ #], 2]]&, tris] ;
		miny = SortFun[ work] ;
		work = Map[ Max[ Part[ Transpose[ #], 2]]&, tris] ;
		maxy = SortFun[ work] ;
		TriangularInterpolating[
			{hull, tri, pnts, minx, maxx, miny, maxy}]
		]

(*
 
 The bounding box info is stored in minx maxx miny and maxy
 
 The format is (eg minx)  { tri-number, xval} 
 the values always increase as you down the list.
  
*)


(*
Colinear data point problem
*)

SortFun[ data_] := 
    	Sort[ 
	    		Transpose[ { Range[ Length[ data]], data}], 
	    		Less[ Part[ #1, 2], Part[#2, 2]]&]


Format[ t_TriangularInterpolating] := "TriangularInterpolating[ <> ]"


(*

FindFirstMin[ test, imin, imax, list]

list {{t1, v1}, {t2, v2}, {t3, v3}, ...}

The vi are the minimum x(y) values of the triangles.  

Return the index of the first for which the test is greater
or equal than.

Return Take[ list, pos] where pos
 
	test >= Part[ list, pos, 2] &&
	test < Part[ list, pos+1, 2]


FindFirstMax[ test, imin, imax, list]

Return the index pos 
	test > Part[ list, pos, 2] &&
	test <= Part[ list, pos+1, 2]
	

*) 


FindFirstMin[ x_, imin_, imax_, list_] :=
	Block[ {pos},
		Which[
			x < Part[ list, 1, 2], {},
			x >=  Part[ list, -1, 2], Map[ First, list],
			True,
				pos = FindFirstMin1[ x, imin, imax, list] ;
				If[  x < Part[ list, pos, 2] ||
					(pos < Length[ list] && x >=  Part[ list, pos+1,2]),
					Print[ "Error in Bounding Box calc"]] ;
				Map[ First, Take[ list, pos]]]
		]

FindFirstMin1[ x_, imin_, imax_, list_] :=
	Block[{pos},
		If[ imin === imax,
			imin,
			pos = Floor[ (imin+imax)/2] ;
			If[ x < Part[ list, pos, 2],
				FindFirstMin1[ x, imin, pos, list],
				If[ x < Part[ list, pos+1, 2],
					pos,
					FindFirstMin1[ x, pos+1, imax, list]
					]
				]
			]
		]
		
FindFirstMax[ x_, imin_, imax_, list_] :=
	Block[ {pos},
		Which[
			x <= Part[ list, 1, 2], Map[ First, list],
			x >  Part[ list, -1, 2], {},
			True,
				pos = FindFirstMax1[ x, imin, imax, list] ;
				If[  x <= Part[ list, pos, 2] ||
					(pos < Length[ list] && x >  Part[ list, pos+1,2]),
					Print[ "Error in Bounding Box calc"]] ;
				Map[ First, Drop[ list, pos]]]
		]


FindFirstMax1[ x_, imin_, imax_, list_] :=
	Block[{pos},
		If[ imin === imax,
			imin,
			pos = Floor[ (imin+imax)/2] ;
			If[ x <= Part[ list, pos, 2],
				FindFirstMax1[ x, imin, pos, list],
				If[ x <= Part[ list, pos+1, 2],
					pos,
					FindFirstMax1[ x, pos+1, imax, list]
					]
				]
			]
		]
		

PointInTri[ 
		{x_, y_}, 
		{{x1_,y1_,z1_}, 
		 {x2_,y2_,z2_},
		 {x3_,y3_,z3_}}] :=
    Block[{t1,t2,t3,eps = 5 $MachineEpsilon},
    	t1 = -x1 y + x2 y + x y1 - x2 y1 - x y2 + x1 y2 ;
    	t2 = -x2 y + x3 y + x y2 - x3 y2 - x y3 + x2 y3 ;
		t3 = x1 y - x3 y - x y1 + x3 y1 + x y3 - x1 y3 ;
    	If[ t1 > -eps && t2 > -eps && t3 > -eps ||
            t1 <  eps && t2 <  eps && t3 <  eps,
	    	True,
			False]
    	]
			

FindTri[ pt_, tris_, pts_, rng_] :=
    Block[{tst},
    	tst = Part[ pts, Part[ tris, First[ rng]]] ;
    	If[ PointInTri[ pt, tst],
		First[ rng],
		If[ Length[ rng] === 1,
			{},
			FindTri[ pt, tris, pts, Rest[ rng]]]]
    ]

InterpolatePointInTri[ 
		{x_, y_}, 
		{{x1_,y1_,z1_}, 
		 {x2_,y2_,z2_},
		 {x3_,y3_,z3_}}] :=
    Block[{den, a, b, c},
    	den = x2 y1 - x3 y1 - x1 y2 + x3 y2 + x1 y3 - x2 y3 ;
		a = -(y2 z1) + y3 z1 + y1 z2 - y3 z2 - y1 z3 + y2 z3 ;
		b = x2 z1 - x3 z1 - x1 z2 + x3 z2 + x1 z3 - x2 z3 ;
		c = x3 y2 z1 - x2 y3 z1 - x3 y1 z2 + x1 y3 z2 + x2 y1 z3 - x1 y2 z3 ;
		a/den x + b/den y + c/den
		]

TriangularInterpolating::dmval =
	"Input value `1` lies outside domain of the interpolating function."


TriangularInterpolating[ 
   {hull_, tris_, pts_, minx_, maxx_, miny_, maxy_}][ x_?NumberQ, y_?NumberQ] :=
    Block[{x0, y0, x1, y1, tri, len},
		len = Length[ minx] ;
		x0 = FindFirstMin[ x, 1, len, minx] ; (* In x0 and above *)
		x1 = FindFirstMax[ x, 1, len, maxx] ; (* In x1 and below *)
		y0 = FindFirstMin[ y, 1, len, miny] ;
		y1 = FindFirstMax[ y, 1, len, maxy] ;
		rng = Intersection[ x0, x1, y0, y1] ;
		If[ rng =!= {},
	    	tri = FindTri[ {x, y}, tris, pts, rng],
	    	tri = {}] ;
		If[ tri === {},
		   		Message[ TriangularInterpolating::dmval, {x,y}];
				Indeterminate
		(* else *) ,
			tri = Part[ pts, Part[ tris, tri]] ;
			InterpolatePointInTri[ {x, y}, tri]]
		]


FixInd[ pts_] :=
    Block[{num},
    	res = Select[ pts, FreeQ[ #, Indeterminate]&] ;
	If[ Length[ res] === 3, 
		res,
		{}]
	]

MinMax[ data_] :=
    Block[{ x},
    	x = Map[ First, data] ;
	{Min[x], Max[x]}
	]

MinMax[ data_, pos_] :=
    Block[{ d},
    	d = Map[ Part[ #, pos]&, data] ;
	{Min[d], Max[d]}
	]


CheckNum[ x_ /; (Head[x] === Integer && x > 1), len_] := x

CheckNum[ x_, len_] := 	Ceiling[ N[ Sqrt[ len] +2]]
	    
		
InterpolateSurface[ 
	SurfaceGraphics3D[ 
				{
				data_, 
				tri_ /; MatrixQ[ tri, NumberQ] && Length[ First[ tri]] === 3
				}, opts___],	smooth_:Automatic] :=
    Block[{n, m, d, fun, x, y, 
           x0, x1, xi, y0, y1, yi, 
	       cnt, num, ntri},
		num = CheckNum[ smooth, Length[ data]] ;
    	fun = TriangularInterpolate[ data] ;
		{x0, x1} = MinMax[ data, 1] ;
		xi = (x1 - x0)/(num-1) ;
		{y0, y1} = MinMax[ data, 2] ;
		yi = (y1 - y0)/(num-1) ;
  		Off[ TriangularInterpolating::dmval] ;
		pts = Table[ {x, y, fun[ x, y]}, 
						{x, x0, x1, xi}, {y, y0, y1, yi}] ;
		On[ TriangularInterpolating::dmval] ;
    	{n, m, d} = Dimensions[ pts] ;
		pts = Flatten[ pts, 1] ;
		cnt = 0 ;
		ntri = 
			Table[ If[ Part[ pts, i, 3] === Indeterminate, 
						Indeterminate,
						++cnt], {i, Length[ pts]}] ;
		ntri = 
   	    	Table[ 
	        	{
	        	Part[ ntri, (i-1)*m+j], Part[ ntri , (i-1)*m+j+1],  
	        	Part[ ntri, (i)*m+j+1], Part[ ntri , (i)*m+j]
	        	}, {i, n-1}, {j, m-1}] ;
		pts = DeleteCases[ pts, {_, _, Indeterminate}] ;
		ntri = Flatten[ ntri, 1] ;
		ntri = Map[ If[ FreeQ[ #, Indeterminate], #, FixInd[ #]]&, ntri] ;
		ntri = DeleteCases[ ntri, {}] ;
		SurfaceGraphics3D[ { pts, ntri}, opts]
		]

End[]

EndPackage[]			

(*:Examples:
	

<<ExtendGraphics`TriangularInterpolate`

pnts = Table[ { x = Random[], y = Random[], x y}, {50}];

fun = TriangularInterpolate[ pnts]

fun[ .5,.7]

ListSurfacePlot3D[ pnts]

Show[ InterpolateSurface[ %, 8]]



*)
