#ifndef INCLUDE_MODULE_DISPLAY_H
#define INCLUDE_MODULE_DISPLAY_H
/**
 * @brief Definitions and prototypes for module_display.c
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation
 * (http://www.gnu.org/licenses/gpl.html)
 *
 * @file    src/module_display.h
 * @author  Matthias Grimm <matthias.grimm@users.sourceforge.net>
 */

#include <pbbtaglist.h>
#include "debug.h"

#define  DEFAULT_FB_DEVICE	"/dev/fb0"

#ifndef KEY_KBDILLUMTOGGLE
#define KEY_KBDILLUMTOGGLE 228
#endif

#ifndef KEY_KBDILLUMDOWN
#define KEY_KBDILLUMDOWN   229
#endif

#ifndef KEY_KBDILLUMUP
#define KEY_KBDILLUMUP     230
#endif

#ifndef FBIOBLANK
#define FBIOBLANK	   0x4611 /* 0 or vesa-level+1 */
#endif

#ifndef VESA_POWERDOWN   /* vesa level for FBIOBLANK */
#define VESA_POWERDOWN		3
#endif

/* Definitions to control ATI Framebuffer, because they aren't
 * available as linux include file. Definitions for the radeon
 * framebuffer are placed in linux/radeonfb.h
 */
#define ATY_MIRROR_CRT_ON	0x00000002
#define FBIO_ATY128_GET_MIRROR	_IOR('@', 1, __u32)
#define FBIO_ATY128_SET_MIRROR	_IOW('@', 2, __u32)

enum {BATTERY, AC};
enum {AUTOADJ_OFF, AUTOADJ_LIN, AUTOADJ_HYS, AUTOADJ_LAST=AUTOADJ_HYS};

/* operation codes (OP) for display_setlevel() */
#define OP_DIM_LIGHT	1
#define OP_DIM_RECOVER	2
#define OP_DIM_OFF	3

#define LCD_BRIGHTNESS_OFF 0  /* lower ranges for light sources */
#define LCD_BRIGHTNESS_MIN 1  /* of 16 steps */
#define KBD_BRIGHTNESS_OFF 0
#define KBD_BRIGHTNESS_MIN 16 /* of 256 steps */

/* definitions for status */
#define STATUS_NORMAL 0
#define STATUS_DIMMED 1
#define STATUS_OFF    2

struct display_flags {
	unsigned int lmu_enable:1;
	unsigned int kbd_switchedoff:1;
	unsigned int ctrl_fb:1;	   /* 1, if framebuffer should be blanked too */
	unsigned int dimfullydark:1;
	unsigned int status:2;
	unsigned int coveropen:1;   /* track signal from TAG_COVERSTATUS */
	unsigned int nobacklight:1;  /* no backlight controller available */
	unsigned int :0;
};

struct autoadj_param {
	unsigned int ambient1;  /* x1 */
	unsigned int level1;    /* y1 */
	unsigned int ambient2;  /* x2 */
	unsigned int level2;    /* y2 */
};

struct display_light {
	int current;     /* current brightness value */
	int target;      /* brightness to set to */
	int max;         /* maximum brightness value */
	int backup;
	int offset;      /* users influence on automatic adjustment */
	int fading;      /* timer interval or 0 for no fading */
	int fadingstep; 
	int autoadj_mode; /* operating mode of the ambient controller: off, linear or hysteresis */
	unsigned int autoadj_hyst_out; /* the hysteresis function needs to know its output status */
	struct autoadj_param ac_params;
	struct autoadj_param battery_params;
	int isrunning;   /* display timer currently running */
#if defined(DEBUG) && DISPLAYTIMER
	struct timeval start;  /* for profiling fading routines */
	int steps;
#endif
};

/* public prototypes */
int display_init ();
void display_exit ();

/* private prototypes */
void display_keyboard (struct tagitem *taglist);
void display_query (struct tagitem *taglist);
void display_configure (struct tagitem *taglist);
/*void display_timer (struct tagitem *taglist);*/
gboolean display_timer (gpointer data);
void display_ambienttimer (struct tagitem *taglist);
void display_handle_tags (int mode, struct tagitem *taglist);
int display_calc_brightness (struct display_light *illu, int pwrsource, int ambient, int ambientmax);
int display_set_lcdbrightness(int level);
int display_set_kbdbrightness(int level);
int display_clip_brightness(struct display_light *illu, int level);
void display_set_fading (struct display_light *illu, int fading);
int display_get_fading (struct display_light *illu);
void display_set_autoadj_from_tag (struct display_light *illu, int pwrsource, tag_t data);
tag_t display_get_autoadj_for_tag (struct display_light *illu, int pwrsource);
void display_set_autoadj_parameter (struct display_light *illu, int pwrsource, int *params);
void display_change_brightness(int op);
void display_framebuffer (int on);
void display_sync_lcdbrightness (void);
int display_switchmirror ();

#endif    /* INCLUDE_MODULE_DISPLAY_H */
