#!/bin/sh
# A simple script that generates loongarch-str.h and loongarch.opt
# from genopt/loongarch-optstr.
#
# Copyright (C) 2021-2024 Free Software Foundation, Inc.
#
# This file is part of GCC.
#
# GCC is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 3, or (at your option) any later
# version.
#
# GCC is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
# License for more details.
#
# You should have received a copy of the GNU General Public License
# along with GCC; see the file COPYING3.  If not see
# <http://www.gnu.org/licenses/>.

cd "$(dirname "$0")"

# Generate a header containing definitions from the string table.
gen_defines() {
    cat <<EOF
/* Generated automatically by "genstr" from "loongarch-strings" and
   "isa-evolution.in".  Please do not edit this file directly.

   Copyright (C) 2021-2024 Free Software Foundation, Inc.
   Contributed by Loongson Ltd.

This file is part of GCC.

GCC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3, or (at your option)
any later version.

GCC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GCC; see the file COPYING3.  If not see
<http://www.gnu.org/licenses/>.  */

#ifndef LOONGARCH_STR_H
#define LOONGARCH_STR_H
EOF

    sed -e '/^$/n' -e 's@#.*$@@' -e '/^$/d' \
	-e 's@^\([^ \t]\+\)[ \t]*\([^ \t]*\)@#define \1 "\2"@' \
	loongarch-strings

    echo

	# Generate the strings from isa-evolution.in.
	awk '{
	  a=$3
	  gsub(/-/, "_", a)
	  print("#define OPTSTR_"toupper(a)"\t\""$3"\"")
	}' isa-evolution.in

    echo
    echo "#endif /* LOONGARCH_STR_H */"
}


# Substitute all "@@<KEY>@@" to "<VALUE>" in loongarch.opt.in
# according to the key-value pairs defined in loongarch-strings.

gen_options() {

    sed -e '/^$/n' -e 's@#.*$@@' -e '/^$/d' \
	-e 's@^\([^ \t]\+\)[ \t]*\([^ \t]*\)@\1="\2"@' \
	loongarch-strings | { \

	# read the definitions
	while read -r line; do
	    eval "$line"
	done

	# print a header
	cat << EOF
; Generated by "genstr" from the template "loongarch.opt.in"
; and definitions from "loongarch-strings" and "isa-evolution.in".
;
; Please do not edit this file directly.
; It will be automatically updated during a gcc build
; if you change "loongarch.opt.in", "loongarch-strings", or
; "isa-evolution.in".
;
EOF

	# make the substitutions
	sed -e 's@"@\\"@g' -e 's/@@\([^@]\+\)@@/${\1}/g' loongarch.opt.in | \
	    while read -r line; do
		eval "echo \"$line\""
	    done
    }

    # Generate the strings from isa-evolution.in.
    awk '{
      print("")
      print("m"$3)
      gsub(/-/, "_", $3)
      print("Target Mask(ISA_"toupper($3)") Var(la_isa_evolution)")
      $1=""; $2=""; $3=""
      sub(/^ */, "", $0)
      print($0)
    }' isa-evolution.in
}

gen_cpucfg_map() {
    cat <<EOF
/* Generated automatically by "genstr" from "isa-evolution.in".
   Please do not edit this file directly.

   Copyright (C) 2023-2024 Free Software Foundation, Inc.

This file is part of GCC.

GCC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3, or (at your option)
any later version.

GCC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GCC; see the file COPYING3.  If not see
<http://www.gnu.org/licenses/>.  */

#ifndef LOONGARCH_CPUCFG_MAP_H
#define LOONGARCH_CPUCFG_MAP_H

#include "options.h"

static constexpr struct {
  int cpucfg_word;
  unsigned int cpucfg_bit;
  HOST_WIDE_INT isa_evolution_bit;
} cpucfg_map[] = {
EOF

    # Generate the strings from isa-evolution.in.
    awk '{
      gsub(/-/, "_", $3)
      print("  { "$1", 1u << "$2", OPTION_MASK_ISA_"toupper($3)" },")
    }' isa-evolution.in

    echo "};"
    echo
    echo "static constexpr int cpucfg_useful_idx[] = {"

    awk 'BEGIN { print("  0,\n  1,\n  2,\n  16,\n  17,\n  18,\n  19,") }
	 {if ($1+0 > max+0) max=$1; print("  "$1",")}' \
	isa-evolution.in | sort -n | uniq

    echo "};"
    echo ""

    awk 'BEGIN { max=19 }
	 { if ($1+0 > max+0) max=$1 }
	 END { print "static constexpr int N_CPUCFG_WORDS = "1+max";" }' \
	isa-evolution.in

    echo "#endif /* LOONGARCH_CPUCFG_MAP_H */"
}

main() {
    case "$1" in
	cpucfg-map) gen_cpucfg_map;;
	header) gen_defines;;
	opt) gen_options;;
	*) echo "Unknown Command: \"$1\". Available: cpucfg-map, header, opt"; exit 1;;
    esac
}

main "$@"
