/*
 * Big5 implementation for libcharset. (This is the character set
 * itself, not any particular multibyte encoding of it. Multibyte
 * encodings of this character set are handled separately.)
 */

#include <assert.h>

#include "charset.h"
#include "internal.h"

/*
 * These tables are generated from the Big5 <-> Unicode character
 * mapping found at
 * 
 * http://www.unicode.org/Public/MAPPINGS/OBSOLETE/EASTASIA/OTHER/BIG5.TXT
 * 
 * In the few cases where Big5 contains two characters whose
 * Unicode equivalents are the same, we map both to those
 * characters.
 */

/*
 * Mapping from Big5 to Unicode. For these purposes Big5 is
 * considered to be a 94x191 square array, with coordinates running
 * from (0,0) to (93,190). (In the multibyte encoding those
 * coordinates are offset by 0xA1 and 0x40 respectively, so they
 * run from (A1,40) to (FE,FE).)
 * 
 * Generated by running the following sh+Perl over BIG5.TXT:

cat BIG5.TXT | \
sed -e $'s/^\\(0xA15A\t\\)0xFFFD/\\10xFF3F/' \
    -e $'s/^\\(0xA1FE\t\\)0xFFFD/\\10x2571/' \
    -e $'s/^\\(0xA240\t\\)0xFFFD/\\10x2572/' \
    -e $'s/^\\(0xA2CC\t\\)0xFFFD/\\10x5341/' \
    -e $'s/^\\(0xA2CE\t\\)0xFFFD/\\10x5345/' \
    -e '/^0xA1C3/d' -e '/^0xA1C5/d' | \
perl -ne '$a{hex$1,hex$2}=$3 if /^0x(\S\S)(\S\S)\s+(0x\S+)\s/;' \
      -e 'END {for $r (0..93) { $o="    {"; for $c (0..190) {' \
      -e '$k=$a{$r+161,$c+64}; $k=" ERROR" unless defined $k;' \
      -e '$o .= ", " if $c > 0; (print "$o\n"), $o="    " if length $o > 70;' \
      -e '$o .= $k; } print "$o},\n"; }}'

 */

static const unsigned short big5_forward[94][191] = {
    {0x3000, 0xFF0C, 0x3001, 0x3002, 0xFF0E, 0x2022, 0xFF1B, 0xFF1A, 0xFF1F, 
    0xFF01, 0xFE30, 0x2026, 0x2025, 0xFE50, 0xFF64, 0xFE52, 0x00B7, 0xFE54, 
    0xFE55, 0xFE56, 0xFE57, 0xFF5C, 0x2013, 0xFE31, 0x2014, 0xFE33, 0xFF3F, 
    0xFE34, 0xFE4F, 0xFF08, 0xFF09, 0xFE35, 0xFE36, 0xFF5B, 0xFF5D, 0xFE37, 
    0xFE38, 0x3014, 0x3015, 0xFE39, 0xFE3A, 0x3010, 0x3011, 0xFE3B, 0xFE3C, 
    0x300A, 0x300B, 0xFE3D, 0xFE3E, 0x3008, 0x3009, 0xFE3F, 0xFE40, 0x300C, 
    0x300D, 0xFE41, 0xFE42, 0x300E, 0x300F, 0xFE43, 0xFE44, 0xFE59, 0xFE5A, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0xFE5B, 0xFE5C, 
    0xFE5D, 0xFE5E, 0x2018, 0x2019, 0x201C, 0x201D, 0x301D, 0x301E, 0x2035, 
    0x2032, 0xFF03, 0xFF06, 0xFF0A, 0x203B, 0x00A7, 0x3003, 0x25CB, 0x25CF, 
    0x25B3, 0x25B2, 0x25CE, 0x2606, 0x2605, 0x25C7, 0x25C6, 0x25A1, 0x25A0, 
    0x25BD, 0x25BC, 0x32A3, 0x2105, 0x203E,  ERROR, 0xFF3F,  ERROR, 0xFE49, 
    0xFE4A, 0xFE4D, 0xFE4E, 0xFE4B, 0xFE4C, 0xFE5F, 0xFE60, 0xFE61, 0xFF0B, 
    0xFF0D, 0x00D7, 0x00F7, 0x00B1, 0x221A, 0xFF1C, 0xFF1E, 0xFF1D, 0x2266, 
    0x2267, 0x2260, 0x221E, 0x2252, 0x2261, 0xFE62, 0xFE63, 0xFE64, 0xFE65, 
    0xFE66, 0x223C, 0x2229, 0x222A, 0x22A5, 0x2220, 0x221F, 0x22BF, 0x33D2, 
    0x33D1, 0x222B, 0x222E, 0x2235, 0x2234, 0x2640, 0x2642, 0x2641, 0x2609, 
    0x2191, 0x2193, 0x2190, 0x2192, 0x2196, 0x2197, 0x2199, 0x2198, 0x2225, 
    0x2223, 0x2571},
    {0x2572, 0xFF0F, 0xFF3C, 0xFF04, 0x00A5, 0x3012, 0x00A2, 0x00A3, 0xFF05, 
    0xFF20, 0x2103, 0x2109, 0xFE69, 0xFE6A, 0xFE6B, 0x33D5, 0x339C, 0x339D, 
    0x339E, 0x33CE, 0x33A1, 0x338E, 0x338F, 0x33C4, 0x00B0, 0x5159, 0x515B, 
    0x515E, 0x515D, 0x5161, 0x5163, 0x55E7, 0x74E9, 0x7CCE, 0x2581, 0x2582, 
    0x2583, 0x2584, 0x2585, 0x2586, 0x2587, 0x2588, 0x258F, 0x258E, 0x258D, 
    0x258C, 0x258B, 0x258A, 0x2589, 0x253C, 0x2534, 0x252C, 0x2524, 0x251C, 
    0x2594, 0x2500, 0x2502, 0x2595, 0x250C, 0x2510, 0x2514, 0x2518, 0x256D, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x256E, 0x2570, 
    0x256F, 0x2550, 0x255E, 0x256A, 0x2561, 0x25E2, 0x25E3, 0x25E5, 0x25E4, 
    0x2571, 0x2572, 0x2573, 0xFF10, 0xFF11, 0xFF12, 0xFF13, 0xFF14, 0xFF15, 
    0xFF16, 0xFF17, 0xFF18, 0xFF19, 0x2160, 0x2161, 0x2162, 0x2163, 0x2164, 
    0x2165, 0x2166, 0x2167, 0x2168, 0x2169, 0x3021, 0x3022, 0x3023, 0x3024, 
    0x3025, 0x3026, 0x3027, 0x3028, 0x3029, 0x5341, 0x5344, 0x5345, 0xFF21, 
    0xFF22, 0xFF23, 0xFF24, 0xFF25, 0xFF26, 0xFF27, 0xFF28, 0xFF29, 0xFF2A, 
    0xFF2B, 0xFF2C, 0xFF2D, 0xFF2E, 0xFF2F, 0xFF30, 0xFF31, 0xFF32, 0xFF33, 
    0xFF34, 0xFF35, 0xFF36, 0xFF37, 0xFF38, 0xFF39, 0xFF3A, 0xFF41, 0xFF42, 
    0xFF43, 0xFF44, 0xFF45, 0xFF46, 0xFF47, 0xFF48, 0xFF49, 0xFF4A, 0xFF4B, 
    0xFF4C, 0xFF4D, 0xFF4E, 0xFF4F, 0xFF50, 0xFF51, 0xFF52, 0xFF53, 0xFF54, 
    0xFF55, 0xFF56},
    {0xFF57, 0xFF58, 0xFF59, 0xFF5A, 0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 
    0x0396, 0x0397, 0x0398, 0x0399, 0x039A, 0x039B, 0x039C, 0x039D, 0x039E, 
    0x039F, 0x03A0, 0x03A1, 0x03A3, 0x03A4, 0x03A5, 0x03A6, 0x03A7, 0x03A8, 
    0x03A9, 0x03B1, 0x03B2, 0x03B3, 0x03B4, 0x03B5, 0x03B6, 0x03B7, 0x03B8, 
    0x03B9, 0x03BA, 0x03BB, 0x03BC, 0x03BD, 0x03BE, 0x03BF, 0x03C0, 0x03C1, 
    0x03C3, 0x03C4, 0x03C5, 0x03C6, 0x03C7, 0x03C8, 0x03C9, 0x3105, 0x3106, 
    0x3107, 0x3108, 0x3109, 0x310A, 0x310B, 0x310C, 0x310D, 0x310E, 0x310F, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x3110, 0x3111, 
    0x3112, 0x3113, 0x3114, 0x3115, 0x3116, 0x3117, 0x3118, 0x3119, 0x311A, 
    0x311B, 0x311C, 0x311D, 0x311E, 0x311F, 0x3120, 0x3121, 0x3122, 0x3123, 
    0x3124, 0x3125, 0x3126, 0x3127, 0x3128, 0x3129, 0x02D9, 0x02C9, 0x02CA, 
    0x02C7, 0x02CB,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR},
    {0x4E00, 0x4E59, 0x4E01, 0x4E03, 0x4E43, 0x4E5D, 0x4E86, 0x4E8C, 0x4EBA, 
    0x513F, 0x5165, 0x516B, 0x51E0, 0x5200, 0x5201, 0x529B, 0x5315, 0x5341, 
    0x535C, 0x53C8, 0x4E09, 0x4E0B, 0x4E08, 0x4E0A, 0x4E2B, 0x4E38, 0x51E1, 
    0x4E45, 0x4E48, 0x4E5F, 0x4E5E, 0x4E8E, 0x4EA1, 0x5140, 0x5203, 0x52FA, 
    0x5343, 0x53C9, 0x53E3, 0x571F, 0x58EB, 0x5915, 0x5927, 0x5973, 0x5B50, 
    0x5B51, 0x5B53, 0x5BF8, 0x5C0F, 0x5C22, 0x5C38, 0x5C71, 0x5DDD, 0x5DE5, 
    0x5DF1, 0x5DF2, 0x5DF3, 0x5DFE, 0x5E72, 0x5EFE, 0x5F0B, 0x5F13, 0x624D, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x4E11, 0x4E10, 
    0x4E0D, 0x4E2D, 0x4E30, 0x4E39, 0x4E4B, 0x5C39, 0x4E88, 0x4E91, 0x4E95, 
    0x4E92, 0x4E94, 0x4EA2, 0x4EC1, 0x4EC0, 0x4EC3, 0x4EC6, 0x4EC7, 0x4ECD, 
    0x4ECA, 0x4ECB, 0x4EC4, 0x5143, 0x5141, 0x5167, 0x516D, 0x516E, 0x516C, 
    0x5197, 0x51F6, 0x5206, 0x5207, 0x5208, 0x52FB, 0x52FE, 0x52FF, 0x5316, 
    0x5339, 0x5348, 0x5347, 0x5345, 0x535E, 0x5384, 0x53CB, 0x53CA, 0x53CD, 
    0x58EC, 0x5929, 0x592B, 0x592A, 0x592D, 0x5B54, 0x5C11, 0x5C24, 0x5C3A, 
    0x5C6F, 0x5DF4, 0x5E7B, 0x5EFF, 0x5F14, 0x5F15, 0x5FC3, 0x6208, 0x6236, 
    0x624B, 0x624E, 0x652F, 0x6587, 0x6597, 0x65A4, 0x65B9, 0x65E5, 0x66F0, 
    0x6708, 0x6728, 0x6B20, 0x6B62, 0x6B79, 0x6BCB, 0x6BD4, 0x6BDB, 0x6C0F, 
    0x6C34, 0x706B, 0x722A, 0x7236, 0x723B, 0x7247, 0x7259, 0x725B, 0x72AC, 
    0x738B, 0x4E19},
    {0x4E16, 0x4E15, 0x4E14, 0x4E18, 0x4E3B, 0x4E4D, 0x4E4F, 0x4E4E, 0x4EE5, 
    0x4ED8, 0x4ED4, 0x4ED5, 0x4ED6, 0x4ED7, 0x4EE3, 0x4EE4, 0x4ED9, 0x4EDE, 
    0x5145, 0x5144, 0x5189, 0x518A, 0x51AC, 0x51F9, 0x51FA, 0x51F8, 0x520A, 
    0x52A0, 0x529F, 0x5305, 0x5306, 0x5317, 0x531D, 0x4EDF, 0x534A, 0x5349, 
    0x5361, 0x5360, 0x536F, 0x536E, 0x53BB, 0x53EF, 0x53E4, 0x53F3, 0x53EC, 
    0x53EE, 0x53E9, 0x53E8, 0x53FC, 0x53F8, 0x53F5, 0x53EB, 0x53E6, 0x53EA, 
    0x53F2, 0x53F1, 0x53F0, 0x53E5, 0x53ED, 0x53FB, 0x56DB, 0x56DA, 0x5916, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x592E, 0x5931, 
    0x5974, 0x5976, 0x5B55, 0x5B83, 0x5C3C, 0x5DE8, 0x5DE7, 0x5DE6, 0x5E02, 
    0x5E03, 0x5E73, 0x5E7C, 0x5F01, 0x5F18, 0x5F17, 0x5FC5, 0x620A, 0x6253, 
    0x6254, 0x6252, 0x6251, 0x65A5, 0x65E6, 0x672E, 0x672C, 0x672A, 0x672B, 
    0x672D, 0x6B63, 0x6BCD, 0x6C11, 0x6C10, 0x6C38, 0x6C41, 0x6C40, 0x6C3E, 
    0x72AF, 0x7384, 0x7389, 0x74DC, 0x74E6, 0x7518, 0x751F, 0x7528, 0x7529, 
    0x7530, 0x7531, 0x7532, 0x7533, 0x758B, 0x767D, 0x76AE, 0x76BF, 0x76EE, 
    0x77DB, 0x77E2, 0x77F3, 0x793A, 0x79BE, 0x7A74, 0x7ACB, 0x4E1E, 0x4E1F, 
    0x4E52, 0x4E53, 0x4E69, 0x4E99, 0x4EA4, 0x4EA6, 0x4EA5, 0x4EFF, 0x4F09, 
    0x4F19, 0x4F0A, 0x4F15, 0x4F0D, 0x4F10, 0x4F11, 0x4F0F, 0x4EF2, 0x4EF6, 
    0x4EFB, 0x4EF0, 0x4EF3, 0x4EFD, 0x4F01, 0x4F0B, 0x5149, 0x5147, 0x5146, 
    0x5148, 0x5168},
    {0x5171, 0x518D, 0x51B0, 0x5217, 0x5211, 0x5212, 0x520E, 0x5216, 0x52A3, 
    0x5308, 0x5321, 0x5320, 0x5370, 0x5371, 0x5409, 0x540F, 0x540C, 0x540A, 
    0x5410, 0x5401, 0x540B, 0x5404, 0x5411, 0x540D, 0x5408, 0x5403, 0x540E, 
    0x5406, 0x5412, 0x56E0, 0x56DE, 0x56DD, 0x5733, 0x5730, 0x5728, 0x572D, 
    0x572C, 0x572F, 0x5729, 0x5919, 0x591A, 0x5937, 0x5938, 0x5984, 0x5978, 
    0x5983, 0x597D, 0x5979, 0x5982, 0x5981, 0x5B57, 0x5B58, 0x5B87, 0x5B88, 
    0x5B85, 0x5B89, 0x5BFA, 0x5C16, 0x5C79, 0x5DDE, 0x5E06, 0x5E76, 0x5E74, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x5F0F, 0x5F1B, 
    0x5FD9, 0x5FD6, 0x620E, 0x620C, 0x620D, 0x6210, 0x6263, 0x625B, 0x6258, 
    0x6536, 0x65E9, 0x65E8, 0x65EC, 0x65ED, 0x66F2, 0x66F3, 0x6709, 0x673D, 
    0x6734, 0x6731, 0x6735, 0x6B21, 0x6B64, 0x6B7B, 0x6C16, 0x6C5D, 0x6C57, 
    0x6C59, 0x6C5F, 0x6C60, 0x6C50, 0x6C55, 0x6C61, 0x6C5B, 0x6C4D, 0x6C4E, 
    0x7070, 0x725F, 0x725D, 0x767E, 0x7AF9, 0x7C73, 0x7CF8, 0x7F36, 0x7F8A, 
    0x7FBD, 0x8001, 0x8003, 0x800C, 0x8012, 0x8033, 0x807F, 0x8089, 0x808B, 
    0x808C, 0x81E3, 0x81EA, 0x81F3, 0x81FC, 0x820C, 0x821B, 0x821F, 0x826E, 
    0x8272, 0x827E, 0x866B, 0x8840, 0x884C, 0x8863, 0x897F, 0x9621, 0x4E32, 
    0x4EA8, 0x4F4D, 0x4F4F, 0x4F47, 0x4F57, 0x4F5E, 0x4F34, 0x4F5B, 0x4F55, 
    0x4F30, 0x4F50, 0x4F51, 0x4F3D, 0x4F3A, 0x4F38, 0x4F43, 0x4F54, 0x4F3C, 
    0x4F46, 0x4F63},
    {0x4F5C, 0x4F60, 0x4F2F, 0x4F4E, 0x4F36, 0x4F59, 0x4F5D, 0x4F48, 0x4F5A, 
    0x514C, 0x514B, 0x514D, 0x5175, 0x51B6, 0x51B7, 0x5225, 0x5224, 0x5229, 
    0x522A, 0x5228, 0x52AB, 0x52A9, 0x52AA, 0x52AC, 0x5323, 0x5373, 0x5375, 
    0x541D, 0x542D, 0x541E, 0x543E, 0x5426, 0x544E, 0x5427, 0x5446, 0x5443, 
    0x5433, 0x5448, 0x5442, 0x541B, 0x5429, 0x544A, 0x5439, 0x543B, 0x5438, 
    0x542E, 0x5435, 0x5436, 0x5420, 0x543C, 0x5440, 0x5431, 0x542B, 0x541F, 
    0x542C, 0x56EA, 0x56F0, 0x56E4, 0x56EB, 0x574A, 0x5751, 0x5740, 0x574D, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x5747, 0x574E, 
    0x573E, 0x5750, 0x574F, 0x573B, 0x58EF, 0x593E, 0x599D, 0x5992, 0x59A8, 
    0x599E, 0x59A3, 0x5999, 0x5996, 0x598D, 0x59A4, 0x5993, 0x598A, 0x59A5, 
    0x5B5D, 0x5B5C, 0x5B5A, 0x5B5B, 0x5B8C, 0x5B8B, 0x5B8F, 0x5C2C, 0x5C40, 
    0x5C41, 0x5C3F, 0x5C3E, 0x5C90, 0x5C91, 0x5C94, 0x5C8C, 0x5DEB, 0x5E0C, 
    0x5E8F, 0x5E87, 0x5E8A, 0x5EF7, 0x5F04, 0x5F1F, 0x5F64, 0x5F62, 0x5F77, 
    0x5F79, 0x5FD8, 0x5FCC, 0x5FD7, 0x5FCD, 0x5FF1, 0x5FEB, 0x5FF8, 0x5FEA, 
    0x6212, 0x6211, 0x6284, 0x6297, 0x6296, 0x6280, 0x6276, 0x6289, 0x626D, 
    0x628A, 0x627C, 0x627E, 0x6279, 0x6273, 0x6292, 0x626F, 0x6298, 0x626E, 
    0x6295, 0x6293, 0x6291, 0x6286, 0x6539, 0x653B, 0x6538, 0x65F1, 0x66F4, 
    0x675F, 0x674E, 0x674F, 0x6750, 0x6751, 0x675C, 0x6756, 0x675E, 0x6749, 
    0x6746, 0x6760},
    {0x6753, 0x6757, 0x6B65, 0x6BCF, 0x6C42, 0x6C5E, 0x6C99, 0x6C81, 0x6C88, 
    0x6C89, 0x6C85, 0x6C9B, 0x6C6A, 0x6C7A, 0x6C90, 0x6C70, 0x6C8C, 0x6C68, 
    0x6C96, 0x6C92, 0x6C7D, 0x6C83, 0x6C72, 0x6C7E, 0x6C74, 0x6C86, 0x6C76, 
    0x6C8D, 0x6C94, 0x6C98, 0x6C82, 0x7076, 0x707C, 0x707D, 0x7078, 0x7262, 
    0x7261, 0x7260, 0x72C4, 0x72C2, 0x7396, 0x752C, 0x752B, 0x7537, 0x7538, 
    0x7682, 0x76EF, 0x77E3, 0x79C1, 0x79C0, 0x79BF, 0x7A76, 0x7CFB, 0x7F55, 
    0x8096, 0x8093, 0x809D, 0x8098, 0x809B, 0x809A, 0x80B2, 0x826F, 0x8292, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x828B, 0x828D, 
    0x898B, 0x89D2, 0x8A00, 0x8C37, 0x8C46, 0x8C55, 0x8C9D, 0x8D64, 0x8D70, 
    0x8DB3, 0x8EAB, 0x8ECA, 0x8F9B, 0x8FB0, 0x8FC2, 0x8FC6, 0x8FC5, 0x8FC4, 
    0x5DE1, 0x9091, 0x90A2, 0x90AA, 0x90A6, 0x90A3, 0x9149, 0x91C6, 0x91CC, 
    0x9632, 0x962E, 0x9631, 0x962A, 0x962C, 0x4E26, 0x4E56, 0x4E73, 0x4E8B, 
    0x4E9B, 0x4E9E, 0x4EAB, 0x4EAC, 0x4F6F, 0x4F9D, 0x4F8D, 0x4F73, 0x4F7F, 
    0x4F6C, 0x4F9B, 0x4F8B, 0x4F86, 0x4F83, 0x4F70, 0x4F75, 0x4F88, 0x4F69, 
    0x4F7B, 0x4F96, 0x4F7E, 0x4F8F, 0x4F91, 0x4F7A, 0x5154, 0x5152, 0x5155, 
    0x5169, 0x5177, 0x5176, 0x5178, 0x51BD, 0x51FD, 0x523B, 0x5238, 0x5237, 
    0x523A, 0x5230, 0x522E, 0x5236, 0x5241, 0x52BE, 0x52BB, 0x5352, 0x5354, 
    0x5353, 0x5351, 0x5366, 0x5377, 0x5378, 0x5379, 0x53D6, 0x53D4, 0x53D7, 
    0x5473, 0x5475},
    {0x5496, 0x5478, 0x5495, 0x5480, 0x547B, 0x5477, 0x5484, 0x5492, 0x5486, 
    0x547C, 0x5490, 0x5471, 0x5476, 0x548C, 0x549A, 0x5462, 0x5468, 0x548B, 
    0x547D, 0x548E, 0x56FA, 0x5783, 0x5777, 0x576A, 0x5769, 0x5761, 0x5766, 
    0x5764, 0x577C, 0x591C, 0x5949, 0x5947, 0x5948, 0x5944, 0x5954, 0x59BE, 
    0x59BB, 0x59D4, 0x59B9, 0x59AE, 0x59D1, 0x59C6, 0x59D0, 0x59CD, 0x59CB, 
    0x59D3, 0x59CA, 0x59AF, 0x59B3, 0x59D2, 0x59C5, 0x5B5F, 0x5B64, 0x5B63, 
    0x5B97, 0x5B9A, 0x5B98, 0x5B9C, 0x5B99, 0x5B9B, 0x5C1A, 0x5C48, 0x5C45, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x5C46, 0x5CB7, 
    0x5CA1, 0x5CB8, 0x5CA9, 0x5CAB, 0x5CB1, 0x5CB3, 0x5E18, 0x5E1A, 0x5E16, 
    0x5E15, 0x5E1B, 0x5E11, 0x5E78, 0x5E9A, 0x5E97, 0x5E9C, 0x5E95, 0x5E96, 
    0x5EF6, 0x5F26, 0x5F27, 0x5F29, 0x5F80, 0x5F81, 0x5F7F, 0x5F7C, 0x5FDD, 
    0x5FE0, 0x5FFD, 0x5FF5, 0x5FFF, 0x600F, 0x6014, 0x602F, 0x6035, 0x6016, 
    0x602A, 0x6015, 0x6021, 0x6027, 0x6029, 0x602B, 0x601B, 0x6216, 0x6215, 
    0x623F, 0x623E, 0x6240, 0x627F, 0x62C9, 0x62CC, 0x62C4, 0x62BF, 0x62C2, 
    0x62B9, 0x62D2, 0x62DB, 0x62AB, 0x62D3, 0x62D4, 0x62CB, 0x62C8, 0x62A8, 
    0x62BD, 0x62BC, 0x62D0, 0x62D9, 0x62C7, 0x62CD, 0x62B5, 0x62DA, 0x62B1, 
    0x62D8, 0x62D6, 0x62D7, 0x62C6, 0x62AC, 0x62CE, 0x653E, 0x65A7, 0x65BC, 
    0x65FA, 0x6614, 0x6613, 0x660C, 0x6606, 0x6602, 0x660E, 0x6600, 0x660F, 
    0x6615, 0x660A},
    {0x6607, 0x670D, 0x670B, 0x676D, 0x678B, 0x6795, 0x6771, 0x679C, 0x6773, 
    0x6777, 0x6787, 0x679D, 0x6797, 0x676F, 0x6770, 0x677F, 0x6789, 0x677E, 
    0x6790, 0x6775, 0x679A, 0x6793, 0x677C, 0x676A, 0x6772, 0x6B23, 0x6B66, 
    0x6B67, 0x6B7F, 0x6C13, 0x6C1B, 0x6CE3, 0x6CE8, 0x6CF3, 0x6CB1, 0x6CCC, 
    0x6CE5, 0x6CB3, 0x6CBD, 0x6CBE, 0x6CBC, 0x6CE2, 0x6CAB, 0x6CD5, 0x6CD3, 
    0x6CB8, 0x6CC4, 0x6CB9, 0x6CC1, 0x6CAE, 0x6CD7, 0x6CC5, 0x6CF1, 0x6CBF, 
    0x6CBB, 0x6CE1, 0x6CDB, 0x6CCA, 0x6CAC, 0x6CEF, 0x6CDC, 0x6CD6, 0x6CE0, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x7095, 0x708E, 
    0x7092, 0x708A, 0x7099, 0x722C, 0x722D, 0x7238, 0x7248, 0x7267, 0x7269, 
    0x72C0, 0x72CE, 0x72D9, 0x72D7, 0x72D0, 0x73A9, 0x73A8, 0x739F, 0x73AB, 
    0x73A5, 0x753D, 0x759D, 0x7599, 0x759A, 0x7684, 0x76C2, 0x76F2, 0x76F4, 
    0x77E5, 0x77FD, 0x793E, 0x7940, 0x7941, 0x79C9, 0x79C8, 0x7A7A, 0x7A79, 
    0x7AFA, 0x7CFE, 0x7F54, 0x7F8C, 0x7F8B, 0x8005, 0x80BA, 0x80A5, 0x80A2, 
    0x80B1, 0x80A1, 0x80AB, 0x80A9, 0x80B4, 0x80AA, 0x80AF, 0x81E5, 0x81FE, 
    0x820D, 0x82B3, 0x829D, 0x8299, 0x82AD, 0x82BD, 0x829F, 0x82B9, 0x82B1, 
    0x82AC, 0x82A5, 0x82AF, 0x82B8, 0x82A3, 0x82B0, 0x82BE, 0x82B7, 0x864E, 
    0x8671, 0x521D, 0x8868, 0x8ECB, 0x8FCE, 0x8FD4, 0x8FD1, 0x90B5, 0x90B8, 
    0x90B1, 0x90B6, 0x91C7, 0x91D1, 0x9577, 0x9580, 0x961C, 0x9640, 0x963F, 
    0x963B, 0x9644},
    {0x9642, 0x96B9, 0x96E8, 0x9752, 0x975E, 0x4E9F, 0x4EAD, 0x4EAE, 0x4FE1, 
    0x4FB5, 0x4FAF, 0x4FBF, 0x4FE0, 0x4FD1, 0x4FCF, 0x4FDD, 0x4FC3, 0x4FB6, 
    0x4FD8, 0x4FDF, 0x4FCA, 0x4FD7, 0x4FAE, 0x4FD0, 0x4FC4, 0x4FC2, 0x4FDA, 
    0x4FCE, 0x4FDE, 0x4FB7, 0x5157, 0x5192, 0x5191, 0x51A0, 0x524E, 0x5243, 
    0x524A, 0x524D, 0x524C, 0x524B, 0x5247, 0x52C7, 0x52C9, 0x52C3, 0x52C1, 
    0x530D, 0x5357, 0x537B, 0x539A, 0x53DB, 0x54AC, 0x54C0, 0x54A8, 0x54CE, 
    0x54C9, 0x54B8, 0x54A6, 0x54B3, 0x54C7, 0x54C2, 0x54BD, 0x54AA, 0x54C1, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x54C4, 0x54C8, 
    0x54AF, 0x54AB, 0x54B1, 0x54BB, 0x54A9, 0x54A7, 0x54BF, 0x56FF, 0x5782, 
    0x578B, 0x57A0, 0x57A3, 0x57A2, 0x57CE, 0x57AE, 0x5793, 0x5955, 0x5951, 
    0x594F, 0x594E, 0x5950, 0x59DC, 0x59D8, 0x59FF, 0x59E3, 0x59E8, 0x5A03, 
    0x59E5, 0x59EA, 0x59DA, 0x59E6, 0x5A01, 0x59FB, 0x5B69, 0x5BA3, 0x5BA6, 
    0x5BA4, 0x5BA2, 0x5BA5, 0x5C01, 0x5C4E, 0x5C4F, 0x5C4D, 0x5C4B, 0x5CD9, 
    0x5CD2, 0x5DF7, 0x5E1D, 0x5E25, 0x5E1F, 0x5E7D, 0x5EA0, 0x5EA6, 0x5EFA, 
    0x5F08, 0x5F2D, 0x5F65, 0x5F88, 0x5F85, 0x5F8A, 0x5F8B, 0x5F87, 0x5F8C, 
    0x5F89, 0x6012, 0x601D, 0x6020, 0x6025, 0x600E, 0x6028, 0x604D, 0x6070, 
    0x6068, 0x6062, 0x6046, 0x6043, 0x606C, 0x606B, 0x606A, 0x6064, 0x6241, 
    0x62DC, 0x6316, 0x6309, 0x62FC, 0x62ED, 0x6301, 0x62EE, 0x62FD, 0x6307, 
    0x62F1, 0x62F7},
    {0x62EF, 0x62EC, 0x62FE, 0x62F4, 0x6311, 0x6302, 0x653F, 0x6545, 0x65AB, 
    0x65BD, 0x65E2, 0x6625, 0x662D, 0x6620, 0x6627, 0x662F, 0x661F, 0x6628, 
    0x6631, 0x6624, 0x66F7, 0x67FF, 0x67D3, 0x67F1, 0x67D4, 0x67D0, 0x67EC, 
    0x67B6, 0x67AF, 0x67F5, 0x67E9, 0x67EF, 0x67C4, 0x67D1, 0x67B4, 0x67DA, 
    0x67E5, 0x67B8, 0x67CF, 0x67DE, 0x67F3, 0x67B0, 0x67D9, 0x67E2, 0x67DD, 
    0x67D2, 0x6B6A, 0x6B83, 0x6B86, 0x6BB5, 0x6BD2, 0x6BD7, 0x6C1F, 0x6CC9, 
    0x6D0B, 0x6D32, 0x6D2A, 0x6D41, 0x6D25, 0x6D0C, 0x6D31, 0x6D1E, 0x6D17, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x6D3B, 0x6D3D, 
    0x6D3E, 0x6D36, 0x6D1B, 0x6CF5, 0x6D39, 0x6D27, 0x6D38, 0x6D29, 0x6D2E, 
    0x6D35, 0x6D0E, 0x6D2B, 0x70AB, 0x70BA, 0x70B3, 0x70AC, 0x70AF, 0x70AD, 
    0x70B8, 0x70AE, 0x70A4, 0x7230, 0x7272, 0x726F, 0x7274, 0x72E9, 0x72E0, 
    0x72E1, 0x73B7, 0x73CA, 0x73BB, 0x73B2, 0x73CD, 0x73C0, 0x73B3, 0x751A, 
    0x752D, 0x754F, 0x754C, 0x754E, 0x754B, 0x75AB, 0x75A4, 0x75A5, 0x75A2, 
    0x75A3, 0x7678, 0x7686, 0x7687, 0x7688, 0x76C8, 0x76C6, 0x76C3, 0x76C5, 
    0x7701, 0x76F9, 0x76F8, 0x7709, 0x770B, 0x76FE, 0x76FC, 0x7707, 0x77DC, 
    0x7802, 0x7814, 0x780C, 0x780D, 0x7946, 0x7949, 0x7948, 0x7947, 0x79B9, 
    0x79BA, 0x79D1, 0x79D2, 0x79CB, 0x7A7F, 0x7A81, 0x7AFF, 0x7AFD, 0x7C7D, 
    0x7D02, 0x7D05, 0x7D00, 0x7D09, 0x7D07, 0x7D04, 0x7D06, 0x7F38, 0x7F8E, 
    0x7FBF, 0x8004},
    {0x8010, 0x800D, 0x8011, 0x8036, 0x80D6, 0x80E5, 0x80DA, 0x80C3, 0x80C4, 
    0x80CC, 0x80E1, 0x80DB, 0x80CE, 0x80DE, 0x80E4, 0x80DD, 0x81F4, 0x8222, 
    0x82E7, 0x8303, 0x8305, 0x82E3, 0x82DB, 0x82E6, 0x8304, 0x82E5, 0x8302, 
    0x8309, 0x82D2, 0x82D7, 0x82F1, 0x8301, 0x82DC, 0x82D4, 0x82D1, 0x82DE, 
    0x82D3, 0x82DF, 0x82EF, 0x8306, 0x8650, 0x8679, 0x867B, 0x867A, 0x884D, 
    0x886B, 0x8981, 0x89D4, 0x8A08, 0x8A02, 0x8A03, 0x8C9E, 0x8CA0, 0x8D74, 
    0x8D73, 0x8DB4, 0x8ECD, 0x8ECC, 0x8FF0, 0x8FE6, 0x8FE2, 0x8FEA, 0x8FE5, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x8FED, 0x8FEB, 
    0x8FE4, 0x8FE8, 0x90CA, 0x90CE, 0x90C1, 0x90C3, 0x914B, 0x914A, 0x91CD, 
    0x9582, 0x9650, 0x964B, 0x964C, 0x964D, 0x9762, 0x9769, 0x97CB, 0x97ED, 
    0x97F3, 0x9801, 0x98A8, 0x98DB, 0x98DF, 0x9996, 0x9999, 0x4E58, 0x4EB3, 
    0x500C, 0x500D, 0x5023, 0x4FEF, 0x5026, 0x5025, 0x4FF8, 0x5029, 0x5016, 
    0x5006, 0x503C, 0x501F, 0x501A, 0x5012, 0x5011, 0x4FFA, 0x5000, 0x5014, 
    0x5028, 0x4FF1, 0x5021, 0x500B, 0x5019, 0x5018, 0x4FF3, 0x4FEE, 0x502D, 
    0x502A, 0x4FFE, 0x502B, 0x5009, 0x517C, 0x51A4, 0x51A5, 0x51A2, 0x51CD, 
    0x51CC, 0x51C6, 0x51CB, 0x5256, 0x525C, 0x5254, 0x525B, 0x525D, 0x532A, 
    0x537F, 0x539F, 0x539D, 0x53DF, 0x54E8, 0x5510, 0x5501, 0x5537, 0x54FC, 
    0x54E5, 0x54F2, 0x5506, 0x54FA, 0x5514, 0x54E9, 0x54ED, 0x54E1, 0x5509, 
    0x54EE, 0x54EA},
    {0x54E6, 0x5527, 0x5507, 0x54FD, 0x550F, 0x5703, 0x5704, 0x57C2, 0x57D4, 
    0x57CB, 0x57C3, 0x5809, 0x590F, 0x5957, 0x5958, 0x595A, 0x5A11, 0x5A18, 
    0x5A1C, 0x5A1F, 0x5A1B, 0x5A13, 0x59EC, 0x5A20, 0x5A23, 0x5A29, 0x5A25, 
    0x5A0C, 0x5A09, 0x5B6B, 0x5C58, 0x5BB0, 0x5BB3, 0x5BB6, 0x5BB4, 0x5BAE, 
    0x5BB5, 0x5BB9, 0x5BB8, 0x5C04, 0x5C51, 0x5C55, 0x5C50, 0x5CED, 0x5CFD, 
    0x5CFB, 0x5CEA, 0x5CE8, 0x5CF0, 0x5CF6, 0x5D01, 0x5CF4, 0x5DEE, 0x5E2D, 
    0x5E2B, 0x5EAB, 0x5EAD, 0x5EA7, 0x5F31, 0x5F92, 0x5F91, 0x5F90, 0x6059, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x6063, 0x6065, 
    0x6050, 0x6055, 0x606D, 0x6069, 0x606F, 0x6084, 0x609F, 0x609A, 0x608D, 
    0x6094, 0x608C, 0x6085, 0x6096, 0x6247, 0x62F3, 0x6308, 0x62FF, 0x634E, 
    0x633E, 0x632F, 0x6355, 0x6342, 0x6346, 0x634F, 0x6349, 0x633A, 0x6350, 
    0x633D, 0x632A, 0x632B, 0x6328, 0x634D, 0x634C, 0x6548, 0x6549, 0x6599, 
    0x65C1, 0x65C5, 0x6642, 0x6649, 0x664F, 0x6643, 0x6652, 0x664C, 0x6645, 
    0x6641, 0x66F8, 0x6714, 0x6715, 0x6717, 0x6821, 0x6838, 0x6848, 0x6846, 
    0x6853, 0x6839, 0x6842, 0x6854, 0x6829, 0x68B3, 0x6817, 0x684C, 0x6851, 
    0x683D, 0x67F4, 0x6850, 0x6840, 0x683C, 0x6843, 0x682A, 0x6845, 0x6813, 
    0x6818, 0x6841, 0x6B8A, 0x6B89, 0x6BB7, 0x6C23, 0x6C27, 0x6C28, 0x6C26, 
    0x6C24, 0x6CF0, 0x6D6A, 0x6D95, 0x6D88, 0x6D87, 0x6D66, 0x6D78, 0x6D77, 
    0x6D59, 0x6D93},
    {0x6D6C, 0x6D89, 0x6D6E, 0x6D5A, 0x6D74, 0x6D69, 0x6D8C, 0x6D8A, 0x6D79, 
    0x6D85, 0x6D65, 0x6D94, 0x70CA, 0x70D8, 0x70E4, 0x70D9, 0x70C8, 0x70CF, 
    0x7239, 0x7279, 0x72FC, 0x72F9, 0x72FD, 0x72F8, 0x72F7, 0x7386, 0x73ED, 
    0x7409, 0x73EE, 0x73E0, 0x73EA, 0x73DE, 0x7554, 0x755D, 0x755C, 0x755A, 
    0x7559, 0x75BE, 0x75C5, 0x75C7, 0x75B2, 0x75B3, 0x75BD, 0x75BC, 0x75B9, 
    0x75C2, 0x75B8, 0x768B, 0x76B0, 0x76CA, 0x76CD, 0x76CE, 0x7729, 0x771F, 
    0x7720, 0x7728, 0x77E9, 0x7830, 0x7827, 0x7838, 0x781D, 0x7834, 0x7837, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x7825, 0x782D, 
    0x7820, 0x781F, 0x7832, 0x7955, 0x7950, 0x7960, 0x795F, 0x7956, 0x795E, 
    0x795D, 0x7957, 0x795A, 0x79E4, 0x79E3, 0x79E7, 0x79DF, 0x79E6, 0x79E9, 
    0x79D8, 0x7A84, 0x7A88, 0x7AD9, 0x7B06, 0x7B11, 0x7C89, 0x7D21, 0x7D17, 
    0x7D0B, 0x7D0A, 0x7D20, 0x7D22, 0x7D14, 0x7D10, 0x7D15, 0x7D1A, 0x7D1C, 
    0x7D0D, 0x7D19, 0x7D1B, 0x7F3A, 0x7F5F, 0x7F94, 0x7FC5, 0x7FC1, 0x8006, 
    0x8018, 0x8015, 0x8019, 0x8017, 0x803D, 0x803F, 0x80F1, 0x8102, 0x80F0, 
    0x8105, 0x80ED, 0x80F4, 0x8106, 0x80F8, 0x80F3, 0x8108, 0x80FD, 0x810A, 
    0x80FC, 0x80EF, 0x81ED, 0x81EC, 0x8200, 0x8210, 0x822A, 0x822B, 0x8228, 
    0x822C, 0x82BB, 0x832B, 0x8352, 0x8354, 0x834A, 0x8338, 0x8350, 0x8349, 
    0x8335, 0x8334, 0x834F, 0x8332, 0x8339, 0x8336, 0x8317, 0x8340, 0x8331, 
    0x8328, 0x8343},
    {0x8654, 0x868A, 0x86AA, 0x8693, 0x86A4, 0x86A9, 0x868C, 0x86A3, 0x869C, 
    0x8870, 0x8877, 0x8881, 0x8882, 0x887D, 0x8879, 0x8A18, 0x8A10, 0x8A0E, 
    0x8A0C, 0x8A15, 0x8A0A, 0x8A17, 0x8A13, 0x8A16, 0x8A0F, 0x8A11, 0x8C48, 
    0x8C7A, 0x8C79, 0x8CA1, 0x8CA2, 0x8D77, 0x8EAC, 0x8ED2, 0x8ED4, 0x8ECF, 
    0x8FB1, 0x9001, 0x9006, 0x8FF7, 0x9000, 0x8FFA, 0x8FF4, 0x9003, 0x8FFD, 
    0x9005, 0x8FF8, 0x9095, 0x90E1, 0x90DD, 0x90E2, 0x9152, 0x914D, 0x914C, 
    0x91D8, 0x91DD, 0x91D7, 0x91DC, 0x91D9, 0x9583, 0x9662, 0x9663, 0x9661, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x965B, 0x965D, 
    0x9664, 0x9658, 0x965E, 0x96BB, 0x98E2, 0x99AC, 0x9AA8, 0x9AD8, 0x9B25, 
    0x9B32, 0x9B3C, 0x4E7E, 0x507A, 0x507D, 0x505C, 0x5047, 0x5043, 0x504C, 
    0x505A, 0x5049, 0x5065, 0x5076, 0x504E, 0x5055, 0x5075, 0x5074, 0x5077, 
    0x504F, 0x500F, 0x506F, 0x506D, 0x515C, 0x5195, 0x51F0, 0x526A, 0x526F, 
    0x52D2, 0x52D9, 0x52D8, 0x52D5, 0x5310, 0x530F, 0x5319, 0x533F, 0x5340, 
    0x533E, 0x53C3, 0x66FC, 0x5546, 0x556A, 0x5566, 0x5544, 0x555E, 0x5561, 
    0x5543, 0x554A, 0x5531, 0x5556, 0x554F, 0x5555, 0x552F, 0x5564, 0x5538, 
    0x552E, 0x555C, 0x552C, 0x5563, 0x5533, 0x5541, 0x5557, 0x5708, 0x570B, 
    0x5709, 0x57DF, 0x5805, 0x580A, 0x5806, 0x57E0, 0x57E4, 0x57FA, 0x5802, 
    0x5835, 0x57F7, 0x57F9, 0x5920, 0x5962, 0x5A36, 0x5A41, 0x5A49, 0x5A66, 
    0x5A6A, 0x5A40},
    {0x5A3C, 0x5A62, 0x5A5A, 0x5A46, 0x5A4A, 0x5B70, 0x5BC7, 0x5BC5, 0x5BC4, 
    0x5BC2, 0x5BBF, 0x5BC6, 0x5C09, 0x5C08, 0x5C07, 0x5C60, 0x5C5C, 0x5C5D, 
    0x5D07, 0x5D06, 0x5D0E, 0x5D1B, 0x5D16, 0x5D22, 0x5D11, 0x5D29, 0x5D14, 
    0x5D19, 0x5D24, 0x5D27, 0x5D17, 0x5DE2, 0x5E38, 0x5E36, 0x5E33, 0x5E37, 
    0x5EB7, 0x5EB8, 0x5EB6, 0x5EB5, 0x5EBE, 0x5F35, 0x5F37, 0x5F57, 0x5F6C, 
    0x5F69, 0x5F6B, 0x5F97, 0x5F99, 0x5F9E, 0x5F98, 0x5FA1, 0x5FA0, 0x5F9C, 
    0x607F, 0x60A3, 0x6089, 0x60A0, 0x60A8, 0x60CB, 0x60B4, 0x60E6, 0x60BD, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x60C5, 0x60BB, 
    0x60B5, 0x60DC, 0x60BC, 0x60D8, 0x60D5, 0x60C6, 0x60DF, 0x60B8, 0x60DA, 
    0x60C7, 0x621A, 0x621B, 0x6248, 0x63A0, 0x63A7, 0x6372, 0x6396, 0x63A2, 
    0x63A5, 0x6377, 0x6367, 0x6398, 0x63AA, 0x6371, 0x63A9, 0x6389, 0x6383, 
    0x639B, 0x636B, 0x63A8, 0x6384, 0x6388, 0x6399, 0x63A1, 0x63AC, 0x6392, 
    0x638F, 0x6380, 0x637B, 0x6369, 0x6368, 0x637A, 0x655D, 0x6556, 0x6551, 
    0x6559, 0x6557, 0x555F, 0x654F, 0x6558, 0x6555, 0x6554, 0x659C, 0x659B, 
    0x65AC, 0x65CF, 0x65CB, 0x65CC, 0x65CE, 0x665D, 0x665A, 0x6664, 0x6668, 
    0x6666, 0x665E, 0x66F9, 0x52D7, 0x671B, 0x6881, 0x68AF, 0x68A2, 0x6893, 
    0x68B5, 0x687F, 0x6876, 0x68B1, 0x68A7, 0x6897, 0x68B0, 0x6883, 0x68C4, 
    0x68AD, 0x6886, 0x6885, 0x6894, 0x689D, 0x68A8, 0x689F, 0x68A1, 0x6882, 
    0x6B32, 0x6BBA},
    {0x6BEB, 0x6BEC, 0x6C2B, 0x6D8E, 0x6DBC, 0x6DF3, 0x6DD9, 0x6DB2, 0x6DE1, 
    0x6DCC, 0x6DE4, 0x6DFB, 0x6DFA, 0x6E05, 0x6DC7, 0x6DCB, 0x6DAF, 0x6DD1, 
    0x6DAE, 0x6DDE, 0x6DF9, 0x6DB8, 0x6DF7, 0x6DF5, 0x6DC5, 0x6DD2, 0x6E1A, 
    0x6DB5, 0x6DDA, 0x6DEB, 0x6DD8, 0x6DEA, 0x6DF1, 0x6DEE, 0x6DE8, 0x6DC6, 
    0x6DC4, 0x6DAA, 0x6DEC, 0x6DBF, 0x6DE6, 0x70F9, 0x7109, 0x710A, 0x70FD, 
    0x70EF, 0x723D, 0x727D, 0x7281, 0x731C, 0x731B, 0x7316, 0x7313, 0x7319, 
    0x7387, 0x7405, 0x740A, 0x7403, 0x7406, 0x73FE, 0x740D, 0x74E0, 0x74F6, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x74F7, 0x751C, 
    0x7522, 0x7565, 0x7566, 0x7562, 0x7570, 0x758F, 0x75D4, 0x75D5, 0x75B5, 
    0x75CA, 0x75CD, 0x768E, 0x76D4, 0x76D2, 0x76DB, 0x7737, 0x773E, 0x773C, 
    0x7736, 0x7738, 0x773A, 0x786B, 0x7843, 0x784E, 0x7965, 0x7968, 0x796D, 
    0x79FB, 0x7A92, 0x7A95, 0x7B20, 0x7B28, 0x7B1B, 0x7B2C, 0x7B26, 0x7B19, 
    0x7B1E, 0x7B2E, 0x7C92, 0x7C97, 0x7C95, 0x7D46, 0x7D43, 0x7D71, 0x7D2E, 
    0x7D39, 0x7D3C, 0x7D40, 0x7D30, 0x7D33, 0x7D44, 0x7D2F, 0x7D42, 0x7D32, 
    0x7D31, 0x7F3D, 0x7F9E, 0x7F9A, 0x7FCC, 0x7FCE, 0x7FD2, 0x801C, 0x804A, 
    0x8046, 0x812F, 0x8116, 0x8123, 0x812B, 0x8129, 0x8130, 0x8124, 0x8202, 
    0x8235, 0x8237, 0x8236, 0x8239, 0x838E, 0x839E, 0x8398, 0x8378, 0x83A2, 
    0x8396, 0x83BD, 0x83AB, 0x8392, 0x838A, 0x8393, 0x8389, 0x83A0, 0x8377, 
    0x837B, 0x837C},
    {0x8386, 0x83A7, 0x8655, 0x5F6A, 0x86C7, 0x86C0, 0x86B6, 0x86C4, 0x86B5, 
    0x86C6, 0x86CB, 0x86B1, 0x86AF, 0x86C9, 0x8853, 0x889E, 0x8888, 0x88AB, 
    0x8892, 0x8896, 0x888D, 0x888B, 0x8993, 0x898F, 0x8A2A, 0x8A1D, 0x8A23, 
    0x8A25, 0x8A31, 0x8A2D, 0x8A1F, 0x8A1B, 0x8A22, 0x8C49, 0x8C5A, 0x8CA9, 
    0x8CAC, 0x8CAB, 0x8CA8, 0x8CAA, 0x8CA7, 0x8D67, 0x8D66, 0x8DBE, 0x8DBA, 
    0x8EDB, 0x8EDF, 0x9019, 0x900D, 0x901A, 0x9017, 0x9023, 0x901F, 0x901D, 
    0x9010, 0x9015, 0x901E, 0x9020, 0x900F, 0x9022, 0x9016, 0x901B, 0x9014, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x90E8, 0x90ED, 
    0x90FD, 0x9157, 0x91CE, 0x91F5, 0x91E6, 0x91E3, 0x91E7, 0x91ED, 0x91E9, 
    0x9589, 0x966A, 0x9675, 0x9673, 0x9678, 0x9670, 0x9674, 0x9676, 0x9677, 
    0x966C, 0x96C0, 0x96EA, 0x96E9, 0x7AE0, 0x7ADF, 0x9802, 0x9803, 0x9B5A, 
    0x9CE5, 0x9E75, 0x9E7F, 0x9EA5, 0x9EBB, 0x50A2, 0x508D, 0x5085, 0x5099, 
    0x5091, 0x5080, 0x5096, 0x5098, 0x509A, 0x6700, 0x51F1, 0x5272, 0x5274, 
    0x5275, 0x5269, 0x52DE, 0x52DD, 0x52DB, 0x535A, 0x53A5, 0x557B, 0x5580, 
    0x55A7, 0x557C, 0x558A, 0x559D, 0x5598, 0x5582, 0x559C, 0x55AA, 0x5594, 
    0x5587, 0x558B, 0x5583, 0x55B3, 0x55AE, 0x559F, 0x553E, 0x55B2, 0x559A, 
    0x55BB, 0x55AC, 0x55B1, 0x557E, 0x5589, 0x55AB, 0x5599, 0x570D, 0x582F, 
    0x582A, 0x5834, 0x5824, 0x5830, 0x5831, 0x5821, 0x581D, 0x5820, 0x58F9, 
    0x58FA, 0x5960},
    {0x5A77, 0x5A9A, 0x5A7F, 0x5A92, 0x5A9B, 0x5AA7, 0x5B73, 0x5B71, 0x5BD2, 
    0x5BCC, 0x5BD3, 0x5BD0, 0x5C0A, 0x5C0B, 0x5C31, 0x5D4C, 0x5D50, 0x5D34, 
    0x5D47, 0x5DFD, 0x5E45, 0x5E3D, 0x5E40, 0x5E43, 0x5E7E, 0x5ECA, 0x5EC1, 
    0x5EC2, 0x5EC4, 0x5F3C, 0x5F6D, 0x5FA9, 0x5FAA, 0x5FA8, 0x60D1, 0x60E1, 
    0x60B2, 0x60B6, 0x60E0, 0x611C, 0x6123, 0x60FA, 0x6115, 0x60F0, 0x60FB, 
    0x60F4, 0x6168, 0x60F1, 0x610E, 0x60F6, 0x6109, 0x6100, 0x6112, 0x621F, 
    0x6249, 0x63A3, 0x638C, 0x63CF, 0x63C0, 0x63E9, 0x63C9, 0x63C6, 0x63CD, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x63D2, 0x63E3, 
    0x63D0, 0x63E1, 0x63D6, 0x63ED, 0x63EE, 0x6376, 0x63F4, 0x63EA, 0x63DB, 
    0x6452, 0x63DA, 0x63F9, 0x655E, 0x6566, 0x6562, 0x6563, 0x6591, 0x6590, 
    0x65AF, 0x666E, 0x6670, 0x6674, 0x6676, 0x666F, 0x6691, 0x667A, 0x667E, 
    0x6677, 0x66FE, 0x66FF, 0x671F, 0x671D, 0x68FA, 0x68D5, 0x68E0, 0x68D8, 
    0x68D7, 0x6905, 0x68DF, 0x68F5, 0x68EE, 0x68E7, 0x68F9, 0x68D2, 0x68F2, 
    0x68E3, 0x68CB, 0x68CD, 0x690D, 0x6912, 0x690E, 0x68C9, 0x68DA, 0x696E, 
    0x68FB, 0x6B3E, 0x6B3A, 0x6B3D, 0x6B98, 0x6B96, 0x6BBC, 0x6BEF, 0x6C2E, 
    0x6C2F, 0x6C2C, 0x6E2F, 0x6E38, 0x6E54, 0x6E21, 0x6E32, 0x6E67, 0x6E4A, 
    0x6E20, 0x6E25, 0x6E23, 0x6E1B, 0x6E5B, 0x6E58, 0x6E24, 0x6E56, 0x6E6E, 
    0x6E2D, 0x6E26, 0x6E6F, 0x6E34, 0x6E4D, 0x6E3A, 0x6E2C, 0x6E43, 0x6E1D, 
    0x6E3E, 0x6ECB},
    {0x6E89, 0x6E19, 0x6E4E, 0x6E63, 0x6E44, 0x6E72, 0x6E69, 0x6E5F, 0x7119, 
    0x711A, 0x7126, 0x7130, 0x7121, 0x7136, 0x716E, 0x711C, 0x724C, 0x7284, 
    0x7280, 0x7336, 0x7325, 0x7334, 0x7329, 0x743A, 0x742A, 0x7433, 0x7422, 
    0x7425, 0x7435, 0x7436, 0x7434, 0x742F, 0x741B, 0x7426, 0x7428, 0x7525, 
    0x7526, 0x756B, 0x756A, 0x75E2, 0x75DB, 0x75E3, 0x75D9, 0x75D8, 0x75DE, 
    0x75E0, 0x767B, 0x767C, 0x7696, 0x7693, 0x76B4, 0x76DC, 0x774F, 0x77ED, 
    0x785D, 0x786C, 0x786F, 0x7A0D, 0x7A08, 0x7A0B, 0x7A05, 0x7A00, 0x7A98, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x7A97, 0x7A96, 
    0x7AE5, 0x7AE3, 0x7B49, 0x7B56, 0x7B46, 0x7B50, 0x7B52, 0x7B54, 0x7B4D, 
    0x7B4B, 0x7B4F, 0x7B51, 0x7C9F, 0x7CA5, 0x7D5E, 0x7D50, 0x7D68, 0x7D55, 
    0x7D2B, 0x7D6E, 0x7D72, 0x7D61, 0x7D66, 0x7D62, 0x7D70, 0x7D73, 0x5584, 
    0x7FD4, 0x7FD5, 0x800B, 0x8052, 0x8085, 0x8155, 0x8154, 0x814B, 0x8151, 
    0x814E, 0x8139, 0x8146, 0x813E, 0x814C, 0x8153, 0x8174, 0x8212, 0x821C, 
    0x83E9, 0x8403, 0x83F8, 0x840D, 0x83E0, 0x83C5, 0x840B, 0x83C1, 0x83EF, 
    0x83F1, 0x83F4, 0x8457, 0x840A, 0x83F0, 0x840C, 0x83CC, 0x83FD, 0x83F2, 
    0x83CA, 0x8438, 0x840E, 0x8404, 0x83DC, 0x8407, 0x83D4, 0x83DF, 0x865B, 
    0x86DF, 0x86D9, 0x86ED, 0x86D4, 0x86DB, 0x86E4, 0x86D0, 0x86DE, 0x8857, 
    0x88C1, 0x88C2, 0x88B1, 0x8983, 0x8996, 0x8A3B, 0x8A60, 0x8A55, 0x8A5E, 
    0x8A3C, 0x8A41},
    {0x8A54, 0x8A5B, 0x8A50, 0x8A46, 0x8A34, 0x8A3A, 0x8A36, 0x8A56, 0x8C61, 
    0x8C82, 0x8CAF, 0x8CBC, 0x8CB3, 0x8CBD, 0x8CC1, 0x8CBB, 0x8CC0, 0x8CB4, 
    0x8CB7, 0x8CB6, 0x8CBF, 0x8CB8, 0x8D8A, 0x8D85, 0x8D81, 0x8DCE, 0x8DDD, 
    0x8DCB, 0x8DDA, 0x8DD1, 0x8DCC, 0x8DDB, 0x8DC6, 0x8EFB, 0x8EF8, 0x8EFC, 
    0x8F9C, 0x902E, 0x9035, 0x9031, 0x9038, 0x9032, 0x9036, 0x9102, 0x90F5, 
    0x9109, 0x90FE, 0x9163, 0x9165, 0x91CF, 0x9214, 0x9215, 0x9223, 0x9209, 
    0x921E, 0x920D, 0x9210, 0x9207, 0x9211, 0x9594, 0x958F, 0x958B, 0x9591, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x9593, 0x9592, 
    0x958E, 0x968A, 0x968E, 0x968B, 0x967D, 0x9685, 0x9686, 0x968D, 0x9672, 
    0x9684, 0x96C1, 0x96C5, 0x96C4, 0x96C6, 0x96C7, 0x96EF, 0x96F2, 0x97CC, 
    0x9805, 0x9806, 0x9808, 0x98E7, 0x98EA, 0x98EF, 0x98E9, 0x98F2, 0x98ED, 
    0x99AE, 0x99AD, 0x9EC3, 0x9ECD, 0x9ED1, 0x4E82, 0x50AD, 0x50B5, 0x50B2, 
    0x50B3, 0x50C5, 0x50BE, 0x50AC, 0x50B7, 0x50BB, 0x50AF, 0x50C7, 0x527F, 
    0x5277, 0x527D, 0x52DF, 0x52E6, 0x52E4, 0x52E2, 0x52E3, 0x532F, 0x55DF, 
    0x55E8, 0x55D3, 0x55E6, 0x55CE, 0x55DC, 0x55C7, 0x55D1, 0x55E3, 0x55E4, 
    0x55EF, 0x55DA, 0x55E1, 0x55C5, 0x55C6, 0x55E5, 0x55C9, 0x5712, 0x5713, 
    0x585E, 0x5851, 0x5858, 0x5857, 0x585A, 0x5854, 0x586B, 0x584C, 0x586D, 
    0x584A, 0x5862, 0x5852, 0x584B, 0x5967, 0x5AC1, 0x5AC9, 0x5ACC, 0x5ABE, 
    0x5ABD, 0x5ABC},
    {0x5AB3, 0x5AC2, 0x5AB2, 0x5D69, 0x5D6F, 0x5E4C, 0x5E79, 0x5EC9, 0x5EC8, 
    0x5F12, 0x5F59, 0x5FAC, 0x5FAE, 0x611A, 0x610F, 0x6148, 0x611F, 0x60F3, 
    0x611B, 0x60F9, 0x6101, 0x6108, 0x614E, 0x614C, 0x6144, 0x614D, 0x613E, 
    0x6134, 0x6127, 0x610D, 0x6106, 0x6137, 0x6221, 0x6222, 0x6413, 0x643E, 
    0x641E, 0x642A, 0x642D, 0x643D, 0x642C, 0x640F, 0x641C, 0x6414, 0x640D, 
    0x6436, 0x6416, 0x6417, 0x6406, 0x656C, 0x659F, 0x65B0, 0x6697, 0x6689, 
    0x6687, 0x6688, 0x6696, 0x6684, 0x6698, 0x668D, 0x6703, 0x6994, 0x696D, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x695A, 0x6977, 
    0x6960, 0x6954, 0x6975, 0x6930, 0x6982, 0x694A, 0x6968, 0x696B, 0x695E, 
    0x6953, 0x6979, 0x6986, 0x695D, 0x6963, 0x695B, 0x6B47, 0x6B72, 0x6BC0, 
    0x6BBF, 0x6BD3, 0x6BFD, 0x6EA2, 0x6EAF, 0x6ED3, 0x6EB6, 0x6EC2, 0x6E90, 
    0x6E9D, 0x6EC7, 0x6EC5, 0x6EA5, 0x6E98, 0x6EBC, 0x6EBA, 0x6EAB, 0x6ED1, 
    0x6E96, 0x6E9C, 0x6EC4, 0x6ED4, 0x6EAA, 0x6EA7, 0x6EB4, 0x714E, 0x7159, 
    0x7169, 0x7164, 0x7149, 0x7167, 0x715C, 0x716C, 0x7166, 0x714C, 0x7165, 
    0x715E, 0x7146, 0x7168, 0x7156, 0x723A, 0x7252, 0x7337, 0x7345, 0x733F, 
    0x733E, 0x746F, 0x745A, 0x7455, 0x745F, 0x745E, 0x7441, 0x743F, 0x7459, 
    0x745B, 0x745C, 0x7576, 0x7578, 0x7600, 0x75F0, 0x7601, 0x75F2, 0x75F1, 
    0x75FA, 0x75FF, 0x75F4, 0x75F3, 0x76DE, 0x76DF, 0x775B, 0x776B, 0x7766, 
    0x775E, 0x7763},
    {0x7779, 0x776A, 0x776C, 0x775C, 0x7765, 0x7768, 0x7762, 0x77EE, 0x788E, 
    0x78B0, 0x7897, 0x7898, 0x788C, 0x7889, 0x787C, 0x7891, 0x7893, 0x787F, 
    0x797A, 0x797F, 0x7981, 0x842C, 0x79BD, 0x7A1C, 0x7A1A, 0x7A20, 0x7A14, 
    0x7A1F, 0x7A1E, 0x7A9F, 0x7AA0, 0x7B77, 0x7BC0, 0x7B60, 0x7B6E, 0x7B67, 
    0x7CB1, 0x7CB3, 0x7CB5, 0x7D93, 0x7D79, 0x7D91, 0x7D81, 0x7D8F, 0x7D5B, 
    0x7F6E, 0x7F69, 0x7F6A, 0x7F72, 0x7FA9, 0x7FA8, 0x7FA4, 0x8056, 0x8058, 
    0x8086, 0x8084, 0x8171, 0x8170, 0x8178, 0x8165, 0x816E, 0x8173, 0x816B, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x8179, 0x817A, 
    0x8166, 0x8205, 0x8247, 0x8482, 0x8477, 0x843D, 0x8431, 0x8475, 0x8466, 
    0x846B, 0x8449, 0x846C, 0x845B, 0x843C, 0x8435, 0x8461, 0x8463, 0x8469, 
    0x846D, 0x8446, 0x865E, 0x865C, 0x865F, 0x86F9, 0x8713, 0x8708, 0x8707, 
    0x8700, 0x86FE, 0x86FB, 0x8702, 0x8703, 0x8706, 0x870A, 0x8859, 0x88DF, 
    0x88D4, 0x88D9, 0x88DC, 0x88D8, 0x88DD, 0x88E1, 0x88CA, 0x88D5, 0x88D2, 
    0x899C, 0x89E3, 0x8A6B, 0x8A72, 0x8A73, 0x8A66, 0x8A69, 0x8A70, 0x8A87, 
    0x8A7C, 0x8A63, 0x8AA0, 0x8A71, 0x8A85, 0x8A6D, 0x8A62, 0x8A6E, 0x8A6C, 
    0x8A79, 0x8A7B, 0x8A3E, 0x8A68, 0x8C62, 0x8C8A, 0x8C89, 0x8CCA, 0x8CC7, 
    0x8CC8, 0x8CC4, 0x8CB2, 0x8CC3, 0x8CC2, 0x8CC5, 0x8DE1, 0x8DDF, 0x8DE8, 
    0x8DEF, 0x8DF3, 0x8DFA, 0x8DEA, 0x8DE4, 0x8DE6, 0x8EB2, 0x8F03, 0x8F09, 
    0x8EFE, 0x8F0A},
    {0x8F9F, 0x8FB2, 0x904B, 0x904A, 0x9053, 0x9042, 0x9054, 0x903C, 0x9055, 
    0x9050, 0x9047, 0x904F, 0x904E, 0x904D, 0x9051, 0x903E, 0x9041, 0x9112, 
    0x9117, 0x916C, 0x916A, 0x9169, 0x91C9, 0x9237, 0x9257, 0x9238, 0x923D, 
    0x9240, 0x923E, 0x925B, 0x924B, 0x9264, 0x9251, 0x9234, 0x9249, 0x924D, 
    0x9245, 0x9239, 0x923F, 0x925A, 0x9598, 0x9698, 0x9694, 0x9695, 0x96CD, 
    0x96CB, 0x96C9, 0x96CA, 0x96F7, 0x96FB, 0x96F9, 0x96F6, 0x9756, 0x9774, 
    0x9776, 0x9810, 0x9811, 0x9813, 0x980A, 0x9812, 0x980C, 0x98FC, 0x98F4, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x98FD, 0x98FE, 
    0x99B3, 0x99B1, 0x99B4, 0x9AE1, 0x9CE9, 0x9E82, 0x9F0E, 0x9F13, 0x9F20, 
    0x50E7, 0x50EE, 0x50E5, 0x50D6, 0x50ED, 0x50DA, 0x50D5, 0x50CF, 0x50D1, 
    0x50F1, 0x50CE, 0x50E9, 0x5162, 0x51F3, 0x5283, 0x5282, 0x5331, 0x53AD, 
    0x55FE, 0x5600, 0x561B, 0x5617, 0x55FD, 0x5614, 0x5606, 0x5609, 0x560D, 
    0x560E, 0x55F7, 0x5616, 0x561F, 0x5608, 0x5610, 0x55F6, 0x5718, 0x5716, 
    0x5875, 0x587E, 0x5883, 0x5893, 0x588A, 0x5879, 0x5885, 0x587D, 0x58FD, 
    0x5925, 0x5922, 0x5924, 0x596A, 0x5969, 0x5AE1, 0x5AE6, 0x5AE9, 0x5AD7, 
    0x5AD6, 0x5AD8, 0x5AE3, 0x5B75, 0x5BDE, 0x5BE7, 0x5BE1, 0x5BE5, 0x5BE6, 
    0x5BE8, 0x5BE2, 0x5BE4, 0x5BDF, 0x5C0D, 0x5C62, 0x5D84, 0x5D87, 0x5E5B, 
    0x5E63, 0x5E55, 0x5E57, 0x5E54, 0x5ED3, 0x5ED6, 0x5F0A, 0x5F46, 0x5F70, 
    0x5FB9, 0x6147},
    {0x613F, 0x614B, 0x6177, 0x6162, 0x6163, 0x615F, 0x615A, 0x6158, 0x6175, 
    0x622A, 0x6487, 0x6458, 0x6454, 0x64A4, 0x6478, 0x645F, 0x647A, 0x6451, 
    0x6467, 0x6434, 0x646D, 0x647B, 0x6572, 0x65A1, 0x65D7, 0x65D6, 0x66A2, 
    0x66A8, 0x669D, 0x699C, 0x69A8, 0x6995, 0x69C1, 0x69AE, 0x69D3, 0x69CB, 
    0x699B, 0x69B7, 0x69BB, 0x69AB, 0x69B4, 0x69D0, 0x69CD, 0x69AD, 0x69CC, 
    0x69A6, 0x69C3, 0x69A3, 0x6B49, 0x6B4C, 0x6C33, 0x6F33, 0x6F14, 0x6EFE, 
    0x6F13, 0x6EF4, 0x6F29, 0x6F3E, 0x6F20, 0x6F2C, 0x6F0F, 0x6F02, 0x6F22, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x6EFF, 0x6EEF, 
    0x6F06, 0x6F31, 0x6F38, 0x6F32, 0x6F23, 0x6F15, 0x6F2B, 0x6F2F, 0x6F88, 
    0x6F2A, 0x6EEC, 0x6F01, 0x6EF2, 0x6ECC, 0x6EF7, 0x7194, 0x7199, 0x717D, 
    0x718A, 0x7184, 0x7192, 0x723E, 0x7292, 0x7296, 0x7344, 0x7350, 0x7464, 
    0x7463, 0x746A, 0x7470, 0x746D, 0x7504, 0x7591, 0x7627, 0x760D, 0x760B, 
    0x7609, 0x7613, 0x76E1, 0x76E3, 0x7784, 0x777D, 0x777F, 0x7761, 0x78C1, 
    0x789F, 0x78A7, 0x78B3, 0x78A9, 0x78A3, 0x798E, 0x798F, 0x798D, 0x7A2E, 
    0x7A31, 0x7AAA, 0x7AA9, 0x7AED, 0x7AEF, 0x7BA1, 0x7B95, 0x7B8B, 0x7B75, 
    0x7B97, 0x7B9D, 0x7B94, 0x7B8F, 0x7BB8, 0x7B87, 0x7B84, 0x7CB9, 0x7CBD, 
    0x7CBE, 0x7DBB, 0x7DB0, 0x7D9C, 0x7DBD, 0x7DBE, 0x7DA0, 0x7DCA, 0x7DB4, 
    0x7DB2, 0x7DB1, 0x7DBA, 0x7DA2, 0x7DBF, 0x7DB5, 0x7DB8, 0x7DAD, 0x7DD2, 
    0x7DC7, 0x7DAC},
    {0x7F70, 0x7FE0, 0x7FE1, 0x7FDF, 0x805E, 0x805A, 0x8087, 0x8150, 0x8180, 
    0x818F, 0x8188, 0x818A, 0x817F, 0x8182, 0x81E7, 0x81FA, 0x8207, 0x8214, 
    0x821E, 0x824B, 0x84C9, 0x84BF, 0x84C6, 0x84C4, 0x8499, 0x849E, 0x84B2, 
    0x849C, 0x84CB, 0x84B8, 0x84C0, 0x84D3, 0x8490, 0x84BC, 0x84D1, 0x84CA, 
    0x873F, 0x871C, 0x873B, 0x8722, 0x8725, 0x8734, 0x8718, 0x8755, 0x8737, 
    0x8729, 0x88F3, 0x8902, 0x88F4, 0x88F9, 0x88F8, 0x88FD, 0x88E8, 0x891A, 
    0x88EF, 0x8AA6, 0x8A8C, 0x8A9E, 0x8AA3, 0x8A8D, 0x8AA1, 0x8A93, 0x8AA4, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x8AAA, 0x8AA5, 
    0x8AA8, 0x8A98, 0x8A91, 0x8A9A, 0x8AA7, 0x8C6A, 0x8C8D, 0x8C8C, 0x8CD3, 
    0x8CD1, 0x8CD2, 0x8D6B, 0x8D99, 0x8D95, 0x8DFC, 0x8F14, 0x8F12, 0x8F15, 
    0x8F13, 0x8FA3, 0x9060, 0x9058, 0x905C, 0x9063, 0x9059, 0x905E, 0x9062, 
    0x905D, 0x905B, 0x9119, 0x9118, 0x911E, 0x9175, 0x9178, 0x9177, 0x9174, 
    0x9278, 0x9280, 0x9285, 0x9298, 0x9296, 0x927B, 0x9293, 0x929C, 0x92A8, 
    0x927C, 0x9291, 0x95A1, 0x95A8, 0x95A9, 0x95A3, 0x95A5, 0x95A4, 0x9699, 
    0x969C, 0x969B, 0x96CC, 0x96D2, 0x9700, 0x977C, 0x9785, 0x97F6, 0x9817, 
    0x9818, 0x98AF, 0x98B1, 0x9903, 0x9905, 0x990C, 0x9909, 0x99C1, 0x9AAF, 
    0x9AB0, 0x9AE6, 0x9B41, 0x9B42, 0x9CF4, 0x9CF6, 0x9CF3, 0x9EBC, 0x9F3B, 
    0x9F4A, 0x5104, 0x5100, 0x50FB, 0x50F5, 0x50F9, 0x5102, 0x5108, 0x5109, 
    0x5105, 0x51DC},
    {0x5287, 0x5288, 0x5289, 0x528D, 0x528A, 0x52F0, 0x53B2, 0x562E, 0x563B, 
    0x5639, 0x5632, 0x563F, 0x5634, 0x5629, 0x5653, 0x564E, 0x5657, 0x5674, 
    0x5636, 0x562F, 0x5630, 0x5880, 0x589F, 0x589E, 0x58B3, 0x589C, 0x58AE, 
    0x58A9, 0x58A6, 0x596D, 0x5B09, 0x5AFB, 0x5B0B, 0x5AF5, 0x5B0C, 0x5B08, 
    0x5BEE, 0x5BEC, 0x5BE9, 0x5BEB, 0x5C64, 0x5C65, 0x5D9D, 0x5D94, 0x5E62, 
    0x5E5F, 0x5E61, 0x5EE2, 0x5EDA, 0x5EDF, 0x5EDD, 0x5EE3, 0x5EE0, 0x5F48, 
    0x5F71, 0x5FB7, 0x5FB5, 0x6176, 0x6167, 0x616E, 0x615D, 0x6155, 0x6182, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x617C, 0x6170, 
    0x616B, 0x617E, 0x61A7, 0x6190, 0x61AB, 0x618E, 0x61AC, 0x619A, 0x61A4, 
    0x6194, 0x61AE, 0x622E, 0x6469, 0x646F, 0x6479, 0x649E, 0x64B2, 0x6488, 
    0x6490, 0x64B0, 0x64A5, 0x6493, 0x6495, 0x64A9, 0x6492, 0x64AE, 0x64AD, 
    0x64AB, 0x649A, 0x64AC, 0x6499, 0x64A2, 0x64B3, 0x6575, 0x6577, 0x6578, 
    0x66AE, 0x66AB, 0x66B4, 0x66B1, 0x6A23, 0x6A1F, 0x69E8, 0x6A01, 0x6A1E, 
    0x6A19, 0x69FD, 0x6A21, 0x6A13, 0x6A0A, 0x69F3, 0x6A02, 0x6A05, 0x69ED, 
    0x6A11, 0x6B50, 0x6B4E, 0x6BA4, 0x6BC5, 0x6BC6, 0x6F3F, 0x6F7C, 0x6F84, 
    0x6F51, 0x6F66, 0x6F54, 0x6F86, 0x6F6D, 0x6F5B, 0x6F78, 0x6F6E, 0x6F8E, 
    0x6F7A, 0x6F70, 0x6F64, 0x6F97, 0x6F58, 0x6ED5, 0x6F6F, 0x6F60, 0x6F5F, 
    0x719F, 0x71AC, 0x71B1, 0x71A8, 0x7256, 0x729B, 0x734E, 0x7357, 0x7469, 
    0x748B, 0x7483},
    {0x747E, 0x7480, 0x757F, 0x7620, 0x7629, 0x761F, 0x7624, 0x7626, 0x7621, 
    0x7622, 0x769A, 0x76BA, 0x76E4, 0x778E, 0x7787, 0x778C, 0x7791, 0x778B, 
    0x78CB, 0x78C5, 0x78BA, 0x78CA, 0x78BE, 0x78D5, 0x78BC, 0x78D0, 0x7A3F, 
    0x7A3C, 0x7A40, 0x7A3D, 0x7A37, 0x7A3B, 0x7AAF, 0x7AAE, 0x7BAD, 0x7BB1, 
    0x7BC4, 0x7BB4, 0x7BC6, 0x7BC7, 0x7BC1, 0x7BA0, 0x7BCC, 0x7CCA, 0x7DE0, 
    0x7DF4, 0x7DEF, 0x7DFB, 0x7DD8, 0x7DEC, 0x7DDD, 0x7DE8, 0x7DE3, 0x7DDA, 
    0x7DDE, 0x7DE9, 0x7D9E, 0x7DD9, 0x7DF2, 0x7DF9, 0x7F75, 0x7F77, 0x7FAF, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x7FE9, 0x8026, 
    0x819B, 0x819C, 0x819D, 0x81A0, 0x819A, 0x8198, 0x8517, 0x853D, 0x851A, 
    0x84EE, 0x852C, 0x852D, 0x8513, 0x8511, 0x8523, 0x8521, 0x8514, 0x84EC, 
    0x8525, 0x84FF, 0x8506, 0x8782, 0x8774, 0x8776, 0x8760, 0x8766, 0x8778, 
    0x8768, 0x8759, 0x8757, 0x874C, 0x8753, 0x885B, 0x885D, 0x8910, 0x8907, 
    0x8912, 0x8913, 0x8915, 0x890A, 0x8ABC, 0x8AD2, 0x8AC7, 0x8AC4, 0x8A95, 
    0x8ACB, 0x8AF8, 0x8AB2, 0x8AC9, 0x8AC2, 0x8ABF, 0x8AB0, 0x8AD6, 0x8ACD, 
    0x8AB6, 0x8AB9, 0x8ADB, 0x8C4C, 0x8C4E, 0x8C6C, 0x8CE0, 0x8CDE, 0x8CE6, 
    0x8CE4, 0x8CEC, 0x8CED, 0x8CE2, 0x8CE3, 0x8CDC, 0x8CEA, 0x8CE1, 0x8D6D, 
    0x8D9F, 0x8DA3, 0x8E2B, 0x8E10, 0x8E1D, 0x8E22, 0x8E0F, 0x8E29, 0x8E1F, 
    0x8E21, 0x8E1E, 0x8EBA, 0x8F1D, 0x8F1B, 0x8F1F, 0x8F29, 0x8F26, 0x8F2A, 
    0x8F1C, 0x8F1E},
    {0x8F25, 0x9069, 0x906E, 0x9068, 0x906D, 0x9077, 0x9130, 0x912D, 0x9127, 
    0x9131, 0x9187, 0x9189, 0x918B, 0x9183, 0x92C5, 0x92BB, 0x92B7, 0x92EA, 
    0x92AC, 0x92E4, 0x92C1, 0x92B3, 0x92BC, 0x92D2, 0x92C7, 0x92F0, 0x92B2, 
    0x95AD, 0x95B1, 0x9704, 0x9706, 0x9707, 0x9709, 0x9760, 0x978D, 0x978B, 
    0x978F, 0x9821, 0x982B, 0x981C, 0x98B3, 0x990A, 0x9913, 0x9912, 0x9918, 
    0x99DD, 0x99D0, 0x99DF, 0x99DB, 0x99D1, 0x99D5, 0x99D2, 0x99D9, 0x9AB7, 
    0x9AEE, 0x9AEF, 0x9B27, 0x9B45, 0x9B44, 0x9B77, 0x9B6F, 0x9D06, 0x9D09, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x9D03, 0x9EA9, 
    0x9EBE, 0x9ECE, 0x58A8, 0x9F52, 0x5112, 0x5118, 0x5114, 0x5110, 0x5115, 
    0x5180, 0x51AA, 0x51DD, 0x5291, 0x5293, 0x52F3, 0x5659, 0x566B, 0x5679, 
    0x5669, 0x5664, 0x5678, 0x566A, 0x5668, 0x5665, 0x5671, 0x566F, 0x566C, 
    0x5662, 0x5676, 0x58C1, 0x58BE, 0x58C7, 0x58C5, 0x596E, 0x5B1D, 0x5B34, 
    0x5B78, 0x5BF0, 0x5C0E, 0x5F4A, 0x61B2, 0x6191, 0x61A9, 0x618A, 0x61CD, 
    0x61B6, 0x61BE, 0x61CA, 0x61C8, 0x6230, 0x64C5, 0x64C1, 0x64CB, 0x64BB, 
    0x64BC, 0x64DA, 0x64C4, 0x64C7, 0x64C2, 0x64CD, 0x64BF, 0x64D2, 0x64D4, 
    0x64BE, 0x6574, 0x66C6, 0x66C9, 0x66B9, 0x66C4, 0x66C7, 0x66B8, 0x6A3D, 
    0x6A38, 0x6A3A, 0x6A59, 0x6A6B, 0x6A58, 0x6A39, 0x6A44, 0x6A62, 0x6A61, 
    0x6A4B, 0x6A47, 0x6A35, 0x6A5F, 0x6A48, 0x6B59, 0x6B77, 0x6C05, 0x6FC2, 
    0x6FB1, 0x6FA1},
    {0x6FC3, 0x6FA4, 0x6FC1, 0x6FA7, 0x6FB3, 0x6FC0, 0x6FB9, 0x6FB6, 0x6FA6, 
    0x6FA0, 0x6FB4, 0x71BE, 0x71C9, 0x71D0, 0x71D2, 0x71C8, 0x71D5, 0x71B9, 
    0x71CE, 0x71D9, 0x71DC, 0x71C3, 0x71C4, 0x7368, 0x749C, 0x74A3, 0x7498, 
    0x749F, 0x749E, 0x74E2, 0x750C, 0x750D, 0x7634, 0x7638, 0x763A, 0x76E7, 
    0x76E5, 0x77A0, 0x779E, 0x779F, 0x77A5, 0x78E8, 0x78DA, 0x78EC, 0x78E7, 
    0x79A6, 0x7A4D, 0x7A4E, 0x7A46, 0x7A4C, 0x7A4B, 0x7ABA, 0x7BD9, 0x7C11, 
    0x7BC9, 0x7BE4, 0x7BDB, 0x7BE1, 0x7BE9, 0x7BE6, 0x7CD5, 0x7CD6, 0x7E0A, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x7E11, 0x7E08, 
    0x7E1B, 0x7E23, 0x7E1E, 0x7E1D, 0x7E09, 0x7E10, 0x7F79, 0x7FB2, 0x7FF0, 
    0x7FF1, 0x7FEE, 0x8028, 0x81B3, 0x81A9, 0x81A8, 0x81FB, 0x8208, 0x8258, 
    0x8259, 0x854A, 0x8559, 0x8548, 0x8568, 0x8569, 0x8543, 0x8549, 0x856D, 
    0x856A, 0x855E, 0x8783, 0x879F, 0x879E, 0x87A2, 0x878D, 0x8861, 0x892A, 
    0x8932, 0x8925, 0x892B, 0x8921, 0x89AA, 0x89A6, 0x8AE6, 0x8AFA, 0x8AEB, 
    0x8AF1, 0x8B00, 0x8ADC, 0x8AE7, 0x8AEE, 0x8AFE, 0x8B01, 0x8B02, 0x8AF7, 
    0x8AED, 0x8AF3, 0x8AF6, 0x8AFC, 0x8C6B, 0x8C6D, 0x8C93, 0x8CF4, 0x8E44, 
    0x8E31, 0x8E34, 0x8E42, 0x8E39, 0x8E35, 0x8F3B, 0x8F2F, 0x8F38, 0x8F33, 
    0x8FA8, 0x8FA6, 0x9075, 0x9074, 0x9078, 0x9072, 0x907C, 0x907A, 0x9134, 
    0x9192, 0x9320, 0x9336, 0x92F8, 0x9333, 0x932F, 0x9322, 0x92FC, 0x932B, 
    0x9304, 0x931A},
    {0x9310, 0x9326, 0x9321, 0x9315, 0x932E, 0x9319, 0x95BB, 0x96A7, 0x96A8, 
    0x96AA, 0x96D5, 0x970E, 0x9711, 0x9716, 0x970D, 0x9713, 0x970F, 0x975B, 
    0x975C, 0x9766, 0x9798, 0x9830, 0x9838, 0x983B, 0x9837, 0x982D, 0x9839, 
    0x9824, 0x9910, 0x9928, 0x991E, 0x991B, 0x9921, 0x991A, 0x99ED, 0x99E2, 
    0x99F1, 0x9AB8, 0x9ABC, 0x9AFB, 0x9AED, 0x9B28, 0x9B91, 0x9D15, 0x9D23, 
    0x9D26, 0x9D28, 0x9D12, 0x9D1B, 0x9ED8, 0x9ED4, 0x9F8D, 0x9F9C, 0x512A, 
    0x511F, 0x5121, 0x5132, 0x52F5, 0x568E, 0x5680, 0x5690, 0x5685, 0x5687, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x568F, 0x58D5, 
    0x58D3, 0x58D1, 0x58CE, 0x5B30, 0x5B2A, 0x5B24, 0x5B7A, 0x5C37, 0x5C68, 
    0x5DBC, 0x5DBA, 0x5DBD, 0x5DB8, 0x5E6B, 0x5F4C, 0x5FBD, 0x61C9, 0x61C2, 
    0x61C7, 0x61E6, 0x61CB, 0x6232, 0x6234, 0x64CE, 0x64CA, 0x64D8, 0x64E0, 
    0x64F0, 0x64E6, 0x64EC, 0x64F1, 0x64E2, 0x64ED, 0x6582, 0x6583, 0x66D9, 
    0x66D6, 0x6A80, 0x6A94, 0x6A84, 0x6AA2, 0x6A9C, 0x6ADB, 0x6AA3, 0x6A7E, 
    0x6A97, 0x6A90, 0x6AA0, 0x6B5C, 0x6BAE, 0x6BDA, 0x6C08, 0x6FD8, 0x6FF1, 
    0x6FDF, 0x6FE0, 0x6FDB, 0x6FE4, 0x6FEB, 0x6FEF, 0x6F80, 0x6FEC, 0x6FE1, 
    0x6FE9, 0x6FD5, 0x6FEE, 0x6FF0, 0x71E7, 0x71DF, 0x71EE, 0x71E6, 0x71E5, 
    0x71ED, 0x71EC, 0x71F4, 0x71E0, 0x7235, 0x7246, 0x7370, 0x7372, 0x74A9, 
    0x74B0, 0x74A6, 0x74A8, 0x7646, 0x7642, 0x764C, 0x76EA, 0x77B3, 0x77AA, 
    0x77B0, 0x77AC},
    {0x77A7, 0x77AD, 0x77EF, 0x78F7, 0x78FA, 0x78F4, 0x78EF, 0x7901, 0x79A7, 
    0x79AA, 0x7A57, 0x7ABF, 0x7C07, 0x7C0D, 0x7BFE, 0x7BF7, 0x7C0C, 0x7BE0, 
    0x7CE0, 0x7CDC, 0x7CDE, 0x7CE2, 0x7CDF, 0x7CD9, 0x7CDD, 0x7E2E, 0x7E3E, 
    0x7E46, 0x7E37, 0x7E32, 0x7E43, 0x7E2B, 0x7E3D, 0x7E31, 0x7E45, 0x7E41, 
    0x7E34, 0x7E39, 0x7E48, 0x7E35, 0x7E3F, 0x7E2F, 0x7F44, 0x7FF3, 0x7FFC, 
    0x8071, 0x8072, 0x8070, 0x806F, 0x8073, 0x81C6, 0x81C3, 0x81BA, 0x81C2, 
    0x81C0, 0x81BF, 0x81BD, 0x81C9, 0x81BE, 0x81E8, 0x8209, 0x8271, 0x85AA, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x8584, 0x857E, 
    0x859C, 0x8591, 0x8594, 0x85AF, 0x859B, 0x8587, 0x85A8, 0x858A, 0x8667, 
    0x87C0, 0x87D1, 0x87B3, 0x87D2, 0x87C6, 0x87AB, 0x87BB, 0x87BA, 0x87C8, 
    0x87CB, 0x893B, 0x8936, 0x8944, 0x8938, 0x893D, 0x89AC, 0x8B0E, 0x8B17, 
    0x8B19, 0x8B1B, 0x8B0A, 0x8B20, 0x8B1D, 0x8B04, 0x8B10, 0x8C41, 0x8C3F, 
    0x8C73, 0x8CFA, 0x8CFD, 0x8CFC, 0x8CF8, 0x8CFB, 0x8DA8, 0x8E49, 0x8E4B, 
    0x8E48, 0x8E4A, 0x8F44, 0x8F3E, 0x8F42, 0x8F45, 0x8F3F, 0x907F, 0x907D, 
    0x9084, 0x9081, 0x9082, 0x9080, 0x9139, 0x91A3, 0x919E, 0x919C, 0x934D, 
    0x9382, 0x9328, 0x9375, 0x934A, 0x9365, 0x934B, 0x9318, 0x937E, 0x936C, 
    0x935B, 0x9370, 0x935A, 0x9354, 0x95CA, 0x95CB, 0x95CC, 0x95C8, 0x95C6, 
    0x96B1, 0x96B8, 0x96D6, 0x971C, 0x971E, 0x97A0, 0x97D3, 0x9846, 0x98B6, 
    0x9935, 0x9A01},
    {0x99FF, 0x9BAE, 0x9BAB, 0x9BAA, 0x9BAD, 0x9D3B, 0x9D3F, 0x9E8B, 0x9ECF, 
    0x9EDE, 0x9EDC, 0x9EDD, 0x9EDB, 0x9F3E, 0x9F4B, 0x53E2, 0x5695, 0x56AE, 
    0x58D9, 0x58D8, 0x5B38, 0x5F5D, 0x61E3, 0x6233, 0x64F4, 0x64F2, 0x64FE, 
    0x6506, 0x64FA, 0x64FB, 0x64F7, 0x65B7, 0x66DC, 0x6726, 0x6AB3, 0x6AAC, 
    0x6AC3, 0x6ABB, 0x6AB8, 0x6AC2, 0x6AAE, 0x6AAF, 0x6B5F, 0x6B78, 0x6BAF, 
    0x7009, 0x700B, 0x6FFE, 0x7006, 0x6FFA, 0x7011, 0x700F, 0x71FB, 0x71FC, 
    0x71FE, 0x71F8, 0x7377, 0x7375, 0x74A7, 0x74BF, 0x7515, 0x7656, 0x7658, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x7652, 0x77BD, 
    0x77BF, 0x77BB, 0x77BC, 0x790E, 0x79AE, 0x7A61, 0x7A62, 0x7A60, 0x7AC4, 
    0x7AC5, 0x7C2B, 0x7C27, 0x7C2A, 0x7C1E, 0x7C23, 0x7C21, 0x7CE7, 0x7E54, 
    0x7E55, 0x7E5E, 0x7E5A, 0x7E61, 0x7E52, 0x7E59, 0x7F48, 0x7FF9, 0x7FFB, 
    0x8077, 0x8076, 0x81CD, 0x81CF, 0x820A, 0x85CF, 0x85A9, 0x85CD, 0x85D0, 
    0x85C9, 0x85B0, 0x85BA, 0x85B9, 0x85A6, 0x87EF, 0x87EC, 0x87F2, 0x87E0, 
    0x8986, 0x89B2, 0x89F4, 0x8B28, 0x8B39, 0x8B2C, 0x8B2B, 0x8C50, 0x8D05, 
    0x8E59, 0x8E63, 0x8E66, 0x8E64, 0x8E5F, 0x8E55, 0x8EC0, 0x8F49, 0x8F4D, 
    0x9087, 0x9083, 0x9088, 0x91AB, 0x91AC, 0x91D0, 0x9394, 0x938A, 0x9396, 
    0x93A2, 0x93B3, 0x93AE, 0x93AC, 0x93B0, 0x9398, 0x939A, 0x9397, 0x95D4, 
    0x95D6, 0x95D0, 0x95D5, 0x96E2, 0x96DC, 0x96D9, 0x96DB, 0x96DE, 0x9724, 
    0x97A3, 0x97A6},
    {0x97AD, 0x97F9, 0x984D, 0x984F, 0x984C, 0x984E, 0x9853, 0x98BA, 0x993E, 
    0x993F, 0x993D, 0x992E, 0x99A5, 0x9A0E, 0x9AC1, 0x9B03, 0x9B06, 0x9B4F, 
    0x9B4E, 0x9B4D, 0x9BCA, 0x9BC9, 0x9BFD, 0x9BC8, 0x9BC0, 0x9D51, 0x9D5D, 
    0x9D60, 0x9EE0, 0x9F15, 0x9F2C, 0x5133, 0x56A5, 0x58DE, 0x58DF, 0x58E2, 
    0x5BF5, 0x9F90, 0x5EEC, 0x61F2, 0x61F7, 0x61F6, 0x61F5, 0x6500, 0x650F, 
    0x66E0, 0x66DD, 0x6AE5, 0x6ADD, 0x6ADA, 0x6AD3, 0x701B, 0x701F, 0x7028, 
    0x701A, 0x701D, 0x7015, 0x7018, 0x7206, 0x720D, 0x7258, 0x72A2, 0x7378, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x737A, 0x74BD, 
    0x74CA, 0x74E3, 0x7587, 0x7586, 0x765F, 0x7661, 0x77C7, 0x7919, 0x79B1, 
    0x7A6B, 0x7A69, 0x7C3E, 0x7C3F, 0x7C38, 0x7C3D, 0x7C37, 0x7C40, 0x7E6B, 
    0x7E6D, 0x7E79, 0x7E69, 0x7E6A, 0x7F85, 0x7E73, 0x7FB6, 0x7FB9, 0x7FB8, 
    0x81D8, 0x85E9, 0x85DD, 0x85EA, 0x85D5, 0x85E4, 0x85E5, 0x85F7, 0x87FB, 
    0x8805, 0x880D, 0x87F9, 0x87FE, 0x8960, 0x895F, 0x8956, 0x895E, 0x8B41, 
    0x8B5C, 0x8B58, 0x8B49, 0x8B5A, 0x8B4E, 0x8B4F, 0x8B46, 0x8B59, 0x8D08, 
    0x8D0A, 0x8E7C, 0x8E72, 0x8E87, 0x8E76, 0x8E6C, 0x8E7A, 0x8E74, 0x8F54, 
    0x8F4E, 0x8FAD, 0x908A, 0x908B, 0x91B1, 0x91AE, 0x93E1, 0x93D1, 0x93DF, 
    0x93C3, 0x93C8, 0x93DC, 0x93DD, 0x93D6, 0x93E2, 0x93CD, 0x93D8, 0x93E4, 
    0x93D7, 0x93E8, 0x95DC, 0x96B4, 0x96E3, 0x972A, 0x9727, 0x9761, 0x97DC, 
    0x97FB, 0x985E},
    {0x9858, 0x985B, 0x98BC, 0x9945, 0x9949, 0x9A16, 0x9A19, 0x9B0D, 0x9BE8, 
    0x9BE7, 0x9BD6, 0x9BDB, 0x9D89, 0x9D61, 0x9D72, 0x9D6A, 0x9D6C, 0x9E92, 
    0x9E97, 0x9E93, 0x9EB4, 0x52F8, 0x56A8, 0x56B7, 0x56B6, 0x56B4, 0x56BC, 
    0x58E4, 0x5B40, 0x5B43, 0x5B7D, 0x5BF6, 0x5DC9, 0x61F8, 0x61FA, 0x6518, 
    0x6514, 0x6519, 0x66E6, 0x6727, 0x6AEC, 0x703E, 0x7030, 0x7032, 0x7210, 
    0x737B, 0x74CF, 0x7662, 0x7665, 0x7926, 0x792A, 0x792C, 0x792B, 0x7AC7, 
    0x7AF6, 0x7C4C, 0x7C43, 0x7C4D, 0x7CEF, 0x7CF0, 0x8FAE, 0x7E7D, 0x7E7C, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x7E82, 0x7F4C, 
    0x8000, 0x81DA, 0x8266, 0x85FB, 0x85F9, 0x8611, 0x85FA, 0x8606, 0x860B, 
    0x8607, 0x860A, 0x8814, 0x8815, 0x8964, 0x89BA, 0x89F8, 0x8B70, 0x8B6C, 
    0x8B66, 0x8B6F, 0x8B5F, 0x8B6B, 0x8D0F, 0x8D0D, 0x8E89, 0x8E81, 0x8E85, 
    0x8E82, 0x91B4, 0x91CB, 0x9418, 0x9403, 0x93FD, 0x95E1, 0x9730, 0x98C4, 
    0x9952, 0x9951, 0x99A8, 0x9A2B, 0x9A30, 0x9A37, 0x9A35, 0x9C13, 0x9C0D, 
    0x9E79, 0x9EB5, 0x9EE8, 0x9F2F, 0x9F5F, 0x9F63, 0x9F61, 0x5137, 0x5138, 
    0x56C1, 0x56C0, 0x56C2, 0x5914, 0x5C6C, 0x5DCD, 0x61FC, 0x61FE, 0x651D, 
    0x651C, 0x6595, 0x66E9, 0x6AFB, 0x6B04, 0x6AFA, 0x6BB2, 0x704C, 0x721B, 
    0x72A7, 0x74D6, 0x74D4, 0x7669, 0x77D3, 0x7C50, 0x7E8F, 0x7E8C, 0x7FBC, 
    0x8617, 0x862D, 0x861A, 0x8823, 0x8822, 0x8821, 0x881F, 0x896A, 0x896C, 
    0x89BD, 0x8B74},
    {0x8B77, 0x8B7D, 0x8D13, 0x8E8A, 0x8E8D, 0x8E8B, 0x8F5F, 0x8FAF, 0x91BA, 
    0x942E, 0x9433, 0x9435, 0x943A, 0x9438, 0x9432, 0x942B, 0x95E2, 0x9738, 
    0x9739, 0x9732, 0x97FF, 0x9867, 0x9865, 0x9957, 0x9A45, 0x9A43, 0x9A40, 
    0x9A3E, 0x9ACF, 0x9B54, 0x9B51, 0x9C2D, 0x9C25, 0x9DAF, 0x9DB4, 0x9DC2, 
    0x9DB8, 0x9E9D, 0x9EEF, 0x9F19, 0x9F5C, 0x9F66, 0x9F67, 0x513C, 0x513B, 
    0x56C8, 0x56CA, 0x56C9, 0x5B7F, 0x5DD4, 0x5DD2, 0x5F4E, 0x61FF, 0x6524, 
    0x6B0A, 0x6B61, 0x7051, 0x7058, 0x7380, 0x74E4, 0x758A, 0x766E, 0x766C, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x79B3, 0x7C60, 
    0x7C5F, 0x807E, 0x807D, 0x81DF, 0x8972, 0x896F, 0x89FC, 0x8B80, 0x8D16, 
    0x8D17, 0x8E91, 0x8E93, 0x8F61, 0x9148, 0x9444, 0x9451, 0x9452, 0x973D, 
    0x973E, 0x97C3, 0x97C1, 0x986B, 0x9955, 0x9A55, 0x9A4D, 0x9AD2, 0x9B1A, 
    0x9C49, 0x9C31, 0x9C3E, 0x9C3B, 0x9DD3, 0x9DD7, 0x9F34, 0x9F6C, 0x9F6A, 
    0x9F94, 0x56CC, 0x5DD6, 0x6200, 0x6523, 0x652B, 0x652A, 0x66EC, 0x6B10, 
    0x74DA, 0x7ACA, 0x7C64, 0x7C63, 0x7C65, 0x7E93, 0x7E96, 0x7E94, 0x81E2, 
    0x8638, 0x863F, 0x8831, 0x8B8A, 0x9090, 0x908F, 0x9463, 0x9460, 0x9464, 
    0x9768, 0x986F, 0x995C, 0x9A5A, 0x9A5B, 0x9A57, 0x9AD3, 0x9AD4, 0x9AD1, 
    0x9C54, 0x9C57, 0x9C56, 0x9DE5, 0x9E9F, 0x9EF4, 0x56D1, 0x58E9, 0x652C, 
    0x705E, 0x7671, 0x7672, 0x77D7, 0x7F50, 0x7F88, 0x8836, 0x8839, 0x8862, 
    0x8B93, 0x8B92},
    {0x8B96, 0x8277, 0x8D1B, 0x91C0, 0x946A, 0x9742, 0x9748, 0x9744, 0x97C6, 
    0x9870, 0x9A5F, 0x9B22, 0x9B58, 0x9C5F, 0x9DF9, 0x9DFA, 0x9E7C, 0x9E7D, 
    0x9F07, 0x9F77, 0x9F72, 0x5EF3, 0x6B16, 0x7063, 0x7C6C, 0x7C6E, 0x883B, 
    0x89C0, 0x8EA1, 0x91C1, 0x9472, 0x9470, 0x9871, 0x995E, 0x9AD6, 0x9B23, 
    0x9ECC, 0x7064, 0x77DA, 0x8B9A, 0x9477, 0x97C9, 0x9A62, 0x9A65, 0x7E9C, 
    0x8B9C, 0x8EAA, 0x91C5, 0x947D, 0x947E, 0x947C, 0x9C77, 0x9C78, 0x9EF7, 
    0x8C54, 0x947F, 0x9E1A, 0x7228, 0x9A6A, 0x9B31, 0x9E1B, 0x9E1E, 0x7C72, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x30FE, 0x309D, 
    0x309E, 0x3005, 0x3041, 0x3042, 0x3043, 0x3044, 0x3045, 0x3046, 0x3047, 
    0x3048, 0x3049, 0x304A, 0x304B, 0x304C, 0x304D, 0x304E, 0x304F, 0x3050, 
    0x3051, 0x3052, 0x3053, 0x3054, 0x3055, 0x3056, 0x3057, 0x3058, 0x3059, 
    0x305A, 0x305B, 0x305C, 0x305D, 0x305E, 0x305F, 0x3060, 0x3061, 0x3062, 
    0x3063, 0x3064, 0x3065, 0x3066, 0x3067, 0x3068, 0x3069, 0x306A, 0x306B, 
    0x306C, 0x306D, 0x306E, 0x306F, 0x3070, 0x3071, 0x3072, 0x3073, 0x3074, 
    0x3075, 0x3076, 0x3077, 0x3078, 0x3079, 0x307A, 0x307B, 0x307C, 0x307D, 
    0x307E, 0x307F, 0x3080, 0x3081, 0x3082, 0x3083, 0x3084, 0x3085, 0x3086, 
    0x3087, 0x3088, 0x3089, 0x308A, 0x308B, 0x308C, 0x308D, 0x308E, 0x308F, 
    0x3090, 0x3091, 0x3092, 0x3093, 0x30A1, 0x30A2, 0x30A3, 0x30A4, 0x30A5, 
    0x30A6, 0x30A7},
    {0x30A8, 0x30A9, 0x30AA, 0x30AB, 0x30AC, 0x30AD, 0x30AE, 0x30AF, 0x30B0, 
    0x30B1, 0x30B2, 0x30B3, 0x30B4, 0x30B5, 0x30B6, 0x30B7, 0x30B8, 0x30B9, 
    0x30BA, 0x30BB, 0x30BC, 0x30BD, 0x30BE, 0x30BF, 0x30C0, 0x30C1, 0x30C2, 
    0x30C3, 0x30C4, 0x30C5, 0x30C6, 0x30C7, 0x30C8, 0x30C9, 0x30CA, 0x30CB, 
    0x30CC, 0x30CD, 0x30CE, 0x30CF, 0x30D0, 0x30D1, 0x30D2, 0x30D3, 0x30D4, 
    0x30D5, 0x30D6, 0x30D7, 0x30D8, 0x30D9, 0x30DA, 0x30DB, 0x30DC, 0x30DD, 
    0x30DE, 0x30DF, 0x30E0, 0x30E1, 0x30E2, 0x30E3, 0x30E4, 0x30E5, 0x30E6, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x30E7, 0x30E8, 
    0x30E9, 0x30EA, 0x30EB, 0x30EC, 0x30ED, 0x30EE, 0x30EF, 0x30F0, 0x30F1, 
    0x30F2, 0x30F3, 0x30F4, 0x30F5, 0x30F6, 0x0414, 0x0415, 0x0401, 0x0416, 
    0x0417, 0x0418, 0x0419, 0x041A, 0x041B, 0x041C, 0x0423, 0x0424, 0x0425, 
    0x0426, 0x0427, 0x0428, 0x0429, 0x042A, 0x042B, 0x042C, 0x042D, 0x042E, 
    0x042F, 0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0451, 0x0436, 
    0x0437, 0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 0x043D, 0x043E, 0x043F, 
    0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447, 0x0448, 
    0x0449, 0x044A, 0x044B, 0x044C, 0x044D, 0x044E, 0x044F, 0x2460, 0x2461, 
    0x2462, 0x2463, 0x2464, 0x2465, 0x2466, 0x2467, 0x2468, 0x2469, 0x2474, 
    0x2475, 0x2476, 0x2477, 0x2478, 0x2479, 0x247A, 0x247B, 0x247C, 0x247D, 
     ERROR,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR},
    {0x4E42, 0x4E5C, 0x51F5, 0x531A, 0x5382, 0x4E07, 0x4E0C, 0x4E47, 0x4E8D, 
    0x56D7, 0xFA0C, 0x5C6E, 0x5F73, 0x4E0F, 0x5187, 0x4E0E, 0x4E2E, 0x4E93, 
    0x4EC2, 0x4EC9, 0x4EC8, 0x5198, 0x52FC, 0x536C, 0x53B9, 0x5720, 0x5903, 
    0x592C, 0x5C10, 0x5DFF, 0x65E1, 0x6BB3, 0x6BCC, 0x6C14, 0x723F, 0x4E31, 
    0x4E3C, 0x4EE8, 0x4EDC, 0x4EE9, 0x4EE1, 0x4EDD, 0x4EDA, 0x520C, 0x531C, 
    0x534C, 0x5722, 0x5723, 0x5917, 0x592F, 0x5B81, 0x5B84, 0x5C12, 0x5C3B, 
    0x5C74, 0x5C73, 0x5E04, 0x5E80, 0x5E82, 0x5FC9, 0x6209, 0x6250, 0x6C15, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x6C36, 0x6C43, 
    0x6C3F, 0x6C3B, 0x72AE, 0x72B0, 0x738A, 0x79B8, 0x808A, 0x961E, 0x4F0E, 
    0x4F18, 0x4F2C, 0x4EF5, 0x4F14, 0x4EF1, 0x4F00, 0x4EF7, 0x4F08, 0x4F1D, 
    0x4F02, 0x4F05, 0x4F22, 0x4F13, 0x4F04, 0x4EF4, 0x4F12, 0x51B1, 0x5213, 
    0x5209, 0x5210, 0x52A6, 0x5322, 0x531F, 0x534D, 0x538A, 0x5407, 0x56E1, 
    0x56DF, 0x572E, 0x572A, 0x5734, 0x593C, 0x5980, 0x597C, 0x5985, 0x597B, 
    0x597E, 0x5977, 0x597F, 0x5B56, 0x5C15, 0x5C25, 0x5C7C, 0x5C7A, 0x5C7B, 
    0x5C7E, 0x5DDF, 0x5E75, 0x5E84, 0x5F02, 0x5F1A, 0x5F74, 0x5FD5, 0x5FD4, 
    0x5FCF, 0x625C, 0x625E, 0x6264, 0x6261, 0x6266, 0x6262, 0x6259, 0x6260, 
    0x625A, 0x6265, 0x65EF, 0x65EE, 0x673E, 0x6739, 0x6738, 0x673B, 0x673A, 
    0x673F, 0x673C, 0x6733, 0x6C18, 0x6C46, 0x6C52, 0x6C5C, 0x6C4F, 0x6C4A, 
    0x6C54, 0x6C4B},
    {0x6C4C, 0x7071, 0x725E, 0x72B4, 0x72B5, 0x738E, 0x752A, 0x767F, 0x7A75, 
    0x7F51, 0x8278, 0x827C, 0x8280, 0x827D, 0x827F, 0x864D, 0x897E, 0x9099, 
    0x9097, 0x9098, 0x909B, 0x9094, 0x9622, 0x9624, 0x9620, 0x9623, 0x4F56, 
    0x4F3B, 0x4F62, 0x4F49, 0x4F53, 0x4F64, 0x4F3E, 0x4F67, 0x4F52, 0x4F5F, 
    0x4F41, 0x4F58, 0x4F2D, 0x4F33, 0x4F3F, 0x4F61, 0x518F, 0x51B9, 0x521C, 
    0x521E, 0x5221, 0x52AD, 0x52AE, 0x5309, 0x5363, 0x5372, 0x538E, 0x538F, 
    0x5430, 0x5437, 0x542A, 0x5454, 0x5445, 0x5419, 0x541C, 0x5425, 0x5418, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x543D, 0x544F, 
    0x5441, 0x5428, 0x5424, 0x5447, 0x56EE, 0x56E7, 0x56E5, 0x5741, 0x5745, 
    0x574C, 0x5749, 0x574B, 0x5752, 0x5906, 0x5940, 0x59A6, 0x5998, 0x59A0, 
    0x5997, 0x598E, 0x59A2, 0x5990, 0x598F, 0x59A7, 0x59A1, 0x5B8E, 0x5B92, 
    0x5C28, 0x5C2A, 0x5C8D, 0x5C8F, 0x5C88, 0x5C8B, 0x5C89, 0x5C92, 0x5C8A, 
    0x5C86, 0x5C93, 0x5C95, 0x5DE0, 0x5E0A, 0x5E0E, 0x5E8B, 0x5E89, 0x5E8C, 
    0x5E88, 0x5E8D, 0x5F05, 0x5F1D, 0x5F78, 0x5F76, 0x5FD2, 0x5FD1, 0x5FD0, 
    0x5FED, 0x5FE8, 0x5FEE, 0x5FF3, 0x5FE1, 0x5FE4, 0x5FE3, 0x5FFA, 0x5FEF, 
    0x5FF7, 0x5FFB, 0x6000, 0x5FF4, 0x623A, 0x6283, 0x628C, 0x628E, 0x628F, 
    0x6294, 0x6287, 0x6271, 0x627B, 0x627A, 0x6270, 0x6281, 0x6288, 0x6277, 
    0x627D, 0x6272, 0x6274, 0x6537, 0x65F0, 0x65F4, 0x65F3, 0x65F2, 0x65F5, 
    0x6745, 0x6747},
    {0x6759, 0x6755, 0x674C, 0x6748, 0x675D, 0x674D, 0x675A, 0x674B, 0x6BD0, 
    0x6C19, 0x6C1A, 0x6C78, 0x6C67, 0x6C6B, 0x6C84, 0x6C8B, 0x6C8F, 0x6C71, 
    0x6C6F, 0x6C69, 0x6C9A, 0x6C6D, 0x6C87, 0x6C95, 0x6C9C, 0x6C66, 0x6C73, 
    0x6C65, 0x6C7B, 0x6C8E, 0x7074, 0x707A, 0x7263, 0x72BF, 0x72BD, 0x72C3, 
    0x72C6, 0x72C1, 0x72BA, 0x72C5, 0x7395, 0x7397, 0x7393, 0x7394, 0x7392, 
    0x753A, 0x7539, 0x7594, 0x7595, 0x7681, 0x793D, 0x8034, 0x8095, 0x8099, 
    0x8090, 0x8092, 0x809C, 0x8290, 0x828F, 0x8285, 0x828E, 0x8291, 0x8293, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x828A, 0x8283, 
    0x8284, 0x8C78, 0x8FC9, 0x8FBF, 0x909F, 0x90A1, 0x90A5, 0x909E, 0x90A7, 
    0x90A0, 0x9630, 0x9628, 0x962F, 0x962D, 0x4E33, 0x4F98, 0x4F7C, 0x4F85, 
    0x4F7D, 0x4F80, 0x4F87, 0x4F76, 0x4F74, 0x4F89, 0x4F84, 0x4F77, 0x4F4C, 
    0x4F97, 0x4F6A, 0x4F9A, 0x4F79, 0x4F81, 0x4F78, 0x4F90, 0x4F9C, 0x4F94, 
    0x4F9E, 0x4F92, 0x4F82, 0x4F95, 0x4F6B, 0x4F6E, 0x519E, 0x51BC, 0x51BE, 
    0x5235, 0x5232, 0x5233, 0x5246, 0x5231, 0x52BC, 0x530A, 0x530B, 0x533C, 
    0x5392, 0x5394, 0x5487, 0x547F, 0x5481, 0x5491, 0x5482, 0x5488, 0x546B, 
    0x547A, 0x547E, 0x5465, 0x546C, 0x5474, 0x5466, 0x548D, 0x546F, 0x5461, 
    0x5460, 0x5498, 0x5463, 0x5467, 0x5464, 0x56F7, 0x56F9, 0x576F, 0x5772, 
    0x576D, 0x576B, 0x5771, 0x5770, 0x5776, 0x5780, 0x5775, 0x577B, 0x5773, 
    0x5774, 0x5762},
    {0x5768, 0x577D, 0x590C, 0x5945, 0x59B5, 0x59BA, 0x59CF, 0x59CE, 0x59B2, 
    0x59CC, 0x59C1, 0x59B6, 0x59BC, 0x59C3, 0x59D6, 0x59B1, 0x59BD, 0x59C0, 
    0x59C8, 0x59B4, 0x59C7, 0x5B62, 0x5B65, 0x5B93, 0x5B95, 0x5C44, 0x5C47, 
    0x5CAE, 0x5CA4, 0x5CA0, 0x5CB5, 0x5CAF, 0x5CA8, 0x5CAC, 0x5C9F, 0x5CA3, 
    0x5CAD, 0x5CA2, 0x5CAA, 0x5CA7, 0x5C9D, 0x5CA5, 0x5CB6, 0x5CB0, 0x5CA6, 
    0x5E17, 0x5E14, 0x5E19, 0x5F28, 0x5F22, 0x5F23, 0x5F24, 0x5F54, 0x5F82, 
    0x5F7E, 0x5F7D, 0x5FDE, 0x5FE5, 0x602D, 0x6026, 0x6019, 0x6032, 0x600B, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x6034, 0x600A, 
    0x6017, 0x6033, 0x601A, 0x601E, 0x602C, 0x6022, 0x600D, 0x6010, 0x602E, 
    0x6013, 0x6011, 0x600C, 0x6009, 0x601C, 0x6214, 0x623D, 0x62AD, 0x62B4, 
    0x62D1, 0x62BE, 0x62AA, 0x62B6, 0x62CA, 0x62AE, 0x62B3, 0x62AF, 0x62BB, 
    0x62A9, 0x62B0, 0x62B8, 0x653D, 0x65A8, 0x65BB, 0x6609, 0x65FC, 0x6604, 
    0x6612, 0x6608, 0x65FB, 0x6603, 0x660B, 0x660D, 0x6605, 0x65FD, 0x6611, 
    0x6610, 0x66F6, 0x670A, 0x6785, 0x676C, 0x678E, 0x6792, 0x6776, 0x677B, 
    0x6798, 0x6786, 0x6784, 0x6774, 0x678D, 0x678C, 0x677A, 0x679F, 0x6791, 
    0x6799, 0x6783, 0x677D, 0x6781, 0x6778, 0x6779, 0x6794, 0x6B25, 0x6B80, 
    0x6B7E, 0x6BDE, 0x6C1D, 0x6C93, 0x6CEC, 0x6CEB, 0x6CEE, 0x6CD9, 0x6CB6, 
    0x6CD4, 0x6CAD, 0x6CE7, 0x6CB7, 0x6CD0, 0x6CC2, 0x6CBA, 0x6CC3, 0x6CC6, 
    0x6CED, 0x6CF2},
    {0x6CD2, 0x6CDD, 0x6CB4, 0x6C8A, 0x6C9D, 0x6C80, 0x6CDE, 0x6CC0, 0x6D30, 
    0x6CCD, 0x6CC7, 0x6CB0, 0x6CF9, 0x6CCF, 0x6CE9, 0x6CD1, 0x7094, 0x7098, 
    0x7085, 0x7093, 0x7086, 0x7084, 0x7091, 0x7096, 0x7082, 0x709A, 0x7083, 
    0x726A, 0x72D6, 0x72CB, 0x72D8, 0x72C9, 0x72DC, 0x72D2, 0x72D4, 0x72DA, 
    0x72CC, 0x72D1, 0x73A4, 0x73A1, 0x73AD, 0x73A6, 0x73A2, 0x73A0, 0x73AC, 
    0x739D, 0x74DD, 0x74E8, 0x753F, 0x7540, 0x753E, 0x758C, 0x7598, 0x76AF, 
    0x76F3, 0x76F1, 0x76F0, 0x76F5, 0x77F8, 0x77FC, 0x77F9, 0x77FB, 0x77FA, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x77F7, 0x7942, 
    0x793F, 0x79C5, 0x7A78, 0x7A7B, 0x7AFB, 0x7C75, 0x7CFD, 0x8035, 0x808F, 
    0x80AE, 0x80A3, 0x80B8, 0x80B5, 0x80AD, 0x8220, 0x82A0, 0x82C0, 0x82AB, 
    0x829A, 0x8298, 0x829B, 0x82B5, 0x82A7, 0x82AE, 0x82BC, 0x829E, 0x82BA, 
    0x82B4, 0x82A8, 0x82A1, 0x82A9, 0x82C2, 0x82A4, 0x82C3, 0x82B6, 0x82A2, 
    0x8670, 0x866F, 0x866D, 0x866E, 0x8C56, 0x8FD2, 0x8FCB, 0x8FD3, 0x8FCD, 
    0x8FD6, 0x8FD5, 0x8FD7, 0x90B2, 0x90B4, 0x90AF, 0x90B3, 0x90B0, 0x9639, 
    0x963D, 0x963C, 0x963A, 0x9643, 0x4FCD, 0x4FC5, 0x4FD3, 0x4FB2, 0x4FC9, 
    0x4FCB, 0x4FC1, 0x4FD4, 0x4FDC, 0x4FD9, 0x4FBB, 0x4FB3, 0x4FDB, 0x4FC7, 
    0x4FD6, 0x4FBA, 0x4FC0, 0x4FB9, 0x4FEC, 0x5244, 0x5249, 0x52C0, 0x52C2, 
    0x533D, 0x537C, 0x5397, 0x5396, 0x5399, 0x5398, 0x54BA, 0x54A1, 0x54AD, 
    0x54A5, 0x54CF},
    {0x54C3, 0x830D, 0x54B7, 0x54AE, 0x54D6, 0x54B6, 0x54C5, 0x54C6, 0x54A0, 
    0x5470, 0x54BC, 0x54A2, 0x54BE, 0x5472, 0x54DE, 0x54B0, 0x57B5, 0x579E, 
    0x579F, 0x57A4, 0x578C, 0x5797, 0x579D, 0x579B, 0x5794, 0x5798, 0x578F, 
    0x5799, 0x57A5, 0x579A, 0x5795, 0x58F4, 0x590D, 0x5953, 0x59E1, 0x59DE, 
    0x59EE, 0x5A00, 0x59F1, 0x59DD, 0x59FA, 0x59FD, 0x59FC, 0x59F6, 0x59E4, 
    0x59F2, 0x59F7, 0x59DB, 0x59E9, 0x59F3, 0x59F5, 0x59E0, 0x59FE, 0x59F4, 
    0x59ED, 0x5BA8, 0x5C4C, 0x5CD0, 0x5CD8, 0x5CCC, 0x5CD7, 0x5CCB, 0x5CDB, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x5CDE, 0x5CDA, 
    0x5CC9, 0x5CC7, 0x5CCA, 0x5CD6, 0x5CD3, 0x5CD4, 0x5CCF, 0x5CC8, 0x5CC6, 
    0x5CCE, 0x5CDF, 0x5CF8, 0x5DF9, 0x5E21, 0x5E22, 0x5E23, 0x5E20, 0x5E24, 
    0x5EB0, 0x5EA4, 0x5EA2, 0x5E9B, 0x5EA3, 0x5EA5, 0x5F07, 0x5F2E, 0x5F56, 
    0x5F86, 0x6037, 0x6039, 0x6054, 0x6072, 0x605E, 0x6045, 0x6053, 0x6047, 
    0x6049, 0x605B, 0x604C, 0x6040, 0x6042, 0x605F, 0x6024, 0x6044, 0x6058, 
    0x6066, 0x606E, 0x6242, 0x6243, 0x62CF, 0x630D, 0x630B, 0x62F5, 0x630E, 
    0x6303, 0x62EB, 0x62F9, 0x630F, 0x630C, 0x62F8, 0x62F6, 0x6300, 0x6313, 
    0x6314, 0x62FA, 0x6315, 0x62FB, 0x62F0, 0x6541, 0x6543, 0x65AA, 0x65BF, 
    0x6636, 0x6621, 0x6632, 0x6635, 0x661C, 0x6626, 0x6622, 0x6633, 0x662B, 
    0x663A, 0x661D, 0x6634, 0x6639, 0x662E, 0x670F, 0x6710, 0x67C1, 0x67F2, 
    0x67C8, 0x67BA},
    {0x67DC, 0x67BB, 0x67F8, 0x67D8, 0x67C0, 0x67B7, 0x67C5, 0x67EB, 0x67E4, 
    0x67DF, 0x67B5, 0x67CD, 0x67B3, 0x67F7, 0x67F6, 0x67EE, 0x67E3, 0x67C2, 
    0x67B9, 0x67CE, 0x67E7, 0x67F0, 0x67B2, 0x67FC, 0x67C6, 0x67ED, 0x67CC, 
    0x67AE, 0x67E6, 0x67DB, 0x67FA, 0x67C9, 0x67CA, 0x67C3, 0x67EA, 0x67CB, 
    0x6B28, 0x6B82, 0x6B84, 0x6BB6, 0x6BD6, 0x6BD8, 0x6BE0, 0x6C20, 0x6C21, 
    0x6D28, 0x6D34, 0x6D2D, 0x6D1F, 0x6D3C, 0x6D3F, 0x6D12, 0x6D0A, 0x6CDA, 
    0x6D33, 0x6D04, 0x6D19, 0x6D3A, 0x6D1A, 0x6D11, 0x6D00, 0x6D1D, 0x6D42, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x6D01, 0x6D18, 
    0x6D37, 0x6D03, 0x6D0F, 0x6D40, 0x6D07, 0x6D20, 0x6D2C, 0x6D08, 0x6D22, 
    0x6D09, 0x6D10, 0x70B7, 0x709F, 0x70BE, 0x70B1, 0x70B0, 0x70A1, 0x70B4, 
    0x70B5, 0x70A9, 0x7241, 0x7249, 0x724A, 0x726C, 0x7270, 0x7273, 0x726E, 
    0x72CA, 0x72E4, 0x72E8, 0x72EB, 0x72DF, 0x72EA, 0x72E6, 0x72E3, 0x7385, 
    0x73CC, 0x73C2, 0x73C8, 0x73C5, 0x73B9, 0x73B6, 0x73B5, 0x73B4, 0x73EB, 
    0x73BF, 0x73C7, 0x73BE, 0x73C3, 0x73C6, 0x73B8, 0x73CB, 0x74EC, 0x74EE, 
    0x752E, 0x7547, 0x7548, 0x75A7, 0x75AA, 0x7679, 0x76C4, 0x7708, 0x7703, 
    0x7704, 0x7705, 0x770A, 0x76F7, 0x76FB, 0x76FA, 0x77E7, 0x77E8, 0x7806, 
    0x7811, 0x7812, 0x7805, 0x7810, 0x780F, 0x780E, 0x7809, 0x7803, 0x7813, 
    0x794A, 0x794C, 0x794B, 0x7945, 0x7944, 0x79D5, 0x79CD, 0x79CF, 0x79D6, 
    0x79CE, 0x7A80},
    {0x7A7E, 0x7AD1, 0x7B00, 0x7B01, 0x7C7A, 0x7C78, 0x7C79, 0x7C7F, 0x7C80, 
    0x7C81, 0x7D03, 0x7D08, 0x7D01, 0x7F58, 0x7F91, 0x7F8D, 0x7FBE, 0x8007, 
    0x800E, 0x800F, 0x8014, 0x8037, 0x80D8, 0x80C7, 0x80E0, 0x80D1, 0x80C8, 
    0x80C2, 0x80D0, 0x80C5, 0x80E3, 0x80D9, 0x80DC, 0x80CA, 0x80D5, 0x80C9, 
    0x80CF, 0x80D7, 0x80E6, 0x80CD, 0x81FF, 0x8221, 0x8294, 0x82D9, 0x82FE, 
    0x82F9, 0x8307, 0x82E8, 0x8300, 0x82D5, 0x833A, 0x82EB, 0x82D6, 0x82F4, 
    0x82EC, 0x82E1, 0x82F2, 0x82F5, 0x830C, 0x82FB, 0x82F6, 0x82F0, 0x82EA, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x82E4, 0x82E0, 
    0x82FA, 0x82F3, 0x82ED, 0x8677, 0x8674, 0x867C, 0x8673, 0x8841, 0x884E, 
    0x8867, 0x886A, 0x8869, 0x89D3, 0x8A04, 0x8A07, 0x8D72, 0x8FE3, 0x8FE1, 
    0x8FEE, 0x8FE0, 0x90F1, 0x90BD, 0x90BF, 0x90D5, 0x90C5, 0x90BE, 0x90C7, 
    0x90CB, 0x90C8, 0x91D4, 0x91D3, 0x9654, 0x964F, 0x9651, 0x9653, 0x964A, 
    0x964E, 0x501E, 0x5005, 0x5007, 0x5013, 0x5022, 0x5030, 0x501B, 0x4FF5, 
    0x4FF4, 0x5033, 0x5037, 0x502C, 0x4FF6, 0x4FF7, 0x5017, 0x501C, 0x5020, 
    0x5027, 0x5035, 0x502F, 0x5031, 0x500E, 0x515A, 0x5194, 0x5193, 0x51CA, 
    0x51C4, 0x51C5, 0x51C8, 0x51CE, 0x5261, 0x525A, 0x5252, 0x525E, 0x525F, 
    0x5255, 0x5262, 0x52CD, 0x530E, 0x539E, 0x5526, 0x54E2, 0x5517, 0x5512, 
    0x54E7, 0x54F3, 0x54E4, 0x551A, 0x54FF, 0x5504, 0x5508, 0x54EB, 0x5511, 
    0x5505, 0x54F1},
    {0x550A, 0x54FB, 0x54F7, 0x54F8, 0x54E0, 0x550E, 0x5503, 0x550B, 0x5701, 
    0x5702, 0x57CC, 0x5832, 0x57D5, 0x57D2, 0x57BA, 0x57C6, 0x57BD, 0x57BC, 
    0x57B8, 0x57B6, 0x57BF, 0x57C7, 0x57D0, 0x57B9, 0x57C1, 0x590E, 0x594A, 
    0x5A19, 0x5A16, 0x5A2D, 0x5A2E, 0x5A15, 0x5A0F, 0x5A17, 0x5A0A, 0x5A1E, 
    0x5A33, 0x5B6C, 0x5BA7, 0x5BAD, 0x5BAC, 0x5C03, 0x5C56, 0x5C54, 0x5CEC, 
    0x5CFF, 0x5CEE, 0x5CF1, 0x5CF7, 0x5D00, 0x5CF9, 0x5E29, 0x5E28, 0x5EA8, 
    0x5EAE, 0x5EAA, 0x5EAC, 0x5F33, 0x5F30, 0x5F67, 0x605D, 0x605A, 0x6067, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x6041, 0x60A2, 
    0x6088, 0x6080, 0x6092, 0x6081, 0x609D, 0x6083, 0x6095, 0x609B, 0x6097, 
    0x6087, 0x609C, 0x608E, 0x6219, 0x6246, 0x62F2, 0x6310, 0x6356, 0x632C, 
    0x6344, 0x6345, 0x6336, 0x6343, 0x63E4, 0x6339, 0x634B, 0x634A, 0x633C, 
    0x6329, 0x6341, 0x6334, 0x6358, 0x6354, 0x6359, 0x632D, 0x6347, 0x6333, 
    0x635A, 0x6351, 0x6338, 0x6357, 0x6340, 0x6348, 0x654A, 0x6546, 0x65C6, 
    0x65C3, 0x65C4, 0x65C2, 0x664A, 0x665F, 0x6647, 0x6651, 0x6712, 0x6713, 
    0x681F, 0x681A, 0x6849, 0x6832, 0x6833, 0x683B, 0x684B, 0x684F, 0x6816, 
    0x6831, 0x681C, 0x6835, 0x682B, 0x682D, 0x682F, 0x684E, 0x6844, 0x6834, 
    0x681D, 0x6812, 0x6814, 0x6826, 0x6828, 0x682E, 0x684D, 0x683A, 0x6825, 
    0x6820, 0x6B2C, 0x6B2F, 0x6B2D, 0x6B31, 0x6B34, 0x6B6D, 0x8082, 0x6B88, 
    0x6BE6, 0x6BE4},
    {0x6BE8, 0x6BE3, 0x6BE2, 0x6BE7, 0x6C25, 0x6D7A, 0x6D63, 0x6D64, 0x6D76, 
    0x6D0D, 0x6D61, 0x6D92, 0x6D58, 0x6D62, 0x6D6D, 0x6D6F, 0x6D91, 0x6D8D, 
    0x6DEF, 0x6D7F, 0x6D86, 0x6D5E, 0x6D67, 0x6D60, 0x6D97, 0x6D70, 0x6D7C, 
    0x6D5F, 0x6D82, 0x6D98, 0x6D2F, 0x6D68, 0x6D8B, 0x6D7E, 0x6D80, 0x6D84, 
    0x6D16, 0x6D83, 0x6D7B, 0x6D7D, 0x6D75, 0x6D90, 0x70DC, 0x70D3, 0x70D1, 
    0x70DD, 0x70CB, 0x7F39, 0x70E2, 0x70D7, 0x70D2, 0x70DE, 0x70E0, 0x70D4, 
    0x70CD, 0x70C5, 0x70C6, 0x70C7, 0x70DA, 0x70CE, 0x70E1, 0x7242, 0x7278, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x7277, 0x7276, 
    0x7300, 0x72FA, 0x72F4, 0x72FE, 0x72F6, 0x72F3, 0x72FB, 0x7301, 0x73D3, 
    0x73D9, 0x73E5, 0x73D6, 0x73BC, 0x73E7, 0x73E3, 0x73E9, 0x73DC, 0x73D2, 
    0x73DB, 0x73D4, 0x73DD, 0x73DA, 0x73D7, 0x73D8, 0x73E8, 0x74DE, 0x74DF, 
    0x74F4, 0x74F5, 0x7521, 0x755B, 0x755F, 0x75B0, 0x75C1, 0x75BB, 0x75C4, 
    0x75C0, 0x75BF, 0x75B6, 0x75BA, 0x768A, 0x76C9, 0x771D, 0x771B, 0x7710, 
    0x7713, 0x7712, 0x7723, 0x7711, 0x7715, 0x7719, 0x771A, 0x7722, 0x7727, 
    0x7823, 0x782C, 0x7822, 0x7835, 0x782F, 0x7828, 0x782E, 0x782B, 0x7821, 
    0x7829, 0x7833, 0x782A, 0x7831, 0x7954, 0x795B, 0x794F, 0x795C, 0x7953, 
    0x7952, 0x7951, 0x79EB, 0x79EC, 0x79E0, 0x79EE, 0x79ED, 0x79EA, 0x79DC, 
    0x79DE, 0x79DD, 0x7A86, 0x7A89, 0x7A85, 0x7A8B, 0x7A8C, 0x7A8A, 0x7A87, 
    0x7AD8, 0x7B10},
    {0x7B04, 0x7B13, 0x7B05, 0x7B0F, 0x7B08, 0x7B0A, 0x7B0E, 0x7B09, 0x7B12, 
    0x7C84, 0x7C91, 0x7C8A, 0x7C8C, 0x7C88, 0x7C8D, 0x7C85, 0x7D1E, 0x7D1D, 
    0x7D11, 0x7D0E, 0x7D18, 0x7D16, 0x7D13, 0x7D1F, 0x7D12, 0x7D0F, 0x7D0C, 
    0x7F5C, 0x7F61, 0x7F5E, 0x7F60, 0x7F5D, 0x7F5B, 0x7F96, 0x7F92, 0x7FC3, 
    0x7FC2, 0x7FC0, 0x8016, 0x803E, 0x8039, 0x80FA, 0x80F2, 0x80F9, 0x80F5, 
    0x8101, 0x80FB, 0x8100, 0x8201, 0x822F, 0x8225, 0x8333, 0x832D, 0x8344, 
    0x8319, 0x8351, 0x8325, 0x8356, 0x833F, 0x8341, 0x8326, 0x831C, 0x8322, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x8342, 0x834E, 
    0x831B, 0x832A, 0x8308, 0x833C, 0x834D, 0x8316, 0x8324, 0x8320, 0x8337, 
    0x832F, 0x8329, 0x8347, 0x8345, 0x834C, 0x8353, 0x831E, 0x832C, 0x834B, 
    0x8327, 0x8348, 0x8653, 0x8652, 0x86A2, 0x86A8, 0x8696, 0x868D, 0x8691, 
    0x869E, 0x8687, 0x8697, 0x8686, 0x868B, 0x869A, 0x8685, 0x86A5, 0x8699, 
    0x86A1, 0x86A7, 0x8695, 0x8698, 0x868E, 0x869D, 0x8690, 0x8694, 0x8843, 
    0x8844, 0x886D, 0x8875, 0x8876, 0x8872, 0x8880, 0x8871, 0x887F, 0x886F, 
    0x8883, 0x887E, 0x8874, 0x887C, 0x8A12, 0x8C47, 0x8C57, 0x8C7B, 0x8CA4, 
    0x8CA3, 0x8D76, 0x8D78, 0x8DB5, 0x8DB7, 0x8DB6, 0x8ED1, 0x8ED3, 0x8FFE, 
    0x8FF5, 0x9002, 0x8FFF, 0x8FFB, 0x9004, 0x8FFC, 0x8FF6, 0x90D6, 0x90E0, 
    0x90D9, 0x90DA, 0x90E3, 0x90DF, 0x90E5, 0x90D8, 0x90DB, 0x90D7, 0x90DC, 
    0x90E4, 0x9150},
    {0x914E, 0x914F, 0x91D5, 0x91E2, 0x91DA, 0x965C, 0x965F, 0x96BC, 0x98E3, 
    0x9ADF, 0x9B2F, 0x4E7F, 0x5070, 0x506A, 0x5061, 0x505E, 0x5060, 0x5053, 
    0x504B, 0x505D, 0x5072, 0x5048, 0x504D, 0x5041, 0x505B, 0x504A, 0x5062, 
    0x5015, 0x5045, 0x505F, 0x5069, 0x506B, 0x5063, 0x5064, 0x5046, 0x5040, 
    0x506E, 0x5073, 0x5057, 0x5051, 0x51D0, 0x526B, 0x526D, 0x526C, 0x526E, 
    0x52D6, 0x52D3, 0x532D, 0x539C, 0x5575, 0x5576, 0x553C, 0x554D, 0x5550, 
    0x5534, 0x552A, 0x5551, 0x5562, 0x5536, 0x5535, 0x5530, 0x5552, 0x5545, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x550C, 0x5532, 
    0x5565, 0x554E, 0x5539, 0x5548, 0x552D, 0x553B, 0x5540, 0x554B, 0x570A, 
    0x5707, 0x57FB, 0x5814, 0x57E2, 0x57F6, 0x57DC, 0x57F4, 0x5800, 0x57ED, 
    0x57FD, 0x5808, 0x57F8, 0x580B, 0x57F3, 0x57CF, 0x5807, 0x57EE, 0x57E3, 
    0x57F2, 0x57E5, 0x57EC, 0x57E1, 0x580E, 0x57FC, 0x5810, 0x57E7, 0x5801, 
    0x580C, 0x57F1, 0x57E9, 0x57F0, 0x580D, 0x5804, 0x595C, 0x5A60, 0x5A58, 
    0x5A55, 0x5A67, 0x5A5E, 0x5A38, 0x5A35, 0x5A6D, 0x5A50, 0x5A5F, 0x5A65, 
    0x5A6C, 0x5A53, 0x5A64, 0x5A57, 0x5A43, 0x5A5D, 0x5A52, 0x5A44, 0x5A5B, 
    0x5A48, 0x5A8E, 0x5A3E, 0x5A4D, 0x5A39, 0x5A4C, 0x5A70, 0x5A69, 0x5A47, 
    0x5A51, 0x5A56, 0x5A42, 0x5A5C, 0x5B72, 0x5B6E, 0x5BC1, 0x5BC0, 0x5C59, 
    0x5D1E, 0x5D0B, 0x5D1D, 0x5D1A, 0x5D20, 0x5D0C, 0x5D28, 0x5D0D, 0x5D26, 
    0x5D25, 0x5D0F},
    {0x5D30, 0x5D12, 0x5D23, 0x5D1F, 0x5D2E, 0x5E3E, 0x5E34, 0x5EB1, 0x5EB4, 
    0x5EB9, 0x5EB2, 0x5EB3, 0x5F36, 0x5F38, 0x5F9B, 0x5F96, 0x5F9F, 0x608A, 
    0x6090, 0x6086, 0x60BE, 0x60B0, 0x60BA, 0x60D3, 0x60D4, 0x60CF, 0x60E4, 
    0x60D9, 0x60DD, 0x60C8, 0x60B1, 0x60DB, 0x60B7, 0x60CA, 0x60BF, 0x60C3, 
    0x60CD, 0x60C0, 0x6332, 0x6365, 0x638A, 0x6382, 0x637D, 0x63BD, 0x639E, 
    0x63AD, 0x639D, 0x6397, 0x63AB, 0x638E, 0x636F, 0x6387, 0x6390, 0x636E, 
    0x63AF, 0x6375, 0x639C, 0x636D, 0x63AE, 0x637C, 0x63A4, 0x633B, 0x639F, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x6378, 0x6385, 
    0x6381, 0x6391, 0x638D, 0x6370, 0x6553, 0x65CD, 0x6665, 0x6661, 0x665B, 
    0x6659, 0x665C, 0x6662, 0x6718, 0x6879, 0x6887, 0x6890, 0x689C, 0x686D, 
    0x686E, 0x68AE, 0x68AB, 0x6956, 0x686F, 0x68A3, 0x68AC, 0x68A9, 0x6875, 
    0x6874, 0x68B2, 0x688F, 0x6877, 0x6892, 0x687C, 0x686B, 0x6872, 0x68AA, 
    0x6880, 0x6871, 0x687E, 0x689B, 0x6896, 0x688B, 0x68A0, 0x6889, 0x68A4, 
    0x6878, 0x687B, 0x6891, 0x688C, 0x688A, 0x687D, 0x6B36, 0x6B33, 0x6B37, 
    0x6B38, 0x6B91, 0x6B8F, 0x6B8D, 0x6B8E, 0x6B8C, 0x6C2A, 0x6DC0, 0x6DAB, 
    0x6DB4, 0x6DB3, 0x6E74, 0x6DAC, 0x6DE9, 0x6DE2, 0x6DB7, 0x6DF6, 0x6DD4, 
    0x6E00, 0x6DC8, 0x6DE0, 0x6DDF, 0x6DD6, 0x6DBE, 0x6DE5, 0x6DDC, 0x6DDD, 
    0x6DDB, 0x6DF4, 0x6DCA, 0x6DBD, 0x6DED, 0x6DF0, 0x6DBA, 0x6DD5, 0x6DC2, 
    0x6DCF, 0x6DC9},
    {0x6DD0, 0x6DF2, 0x6DD3, 0x6DFD, 0x6DD7, 0x6DCD, 0x6DE3, 0x6DBB, 0x70FA, 
    0x710D, 0x70F7, 0x7117, 0x70F4, 0x710C, 0x70F0, 0x7104, 0x70F3, 0x7110, 
    0x70FC, 0x70FF, 0x7106, 0x7113, 0x7100, 0x70F8, 0x70F6, 0x710B, 0x7102, 
    0x710E, 0x727E, 0x727B, 0x727C, 0x727F, 0x731D, 0x7317, 0x7307, 0x7311, 
    0x7318, 0x730A, 0x7308, 0x72FF, 0x730F, 0x731E, 0x7388, 0x73F6, 0x73F8, 
    0x73F5, 0x7404, 0x7401, 0x73FD, 0x7407, 0x7400, 0x73FA, 0x73FC, 0x73FF, 
    0x740C, 0x740B, 0x73F4, 0x7408, 0x7564, 0x7563, 0x75CE, 0x75D2, 0x75CF, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x75CB, 0x75CC, 
    0x75D1, 0x75D0, 0x768F, 0x7689, 0x76D3, 0x7739, 0x772F, 0x772D, 0x7731, 
    0x7732, 0x7734, 0x7733, 0x773D, 0x7725, 0x773B, 0x7735, 0x7848, 0x7852, 
    0x7849, 0x784D, 0x784A, 0x784C, 0x7826, 0x7845, 0x7850, 0x7964, 0x7967, 
    0x7969, 0x796A, 0x7963, 0x796B, 0x7961, 0x79BB, 0x79FA, 0x79F8, 0x79F6, 
    0x79F7, 0x7A8F, 0x7A94, 0x7A90, 0x7B35, 0x7B47, 0x7B34, 0x7B25, 0x7B30, 
    0x7B22, 0x7B24, 0x7B33, 0x7B18, 0x7B2A, 0x7B1D, 0x7B31, 0x7B2B, 0x7B2D, 
    0x7B2F, 0x7B32, 0x7B38, 0x7B1A, 0x7B23, 0x7C94, 0x7C98, 0x7C96, 0x7CA3, 
    0x7D35, 0x7D3D, 0x7D38, 0x7D36, 0x7D3A, 0x7D45, 0x7D2C, 0x7D29, 0x7D41, 
    0x7D47, 0x7D3E, 0x7D3F, 0x7D4A, 0x7D3B, 0x7D28, 0x7F63, 0x7F95, 0x7F9C, 
    0x7F9D, 0x7F9B, 0x7FCA, 0x7FCB, 0x7FCD, 0x7FD0, 0x7FD1, 0x7FC7, 0x7FCF, 
    0x7FC9, 0x801F},
    {0x801E, 0x801B, 0x8047, 0x8043, 0x8048, 0x8118, 0x8125, 0x8119, 0x811B, 
    0x812D, 0x811F, 0x812C, 0x811E, 0x8121, 0x8115, 0x8127, 0x811D, 0x8122, 
    0x8211, 0x8238, 0x8233, 0x823A, 0x8234, 0x8232, 0x8274, 0x8390, 0x83A3, 
    0x83A8, 0x838D, 0x837A, 0x8373, 0x83A4, 0x8374, 0x838F, 0x8381, 0x8395, 
    0x8399, 0x8375, 0x8394, 0x83A9, 0x837D, 0x8383, 0x838C, 0x839D, 0x839B, 
    0x83AA, 0x838B, 0x837E, 0x83A5, 0x83AF, 0x8388, 0x8397, 0x83B0, 0x837F, 
    0x83A6, 0x8387, 0x83AE, 0x8376, 0x839A, 0x8659, 0x8656, 0x86BF, 0x86B7, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x86C2, 0x86C1, 
    0x86C5, 0x86BA, 0x86B0, 0x86C8, 0x86B9, 0x86B3, 0x86B8, 0x86CC, 0x86B4, 
    0x86BB, 0x86BC, 0x86C3, 0x86BD, 0x86BE, 0x8852, 0x8889, 0x8895, 0x88A8, 
    0x88A2, 0x88AA, 0x889A, 0x8891, 0x88A1, 0x889F, 0x8898, 0x88A7, 0x8899, 
    0x889B, 0x8897, 0x88A4, 0x88AC, 0x888C, 0x8893, 0x888E, 0x8982, 0x89D6, 
    0x89D9, 0x89D5, 0x8A30, 0x8A27, 0x8A2C, 0x8A1E, 0x8C39, 0x8C3B, 0x8C5C, 
    0x8C5D, 0x8C7D, 0x8CA5, 0x8D7D, 0x8D7B, 0x8D79, 0x8DBC, 0x8DC2, 0x8DB9, 
    0x8DBF, 0x8DC1, 0x8ED8, 0x8EDE, 0x8EDD, 0x8EDC, 0x8ED7, 0x8EE0, 0x8EE1, 
    0x9024, 0x900B, 0x9011, 0x901C, 0x900C, 0x9021, 0x90EF, 0x90EA, 0x90F0, 
    0x90F4, 0x90F2, 0x90F3, 0x90D4, 0x90EB, 0x90EC, 0x90E9, 0x9156, 0x9158, 
    0x915A, 0x9153, 0x9155, 0x91EC, 0x91F4, 0x91F1, 0x91F3, 0x91F8, 0x91E4, 
    0x91F9, 0x91EA},
    {0x91EB, 0x91F7, 0x91E8, 0x91EE, 0x957A, 0x9586, 0x9588, 0x967C, 0x966D, 
    0x966B, 0x9671, 0x966F, 0x96BF, 0x976A, 0x9804, 0x98E5, 0x9997, 0x509B, 
    0x5095, 0x5094, 0x509E, 0x508B, 0x50A3, 0x5083, 0x508C, 0x508E, 0x509D, 
    0x5068, 0x509C, 0x5092, 0x5082, 0x5087, 0x515F, 0x51D4, 0x5312, 0x5311, 
    0x53A4, 0x53A7, 0x5591, 0x55A8, 0x55A5, 0x55AD, 0x5577, 0x5645, 0x55A2, 
    0x5593, 0x5588, 0x558F, 0x55B5, 0x5581, 0x55A3, 0x5592, 0x55A4, 0x557D, 
    0x558C, 0x55A6, 0x557F, 0x5595, 0x55A1, 0x558E, 0x570C, 0x5829, 0x5837, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x5819, 0x581E, 
    0x5827, 0x5823, 0x5828, 0x57F5, 0x5848, 0x5825, 0x581C, 0x581B, 0x5833, 
    0x583F, 0x5836, 0x582E, 0x5839, 0x5838, 0x582D, 0x582C, 0x583B, 0x5961, 
    0x5AAF, 0x5A94, 0x5A9F, 0x5A7A, 0x5AA2, 0x5A9E, 0x5A78, 0x5AA6, 0x5A7C, 
    0x5AA5, 0x5AAC, 0x5A95, 0x5AAE, 0x5A37, 0x5A84, 0x5A8A, 0x5A97, 0x5A83, 
    0x5A8B, 0x5AA9, 0x5A7B, 0x5A7D, 0x5A8C, 0x5A9C, 0x5A8F, 0x5A93, 0x5A9D, 
    0x5BEA, 0x5BCD, 0x5BCB, 0x5BD4, 0x5BD1, 0x5BCA, 0x5BCE, 0x5C0C, 0x5C30, 
    0x5D37, 0x5D43, 0x5D6B, 0x5D41, 0x5D4B, 0x5D3F, 0x5D35, 0x5D51, 0x5D4E, 
    0x5D55, 0x5D33, 0x5D3A, 0x5D52, 0x5D3D, 0x5D31, 0x5D59, 0x5D42, 0x5D39, 
    0x5D49, 0x5D38, 0x5D3C, 0x5D32, 0x5D36, 0x5D40, 0x5D45, 0x5E44, 0x5E41, 
    0x5F58, 0x5FA6, 0x5FA5, 0x5FAB, 0x60C9, 0x60B9, 0x60CC, 0x60E2, 0x60CE, 
    0x60C4, 0x6114},
    {0x60F2, 0x610A, 0x6116, 0x6105, 0x60F5, 0x6113, 0x60F8, 0x60FC, 0x60FE, 
    0x60C1, 0x6103, 0x6118, 0x611D, 0x6110, 0x60FF, 0x6104, 0x610B, 0x624A, 
    0x6394, 0x63B1, 0x63B0, 0x63CE, 0x63E5, 0x63E8, 0x63EF, 0x63C3, 0x649D, 
    0x63F3, 0x63CA, 0x63E0, 0x63F6, 0x63D5, 0x63F2, 0x63F5, 0x6461, 0x63DF, 
    0x63BE, 0x63DD, 0x63DC, 0x63C4, 0x63D8, 0x63D3, 0x63C2, 0x63C7, 0x63CC, 
    0x63CB, 0x63C8, 0x63F0, 0x63D7, 0x63D9, 0x6532, 0x6567, 0x656A, 0x6564, 
    0x655C, 0x6568, 0x6565, 0x658C, 0x659D, 0x659E, 0x65AE, 0x65D0, 0x65D2, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x667C, 0x666C, 
    0x667B, 0x6680, 0x6671, 0x6679, 0x666A, 0x6672, 0x6701, 0x690C, 0x68D3, 
    0x6904, 0x68DC, 0x692A, 0x68EC, 0x68EA, 0x68F1, 0x690F, 0x68D6, 0x68F7, 
    0x68EB, 0x68E4, 0x68F6, 0x6913, 0x6910, 0x68F3, 0x68E1, 0x6907, 0x68CC, 
    0x6908, 0x6970, 0x68B4, 0x6911, 0x68EF, 0x68C6, 0x6914, 0x68F8, 0x68D0, 
    0x68FD, 0x68FC, 0x68E8, 0x690B, 0x690A, 0x6917, 0x68CE, 0x68C8, 0x68DD, 
    0x68DE, 0x68E6, 0x68F4, 0x68D1, 0x6906, 0x68D4, 0x68E9, 0x6915, 0x6925, 
    0x68C7, 0x6B39, 0x6B3B, 0x6B3F, 0x6B3C, 0x6B94, 0x6B97, 0x6B99, 0x6B95, 
    0x6BBD, 0x6BF0, 0x6BF2, 0x6BF3, 0x6C30, 0x6DFC, 0x6E46, 0x6E47, 0x6E1F, 
    0x6E49, 0x6E88, 0x6E3C, 0x6E3D, 0x6E45, 0x6E62, 0x6E2B, 0x6E3F, 0x6E41, 
    0x6E5D, 0x6E73, 0x6E1C, 0x6E33, 0x6E4B, 0x6E40, 0x6E51, 0x6E3B, 0x6E03, 
    0x6E2E, 0x6E5E},
    {0x6E68, 0x6E5C, 0x6E61, 0x6E31, 0x6E28, 0x6E60, 0x6E71, 0x6E6B, 0x6E39, 
    0x6E22, 0x6E30, 0x6E53, 0x6E65, 0x6E27, 0x6E78, 0x6E64, 0x6E77, 0x6E55, 
    0x6E79, 0x6E52, 0x6E66, 0x6E35, 0x6E36, 0x6E5A, 0x7120, 0x711E, 0x712F, 
    0x70FB, 0x712E, 0x7131, 0x7123, 0x7125, 0x7122, 0x7132, 0x711F, 0x7128, 
    0x713A, 0x711B, 0x724B, 0x725A, 0x7288, 0x7289, 0x7286, 0x7285, 0x728B, 
    0x7312, 0x730B, 0x7330, 0x7322, 0x7331, 0x7333, 0x7327, 0x7332, 0x732D, 
    0x7326, 0x7323, 0x7335, 0x730C, 0x742E, 0x742C, 0x7430, 0x742B, 0x7416, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x741A, 0x7421, 
    0x742D, 0x7431, 0x7424, 0x7423, 0x741D, 0x7429, 0x7420, 0x7432, 0x74FB, 
    0x752F, 0x756F, 0x756C, 0x75E7, 0x75DA, 0x75E1, 0x75E6, 0x75DD, 0x75DF, 
    0x75E4, 0x75D7, 0x7695, 0x7692, 0x76DA, 0x7746, 0x7747, 0x7744, 0x774D, 
    0x7745, 0x774A, 0x774E, 0x774B, 0x774C, 0x77DE, 0x77EC, 0x7860, 0x7864, 
    0x7865, 0x785C, 0x786D, 0x7871, 0x786A, 0x786E, 0x7870, 0x7869, 0x7868, 
    0x785E, 0x7862, 0x7974, 0x7973, 0x7972, 0x7970, 0x7A02, 0x7A0A, 0x7A03, 
    0x7A0C, 0x7A04, 0x7A99, 0x7AE6, 0x7AE4, 0x7B4A, 0x7B3B, 0x7B44, 0x7B48, 
    0x7B4C, 0x7B4E, 0x7B40, 0x7B58, 0x7B45, 0x7CA2, 0x7C9E, 0x7CA8, 0x7CA1, 
    0x7D58, 0x7D6F, 0x7D63, 0x7D53, 0x7D56, 0x7D67, 0x7D6A, 0x7D4F, 0x7D6D, 
    0x7D5C, 0x7D6B, 0x7D52, 0x7D54, 0x7D69, 0x7D51, 0x7D5F, 0x7D4E, 0x7F3E, 
    0x7F3F, 0x7F65},
    {0x7F66, 0x7FA2, 0x7FA0, 0x7FA1, 0x7FD7, 0x8051, 0x804F, 0x8050, 0x80FE, 
    0x80D4, 0x8143, 0x814A, 0x8152, 0x814F, 0x8147, 0x813D, 0x814D, 0x813A, 
    0x81E6, 0x81EE, 0x81F7, 0x81F8, 0x81F9, 0x8204, 0x823C, 0x823D, 0x823F, 
    0x8275, 0x833B, 0x83CF, 0x83F9, 0x8423, 0x83C0, 0x83E8, 0x8412, 0x83E7, 
    0x83E4, 0x83FC, 0x83F6, 0x8410, 0x83C6, 0x83C8, 0x83EB, 0x83E3, 0x83BF, 
    0x8401, 0x83DD, 0x83E5, 0x83D8, 0x83FF, 0x83E1, 0x83CB, 0x83CE, 0x83D6, 
    0x83F5, 0x83C9, 0x8409, 0x840F, 0x83DE, 0x8411, 0x8406, 0x83C2, 0x83F3, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x83D5, 0x83FA, 
    0x83C7, 0x83D1, 0x83EA, 0x8413, 0x83C3, 0x83EC, 0x83EE, 0x83C4, 0x83FB, 
    0x83D7, 0x83E2, 0x841B, 0x83DB, 0x83FE, 0x86D8, 0x86E2, 0x86E6, 0x86D3, 
    0x86E3, 0x86DA, 0x86EA, 0x86DD, 0x86EB, 0x86DC, 0x86EC, 0x86E9, 0x86D7, 
    0x86E8, 0x86D1, 0x8848, 0x8856, 0x8855, 0x88BA, 0x88D7, 0x88B9, 0x88B8, 
    0x88C0, 0x88BE, 0x88B6, 0x88BC, 0x88B7, 0x88BD, 0x88B2, 0x8901, 0x88C9, 
    0x8995, 0x8998, 0x8997, 0x89DD, 0x89DA, 0x89DB, 0x8A4E, 0x8A4D, 0x8A39, 
    0x8A59, 0x8A40, 0x8A57, 0x8A58, 0x8A44, 0x8A45, 0x8A52, 0x8A48, 0x8A51, 
    0x8A4A, 0x8A4C, 0x8A4F, 0x8C5F, 0x8C81, 0x8C80, 0x8CBA, 0x8CBE, 0x8CB0, 
    0x8CB9, 0x8CB5, 0x8D84, 0x8D80, 0x8D89, 0x8DD8, 0x8DD3, 0x8DCD, 0x8DC7, 
    0x8DD6, 0x8DDC, 0x8DCF, 0x8DD5, 0x8DD9, 0x8DC8, 0x8DD7, 0x8DC5, 0x8EEF, 
    0x8EF7, 0x8EFA},
    {0x8EF9, 0x8EE6, 0x8EEE, 0x8EE5, 0x8EF5, 0x8EE7, 0x8EE8, 0x8EF6, 0x8EEB, 
    0x8EF1, 0x8EEC, 0x8EF4, 0x8EE9, 0x902D, 0x9034, 0x902F, 0x9106, 0x912C, 
    0x9104, 0x90FF, 0x90FC, 0x9108, 0x90F9, 0x90FB, 0x9101, 0x9100, 0x9107, 
    0x9105, 0x9103, 0x9161, 0x9164, 0x915F, 0x9162, 0x9160, 0x9201, 0x920A, 
    0x9225, 0x9203, 0x921A, 0x9226, 0x920F, 0x920C, 0x9200, 0x9212, 0x91FF, 
    0x91FD, 0x9206, 0x9204, 0x9227, 0x9202, 0x921C, 0x9224, 0x9219, 0x9217, 
    0x9205, 0x9216, 0x957B, 0x958D, 0x958C, 0x9590, 0x9687, 0x967E, 0x9688, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x9689, 0x9683, 
    0x9680, 0x96C2, 0x96C8, 0x96C3, 0x96F1, 0x96F0, 0x976C, 0x9770, 0x976E, 
    0x9807, 0x98A9, 0x98EB, 0x9CE6, 0x9EF9, 0x4E83, 0x4E84, 0x4EB6, 0x50BD, 
    0x50BF, 0x50C6, 0x50AE, 0x50C4, 0x50CA, 0x50B4, 0x50C8, 0x50C2, 0x50B0, 
    0x50C1, 0x50BA, 0x50B1, 0x50CB, 0x50C9, 0x50B6, 0x50B8, 0x51D7, 0x527A, 
    0x5278, 0x527B, 0x527C, 0x55C3, 0x55DB, 0x55CC, 0x55D0, 0x55CB, 0x55CA, 
    0x55DD, 0x55C0, 0x55D4, 0x55C4, 0x55E9, 0x55BF, 0x55D2, 0x558D, 0x55CF, 
    0x55D5, 0x55E2, 0x55D6, 0x55C8, 0x55F2, 0x55CD, 0x55D9, 0x55C2, 0x5714, 
    0x5853, 0x5868, 0x5864, 0x584F, 0x584D, 0x5849, 0x586F, 0x5855, 0x584E, 
    0x585D, 0x5859, 0x5865, 0x585B, 0x583D, 0x5863, 0x5871, 0x58FC, 0x5AC7, 
    0x5AC4, 0x5ACB, 0x5ABA, 0x5AB8, 0x5AB1, 0x5AB5, 0x5AB0, 0x5ABF, 0x5AC8, 
    0x5ABB, 0x5AC6},
    {0x5AB7, 0x5AC0, 0x5ACA, 0x5AB4, 0x5AB6, 0x5ACD, 0x5AB9, 0x5A90, 0x5BD6, 
    0x5BD8, 0x5BD9, 0x5C1F, 0x5C33, 0x5D71, 0x5D63, 0x5D4A, 0x5D65, 0x5D72, 
    0x5D6C, 0x5D5E, 0x5D68, 0x5D67, 0x5D62, 0x5DF0, 0x5E4F, 0x5E4E, 0x5E4A, 
    0x5E4D, 0x5E4B, 0x5EC5, 0x5ECC, 0x5EC6, 0x5ECB, 0x5EC7, 0x5F40, 0x5FAF, 
    0x5FAD, 0x60F7, 0x6149, 0x614A, 0x612B, 0x6145, 0x6136, 0x6132, 0x612E, 
    0x6146, 0x612F, 0x614F, 0x6129, 0x6140, 0x6220, 0x9168, 0x6223, 0x6225, 
    0x6224, 0x63C5, 0x63F1, 0x63EB, 0x6410, 0x6412, 0x6409, 0x6420, 0x6424, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x6433, 0x6443, 
    0x641F, 0x6415, 0x6418, 0x6439, 0x6437, 0x6422, 0x6423, 0x640C, 0x6426, 
    0x6430, 0x6428, 0x6441, 0x6435, 0x642F, 0x640A, 0x641A, 0x6440, 0x6425, 
    0x6427, 0x640B, 0x63E7, 0x641B, 0x642E, 0x6421, 0x640E, 0x656F, 0x6592, 
    0x65D3, 0x6686, 0x668C, 0x6695, 0x6690, 0x668B, 0x668A, 0x6699, 0x6694, 
    0x6678, 0x6720, 0x6966, 0x695F, 0x6938, 0x694E, 0x6962, 0x6971, 0x693F, 
    0x6945, 0x696A, 0x6939, 0x6942, 0x6957, 0x6959, 0x697A, 0x6948, 0x6949, 
    0x6935, 0x696C, 0x6933, 0x693D, 0x6965, 0x68F0, 0x6978, 0x6934, 0x6969, 
    0x6940, 0x696F, 0x6944, 0x6976, 0x6958, 0x6941, 0x6974, 0x694C, 0x693B, 
    0x694B, 0x6937, 0x695C, 0x694F, 0x6951, 0x6932, 0x6952, 0x692F, 0x697B, 
    0x693C, 0x6B46, 0x6B45, 0x6B43, 0x6B42, 0x6B48, 0x6B41, 0x6B9B, 0xFA0D, 
    0x6BFB, 0x6BFC},
    {0x6BF9, 0x6BF7, 0x6BF8, 0x6E9B, 0x6ED6, 0x6EC8, 0x6E8F, 0x6EC0, 0x6E9F, 
    0x6E93, 0x6E94, 0x6EA0, 0x6EB1, 0x6EB9, 0x6EC6, 0x6ED2, 0x6EBD, 0x6EC1, 
    0x6E9E, 0x6EC9, 0x6EB7, 0x6EB0, 0x6ECD, 0x6EA6, 0x6ECF, 0x6EB2, 0x6EBE, 
    0x6EC3, 0x6EDC, 0x6ED8, 0x6E99, 0x6E92, 0x6E8E, 0x6E8D, 0x6EA4, 0x6EA1, 
    0x6EBF, 0x6EB3, 0x6ED0, 0x6ECA, 0x6E97, 0x6EAE, 0x6EA3, 0x7147, 0x7154, 
    0x7152, 0x7163, 0x7160, 0x7141, 0x715D, 0x7162, 0x7172, 0x7178, 0x716A, 
    0x7161, 0x7142, 0x7158, 0x7143, 0x714B, 0x7170, 0x715F, 0x7150, 0x7153, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x7144, 0x714D, 
    0x715A, 0x724F, 0x728D, 0x728C, 0x7291, 0x7290, 0x728E, 0x733C, 0x7342, 
    0x733B, 0x733A, 0x7340, 0x734A, 0x7349, 0x7444, 0x744A, 0x744B, 0x7452, 
    0x7451, 0x7457, 0x7440, 0x744F, 0x7450, 0x744E, 0x7442, 0x7446, 0x744D, 
    0x7454, 0x74E1, 0x74FF, 0x74FE, 0x74FD, 0x751D, 0x7579, 0x7577, 0x6983, 
    0x75EF, 0x760F, 0x7603, 0x75F7, 0x75FE, 0x75FC, 0x75F9, 0x75F8, 0x7610, 
    0x75FB, 0x75F6, 0x75ED, 0x75F5, 0x75FD, 0x7699, 0x76B5, 0x76DD, 0x7755, 
    0x775F, 0x7760, 0x7752, 0x7756, 0x775A, 0x7769, 0x7767, 0x7754, 0x7759, 
    0x776D, 0x77E0, 0x7887, 0x789A, 0x7894, 0x788F, 0x7884, 0x7895, 0x7885, 
    0x7886, 0x78A1, 0x7883, 0x7879, 0x7899, 0x7880, 0x7896, 0x787B, 0x797C, 
    0x7982, 0x797D, 0x7979, 0x7A11, 0x7A18, 0x7A19, 0x7A12, 0x7A17, 0x7A15, 
    0x7A22, 0x7A13},
    {0x7A1B, 0x7A10, 0x7AA3, 0x7AA2, 0x7A9E, 0x7AEB, 0x7B66, 0x7B64, 0x7B6D, 
    0x7B74, 0x7B69, 0x7B72, 0x7B65, 0x7B73, 0x7B71, 0x7B70, 0x7B61, 0x7B78, 
    0x7B76, 0x7B63, 0x7CB2, 0x7CB4, 0x7CAF, 0x7D88, 0x7D86, 0x7D80, 0x7D8D, 
    0x7D7F, 0x7D85, 0x7D7A, 0x7D8E, 0x7D7B, 0x7D83, 0x7D7C, 0x7D8C, 0x7D94, 
    0x7D84, 0x7D7D, 0x7D92, 0x7F6D, 0x7F6B, 0x7F67, 0x7F68, 0x7F6C, 0x7FA6, 
    0x7FA5, 0x7FA7, 0x7FDB, 0x7FDC, 0x8021, 0x8164, 0x8160, 0x8177, 0x815C, 
    0x8169, 0x815B, 0x8162, 0x8172, 0x6721, 0x815E, 0x8176, 0x8167, 0x816F, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x8144, 0x8161, 
    0x821D, 0x8249, 0x8244, 0x8240, 0x8242, 0x8245, 0x84F1, 0x843F, 0x8456, 
    0x8476, 0x8479, 0x848F, 0x848D, 0x8465, 0x8451, 0x8440, 0x8486, 0x8467, 
    0x8430, 0x844D, 0x847D, 0x845A, 0x8459, 0x8474, 0x8473, 0x845D, 0x8507, 
    0x845E, 0x8437, 0x843A, 0x8434, 0x847A, 0x8443, 0x8478, 0x8432, 0x8445, 
    0x8429, 0x83D9, 0x844B, 0x842F, 0x8442, 0x842D, 0x845F, 0x8470, 0x8439, 
    0x844E, 0x844C, 0x8452, 0x846F, 0x84C5, 0x848E, 0x843B, 0x8447, 0x8436, 
    0x8433, 0x8468, 0x847E, 0x8444, 0x842B, 0x8460, 0x8454, 0x846E, 0x8450, 
    0x870B, 0x8704, 0x86F7, 0x870C, 0x86FA, 0x86D6, 0x86F5, 0x874D, 0x86F8, 
    0x870E, 0x8709, 0x8701, 0x86F6, 0x870D, 0x8705, 0x88D6, 0x88CB, 0x88CD, 
    0x88CE, 0x88DE, 0x88DB, 0x88DA, 0x88CC, 0x88D0, 0x8985, 0x899B, 0x89DF, 
    0x89E5, 0x89E4},
    {0x89E1, 0x89E0, 0x89E2, 0x89DC, 0x89E6, 0x8A76, 0x8A86, 0x8A7F, 0x8A61, 
    0x8A3F, 0x8A77, 0x8A82, 0x8A84, 0x8A75, 0x8A83, 0x8A81, 0x8A74, 0x8A7A, 
    0x8C3C, 0x8C4B, 0x8C4A, 0x8C65, 0x8C64, 0x8C66, 0x8C86, 0x8C84, 0x8C85, 
    0x8CCC, 0x8D68, 0x8D69, 0x8D91, 0x8D8C, 0x8D8E, 0x8D8F, 0x8D8D, 0x8D93, 
    0x8D94, 0x8D90, 0x8D92, 0x8DF0, 0x8DE0, 0x8DEC, 0x8DF1, 0x8DEE, 0x8DD0, 
    0x8DE9, 0x8DE3, 0x8DE2, 0x8DE7, 0x8DF2, 0x8DEB, 0x8DF4, 0x8F06, 0x8EFF, 
    0x8F01, 0x8F00, 0x8F05, 0x8F07, 0x8F08, 0x8F02, 0x8F0B, 0x9052, 0x903F, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x9044, 0x9049, 
    0x903D, 0x9110, 0x910D, 0x910F, 0x9111, 0x9116, 0x9114, 0x910B, 0x910E, 
    0x916E, 0x916F, 0x9248, 0x9252, 0x9230, 0x923A, 0x9266, 0x9233, 0x9265, 
    0x925E, 0x9283, 0x922E, 0x924A, 0x9246, 0x926D, 0x926C, 0x924F, 0x9260, 
    0x9267, 0x926F, 0x9236, 0x9261, 0x9270, 0x9231, 0x9254, 0x9263, 0x9250, 
    0x9272, 0x924E, 0x9253, 0x924C, 0x9256, 0x9232, 0x959F, 0x959C, 0x959E, 
    0x959B, 0x9692, 0x9693, 0x9691, 0x9697, 0x96CE, 0x96FA, 0x96FD, 0x96F8, 
    0x96F5, 0x9773, 0x9777, 0x9778, 0x9772, 0x980F, 0x980D, 0x980E, 0x98AC, 
    0x98F6, 0x98F9, 0x99AF, 0x99B2, 0x99B0, 0x99B5, 0x9AAD, 0x9AAB, 0x9B5B, 
    0x9CEA, 0x9CED, 0x9CE7, 0x9E80, 0x9EFD, 0x50E6, 0x50D4, 0x50D7, 0x50E8, 
    0x50F3, 0x50DB, 0x50EA, 0x50DD, 0x50E4, 0x50D3, 0x50EC, 0x50F0, 0x50EF, 
    0x50E3, 0x50E0},
    {0x51D8, 0x5280, 0x5281, 0x52E9, 0x52EB, 0x5330, 0x53AC, 0x5627, 0x5615, 
    0x560C, 0x5612, 0x55FC, 0x560F, 0x561C, 0x5601, 0x5613, 0x5602, 0x55FA, 
    0x561D, 0x5604, 0x55FF, 0x55F9, 0x5889, 0x587C, 0x5890, 0x5898, 0x5886, 
    0x5881, 0x587F, 0x5874, 0x588B, 0x587A, 0x5887, 0x5891, 0x588E, 0x5876, 
    0x5882, 0x5888, 0x587B, 0x5894, 0x588F, 0x58FE, 0x596B, 0x5ADC, 0x5AEE, 
    0x5AE5, 0x5AD5, 0x5AEA, 0x5ADA, 0x5AED, 0x5AEB, 0x5AF3, 0x5AE2, 0x5AE0, 
    0x5ADB, 0x5AEC, 0x5ADE, 0x5ADD, 0x5AD9, 0x5AE8, 0x5ADF, 0x5B77, 0x5BE0, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x5BE3, 0x5C63, 
    0x5D82, 0x5D80, 0x5D7D, 0x5D86, 0x5D7A, 0x5D81, 0x5D77, 0x5D8A, 0x5D89, 
    0x5D88, 0x5D7E, 0x5D7C, 0x5D8D, 0x5D79, 0x5D7F, 0x5E58, 0x5E59, 0x5E53, 
    0x5ED8, 0x5ED1, 0x5ED7, 0x5ECE, 0x5EDC, 0x5ED5, 0x5ED9, 0x5ED2, 0x5ED4, 
    0x5F44, 0x5F43, 0x5F6F, 0x5FB6, 0x612C, 0x6128, 0x6141, 0x615E, 0x6171, 
    0x6173, 0x6152, 0x6153, 0x6172, 0x616C, 0x6180, 0x6174, 0x6154, 0x617A, 
    0x615B, 0x6165, 0x613B, 0x616A, 0x6161, 0x6156, 0x6229, 0x6227, 0x622B, 
    0x642B, 0x644D, 0x645B, 0x645D, 0x6474, 0x6476, 0x6472, 0x6473, 0x647D, 
    0x6475, 0x6466, 0x64A6, 0x644E, 0x6482, 0x645E, 0x645C, 0x644B, 0x6453, 
    0x6460, 0x6450, 0x647F, 0x643F, 0x646C, 0x646B, 0x6459, 0x6465, 0x6477, 
    0x6573, 0x65A0, 0x66A1, 0x66A0, 0x669F, 0x6705, 0x6704, 0x6722, 0x69B1, 
    0x69B6, 0x69C9},
    {0x69A0, 0x69CE, 0x6996, 0x69B0, 0x69AC, 0x69BC, 0x6991, 0x6999, 0x698E, 
    0x69A7, 0x698D, 0x69A9, 0x69BE, 0x69AF, 0x69BF, 0x69C4, 0x69BD, 0x69A4, 
    0x69D4, 0x69B9, 0x69CA, 0x699A, 0x69CF, 0x69B3, 0x6993, 0x69AA, 0x69A1, 
    0x699E, 0x69D9, 0x6997, 0x6990, 0x69C2, 0x69B5, 0x69A5, 0x69C6, 0x6B4A, 
    0x6B4D, 0x6B4B, 0x6B9E, 0x6B9F, 0x6BA0, 0x6BC3, 0x6BC4, 0x6BFE, 0x6ECE, 
    0x6EF5, 0x6EF1, 0x6F03, 0x6F25, 0x6EF8, 0x6F37, 0x6EFB, 0x6F2E, 0x6F09, 
    0x6F4E, 0x6F19, 0x6F1A, 0x6F27, 0x6F18, 0x6F3B, 0x6F12, 0x6EED, 0x6F0A, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x6F36, 0x6F73, 
    0x6EF9, 0x6EEE, 0x6F2D, 0x6F40, 0x6F30, 0x6F3C, 0x6F35, 0x6EEB, 0x6F07, 
    0x6F0E, 0x6F43, 0x6F05, 0x6EFD, 0x6EF6, 0x6F39, 0x6F1C, 0x6EFC, 0x6F3A, 
    0x6F1F, 0x6F0D, 0x6F1E, 0x6F08, 0x6F21, 0x7187, 0x7190, 0x7189, 0x7180, 
    0x7185, 0x7182, 0x718F, 0x717B, 0x7186, 0x7181, 0x7197, 0x7244, 0x7253, 
    0x7297, 0x7295, 0x7293, 0x7343, 0x734D, 0x7351, 0x734C, 0x7462, 0x7473, 
    0x7471, 0x7475, 0x7472, 0x7467, 0x746E, 0x7500, 0x7502, 0x7503, 0x757D, 
    0x7590, 0x7616, 0x7608, 0x760C, 0x7615, 0x7611, 0x760A, 0x7614, 0x76B8, 
    0x7781, 0x777C, 0x7785, 0x7782, 0x776E, 0x7780, 0x776F, 0x777E, 0x7783, 
    0x78B2, 0x78AA, 0x78B4, 0x78AD, 0x78A8, 0x787E, 0x78AB, 0x789E, 0x78A5, 
    0x78A0, 0x78AC, 0x78A2, 0x78A4, 0x7998, 0x798A, 0x798B, 0x7996, 0x7995, 
    0x7994, 0x7993},
    {0x7997, 0x7988, 0x7992, 0x7990, 0x7A2B, 0x7A4A, 0x7A30, 0x7A2F, 0x7A28, 
    0x7A26, 0x7AA8, 0x7AAB, 0x7AAC, 0x7AEE, 0x7B88, 0x7B9C, 0x7B8A, 0x7B91, 
    0x7B90, 0x7B96, 0x7B8D, 0x7B8C, 0x7B9B, 0x7B8E, 0x7B85, 0x7B98, 0x5284, 
    0x7B99, 0x7BA4, 0x7B82, 0x7CBB, 0x7CBF, 0x7CBC, 0x7CBA, 0x7DA7, 0x7DB7, 
    0x7DC2, 0x7DA3, 0x7DAA, 0x7DC1, 0x7DC0, 0x7DC5, 0x7D9D, 0x7DCE, 0x7DC4, 
    0x7DC6, 0x7DCB, 0x7DCC, 0x7DAF, 0x7DB9, 0x7D96, 0x7DBC, 0x7D9F, 0x7DA6, 
    0x7DAE, 0x7DA9, 0x7DA1, 0x7DC9, 0x7F73, 0x7FE2, 0x7FE3, 0x7FE5, 0x7FDE, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x8024, 0x805D, 
    0x805C, 0x8189, 0x8186, 0x8183, 0x8187, 0x818D, 0x818C, 0x818B, 0x8215, 
    0x8497, 0x84A4, 0x84A1, 0x849F, 0x84BA, 0x84CE, 0x84C2, 0x84AC, 0x84AE, 
    0x84AB, 0x84B9, 0x84B4, 0x84C1, 0x84CD, 0x84AA, 0x849A, 0x84B1, 0x84D0, 
    0x849D, 0x84A7, 0x84BB, 0x84A2, 0x8494, 0x84C7, 0x84CC, 0x849B, 0x84A9, 
    0x84AF, 0x84A8, 0x84D6, 0x8498, 0x84B6, 0x84CF, 0x84A0, 0x84D7, 0x84D4, 
    0x84D2, 0x84DB, 0x84B0, 0x8491, 0x8661, 0x8733, 0x8723, 0x8728, 0x876B, 
    0x8740, 0x872E, 0x871E, 0x8721, 0x8719, 0x871B, 0x8743, 0x872C, 0x8741, 
    0x873E, 0x8746, 0x8720, 0x8732, 0x872A, 0x872D, 0x873C, 0x8712, 0x873A, 
    0x8731, 0x8735, 0x8742, 0x8726, 0x8727, 0x8738, 0x8724, 0x871A, 0x8730, 
    0x8711, 0x88F7, 0x88E7, 0x88F1, 0x88F2, 0x88FA, 0x88FE, 0x88EE, 0x88FC, 
    0x88F6, 0x88FB},
    {0x88F0, 0x88EC, 0x88EB, 0x899D, 0x89A1, 0x899F, 0x899E, 0x89E9, 0x89EB, 
    0x89E8, 0x8AAB, 0x8A99, 0x8A8B, 0x8A92, 0x8A8F, 0x8A96, 0x8C3D, 0x8C68, 
    0x8C69, 0x8CD5, 0x8CCF, 0x8CD7, 0x8D96, 0x8E09, 0x8E02, 0x8DFF, 0x8E0D, 
    0x8DFD, 0x8E0A, 0x8E03, 0x8E07, 0x8E06, 0x8E05, 0x8DFE, 0x8E00, 0x8E04, 
    0x8F10, 0x8F11, 0x8F0E, 0x8F0D, 0x9123, 0x911C, 0x9120, 0x9122, 0x911F, 
    0x911D, 0x911A, 0x9124, 0x9121, 0x911B, 0x917A, 0x9172, 0x9179, 0x9173, 
    0x92A5, 0x92A4, 0x9276, 0x929B, 0x927A, 0x92A0, 0x9294, 0x92AA, 0x928D, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x92A6, 0x929A, 
    0x92AB, 0x9279, 0x9297, 0x927F, 0x92A3, 0x92EE, 0x928E, 0x9282, 0x9295, 
    0x92A2, 0x927D, 0x9288, 0x92A1, 0x928A, 0x9286, 0x928C, 0x9299, 0x92A7, 
    0x927E, 0x9287, 0x92A9, 0x929D, 0x928B, 0x922D, 0x969E, 0x96A1, 0x96FF, 
    0x9758, 0x977D, 0x977A, 0x977E, 0x9783, 0x9780, 0x9782, 0x977B, 0x9784, 
    0x9781, 0x977F, 0x97CE, 0x97CD, 0x9816, 0x98AD, 0x98AE, 0x9902, 0x9900, 
    0x9907, 0x999D, 0x999C, 0x99C3, 0x99B9, 0x99BB, 0x99BA, 0x99C2, 0x99BD, 
    0x99C7, 0x9AB1, 0x9AE3, 0x9AE7, 0x9B3E, 0x9B3F, 0x9B60, 0x9B61, 0x9B5F, 
    0x9CF1, 0x9CF2, 0x9CF5, 0x9EA7, 0x50FF, 0x5103, 0x5130, 0x50F8, 0x5106, 
    0x5107, 0x50F6, 0x50FE, 0x510B, 0x510C, 0x50FD, 0x510A, 0x528B, 0x528C, 
    0x52F1, 0x52EF, 0x5648, 0x5642, 0x564C, 0x5635, 0x5641, 0x564A, 0x5649, 
    0x5646, 0x5658},
    {0x565A, 0x5640, 0x5633, 0x563D, 0x562C, 0x563E, 0x5638, 0x562A, 0x563A, 
    0x571A, 0x58AB, 0x589D, 0x58B1, 0x58A0, 0x58A3, 0x58AF, 0x58AC, 0x58A5, 
    0x58A1, 0x58FF, 0x5AFF, 0x5AF4, 0x5AFD, 0x5AF7, 0x5AF6, 0x5B03, 0x5AF8, 
    0x5B02, 0x5AF9, 0x5B01, 0x5B07, 0x5B05, 0x5B0F, 0x5C67, 0x5D99, 0x5D97, 
    0x5D9F, 0x5D92, 0x5DA2, 0x5D93, 0x5D95, 0x5DA0, 0x5D9C, 0x5DA1, 0x5D9A, 
    0x5D9E, 0x5E69, 0x5E5D, 0x5E60, 0x5E5C, 0x7DF3, 0x5EDB, 0x5EDE, 0x5EE1, 
    0x5F49, 0x5FB2, 0x618B, 0x6183, 0x6179, 0x61B1, 0x61B0, 0x61A2, 0x6189, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x619B, 0x6193, 
    0x61AF, 0x61AD, 0x619F, 0x6192, 0x61AA, 0x61A1, 0x618D, 0x6166, 0x61B3, 
    0x622D, 0x646E, 0x6470, 0x6496, 0x64A0, 0x6485, 0x6497, 0x649C, 0x648F, 
    0x648B, 0x648A, 0x648C, 0x64A3, 0x649F, 0x6468, 0x64B1, 0x6498, 0x6576, 
    0x657A, 0x6579, 0x657B, 0x65B2, 0x65B3, 0x66B5, 0x66B0, 0x66A9, 0x66B2, 
    0x66B7, 0x66AA, 0x66AF, 0x6A00, 0x6A06, 0x6A17, 0x69E5, 0x69F8, 0x6A15, 
    0x69F1, 0x69E4, 0x6A20, 0x69FF, 0x69EC, 0x69E2, 0x6A1B, 0x6A1D, 0x69FE, 
    0x6A27, 0x69F2, 0x69EE, 0x6A14, 0x69F7, 0x69E7, 0x6A40, 0x6A08, 0x69E6, 
    0x69FB, 0x6A0D, 0x69FC, 0x69EB, 0x6A09, 0x6A04, 0x6A18, 0x6A25, 0x6A0F, 
    0x69F6, 0x6A26, 0x6A07, 0x69F4, 0x6A16, 0x6B51, 0x6BA5, 0x6BA3, 0x6BA2, 
    0x6BA6, 0x6C01, 0x6C00, 0x6BFF, 0x6C02, 0x6F41, 0x6F26, 0x6F7E, 0x6F87, 
    0x6FC6, 0x6F92},
    {0x6F8D, 0x6F89, 0x6F8C, 0x6F62, 0x6F4F, 0x6F85, 0x6F5A, 0x6F96, 0x6F76, 
    0x6F6C, 0x6F82, 0x6F55, 0x6F72, 0x6F52, 0x6F50, 0x6F57, 0x6F94, 0x6F93, 
    0x6F5D, 0x6F00, 0x6F61, 0x6F6B, 0x6F7D, 0x6F67, 0x6F90, 0x6F53, 0x6F8B, 
    0x6F69, 0x6F7F, 0x6F95, 0x6F63, 0x6F77, 0x6F6A, 0x6F7B, 0x71B2, 0x71AF, 
    0x719B, 0x71B0, 0x71A0, 0x719A, 0x71A9, 0x71B5, 0x719D, 0x71A5, 0x719E, 
    0x71A4, 0x71A1, 0x71AA, 0x719C, 0x71A7, 0x71B3, 0x7298, 0x729A, 0x7358, 
    0x7352, 0x735E, 0x735F, 0x7360, 0x735D, 0x735B, 0x7361, 0x735A, 0x7359, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x7362, 0x7487, 
    0x7489, 0x748A, 0x7486, 0x7481, 0x747D, 0x7485, 0x7488, 0x747C, 0x7479, 
    0x7508, 0x7507, 0x757E, 0x7625, 0x761E, 0x7619, 0x761D, 0x761C, 0x7623, 
    0x761A, 0x7628, 0x761B, 0x769C, 0x769D, 0x769E, 0x769B, 0x778D, 0x778F, 
    0x7789, 0x7788, 0x78CD, 0x78BB, 0x78CF, 0x78CC, 0x78D1, 0x78CE, 0x78D4, 
    0x78C8, 0x78C3, 0x78C4, 0x78C9, 0x799A, 0x79A1, 0x79A0, 0x799C, 0x79A2, 
    0x799B, 0x6B76, 0x7A39, 0x7AB2, 0x7AB4, 0x7AB3, 0x7BB7, 0x7BCB, 0x7BBE, 
    0x7BAC, 0x7BCE, 0x7BAF, 0x7BB9, 0x7BCA, 0x7BB5, 0x7CC5, 0x7CC8, 0x7CCC, 
    0x7CCB, 0x7DF7, 0x7DDB, 0x7DEA, 0x7DE7, 0x7DD7, 0x7DE1, 0x7E03, 0x7DFA, 
    0x7DE6, 0x7DF6, 0x7DF1, 0x7DF0, 0x7DEE, 0x7DDF, 0x7F76, 0x7FAC, 0x7FB0, 
    0x7FAD, 0x7FED, 0x7FEB, 0x7FEA, 0x7FEC, 0x7FE6, 0x7FE8, 0x8064, 0x8067, 
    0x81A3, 0x819F},
    {0x819E, 0x8195, 0x81A2, 0x8199, 0x8197, 0x8216, 0x824F, 0x8253, 0x8252, 
    0x8250, 0x824E, 0x8251, 0x8524, 0x853B, 0x850F, 0x8500, 0x8529, 0x850E, 
    0x8509, 0x850D, 0x851F, 0x850A, 0x8527, 0x851C, 0x84FB, 0x852B, 0x84FA, 
    0x8508, 0x850C, 0x84F4, 0x852A, 0x84F2, 0x8515, 0x84F7, 0x84EB, 0x84F3, 
    0x84FC, 0x8512, 0x84EA, 0x84E9, 0x8516, 0x84FE, 0x8528, 0x851D, 0x852E, 
    0x8502, 0x84FD, 0x851E, 0x84F6, 0x8531, 0x8526, 0x84E7, 0x84E8, 0x84F0, 
    0x84EF, 0x84F9, 0x8518, 0x8520, 0x8530, 0x850B, 0x8519, 0x852F, 0x8662, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x8756, 0x8763, 
    0x8764, 0x8777, 0x87E1, 0x8773, 0x8758, 0x8754, 0x875B, 0x8752, 0x8761, 
    0x875A, 0x8751, 0x875E, 0x876D, 0x876A, 0x8750, 0x874E, 0x875F, 0x875D, 
    0x876F, 0x876C, 0x877A, 0x876E, 0x875C, 0x8765, 0x874F, 0x877B, 0x8775, 
    0x8762, 0x8767, 0x8769, 0x885A, 0x8905, 0x890C, 0x8914, 0x890B, 0x8917, 
    0x8918, 0x8919, 0x8906, 0x8916, 0x8911, 0x890E, 0x8909, 0x89A2, 0x89A4, 
    0x89A3, 0x89ED, 0x89F0, 0x89EC, 0x8ACF, 0x8AC6, 0x8AB8, 0x8AD3, 0x8AD1, 
    0x8AD4, 0x8AD5, 0x8ABB, 0x8AD7, 0x8ABE, 0x8AC0, 0x8AC5, 0x8AD8, 0x8AC3, 
    0x8ABA, 0x8ABD, 0x8AD9, 0x8C3E, 0x8C4D, 0x8C8F, 0x8CE5, 0x8CDF, 0x8CD9, 
    0x8CE8, 0x8CDA, 0x8CDD, 0x8CE7, 0x8DA0, 0x8D9C, 0x8DA1, 0x8D9B, 0x8E20, 
    0x8E23, 0x8E25, 0x8E24, 0x8E2E, 0x8E15, 0x8E1B, 0x8E16, 0x8E11, 0x8E19, 
    0x8E26, 0x8E27},
    {0x8E14, 0x8E12, 0x8E18, 0x8E13, 0x8E1C, 0x8E17, 0x8E1A, 0x8F2C, 0x8F24, 
    0x8F18, 0x8F1A, 0x8F20, 0x8F23, 0x8F16, 0x8F17, 0x9073, 0x9070, 0x906F, 
    0x9067, 0x906B, 0x912F, 0x912B, 0x9129, 0x912A, 0x9132, 0x9126, 0x912E, 
    0x9185, 0x9186, 0x918A, 0x9181, 0x9182, 0x9184, 0x9180, 0x92D0, 0x92C3, 
    0x92C4, 0x92C0, 0x92D9, 0x92B6, 0x92CF, 0x92F1, 0x92DF, 0x92D8, 0x92E9, 
    0x92D7, 0x92DD, 0x92CC, 0x92EF, 0x92C2, 0x92E8, 0x92CA, 0x92C8, 0x92CE, 
    0x92E6, 0x92CD, 0x92D5, 0x92C9, 0x92E0, 0x92DE, 0x92E7, 0x92D1, 0x92D3, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x92B5, 0x92E1, 
    0x92C6, 0x92B4, 0x957C, 0x95AC, 0x95AB, 0x95AE, 0x95B0, 0x96A4, 0x96A2, 
    0x96D3, 0x9705, 0x9708, 0x9702, 0x975A, 0x978A, 0x978E, 0x9788, 0x97D0, 
    0x97CF, 0x981E, 0x981D, 0x9826, 0x9829, 0x9828, 0x9820, 0x981B, 0x9827, 
    0x98B2, 0x9908, 0x98FA, 0x9911, 0x9914, 0x9916, 0x9917, 0x9915, 0x99DC, 
    0x99CD, 0x99CF, 0x99D3, 0x99D4, 0x99CE, 0x99C9, 0x99D6, 0x99D8, 0x99CB, 
    0x99D7, 0x99CC, 0x9AB3, 0x9AEC, 0x9AEB, 0x9AF3, 0x9AF2, 0x9AF1, 0x9B46, 
    0x9B43, 0x9B67, 0x9B74, 0x9B71, 0x9B66, 0x9B76, 0x9B75, 0x9B70, 0x9B68, 
    0x9B64, 0x9B6C, 0x9CFC, 0x9CFA, 0x9CFD, 0x9CFF, 0x9CF7, 0x9D07, 0x9D00, 
    0x9CF9, 0x9CFB, 0x9D08, 0x9D05, 0x9D04, 0x9E83, 0x9ED3, 0x9F0F, 0x9F10, 
    0x511C, 0x5113, 0x5117, 0x511A, 0x5111, 0x51DE, 0x5334, 0x53E1, 0x5670, 
    0x5660, 0x566E},
    {0x5673, 0x5666, 0x5663, 0x566D, 0x5672, 0x565E, 0x5677, 0x571C, 0x571B, 
    0x58C8, 0x58BD, 0x58C9, 0x58BF, 0x58BA, 0x58C2, 0x58BC, 0x58C6, 0x5B17, 
    0x5B19, 0x5B1B, 0x5B21, 0x5B14, 0x5B13, 0x5B10, 0x5B16, 0x5B28, 0x5B1A, 
    0x5B20, 0x5B1E, 0x5BEF, 0x5DAC, 0x5DB1, 0x5DA9, 0x5DA7, 0x5DB5, 0x5DB0, 
    0x5DAE, 0x5DAA, 0x5DA8, 0x5DB2, 0x5DAD, 0x5DAF, 0x5DB4, 0x5E67, 0x5E68, 
    0x5E66, 0x5E6F, 0x5EE9, 0x5EE7, 0x5EE6, 0x5EE8, 0x5EE5, 0x5F4B, 0x5FBC, 
    0x619D, 0x61A8, 0x6196, 0x61C5, 0x61B4, 0x61C6, 0x61C1, 0x61CC, 0x61BA, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x61BF, 0x61B8, 
    0x618C, 0x64D7, 0x64D6, 0x64D0, 0x64CF, 0x64C9, 0x64BD, 0x6489, 0x64C3, 
    0x64DB, 0x64F3, 0x64D9, 0x6533, 0x657F, 0x657C, 0x65A2, 0x66C8, 0x66BE, 
    0x66C0, 0x66CA, 0x66CB, 0x66CF, 0x66BD, 0x66BB, 0x66BA, 0x66CC, 0x6723, 
    0x6A34, 0x6A66, 0x6A49, 0x6A67, 0x6A32, 0x6A68, 0x6A3E, 0x6A5D, 0x6A6D, 
    0x6A76, 0x6A5B, 0x6A51, 0x6A28, 0x6A5A, 0x6A3B, 0x6A3F, 0x6A41, 0x6A6A, 
    0x6A64, 0x6A50, 0x6A4F, 0x6A54, 0x6A6F, 0x6A69, 0x6A60, 0x6A3C, 0x6A5E, 
    0x6A56, 0x6A55, 0x6A4D, 0x6A4E, 0x6A46, 0x6B55, 0x6B54, 0x6B56, 0x6BA7, 
    0x6BAA, 0x6BAB, 0x6BC8, 0x6BC7, 0x6C04, 0x6C03, 0x6C06, 0x6FAD, 0x6FCB, 
    0x6FA3, 0x6FC7, 0x6FBC, 0x6FCE, 0x6FC8, 0x6F5E, 0x6FC4, 0x6FBD, 0x6F9E, 
    0x6FCA, 0x6FA8, 0x7004, 0x6FA5, 0x6FAE, 0x6FBA, 0x6FAC, 0x6FAA, 0x6FCF, 
    0x6FBF, 0x6FB8},
    {0x6FA2, 0x6FC9, 0x6FAB, 0x6FCD, 0x6FAF, 0x6FB2, 0x6FB0, 0x71C5, 0x71C2, 
    0x71BF, 0x71B8, 0x71D6, 0x71C0, 0x71C1, 0x71CB, 0x71D4, 0x71CA, 0x71C7, 
    0x71CF, 0x71BD, 0x71D8, 0x71BC, 0x71C6, 0x71DA, 0x71DB, 0x729D, 0x729E, 
    0x7369, 0x7366, 0x7367, 0x736C, 0x7365, 0x736B, 0x736A, 0x747F, 0x749A, 
    0x74A0, 0x7494, 0x7492, 0x7495, 0x74A1, 0x750B, 0x7580, 0x762F, 0x762D, 
    0x7631, 0x763D, 0x7633, 0x763C, 0x7635, 0x7632, 0x7630, 0x76BB, 0x76E6, 
    0x779A, 0x779D, 0x77A1, 0x779C, 0x779B, 0x77A2, 0x77A3, 0x7795, 0x7799, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x7797, 0x78DD, 
    0x78E9, 0x78E5, 0x78EA, 0x78DE, 0x78E3, 0x78DB, 0x78E1, 0x78E2, 0x78ED, 
    0x78DF, 0x78E0, 0x79A4, 0x7A44, 0x7A48, 0x7A47, 0x7AB6, 0x7AB8, 0x7AB5, 
    0x7AB1, 0x7AB7, 0x7BDE, 0x7BE3, 0x7BE7, 0x7BDD, 0x7BD5, 0x7BE5, 0x7BDA, 
    0x7BE8, 0x7BF9, 0x7BD4, 0x7BEA, 0x7BE2, 0x7BDC, 0x7BEB, 0x7BD8, 0x7BDF, 
    0x7CD2, 0x7CD4, 0x7CD7, 0x7CD0, 0x7CD1, 0x7E12, 0x7E21, 0x7E17, 0x7E0C, 
    0x7E1F, 0x7E20, 0x7E13, 0x7E0E, 0x7E1C, 0x7E15, 0x7E1A, 0x7E22, 0x7E0B, 
    0x7E0F, 0x7E16, 0x7E0D, 0x7E14, 0x7E25, 0x7E24, 0x7F43, 0x7F7B, 0x7F7C, 
    0x7F7A, 0x7FB1, 0x7FEF, 0x802A, 0x8029, 0x806C, 0x81B1, 0x81A6, 0x81AE, 
    0x81B9, 0x81B5, 0x81AB, 0x81B0, 0x81AC, 0x81B4, 0x81B2, 0x81B7, 0x81A7, 
    0x81F2, 0x8255, 0x8256, 0x8257, 0x8556, 0x8545, 0x856B, 0x854D, 0x8553, 
    0x8561, 0x8558},
    {0x8540, 0x8546, 0x8564, 0x8541, 0x8562, 0x8544, 0x8551, 0x8547, 0x8563, 
    0x853E, 0x855B, 0x8571, 0x854E, 0x856E, 0x8575, 0x8555, 0x8567, 0x8560, 
    0x858C, 0x8566, 0x855D, 0x8554, 0x8565, 0x856C, 0x8663, 0x8665, 0x8664, 
    0x879B, 0x878F, 0x8797, 0x8793, 0x8792, 0x8788, 0x8781, 0x8796, 0x8798, 
    0x8779, 0x8787, 0x87A3, 0x8785, 0x8790, 0x8791, 0x879D, 0x8784, 0x8794, 
    0x879C, 0x879A, 0x8789, 0x891E, 0x8926, 0x8930, 0x892D, 0x892E, 0x8927, 
    0x8931, 0x8922, 0x8929, 0x8923, 0x892F, 0x892C, 0x891F, 0x89F1, 0x8AE0, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x8AE2, 0x8AF2, 
    0x8AF4, 0x8AF5, 0x8ADD, 0x8B14, 0x8AE4, 0x8ADF, 0x8AF0, 0x8AC8, 0x8ADE, 
    0x8AE1, 0x8AE8, 0x8AFF, 0x8AEF, 0x8AFB, 0x8C91, 0x8C92, 0x8C90, 0x8CF5, 
    0x8CEE, 0x8CF1, 0x8CF0, 0x8CF3, 0x8D6C, 0x8D6E, 0x8DA5, 0x8DA7, 0x8E33, 
    0x8E3E, 0x8E38, 0x8E40, 0x8E45, 0x8E36, 0x8E3C, 0x8E3D, 0x8E41, 0x8E30, 
    0x8E3F, 0x8EBD, 0x8F36, 0x8F2E, 0x8F35, 0x8F32, 0x8F39, 0x8F37, 0x8F34, 
    0x9076, 0x9079, 0x907B, 0x9086, 0x90FA, 0x9133, 0x9135, 0x9136, 0x9193, 
    0x9190, 0x9191, 0x918D, 0x918F, 0x9327, 0x931E, 0x9308, 0x931F, 0x9306, 
    0x930F, 0x937A, 0x9338, 0x933C, 0x931B, 0x9323, 0x9312, 0x9301, 0x9346, 
    0x932D, 0x930E, 0x930D, 0x92CB, 0x931D, 0x92FA, 0x9325, 0x9313, 0x92F9, 
    0x92F7, 0x9334, 0x9302, 0x9324, 0x92FF, 0x9329, 0x9339, 0x9335, 0x932A, 
    0x9314, 0x930C},
    {0x930B, 0x92FE, 0x9309, 0x9300, 0x92FB, 0x9316, 0x95BC, 0x95CD, 0x95BE, 
    0x95B9, 0x95BA, 0x95B6, 0x95BF, 0x95B5, 0x95BD, 0x96A9, 0x96D4, 0x970B, 
    0x9712, 0x9710, 0x9799, 0x9797, 0x9794, 0x97F0, 0x97F8, 0x9835, 0x982F, 
    0x9832, 0x9924, 0x991F, 0x9927, 0x9929, 0x999E, 0x99EE, 0x99EC, 0x99E5, 
    0x99E4, 0x99F0, 0x99E3, 0x99EA, 0x99E9, 0x99E7, 0x9AB9, 0x9ABF, 0x9AB4, 
    0x9ABB, 0x9AF6, 0x9AFA, 0x9AF9, 0x9AF7, 0x9B33, 0x9B80, 0x9B85, 0x9B87, 
    0x9B7C, 0x9B7E, 0x9B7B, 0x9B82, 0x9B93, 0x9B92, 0x9B90, 0x9B7A, 0x9B95, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x9B7D, 0x9B88, 
    0x9D25, 0x9D17, 0x9D20, 0x9D1E, 0x9D14, 0x9D29, 0x9D1D, 0x9D18, 0x9D22, 
    0x9D10, 0x9D19, 0x9D1F, 0x9E88, 0x9E86, 0x9E87, 0x9EAE, 0x9EAD, 0x9ED5, 
    0x9ED6, 0x9EFA, 0x9F12, 0x9F3D, 0x5126, 0x5125, 0x5122, 0x5124, 0x5120, 
    0x5129, 0x52F4, 0x5693, 0x568C, 0x568D, 0x5686, 0x5684, 0x5683, 0x567E, 
    0x5682, 0x567F, 0x5681, 0x58D6, 0x58D4, 0x58CF, 0x58D2, 0x5B2D, 0x5B25, 
    0x5B32, 0x5B23, 0x5B2C, 0x5B27, 0x5B26, 0x5B2F, 0x5B2E, 0x5B7B, 0x5BF1, 
    0x5BF2, 0x5DB7, 0x5E6C, 0x5E6A, 0x5FBE, 0x5FBB, 0x61C3, 0x61B5, 0x61BC, 
    0x61E7, 0x61E0, 0x61E5, 0x61E4, 0x61E8, 0x61DE, 0x64EF, 0x64E9, 0x64E3, 
    0x64EB, 0x64E4, 0x64E8, 0x6581, 0x6580, 0x65B6, 0x65DA, 0x66D2, 0x6A8D, 
    0x6A96, 0x6A81, 0x6AA5, 0x6A89, 0x6A9F, 0x6A9B, 0x6AA1, 0x6A9E, 0x6A87, 
    0x6A93, 0x6A8E},
    {0x6A95, 0x6A83, 0x6AA8, 0x6AA4, 0x6A91, 0x6A7F, 0x6AA6, 0x6A9A, 0x6A85, 
    0x6A8C, 0x6A92, 0x6B5B, 0x6BAD, 0x6C09, 0x6FCC, 0x6FA9, 0x6FF4, 0x6FD4, 
    0x6FE3, 0x6FDC, 0x6FED, 0x6FE7, 0x6FE6, 0x6FDE, 0x6FF2, 0x6FDD, 0x6FE2, 
    0x6FE8, 0x71E1, 0x71F1, 0x71E8, 0x71F2, 0x71E4, 0x71F0, 0x71E2, 0x7373, 
    0x736E, 0x736F, 0x7497, 0x74B2, 0x74AB, 0x7490, 0x74AA, 0x74AD, 0x74B1, 
    0x74A5, 0x74AF, 0x7510, 0x7511, 0x7512, 0x750F, 0x7584, 0x7643, 0x7648, 
    0x7649, 0x7647, 0x76A4, 0x76E9, 0x77B5, 0x77AB, 0x77B2, 0x77B7, 0x77B6, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x77B4, 0x77B1, 
    0x77A8, 0x77F0, 0x78F3, 0x78FD, 0x7902, 0x78FB, 0x78FC, 0x78F2, 0x7905, 
    0x78F9, 0x78FE, 0x7904, 0x79AB, 0x79A8, 0x7A5C, 0x7A5B, 0x7A56, 0x7A58, 
    0x7A54, 0x7A5A, 0x7ABE, 0x7AC0, 0x7AC1, 0x7C05, 0x7C0F, 0x7BF2, 0x7C00, 
    0x7BFF, 0x7BFB, 0x7C0E, 0x7BF4, 0x7C0B, 0x7BF3, 0x7C02, 0x7C09, 0x7C03, 
    0x7C01, 0x7BF8, 0x7BFD, 0x7C06, 0x7BF0, 0x7BF1, 0x7C10, 0x7C0A, 0x7CE8, 
    0x7E2D, 0x7E3C, 0x7E42, 0x7E33, 0x9848, 0x7E38, 0x7E2A, 0x7E49, 0x7E40, 
    0x7E47, 0x7E29, 0x7E4C, 0x7E30, 0x7E3B, 0x7E36, 0x7E44, 0x7E3A, 0x7F45, 
    0x7F7F, 0x7F7E, 0x7F7D, 0x7FF4, 0x7FF2, 0x802C, 0x81BB, 0x81C4, 0x81CC, 
    0x81CA, 0x81C5, 0x81C7, 0x81BC, 0x81E9, 0x825B, 0x825A, 0x825C, 0x8583, 
    0x8580, 0x858F, 0x85A7, 0x8595, 0x85A0, 0x858B, 0x85A3, 0x857B, 0x85A4, 
    0x859A, 0x859E},
    {0x8577, 0x857C, 0x8589, 0x85A1, 0x857A, 0x8578, 0x8557, 0x858E, 0x8596, 
    0x8586, 0x858D, 0x8599, 0x859D, 0x8581, 0x85A2, 0x8582, 0x8588, 0x8585, 
    0x8579, 0x8576, 0x8598, 0x8590, 0x859F, 0x8668, 0x87BE, 0x87AA, 0x87AD, 
    0x87C5, 0x87B0, 0x87AC, 0x87B9, 0x87B5, 0x87BC, 0x87AE, 0x87C9, 0x87C3, 
    0x87C2, 0x87CC, 0x87B7, 0x87AF, 0x87C4, 0x87CA, 0x87B4, 0x87B6, 0x87BF, 
    0x87B8, 0x87BD, 0x87DE, 0x87B2, 0x8935, 0x8933, 0x893C, 0x893E, 0x8941, 
    0x8952, 0x8937, 0x8942, 0x89AD, 0x89AF, 0x89AE, 0x89F2, 0x89F3, 0x8B1E, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x8B18, 0x8B16, 
    0x8B11, 0x8B05, 0x8B0B, 0x8B22, 0x8B0F, 0x8B12, 0x8B15, 0x8B07, 0x8B0D, 
    0x8B08, 0x8B06, 0x8B1C, 0x8B13, 0x8B1A, 0x8C4F, 0x8C70, 0x8C72, 0x8C71, 
    0x8C6F, 0x8C95, 0x8C94, 0x8CF9, 0x8D6F, 0x8E4E, 0x8E4D, 0x8E53, 0x8E50, 
    0x8E4C, 0x8E47, 0x8F43, 0x8F40, 0x9085, 0x907E, 0x9138, 0x919A, 0x91A2, 
    0x919B, 0x9199, 0x919F, 0x91A1, 0x919D, 0x91A0, 0x93A1, 0x9383, 0x93AF, 
    0x9364, 0x9356, 0x9347, 0x937C, 0x9358, 0x935C, 0x9376, 0x9349, 0x9350, 
    0x9351, 0x9360, 0x936D, 0x938F, 0x934C, 0x936A, 0x9379, 0x9357, 0x9355, 
    0x9352, 0x934F, 0x9371, 0x9377, 0x937B, 0x9361, 0x935E, 0x9363, 0x9367, 
    0x9380, 0x934E, 0x9359, 0x95C7, 0x95C0, 0x95C9, 0x95C3, 0x95C5, 0x95B7, 
    0x96AE, 0x96B0, 0x96AC, 0x9720, 0x971F, 0x9718, 0x971D, 0x9719, 0x979A, 
    0x97A1, 0x979C},
    {0x979E, 0x979D, 0x97D5, 0x97D4, 0x97F1, 0x9841, 0x9844, 0x984A, 0x9849, 
    0x9845, 0x9843, 0x9925, 0x992B, 0x992C, 0x992A, 0x9933, 0x9932, 0x992F, 
    0x992D, 0x9931, 0x9930, 0x9998, 0x99A3, 0x99A1, 0x9A02, 0x99FA, 0x99F4, 
    0x99F7, 0x99F9, 0x99F8, 0x99F6, 0x99FB, 0x99FD, 0x99FE, 0x99FC, 0x9A03, 
    0x9ABE, 0x9AFE, 0x9AFD, 0x9B01, 0x9AFC, 0x9B48, 0x9B9A, 0x9BA8, 0x9B9E, 
    0x9B9B, 0x9BA6, 0x9BA1, 0x9BA5, 0x9BA4, 0x9B86, 0x9BA2, 0x9BA0, 0x9BAF, 
    0x9D33, 0x9D41, 0x9D67, 0x9D36, 0x9D2E, 0x9D2F, 0x9D31, 0x9D38, 0x9D30, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x9D45, 0x9D42, 
    0x9D43, 0x9D3E, 0x9D37, 0x9D40, 0x9D3D, 0x7FF5, 0x9D2D, 0x9E8A, 0x9E89, 
    0x9E8D, 0x9EB0, 0x9EC8, 0x9EDA, 0x9EFB, 0x9EFF, 0x9F24, 0x9F23, 0x9F22, 
    0x9F54, 0x9FA0, 0x5131, 0x512D, 0x512E, 0x5698, 0x569C, 0x5697, 0x569A, 
    0x569D, 0x5699, 0x5970, 0x5B3C, 0x5C69, 0x5C6A, 0x5DC0, 0x5E6D, 0x5E6E, 
    0x61D8, 0x61DF, 0x61ED, 0x61EE, 0x61F1, 0x61EA, 0x61F0, 0x61EB, 0x61D6, 
    0x61E9, 0x64FF, 0x6504, 0x64FD, 0x64F8, 0x6501, 0x6503, 0x64FC, 0x6594, 
    0x65DB, 0x66DA, 0x66DB, 0x66D8, 0x6AC5, 0x6AB9, 0x6ABD, 0x6AE1, 0x6AC6, 
    0x6ABA, 0x6AB6, 0x6AB7, 0x6AC7, 0x6AB4, 0x6AAD, 0x6B5E, 0x6BC9, 0x6C0B, 
    0x7007, 0x700C, 0x700D, 0x7001, 0x7005, 0x7014, 0x700E, 0x6FFF, 0x7000, 
    0x6FFB, 0x7026, 0x6FFC, 0x6FF7, 0x700A, 0x7201, 0x71FF, 0x71F9, 0x7203, 
    0x71FD, 0x7376},
    {0x74B8, 0x74C0, 0x74B5, 0x74C1, 0x74BE, 0x74B6, 0x74BB, 0x74C2, 0x7514, 
    0x7513, 0x765C, 0x7664, 0x7659, 0x7650, 0x7653, 0x7657, 0x765A, 0x76A6, 
    0x76BD, 0x76EC, 0x77C2, 0x77BA, 0x78FF, 0x790C, 0x7913, 0x7914, 0x7909, 
    0x7910, 0x7912, 0x7911, 0x79AD, 0x79AC, 0x7A5F, 0x7C1C, 0x7C29, 0x7C19, 
    0x7C20, 0x7C1F, 0x7C2D, 0x7C1D, 0x7C26, 0x7C28, 0x7C22, 0x7C25, 0x7C30, 
    0x7E5C, 0x7E50, 0x7E56, 0x7E63, 0x7E58, 0x7E62, 0x7E5F, 0x7E51, 0x7E60, 
    0x7E57, 0x7E53, 0x7FB5, 0x7FB3, 0x7FF7, 0x7FF8, 0x8075, 0x81D1, 0x81D2, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x81D0, 0x825F, 
    0x825E, 0x85B4, 0x85C6, 0x85C0, 0x85C3, 0x85C2, 0x85B3, 0x85B5, 0x85BD, 
    0x85C7, 0x85C4, 0x85BF, 0x85CB, 0x85CE, 0x85C8, 0x85C5, 0x85B1, 0x85B6, 
    0x85D2, 0x8624, 0x85B8, 0x85B7, 0x85BE, 0x8669, 0x87E7, 0x87E6, 0x87E2, 
    0x87DB, 0x87EB, 0x87EA, 0x87E5, 0x87DF, 0x87F3, 0x87E4, 0x87D4, 0x87DC, 
    0x87D3, 0x87ED, 0x87D8, 0x87E3, 0x87A4, 0x87D7, 0x87D9, 0x8801, 0x87F4, 
    0x87E8, 0x87DD, 0x8953, 0x894B, 0x894F, 0x894C, 0x8946, 0x8950, 0x8951, 
    0x8949, 0x8B2A, 0x8B27, 0x8B23, 0x8B33, 0x8B30, 0x8B35, 0x8B47, 0x8B2F, 
    0x8B3C, 0x8B3E, 0x8B31, 0x8B25, 0x8B37, 0x8B26, 0x8B36, 0x8B2E, 0x8B24, 
    0x8B3B, 0x8B3D, 0x8B3A, 0x8C42, 0x8C75, 0x8C99, 0x8C98, 0x8C97, 0x8CFE, 
    0x8D04, 0x8D02, 0x8D00, 0x8E5C, 0x8E62, 0x8E60, 0x8E57, 0x8E56, 0x8E5E, 
    0x8E65, 0x8E67},
    {0x8E5B, 0x8E5A, 0x8E61, 0x8E5D, 0x8E69, 0x8E54, 0x8F46, 0x8F47, 0x8F48, 
    0x8F4B, 0x9128, 0x913A, 0x913B, 0x913E, 0x91A8, 0x91A5, 0x91A7, 0x91AF, 
    0x91AA, 0x93B5, 0x938C, 0x9392, 0x93B7, 0x939B, 0x939D, 0x9389, 0x93A7, 
    0x938E, 0x93AA, 0x939E, 0x93A6, 0x9395, 0x9388, 0x9399, 0x939F, 0x938D, 
    0x93B1, 0x9391, 0x93B2, 0x93A4, 0x93A8, 0x93B4, 0x93A3, 0x93A5, 0x95D2, 
    0x95D3, 0x95D1, 0x96B3, 0x96D7, 0x96DA, 0x5DC2, 0x96DF, 0x96D8, 0x96DD, 
    0x9723, 0x9722, 0x9725, 0x97AC, 0x97AE, 0x97A8, 0x97AB, 0x97A4, 0x97AA, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x97A2, 0x97A5, 
    0x97D7, 0x97D9, 0x97D6, 0x97D8, 0x97FA, 0x9850, 0x9851, 0x9852, 0x98B8, 
    0x9941, 0x993C, 0x993A, 0x9A0F, 0x9A0B, 0x9A09, 0x9A0D, 0x9A04, 0x9A11, 
    0x9A0A, 0x9A05, 0x9A07, 0x9A06, 0x9AC0, 0x9ADC, 0x9B08, 0x9B04, 0x9B05, 
    0x9B29, 0x9B35, 0x9B4A, 0x9B4C, 0x9B4B, 0x9BC7, 0x9BC6, 0x9BC3, 0x9BBF, 
    0x9BC1, 0x9BB5, 0x9BB8, 0x9BD3, 0x9BB6, 0x9BC4, 0x9BB9, 0x9BBD, 0x9D5C, 
    0x9D53, 0x9D4F, 0x9D4A, 0x9D5B, 0x9D4B, 0x9D59, 0x9D56, 0x9D4C, 0x9D57, 
    0x9D52, 0x9D54, 0x9D5F, 0x9D58, 0x9D5A, 0x9E8E, 0x9E8C, 0x9EDF, 0x9F01, 
    0x9F00, 0x9F16, 0x9F25, 0x9F2B, 0x9F2A, 0x9F29, 0x9F28, 0x9F4C, 0x9F55, 
    0x5134, 0x5135, 0x5296, 0x52F7, 0x53B4, 0x56AB, 0x56AD, 0x56A6, 0x56A7, 
    0x56AA, 0x56AC, 0x58DA, 0x58DD, 0x58DB, 0x5912, 0x5B3D, 0x5B3E, 0x5B3F, 
    0x5DC3, 0x5E70},
    {0x5FBF, 0x61FB, 0x6507, 0x6510, 0x650D, 0x6509, 0x650C, 0x650E, 0x6584, 
    0x65DE, 0x65DD, 0x66DE, 0x6AE7, 0x6AE0, 0x6ACC, 0x6AD1, 0x6AD9, 0x6ACB, 
    0x6ADF, 0x6ADC, 0x6AD0, 0x6AEB, 0x6ACF, 0x6ACD, 0x6ADE, 0x6B60, 0x6BB0, 
    0x6C0C, 0x7019, 0x7027, 0x7020, 0x7016, 0x702B, 0x7021, 0x7022, 0x7023, 
    0x7029, 0x7017, 0x7024, 0x701C, 0x702A, 0x720C, 0x720A, 0x7207, 0x7202, 
    0x7205, 0x72A5, 0x72A6, 0x72A4, 0x72A3, 0x72A1, 0x74CB, 0x74C5, 0x74B7, 
    0x74C3, 0x7516, 0x7660, 0x77C9, 0x77CA, 0x77C4, 0x77F1, 0x791D, 0x791B, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x7921, 0x791C, 
    0x7917, 0x791E, 0x79B0, 0x7A67, 0x7A68, 0x7C33, 0x7C3C, 0x7C39, 0x7C2C, 
    0x7C3B, 0x7CEC, 0x7CEA, 0x7E76, 0x7E75, 0x7E78, 0x7E70, 0x7E77, 0x7E6F, 
    0x7E7A, 0x7E72, 0x7E74, 0x7E68, 0x7F4B, 0x7F4A, 0x7F83, 0x7F86, 0x7FB7, 
    0x7FFD, 0x7FFE, 0x8078, 0x81D7, 0x81D5, 0x8264, 0x8261, 0x8263, 0x85EB, 
    0x85F1, 0x85ED, 0x85D9, 0x85E1, 0x85E8, 0x85DA, 0x85D7, 0x85EC, 0x85F2, 
    0x85F8, 0x85D8, 0x85DF, 0x85E3, 0x85DC, 0x85D1, 0x85F0, 0x85E6, 0x85EF, 
    0x85DE, 0x85E2, 0x8800, 0x87FA, 0x8803, 0x87F6, 0x87F7, 0x8809, 0x880C, 
    0x880B, 0x8806, 0x87FC, 0x8808, 0x87FF, 0x880A, 0x8802, 0x8962, 0x895A, 
    0x895B, 0x8957, 0x8961, 0x895C, 0x8958, 0x895D, 0x8959, 0x8988, 0x89B7, 
    0x89B6, 0x89F6, 0x8B50, 0x8B48, 0x8B4A, 0x8B40, 0x8B53, 0x8B56, 0x8B54, 
    0x8B4B, 0x8B55},
    {0x8B51, 0x8B42, 0x8B52, 0x8B57, 0x8C43, 0x8C77, 0x8C76, 0x8C9A, 0x8D06, 
    0x8D07, 0x8D09, 0x8DAC, 0x8DAA, 0x8DAD, 0x8DAB, 0x8E6D, 0x8E78, 0x8E73, 
    0x8E6A, 0x8E6F, 0x8E7B, 0x8EC2, 0x8F52, 0x8F51, 0x8F4F, 0x8F50, 0x8F53, 
    0x8FB4, 0x9140, 0x913F, 0x91B0, 0x91AD, 0x93DE, 0x93C7, 0x93CF, 0x93C2, 
    0x93DA, 0x93D0, 0x93F9, 0x93EC, 0x93CC, 0x93D9, 0x93A9, 0x93E6, 0x93CA, 
    0x93D4, 0x93EE, 0x93E3, 0x93D5, 0x93C4, 0x93CE, 0x93C0, 0x93D2, 0x93E7, 
    0x957D, 0x95DA, 0x95DB, 0x96E1, 0x9729, 0x972B, 0x972C, 0x9728, 0x9726, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x97B3, 0x97B7, 
    0x97B6, 0x97DD, 0x97DE, 0x97DF, 0x985C, 0x9859, 0x985D, 0x9857, 0x98BF, 
    0x98BD, 0x98BB, 0x98BE, 0x9948, 0x9947, 0x9943, 0x99A6, 0x99A7, 0x9A1A, 
    0x9A15, 0x9A25, 0x9A1D, 0x9A24, 0x9A1B, 0x9A22, 0x9A20, 0x9A27, 0x9A23, 
    0x9A1E, 0x9A1C, 0x9A14, 0x9AC2, 0x9B0B, 0x9B0A, 0x9B0E, 0x9B0C, 0x9B37, 
    0x9BEA, 0x9BEB, 0x9BE0, 0x9BDE, 0x9BE4, 0x9BE6, 0x9BE2, 0x9BF0, 0x9BD4, 
    0x9BD7, 0x9BEC, 0x9BDC, 0x9BD9, 0x9BE5, 0x9BD5, 0x9BE1, 0x9BDA, 0x9D77, 
    0x9D81, 0x9D8A, 0x9D84, 0x9D88, 0x9D71, 0x9D80, 0x9D78, 0x9D86, 0x9D8B, 
    0x9D8C, 0x9D7D, 0x9D6B, 0x9D74, 0x9D75, 0x9D70, 0x9D69, 0x9D85, 0x9D73, 
    0x9D7B, 0x9D82, 0x9D6F, 0x9D79, 0x9D7F, 0x9D87, 0x9D68, 0x9E94, 0x9E91, 
    0x9EC0, 0x9EFC, 0x9F2D, 0x9F40, 0x9F41, 0x9F4D, 0x9F56, 0x9F57, 0x9F58, 
    0x5337, 0x56B2},
    {0x56B5, 0x56B3, 0x58E3, 0x5B45, 0x5DC6, 0x5DC7, 0x5EEE, 0x5EEF, 0x5FC0, 
    0x5FC1, 0x61F9, 0x6517, 0x6516, 0x6515, 0x6513, 0x65DF, 0x66E8, 0x66E3, 
    0x66E4, 0x6AF3, 0x6AF0, 0x6AEA, 0x6AE8, 0x6AF9, 0x6AF1, 0x6AEE, 0x6AEF, 
    0x703C, 0x7035, 0x702F, 0x7037, 0x7034, 0x7031, 0x7042, 0x7038, 0x703F, 
    0x703A, 0x7039, 0x7040, 0x703B, 0x7033, 0x7041, 0x7213, 0x7214, 0x72A8, 
    0x737D, 0x737C, 0x74BA, 0x76AB, 0x76AA, 0x76BE, 0x76ED, 0x77CC, 0x77CE, 
    0x77CF, 0x77CD, 0x77F2, 0x7925, 0x7923, 0x7927, 0x7928, 0x7924, 0x7929, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x79B2, 0x7A6E, 
    0x7A6C, 0x7A6D, 0x7AF7, 0x7C49, 0x7C48, 0x7C4A, 0x7C47, 0x7C45, 0x7CEE, 
    0x7E7B, 0x7E7E, 0x7E81, 0x7E80, 0x7FBA, 0x7FFF, 0x8079, 0x81DB, 0x81D9, 
    0x820B, 0x8268, 0x8269, 0x8622, 0x85FF, 0x8601, 0x85FE, 0x861B, 0x8600, 
    0x85F6, 0x8604, 0x8609, 0x8605, 0x860C, 0x85FD, 0x8819, 0x8810, 0x8811, 
    0x8817, 0x8813, 0x8816, 0x8963, 0x8966, 0x89B9, 0x89F7, 0x8B60, 0x8B6A, 
    0x8B5D, 0x8B68, 0x8B63, 0x8B65, 0x8B67, 0x8B6D, 0x8DAE, 0x8E86, 0x8E88, 
    0x8E84, 0x8F59, 0x8F56, 0x8F57, 0x8F55, 0x8F58, 0x8F5A, 0x908D, 0x9143, 
    0x9141, 0x91B7, 0x91B5, 0x91B2, 0x91B3, 0x940B, 0x9413, 0x93FB, 0x9420, 
    0x940F, 0x9414, 0x93FE, 0x9415, 0x9410, 0x9428, 0x9419, 0x940D, 0x93F5, 
    0x9400, 0x93F7, 0x9407, 0x940E, 0x9416, 0x9412, 0x93FA, 0x9409, 0x93F8, 
    0x940A, 0x93FF},
    {0x93FC, 0x940C, 0x93F6, 0x9411, 0x9406, 0x95DE, 0x95E0, 0x95DF, 0x972E, 
    0x972F, 0x97B9, 0x97BB, 0x97FD, 0x97FE, 0x9860, 0x9862, 0x9863, 0x985F, 
    0x98C1, 0x98C2, 0x9950, 0x994E, 0x9959, 0x994C, 0x994B, 0x9953, 0x9A32, 
    0x9A34, 0x9A31, 0x9A2C, 0x9A2A, 0x9A36, 0x9A29, 0x9A2E, 0x9A38, 0x9A2D, 
    0x9AC7, 0x9ACA, 0x9AC6, 0x9B10, 0x9B12, 0x9B11, 0x9C0B, 0x9C08, 0x9BF7, 
    0x9C05, 0x9C12, 0x9BF8, 0x9C40, 0x9C07, 0x9C0E, 0x9C06, 0x9C17, 0x9C14, 
    0x9C09, 0x9D9F, 0x9D99, 0x9DA4, 0x9D9D, 0x9D92, 0x9D98, 0x9D90, 0x9D9B, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x9DA0, 0x9D94, 
    0x9D9C, 0x9DAA, 0x9D97, 0x9DA1, 0x9D9A, 0x9DA2, 0x9DA8, 0x9D9E, 0x9DA3, 
    0x9DBF, 0x9DA9, 0x9D96, 0x9DA6, 0x9DA7, 0x9E99, 0x9E9B, 0x9E9A, 0x9EE5, 
    0x9EE4, 0x9EE7, 0x9EE6, 0x9F30, 0x9F2E, 0x9F5B, 0x9F60, 0x9F5E, 0x9F5D, 
    0x9F59, 0x9F91, 0x513A, 0x5139, 0x5298, 0x5297, 0x56C3, 0x56BD, 0x56BE, 
    0x5B48, 0x5B47, 0x5DCB, 0x5DCF, 0x5EF1, 0x61FD, 0x651B, 0x6B02, 0x6AFC, 
    0x6B03, 0x6AF8, 0x6B00, 0x7043, 0x7044, 0x704A, 0x7048, 0x7049, 0x7045, 
    0x7046, 0x721D, 0x721A, 0x7219, 0x737E, 0x7517, 0x766A, 0x77D0, 0x792D, 
    0x7931, 0x792F, 0x7C54, 0x7C53, 0x7CF2, 0x7E8A, 0x7E87, 0x7E88, 0x7E8B, 
    0x7E86, 0x7E8D, 0x7F4D, 0x7FBB, 0x8030, 0x81DD, 0x8618, 0x862A, 0x8626, 
    0x861F, 0x8623, 0x861C, 0x8619, 0x8627, 0x862E, 0x8621, 0x8620, 0x8629, 
    0x861E, 0x8625},
    {0x8829, 0x881D, 0x881B, 0x8820, 0x8824, 0x881C, 0x882B, 0x884A, 0x896D, 
    0x8969, 0x896E, 0x896B, 0x89FA, 0x8B79, 0x8B78, 0x8B45, 0x8B7A, 0x8B7B, 
    0x8D10, 0x8D14, 0x8DAF, 0x8E8E, 0x8E8C, 0x8F5E, 0x8F5B, 0x8F5D, 0x9146, 
    0x9144, 0x9145, 0x91B9, 0x943F, 0x943B, 0x9436, 0x9429, 0x943D, 0x943C, 
    0x9430, 0x9439, 0x942A, 0x9437, 0x942C, 0x9440, 0x9431, 0x95E5, 0x95E4, 
    0x95E3, 0x9735, 0x973A, 0x97BF, 0x97E1, 0x9864, 0x98C9, 0x98C6, 0x98C0, 
    0x9958, 0x9956, 0x9A39, 0x9A3D, 0x9A46, 0x9A44, 0x9A42, 0x9A41, 0x9A3A, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x9A3F, 0x9ACD, 
    0x9B15, 0x9B17, 0x9B18, 0x9B16, 0x9B3A, 0x9B52, 0x9C2B, 0x9C1D, 0x9C1C, 
    0x9C2C, 0x9C23, 0x9C28, 0x9C29, 0x9C24, 0x9C21, 0x9DB7, 0x9DB6, 0x9DBC, 
    0x9DC1, 0x9DC7, 0x9DCA, 0x9DCF, 0x9DBE, 0x9DC5, 0x9DC3, 0x9DBB, 0x9DB5, 
    0x9DCE, 0x9DB9, 0x9DBA, 0x9DAC, 0x9DC8, 0x9DB1, 0x9DAD, 0x9DCC, 0x9DB3, 
    0x9DCD, 0x9DB2, 0x9E7A, 0x9E9C, 0x9EEB, 0x9EEE, 0x9EED, 0x9F1B, 0x9F18, 
    0x9F1A, 0x9F31, 0x9F4E, 0x9F65, 0x9F64, 0x9F92, 0x4EB9, 0x56C6, 0x56C5, 
    0x56CB, 0x5971, 0x5B4B, 0x5B4C, 0x5DD5, 0x5DD1, 0x5EF2, 0x6521, 0x6520, 
    0x6526, 0x6522, 0x6B0B, 0x6B08, 0x6B09, 0x6C0D, 0x7055, 0x7056, 0x7057, 
    0x7052, 0x721E, 0x721F, 0x72A9, 0x737F, 0x74D8, 0x74D5, 0x74D9, 0x74D7, 
    0x766D, 0x76AD, 0x7935, 0x79B4, 0x7A70, 0x7A71, 0x7C57, 0x7C5C, 0x7C59, 
    0x7C5B, 0x7C5A},
    {0x7CF4, 0x7CF1, 0x7E91, 0x7F4F, 0x7F87, 0x81DE, 0x826B, 0x8634, 0x8635, 
    0x8633, 0x862C, 0x8632, 0x8636, 0x882C, 0x8828, 0x8826, 0x882A, 0x8825, 
    0x8971, 0x89BF, 0x89BE, 0x89FB, 0x8B7E, 0x8B84, 0x8B82, 0x8B86, 0x8B85, 
    0x8B7F, 0x8D15, 0x8E95, 0x8E94, 0x8E9A, 0x8E92, 0x8E90, 0x8E96, 0x8E97, 
    0x8F60, 0x8F62, 0x9147, 0x944C, 0x9450, 0x944A, 0x944B, 0x944F, 0x9447, 
    0x9445, 0x9448, 0x9449, 0x9446, 0x973F, 0x97E3, 0x986A, 0x9869, 0x98CB, 
    0x9954, 0x995B, 0x9A4E, 0x9A53, 0x9A54, 0x9A4C, 0x9A4F, 0x9A48, 0x9A4A, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x9A49, 0x9A52, 
    0x9A50, 0x9AD0, 0x9B19, 0x9B2B, 0x9B3B, 0x9B56, 0x9B55, 0x9C46, 0x9C48, 
    0x9C3F, 0x9C44, 0x9C39, 0x9C33, 0x9C41, 0x9C3C, 0x9C37, 0x9C34, 0x9C32, 
    0x9C3D, 0x9C36, 0x9DDB, 0x9DD2, 0x9DDE, 0x9DDA, 0x9DCB, 0x9DD0, 0x9DDC, 
    0x9DD1, 0x9DDF, 0x9DE9, 0x9DD9, 0x9DD8, 0x9DD6, 0x9DF5, 0x9DD5, 0x9DDD, 
    0x9EB6, 0x9EF0, 0x9F35, 0x9F33, 0x9F32, 0x9F42, 0x9F6B, 0x9F95, 0x9FA2, 
    0x513D, 0x5299, 0x58E8, 0x58E7, 0x5972, 0x5B4D, 0x5DD8, 0x882F, 0x5F4F, 
    0x6201, 0x6203, 0x6204, 0x6529, 0x6525, 0x6596, 0x66EB, 0x6B11, 0x6B12, 
    0x6B0F, 0x6BCA, 0x705B, 0x705A, 0x7222, 0x7382, 0x7381, 0x7383, 0x7670, 
    0x77D4, 0x7C67, 0x7C66, 0x7E95, 0x826C, 0x863A, 0x8640, 0x8639, 0x863C, 
    0x8631, 0x863B, 0x863E, 0x8830, 0x8832, 0x882E, 0x8833, 0x8976, 0x8974, 
    0x8973, 0x89FE},
    {0x8B8C, 0x8B8E, 0x8B8B, 0x8B88, 0x8C45, 0x8D19, 0x8E98, 0x8F64, 0x8F63, 
    0x91BC, 0x9462, 0x9455, 0x945D, 0x9457, 0x945E, 0x97C4, 0x97C5, 0x9800, 
    0x9A56, 0x9A59, 0x9B1E, 0x9B1F, 0x9B20, 0x9C52, 0x9C58, 0x9C50, 0x9C4A, 
    0x9C4D, 0x9C4B, 0x9C55, 0x9C59, 0x9C4C, 0x9C4E, 0x9DFB, 0x9DF7, 0x9DEF, 
    0x9DE3, 0x9DEB, 0x9DF8, 0x9DE4, 0x9DF6, 0x9DE1, 0x9DEE, 0x9DE6, 0x9DF2, 
    0x9DF0, 0x9DE2, 0x9DEC, 0x9DF4, 0x9DF3, 0x9DE8, 0x9DED, 0x9EC2, 0x9ED0, 
    0x9EF2, 0x9EF3, 0x9F06, 0x9F1C, 0x9F38, 0x9F37, 0x9F36, 0x9F43, 0x9F4F, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x9F71, 0x9F70, 
    0x9F6E, 0x9F6F, 0x56D3, 0x56CD, 0x5B4E, 0x5C6D, 0x652D, 0x66ED, 0x66EE, 
    0x6B13, 0x705F, 0x7061, 0x705D, 0x7060, 0x7223, 0x74DB, 0x74E5, 0x77D5, 
    0x7938, 0x79B7, 0x79B6, 0x7C6A, 0x7E97, 0x7F89, 0x826D, 0x8643, 0x8838, 
    0x8837, 0x8835, 0x884B, 0x8B94, 0x8B95, 0x8E9E, 0x8E9F, 0x8EA0, 0x8E9D, 
    0x91BE, 0x91BD, 0x91C2, 0x946B, 0x9468, 0x9469, 0x96E5, 0x9746, 0x9743, 
    0x9747, 0x97C7, 0x97E5, 0x9A5E, 0x9AD5, 0x9B59, 0x9C63, 0x9C67, 0x9C66, 
    0x9C62, 0x9C5E, 0x9C60, 0x9E02, 0x9DFE, 0x9E07, 0x9E03, 0x9E06, 0x9E05, 
    0x9E00, 0x9E01, 0x9E09, 0x9DFF, 0x9DFD, 0x9E04, 0x9EA0, 0x9F1E, 0x9F46, 
    0x9F74, 0x9F75, 0x9F76, 0x56D4, 0x652E, 0x65B8, 0x6B18, 0x6B19, 0x6B17, 
    0x6B1A, 0x7062, 0x7226, 0x72AA, 0x77D8, 0x77D9, 0x7939, 0x7C69, 0x7C6B, 
    0x7CF6, 0x7E9A},
    {0x7E98, 0x7E9B, 0x7E99, 0x81E0, 0x81E1, 0x8646, 0x8647, 0x8648, 0x8979, 
    0x897A, 0x897C, 0x897B, 0x89FF, 0x8B98, 0x8B99, 0x8EA5, 0x8EA4, 0x8EA3, 
    0x946E, 0x946D, 0x946F, 0x9471, 0x9473, 0x9749, 0x9872, 0x995F, 0x9C68, 
    0x9C6E, 0x9C6D, 0x9E0B, 0x9E0D, 0x9E10, 0x9E0F, 0x9E12, 0x9E11, 0x9EA1, 
    0x9EF5, 0x9F09, 0x9F47, 0x9F78, 0x9F7B, 0x9F7A, 0x9F79, 0x571E, 0x7066, 
    0x7C6F, 0x883C, 0x8DB2, 0x8EA6, 0x91C3, 0x9474, 0x9478, 0x9476, 0x9475, 
    0x9A60, 0x9C74, 0x9C73, 0x9C71, 0x9C75, 0x9E14, 0x9E13, 0x9EF6, 0x9F0A, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 0x9FA4, 0x7068, 
    0x7065, 0x7CF7, 0x866A, 0x883E, 0x883D, 0x883F, 0x8B9E, 0x8C9C, 0x8EA9, 
    0x8EC9, 0x974B, 0x9873, 0x9874, 0x98CC, 0x9961, 0x99AB, 0x9A64, 0x9A66, 
    0x9A67, 0x9B24, 0x9E15, 0x9E17, 0x9F48, 0x6207, 0x6B1E, 0x7227, 0x864C, 
    0x8EA8, 0x9482, 0x9480, 0x9481, 0x9A69, 0x9A68, 0x9B2E, 0x9E19, 0x7229, 
    0x864B, 0x8B9F, 0x9483, 0x9C79, 0x9EB7, 0x7675, 0x9A6B, 0x9C7A, 0x9E1D, 
    0x7069, 0x706A, 0x9EA4, 0x9F7E, 0x9F49, 0x9F98,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR},
    { ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR,  ERROR, 
     ERROR,  ERROR},
};

/*
 * Mapping from Unicode back to Big5.
 * 
 * This inverse is done the same way the SBCS ones are done: I
 * provide a list of Big5 positions, sorted into ascending order of
 * their Unicode values. Then we can binary-search through this
 * list, referring to the above table for each comparison, to find
 * the coordinates for a given code point.
 * 
 * Generated by running the following sh+Perl over BIG5.TXT:

cat BIG5.TXT | \
sed -e $'s/^\\(0xA15A\t\\)0xFFFD/\\10xFF3F/' \
    -e $'s/^\\(0xA1FE\t\\)0xFFFD/\\10x2571/' \
    -e $'s/^\\(0xA240\t\\)0xFFFD/\\10x2572/' \
    -e $'s/^\\(0xA2CC\t\\)0xFFFD/\\10x5341/' \
    -e $'s/^\\(0xA2CE\t\\)0xFFFD/\\10x5345/' \
    -e '/^0xA1C3/d' -e '/^0xA1C5/d' | \
perl -ne '$a{$3}=[hex$1,hex$2] if /^0x(\S\S)(\S\S)\s+(0x\S+)\s/;' \
      -e 'END {$o="    ";for $k (sort keys %a) { ($r,$c) = @{$a{$k}}; ' \
      -e '$o .= $C; $C = ", "; (print "$o\n"), $o="    " if length $o > 70;' \
      -e '$o .= sprintf "{%d,%d}", $r-161,$c-64; } print "$o\n"; }'

 */

static const struct { unsigned char r, c; } big5_backward[] = {
    {1,6}, {1,7}, {1,4}, {0,113}, {1,24}, {0,147}, {0,16}, {0,145}, {0,146}, 
    {2,126}, {2,124}, {2,125}, {2,127}, {2,123}, {2,4}, {2,5}, {2,6}, {2,7}, 
    {2,8}, {2,9}, {2,10}, {2,11}, {2,12}, {2,13}, {2,14}, {2,15}, {2,16}, 
    {2,17}, {2,18}, {2,19}, {2,20}, {2,21}, {2,22}, {2,23}, {2,24}, {2,25}, 
    {2,26}, {2,27}, {2,28}, {2,29}, {2,30}, {2,31}, {2,32}, {2,33}, {2,34}, 
    {2,35}, {2,36}, {2,37}, {2,38}, {2,39}, {2,40}, {2,41}, {2,42}, {2,43}, 
    {2,44}, {2,45}, {2,46}, {2,47}, {2,48}, {2,49}, {2,50}, {2,51}, {38,115}, 
    {38,113}, {38,114}, {38,116}, {38,117}, {38,118}, {38,119}, {38,120}, 
    {38,121}, {38,122}, {38,123}, {38,124}, {38,125}, {38,126}, {38,127}, 
    {38,128}, {38,129}, {38,130}, {38,131}, {38,132}, {38,133}, {38,134}, 
    {38,135}, {38,136}, {38,137}, {38,138}, {38,139}, {38,140}, {38,141}, 
    {38,143}, {38,144}, {38,145}, {38,146}, {38,147}, {38,148}, {38,149}, 
    {38,150}, {38,151}, {38,152}, {38,153}, {38,154}, {38,155}, {38,156}, 
    {38,157}, {38,158}, {38,159}, {38,160}, {38,161}, {38,162}, {38,163}, 
    {38,164}, {38,165}, {38,166}, {38,167}, {38,168}, {38,142}, {0,22}, 
    {0,24}, {0,101}, {0,102}, {0,103}, {0,104}, {0,5}, {0,12}, {0,11}, 
    {0,108}, {0,107}, {0,112}, {0,130}, {1,10}, {0,129}, {1,11}, {1,121}, 
    {1,122}, {1,123}, {1,124}, {1,125}, {1,126}, {1,127}, {1,128}, {1,129}, 
    {1,130}, {0,182}, {0,180}, {0,183}, {0,181}, {0,184}, {0,185}, {0,187}, 
    {0,186}, {0,148}, {0,155}, {0,168}, {0,167}, {0,189}, {0,188}, {0,164}, 
    {0,165}, {0,172}, {0,173}, {0,175}, {0,174}, {0,163}, {0,156}, {0,154}, 
    {0,157}, {0,152}, {0,153}, {0,166}, {0,169}, {38,169}, {38,170}, {38,171}, 
    {38,172}, {38,173}, {38,174}, {38,175}, {38,176}, {38,177}, {38,178}, 
    {38,179}, {38,180}, {38,181}, {38,182}, {38,183}, {38,184}, {38,185}, 
    {38,186}, {38,187}, {38,188}, {1,55}, {1,56}, {1,58}, {1,59}, {1,60}, 
    {1,61}, {1,53}, {1,52}, {1,51}, {1,50}, {1,49}, {1,100}, {1,101}, {1,103}, 
    {1,102}, {1,62}, {1,97}, {1,99}, {1,98}, {1,108}, {1,109}, {1,110}, 
    {1,34}, {1,35}, {1,36}, {1,37}, {1,38}, {1,39}, {1,40}, {1,41}, {1,48}, 
    {1,47}, {1,46}, {1,45}, {1,44}, {1,43}, {1,42}, {1,54}, {1,57}, {0,125}, 
    {0,124}, {0,118}, {0,117}, {0,127}, {0,126}, {0,123}, {0,122}, {0,115}, 
    {0,119}, {0,116}, {1,104}, {1,105}, {1,107}, {1,106}, {0,121}, {0,120}, 
    {0,179}, {0,176}, {0,178}, {0,177}, {0,0}, {0,2}, {0,3}, {0,114}, {37,100}, 
    {0,49}, {0,50}, {0,45}, {0,46}, {0,53}, {0,54}, {0,57}, {0,58}, {0,41}, 
    {0,42}, {1,5}, {0,37}, {0,38}, {0,105}, {0,106}, {1,131}, {1,132}, 
    {1,133}, {1,134}, {1,135}, {1,136}, {1,137}, {1,138}, {1,139}, {37,101}, 
    {37,102}, {37,103}, {37,104}, {37,105}, {37,106}, {37,107}, {37,108}, 
    {37,109}, {37,110}, {37,111}, {37,112}, {37,113}, {37,114}, {37,115}, 
    {37,116}, {37,117}, {37,118}, {37,119}, {37,120}, {37,121}, {37,122}, 
    {37,123}, {37,124}, {37,125}, {37,126}, {37,127}, {37,128}, {37,129}, 
    {37,130}, {37,131}, {37,132}, {37,133}, {37,134}, {37,135}, {37,136}, 
    {37,137}, {37,138}, {37,139}, {37,140}, {37,141}, {37,142}, {37,143}, 
    {37,144}, {37,145}, {37,146}, {37,147}, {37,148}, {37,149}, {37,150}, 
    {37,151}, {37,152}, {37,153}, {37,154}, {37,155}, {37,156}, {37,157}, 
    {37,158}, {37,159}, {37,160}, {37,161}, {37,162}, {37,163}, {37,164}, 
    {37,165}, {37,166}, {37,167}, {37,168}, {37,169}, {37,170}, {37,171}, 
    {37,172}, {37,173}, {37,174}, {37,175}, {37,176}, {37,177}, {37,178}, 
    {37,179}, {37,180}, {37,181}, {37,182}, {37,183}, {37,98}, {37,99}, 
    {37,184}, {37,185}, {37,186}, {37,187}, {37,188}, {37,189}, {37,190}, 
    {38,0}, {38,1}, {38,2}, {38,3}, {38,4}, {38,5}, {38,6}, {38,7}, {38,8}, 
    {38,9}, {38,10}, {38,11}, {38,12}, {38,13}, {38,14}, {38,15}, {38,16}, 
    {38,17}, {38,18}, {38,19}, {38,20}, {38,21}, {38,22}, {38,23}, {38,24}, 
    {38,25}, {38,26}, {38,27}, {38,28}, {38,29}, {38,30}, {38,31}, {38,32}, 
    {38,33}, {38,34}, {38,35}, {38,36}, {38,37}, {38,38}, {38,39}, {38,40}, 
    {38,41}, {38,42}, {38,43}, {38,44}, {38,45}, {38,46}, {38,47}, {38,48}, 
    {38,49}, {38,50}, {38,51}, {38,52}, {38,53}, {38,54}, {38,55}, {38,56}, 
    {38,57}, {38,58}, {38,59}, {38,60}, {38,61}, {38,62}, {38,97}, {38,98}, 
    {38,99}, {38,100}, {38,101}, {38,102}, {38,103}, {38,104}, {38,105}, 
    {38,106}, {38,107}, {38,108}, {38,109}, {38,110}, {38,111}, {38,112}, 
    {37,97}, {2,52}, {2,53}, {2,54}, {2,55}, {2,56}, {2,57}, {2,58}, {2,59}, 
    {2,60}, {2,61}, {2,62}, {2,97}, {2,98}, {2,99}, {2,100}, {2,101}, {2,102}, 
    {2,103}, {2,104}, {2,105}, {2,106}, {2,107}, {2,108}, {2,109}, {2,110}, 
    {2,111}, {2,112}, {2,113}, {2,114}, {2,115}, {2,116}, {2,117}, {2,118}, 
    {2,119}, {2,120}, {2,121}, {2,122}, {0,128}, {1,21}, {1,22}, {1,16}, 
    {1,17}, {1,18}, {1,20}, {1,23}, {1,19}, {0,171}, {0,170}, {1,15}, {3,0}, 
    {3,2}, {3,3}, {40,5}, {3,22}, {3,20}, {3,23}, {3,21}, {40,6}, {3,99}, 
    {40,15}, {40,13}, {3,98}, {3,97}, {4,2}, {4,1}, {4,0}, {4,3}, {3,190}, 
    {4,160}, {4,161}, {7,131}, {3,24}, {3,100}, {40,16}, {3,101}, {40,35}, 
    {5,170}, {42,113}, {3,25}, {3,102}, {4,4}, {40,36}, {40,0}, {3,4}, 
    {3,27}, {40,7}, {3,28}, {3,103}, {4,5}, {4,7}, {4,6}, {4,162}, {4,163}, 
    {7,132}, {12,124}, {3,1}, {40,1}, {3,5}, {3,30}, {3,29}, {4,164}, {7,133}, 
    {15,110}, {51,11}, {21,131}, {59,113}, {59,114}, {3,6}, {3,105}, {7,134}, 
    {3,7}, {40,8}, {3,31}, {3,106}, {3,108}, {40,17}, {3,109}, {3,107}, 
    {4,165}, {7,135}, {7,136}, {10,5}, {3,32}, {3,110}, {4,166}, {4,168}, 
    {4,167}, {5,171}, {7,137}, {7,138}, {10,6}, {10,7}, {12,125}, {59,115}, 
    {85,150}, {3,8}, {3,112}, {3,111}, {40,18}, {3,113}, {3,119}, {3,114}, 
    {3,115}, {40,20}, {40,19}, {3,117}, {3,118}, {3,116}, {4,10}, {4,11}, 
    {4,12}, {4,13}, {4,9}, {4,16}, {40,42}, {40,38}, {40,41}, {4,17}, {4,33}, 
    {40,40}, {4,14}, {4,15}, {4,8}, {40,37}, {40,39}, {4,181}, {40,112}, 
    {4,178}, {4,182}, {40,122}, {40,110}, {4,179}, {40,114}, {4,180}, {4,183}, 
    {4,169}, {40,113}, {4,184}, {40,117}, {40,121}, {40,118}, {40,115}, 
    {4,170}, {4,172}, {4,185}, {4,174}, {40,107}, {4,177}, {4,175}, {4,176}, 
    {40,123}, {40,120}, {40,111}, {4,173}, {40,108}, {4,171}, {40,116}, 
    {40,119}, {40,109}, {41,38}, {6,2}, {5,180}, {41,39}, {5,177}, {6,4}, 
    {5,185}, {5,184}, {41,27}, {5,188}, {5,183}, {41,32}, {41,40}, {41,36}, 
    {5,186}, {5,189}, {5,174}, {6,7}, {41,29}, {42,125}, {5,172}, {6,3}, 
    {5,173}, {5,181}, {5,182}, {41,34}, {41,30}, {5,187}, {5,179}, {41,26}, 
    {5,175}, {41,37}, {6,5}, {6,8}, {5,178}, {6,0}, {6,6}, {5,176}, {41,35}, 
    {6,1}, {41,41}, {41,28}, {5,190}, {41,31}, {41,33}, {7,152}, {42,127}, 
    {42,139}, {7,144}, {42,140}, {7,139}, {7,149}, {7,142}, {42,121}, {7,150}, 
    {42,120}, {42,124}, {42,131}, {42,129}, {7,158}, {7,153}, {42,115}, 
    {42,117}, {7,155}, {7,143}, {42,118}, {42,130}, {42,137}, {7,148}, 
    {42,123}, {42,116}, {7,147}, {42,119}, {7,151}, {42,122}, {7,146}, 
    {7,141}, {7,156}, {42,132}, {7,157}, {42,136}, {42,134}, {42,138}, 
    {7,154}, {42,126}, {42,114}, {42,128}, {7,145}, {42,133}, {7,140}, 
    {42,135}, {10,22}, {10,10}, {44,160}, {44,168}, {10,9}, {10,17}, {10,29}, 
    {44,174}, {44,172}, {44,167}, {10,11}, {44,173}, {44,163}, {10,25}, 
    {10,16}, {10,24}, {44,158}, {44,170}, {44,161}, {10,20}, {44,162}, 
    {44,157}, {10,27}, {10,14}, {10,23}, {10,13}, {44,159}, {44,164}, {44,171}, 
    {10,21}, {10,18}, {44,166}, {10,26}, {44,169}, {44,165}, {10,15}, {10,28}, 
    {10,19}, {10,12}, {10,8}, {44,175}, {12,151}, {12,129}, {12,145}, {12,150}, 
    {47,144}, {47,143}, {47,148}, {47,149}, {12,132}, {12,141}, {12,154}, 
    {12,142}, {47,137}, {12,135}, {47,138}, {12,156}, {12,147}, {12,126}, 
    {12,127}, {47,157}, {15,127}, {12,140}, {12,139}, {47,139}, {12,143}, 
    {51,27}, {12,134}, {47,150}, {12,149}, {12,148}, {12,138}, {47,142}, 
    {47,151}, {47,136}, {12,137}, {47,152}, {12,146}, {47,140}, {12,128}, 
    {12,131}, {12,130}, {47,153}, {12,144}, {12,133}, {12,153}, {12,155}, 
    {47,147}, {12,152}, {47,155}, {47,141}, {47,156}, {47,145}, {47,154}, 
    {47,146}, {12,136}, {51,35}, {51,23}, {15,115}, {51,28}, {51,34}, {15,114}, 
    {51,21}, {15,118}, {51,25}, {51,18}, {15,116}, {51,22}, {15,121}, {15,126}, 
    {51,39}, {51,17}, {15,122}, {51,38}, {15,117}, {51,24}, {15,113}, {51,19}, 
    {51,15}, {51,29}, {51,16}, {51,14}, {51,26}, {51,32}, {51,33}, {15,119}, 
    {55,27}, {51,30}, {51,13}, {51,31}, {15,129}, {51,36}, {15,128}, {51,12}, 
    {51,20}, {51,37}, {15,124}, {15,123}, {15,120}, {15,125}, {15,111}, 
    {15,112}, {18,136}, {55,30}, {55,23}, {18,133}, {55,31}, {55,21}, {55,24}, 
    {18,132}, {55,25}, {18,135}, {55,29}, {55,19}, {55,18}, {18,137}, {18,138}, 
    {18,134}, {18,139}, {55,17}, {55,28}, {55,26}, {55,20}, {18,131}, {55,22}, 
    {21,138}, {21,132}, {59,119}, {21,141}, {59,125}, {59,128}, {21,134}, 
    {21,135}, {59,122}, {21,133}, {59,131}, {21,139}, {59,132}, {59,127}, 
    {21,140}, {59,116}, {21,137}, {59,117}, {59,126}, {59,124}, {59,120}, 
    {21,136}, {59,118}, {21,142}, {59,123}, {59,130}, {59,121}, {59,129}, 
    {24,118}, {24,115}, {24,116}, {63,185}, {63,177}, {24,114}, {24,111}, 
    {63,178}, {24,113}, {63,181}, {63,183}, {63,190}, {63,189}, {63,184}, 
    {24,110}, {63,176}, {24,108}, {63,179}, {24,119}, {63,182}, {63,186}, 
    {24,112}, {24,109}, {63,188}, {63,187}, {24,117}, {63,180}, {26,184}, 
    {67,172}, {67,169}, {26,185}, {26,183}, {67,176}, {67,173}, {67,166}, 
    {26,182}, {26,186}, {67,167}, {26,181}, {26,189}, {67,170}, {67,171}, 
    {26,187}, {26,188}, {67,177}, {67,174}, {67,175}, {29,106}, {71,184}, 
    {29,103}, {71,181}, {29,105}, {29,107}, {71,182}, {29,104}, {71,183}, 
    {71,180}, {31,54}, {75,125}, {31,55}, {75,123}, {75,124}, {75,122}, 
    {75,121}, {75,126}, {31,53}, {78,120}, {78,121}, {67,168}, {78,119}, 
    {31,56}, {34,31}, {80,171}, {80,172}, {35,151}, {35,152}, {84,129}, 
    {84,128}, {36,44}, {36,43}, {86,144}, {3,9}, {3,33}, {3,121}, {3,120}, 
    {4,19}, {4,18}, {4,188}, {4,187}, {4,189}, {4,186}, {6,10}, {6,9}, 
    {6,11}, {7,160}, {7,159}, {7,161}, {10,30}, {1,25}, {47,158}, {1,26}, 
    {15,130}, {1,28}, {1,27}, {55,32}, {1,29}, {24,120}, {1,30}, {3,10}, 
    {3,122}, {4,190}, {7,162}, {3,11}, {3,125}, {3,123}, {3,124}, {5,0}, 
    {6,12}, {7,164}, {7,163}, {7,165}, {12,157}, {29,108}, {40,14}, {4,20}, 
    {4,21}, {5,1}, {41,42}, {10,32}, {10,31}, {47,160}, {47,159}, {15,131}, 
    {3,126}, {40,21}, {42,141}, {10,33}, {12,160}, {12,158}, {12,159}, 
    {29,109}, {4,22}, {5,2}, {40,124}, {6,13}, {6,14}, {41,43}, {42,142}, 
    {7,166}, {42,143}, {47,162}, {47,163}, {12,163}, {47,164}, {47,161}, 
    {12,164}, {12,162}, {12,161}, {47,165}, {51,40}, {55,33}, {59,133}, 
    {64,0}, {26,190}, {29,110}, {71,185}, {3,12}, {3,26}, {15,132}, {18,141}, 
    {24,121}, {40,2}, {3,127}, {4,25}, {4,23}, {4,24}, {7,167}, {3,13}, 
    {3,14}, {3,34}, {3,128}, {3,129}, {3,130}, {40,126}, {4,26}, {40,43}, 
    {5,6}, {40,127}, {5,4}, {5,5}, {40,125}, {5,7}, {5,3}, {41,44}, {9,172}, 
    {41,45}, {41,46}, {6,16}, {6,15}, {6,19}, {6,17}, {6,18}, {7,173}, 
    {7,172}, {42,148}, {42,145}, {42,146}, {42,144}, {7,174}, {7,170}, 
    {7,169}, {7,171}, {7,168}, {7,175}, {10,35}, {44,176}, {42,147}, {10,40}, 
    {44,177}, {10,36}, {10,39}, {10,38}, {10,37}, {10,34}, {47,168}, {12,167}, 
    {47,171}, {12,165}, {47,167}, {12,168}, {12,166}, {12,169}, {47,169}, 
    {47,170}, {47,166}, {47,172}, {18,145}, {15,133}, {51,41}, {51,43}, 
    {51,42}, {51,44}, {15,134}, {18,142}, {18,143}, {18,144}, {21,144}, 
    {59,135}, {59,134}, {59,136}, {59,137}, {21,145}, {21,143}, {64,1}, 
    {64,2}, {24,123}, {24,122}, {66,26}, {27,0}, {27,1}, {27,2}, {27,4}, 
    {67,178}, {67,179}, {27,3}, {29,111}, {29,112}, {80,173}, {84,131}, 
    {84,130}, {86,145}, {3,15}, {4,28}, {4,27}, {5,8}, {40,128}, {6,21}, 
    {6,22}, {6,20}, {6,23}, {41,47}, {41,48}, {7,177}, {42,149}, {7,176}, 
    {44,178}, {10,44}, {44,179}, {10,43}, {10,41}, {10,42}, {47,173}, {15,135}, 
    {51,46}, {15,138}, {51,45}, {16,165}, {15,137}, {15,136}, {18,148}, 
    {18,147}, {18,146}, {21,146}, {21,149}, {21,150}, {21,148}, {21,147}, 
    {64,3}, {64,4}, {67,181}, {27,5}, {67,180}, {29,113}, {75,127}, {31,57}, 
    {80,174}, {35,21}, {3,35}, {3,131}, {40,22}, {3,132}, {3,133}, {4,29}, 
    {4,30}, {5,9}, {41,49}, {42,150}, {42,151}, {10,45}, {47,174}, {15,140}, 
    {15,139}, {55,35}, {55,34}, {3,16}, {3,134}, {4,31}, {15,141}, {40,3}, 
    {40,44}, {4,32}, {40,130}, {5,11}, {5,10}, {40,129}, {6,24}, {12,170}, 
    {51,47}, {21,151}, {64,5}, {24,124}, {71,186}, {82,189}, {3,135}, {42,152}, 
    {44,180}, {15,144}, {15,142}, {15,143}, {3,17}, {3,36}, {1,141}, {3,138}, 
    {3,137}, {3,136}, {4,35}, {4,34}, {40,45}, {40,131}, {7,181}, {7,178}, 
    {7,180}, {7,179}, {10,46}, {18,149}, {3,18}, {3,139}, {4,37}, {4,36}, 
    {41,50}, {7,182}, {40,23}, {4,39}, {4,38}, {5,12}, {5,13}, {41,51}, 
    {6,25}, {6,26}, {7,183}, {7,184}, {7,185}, {10,47}, {44,181}, {12,171}, 
    {40,4}, {3,140}, {40,132}, {41,52}, {41,53}, {42,153}, {42,154}, {44,183}, 
    {44,182}, {44,185}, {44,184}, {10,48}, {51,48}, {12,173}, {47,175}, 
    {12,172}, {55,36}, {18,150}, {55,37}, {64,6}, {24,125}, {27,6}, {80,175}, 
    {40,24}, {4,40}, {15,145}, {3,19}, {3,37}, {3,142}, {3,141}, {3,143}, 
    {7,187}, {7,186}, {7,188}, {10,49}, {12,174}, {71,187}, {33,15}, {3,38}, 
    {4,42}, {4,57}, {4,52}, {4,47}, {4,46}, {4,53}, {4,51}, {4,44}, {4,58}, 
    {4,45}, {4,41}, {4,56}, {4,55}, {4,54}, {4,43}, {4,50}, {4,49}, {4,59}, 
    {4,48}, {5,19}, {5,25}, {5,21}, {5,27}, {40,133}, {5,24}, {5,14}, {5,17}, 
    {5,20}, {5,16}, {5,23}, {5,26}, {5,15}, {5,18}, {5,22}, {5,28}, {41,62}, 
    {41,59}, {6,39}, {41,60}, {6,27}, {6,29}, {6,53}, {6,48}, {41,101}, 
    {41,61}, {6,31}, {6,33}, {41,100}, {6,40}, {41,56}, {6,52}, {6,54}, 
    {6,28}, {6,45}, {41,54}, {6,51}, {6,36}, {6,46}, {6,47}, {41,55}, {6,44}, 
    {6,42}, {6,43}, {6,49}, {41,97}, {6,30}, {6,50}, {41,99}, {6,38}, {6,35}, 
    {41,58}, {6,34}, {41,102}, {6,37}, {6,41}, {6,32}, {41,98}, {41,57}, 
    {42,171}, {42,170}, {8,15}, {42,173}, {42,175}, {42,164}, {42,167}, 
    {42,174}, {8,16}, {42,161}, {42,165}, {42,169}, {45,9}, {8,11}, {45,13}, 
    {7,189}, {42,166}, {7,190}, {8,12}, {8,5}, {8,1}, {42,162}, {8,4}, 
    {8,9}, {8,18}, {42,163}, {42,156}, {8,3}, {42,157}, {42,159}, {8,6}, 
    {8,8}, {42,155}, {42,160}, {8,17}, {8,13}, {42,168}, {8,19}, {8,10}, 
    {42,158}, {8,7}, {8,2}, {8,0}, {42,172}, {8,14}, {45,8}, {44,187}, 
    {45,11}, {44,189}, {10,56}, {10,104}, {10,52}, {10,103}, {10,61}, {10,100}, 
    {10,50}, {44,188}, {45,3}, {10,99}, {45,15}, {10,101}, {10,57}, {45,5}, 
    {45,2}, {10,55}, {44,186}, {10,102}, {45,10}, {10,60}, {45,12}, {10,105}, 
    {10,51}, {10,62}, {10,59}, {45,0}, {10,97}, {45,6}, {45,7}, {10,58}, 
    {10,98}, {10,54}, {10,53}, {44,190}, {45,4}, {45,14}, {48,4}, {12,187}, 
    {47,177}, {47,182}, {12,180}, {13,0}, {47,180}, {12,175}, {12,185}, 
    {12,190}, {47,187}, {12,186}, {12,189}, {47,190}, {12,181}, {47,181}, 
    {48,2}, {48,3}, {12,183}, {48,1}, {12,179}, {13,3}, {47,184}, {12,177}, 
    {48,6}, {47,185}, {47,189}, {12,182}, {13,2}, {47,186}, {12,188}, {48,0}, 
    {48,7}, {51,97}, {48,5}, {13,4}, {12,176}, {47,188}, {47,179}, {12,184}, 
    {47,178}, {47,183}, {47,176}, {13,1}, {51,55}, {15,164}, {51,103}, 
    {15,162}, {15,159}, {51,60}, {15,155}, {51,98}, {15,166}, {51,54}, 
    {51,59}, {51,58}, {12,178}, {15,161}, {51,101}, {51,104}, {51,51}, 
    {18,168}, {51,105}, {15,167}, {15,153}, {15,150}, {51,62}, {15,147}, 
    {51,102}, {15,154}, {51,106}, {51,52}, {51,100}, {15,157}, {51,53}, 
    {51,56}, {51,61}, {15,158}, {15,156}, {15,168}, {15,163}, {15,151}, 
    {16,146}, {15,152}, {51,57}, {15,165}, {15,160}, {51,99}, {15,149}, 
    {15,148}, {51,49}, {51,50}, {55,42}, {18,151}, {18,154}, {55,53}, {18,174}, 
    {55,56}, {18,152}, {55,49}, {18,158}, {18,164}, {20,125}, {18,162}, 
    {55,46}, {18,175}, {18,155}, {18,163}, {55,54}, {59,151}, {55,59}, 
    {55,47}, {55,38}, {55,51}, {55,45}, {18,161}, {55,57}, {18,157}, {18,177}, 
    {18,170}, {18,159}, {18,156}, {18,167}, {55,58}, {55,44}, {55,50}, 
    {55,52}, {55,40}, {55,55}, {18,153}, {55,39}, {18,160}, {18,176}, {18,172}, 
    {55,41}, {18,166}, {18,173}, {18,169}, {18,165}, {55,48}, {18,171}, 
    {59,149}, {59,145}, {59,160}, {59,138}, {59,147}, {21,165}, {21,166}, 
    {21,158}, {59,156}, {21,168}, {59,143}, {59,142}, {59,140}, {59,158}, 
    {21,156}, {59,152}, {59,141}, {21,159}, {59,150}, {21,154}, {59,146}, 
    {59,153}, {59,155}, {59,159}, {21,163}, {59,139}, {21,157}, {59,144}, 
    {21,152}, {21,164}, {59,154}, {21,160}, {21,161}, {21,167}, {21,155}, 
    {1,31}, {21,153}, {59,148}, {21,162}, {59,157}, {24,141}, {24,136}, 
    {64,21}, {64,17}, {64,11}, {24,130}, {24,126}, {64,20}, {24,127}, {64,14}, 
    {64,16}, {64,19}, {24,132}, {24,139}, {24,133}, {64,9}, {24,134}, {24,135}, 
    {64,12}, {24,140}, {64,10}, {64,15}, {24,131}, {64,8}, {24,137}, {24,129}, 
    {24,128}, {64,13}, {64,18}, {24,138}, {64,7}, {27,13}, {68,7}, {68,4}, 
    {27,7}, {27,19}, {27,20}, {27,10}, {68,2}, {27,12}, {67,185}, {27,18}, 
    {68,6}, {27,9}, {68,8}, {27,8}, {68,3}, {68,5}, {27,11}, {68,1}, {67,186}, 
    {67,183}, {55,43}, {67,189}, {67,182}, {67,188}, {67,187}, {67,184}, 
    {27,15}, {27,14}, {27,16}, {67,190}, {29,114}, {68,0}, {72,5}, {71,189}, 
    {29,126}, {72,2}, {29,118}, {29,122}, {72,1}, {29,121}, {29,117}, {29,120}, 
    {29,115}, {29,125}, {72,3}, {71,190}, {29,124}, {71,188}, {29,123}, 
    {72,4}, {72,0}, {27,17}, {29,127}, {72,6}, {29,119}, {29,116}, {75,134}, 
    {75,136}, {31,59}, {75,137}, {75,135}, {75,133}, {75,132}, {31,61}, 
    {75,131}, {31,62}, {75,129}, {75,130}, {31,58}, {31,97}, {31,60}, {75,128}, 
    {33,16}, {78,124}, {78,122}, {78,127}, {78,125}, {78,123}, {78,126}, 
    {34,32}, {80,178}, {80,179}, {35,22}, {80,180}, {80,176}, {80,181}, 
    {80,177}, {33,17}, {82,190}, {83,1}, {35,25}, {83,0}, {35,24}, {35,23}, 
    {35,26}, {84,133}, {84,134}, {35,154}, {35,153}, {35,155}, {84,132}, 
    {85,152}, {85,151}, {36,45}, {36,47}, {36,46}, {85,153}, {36,136}, 
    {87,102}, {36,177}, {87,101}, {87,174}, {40,9}, {4,61}, {4,60}, {5,31}, 
    {5,30}, {40,135}, {5,29}, {40,134}, {6,57}, {41,105}, {41,104}, {6,55}, 
    {6,58}, {41,103}, {6,56}, {42,176}, {42,177}, {8,20}, {10,106}, {48,8}, 
    {48,9}, {13,5}, {13,6}, {51,108}, {15,169}, {15,171}, {51,107}, {15,170}, 
    {55,60}, {18,178}, {21,169}, {21,170}, {59,161}, {24,143}, {24,142}, 
    {68,9}, {72,8}, {72,7}, {88,43}, {3,39}, {40,25}, {40,46}, {40,47}, 
    {5,34}, {5,38}, {40,137}, {5,36}, {5,35}, {40,136}, {5,37}, {5,33}, 
    {5,32}, {40,138}, {6,102}, {6,99}, {6,61}, {41,106}, {41,107}, {6,97}, 
    {41,109}, {6,59}, {41,110}, {41,108}, {6,62}, {6,98}, {6,101}, {6,100}, 
    {6,60}, {41,111}, {8,25}, {42,190}, {8,27}, {8,26}, {43,0}, {8,24}, 
    {8,23}, {42,181}, {42,180}, {42,178}, {42,183}, {42,182}, {42,179}, 
    {42,188}, {42,189}, {42,186}, {42,184}, {8,22}, {42,187}, {8,28}, {43,1}, 
    {42,185}, {10,107}, {8,21}, {10,108}, {45,20}, {45,26}, {10,114}, {45,24}, 
    {45,30}, {45,21}, {45,25}, {45,27}, {45,29}, {45,23}, {45,22}, {45,17}, 
    {45,18}, {10,109}, {10,111}, {10,110}, {45,19}, {45,28}, {10,113}, 
    {45,16}, {48,19}, {48,18}, {48,23}, {48,14}, {48,17}, {48,16}, {48,20}, 
    {48,24}, {13,7}, {13,10}, {48,15}, {48,21}, {13,9}, {48,10}, {10,112}, 
    {51,122}, {48,22}, {48,13}, {13,8}, {48,12}, {51,113}, {15,172}, {15,176}, 
    {51,129}, {51,111}, {51,125}, {15,177}, {51,127}, {51,133}, {51,137}, 
    {51,128}, {51,116}, {51,124}, {51,138}, {51,136}, {51,126}, {51,121}, 
    {51,114}, {55,102}, {51,112}, {15,181}, {51,119}, {15,182}, {15,178}, 
    {51,109}, {51,131}, {51,117}, {51,115}, {51,134}, {15,179}, {51,140}, 
    {15,173}, {15,175}, {51,123}, {51,118}, {13,11}, {15,174}, {51,120}, 
    {51,135}, {51,139}, {51,130}, {51,132}, {51,110}, {55,97}, {55,106}, 
    {55,105}, {18,186}, {55,98}, {18,187}, {18,185}, {55,100}, {18,182}, 
    {55,104}, {55,99}, {55,101}, {55,61}, {18,180}, {55,114}, {55,113}, 
    {55,110}, {18,179}, {18,183}, {18,184}, {48,11}, {55,107}, {18,181}, 
    {15,180}, {55,109}, {55,62}, {55,112}, {55,111}, {55,115}, {59,175}, 
    {55,108}, {55,103}, {59,167}, {21,180}, {21,183}, {21,178}, {59,166}, 
    {59,170}, {59,165}, {21,172}, {21,182}, {59,162}, {21,176}, {59,169}, 
    {21,174}, {21,173}, {59,172}, {21,175}, {59,174}, {59,171}, {21,171}, 
    {21,181}, {59,176}, {59,164}, {59,173}, {59,163}, {21,177}, {21,179}, 
    {59,168}, {59,177}, {64,29}, {24,144}, {64,35}, {24,149}, {64,31}, 
    {64,38}, {64,23}, {24,151}, {24,145}, {64,28}, {27,21}, {64,27}, {64,36}, 
    {24,146}, {24,150}, {64,26}, {64,32}, {64,37}, {64,22}, {24,148}, {64,30}, 
    {64,34}, {64,40}, {64,24}, {64,33}, {24,147}, {64,39}, {64,25}, {27,25}, 
    {68,11}, {27,23}, {27,22}, {68,13}, {68,18}, {68,14}, {68,17}, {27,28}, 
    {29,101}, {27,27}, {68,10}, {68,16}, {27,26}, {68,15}, {68,12}, {27,24}, 
    {72,13}, {72,15}, {72,10}, {29,129}, {72,12}, {29,128}, {72,14}, {29,131}, 
    {72,16}, {29,130}, {72,9}, {72,11}, {31,101}, {75,140}, {31,100}, {75,141}, 
    {31,99}, {75,139}, {31,98}, {75,138}, {33,19}, {33,18}, {80,182}, {80,184}, 
    {80,183}, {34,33}, {34,34}, {34,35}, {83,2}, {35,27}, {86,147}, {86,146}, 
    {36,178}, {3,40}, {3,144}, {6,103}, {45,31}, {18,188}, {18,189}, {59,178}, 
    {24,152}, {64,41}, {68,19}, {40,26}, {41,112}, {43,2}, {45,32}, {48,25}, 
    {13,12}, {80,185}, {35,156}, {3,41}, {4,62}, {40,48}, {5,39}, {5,40}, 
    {8,29}, {15,183}, {24,154}, {24,155}, {24,153}, {3,42}, {3,145}, {3,147}, 
    {3,146}, {40,27}, {3,148}, {4,97}, {40,49}, {4,98}, {5,41}, {5,42}, 
    {40,139}, {6,104}, {41,113}, {8,33}, {43,3}, {8,31}, {8,32}, {8,30}, 
    {48,26}, {10,118}, {10,117}, {10,119}, {10,116}, {45,33}, {8,34}, {10,115}, 
    {13,13}, {13,14}, {13,15}, {51,141}, {18,190}, {55,116}, {15,184}, 
    {21,184}, {24,157}, {24,156}, {64,42}, {27,29}, {29,132}, {78,128}, 
    {85,154}, {86,148}, {3,43}, {4,99}, {4,100}, {40,145}, {5,44}, {5,47}, 
    {40,143}, {40,141}, {5,46}, {40,144}, {40,146}, {40,140}, {5,49}, {5,48}, 
    {5,45}, {5,43}, {40,142}, {6,115}, {6,112}, {41,118}, {41,121}, {41,120}, 
    {6,106}, {6,114}, {6,111}, {41,117}, {41,115}, {6,110}, {6,105}, {6,108}, 
    {41,116}, {41,123}, {41,119}, {6,109}, {6,113}, {6,116}, {41,114}, 
    {41,122}, {6,107}, {8,39}, {8,47}, {43,15}, {43,8}, {8,48}, {43,19}, 
    {43,4}, {43,11}, {8,38}, {43,5}, {8,36}, {43,12}, {43,16}, {8,35}, 
    {43,17}, {43,10}, {43,13}, {8,50}, {8,41}, {43,20}, {43,18}, {8,46}, 
    {8,44}, {43,9}, {8,43}, {43,7}, {43,6}, {8,42}, {8,40}, {8,49}, {8,45}, 
    {8,37}, {43,14}, {10,121}, {10,128}, {45,47}, {10,120}, {45,39}, {45,35}, 
    {45,51}, {45,34}, {10,123}, {45,44}, {10,126}, {10,129}, {10,124}, 
    {45,48}, {10,127}, {13,22}, {45,54}, {45,36}, {45,38}, {45,45}, {45,49}, 
    {45,53}, {45,50}, {45,43}, {45,46}, {45,40}, {10,131}, {45,42}, {45,41}, 
    {45,52}, {10,122}, {45,37}, {10,130}, {10,125}, {13,28}, {48,34}, {13,27}, 
    {48,32}, {13,16}, {13,21}, {48,31}, {48,28}, {48,33}, {13,17}, {48,27}, 
    {13,20}, {13,18}, {48,35}, {13,19}, {13,23}, {13,24}, {13,26}, {13,25}, 
    {48,29}, {48,30}, {48,36}, {51,148}, {15,185}, {55,130}, {51,147}, 
    {51,166}, {16,0}, {51,164}, {15,190}, {15,186}, {51,173}, {51,157}, 
    {51,160}, {16,3}, {51,170}, {51,162}, {15,187}, {16,4}, {51,167}, {51,165}, 
    {51,150}, {51,171}, {51,159}, {51,154}, {51,144}, {51,172}, {51,156}, 
    {51,143}, {16,2}, {51,161}, {51,174}, {51,158}, {51,146}, {51,151}, 
    {51,142}, {16,1}, {51,155}, {51,152}, {15,188}, {51,145}, {51,169}, 
    {15,189}, {51,153}, {51,149}, {51,168}, {19,0}, {55,123}, {55,120}, 
    {55,137}, {55,125}, {55,138}, {19,2}, {55,134}, {55,131}, {55,132}, 
    {55,135}, {55,139}, {51,163}, {55,141}, {60,7}, {19,3}, {55,142}, {55,118}, 
    {55,128}, {55,133}, {19,1}, {19,4}, {55,140}, {55,143}, {55,122}, {55,119}, 
    {55,121}, {55,126}, {55,124}, {19,5}, {55,136}, {55,127}, {55,129}, 
    {55,117}, {59,186}, {59,184}, {22,2}, {22,0}, {60,3}, {59,185}, {60,4}, 
    {60,0}, {59,183}, {60,6}, {59,182}, {59,189}, {21,190}, {21,189}, {21,188}, 
    {59,187}, {60,1}, {21,185}, {22,1}, {59,180}, {59,190}, {59,179}, {59,188}, 
    {21,186}, {60,2}, {59,181}, {21,187}, {60,5}, {64,46}, {24,162}, {24,161}, 
    {24,163}, {64,58}, {64,48}, {64,54}, {64,43}, {64,57}, {64,56}, {64,60}, 
    {64,53}, {24,158}, {64,52}, {24,164}, {64,45}, {24,159}, {64,59}, {24,160}, 
    {64,47}, {64,50}, {64,55}, {64,49}, {64,44}, {64,51}, {68,21}, {27,33}, 
    {68,24}, {68,23}, {68,26}, {68,28}, {27,31}, {68,22}, {68,20}, {68,29}, 
    {68,27}, {68,25}, {68,31}, {68,30}, {27,35}, {27,30}, {27,32}, {27,34}, 
    {68,32}, {72,23}, {72,22}, {72,21}, {72,24}, {72,17}, {72,18}, {72,26}, 
    {72,19}, {29,133}, {72,28}, {72,27}, {72,20}, {75,145}, {31,104}, {75,143}, 
    {75,148}, {75,147}, {72,25}, {31,103}, {75,146}, {75,142}, {75,150}, 
    {75,149}, {31,102}, {75,144}, {29,134}, {33,20}, {78,129}, {80,186}, 
    {80,187}, {80,188}, {35,28}, {35,29}, {83,3}, {84,136}, {84,135}, {85,155}, 
    {85,156}, {86,149}, {87,103}, {3,44}, {3,45}, {3,46}, {3,149}, {4,101}, 
    {40,147}, {5,50}, {5,51}, {6,119}, {6,120}, {6,118}, {6,117}, {8,51}, 
    {43,21}, {8,53}, {8,52}, {43,22}, {10,132}, {13,29}, {48,37}, {51,176}, 
    {16,5}, {19,7}, {51,175}, {19,6}, {24,165}, {64,61}, {29,135}, {31,105}, 
    {75,151}, {35,30}, {36,48}, {40,50}, {4,102}, {40,51}, {5,54}, {5,52}, 
    {5,53}, {5,55}, {6,122}, {6,121}, {41,124}, {6,123}, {41,125}, {43,23}, 
    {43,24}, {8,54}, {8,56}, {8,58}, {8,55}, {8,59}, {8,57}, {10,136}, 
    {10,133}, {10,135}, {10,137}, {10,134}, {48,38}, {45,55}, {48,40}, 
    {48,39}, {13,35}, {13,31}, {13,32}, {13,34}, {13,36}, {13,33}, {13,38}, 
    {13,37}, {16,10}, {51,178}, {51,177}, {16,9}, {16,8}, {16,7}, {16,11}, 
    {16,6}, {55,149}, {55,146}, {19,9}, {55,145}, {55,150}, {19,11}, {55,148}, 
    {19,8}, {19,10}, {55,147}, {60,8}, {60,9}, {60,10}, {24,166}, {24,174}, 
    {64,62}, {24,168}, {24,172}, {64,97}, {24,173}, {24,169}, {24,170}, 
    {24,167}, {24,171}, {27,38}, {55,144}, {27,39}, {27,37}, {27,36}, {72,29}, 
    {29,136}, {75,152}, {75,153}, {34,36}, {35,31}, {3,47}, {5,56}, {10,138}, 
    {48,41}, {13,39}, {16,14}, {16,13}, {16,12}, {19,12}, {19,13}, {55,151}, 
    {24,175}, {29,137}, {3,48}, {40,28}, {3,150}, {40,52}, {40,148}, {5,57}, 
    {8,60}, {60,11}, {3,49}, {3,151}, {40,149}, {41,126}, {41,127}, {6,124}, 
    {55,152}, {19,14}, {60,12}, {31,106}, {3,50}, {3,104}, {3,152}, {40,53}, 
    {4,103}, {6,128}, {6,127}, {6,125}, {6,126}, {43,25}, {8,62}, {8,97}, 
    {43,26}, {8,61}, {10,142}, {45,56}, {10,141}, {10,139}, {10,140}, {13,42}, 
    {13,40}, {48,43}, {13,41}, {48,42}, {13,30}, {51,179}, {16,16}, {16,17}, 
    {16,15}, {24,176}, {64,98}, {27,40}, {27,41}, {68,33}, {31,107}, {78,130}, 
    {78,131}, {35,157}, {87,104}, {40,11}, {3,153}, {3,51}, {40,55}, {40,54}, 
    {5,58}, {40,151}, {40,152}, {40,150}, {40,153}, {41,135}, {41,130}, 
    {41,132}, {41,134}, {41,131}, {6,132}, {41,128}, {41,129}, {6,129}, 
    {6,130}, {41,133}, {41,136}, {6,131}, {41,137}, {43,40}, {43,34}, {43,29}, 
    {8,99}, {43,37}, {43,35}, {43,28}, {43,41}, {43,44}, {43,39}, {43,32}, 
    {8,101}, {43,38}, {8,102}, {43,33}, {43,36}, {43,27}, {43,31}, {43,43}, 
    {8,103}, {8,104}, {43,30}, {43,42}, {8,98}, {8,100}, {45,107}, {45,100}, 
    {45,106}, {45,99}, {45,101}, {45,61}, {45,59}, {45,108}, {45,105}, 
    {45,57}, {10,144}, {45,103}, {45,104}, {45,102}, {45,60}, {45,58}, 
    {10,143}, {45,98}, {45,62}, {45,97}, {45,109}, {13,47}, {13,46}, {48,44}, 
    {13,43}, {48,46}, {13,48}, {48,47}, {13,51}, {13,49}, {48,48}, {45,110}, 
    {48,50}, {13,45}, {13,44}, {48,45}, {48,49}, {13,50}, {16,19}, {16,18}, 
    {51,181}, {51,185}, {51,187}, {16,20}, {51,190}, {16,24}, {52,1}, {16,26}, 
    {16,22}, {16,30}, {16,27}, {51,183}, {16,21}, {51,182}, {51,180}, {52,3}, 
    {51,184}, {16,23}, {52,2}, {16,28}, {51,189}, {51,188}, {16,29}, {51,186}, 
    {16,25}, {52,4}, {52,0}, {55,167}, {55,174}, {55,163}, {19,17}, {55,159}, 
    {55,175}, {55,153}, {55,172}, {55,170}, {55,164}, {55,173}, {55,166}, 
    {55,158}, {55,176}, {55,156}, {55,169}, {55,154}, {55,177}, {19,18}, 
    {55,171}, {60,15}, {55,157}, {19,15}, {55,161}, {19,16}, {55,160}, 
    {55,165}, {55,162}, {55,168}, {60,19}, {60,22}, {60,14}, {60,16}, {60,21}, 
    {60,20}, {22,3}, {55,155}, {60,18}, {22,4}, {60,13}, {60,17}, {64,105}, 
    {64,112}, {64,103}, {64,110}, {64,101}, {64,109}, {64,113}, {64,100}, 
    {64,104}, {64,99}, {24,177}, {64,102}, {24,178}, {64,108}, {64,107}, 
    {64,106}, {64,111}, {68,37}, {68,39}, {27,43}, {68,40}, {68,35}, {68,34}, 
    {68,44}, {68,42}, {27,42}, {68,45}, {68,36}, {68,41}, {68,43}, {68,38}, 
    {72,33}, {72,38}, {72,32}, {72,37}, {72,30}, {72,40}, {72,36}, {72,41}, 
    {72,35}, {72,31}, {72,39}, {72,42}, {72,34}, {75,154}, {31,111}, {31,109}, 
    {31,108}, {31,110}, {78,132}, {80,50}, {80,189}, {83,4}, {83,5}, {35,32}, 
    {84,137}, {35,158}, {84,138}, {85,158}, {36,50}, {36,49}, {85,157}, 
    {36,137}, {86,150}, {3,52}, {5,59}, {40,154}, {41,138}, {7,117}, {16,31}, 
    {3,53}, {4,106}, {4,105}, {4,104}, {6,133}, {13,52}, {60,23}, {3,54}, 
    {3,55}, {3,56}, {3,154}, {10,145}, {45,111}, {19,19}, {3,57}, {40,29}, 
    {4,107}, {4,108}, {40,56}, {5,60}, {41,139}, {6,134}, {41,140}, {8,110}, 
    {43,46}, {8,108}, {8,107}, {43,45}, {8,105}, {43,47}, {8,106}, {8,109}, 
    {10,146}, {10,148}, {45,115}, {45,112}, {45,113}, {45,114}, {45,116}, 
    {10,147}, {48,52}, {48,51}, {13,54}, {13,53}, {16,34}, {52,6}, {16,33}, 
    {16,35}, {16,32}, {19,21}, {52,5}, {19,22}, {55,179}, {19,23}, {55,178}, 
    {19,20}, {60,26}, {60,28}, {22,5}, {60,27}, {60,25}, {60,24}, {64,116}, 
    {24,183}, {24,181}, {24,182}, {64,114}, {64,115}, {24,179}, {68,49}, 
    {68,47}, {27,45}, {68,48}, {27,46}, {27,44}, {24,180}, {72,45}, {72,43}, 
    {72,44}, {68,46}, {75,156}, {31,112}, {75,155}, {78,133}, {78,134}, 
    {72,46}, {80,190}, {3,58}, {4,109}, {5,62}, {40,155}, {5,61}, {8,111}, 
    {22,6}, {3,155}, {4,110}, {10,149}, {19,24}, {40,57}, {40,58}, {40,156}, 
    {6,136}, {41,144}, {41,142}, {6,137}, {41,141}, {41,143}, {41,145}, 
    {6,135}, {8,115}, {8,116}, {8,113}, {8,112}, {45,120}, {8,114}, {10,150}, 
    {45,119}, {45,121}, {45,118}, {45,122}, {10,151}, {13,57}, {48,53}, 
    {48,55}, {13,55}, {48,56}, {13,56}, {48,54}, {45,117}, {52,7}, {52,10}, 
    {52,11}, {52,8}, {16,39}, {16,38}, {16,36}, {16,37}, {52,9}, {16,40}, 
    {19,26}, {19,27}, {19,28}, {60,29}, {60,31}, {60,33}, {22,8}, {22,7}, 
    {19,25}, {60,32}, {60,30}, {64,120}, {64,118}, {64,124}, {24,184}, 
    {64,125}, {64,122}, {24,185}, {64,119}, {64,117}, {64,123}, {27,48}, 
    {68,51}, {64,121}, {27,50}, {68,52}, {27,49}, {27,52}, {68,53}, {27,47}, 
    {27,51}, {72,51}, {72,49}, {72,48}, {72,50}, {72,47}, {34,38}, {83,6}, 
    {83,7}, {84,139}, {85,159}, {37,21}, {8,117}, {6,138}, {10,152}, {3,59}, 
    {3,156}, {4,111}, {40,157}, {6,139}, {41,146}, {45,123}, {10,153}, 
    {24,186}, {3,60}, {5,97}, {22,9}, {3,61}, {3,157}, {3,158}, {4,113}, 
    {4,112}, {40,158}, {5,98}, {41,147}, {6,140}, {43,49}, {43,50}, {43,51}, 
    {8,118}, {8,119}, {43,48}, {8,120}, {10,154}, {45,124}, {48,58}, {13,58}, 
    {48,57}, {16,41}, {52,12}, {16,42}, {52,13}, {19,29}, {60,34}, {64,127}, 
    {64,126}, {24,187}, {27,53}, {68,54}, {29,138}, {72,52}, {31,113}, 
    {36,51}, {86,152}, {43,52}, {45,125}, {16,43}, {55,180}, {22,10}, {33,21}, 
    {6,142}, {6,141}, {10,155}, {48,59}, {16,45}, {18,3}, {16,46}, {16,44}, 
    {19,30}, {64,128}, {24,188}, {27,54}, {40,12}, {40,159}, {41,149}, 
    {6,143}, {41,148}, {6,144}, {8,124}, {43,55}, {43,54}, {8,123}, {8,121}, 
    {8,122}, {43,53}, {10,157}, {45,126}, {10,160}, {10,156}, {10,162}, 
    {10,158}, {10,159}, {10,161}, {13,61}, {13,60}, {13,59}, {52,15}, {16,47}, 
    {16,50}, {16,48}, {52,14}, {16,53}, {16,49}, {52,16}, {16,52}, {16,51}, 
    {55,182}, {55,181}, {19,33}, {19,31}, {19,32}, {55,183}, {22,11}, {60,36}, 
    {22,12}, {60,35}, {68,55}, {27,56}, {64,129}, {27,55}, {24,189}, {75,158}, 
    {72,53}, {31,114}, {75,157}, {81,0}, {83,8}, {83,9}, {3,159}, {4,114}, 
    {40,59}, {6,146}, {6,148}, {40,162}, {41,152}, {41,151}, {41,150}, 
    {40,161}, {40,160}, {5,100}, {6,147}, {6,145}, {5,99}, {8,125}, {43,56}, 
    {8,126}, {41,157}, {41,159}, {41,158}, {43,57}, {41,154}, {6,152}, 
    {6,150}, {41,153}, {41,155}, {41,161}, {6,149}, {41,156}, {41,165}, 
    {8,128}, {41,162}, {6,151}, {41,160}, {41,163}, {8,127}, {8,129}, {41,164}, 
    {43,111}, {43,98}, {43,62}, {43,110}, {43,105}, {10,167}, {8,130}, 
    {43,106}, {43,109}, {10,163}, {43,108}, {8,131}, {8,136}, {8,134}, 
    {43,99}, {43,60}, {43,101}, {8,141}, {43,112}, {10,164}, {43,102}, 
    {10,165}, {8,137}, {43,104}, {45,141}, {10,166}, {43,59}, {8,138}, 
    {10,168}, {8,139}, {8,135}, {8,140}, {43,103}, {43,58}, {43,107}, {8,132}, 
    {43,61}, {43,100}, {43,97}, {8,133}, {45,127}, {45,128}, {45,138}, 
    {48,97}, {45,139}, {10,174}, {45,142}, {45,132}, {10,173}, {45,134}, 
    {45,135}, {45,137}, {10,169}, {13,99}, {45,133}, {45,129}, {13,100}, 
    {45,143}, {13,62}, {48,61}, {45,136}, {48,60}, {45,131}, {45,140}, 
    {10,172}, {13,97}, {10,178}, {13,98}, {45,144}, {48,62}, {10,171}, 
    {13,102}, {10,177}, {10,176}, {10,175}, {13,101}, {45,145}, {13,103}, 
    {10,170}, {45,130}, {16,54}, {48,100}, {48,102}, {48,104}, {13,104}, 
    {13,110}, {52,19}, {48,108}, {48,99}, {16,56}, {52,17}, {13,109}, {13,107}, 
    {48,110}, {52,18}, {48,101}, {13,108}, {48,105}, {13,111}, {48,107}, 
    {13,106}, {48,106}, {48,109}, {48,103}, {13,105}, {16,57}, {48,98}, 
    {16,55}, {16,58}, {52,21}, {52,30}, {19,36}, {16,60}, {16,99}, {19,37}, 
    {52,32}, {16,106}, {55,185}, {52,22}, {16,98}, {16,101}, {16,62}, {52,20}, 
    {52,34}, {52,37}, {56,9}, {52,35}, {55,189}, {16,97}, {16,104}, {16,108}, 
    {52,29}, {55,184}, {52,33}, {16,59}, {55,186}, {52,36}, {55,188}, {52,25}, 
    {19,34}, {52,23}, {52,24}, {16,103}, {16,102}, {52,27}, {16,107}, {52,31}, 
    {16,100}, {52,28}, {16,105}, {19,38}, {19,35}, {55,187}, {52,26}, {16,61}, 
    {19,43}, {19,47}, {56,0}, {22,17}, {19,45}, {56,4}, {19,49}, {60,37}, 
    {56,6}, {22,19}, {19,41}, {19,44}, {56,7}, {56,8}, {56,14}, {19,51}, 
    {22,20}, {56,10}, {56,15}, {56,3}, {22,30}, {22,21}, {19,50}, {56,1}, 
    {56,16}, {22,29}, {19,48}, {22,14}, {56,13}, {19,52}, {56,5}, {55,190}, 
    {19,42}, {56,2}, {56,11}, {22,13}, {22,18}, {19,39}, {56,12}, {22,16}, 
    {19,40}, {22,28}, {64,131}, {60,48}, {60,40}, {64,130}, {60,44}, {60,46}, 
    {60,43}, {22,27}, {60,42}, {22,31}, {64,146}, {22,26}, {25,0}, {60,49}, 
    {64,132}, {22,24}, {60,41}, {60,45}, {24,190}, {22,15}, {60,38}, {60,39}, 
    {25,1}, {22,23}, {22,25}, {22,22}, {60,47}, {64,136}, {64,137}, {64,142}, 
    {27,61}, {64,149}, {25,7}, {25,6}, {64,144}, {27,60}, {64,133}, {25,5}, 
    {64,148}, {25,3}, {25,4}, {64,145}, {68,106}, {27,58}, {19,46}, {64,147}, 
    {27,99}, {64,139}, {27,59}, {27,98}, {64,134}, {64,138}, {64,135}, 
    {64,141}, {25,8}, {27,57}, {25,2}, {68,58}, {64,143}, {27,97}, {27,100}, 
    {64,140}, {27,62}, {68,57}, {68,62}, {29,142}, {68,56}, {72,99}, {68,105}, 
    {27,104}, {27,102}, {29,140}, {68,102}, {68,98}, {27,108}, {72,56}, 
    {27,106}, {68,97}, {72,54}, {68,101}, {68,104}, {68,61}, {27,107}, 
    {27,101}, {72,55}, {29,141}, {68,103}, {27,103}, {27,105}, {68,100}, 
    {27,109}, {68,99}, {68,60}, {68,59}, {29,139}, {68,107}, {72,58}, {75,160}, 
    {29,144}, {72,98}, {72,62}, {75,161}, {29,145}, {72,97}, {72,60}, {31,116}, 
    {75,159}, {72,57}, {72,59}, {31,117}, {29,147}, {31,115}, {29,146}, 
    {31,119}, {72,61}, {29,143}, {78,143}, {78,135}, {75,167}, {78,136}, 
    {75,163}, {33,22}, {75,165}, {75,164}, {31,118}, {75,162}, {75,166}, 
    {78,144}, {78,140}, {78,142}, {78,137}, {78,138}, {78,141}, {78,139}, 
    {34,39}, {34,42}, {34,41}, {34,40}, {35,33}, {83,10}, {35,34}, {81,1}, 
    {35,159}, {84,140}, {35,160}, {36,52}, {36,138}, {86,153}, {86,154}, 
    {86,155}, {88,122}, {3,160}, {40,60}, {4,115}, {5,102}, {5,103}, {5,101}, 
    {5,104}, {6,154}, {6,153}, {43,113}, {8,143}, {8,142}, {48,111}, {16,109}, 
    {16,110}, {19,53}, {60,50}, {22,32}, {22,33}, {60,52}, {60,54}, {60,53}, 
    {64,151}, {64,150}, {25,9}, {64,152}, {68,108}, {27,110}, {29,148}, 
    {31,120}, {33,23}, {31,121}, {3,161}, {41,166}, {43,114}, {8,145}, 
    {8,144}, {8,146}, {10,179}, {45,146}, {45,147}, {48,112}, {13,112}, 
    {16,111}, {19,54}, {56,17}, {3,162}, {3,62}, {3,163}, {40,61}, {4,119}, 
    {4,118}, {4,116}, {4,117}, {5,107}, {40,169}, {40,171}, {5,106}, {40,163}, 
    {40,164}, {40,170}, {40,166}, {40,168}, {5,105}, {40,165}, {40,172}, 
    {40,167}, {6,161}, {6,170}, {6,168}, {41,176}, {41,173}, {41,181}, 
    {6,166}, {41,182}, {6,159}, {41,179}, {6,165}, {41,175}, {41,174}, 
    {6,163}, {41,180}, {6,164}, {8,147}, {6,158}, {41,177}, {41,167}, {6,155}, 
    {6,174}, {41,172}, {41,178}, {6,160}, {6,162}, {41,168}, {41,169}, 
    {41,170}, {6,173}, {6,167}, {6,172}, {41,171}, {6,171}, {6,157}, {6,156}, 
    {6,169}, {8,161}, {43,126}, {43,119}, {8,156}, {8,175}, {43,115}, {43,122}, 
    {43,124}, {43,127}, {8,170}, {43,123}, {43,116}, {8,168}, {43,120}, 
    {43,128}, {8,153}, {43,125}, {8,163}, {8,162}, {43,118}, {8,151}, {8,152}, 
    {8,150}, {8,174}, {8,166}, {8,160}, {8,148}, {43,121}, {8,159}, {8,149}, 
    {8,167}, {8,176}, {45,148}, {8,164}, {43,117}, {8,154}, {8,157}, {8,158}, 
    {8,172}, {8,173}, {8,171}, {8,165}, {8,169}, {8,155}, {10,180}, {45,154}, 
    {11,1}, {10,184}, {10,186}, {11,0}, {45,166}, {10,189}, {48,113}, {13,113}, 
    {11,3}, {45,151}, {45,159}, {10,190}, {45,158}, {45,155}, {45,163}, 
    {45,165}, {10,183}, {10,187}, {11,2}, {13,115}, {45,160}, {10,185}, 
    {11,5}, {45,153}, {10,188}, {13,114}, {10,182}, {45,150}, {45,157}, 
    {45,149}, {45,152}, {45,156}, {48,114}, {11,4}, {45,161}, {45,162}, 
    {45,164}, {10,181}, {13,129}, {48,126}, {13,127}, {13,128}, {48,116}, 
    {48,132}, {13,118}, {52,38}, {48,134}, {48,128}, {48,119}, {48,137}, 
    {48,122}, {13,124}, {52,61}, {48,125}, {13,126}, {13,117}, {48,139}, 
    {48,127}, {13,120}, {48,120}, {48,117}, {48,118}, {13,121}, {48,133}, 
    {48,140}, {13,123}, {48,124}, {48,123}, {13,131}, {13,130}, {13,116}, 
    {13,122}, {13,125}, {48,136}, {48,130}, {13,119}, {48,115}, {48,138}, 
    {48,129}, {48,131}, {48,135}, {52,39}, {16,119}, {16,139}, {16,138}, 
    {16,127}, {52,57}, {52,53}, {52,50}, {52,102}, {16,122}, {16,114}, 
    {52,55}, {19,104}, {16,118}, {52,97}, {16,140}, {16,137}, {52,59}, 
    {52,42}, {16,136}, {52,99}, {52,41}, {16,125}, {16,129}, {52,98}, {52,51}, 
    {16,130}, {16,124}, {52,40}, {19,56}, {52,101}, {52,49}, {16,135}, 
    {52,52}, {52,100}, {16,134}, {56,18}, {16,115}, {52,47}, {16,120}, 
    {16,131}, {16,126}, {52,56}, {52,46}, {52,44}, {52,62}, {16,112}, {16,132}, 
    {16,116}, {19,55}, {52,60}, {16,117}, {16,113}, {16,128}, {16,123}, 
    {16,121}, {52,48}, {16,133}, {52,45}, {52,58}, {52,54}, {56,20}, {56,19}, 
    {52,43}, {56,36}, {19,58}, {56,42}, {56,25}, {56,39}, {60,55}, {19,61}, 
    {56,43}, {56,46}, {19,60}, {56,28}, {56,45}, {56,44}, {19,62}, {56,21}, 
    {19,57}, {19,99}, {19,97}, {56,41}, {56,31}, {19,101}, {56,48}, {56,40}, 
    {56,49}, {19,109}, {19,107}, {56,38}, {56,37}, {56,35}, {56,29}, {19,100}, 
    {19,98}, {48,121}, {56,22}, {60,119}, {56,23}, {19,59}, {19,106}, {60,57}, 
    {19,102}, {19,103}, {56,24}, {56,47}, {60,56}, {56,32}, {56,27}, {19,105}, 
    {56,33}, {56,30}, {19,110}, {22,48}, {60,60}, {60,113}, {60,118}, {60,106}, 
    {22,44}, {60,123}, {22,41}, {60,58}, {60,59}, {22,34}, {22,43}, {60,100}, 
    {22,46}, {22,47}, {60,101}, {60,114}, {60,120}, {22,42}, {22,36}, {60,99}, 
    {60,61}, {60,122}, {60,104}, {60,105}, {60,62}, {60,116}, {60,107}, 
    {60,117}, {60,109}, {22,37}, {64,153}, {22,40}, {22,38}, {60,121}, 
    {60,112}, {60,108}, {60,97}, {25,19}, {60,111}, {22,45}, {60,103}, 
    {60,102}, {22,39}, {22,35}, {64,174}, {60,115}, {60,110}, {60,98}, 
    {64,169}, {64,154}, {64,165}, {64,172}, {25,17}, {19,108}, {64,170}, 
    {25,12}, {25,11}, {64,177}, {64,155}, {64,168}, {64,156}, {64,167}, 
    {25,15}, {64,171}, {56,34}, {64,178}, {64,163}, {25,18}, {68,122}, 
    {27,111}, {64,176}, {64,175}, {25,20}, {68,109}, {27,112}, {68,110}, 
    {64,159}, {64,160}, {64,157}, {64,162}, {64,158}, {64,179}, {25,14}, 
    {27,113}, {25,16}, {25,21}, {64,161}, {64,173}, {64,166}, {68,113}, 
    {25,10}, {27,116}, {72,106}, {68,118}, {68,117}, {68,119}, {68,116}, 
    {27,117}, {27,123}, {27,120}, {27,121}, {68,111}, {68,114}, {68,124}, 
    {27,129}, {27,127}, {68,115}, {56,26}, {27,114}, {68,121}, {68,112}, 
    {27,130}, {68,120}, {25,13}, {27,119}, {64,164}, {27,122}, {27,126}, 
    {27,128}, {27,125}, {27,124}, {27,118}, {68,123}, {27,115}, {27,131}, 
    {29,152}, {29,153}, {72,105}, {29,162}, {29,159}, {29,150}, {29,157}, 
    {72,107}, {29,155}, {29,149}, {29,156}, {72,104}, {31,123}, {29,151}, 
    {29,158}, {31,122}, {72,103}, {72,102}, {29,160}, {29,161}, {72,101}, 
    {72,100}, {31,124}, {72,110}, {29,154}, {72,108}, {31,125}, {31,130}, 
    {75,170}, {75,172}, {31,127}, {75,173}, {75,169}, {75,171}, {31,128}, 
    {31,131}, {75,168}, {31,126}, {31,129}, {33,25}, {72,109}, {33,24}, 
    {33,30}, {78,148}, {33,28}, {33,29}, {78,151}, {78,147}, {33,26}, {78,145}, 
    {34,43}, {78,149}, {78,150}, {78,146}, {33,27}, {81,2}, {81,5}, {81,6}, 
    {81,4}, {81,7}, {34,44}, {81,3}, {83,14}, {35,36}, {83,13}, {83,12}, 
    {83,11}, {35,35}, {35,37}, {84,141}, {35,162}, {35,161}, {85,161}, 
    {85,160}, {85,163}, {36,139}, {36,53}, {86,157}, {85,162}, {86,156}, 
    {36,141}, {36,140}, {36,179}, {87,105}, {87,175}, {3,164}, {56,50}, 
    {72,111}, {5,108}, {41,183}, {6,177}, {6,175}, {6,176}, {43,129}, {8,177}, 
    {11,6}, {45,167}, {45,168}, {11,7}, {48,142}, {13,132}, {13,133}, {48,141}, 
    {16,147}, {16,143}, {52,103}, {16,150}, {16,149}, {16,142}, {16,145}, 
    {16,148}, {16,144}, {56,54}, {16,141}, {19,111}, {19,113}, {19,114}, 
    {56,53}, {56,56}, {19,112}, {56,51}, {56,55}, {56,52}, {22,49}, {60,124}, 
    {25,22}, {64,180}, {29,163}, {27,132}, {68,125}, {27,133}, {27,134}, 
    {68,127}, {68,126}, {68,128}, {72,113}, {72,112}, {75,175}, {75,174}, 
    {31,132}, {31,133}, {81,8}, {3,165}, {56,57}, {19,116}, {19,115}, {60,125}, 
    {78,152}, {35,163}, {86,158}, {3,166}, {13,134}, {16,152}, {16,151}, 
    {56,58}, {56,59}, {22,50}, {64,181}, {25,23}, {72,114}, {3,167}, {4,120}, 
    {8,178}, {43,130}, {45,169}, {11,8}, {16,153}, {56,60}, {19,117}, {22,51}, 
    {68,129}, {68,130}, {75,176}, {33,31}, {87,176}, {3,168}, {43,131}, 
    {8,179}, {11,9}, {45,170}, {13,135}, {48,146}, {48,144}, {48,145}, 
    {13,136}, {48,143}, {16,155}, {16,156}, {52,104}, {16,157}, {16,154}, 
    {56,61}, {56,62}, {60,126}, {25,25}, {25,24}, {75,177}, {78,153}, {81,10}, 
    {81,9}, {83,15}, {40,30}, {11,10}, {3,169}, {4,121}, {5,110}, {5,109}, 
    {5,111}, {5,112}, {40,174}, {40,173}, {41,184}, {6,178}, {41,187}, 
    {41,186}, {41,185}, {41,188}, {8,180}, {43,137}, {43,133}, {43,142}, 
    {8,187}, {8,185}, {43,138}, {43,134}, {43,141}, {8,184}, {9,0}, {43,136}, 
    {43,132}, {8,190}, {43,139}, {8,183}, {43,140}, {8,186}, {8,188}, {43,144}, 
    {43,143}, {43,135}, {8,182}, {8,181}, {8,189}, {45,175}, {45,181}, 
    {11,16}, {11,13}, {45,172}, {45,177}, {11,19}, {11,11}, {45,176}, {11,14}, 
    {11,17}, {45,179}, {11,12}, {45,184}, {11,15}, {11,18}, {45,173}, {45,178}, 
    {45,182}, {45,174}, {45,171}, {45,183}, {45,180}, {13,144}, {13,137}, 
    {13,140}, {13,143}, {48,149}, {13,138}, {48,147}, {13,142}, {13,139}, 
    {48,150}, {13,141}, {52,108}, {16,159}, {52,107}, {52,109}, {16,158}, 
    {16,163}, {48,148}, {52,106}, {52,110}, {16,160}, {52,105}, {16,162}, 
    {16,161}, {56,103}, {56,98}, {19,118}, {19,122}, {19,119}, {56,101}, 
    {56,104}, {19,120}, {19,121}, {19,126}, {60,135}, {56,102}, {19,124}, 
    {56,99}, {56,97}, {19,125}, {56,100}, {22,57}, {60,127}, {22,54}, {22,55}, 
    {22,53}, {60,132}, {60,131}, {60,128}, {22,59}, {60,130}, {19,123}, 
    {60,134}, {60,129}, {22,56}, {22,52}, {22,58}, {60,133}, {25,28}, {64,184}, 
    {64,183}, {64,182}, {25,26}, {25,27}, {68,133}, {68,136}, {27,136}, 
    {27,135}, {68,137}, {68,132}, {27,138}, {68,134}, {27,137}, {68,131}, 
    {68,135}, {29,169}, {29,166}, {72,123}, {72,122}, {72,121}, {72,116}, 
    {72,117}, {29,167}, {29,164}, {29,168}, {72,115}, {29,165}, {72,118}, 
    {72,119}, {72,124}, {72,120}, {75,178}, {31,135}, {78,156}, {31,134}, 
    {78,154}, {78,155}, {33,32}, {34,46}, {81,11}, {34,45}, {83,17}, {83,18}, 
    {35,38}, {83,16}, {35,164}, {86,159}, {36,142}, {87,106}, {87,107}, 
    {3,170}, {5,113}, {5,114}, {6,179}, {43,145}, {11,20}, {13,145}, {16,164}, 
    {15,146}, {19,127}, {19,128}, {18,140}, {56,105}, {22,60}, {64,186}, 
    {64,185}, {3,171}, {5,115}, {43,146}, {9,2}, {9,1}, {45,185}, {45,186}, 
    {48,151}, {48,152}, {13,146}, {13,147}, {13,148}, {52,111}, {16,166}, 
    {19,130}, {19,129}, {60,136}, {62,58}, {64,187}, {72,125}, {33,33}, 
    {35,39}, {3,172}, {4,124}, {4,125}, {4,123}, {4,126}, {4,122}, {5,118}, 
    {40,182}, {5,117}, {5,119}, {40,177}, {40,176}, {40,179}, {40,178}, 
    {40,181}, {5,116}, {40,175}, {40,180}, {41,189}, {6,189}, {41,190}, 
    {42,3}, {6,188}, {42,7}, {42,2}, {42,5}, {6,181}, {6,182}, {6,183}, 
    {6,184}, {7,0}, {42,1}, {6,186}, {7,1}, {42,0}, {42,6}, {6,185}, {42,4}, 
    {6,187}, {6,180}, {6,190}, {9,23}, {43,148}, {9,3}, {9,13}, {9,14}, 
    {9,6}, {9,24}, {9,8}, {43,156}, {9,19}, {43,151}, {9,9}, {43,166}, 
    {43,167}, {43,159}, {43,152}, {9,22}, {43,164}, {9,17}, {9,15}, {43,165}, 
    {43,163}, {43,155}, {43,147}, {43,154}, {9,10}, {9,16}, {9,4}, {43,158}, 
    {43,157}, {43,149}, {9,18}, {43,161}, {43,150}, {9,21}, {43,168}, {9,5}, 
    {9,12}, {43,153}, {43,162}, {9,20}, {9,7}, {9,11}, {43,160}, {46,27}, 
    {11,28}, {11,41}, {46,22}, {46,12}, {11,34}, {46,10}, {11,27}, {46,5}, 
    {11,37}, {46,18}, {45,190}, {46,1}, {46,4}, {45,187}, {46,17}, {46,33}, 
    {11,32}, {46,6}, {46,24}, {45,189}, {46,31}, {46,32}, {46,35}, {46,26}, 
    {46,11}, {46,19}, {11,38}, {11,25}, {11,33}, {11,45}, {11,22}, {11,24}, 
    {46,3}, {11,42}, {11,35}, {46,29}, {46,0}, {11,44}, {11,39}, {46,9}, 
    {11,43}, {46,16}, {46,8}, {11,36}, {46,28}, {46,20}, {11,30}, {46,34}, 
    {46,7}, {11,26}, {46,25}, {46,15}, {11,31}, {46,21}, {11,23}, {45,188}, 
    {11,40}, {13,163}, {11,29}, {46,14}, {46,13}, {46,2}, {46,30}, {46,23}, 
    {11,21}, {48,172}, {13,170}, {48,173}, {48,161}, {13,159}, {13,171}, 
    {48,154}, {48,163}, {48,171}, {48,153}, {48,180}, {13,149}, {48,179}, 
    {48,174}, {48,175}, {13,157}, {13,168}, {48,165}, {48,166}, {48,176}, 
    {48,167}, {48,162}, {48,156}, {48,157}, {48,170}, {48,164}, {13,150}, 
    {13,154}, {48,178}, {48,158}, {13,166}, {13,162}, {13,165}, {13,172}, 
    {13,155}, {13,167}, {48,169}, {13,169}, {13,152}, {13,151}, {48,155}, 
    {48,159}, {13,160}, {48,177}, {48,168}, {48,160}, {13,164}, {13,161}, 
    {13,153}, {13,156}, {52,132}, {52,116}, {52,117}, {52,121}, {52,136}, 
    {52,133}, {52,126}, {52,125}, {16,173}, {52,129}, {52,144}, {52,112}, 
    {52,145}, {52,131}, {52,149}, {52,137}, {16,172}, {52,135}, {16,167}, 
    {16,188}, {16,178}, {16,182}, {16,181}, {52,113}, {52,142}, {52,148}, 
    {52,140}, {52,147}, {52,128}, {52,114}, {52,146}, {52,130}, {16,170}, 
    {16,183}, {52,139}, {16,176}, {52,138}, {52,115}, {16,184}, {16,186}, 
    {52,141}, {16,187}, {16,169}, {52,122}, {52,143}, {16,175}, {16,185}, 
    {52,124}, {52,134}, {52,119}, {52,123}, {16,180}, {52,118}, {16,168}, 
    {16,177}, {16,174}, {52,127}, {13,158}, {56,128}, {16,171}, {16,179}, 
    {56,131}, {56,153}, {56,142}, {19,150}, {19,145}, {56,125}, {19,146}, 
    {56,141}, {56,134}, {56,147}, {19,142}, {56,107}, {56,149}, {19,132}, 
    {56,115}, {19,135}, {19,134}, {19,151}, {56,109}, {56,143}, {56,144}, 
    {19,137}, {19,133}, {56,123}, {19,144}, {56,118}, {56,145}, {19,140}, 
    {56,137}, {56,150}, {56,112}, {56,117}, {56,111}, {19,139}, {56,130}, 
    {60,158}, {56,113}, {19,143}, {56,122}, {56,146}, {19,138}, {56,119}, 
    {56,116}, {56,133}, {19,141}, {19,131}, {19,153}, {56,136}, {56,135}, 
    {56,108}, {19,136}, {56,148}, {56,124}, {56,126}, {56,139}, {56,138}, 
    {56,106}, {19,147}, {19,149}, {56,114}, {56,121}, {56,129}, {19,148}, 
    {56,120}, {56,132}, {56,151}, {56,140}, {56,152}, {56,110}, {60,178}, 
    {22,102}, {60,176}, {60,155}, {60,160}, {60,153}, {60,172}, {60,139}, 
    {60,146}, {60,170}, {60,180}, {60,156}, {60,143}, {60,162}, {60,167}, 
    {60,147}, {60,164}, {60,144}, {60,151}, {60,152}, {22,104}, {60,171}, 
    {60,169}, {60,140}, {60,174}, {60,175}, {60,177}, {22,108}, {22,100}, 
    {52,120}, {60,148}, {60,166}, {60,149}, {22,97}, {22,113}, {60,173}, 
    {22,111}, {22,107}, {60,138}, {22,99}, {60,141}, {22,112}, {60,157}, 
    {60,137}, {22,105}, {60,161}, {60,145}, {22,106}, {60,154}, {22,62}, 
    {19,152}, {60,163}, {56,127}, {60,142}, {60,168}, {22,101}, {60,165}, 
    {22,98}, {60,159}, {22,109}, {60,150}, {60,179}, {22,103}, {61,134}, 
    {22,110}, {65,10}, {65,8}, {65,30}, {65,6}, {65,24}, {22,61}, {25,31}, 
    {65,2}, {65,29}, {65,7}, {65,21}, {25,36}, {25,29}, {65,27}, {65,0}, 
    {65,26}, {25,47}, {65,17}, {65,33}, {25,45}, {65,9}, {25,30}, {65,11}, 
    {65,25}, {25,39}, {65,4}, {25,43}, {25,33}, {65,13}, {65,3}, {64,188}, 
    {65,23}, {25,40}, {65,32}, {64,189}, {25,37}, {65,19}, {25,38}, {65,5}, 
    {65,16}, {65,12}, {65,14}, {25,32}, {65,31}, {25,46}, {65,15}, {65,34}, 
    {64,190}, {65,20}, {25,35}, {25,44}, {25,42}, {65,1}, {65,22}, {25,41}, 
    {25,34}, {65,18}, {65,28}, {68,149}, {68,145}, {68,141}, {68,161}, 
    {68,158}, {27,141}, {68,165}, {68,148}, {27,152}, {68,155}, {68,144}, 
    {68,154}, {27,149}, {68,174}, {68,171}, {68,157}, {68,142}, {68,162}, 
    {68,164}, {27,145}, {68,152}, {68,147}, {68,138}, {27,142}, {27,150}, 
    {68,167}, {27,151}, {68,139}, {68,173}, {68,160}, {68,166}, {27,148}, 
    {68,163}, {68,170}, {27,153}, {27,147}, {68,156}, {68,143}, {68,175}, 
    {68,140}, {68,168}, {27,144}, {68,150}, {68,151}, {27,143}, {27,140}, 
    {68,146}, {27,146}, {27,139}, {68,169}, {68,172}, {68,153}, {72,138}, 
    {72,130}, {72,126}, {29,182}, {29,171}, {29,176}, {29,172}, {72,140}, 
    {72,151}, {29,170}, {72,132}, {72,141}, {68,159}, {72,142}, {29,177}, 
    {72,157}, {29,181}, {29,184}, {72,128}, {29,180}, {72,155}, {72,156}, 
    {72,146}, {72,145}, {72,137}, {72,147}, {72,154}, {72,153}, {29,175}, 
    {29,173}, {72,139}, {72,136}, {72,133}, {72,152}, {29,183}, {72,150}, 
    {29,179}, {29,178}, {72,144}, {72,127}, {72,129}, {72,131}, {72,149}, 
    {72,143}, {29,174}, {72,134}, {72,148}, {72,135}, {31,143}, {76,5}, 
    {31,136}, {75,181}, {76,1}, {31,138}, {76,8}, {75,188}, {75,183}, {76,9}, 
    {75,179}, {75,190}, {31,145}, {76,4}, {76,10}, {75,189}, {31,137}, 
    {76,0}, {75,180}, {31,144}, {76,7}, {75,185}, {31,140}, {75,187}, {75,184}, 
    {31,146}, {75,186}, {31,139}, {31,142}, {76,3}, {75,182}, {76,6}, {76,2}, 
    {33,35}, {78,167}, {33,40}, {33,41}, {33,34}, {78,166}, {78,163}, {78,164}, 
    {33,38}, {78,158}, {78,162}, {33,37}, {78,159}, {33,39}, {33,36}, {78,157}, 
    {78,161}, {78,165}, {81,17}, {81,14}, {81,23}, {81,22}, {81,20}, {81,15}, 
    {34,50}, {81,16}, {34,49}, {31,141}, {81,19}, {34,48}, {81,24}, {81,18}, 
    {81,13}, {78,160}, {34,47}, {81,12}, {83,22}, {83,21}, {81,21}, {35,40}, 
    {83,25}, {83,26}, {83,20}, {83,24}, {83,19}, {84,145}, {83,23}, {35,167}, 
    {35,165}, {84,143}, {84,146}, {84,142}, {84,144}, {35,166}, {85,165}, 
    {85,166}, {36,54}, {85,164}, {86,162}, {36,143}, {86,160}, {86,161}, 
    {87,108}, {37,22}, {87,179}, {87,177}, {87,178}, {87,180}, {88,123}, 
    {3,173}, {5,120}, {9,25}, {43,169}, {46,36}, {48,181}, {48,183}, {48,182}, 
    {48,184}, {16,189}, {52,151}, {48,185}, {52,150}, {52,152}, {52,153}, 
    {56,154}, {19,155}, {56,155}, {56,157}, {19,156}, {19,154}, {56,156}, 
    {60,186}, {60,184}, {60,183}, {60,182}, {60,181}, {22,114}, {60,185}, 
    {25,48}, {65,35}, {65,37}, {25,49}, {65,36}, {27,155}, {27,154}, {68,176}, 
    {72,159}, {72,158}, {72,160}, {29,185}, {76,11}, {31,147}, {78,168}, 
    {33,42}, {81,25}, {36,55}, {3,174}, {4,127}, {5,121}, {7,2}, {9,26}, 
    {9,27}, {11,46}, {48,186}, {22,115}, {69,145}, {29,186}, {33,43}, {3,175}, 
    {5,122}, {43,171}, {9,28}, {43,170}, {46,37}, {11,47}, {46,38}, {11,48}, 
    {48,188}, {13,174}, {13,173}, {52,158}, {52,156}, {52,157}, {52,155}, 
    {52,154}, {56,158}, {56,161}, {19,158}, {56,159}, {19,157}, {56,160}, 
    {60,187}, {65,38}, {65,39}, {65,40}, {68,179}, {68,178}, {27,156}, 
    {68,177}, {68,180}, {72,161}, {72,162}, {72,163}, {76,12}, {31,148}, 
    {33,44}, {81,26}, {35,168}, {40,31}, {11,49}, {46,39}, {13,175}, {16,190}, 
    {19,159}, {56,162}, {22,117}, {22,116}, {65,41}, {65,42}, {27,157}, 
    {27,158}, {72,165}, {72,164}, {78,169}, {86,163}, {3,176}, {40,32}, 
    {4,128}, {7,3}, {42,8}, {11,50}, {22,118}, {3,177}, {46,40}, {11,51}, 
    {46,41}, {31,149}, {3,178}, {43,172}, {46,42}, {49,2}, {49,1}, {48,190}, 
    {48,189}, {49,3}, {49,0}, {17,0}, {17,1}, {19,160}, {56,163}, {56,164}, 
    {56,165}, {61,1}, {61,2}, {61,0}, {60,189}, {60,190}, {22,119}, {65,43}, 
    {68,183}, {68,182}, {68,181}, {68,184}, {72,167}, {72,166}, {29,187}, 
    {72,168}, {31,150}, {76,13}, {78,170}, {81,27}, {85,167}, {3,179}, 
    {4,130}, {4,129}, {9,29}, {40,33}, {40,62}, {5,123}, {40,183}, {42,9}, 
    {42,10}, {9,30}, {43,173}, {11,52}, {46,43}, {46,44}, {13,176}, {13,180}, 
    {49,4}, {13,179}, {13,177}, {13,178}, {52,159}, {17,2}, {19,163}, {19,161}, 
    {19,162}, {56,166}, {25,50}, {3,180}, {40,97}, {4,131}, {40,100}, {4,134}, 
    {40,99}, {4,133}, {4,132}, {7,4}, {40,98}, {40,184}, {40,188}, {40,190}, 
    {41,0}, {5,133}, {5,134}, {40,187}, {5,129}, {40,185}, {40,189}, {5,130}, 
    {5,125}, {5,126}, {5,132}, {40,186}, {5,124}, {7,5}, {5,127}, {5,128}, 
    {5,131}, {42,27}, {42,25}, {42,12}, {7,17}, {42,19}, {7,12}, {42,13}, 
    {42,21}, {42,18}, {7,15}, {42,17}, {7,22}, {42,26}, {7,24}, {7,26}, 
    {42,11}, {7,13}, {42,28}, {7,20}, {7,23}, {44,5}, {7,7}, {7,30}, {7,21}, 
    {42,14}, {7,10}, {7,25}, {42,22}, {7,8}, {7,9}, {44,3}, {42,15}, {7,16}, 
    {7,27}, {42,29}, {42,16}, {7,14}, {7,19}, {43,174}, {7,28}, {42,23}, 
    {7,18}, {7,29}, {7,6}, {42,20}, {7,11}, {42,24}, {44,4}, {9,42}, {9,58}, 
    {43,181}, {9,49}, {44,11}, {9,34}, {9,37}, {44,2}, {43,179}, {43,183}, 
    {9,45}, {9,47}, {43,186}, {9,54}, {9,40}, {9,38}, {9,39}, {9,53}, {44,7}, 
    {9,48}, {43,185}, {43,187}, {9,46}, {9,51}, {43,188}, {44,10}, {11,53}, 
    {9,57}, {9,35}, {44,9}, {44,13}, {43,184}, {44,15}, {44,0}, {9,44}, 
    {43,180}, {9,43}, {9,61}, {9,50}, {43,178}, {46,53}, {9,56}, {9,60}, 
    {44,1}, {44,6}, {9,62}, {9,55}, {9,41}, {9,31}, {9,36}, {43,182}, {9,32}, 
    {44,14}, {43,176}, {43,175}, {43,189}, {43,177}, {9,59}, {13,181}, 
    {9,52}, {43,190}, {9,33}, {11,102}, {44,12}, {46,60}, {46,97}, {46,100}, 
    {46,55}, {46,103}, {46,106}, {46,108}, {46,52}, {11,54}, {11,59}, {49,9}, 
    {11,109}, {46,101}, {46,109}, {46,59}, {46,51}, {49,36}, {11,62}, {46,98}, 
    {46,56}, {46,58}, {11,101}, {46,61}, {11,61}, {46,48}, {46,104}, {46,107}, 
    {11,58}, {11,104}, {46,45}, {11,106}, {11,56}, {11,110}, {46,105}, 
    {46,47}, {11,107}, {49,30}, {44,8}, {11,60}, {11,55}, {46,54}, {46,46}, 
    {11,108}, {11,100}, {46,99}, {11,105}, {11,103}, {46,57}, {11,97}, 
    {46,49}, {11,98}, {11,99}, {46,50}, {46,102}, {11,57}, {46,62}, {49,12}, 
    {13,189}, {14,3}, {49,21}, {49,27}, {49,23}, {49,10}, {49,13}, {49,6}, 
    {49,7}, {14,10}, {13,186}, {49,22}, {49,31}, {14,5}, {13,182}, {14,0}, 
    {49,14}, {14,2}, {49,15}, {49,25}, {14,4}, {49,40}, {49,8}, {13,188}, 
    {13,187}, {14,8}, {49,5}, {49,38}, {49,26}, {49,39}, {49,33}, {49,19}, 
    {49,34}, {49,28}, {49,37}, {49,35}, {14,9}, {49,20}, {13,185}, {13,184}, 
    {14,1}, {14,7}, {49,32}, {14,6}, {49,17}, {17,3}, {49,41}, {49,16}, 
    {49,11}, {13,190}, {14,11}, {13,183}, {49,24}, {49,29}, {17,37}, {52,161}, 
    {52,165}, {17,18}, {17,16}, {17,7}, {52,163}, {52,162}, {17,27}, {52,168}, 
    {17,21}, {52,186}, {53,7}, {17,4}, {52,183}, {52,176}, {17,39}, {52,160}, 
    {52,188}, {17,36}, {17,24}, {17,35}, {17,14}, {52,172}, {52,190}, {52,182}, 
    {17,15}, {17,9}, {53,5}, {52,189}, {53,0}, {17,17}, {17,25}, {53,2}, 
    {52,170}, {52,187}, {52,175}, {53,4}, {17,30}, {17,6}, {17,28}, {52,180}, 
    {52,178}, {52,179}, {17,19}, {52,174}, {52,173}, {17,8}, {52,167}, 
    {53,6}, {17,10}, {52,177}, {17,40}, {17,34}, {52,166}, {17,31}, {17,29}, 
    {17,38}, {52,184}, {17,33}, {49,18}, {52,185}, {17,32}, {53,1}, {17,5}, 
    {52,181}, {17,23}, {52,169}, {17,22}, {17,20}, {17,12}, {17,11}, {56,167}, 
    {53,3}, {52,171}, {56,188}, {17,13}, {20,1}, {17,26}, {19,174}, {56,182}, 
    {19,188}, {56,170}, {19,171}, {19,167}, {57,9}, {19,173}, {19,177}, 
    {19,172}, {19,181}, {57,13}, {57,4}, {56,177}, {19,186}, {19,180}, 
    {56,189}, {19,164}, {57,10}, {57,3}, {19,168}, {56,183}, {19,183}, 
    {57,21}, {57,22}, {19,165}, {57,8}, {19,185}, {56,187}, {56,173}, {56,174}, 
    {19,189}, {56,178}, {56,185}, {56,179}, {19,187}, {20,4}, {56,175}, 
    {56,168}, {56,169}, {56,171}, {19,170}, {56,184}, {19,184}, {20,2}, 
    {56,186}, {57,19}, {57,11}, {19,166}, {57,17}, {19,178}, {19,176}, 
    {57,23}, {19,175}, {57,1}, {56,180}, {56,190}, {20,7}, {57,5}, {57,2}, 
    {56,176}, {20,3}, {57,15}, {57,12}, {57,20}, {19,169}, {57,0}, {20,6}, 
    {57,7}, {19,179}, {19,182}, {57,6}, {20,5}, {56,181}, {52,164}, {57,16}, 
    {57,14}, {57,18}, {56,172}, {20,0}, {61,33}, {61,32}, {61,6}, {22,125}, 
    {61,31}, {61,9}, {61,10}, {22,135}, {61,40}, {22,130}, {61,30}, {61,3}, 
    {22,136}, {22,126}, {61,18}, {61,8}, {61,11}, {61,35}, {22,120}, {61,42}, 
    {61,34}, {22,129}, {61,23}, {22,140}, {22,139}, {22,133}, {61,41}, 
    {22,121}, {61,21}, {61,12}, {61,25}, {61,37}, {22,141}, {22,123}, {61,20}, 
    {61,13}, {22,132}, {22,131}, {61,16}, {61,26}, {61,36}, {61,7}, {61,17}, 
    {22,124}, {61,27}, {22,137}, {22,128}, {61,14}, {22,127}, {61,5}, {61,19}, 
    {61,39}, {19,190}, {25,112}, {61,22}, {65,44}, {61,24}, {61,38}, {22,134}, 
    {61,15}, {22,122}, {22,138}, {27,176}, {61,4}, {61,29}, {61,28}, {65,106}, 
    {25,109}, {65,61}, {65,100}, {25,98}, {65,46}, {25,111}, {25,55}, {65,45}, 
    {65,112}, {25,113}, {65,49}, {65,99}, {65,51}, {65,115}, {65,111}, 
    {25,53}, {25,97}, {69,19}, {25,110}, {25,61}, {65,47}, {65,110}, {25,99}, 
    {65,107}, {65,120}, {65,53}, {65,62}, {65,118}, {65,108}, {25,60}, 
    {65,60}, {25,54}, {25,52}, {25,104}, {65,58}, {65,55}, {65,56}, {65,114}, 
    {65,119}, {65,117}, {25,58}, {65,121}, {25,62}, {25,103}, {65,48}, 
    {68,186}, {65,57}, {25,56}, {25,108}, {25,105}, {25,59}, {65,101}, 
    {65,52}, {25,106}, {65,103}, {25,100}, {25,102}, {25,51}, {65,105}, 
    {65,97}, {65,50}, {25,101}, {65,113}, {65,116}, {65,59}, {65,104}, 
    {25,57}, {27,159}, {65,102}, {68,185}, {65,109}, {65,54}, {69,4}, {69,14}, 
    {27,162}, {69,13}, {69,25}, {27,164}, {69,11}, {69,15}, {27,175}, {69,6}, 
    {27,167}, {69,18}, {72,176}, {27,179}, {27,178}, {69,20}, {69,3}, {69,30}, 
    {27,173}, {27,163}, {69,23}, {69,27}, {69,32}, {69,21}, {69,9}, {27,166}, 
    {27,169}, {27,177}, {27,172}, {69,12}, {65,98}, {69,8}, {69,31}, {27,168}, 
    {27,171}, {69,33}, {27,160}, {69,22}, {68,187}, {69,28}, {31,159}, 
    {69,10}, {27,161}, {69,5}, {27,165}, {68,188}, {25,107}, {69,1}, {69,26}, 
    {69,2}, {69,0}, {27,170}, {69,24}, {68,190}, {69,17}, {69,16}, {69,29}, 
    {69,7}, {27,174}, {72,179}, {30,9}, {29,190}, {73,0}, {72,171}, {30,1}, 
    {72,183}, {30,8}, {30,3}, {72,181}, {76,15}, {72,187}, {73,2}, {72,186}, 
    {72,169}, {72,184}, {73,4}, {73,6}, {29,189}, {73,5}, {30,4}, {30,10}, 
    {30,7}, {72,190}, {30,6}, {72,185}, {72,173}, {72,178}, {72,189}, {30,5}, 
    {30,2}, {29,188}, {30,0}, {72,177}, {68,189}, {72,172}, {72,175}, {73,1}, 
    {72,180}, {72,170}, {76,14}, {73,3}, {72,174}, {72,188}, {76,17}, {31,163}, 
    {31,151}, {31,155}, {76,19}, {76,25}, {76,23}, {31,153}, {31,154}, 
    {31,161}, {76,26}, {76,18}, {31,156}, {76,22}, {76,21}, {76,27}, {31,162}, 
    {31,157}, {31,160}, {76,20}, {31,164}, {31,158}, {31,165}, {31,152}, 
    {76,24}, {76,16}, {78,183}, {33,49}, {78,180}, {78,182}, {33,47}, {78,178}, 
    {78,179}, {78,174}, {72,182}, {78,175}, {33,48}, {78,171}, {33,45}, 
    {78,184}, {33,46}, {78,172}, {78,173}, {78,177}, {33,51}, {33,50}, 
    {78,176}, {34,56}, {81,31}, {81,37}, {34,57}, {81,28}, {34,54}, {34,51}, 
    {81,39}, {34,55}, {34,52}, {81,30}, {81,33}, {81,34}, {81,35}, {81,38}, 
    {78,181}, {81,29}, {34,53}, {81,36}, {81,40}, {81,32}, {83,29}, {35,42}, 
    {83,32}, {35,43}, {83,40}, {83,31}, {83,28}, {83,30}, {83,34}, {83,37}, 
    {83,36}, {83,39}, {83,27}, {35,41}, {83,35}, {83,38}, {83,41}, {83,33}, 
    {84,147}, {84,148}, {84,152}, {84,153}, {84,150}, {84,151}, {84,149}, 
    {35,169}, {36,56}, {85,171}, {85,168}, {85,169}, {85,170}, {36,57}, 
    {86,165}, {86,164}, {87,111}, {36,180}, {87,109}, {87,112}, {87,110}, 
    {87,181}, {37,23}, {37,37}, {88,99}, {88,44}, {88,98}, {88,144}, {88,145}, 
    {3,181}, {5,135}, {41,1}, {42,30}, {7,31}, {7,34}, {42,31}, {7,32}, 
    {7,33}, {44,24}, {44,26}, {44,21}, {44,18}, {44,20}, {9,100}, {9,98}, 
    {44,22}, {9,99}, {44,19}, {44,16}, {9,97}, {44,23}, {44,17}, {9,101}, 
    {44,25}, {46,111}, {46,115}, {11,119}, {46,118}, {11,111}, {11,114}, 
    {11,116}, {11,118}, {11,115}, {46,114}, {46,113}, {11,113}, {46,116}, 
    {46,117}, {46,110}, {11,117}, {11,112}, {46,112}, {49,55}, {49,56}, 
    {49,57}, {14,16}, {14,12}, {49,46}, {49,54}, {49,59}, {14,17}, {49,44}, 
    {49,50}, {49,43}, {49,53}, {49,49}, {14,13}, {14,15}, {49,58}, {49,42}, 
    {49,45}, {49,51}, {49,52}, {49,60}, {49,48}, {14,14}, {17,45}, {53,14}, 
    {53,16}, {53,12}, {53,24}, {53,10}, {53,23}, {17,41}, {53,8}, {57,27}, 
    {53,18}, {17,44}, {53,19}, {53,22}, {53,26}, {53,15}, {53,20}, {17,42}, 
    {17,43}, {53,25}, {53,13}, {53,9}, {53,27}, {53,17}, {53,21}, {53,11}, 
    {20,8}, {20,9}, {57,37}, {20,15}, {57,25}, {57,34}, {57,24}, {20,12}, 
    {57,32}, {57,30}, {57,31}, {20,10}, {57,35}, {57,28}, {57,26}, {20,11}, 
    {57,29}, {57,33}, {20,13}, {57,36}, {61,48}, {61,55}, {61,57}, {61,97}, 
    {22,154}, {61,43}, {22,146}, {61,58}, {22,151}, {61,98}, {22,142}, 
    {61,61}, {61,45}, {61,62}, {61,44}, {22,156}, {61,56}, {22,143}, {61,99}, 
    {22,148}, {61,49}, {22,153}, {61,60}, {61,47}, {61,54}, {61,50}, {61,46}, 
    {22,145}, {22,152}, {22,150}, {22,147}, {22,155}, {22,144}, {61,53}, 
    {22,149}, {20,14}, {61,59}, {61,51}, {61,52}, {65,129}, {25,116}, {65,125}, 
    {65,131}, {65,127}, {25,118}, {65,126}, {65,130}, {65,122}, {65,124}, 
    {25,117}, {65,128}, {65,123}, {25,119}, {25,114}, {65,132}, {25,115}, 
    {69,39}, {69,36}, {69,48}, {69,42}, {69,44}, {27,180}, {69,38}, {69,46}, 
    {69,45}, {69,43}, {69,49}, {27,183}, {69,40}, {69,47}, {27,181}, {69,35}, 
    {69,37}, {27,182}, {69,34}, {69,50}, {69,41}, {73,10}, {30,17}, {73,21}, 
    {73,19}, {30,11}, {73,9}, {73,12}, {73,13}, {73,8}, {30,21}, {30,22}, 
    {73,7}, {73,22}, {73,17}, {30,15}, {30,12}, {73,16}, {73,14}, {30,18}, 
    {73,18}, {30,13}, {30,14}, {73,15}, {30,16}, {73,11}, {73,20}, {30,19}, 
    {73,23}, {73,24}, {30,20}, {31,167}, {31,174}, {76,28}, {76,34}, {76,32}, 
    {31,170}, {31,169}, {31,166}, {76,30}, {31,172}, {31,171}, {31,168}, 
    {76,33}, {76,29}, {76,31}, {31,173}, {33,55}, {78,187}, {33,52}, {33,53}, 
    {78,189}, {33,54}, {78,186}, {78,185}, {81,44}, {78,188}, {81,45}, 
    {34,58}, {81,43}, {81,42}, {81,41}, {34,59}, {35,44}, {83,42}, {83,43}, 
    {84,156}, {84,155}, {35,170}, {84,154}, {85,172}, {85,173}, {86,166}, 
    {87,113}, {87,182}, {88,124}, {37,57}, {88,134}, {3,182}, {9,102}, 
    {9,103}, {11,120}, {31,175}, {3,183}, {9,104}, {14,18}, {22,157}, {3,184}, 
    {17,46}, {25,120}, {40,34}, {46,119}, {49,61}, {65,133}, {31,176}, 
    {3,185}, {9,105}, {46,120}, {46,121}, {57,38}, {20,16}, {61,100}, {22,158}, 
    {65,134}, {27,184}, {34,60}, {3,186}, {57,39}, {3,187}, {5,137}, {41,2}, 
    {5,136}, {7,37}, {7,36}, {7,35}, {42,32}, {9,106}, {9,107}, {44,27}, 
    {46,122}, {46,125}, {11,122}, {46,123}, {11,121}, {46,124}, {11,123}, 
    {49,98}, {49,97}, {49,62}, {14,19}, {53,29}, {53,30}, {17,47}, {53,28}, 
    {53,31}, {20,18}, {17,48}, {20,17}, {57,43}, {57,42}, {57,40}, {57,41}, 
    {57,44}, {61,102}, {61,101}, {61,105}, {61,104}, {61,103}, {25,121}, 
    {65,137}, {65,136}, {25,122}, {65,135}, {69,51}, {69,52}, {27,185}, 
    {73,25}, {73,26}, {81,50}, {34,61}, {81,49}, {81,48}, {81,46}, {81,47}, 
    {35,171}, {83,44}, {85,174}, {87,183}, {3,188}, {40,101}, {4,135}, 
    {40,102}, {41,3}, {41,4}, {42,38}, {42,34}, {42,33}, {9,108}, {42,37}, 
    {7,39}, {42,35}, {7,38}, {42,39}, {42,36}, {44,31}, {46,126}, {44,29}, 
    {44,36}, {9,109}, {9,112}, {44,37}, {44,33}, {44,34}, {44,28}, {9,111}, 
    {44,30}, {9,110}, {44,35}, {44,32}, {46,130}, {11,125}, {11,126}, {46,133}, 
    {46,127}, {46,132}, {46,128}, {11,124}, {46,131}, {46,129}, {49,104}, 
    {49,101}, {49,103}, {14,24}, {14,23}, {14,21}, {49,100}, {49,105}, 
    {14,20}, {14,22}, {49,102}, {53,39}, {49,99}, {49,106}, {53,34}, {53,38}, 
    {53,37}, {57,46}, {57,57}, {53,40}, {53,35}, {57,45}, {17,52}, {17,51}, 
    {53,33}, {53,36}, {17,53}, {17,50}, {17,49}, {53,32}, {53,41}, {57,48}, 
    {57,55}, {20,20}, {57,54}, {57,51}, {20,22}, {57,53}, {57,47}, {57,49}, 
    {57,52}, {57,50}, {20,21}, {57,56}, {20,19}, {22,159}, {61,109}, {61,108}, 
    {61,106}, {22,162}, {22,161}, {61,110}, {61,107}, {65,138}, {25,123}, 
    {22,160}, {61,112}, {61,111}, {65,141}, {65,139}, {27,186}, {25,124}, 
    {65,140}, {69,54}, {27,187}, {69,53}, {69,62}, {69,61}, {69,59}, {69,58}, 
    {69,55}, {69,56}, {69,57}, {69,60}, {69,97}, {73,31}, {73,28}, {73,29}, 
    {30,23}, {73,27}, {73,33}, {73,32}, {73,30}, {76,36}, {76,37}, {31,177}, 
    {31,178}, {76,35}, {33,57}, {78,190}, {33,56}, {34,62}, {34,97}, {35,45}, 
    {83,46}, {83,45}, {84,157}, {85,175}, {36,58}, {86,168}, {86,167}, 
    {86,169}, {4,136}, {46,134}, {14,25}, {17,54}, {53,42}, {4,137}, {40,103}, 
    {3,189}, {41,5}, {42,44}, {42,42}, {42,43}, {42,40}, {7,40}, {42,41}, 
    {44,45}, {9,115}, {44,43}, {44,39}, {44,42}, {44,38}, {9,117}, {44,41}, 
    {9,114}, {9,113}, {9,116}, {44,44}, {44,40}, {11,130}, {11,133}, {46,142}, 
    {46,141}, {46,140}, {11,127}, {46,149}, {46,139}, {11,129}, {49,111}, 
    {46,146}, {46,144}, {11,132}, {46,136}, {46,147}, {46,138}, {46,148}, 
    {46,145}, {46,137}, {11,128}, {46,150}, {46,135}, {11,131}, {49,116}, 
    {49,107}, {49,118}, {49,110}, {49,121}, {49,122}, {49,108}, {49,120}, 
    {49,117}, {49,115}, {49,119}, {14,31}, {14,29}, {49,113}, {49,109}, 
    {49,112}, {49,123}, {49,114}, {14,30}, {46,143}, {14,26}, {14,28}, 
    {53,56}, {53,45}, {53,43}, {53,44}, {53,51}, {53,52}, {53,48}, {17,59}, 
    {53,53}, {53,50}, {53,47}, {17,57}, {53,46}, {17,55}, {17,58}, {53,49}, 
    {53,57}, {14,27}, {17,56}, {53,55}, {53,54}, {17,60}, {57,62}, {57,97}, 
    {20,32}, {57,103}, {57,105}, {57,98}, {20,26}, {57,102}, {57,101}, 
    {20,27}, {20,33}, {20,34}, {57,104}, {20,24}, {57,61}, {57,59}, {57,99}, 
    {57,58}, {20,31}, {57,60}, {57,100}, {57,106}, {20,25}, {20,30}, {20,28}, 
    {20,29}, {20,23}, {22,169}, {61,119}, {22,168}, {61,123}, {61,113}, 
    {61,124}, {61,114}, {61,115}, {61,125}, {61,122}, {61,120}, {61,121}, 
    {61,117}, {61,116}, {61,126}, {22,165}, {61,118}, {22,170}, {22,164}, 
    {22,171}, {22,172}, {22,167}, {22,166}, {65,142}, {25,126}, {25,125}, 
    {65,147}, {27,188}, {25,127}, {25,129}, {65,148}, {22,163}, {25,128}, 
    {65,144}, {65,146}, {65,143}, {65,145}, {69,107}, {69,106}, {69,103}, 
    {28,0}, {73,34}, {28,1}, {69,102}, {27,190}, {69,104}, {69,101}, {69,98}, 
    {69,105}, {69,99}, {69,100}, {27,189}, {76,41}, {73,38}, {73,37}, {73,39}, 
    {76,38}, {30,26}, {73,35}, {30,24}, {30,28}, {30,27}, {73,36}, {73,40}, 
    {30,25}, {76,45}, {31,181}, {33,58}, {31,182}, {31,179}, {76,42}, {76,40}, 
    {76,43}, {76,46}, {31,180}, {76,44}, {76,39}, {79,2}, {79,5}, {81,53}, 
    {79,0}, {83,47}, {79,6}, {34,98}, {79,4}, {33,59}, {79,1}, {79,3}, 
    {79,7}, {81,54}, {81,52}, {34,99}, {81,51}, {35,46}, {35,173}, {85,177}, 
    {35,172}, {85,179}, {85,176}, {85,178}, {36,144}, {87,114}, {4,138}, 
    {44,46}, {49,124}, {49,125}, {17,61}, {61,127}, {30,29}, {34,100}, 
    {36,59}, {87,115}, {4,139}, {44,47}, {1,32}, {46,151}, {46,152}, {49,126}, 
    {49,127}, {17,62}, {17,97}, {57,107}, {61,130}, {61,129}, {61,128}, 
    {65,149}, {65,150}, {65,151}, {25,130}, {69,109}, {69,108}, {73,41}, 
    {30,30}, {30,31}, {76,50}, {76,47}, {76,48}, {76,49}, {79,9}, {79,8}, 
    {33,60}, {81,55}, {84,158}, {4,140}, {11,134}, {17,98}, {61,131}, {4,141}, 
    {49,128}, {17,99}, {20,35}, {20,36}, {4,142}, {4,143}, {41,6}, {7,42}, 
    {7,41}, {11,135}, {46,153}, {57,108}, {4,144}, {4,145}, {4,146}, {4,147}, 
    {7,43}, {7,44}, {42,46}, {42,45}, {9,118}, {44,50}, {44,48}, {44,49}, 
    {46,154}, {46,155}, {11,139}, {11,137}, {11,138}, {11,136}, {14,32}, 
    {14,36}, {14,35}, {49,129}, {14,34}, {14,33}, {49,130}, {17,102}, {53,59}, 
    {53,58}, {17,100}, {17,101}, {20,38}, {20,37}, {57,110}, {57,109}, 
    {17,103}, {22,173}, {61,133}, {22,174}, {61,132}, {65,152}, {69,110}, 
    {28,2}, {73,42}, {76,51}, {34,102}, {34,101}, {36,60}, {4,148}, {44,51}, 
    {17,104}, {65,153}, {25,131}, {42,47}, {42,48}, {44,52}, {9,120}, {9,121}, 
    {9,119}, {11,143}, {11,144}, {11,141}, {11,142}, {46,156}, {46,157}, 
    {11,140}, {49,131}, {14,40}, {14,41}, {17,107}, {49,137}, {14,46}, 
    {14,44}, {49,138}, {49,133}, {14,43}, {14,42}, {14,37}, {49,136}, {49,135}, 
    {49,132}, {14,45}, {49,134}, {14,38}, {14,39}, {17,108}, {53,97}, {53,98}, 
    {17,109}, {53,60}, {53,62}, {53,100}, {53,99}, {53,61}, {17,105}, {17,106}, 
    {57,118}, {20,43}, {20,42}, {57,112}, {20,40}, {57,115}, {20,44}, {57,116}, 
    {20,45}, {57,113}, {20,39}, {20,41}, {57,117}, {57,114}, {57,111}, 
    {61,146}, {61,135}, {22,176}, {22,179}, {22,178}, {22,183}, {22,182}, 
    {61,147}, {61,145}, {61,138}, {61,142}, {61,141}, {22,180}, {61,144}, 
    {61,140}, {61,148}, {61,139}, {22,181}, {22,175}, {22,177}, {61,137}, 
    {65,155}, {25,135}, {65,159}, {25,134}, {65,156}, {25,133}, {61,136}, 
    {61,143}, {65,158}, {25,136}, {65,160}, {65,157}, {65,154}, {69,113}, 
    {69,117}, {69,119}, {69,115}, {69,114}, {69,112}, {28,5}, {28,3}, {28,8}, 
    {28,9}, {69,116}, {28,6}, {69,111}, {28,7}, {25,132}, {69,118}, {28,4}, 
    {73,44}, {73,43}, {73,51}, {73,45}, {73,50}, {73,47}, {30,32}, {73,49}, 
    {30,33}, {30,34}, {73,48}, {73,46}, {31,184}, {76,52}, {31,183}, {76,55}, 
    {76,53}, {76,54}, {31,185}, {79,13}, {33,97}, {79,14}, {33,61}, {79,15}, 
    {33,62}, {79,12}, {79,16}, {79,10}, {34,103}, {81,56}, {34,104}, {35,47}, 
    {79,11}, {35,48}, {35,174}, {84,159}, {36,62}, {85,180}, {36,61}, {86,170}, 
    {36,181}, {36,182}, {88,140}, {11,145}, {46,158}, {20,46}, {20,47}, 
    {4,149}, {5,138}, {41,7}, {42,49}, {7,45}, {9,122}, {11,146}, {11,147}, 
    {11,148}, {53,102}, {49,139}, {14,47}, {17,110}, {53,101}, {57,120}, 
    {20,49}, {57,119}, {20,48}, {61,149}, {28,10}, {69,123}, {69,120}, 
    {69,121}, {69,122}, {76,56}, {79,17}, {83,49}, {83,48}, {85,181}, {4,150}, 
    {44,53}, {14,48}, {20,50}, {61,150}, {65,161}, {28,11}, {73,52}, {79,18}, 
    {83,50}, {4,151}, {9,123}, {11,151}, {46,159}, {11,152}, {11,150}, 
    {11,149}, {49,140}, {14,49}, {14,50}, {14,51}, {17,112}, {53,103}, 
    {17,111}, {57,121}, {17,113}, {20,51}, {61,151}, {22,184}, {22,185}, 
    {25,137}, {25,138}, {28,12}, {30,36}, {73,53}, {30,35}, {76,57}, {31,186}, 
    {79,19}, {83,51}, {4,152}, {7,46}, {44,56}, {44,55}, {9,124}, {44,54}, 
    {9,125}, {44,57}, {46,165}, {11,155}, {11,154}, {46,167}, {46,166}, 
    {11,159}, {11,158}, {11,153}, {46,161}, {46,162}, {46,163}, {11,160}, 
    {46,160}, {11,156}, {46,164}, {11,157}, {49,143}, {49,147}, {49,145}, 
    {49,144}, {49,148}, {49,149}, {49,150}, {49,142}, {49,141}, {14,53}, 
    {14,54}, {49,151}, {49,146}, {53,112}, {49,152}, {14,55}, {14,52}, 
    {53,106}, {53,105}, {53,107}, {53,108}, {53,110}, {53,109}, {53,114}, 
    {17,117}, {17,114}, {17,118}, {53,104}, {17,119}, {53,113}, {17,116}, 
    {53,111}, {17,115}, {57,124}, {57,126}, {57,122}, {57,123}, {57,127}, 
    {57,129}, {57,130}, {57,125}, {57,128}, {20,52}, {61,155}, {61,160}, 
    {61,152}, {61,156}, {61,161}, {61,157}, {22,186}, {23,3}, {22,189}, 
    {61,153}, {61,154}, {25,142}, {23,6}, {22,190}, {23,4}, {22,188}, {61,159}, 
    {23,5}, {61,158}, {23,1}, {22,187}, {23,2}, {61,162}, {65,166}, {65,168}, 
    {23,0}, {65,163}, {25,140}, {65,169}, {25,141}, {65,167}, {65,162}, 
    {65,165}, {65,170}, {25,139}, {65,164}, {28,14}, {69,127}, {69,126}, 
    {28,17}, {28,15}, {69,124}, {28,13}, {69,125}, {28,16}, {73,61}, {73,97}, 
    {73,62}, {73,54}, {73,58}, {73,57}, {73,55}, {30,38}, {30,39}, {30,37}, 
    {73,56}, {73,59}, {73,60}, {30,40}, {32,0}, {76,99}, {31,188}, {76,59}, 
    {31,190}, {32,1}, {31,189}, {76,98}, {76,60}, {31,187}, {76,97}, {76,58}, 
    {76,62}, {76,61}, {79,21}, {33,100}, {33,101}, {33,98}, {33,99}, {79,20}, 
    {81,59}, {34,105}, {81,57}, {81,58}, {83,52}, {83,55}, {83,53}, {83,54}, 
    {84,160}, {35,175}, {86,171}, {87,116}, {36,183}, {87,184}, {87,185}, 
    {37,38}, {4,153}, {11,161}, {57,131}, {61,163}, {4,154}, {7,47}, {9,126}, 
    {46,168}, {46,169}, {14,56}, {57,132}, {20,53}, {23,7}, {32,2}, {76,100}, 
    {81,60}, {83,56}, {4,155}, {44,97}, {44,58}, {44,60}, {44,62}, {44,61}, 
    {44,59}, {9,127}, {11,162}, {46,178}, {46,173}, {46,170}, {46,177}, 
    {11,164}, {11,165}, {46,176}, {46,175}, {46,174}, {46,171}, {46,172}, 
    {46,179}, {11,163}, {14,60}, {14,100}, {14,99}, {49,161}, {49,155}, 
    {49,153}, {14,97}, {53,121}, {14,58}, {49,158}, {49,162}, {49,164}, 
    {49,160}, {49,154}, {14,98}, {49,159}, {49,157}, {14,57}, {49,165}, 
    {14,101}, {49,163}, {14,61}, {49,156}, {14,62}, {14,59}, {17,121}, 
    {53,122}, {53,115}, {53,117}, {53,119}, {53,120}, {53,118}, {17,122}, 
    {53,123}, {53,116}, {57,136}, {20,54}, {57,144}, {57,133}, {57,145}, 
    {57,134}, {57,135}, {57,143}, {57,142}, {57,139}, {17,120}, {20,55}, 
    {57,137}, {57,140}, {20,56}, {57,141}, {57,138}, {61,174}, {61,178}, 
    {23,14}, {65,176}, {23,17}, {61,176}, {61,173}, {61,168}, {61,170}, 
    {61,171}, {61,164}, {23,13}, {23,12}, {23,8}, {61,167}, {23,15}, {23,16}, 
    {61,166}, {61,169}, {61,177}, {23,10}, {23,11}, {61,175}, {61,165}, 
    {65,178}, {25,144}, {65,180}, {61,172}, {65,182}, {25,148}, {65,183}, 
    {65,179}, {25,145}, {65,175}, {25,147}, {65,172}, {65,177}, {65,181}, 
    {65,174}, {23,9}, {65,171}, {25,146}, {65,173}, {28,20}, {69,129}, 
    {28,24}, {28,22}, {25,143}, {69,136}, {69,137}, {28,19}, {69,135}, 
    {69,138}, {28,21}, {28,18}, {69,131}, {69,128}, {69,133}, {69,130}, 
    {28,25}, {69,132}, {69,134}, {28,23}, {30,42}, {73,104}, {73,98}, {73,102}, 
    {73,108}, {73,109}, {73,105}, {73,106}, {73,103}, {73,100}, {30,44}, 
    {30,41}, {73,99}, {73,101}, {30,43}, {73,107}, {32,6}, {76,106}, {76,101}, 
    {32,5}, {32,3}, {76,108}, {32,4}, {76,104}, {76,105}, {76,102}, {76,109}, 
    {79,22}, {32,7}, {76,103}, {76,110}, {76,107}, {79,26}, {79,23}, {33,102}, 
    {79,27}, {79,29}, {79,28}, {79,24}, {79,25}, {81,99}, {34,106}, {81,62}, 
    {81,98}, {81,61}, {81,100}, {81,97}, {83,58}, {83,61}, {83,57}, {35,49}, 
    {83,59}, {83,60}, {83,62}, {35,50}, {35,52}, {35,51}, {84,161}, {84,163}, 
    {84,162}, {85,182}, {87,117}, {87,186}, {4,156}, {42,50}, {9,128}, 
    {44,99}, {9,129}, {9,130}, {44,98}, {46,184}, {46,183}, {11,166}, {11,169}, 
    {11,168}, {11,167}, {46,180}, {46,182}, {46,181}, {49,168}, {14,103}, 
    {49,172}, {49,171}, {49,170}, {49,166}, {14,102}, {14,106}, {14,109}, 
    {14,110}, {49,167}, {49,169}, {14,108}, {14,107}, {14,105}, {14,104}, 
    {53,130}, {53,128}, {53,124}, {17,123}, {53,125}, {17,124}, {53,126}, 
    {53,127}, {53,129}, {17,125}, {57,149}, {57,148}, {57,147}, {57,146}, 
    {61,182}, {23,18}, {61,179}, {61,181}, {23,19}, {23,20}, {61,180}, 
    {66,1}, {65,185}, {65,186}, {25,151}, {25,149}, {25,150}, {66,3}, {66,2}, 
    {65,190}, {65,189}, {65,188}, {65,187}, {66,0}, {65,184}, {69,139}, 
    {69,144}, {69,142}, {69,141}, {69,140}, {69,143}, {73,110}, {30,45}, 
    {32,8}, {76,112}, {32,9}, {76,111}, {79,31}, {79,30}, {33,103}, {81,101}, 
    {34,107}, {83,97}, {36,97}, {85,183}, {87,119}, {87,118}, {40,104}, 
    {11,170}, {11,171}, {53,131}, {23,22}, {4,157}, {7,50}, {7,49}, {7,48}, 
    {44,100}, {9,132}, {9,131}, {11,174}, {46,186}, {46,189}, {46,187}, 
    {11,172}, {11,173}, {46,185}, {46,188}, {14,117}, {49,179}, {49,181}, 
    {49,180}, {14,114}, {49,175}, {14,112}, {14,111}, {14,115}, {14,113}, 
    {14,116}, {49,178}, {49,173}, {49,174}, {49,177}, {49,176}, {53,134}, 
    {53,135}, {53,133}, {53,132}, {17,126}, {20,61}, {57,150}, {57,152}, 
    {57,154}, {20,60}, {20,58}, {57,151}, {20,59}, {57,153}, {20,57}, {62,1}, 
    {61,183}, {61,186}, {61,190}, {23,26}, {61,188}, {61,187}, {61,184}, 
    {61,185}, {23,24}, {62,0}, {23,23}, {23,28}, {23,27}, {23,25}, {61,189}, 
    {66,9}, {66,8}, {66,4}, {25,152}, {66,7}, {66,6}, {25,153}, {28,30}, 
    {69,146}, {28,31}, {28,27}, {28,29}, {28,26}, {28,28}, {73,111}, {30,48}, 
    {73,113}, {73,112}, {66,5}, {30,50}, {30,49}, {30,46}, {30,47}, {76,117}, 
    {76,115}, {32,10}, {76,116}, {76,118}, {76,114}, {76,113}, {79,32}, 
    {33,106}, {33,104}, {33,105}, {81,102}, {81,103}, {34,109}, {34,108}, 
    {83,99}, {83,100}, {83,98}, {85,184}, {85,185}, {4,158}, {41,8}, {7,51}, 
    {44,101}, {9,134}, {9,133}, {44,102}, {47,0}, {11,175}, {46,190}, {11,176}, 
    {14,118}, {49,184}, {49,182}, {49,188}, {14,119}, {49,183}, {49,187}, 
    {49,185}, {49,186}, {53,136}, {53,138}, {17,127}, {53,137}, {17,128}, 
    {20,98}, {20,97}, {20,62}, {57,155}, {62,4}, {23,29}, {23,30}, {62,3}, 
    {62,2}, {66,10}, {25,155}, {25,154}, {66,11}, {66,12}, {28,33}, {28,32}, 
    {73,117}, {69,147}, {69,149}, {69,148}, {73,116}, {73,114}, {73,118}, 
    {73,115}, {30,51}, {76,119}, {32,11}, {76,120}, {76,121}, {33,107}, 
    {33,108}, {35,53}, {36,145}, {4,159}, {47,1}, {49,189}, {14,120}, {18,122}, 
    {18,121}, {20,100}, {57,157}, {20,99}, {57,156}, {62,5}, {25,156}, 
    {66,13}, {25,157}, {35,54}, {83,101}, {5,139}, {9,135}, {44,103}, {11,178}, 
    {11,177}, {47,2}, {47,3}, {50,0}, {50,2}, {14,121}, {50,4}, {50,7}, 
    {50,5}, {50,6}, {50,3}, {49,190}, {14,122}, {50,8}, {50,1}, {53,147}, 
    {17,134}, {53,156}, {17,131}, {53,149}, {17,135}, {17,129}, {53,144}, 
    {53,157}, {53,145}, {53,142}, {17,133}, {17,130}, {53,148}, {53,151}, 
    {17,132}, {53,152}, {17,136}, {53,153}, {53,143}, {53,150}, {53,154}, 
    {53,146}, {53,141}, {53,139}, {53,155}, {57,159}, {57,164}, {57,160}, 
    {57,166}, {20,103}, {53,140}, {57,161}, {20,101}, {57,158}, {20,108}, 
    {57,162}, {20,107}, {57,163}, {20,109}, {20,104}, {20,110}, {20,105}, 
    {20,106}, {20,102}, {57,165}, {23,33}, {62,16}, {62,19}, {62,7}, {62,12}, 
    {62,6}, {23,35}, {62,10}, {62,8}, {23,34}, {62,15}, {62,14}, {62,11}, 
    {62,13}, {62,9}, {25,161}, {62,18}, {23,31}, {62,17}, {66,29}, {25,168}, 
    {66,24}, {25,167}, {66,14}, {66,16}, {25,160}, {66,21}, {66,20}, {66,23}, 
    {25,165}, {66,18}, {66,17}, {25,164}, {25,159}, {66,19}, {25,162}, 
    {66,25}, {66,27}, {66,22}, {66,15}, {25,163}, {28,41}, {25,158}, {66,28}, 
    {69,153}, {28,34}, {69,155}, {28,35}, {28,37}, {69,158}, {69,150}, 
    {25,166}, {69,156}, {69,152}, {23,32}, {28,40}, {28,36}, {28,38}, {28,39}, 
    {30,54}, {69,157}, {69,151}, {28,42}, {69,154}, {73,128}, {73,123}, 
    {73,133}, {30,52}, {73,125}, {30,56}, {73,131}, {73,122}, {73,119}, 
    {73,134}, {32,17}, {30,57}, {73,130}, {73,120}, {30,55}, {73,124}, 
    {30,59}, {73,121}, {73,126}, {30,58}, {73,129}, {73,132}, {76,139}, 
    {76,140}, {76,124}, {76,131}, {76,129}, {32,15}, {76,136}, {73,127}, 
    {76,127}, {76,137}, {32,14}, {76,126}, {76,125}, {76,135}, {76,132}, 
    {76,134}, {76,122}, {76,138}, {32,12}, {76,133}, {76,142}, {76,130}, 
    {32,16}, {32,13}, {76,128}, {76,123}, {76,141}, {30,53}, {79,35}, {79,33}, 
    {79,39}, {33,112}, {79,37}, {79,36}, {33,114}, {79,42}, {33,113}, {79,43}, 
    {79,40}, {33,110}, {79,41}, {79,34}, {33,111}, {33,109}, {81,107}, 
    {79,38}, {79,44}, {81,104}, {34,114}, {34,112}, {81,106}, {81,108}, 
    {81,105}, {34,113}, {34,110}, {34,111}, {34,115}, {35,56}, {83,106}, 
    {83,105}, {83,103}, {83,102}, {83,104}, {35,55}, {35,57}, {35,176}, 
    {84,165}, {84,164}, {85,186}, {85,188}, {85,190}, {85,189}, {85,187}, 
    {36,99}, {36,98}, {36,147}, {36,146}, {36,148}, {86,173}, {86,172}, 
    {87,187}, {87,120}, {87,188}, {37,24}, {37,25}, {88,45}, {37,62}, {5,140}, 
    {44,104}, {47,5}, {47,6}, {47,4}, {11,179}, {47,7}, {47,8}, {47,9}, 
    {50,9}, {50,15}, {50,13}, {14,123}, {50,11}, {50,12}, {50,14}, {50,10}, 
    {17,137}, {53,158}, {17,139}, {53,160}, {17,138}, {53,159}, {57,168}, 
    {20,111}, {57,170}, {57,167}, {53,161}, {20,112}, {57,169}, {62,22}, 
    {23,36}, {62,20}, {23,37}, {62,21}, {23,38}, {25,169}, {66,33}, {66,30}, 
    {66,32}, {25,170}, {25,171}, {66,31}, {69,159}, {69,160}, {28,43}, 
    {69,162}, {69,161}, {1,33}, {73,138}, {73,139}, {73,135}, {73,136}, 
    {30,60}, {30,61}, {73,137}, {32,23}, {32,19}, {32,24}, {32,20}, {32,22}, 
    {32,18}, {32,21}, {33,115}, {76,143}, {81,110}, {81,109}, {83,107}, 
    {35,58}, {35,59}, {86,1}, {84,166}, {86,0}, {87,189}, {88,100}, {5,141}, 
    {7,52}, {44,105}, {9,136}, {11,182}, {47,12}, {11,180}, {47,10}, {11,185}, 
    {11,181}, {11,186}, {11,184}, {47,11}, {11,183}, {14,127}, {14,126}, 
    {50,26}, {14,135}, {50,19}, {50,25}, {14,131}, {50,18}, {50,24}, {50,22}, 
    {14,130}, {14,132}, {50,21}, {14,125}, {50,20}, {14,136}, {14,133}, 
    {14,137}, {14,134}, {50,17}, {50,16}, {50,23}, {14,128}, {14,124}, 
    {14,129}, {53,176}, {53,169}, {20,117}, {53,168}, {17,143}, {17,150}, 
    {17,147}, {17,153}, {17,152}, {17,148}, {53,162}, {53,165}, {53,164}, 
    {17,144}, {53,166}, {53,175}, {17,145}, {53,163}, {53,172}, {53,173}, 
    {17,146}, {53,170}, {17,151}, {17,141}, {17,149}, {53,167}, {17,140}, 
    {53,171}, {53,174}, {57,187}, {57,178}, {20,114}, {57,185}, {57,182}, 
    {57,174}, {57,183}, {20,116}, {57,175}, {57,171}, {23,44}, {57,180}, 
    {20,113}, {57,186}, {20,120}, {20,122}, {57,173}, {20,121}, {57,176}, 
    {20,115}, {57,184}, {57,177}, {57,181}, {57,179}, {20,118}, {57,172}, 
    {20,123}, {17,142}, {20,119}, {20,124}, {23,40}, {62,29}, {62,31}, 
    {62,33}, {62,37}, {62,27}, {62,25}, {23,42}, {62,32}, {62,36}, {62,28}, 
    {62,24}, {62,23}, {62,34}, {62,26}, {62,30}, {23,43}, {23,41}, {62,38}, 
    {23,39}, {62,35}, {66,50}, {25,174}, {66,42}, {28,56}, {66,52}, {25,177}, 
    {66,56}, {25,183}, {66,37}, {66,53}, {66,34}, {66,55}, {66,38}, {25,190}, 
    {25,187}, {66,54}, {66,48}, {25,173}, {25,181}, {25,180}, {25,179}, 
    {25,185}, {66,35}, {25,186}, {66,49}, {25,182}, {25,172}, {66,51}, 
    {25,175}, {25,176}, {25,184}, {66,40}, {66,39}, {66,36}, {66,44}, {66,41}, 
    {66,45}, {25,189}, {66,57}, {25,178}, {66,46}, {66,47}, {66,43}, {25,188}, 
    {69,167}, {28,48}, {28,57}, {28,53}, {69,164}, {28,50}, {28,54}, {69,176}, 
    {28,44}, {69,168}, {28,52}, {69,171}, {69,166}, {28,51}, {28,55}, {69,165}, 
    {28,49}, {69,175}, {28,46}, {69,174}, {69,173}, {28,58}, {68,50}, {28,45}, 
    {69,172}, {69,163}, {28,59}, {69,170}, {28,47}, {69,169}, {30,98}, 
    {30,103}, {30,62}, {73,152}, {73,143}, {73,155}, {73,147}, {73,153}, 
    {30,104}, {30,97}, {73,140}, {73,146}, {73,156}, {73,149}, {73,154}, 
    {73,142}, {73,150}, {30,99}, {73,148}, {30,102}, {30,101}, {73,144}, 
    {73,145}, {73,141}, {73,151}, {30,100}, {73,158}, {73,157}, {76,154}, 
    {76,150}, {32,31}, {76,144}, {32,25}, {32,41}, {76,156}, {32,33}, {32,29}, 
    {76,147}, {32,36}, {32,39}, {76,158}, {32,28}, {76,149}, {32,37}, {76,160}, 
    {76,157}, {76,145}, {32,32}, {32,26}, {32,40}, {76,152}, {32,35}, {76,146}, 
    {32,30}, {76,159}, {32,34}, {32,27}, {76,153}, {32,38}, {76,151}, {76,155}, 
    {79,46}, {79,52}, {33,121}, {79,55}, {33,116}, {33,117}, {79,47}, {79,54}, 
    {79,49}, {33,122}, {33,119}, {79,45}, {33,118}, {79,51}, {79,53}, {33,120}, 
    {79,50}, {79,48}, {81,120}, {34,119}, {34,120}, {34,116}, {34,117}, 
    {81,116}, {81,114}, {81,118}, {34,122}, {81,119}, {81,112}, {81,111}, 
    {81,115}, {81,113}, {34,118}, {81,117}, {83,108}, {35,62}, {35,61}, 
    {83,109}, {83,111}, {83,110}, {35,97}, {84,171}, {84,168}, {84,169}, 
    {84,167}, {84,170}, {35,178}, {84,172}, {35,177}, {86,2}, {36,149}, 
    {36,151}, {86,174}, {36,150}, {87,121}, {88,0}, {88,2}, {87,190}, {88,1}, 
    {37,44}, {5,142}, {11,187}, {49,47}, {14,138}, {17,154}, {57,188}, 
    {57,189}, {73,159}, {32,42}, {76,161}, {33,123}, {81,122}, {81,121}, 
    {35,98}, {84,173}, {86,3}, {36,184}, {41,9}, {9,137}, {7,53}, {47,13}, 
    {50,32}, {50,27}, {50,31}, {50,29}, {14,139}, {50,30}, {50,28}, {53,177}, 
    {57,190}, {58,0}, {62,41}, {62,42}, {23,46}, {23,47}, {62,40}, {62,43}, 
    {62,39}, {23,45}, {26,0}, {23,48}, {66,58}, {28,60}, {69,177}, {28,61}, 
    {30,105}, {73,162}, {73,160}, {73,161}, {76,164}, {76,163}, {76,162}, 
    {81,123}, {34,121}, {81,124}, {86,4}, {36,185}, {87,122}, {5,143}, 
    {9,139}, {9,138}, {47,15}, {11,188}, {47,14}, {50,34}, {14,140}, {53,178}, 
    {50,33}, {17,156}, {53,181}, {53,179}, {53,180}, {17,155}, {58,2}, 
    {58,3}, {58,1}, {23,51}, {62,45}, {62,44}, {62,46}, {23,50}, {23,49}, 
    {69,178}, {69,180}, {28,62}, {69,179}, {73,163}, {30,106}, {79,57}, 
    {79,56}, {34,123}, {81,125}, {34,125}, {34,124}, {83,112}, {84,174}, 
    {35,179}, {5,144}, {47,16}, {11,189}, {50,37}, {14,142}, {50,36}, {50,35}, 
    {14,141}, {53,187}, {53,189}, {53,182}, {53,183}, {17,157}, {53,184}, 
    {17,158}, {53,188}, {53,185}, {53,186}, {17,159}, {20,126}, {20,127}, 
    {58,4}, {62,47}, {62,48}, {66,62}, {26,3}, {26,1}, {26,2}, {66,59}, 
    {66,60}, {66,61}, {69,185}, {69,186}, {28,97}, {69,183}, {69,182}, 
    {69,184}, {69,181}, {30,109}, {73,164}, {30,107}, {30,108}, {76,166}, 
    {32,43}, {76,165}, {78,104}, {79,58}, {79,59}, {33,124}, {33,125}, 
    {32,44}, {81,126}, {81,127}, {83,113}, {35,99}, {5,145}, {5,146}, {11,190}, 
    {9,140}, {14,143}, {47,17}, {20,128}, {5,147}, {12,1}, {47,18}, {47,19}, 
    {12,0}, {12,2}, {5,148}, {47,20}, {14,145}, {50,38}, {14,147}, {14,144}, 
    {14,146}, {54,1}, {17,160}, {54,0}, {53,190}, {62,49}, {66,97}, {28,98}, 
    {30,110}, {73,166}, {73,165}, {76,167}, {84,175}, {5,149}, {42,51}, 
    {44,106}, {12,3}, {47,21}, {50,40}, {14,148}, {50,39}, {14,149}, {54,3}, 
    {17,162}, {54,2}, {54,4}, {17,161}, {58,6}, {58,7}, {58,5}, {20,129}, 
    {23,52}, {23,53}, {26,5}, {66,99}, {66,98}, {26,4}, {69,187}, {69,188}, 
    {73,167}, {32,48}, {32,47}, {32,45}, {32,46}, {32,49}, {79,60}, {33,127}, 
    {33,126}, {81,128}, {83,114}, {36,101}, {36,100}, {5,150}, {48,187}, 
    {23,55}, {20,130}, {23,54}, {26,6}, {5,151}, {40,105}, {5,152}, {5,153}, 
    {44,107}, {42,54}, {42,55}, {7,55}, {42,52}, {7,54}, {7,57}, {42,53}, 
    {7,59}, {7,58}, {42,56}, {7,56}, {9,145}, {9,143}, {44,109}, {9,142}, 
    {9,147}, {9,149}, {9,146}, {44,112}, {44,108}, {9,150}, {9,144}, {7,60}, 
    {9,148}, {44,111}, {44,110}, {9,141}, {47,27}, {12,7}, {12,8}, {47,29}, 
    {47,23}, {47,26}, {47,35}, {47,33}, {12,9}, {47,39}, {12,12}, {47,36}, 
    {47,28}, {47,25}, {58,9}, {47,34}, {12,4}, {47,37}, {47,22}, {47,31}, 
    {12,6}, {12,11}, {47,32}, {12,15}, {12,13}, {47,24}, {12,10}, {47,30}, 
    {12,14}, {12,5}, {47,38}, {14,154}, {14,163}, {14,152}, {14,150}, {50,42}, 
    {14,158}, {14,155}, {50,44}, {14,157}, {50,43}, {50,41}, {50,46}, {14,162}, 
    {14,160}, {58,8}, {50,47}, {50,45}, {14,151}, {14,153}, {14,156}, {14,159}, 
    {14,161}, {54,14}, {17,164}, {54,5}, {54,7}, {54,8}, {54,16}, {54,12}, 
    {54,10}, {54,13}, {54,17}, {17,165}, {17,169}, {54,6}, {54,15}, {17,167}, 
    {17,166}, {54,11}, {54,9}, {17,163}, {17,168}, {20,136}, {58,17}, {58,15}, 
    {20,138}, {58,10}, {62,97}, {20,137}, {58,14}, {58,11}, {20,133}, {20,139}, 
    {58,16}, {20,135}, {58,13}, {26,7}, {20,134}, {58,12}, {20,140}, {20,132}, 
    {20,131}, {62,55}, {62,53}, {62,59}, {62,51}, {62,98}, {62,56}, {62,50}, 
    {23,59}, {23,99}, {62,61}, {62,54}, {23,62}, {23,60}, {62,62}, {23,57}, 
    {23,56}, {62,57}, {23,61}, {20,141}, {62,60}, {62,52}, {23,58}, {23,97}, 
    {23,98}, {26,12}, {26,8}, {26,13}, {66,102}, {66,101}, {66,103}, {26,10}, 
    {66,100}, {26,11}, {66,106}, {66,105}, {66,104}, {26,9}, {70,1}, {70,4}, 
    {28,104}, {70,3}, {28,103}, {28,99}, {28,100}, {28,101}, {70,0}, {69,190}, 
    {28,102}, {70,2}, {69,189}, {73,169}, {73,179}, {30,113}, {30,112}, 
    {73,173}, {73,175}, {73,170}, {73,174}, {73,168}, {73,177}, {30,111}, 
    {73,176}, {73,172}, {73,178}, {73,171}, {32,52}, {76,168}, {76,174}, 
    {32,56}, {32,58}, {32,55}, {32,54}, {32,53}, {32,51}, {76,169}, {76,172}, 
    {32,50}, {76,173}, {32,57}, {76,171}, {76,170}, {33,128}, {33,129}, 
    {79,97}, {79,61}, {79,62}, {81,130}, {81,129}, {34,126}, {83,116}, 
    {35,100}, {83,115}, {84,176}, {86,5}, {36,102}, {88,3}, {88,4}, {36,152}, 
    {5,154}, {9,151}, {58,18}, {26,14}, {32,59}, {76,175}, {5,155}, {14,165}, 
    {14,164}, {58,19}, {73,180}, {5,156}, {12,16}, {58,20}, {58,21}, {58,22}, 
    {26,15}, {30,114}, {5,157}, {9,152}, {47,40}, {14,166}, {50,48}, {17,170}, 
    {58,23}, {23,100}, {26,16}, {30,115}, {32,60}, {33,130}, {83,117}, 
    {5,158}, {9,153}, {14,167}, {54,18}, {20,142}, {26,17}, {66,107}, {70,5}, 
    {5,159}, {20,143}, {62,99}, {26,18}, {5,160}, {44,113}, {47,41}, {12,17}, 
    {50,50}, {14,170}, {14,168}, {14,169}, {14,171}, {50,49}, {54,23}, 
    {54,20}, {54,22}, {17,171}, {17,173}, {17,172}, {54,19}, {17,174}, 
    {54,21}, {58,24}, {58,25}, {58,26}, {62,102}, {62,103}, {62,101}, {62,104}, 
    {23,101}, {62,100}, {26,19}, {70,10}, {70,6}, {70,9}, {70,11}, {70,8}, 
    {70,7}, {73,181}, {73,182}, {73,183}, {30,116}, {30,117}, {76,177}, 
    {76,176}, {76,178}, {79,99}, {79,98}, {81,132}, {81,133}, {81,131}, 
    {35,101}, {83,118}, {83,119}, {86,6}, {86,175}, {87,123}, {5,161}, 
    {7,61}, {32,61}, {5,162}, {54,24}, {58,27}, {37,1}, {41,10}, {41,11}, 
    {41,13}, {5,163}, {41,14}, {41,12}, {42,98}, {42,99}, {42,59}, {42,97}, 
    {7,97}, {7,98}, {42,60}, {42,58}, {42,57}, {42,61}, {7,62}, {42,62}, 
    {47,42}, {44,118}, {9,156}, {44,117}, {44,119}, {9,155}, {44,124}, 
    {9,159}, {44,114}, {44,128}, {44,134}, {9,166}, {44,131}, {9,163}, 
    {44,121}, {44,127}, {44,129}, {44,116}, {9,162}, {9,157}, {44,122}, 
    {9,164}, {9,167}, {9,161}, {9,154}, {44,126}, {44,120}, {44,133}, {9,169}, 
    {9,165}, {9,160}, {44,125}, {14,172}, {44,123}, {9,158}, {9,168}, {44,115}, 
    {44,130}, {44,132}, {12,34}, {12,28}, {12,36}, {12,33}, {47,49}, {47,52}, 
    {12,29}, {47,43}, {12,22}, {12,32}, {12,35}, {12,37}, {47,98}, {47,55}, 
    {12,21}, {47,97}, {12,25}, {12,23}, {12,18}, {47,47}, {47,62}, {47,51}, 
    {47,54}, {47,101}, {12,38}, {47,61}, {12,30}, {47,56}, {47,100}, {47,53}, 
    {47,57}, {47,60}, {47,45}, {47,99}, {47,59}, {47,44}, {47,48}, {12,31}, 
    {12,26}, {12,19}, {12,24}, {12,20}, {12,39}, {47,46}, {50,101}, {12,27}, 
    {47,58}, {45,1}, {50,104}, {14,186}, {50,54}, {50,99}, {50,61}, {50,114}, 
    {50,106}, {50,62}, {50,105}, {50,56}, {50,60}, {50,117}, {14,189}, 
    {50,109}, {50,100}, {14,173}, {50,115}, {50,52}, {50,108}, {14,188}, 
    {14,183}, {50,51}, {14,181}, {14,180}, {14,185}, {50,107}, {14,177}, 
    {14,184}, {47,50}, {58,28}, {50,102}, {50,58}, {14,187}, {50,59}, {50,97}, 
    {14,190}, {50,53}, {50,111}, {50,110}, {50,118}, {14,179}, {14,176}, 
    {50,116}, {50,112}, {50,103}, {50,98}, {14,182}, {14,178}, {50,55}, 
    {14,174}, {50,113}, {14,175}, {50,57}, {54,30}, {54,32}, {54,37}, {54,57}, 
    {17,188}, {17,178}, {54,29}, {17,189}, {17,190}, {54,40}, {54,47}, 
    {54,53}, {54,34}, {54,41}, {18,0}, {54,55}, {54,50}, {17,186}, {17,184}, 
    {54,46}, {54,42}, {54,28}, {17,175}, {54,33}, {54,25}, {17,183}, {17,185}, 
    {54,38}, {54,35}, {17,180}, {54,51}, {17,177}, {54,36}, {54,58}, {54,44}, 
    {54,43}, {17,176}, {17,187}, {17,179}, {54,26}, {54,31}, {54,48}, {54,54}, 
    {18,1}, {54,27}, {54,39}, {54,45}, {17,182}, {54,56}, {54,49}, {54,52}, 
    {17,181}, {58,44}, {58,32}, {20,151}, {58,61}, {58,103}, {58,106}, 
    {20,149}, {58,40}, {58,99}, {58,41}, {58,55}, {20,162}, {58,51}, {20,159}, 
    {58,52}, {58,29}, {58,100}, {20,168}, {58,97}, {58,53}, {58,108}, {58,48}, 
    {62,136}, {58,111}, {20,166}, {58,46}, {58,58}, {20,169}, {20,148}, 
    {58,50}, {58,109}, {58,43}, {58,36}, {58,47}, {58,35}, {58,33}, {20,144}, 
    {58,101}, {58,42}, {58,104}, {58,105}, {20,152}, {20,157}, {20,153}, 
    {20,161}, {58,62}, {20,154}, {58,54}, {58,38}, {20,146}, {58,30}, {58,98}, 
    {58,107}, {58,37}, {20,160}, {58,112}, {58,49}, {58,45}, {20,145}, 
    {20,165}, {58,60}, {20,167}, {58,56}, {20,156}, {20,150}, {20,158}, 
    {20,147}, {20,164}, {58,57}, {58,39}, {58,59}, {58,34}, {58,102}, {58,110}, 
    {58,31}, {62,135}, {62,157}, {23,21}, {62,140}, {62,138}, {62,117}, 
    {23,105}, {62,133}, {62,153}, {62,129}, {23,113}, {62,152}, {62,127}, 
    {20,163}, {62,143}, {62,128}, {62,150}, {23,112}, {23,104}, {62,106}, 
    {62,114}, {62,139}, {62,131}, {62,156}, {62,134}, {23,118}, {62,151}, 
    {23,109}, {62,137}, {62,145}, {62,118}, {62,144}, {62,161}, {62,113}, 
    {62,146}, {62,159}, {62,107}, {20,155}, {62,121}, {62,120}, {23,111}, 
    {62,124}, {62,126}, {62,141}, {62,158}, {23,114}, {23,115}, {62,112}, 
    {23,107}, {62,116}, {62,154}, {23,116}, {23,108}, {23,110}, {23,117}, 
    {62,160}, {62,147}, {62,142}, {62,123}, {62,122}, {23,106}, {62,108}, 
    {23,103}, {62,132}, {62,109}, {62,130}, {62,119}, {62,155}, {23,102}, 
    {62,115}, {62,111}, {62,149}, {62,110}, {26,32}, {66,147}, {66,130}, 
    {66,108}, {66,138}, {26,24}, {66,123}, {66,133}, {26,27}, {66,126}, 
    {26,25}, {66,111}, {66,141}, {66,110}, {66,129}, {66,109}, {66,127}, 
    {66,136}, {66,134}, {66,122}, {66,117}, {66,115}, {66,116}, {66,135}, 
    {66,146}, {66,124}, {26,26}, {66,119}, {66,139}, {26,29}, {66,118}, 
    {66,112}, {66,128}, {26,33}, {26,21}, {26,30}, {66,120}, {66,114}, 
    {26,23}, {62,148}, {26,22}, {66,131}, {26,20}, {26,35}, {26,28}, {66,132}, 
    {66,121}, {66,113}, {66,140}, {66,125}, {26,34}, {66,144}, {26,31}, 
    {66,143}, {66,137}, {66,142}, {66,145}, {70,51}, {70,52}, {70,39}, 
    {70,38}, {70,34}, {28,116}, {28,108}, {70,54}, {70,53}, {62,105}, {70,31}, 
    {70,35}, {70,29}, {70,48}, {70,33}, {70,55}, {70,26}, {70,24}, {70,36}, 
    {70,46}, {70,41}, {28,118}, {70,15}, {70,45}, {28,119}, {62,125}, {70,27}, 
    {70,18}, {70,21}, {70,59}, {70,28}, {70,19}, {70,17}, {70,14}, {28,112}, 
    {70,37}, {28,111}, {28,115}, {70,32}, {70,40}, {28,105}, {70,56}, {70,60}, 
    {28,107}, {70,23}, {70,43}, {70,47}, {70,20}, {70,57}, {28,114}, {28,113}, 
    {70,12}, {28,117}, {70,50}, {70,22}, {70,42}, {70,16}, {70,30}, {70,25}, 
    {28,109}, {28,110}, {70,44}, {70,61}, {70,58}, {70,49}, {70,13}, {28,106}, 
    {74,9}, {74,0}, {74,3}, {30,123}, {74,5}, {73,185}, {74,1}, {74,7}, 
    {30,120}, {30,124}, {30,118}, {73,187}, {74,12}, {74,6}, {73,188}, 
    {74,21}, {74,15}, {73,184}, {77,6}, {73,190}, {30,119}, {74,10}, {74,20}, 
    {30,127}, {74,17}, {73,189}, {74,4}, {74,8}, {74,2}, {74,22}, {74,19}, 
    {74,16}, {30,121}, {30,122}, {30,126}, {73,186}, {74,23}, {30,125}, 
    {74,13}, {74,11}, {74,14}, {77,19}, {77,0}, {77,5}, {77,18}, {77,4}, 
    {76,187}, {77,1}, {32,98}, {76,180}, {77,13}, {77,15}, {76,179}, {32,97}, 
    {77,17}, {77,9}, {32,104}, {77,16}, {77,2}, {32,106}, {76,185}, {74,18}, 
    {77,10}, {77,7}, {76,181}, {77,21}, {32,100}, {32,101}, {76,183}, {77,8}, 
    {77,20}, {77,11}, {76,189}, {32,103}, {32,99}, {77,12}, {76,190}, {77,22}, 
    {76,184}, {77,3}, {77,14}, {76,186}, {76,188}, {33,139}, {76,182}, 
    {32,105}, {33,132}, {32,62}, {32,102}, {33,136}, {79,115}, {79,105}, 
    {79,100}, {79,106}, {79,116}, {79,120}, {79,119}, {33,138}, {33,137}, 
    {79,107}, {79,121}, {79,110}, {79,102}, {79,104}, {79,103}, {79,109}, 
    {79,114}, {79,101}, {79,108}, {79,113}, {33,135}, {79,111}, {33,133}, 
    {79,112}, {33,131}, {33,134}, {81,149}, {79,117}, {34,130}, {81,141}, 
    {81,145}, {81,137}, {81,140}, {81,148}, {34,128}, {81,153}, {81,146}, 
    {81,138}, {81,154}, {81,147}, {34,131}, {34,132}, {81,151}, {81,139}, 
    {34,127}, {34,129}, {81,134}, {81,142}, {81,136}, {81,152}, {81,150}, 
    {81,135}, {81,143}, {83,126}, {34,133}, {81,144}, {35,103}, {35,105}, 
    {35,102}, {83,131}, {83,123}, {83,121}, {83,125}, {83,122}, {83,127}, 
    {83,129}, {35,106}, {35,108}, {83,128}, {35,109}, {35,107}, {83,130}, 
    {35,104}, {35,180}, {84,177}, {84,183}, {35,182}, {83,124}, {84,182}, 
    {84,189}, {84,180}, {84,187}, {84,186}, {83,120}, {84,181}, {79,118}, 
    {84,190}, {84,179}, {84,184}, {84,188}, {84,178}, {86,10}, {35,181}, 
    {84,185}, {86,180}, {86,11}, {86,9}, {86,7}, {86,8}, {86,12}, {36,153}, 
    {86,178}, {86,176}, {86,181}, {86,179}, {86,182}, {36,154}, {86,177}, 
    {87,124}, {88,5}, {88,6}, {88,7}, {88,135}, {88,125}, {41,15}, {9,170}, 
    {12,40}, {50,120}, {50,119}, {15,0}, {18,2}, {54,60}, {54,59}, {20,170}, 
    {23,120}, {23,119}, {23,121}, {66,148}, {70,62}, {74,24}, {74,26}, 
    {74,25}, {32,107}, {77,23}, {79,122}, {88,101}, {5,164}, {44,137}, 
    {44,138}, {44,136}, {44,135}, {9,171}, {47,105}, {47,103}, {47,102}, 
    {12,41}, {12,43}, {12,42}, {47,104}, {50,132}, {50,129}, {50,127}, 
    {15,1}, {50,130}, {15,6}, {50,124}, {50,139}, {50,141}, {50,125}, {15,3}, 
    {50,142}, {50,137}, {50,123}, {50,128}, {50,138}, {50,134}, {50,131}, 
    {15,8}, {50,140}, {50,126}, {50,135}, {50,121}, {15,7}, {15,4}, {50,133}, 
    {50,136}, {50,122}, {15,5}, {15,2}, {18,12}, {54,101}, {18,11}, {54,104}, 
    {54,107}, {18,8}, {18,6}, {54,62}, {54,105}, {54,103}, {54,100}, {54,108}, 
    {54,109}, {54,111}, {54,112}, {54,61}, {18,5}, {54,98}, {54,97}, {54,110}, 
    {18,7}, {54,99}, {18,9}, {18,4}, {54,102}, {18,13}, {18,10}, {54,106}, 
    {20,177}, {58,127}, {58,116}, {20,174}, {62,167}, {58,125}, {58,113}, 
    {20,172}, {58,118}, {20,175}, {58,122}, {58,120}, {20,178}, {20,171}, 
    {58,114}, {58,117}, {20,176}, {58,115}, {58,126}, {58,124}, {58,119}, 
    {58,121}, {58,123}, {20,173}, {62,168}, {62,174}, {62,164}, {62,170}, 
    {23,122}, {62,166}, {23,128}, {23,127}, {23,126}, {62,173}, {23,129}, 
    {23,130}, {62,163}, {62,176}, {23,131}, {23,125}, {23,124}, {62,172}, 
    {23,132}, {62,162}, {62,165}, {62,175}, {62,171}, {66,180}, {66,169}, 
    {23,123}, {26,42}, {66,157}, {66,178}, {66,158}, {26,37}, {66,155}, 
    {66,164}, {66,156}, {26,39}, {66,150}, {66,177}, {26,40}, {66,174}, 
    {66,175}, {66,151}, {26,45}, {66,166}, {66,160}, {66,167}, {66,154}, 
    {66,179}, {66,171}, {66,165}, {66,149}, {26,41}, {66,172}, {26,44}, 
    {66,176}, {66,170}, {26,38}, {66,168}, {66,162}, {26,36}, {66,153}, 
    {66,161}, {66,173}, {66,159}, {66,163}, {28,129}, {62,169}, {70,114}, 
    {70,123}, {70,113}, {70,109}, {70,106}, {28,130}, {70,104}, {26,43}, 
    {70,97}, {28,128}, {70,103}, {28,127}, {70,108}, {70,105}, {70,121}, 
    {70,116}, {70,110}, {70,115}, {28,123}, {70,107}, {70,126}, {70,98}, 
    {70,99}, {70,122}, {28,124}, {70,127}, {28,126}, {70,128}, {70,112}, 
    {66,152}, {70,118}, {70,111}, {70,120}, {70,117}, {70,102}, {28,121}, 
    {70,125}, {28,122}, {70,100}, {28,125}, {74,36}, {70,119}, {70,124}, 
    {74,33}, {28,120}, {30,128}, {74,43}, {74,39}, {74,37}, {74,32}, {74,47}, 
    {30,132}, {74,28}, {74,40}, {74,41}, {74,31}, {74,30}, {74,44}, {74,34}, 
    {74,29}, {74,35}, {74,46}, {74,27}, {74,45}, {74,42}, {30,130}, {30,129}, 
    {30,131}, {74,38}, {79,139}, {77,25}, {32,113}, {77,29}, {77,26}, {77,33}, 
    {77,39}, {77,28}, {77,48}, {32,110}, {77,42}, {77,31}, {77,43}, {77,38}, 
    {77,45}, {77,30}, {32,115}, {32,114}, {77,32}, {77,46}, {77,24}, {77,44}, 
    {32,108}, {77,36}, {77,35}, {77,40}, {77,27}, {32,112}, {32,116}, {77,34}, 
    {77,41}, {32,117}, {77,37}, {32,109}, {32,111}, {79,135}, {79,133}, 
    {79,140}, {79,137}, {79,141}, {79,126}, {79,134}, {79,145}, {77,47}, 
    {79,130}, {33,143}, {70,101}, {79,125}, {79,138}, {79,132}, {79,129}, 
    {79,124}, {79,123}, {79,144}, {79,128}, {79,127}, {33,141}, {79,136}, 
    {33,140}, {33,142}, {79,131}, {79,143}, {81,158}, {81,159}, {34,137}, 
    {81,156}, {34,134}, {81,164}, {34,138}, {81,166}, {81,155}, {79,142}, 
    {81,168}, {81,157}, {34,135}, {81,163}, {81,165}, {81,160}, {81,167}, 
    {81,162}, {81,161}, {34,136}, {83,133}, {83,134}, {83,136}, {35,110}, 
    {35,111}, {83,137}, {83,135}, {83,132}, {85,2}, {85,5}, {85,1}, {35,186}, 
    {85,3}, {35,185}, {35,184}, {35,183}, {85,4}, {86,17}, {86,15}, {86,14}, 
    {85,0}, {86,16}, {85,6}, {86,13}, {86,185}, {86,151}, {86,183}, {36,155}, 
    {86,184}, {86,186}, {87,127}, {36,186}, {87,126}, {87,125}, {36,187}, 
    {37,26}, {88,46}, {88,103}, {88,102}, {88,104}, {5,165}, {47,106}, 
    {50,143}, {50,144}, {58,128}, {85,7}, {87,128}, {5,166}, {12,44}, {47,107}, 
    {54,113}, {18,14}, {58,130}, {58,129}, {20,179}, {23,133}, {70,129}, 
    {28,131}, {28,132}, {30,133}, {36,188}, {5,167}, {47,108}, {9,173}, 
    {47,110}, {47,109}, {12,45}, {50,145}, {50,152}, {15,9}, {50,150}, 
    {50,148}, {50,155}, {50,146}, {50,147}, {15,10}, {15,14}, {50,156}, 
    {15,13}, {50,154}, {50,151}, {50,149}, {15,11}, {15,12}, {50,153}, 
    {18,16}, {54,114}, {18,21}, {54,130}, {18,20}, {54,132}, {54,120}, 
    {18,18}, {54,131}, {54,115}, {18,19}, {54,127}, {54,123}, {54,125}, 
    {54,119}, {54,126}, {18,15}, {54,122}, {54,121}, {54,117}, {54,128}, 
    {54,124}, {54,116}, {54,118}, {18,17}, {54,129}, {20,182}, {58,141}, 
    {58,137}, {58,139}, {58,134}, {58,133}, {58,131}, {58,138}, {58,140}, 
    {58,136}, {58,135}, {20,180}, {20,181}, {58,143}, {23,141}, {62,178}, 
    {62,184}, {62,179}, {62,180}, {62,185}, {23,143}, {23,135}, {23,142}, 
    {62,177}, {58,132}, {23,138}, {23,136}, {62,183}, {62,182}, {23,137}, 
    {23,139}, {62,181}, {23,134}, {23,140}, {66,182}, {26,52}, {67,2}, 
    {67,1}, {66,187}, {26,54}, {67,0}, {66,183}, {66,184}, {26,46}, {26,48}, 
    {66,189}, {66,181}, {26,50}, {26,49}, {66,185}, {66,190}, {66,188}, 
    {26,51}, {66,186}, {58,142}, {26,47}, {70,130}, {70,137}, {28,134}, 
    {70,141}, {28,138}, {70,133}, {70,131}, {70,140}, {28,133}, {70,139}, 
    {28,135}, {28,136}, {70,132}, {28,137}, {70,138}, {70,134}, {70,135}, 
    {70,136}, {26,53}, {74,48}, {74,60}, {30,138}, {74,55}, {74,57}, {30,136}, 
    {74,49}, {74,53}, {74,56}, {30,134}, {30,137}, {74,59}, {74,51}, {74,52}, 
    {74,58}, {74,50}, {74,54}, {30,135}, {77,50}, {77,49}, {32,119}, {77,55}, 
    {32,121}, {32,118}, {77,51}, {32,122}, {77,52}, {77,53}, {77,56}, {32,120}, 
    {79,150}, {79,153}, {79,147}, {79,149}, {79,148}, {79,151}, {79,152}, 
    {77,54}, {79,146}, {34,141}, {81,172}, {81,175}, {81,177}, {81,170}, 
    {81,171}, {81,174}, {81,176}, {34,142}, {34,140}, {34,139}, {81,173}, 
    {81,169}, {83,138}, {35,112}, {83,139}, {85,9}, {35,187}, {85,11}, 
    {35,188}, {85,8}, {85,10}, {36,104}, {86,18}, {36,103}, {86,189}, {86,188}, 
    {86,187}, {88,8}, {88,9}, {88,11}, {88,10}, {41,16}, {5,168}, {12,46}, 
    {54,133}, {20,183}, {62,186}, {33,144}, {81,178}, {7,99}, {18,23}, 
    {18,22}, {58,144}, {20,184}, {58,146}, {58,145}, {62,187}, {23,144}, 
    {67,3}, {67,6}, {67,5}, {67,4}, {70,142}, {70,144}, {70,143}, {30,140}, 
    {30,139}, {32,123}, {77,57}, {77,59}, {77,58}, {33,145}, {81,180}, 
    {81,179}, {83,140}, {35,113}, {35,189}, {86,20}, {86,19}, {37,27}, 
    {7,100}, {47,111}, {12,47}, {54,136}, {54,134}, {54,135}, {58,148}, 
    {58,149}, {63,3}, {58,147}, {62,188}, {63,1}, {63,0}, {63,2}, {23,145}, 
    {62,190}, {62,189}, {63,4}, {67,9}, {67,7}, {67,8}, {70,147}, {70,145}, 
    {70,146}, {74,61}, {77,60}, {77,61}, {33,146}, {81,181}, {83,141}, 
    {35,114}, {85,12}, {86,21}, {36,105}, {86,190}, {88,12}, {7,101}, {12,49}, 
    {12,50}, {47,112}, {47,113}, {12,48}, {15,20}, {15,18}, {15,17}, {15,24}, 
    {15,16}, {15,25}, {50,157}, {15,22}, {15,19}, {15,23}, {15,21}, {15,15}, 
    {18,31}, {18,25}, {54,140}, {18,30}, {18,32}, {18,26}, {18,27}, {54,138}, 
    {18,24}, {54,139}, {18,29}, {54,137}, {18,28}, {21,4}, {21,6}, {58,152}, 
    {21,5}, {20,185}, {20,189}, {23,164}, {63,9}, {58,154}, {20,190}, {58,157}, 
    {58,158}, {21,3}, {58,160}, {58,162}, {58,163}, {58,151}, {58,150}, 
    {58,164}, {21,2}, {58,161}, {58,159}, {21,0}, {20,187}, {21,7}, {58,155}, 
    {58,156}, {58,153}, {21,1}, {20,188}, {20,186}, {63,8}, {23,159}, {23,154}, 
    {23,149}, {23,165}, {23,150}, {23,146}, {23,161}, {23,158}, {23,160}, 
    {23,151}, {23,156}, {23,147}, {23,148}, {63,16}, {63,13}, {63,5}, {63,10}, 
    {23,162}, {63,17}, {23,163}, {23,153}, {63,7}, {63,15}, {63,11}, {63,14}, 
    {63,12}, {23,157}, {63,6}, {23,152}, {67,12}, {26,56}, {26,59}, {67,14}, 
    {26,101}, {67,13}, {26,61}, {28,143}, {67,15}, {26,100}, {67,11}, {26,102}, 
    {26,57}, {23,155}, {26,60}, {26,58}, {26,62}, {26,98}, {26,55}, {26,103}, 
    {26,99}, {26,97}, {67,10}, {28,150}, {28,146}, {28,153}, {70,150}, 
    {28,154}, {70,162}, {70,155}, {28,139}, {70,163}, {70,157}, {28,149}, 
    {70,158}, {28,148}, {70,161}, {28,142}, {70,159}, {70,149}, {28,141}, 
    {74,106}, {28,147}, {28,144}, {28,152}, {70,148}, {70,152}, {28,140}, 
    {70,151}, {70,153}, {70,154}, {28,151}, {70,156}, {70,160}, {70,164}, 
    {28,155}, {30,146}, {74,101}, {74,107}, {74,104}, {74,62}, {74,108}, 
    {74,97}, {74,103}, {30,141}, {30,147}, {74,109}, {30,143}, {30,153}, 
    {30,148}, {74,111}, {74,105}, {30,144}, {74,98}, {30,154}, {74,99}, 
    {74,100}, {30,155}, {30,152}, {28,145}, {30,142}, {74,112}, {30,156}, 
    {30,149}, {74,110}, {30,145}, {30,150}, {30,151}, {32,131}, {77,100}, 
    {77,109}, {77,106}, {77,108}, {32,128}, {77,101}, {77,107}, {32,124}, 
    {77,103}, {32,132}, {77,99}, {77,104}, {77,111}, {74,102}, {77,105}, 
    {77,98}, {32,125}, {77,97}, {32,126}, {77,112}, {32,127}, {77,110}, 
    {32,130}, {77,62}, {32,129}, {77,102}, {79,156}, {79,170}, {79,165}, 
    {79,167}, {79,155}, {33,147}, {79,154}, {33,150}, {33,149}, {79,169}, 
    {79,161}, {79,158}, {79,164}, {79,157}, {79,159}, {79,168}, {79,166}, 
    {33,148}, {79,173}, {79,171}, {79,162}, {79,172}, {79,163}, {81,185}, 
    {34,143}, {82,1}, {85,15}, {34,150}, {79,160}, {81,183}, {34,146}, 
    {81,184}, {81,189}, {34,148}, {34,149}, {81,182}, {82,0}, {82,2}, {81,186}, 
    {81,188}, {81,190}, {81,187}, {82,3}, {34,145}, {34,151}, {34,147}, 
    {34,144}, {83,144}, {35,119}, {83,142}, {83,146}, {83,147}, {35,117}, 
    {83,148}, {83,145}, {83,143}, {35,120}, {35,116}, {83,149}, {35,118}, 
    {35,115}, {35,190}, {36,0}, {85,14}, {85,13}, {85,16}, {85,17}, {36,1}, 
    {86,22}, {86,27}, {36,106}, {86,24}, {86,23}, {86,26}, {86,25}, {87,3}, 
    {36,156}, {87,2}, {87,0}, {87,1}, {36,190}, {36,189}, {87,129}, {87,130}, 
    {37,0}, {88,13}, {88,14}, {37,39}, {37,45}, {88,105}, {88,136}, {7,102}, 
    {54,141}, {54,142}, {63,18}, {67,16}, {70,165}, {32,134}, {32,133}, 
    {79,174}, {82,4}, {87,4}, {7,103}, {50,158}, {15,26}, {18,33}, {63,20}, 
    {63,19}, {28,156}, {70,166}, {28,157}, {77,113}, {33,151}, {37,54}, 
    {7,104}, {44,139}, {50,159}, {18,34}, {54,143}, {54,144}, {58,165}, 
    {21,8}, {23,166}, {63,22}, {63,21}, {63,23}, {67,17}, {67,18}, {26,104}, 
    {30,157}, {28,158}, {30,158}, {77,117}, {77,114}, {77,116}, {77,115}, 
    {32,135}, {79,175}, {82,6}, {82,5}, {42,100}, {15,28}, {15,27}, {50,160}, 
    {54,145}, {58,167}, {58,166}, {21,9}, {63,25}, {63,26}, {63,24}, {23,168}, 
    {23,167}, {26,106}, {26,105}, {70,167}, {74,115}, {74,113}, {74,114}, 
    {30,159}, {77,119}, {77,118}, {79,178}, {79,177}, {79,176}, {82,7}, 
    {88,106}, {7,105}, {12,51}, {12,52}, {15,29}, {15,30}, {50,162}, {50,161}, 
    {54,146}, {18,40}, {18,38}, {18,35}, {18,39}, {18,37}, {18,36}, {21,10}, 
    {58,170}, {23,173}, {21,12}, {21,17}, {58,172}, {21,19}, {21,18}, {21,21}, 
    {58,171}, {58,168}, {21,15}, {21,11}, {21,13}, {58,169}, {21,20}, {21,16}, 
    {21,14}, {23,175}, {23,174}, {23,172}, {23,176}, {23,170}, {23,171}, 
    {23,169}, {63,27}, {67,20}, {26,108}, {26,109}, {26,107}, {67,19}, 
    {67,21}, {70,170}, {70,172}, {28,167}, {70,173}, {28,160}, {70,169}, 
    {28,159}, {28,169}, {28,165}, {28,166}, {28,162}, {70,168}, {28,161}, 
    {70,174}, {70,171}, {28,168}, {28,163}, {28,164}, {74,117}, {74,119}, 
    {74,118}, {74,120}, {30,160}, {74,116}, {32,139}, {77,120}, {32,136}, 
    {32,140}, {32,138}, {32,137}, {79,179}, {79,182}, {79,181}, {79,180}, 
    {33,152}, {82,8}, {82,9}, {34,152}, {82,10}, {34,153}, {35,122}, {35,121}, 
    {85,18}, {36,2}, {85,19}, {86,28}, {36,107}, {36,108}, {87,5}, {37,2}, 
    {7,106}, {18,42}, {18,41}, {63,28}, {63,29}, {26,110}, {74,121}, {28,170}, 
    {74,122}, {77,121}, {7,107}, {47,114}, {12,54}, {12,53}, {50,163}, 
    {15,31}, {50,164}, {54,149}, {54,148}, {54,147}, {58,174}, {21,24}, 
    {58,173}, {21,23}, {58,175}, {21,22}, {63,31}, {63,34}, {63,32}, {63,33}, 
    {63,37}, {63,30}, {63,38}, {63,35}, {63,36}, {26,112}, {67,22}, {26,111}, 
    {70,178}, {70,176}, {28,171}, {70,175}, {70,177}, {28,172}, {74,123}, 
    {74,124}, {32,141}, {82,12}, {82,14}, {82,11}, {82,13}, {83,150}, {85,20}, 
    {88,47}, {7,108}, {12,55}, {50,165}, {50,167}, {50,166}, {54,152}, 
    {18,44}, {54,150}, {18,43}, {54,153}, {54,154}, {54,151}, {58,187}, 
    {21,32}, {58,179}, {58,185}, {21,27}, {21,30}, {58,178}, {21,25}, {58,182}, 
    {63,44}, {21,29}, {58,177}, {58,183}, {58,180}, {58,186}, {58,176}, 
    {58,184}, {21,28}, {21,31}, {58,181}, {21,26}, {23,178}, {63,40}, {23,177}, 
    {63,47}, {63,46}, {23,184}, {23,185}, {63,48}, {23,179}, {63,45}, {23,183}, 
    {63,50}, {63,41}, {63,43}, {23,180}, {63,39}, {63,42}, {63,49}, {23,181}, 
    {63,51}, {23,182}, {26,113}, {67,27}, {67,33}, {67,25}, {67,34}, {67,24}, 
    {67,29}, {67,35}, {67,32}, {67,31}, {67,30}, {67,23}, {67,28}, {67,26}, 
    {28,177}, {28,174}, {70,187}, {71,1}, {71,3}, {71,0}, {70,184}, {70,186}, 
    {71,5}, {71,2}, {70,188}, {71,6}, {70,185}, {71,4}, {28,175}, {28,181}, 
    {28,179}, {70,179}, {28,180}, {28,176}, {70,180}, {70,182}, {70,181}, 
    {70,189}, {70,190}, {28,178}, {28,173}, {70,183}, {74,134}, {30,162}, 
    {74,125}, {30,163}, {30,166}, {74,130}, {74,127}, {30,165}, {74,131}, 
    {74,132}, {74,126}, {74,135}, {74,128}, {74,133}, {30,164}, {30,161}, 
    {74,129}, {77,127}, {32,144}, {32,142}, {32,145}, {32,143}, {77,126}, 
    {77,123}, {77,122}, {77,125}, {77,124}, {80,5}, {33,158}, {79,187}, 
    {79,186}, {33,153}, {80,1}, {80,0}, {79,183}, {80,3}, {79,188}, {33,157}, 
    {79,185}, {80,2}, {79,184}, {33,154}, {33,156}, {79,189}, {33,155}, 
    {79,190}, {80,4}, {82,18}, {34,158}, {82,15}, {82,19}, {34,155}, {82,17}, 
    {34,160}, {34,157}, {82,16}, {34,159}, {82,20}, {34,154}, {35,124}, 
    {35,126}, {83,153}, {35,125}, {83,151}, {34,156}, {83,152}, {35,123}, 
    {36,3}, {36,5}, {85,22}, {36,4}, {85,21}, {86,33}, {36,109}, {86,32}, 
    {36,110}, {86,30}, {86,29}, {86,34}, {86,35}, {87,6}, {86,31}, {87,134}, 
    {87,131}, {87,132}, {87,133}, {37,28}, {88,17}, {88,16}, {88,15}, {88,48}, 
    {88,126}, {88,107}, {37,46}, {7,109}, {15,32}, {23,186}, {28,182}, 
    {74,136}, {33,159}, {82,21}, {88,108}, {7,110}, {9,174}, {12,57}, {12,56}, 
    {15,35}, {50,168}, {15,33}, {50,169}, {15,34}, {54,159}, {54,155}, 
    {18,45}, {54,158}, {54,157}, {54,156}, {18,46}, {54,160}, {54,161}, 
    {59,3}, {59,1}, {59,5}, {59,6}, {59,12}, {59,8}, {59,10}, {59,2}, {58,188}, 
    {59,9}, {59,11}, {59,4}, {59,7}, {58,189}, {21,34}, {59,0}, {58,190}, 
    {21,33}, {21,35}, {23,189}, {63,53}, {63,55}, {63,54}, {63,59}, {23,187}, 
    {63,56}, {63,52}, {63,57}, {63,58}, {23,188}, {23,190}, {63,60}, {67,39}, 
    {67,38}, {67,36}, {67,37}, {26,115}, {26,117}, {26,114}, {26,116}, 
    {71,13}, {71,14}, {71,9}, {71,10}, {28,184}, {28,189}, {28,183}, {28,190}, 
    {28,185}, {71,11}, {71,12}, {71,8}, {29,0}, {28,187}, {28,186}, {28,188}, 
    {71,7}, {74,138}, {30,168}, {74,140}, {30,170}, {74,143}, {74,139}, 
    {74,137}, {74,142}, {30,169}, {74,141}, {30,167}, {32,147}, {32,150}, 
    {77,129}, {32,148}, {77,128}, {32,146}, {32,149}, {80,6}, {80,7}, {80,8}, 
    {33,160}, {80,9}, {33,161}, {34,162}, {82,24}, {82,25}, {82,23}, {82,22}, 
    {82,26}, {34,161}, {83,157}, {83,155}, {83,156}, {83,158}, {83,154}, 
    {83,159}, {85,24}, {85,25}, {85,23}, {36,6}, {86,36}, {36,111}, {86,37}, 
    {87,8}, {87,7}, {7,111}, {21,36}, {24,0}, {26,118}, {30,172}, {30,171}, 
    {34,163}, {35,60}, {36,7}, {7,112}, {15,36}, {24,1}, {82,27}, {42,102}, 
    {7,113}, {7,116}, {7,115}, {7,114}, {42,101}, {44,141}, {44,143}, {9,175}, 
    {9,177}, {44,140}, {44,142}, {9,176}, {44,145}, {44,144}, {44,146}, 
    {47,118}, {47,116}, {12,60}, {47,115}, {12,99}, {12,62}, {12,59}, {12,100}, 
    {12,61}, {12,98}, {12,97}, {47,117}, {12,58}, {15,42}, {50,171}, {50,177}, 
    {15,39}, {15,46}, {15,41}, {50,174}, {50,176}, {15,44}, {50,170}, {50,173}, 
    {15,40}, {15,37}, {50,172}, {15,43}, {50,175}, {15,45}, {15,38}, {54,163}, 
    {54,166}, {18,48}, {18,58}, {18,54}, {54,164}, {18,62}, {18,55}, {18,60}, 
    {18,50}, {18,47}, {18,49}, {18,61}, {54,165}, {18,53}, {18,56}, {18,52}, 
    {18,57}, {54,167}, {18,59}, {18,51}, {54,162}, {59,13}, {21,37}, {59,15}, 
    {21,39}, {21,41}, {59,14}, {21,38}, {21,42}, {21,40}, {24,7}, {63,99}, 
    {24,15}, {63,62}, {24,16}, {24,5}, {63,97}, {24,10}, {63,98}, {24,3}, 
    {24,2}, {24,13}, {24,12}, {24,11}, {24,9}, {24,14}, {63,61}, {24,4}, 
    {24,6}, {24,8}, {26,120}, {26,123}, {26,127}, {26,121}, {26,126}, {26,124}, 
    {26,119}, {26,125}, {26,122}, {71,18}, {29,3}, {29,1}, {71,19}, {29,4}, 
    {29,2}, {71,17}, {71,16}, {30,176}, {71,15}, {30,174}, {30,173}, {74,144}, 
    {29,5}, {30,175}, {74,145}, {30,178}, {74,146}, {30,177}, {32,152}, 
    {77,131}, {32,151}, {32,156}, {32,154}, {32,155}, {33,163}, {32,153}, 
    {77,130}, {74,147}, {33,162}, {33,164}, {34,164}, {34,165}, {83,160}, 
    {36,158}, {36,157}, {7,118}, {41,21}, {15,47}, {41,18}, {41,19}, {41,17}, 
    {41,20}, {42,106}, {42,103}, {42,108}, {42,104}, {7,119}, {7,122}, 
    {42,105}, {7,121}, {42,107}, {7,120}, {44,149}, {44,151}, {9,180}, 
    {44,147}, {44,150}, {44,148}, {9,178}, {9,181}, {9,179}, {47,120}, 
    {47,124}, {47,121}, {12,103}, {12,104}, {47,123}, {47,125}, {47,127}, 
    {12,101}, {47,126}, {12,102}, {54,174}, {47,122}, {50,178}, {50,187}, 
    {50,185}, {50,180}, {50,181}, {50,186}, {50,188}, {15,49}, {50,183}, 
    {50,179}, {15,48}, {15,50}, {50,182}, {50,189}, {50,184}, {18,97}, 
    {54,177}, {54,169}, {54,175}, {54,176}, {18,98}, {54,168}, {54,170}, 
    {47,119}, {54,172}, {54,173}, {54,171}, {21,44}, {59,22}, {74,148}, 
    {59,23}, {59,20}, {18,99}, {21,46}, {59,19}, {59,25}, {59,24}, {21,43}, 
    {59,28}, {59,18}, {59,27}, {59,16}, {59,26}, {59,21}, {21,45}, {63,106}, 
    {63,101}, {63,107}, {63,102}, {63,100}, {63,103}, {24,17}, {63,105}, 
    {63,104}, {24,18}, {26,129}, {26,128}, {67,46}, {67,49}, {67,41}, {67,45}, 
    {26,130}, {67,44}, {67,42}, {67,48}, {67,43}, {67,40}, {67,47}, {71,25}, 
    {29,8}, {80,10}, {71,22}, {71,23}, {71,21}, {59,17}, {29,7}, {71,26}, 
    {71,20}, {29,6}, {29,9}, {71,24}, {74,149}, {30,179}, {74,150}, {74,151}, 
    {77,132}, {32,157}, {80,11}, {80,12}, {80,13}, {82,29}, {82,28}, {83,162}, 
    {83,161}, {85,27}, {85,28}, {85,26}, {86,38}, {36,112}, {7,123}, {12,106}, 
    {12,105}, {15,53}, {15,52}, {51,0}, {51,1}, {50,190}, {15,51}, {54,181}, 
    {54,182}, {54,178}, {18,100}, {54,179}, {54,180}, {59,31}, {59,33}, 
    {59,29}, {59,32}, {21,47}, {59,30}, {21,48}, {60,51}, {24,21}, {24,20}, 
    {24,19}, {63,108}, {63,109}, {67,51}, {67,53}, {26,134}, {26,131}, 
    {26,133}, {26,132}, {67,52}, {67,50}, {71,33}, {71,30}, {71,31}, {29,13}, 
    {71,32}, {71,27}, {71,28}, {29,10}, {29,11}, {71,29}, {29,12}, {74,155}, 
    {74,156}, {74,153}, {74,154}, {30,180}, {74,152}, {77,136}, {77,133}, 
    {77,135}, {32,160}, {77,139}, {32,159}, {77,137}, {77,140}, {77,138}, 
    {77,134}, {32,158}, {80,15}, {80,16}, {80,14}, {80,18}, {33,165}, {33,166}, 
    {82,31}, {34,167}, {80,17}, {82,30}, {34,166}, {83,165}, {83,166}, 
    {35,127}, {83,164}, {83,163}, {85,29}, {36,8}, {87,9}, {87,136}, {87,135}, 
    {37,3}, {37,29}, {87,137}, {88,49}, {37,47}, {7,124}, {9,182}, {24,22}, 
    {35,128}, {7,125}, {12,107}, {18,101}, {21,49}, {33,167}, {9,183}, 
    {47,129}, {47,128}, {51,2}, {15,56}, {15,54}, {15,58}, {51,4}, {15,57}, 
    {15,55}, {51,3}, {18,104}, {54,188}, {18,103}, {18,105}, {55,2}, {18,107}, 
    {54,190}, {55,0}, {54,183}, {18,106}, {55,3}, {54,185}, {54,186}, {54,184}, 
    {18,102}, {55,1}, {54,187}, {54,189}, {59,45}, {59,44}, {59,42}, {59,34}, 
    {59,49}, {59,37}, {59,47}, {59,54}, {59,46}, {21,57}, {21,53}, {59,35}, 
    {59,41}, {21,55}, {59,40}, {21,56}, {21,58}, {59,43}, {21,50}, {21,51}, 
    {59,55}, {59,53}, {59,52}, {59,38}, {59,50}, {21,54}, {21,52}, {59,51}, 
    {59,36}, {59,39}, {59,48}, {67,122}, {63,119}, {63,112}, {63,131}, 
    {63,140}, {63,115}, {24,33}, {63,128}, {24,23}, {24,25}, {24,37}, {63,113}, 
    {24,26}, {24,28}, {24,38}, {24,27}, {24,36}, {63,121}, {63,110}, {24,34}, 
    {63,120}, {24,30}, {63,138}, {24,35}, {63,136}, {63,124}, {63,134}, 
    {24,32}, {63,111}, {63,137}, {63,132}, {63,139}, {24,24}, {24,39}, 
    {24,29}, {63,117}, {63,125}, {63,129}, {63,133}, {24,31}, {63,116}, 
    {63,114}, {63,126}, {63,123}, {63,122}, {63,127}, {63,130}, {63,135}, 
    {67,56}, {26,135}, {67,100}, {67,58}, {26,140}, {26,144}, {67,109}, 
    {67,117}, {67,102}, {26,136}, {67,106}, {63,118}, {26,137}, {67,113}, 
    {67,118}, {67,110}, {67,112}, {67,121}, {67,114}, {67,62}, {67,105}, 
    {26,145}, {26,141}, {67,60}, {67,107}, {26,139}, {67,101}, {26,138}, 
    {67,115}, {67,98}, {67,57}, {26,142}, {67,120}, {67,59}, {67,111}, 
    {67,108}, {67,103}, {67,55}, {67,54}, {67,97}, {67,116}, {26,143}, 
    {67,119}, {67,61}, {67,99}, {29,18}, {29,26}, {29,21}, {71,100}, {71,97}, 
    {71,39}, {29,16}, {29,15}, {29,22}, {71,37}, {29,20}, {71,49}, {71,35}, 
    {71,36}, {29,14}, {71,99}, {29,24}, {71,52}, {71,57}, {71,51}, {74,174}, 
    {71,47}, {71,55}, {71,53}, {71,40}, {71,34}, {71,61}, {29,23}, {71,62}, 
    {71,56}, {71,45}, {71,43}, {71,38}, {71,46}, {71,59}, {71,42}, {71,58}, 
    {71,98}, {29,19}, {71,54}, {71,60}, {71,50}, {71,44}, {29,17}, {67,104}, 
    {71,48}, {29,25}, {71,41}, {74,180}, {30,183}, {74,179}, {74,176}, 
    {75,4}, {30,187}, {75,1}, {74,184}, {75,3}, {74,169}, {74,182}, {30,189}, 
    {74,161}, {74,159}, {75,2}, {75,0}, {74,190}, {74,173}, {74,172}, {74,162}, 
    {31,0}, {74,168}, {74,178}, {74,189}, {31,3}, {75,5}, {32,168}, {31,5}, 
    {30,190}, {74,166}, {74,175}, {74,158}, {74,160}, {30,181}, {31,2}, 
    {30,186}, {74,167}, {74,183}, {74,177}, {31,1}, {74,157}, {32,163}, 
    {74,185}, {74,188}, {30,188}, {74,171}, {31,4}, {30,185}, {30,184}, 
    {74,181}, {74,187}, {30,182}, {74,164}, {74,186}, {74,165}, {74,170}, 
    {77,146}, {77,151}, {32,165}, {32,167}, {77,157}, {32,161}, {77,172}, 
    {77,163}, {77,152}, {77,153}, {77,162}, {32,174}, {77,161}, {77,145}, 
    {77,160}, {77,148}, {77,173}, {32,173}, {32,171}, {77,149}, {77,168}, 
    {77,154}, {77,167}, {77,169}, {77,144}, {32,166}, {77,170}, {77,158}, 
    {32,170}, {77,155}, {32,172}, {77,164}, {32,164}, {77,150}, {77,165}, 
    {77,159}, {74,163}, {77,166}, {77,147}, {32,169}, {77,171}, {32,162}, 
    {77,142}, {80,32}, {80,25}, {33,169}, {80,20}, {80,35}, {80,27}, {77,156}, 
    {80,37}, {80,21}, {33,168}, {80,31}, {33,170}, {33,178}, {33,176}, 
    {80,33}, {33,177}, {80,23}, {80,24}, {80,29}, {80,34}, {77,141}, {33,171}, 
    {80,42}, {80,39}, {80,43}, {80,30}, {80,26}, {80,40}, {82,42}, {80,28}, 
    {33,174}, {33,173}, {77,143}, {33,175}, {80,36}, {80,38}, {33,172}, 
    {80,41}, {80,19}, {80,22}, {82,51}, {82,35}, {34,171}, {82,49}, {82,33}, 
    {34,172}, {82,44}, {82,40}, {34,177}, {82,50}, {82,34}, {82,37}, {34,169}, 
    {82,52}, {82,45}, {82,48}, {34,175}, {34,180}, {34,178}, {82,41}, {82,36}, 
    {34,173}, {34,174}, {82,32}, {34,170}, {34,168}, {34,176}, {82,47}, 
    {34,179}, {82,43}, {82,53}, {34,181}, {82,39}, {82,46}, {83,179}, {84,2}, 
    {83,181}, {83,188}, {82,38}, {83,186}, {83,169}, {84,0}, {35,131}, 
    {83,173}, {83,190}, {83,180}, {35,130}, {84,4}, {83,182}, {83,187}, 
    {83,189}, {83,167}, {84,1}, {83,178}, {83,183}, {83,171}, {83,175}, 
    {84,3}, {83,185}, {83,168}, {83,172}, {83,174}, {83,184}, {35,129}, 
    {83,177}, {83,170}, {83,176}, {85,33}, {85,38}, {36,15}, {85,40}, {36,9}, 
    {85,36}, {85,42}, {36,14}, {36,10}, {36,11}, {85,32}, {85,39}, {36,13}, 
    {85,37}, {36,12}, {85,31}, {85,35}, {85,34}, {85,30}, {85,41}, {36,113}, 
    {86,45}, {86,48}, {86,44}, {86,46}, {86,47}, {86,41}, {86,42}, {86,39}, 
    {86,43}, {86,40}, {36,114}, {36,115}, {87,11}, {87,13}, {87,12}, {87,14}, 
    {36,160}, {87,10}, {36,159}, {36,161}, {87,139}, {87,140}, {37,4}, 
    {87,138}, {88,19}, {88,18}, {88,20}, {37,31}, {88,21}, {37,30}, {88,22}, 
    {88,50}, {88,53}, {88,52}, {37,40}, {88,51}, {37,50}, {37,48}, {37,49}, 
    {37,55}, {88,128}, {88,129}, {88,127}, {88,137}, {9,184}, {55,4}, {59,56}, 
    {71,101}, {82,54}, {9,185}, {12,108}, {15,59}, {55,5}, {55,6}, {18,108}, 
    {21,61}, {59,58}, {59,57}, {21,99}, {21,60}, {59,59}, {21,62}, {21,98}, 
    {21,97}, {21,59}, {24,40}, {63,144}, {63,142}, {63,143}, {63,141}, 
    {26,146}, {26,149}, {26,151}, {26,150}, {26,147}, {26,148}, {71,103}, 
    {71,102}, {29,27}, {71,104}, {71,105}, {29,28}, {75,13}, {75,11}, {77,179}, 
    {75,9}, {75,10}, {31,6}, {75,6}, {75,14}, {75,8}, {75,12}, {77,175}, 
    {77,177}, {77,178}, {32,179}, {77,174}, {32,178}, {77,176}, {32,175}, 
    {32,176}, {32,177}, {75,7}, {33,181}, {80,46}, {80,44}, {80,45}, {33,179}, 
    {33,182}, {33,180}, {82,55}, {82,56}, {34,182}, {84,5}, {84,7}, {84,6}, 
    {35,132}, {36,16}, {85,45}, {85,44}, {85,43}, {9,186}, {40,106}, {41,24}, 
    {5,169}, {41,22}, {41,25}, {41,23}, {42,110}, {7,129}, {7,130}, {42,112}, 
    {7,127}, {42,111}, {42,109}, {7,128}, {7,126}, {44,152}, {44,155}, 
    {9,189}, {44,154}, {44,153}, {9,188}, {9,187}, {10,0}, {44,156}, {9,190}, 
    {47,134}, {12,110}, {12,111}, {12,112}, {47,135}, {47,131}, {12,109}, 
    {47,132}, {47,133}, {47,130}, {15,100}, {15,97}, {51,5}, {15,98}, {15,101}, 
    {51,6}, {15,62}, {15,60}, {15,61}, {15,99}, {18,109}, {55,9}, {18,117}, 
    {55,8}, {55,11}, {18,113}, {55,10}, {21,107}, {18,111}, {18,114}, {18,110}, 
    {18,115}, {18,116}, {18,112}, {55,7}, {21,103}, {59,61}, {59,99}, {59,98}, 
    {21,108}, {21,104}, {21,105}, {59,60}, {59,62}, {59,97}, {21,100}, 
    {21,102}, {21,106}, {21,101}, {63,147}, {63,145}, {63,146}, {24,42}, 
    {24,43}, {63,148}, {24,41}, {26,152}, {26,154}, {26,153}, {67,123}, 
    {67,124}, {71,107}, {71,106}, {31,7}, {31,8}, {75,15}, {31,9}, {77,182}, 
    {77,180}, {77,181}, {32,180}, {80,47}, {34,183}, {32,181}, {10,1}, 
    {15,102}, {51,7}, {55,12}, {18,118}, {21,109}, {59,100}, {59,102}, 
    {21,111}, {21,110}, {21,112}, {21,113}, {59,101}, {24,46}, {24,47}, 
    {24,45}, {26,155}, {24,44}, {63,149}, {26,156}, {71,108}, {75,16}, 
    {31,10}, {32,182}, {80,48}, {80,52}, {33,185}, {80,49}, {33,186}, {33,184}, 
    {80,53}, {33,187}, {80,51}, {82,57}, {33,183}, {34,184}, {87,141}, 
    {10,2}, {18,120}, {18,119}, {21,114}, {59,104}, {59,103}, {21,115}, 
    {63,153}, {24,51}, {24,48}, {63,152}, {24,50}, {63,150}, {24,49}, {63,151}, 
    {67,125}, {26,157}, {71,111}, {29,29}, {71,109}, {29,30}, {29,31}, 
    {71,110}, {29,32}, {75,17}, {31,14}, {31,11}, {31,16}, {75,19}, {31,12}, 
    {75,18}, {31,15}, {31,13}, {77,185}, {77,187}, {32,183}, {77,186}, 
    {32,184}, {77,184}, {77,183}, {80,55}, {80,54}, {33,188}, {80,56}, 
    {82,62}, {34,186}, {82,61}, {82,58}, {34,185}, {82,59}, {82,60}, {84,8}, 
    {84,9}, {35,133}, {36,19}, {85,46}, {36,17}, {36,18}, {85,47}, {36,116}, 
    {36,117}, {86,49}, {37,5}, {87,143}, {37,7}, {87,142}, {87,144}, {37,6}, 
    {88,23}, {88,109}, {10,3}, {24,52}, {67,126}, {71,112}, {31,17}, {31,18}, 
    {10,4}, {29,33}, {34,187}, {12,113}, {31,19}, {36,162}, {12,114}, {55,13}, 
    {59,105}, {59,107}, {59,106}, {63,157}, {63,154}, {24,53}, {24,54}, 
    {63,155}, {63,156}, {67,128}, {67,133}, {26,158}, {67,127}, {67,129}, 
    {67,136}, {67,131}, {67,135}, {67,132}, {67,130}, {67,134}, {26,159}, 
    {71,115}, {71,113}, {29,35}, {29,34}, {71,114}, {29,36}, {75,22}, {75,21}, 
    {31,20}, {75,20}, {77,188}, {77,190}, {78,1}, {78,0}, {32,185}, {77,189}, 
    {80,97}, {33,189}, {80,61}, {80,98}, {33,190}, {80,59}, {80,62}, {80,60}, 
    {80,57}, {34,0}, {80,58}, {82,97}, {82,99}, {82,98}, {84,10}, {84,11}, 
    {85,48}, {36,119}, {36,118}, {87,15}, {87,16}, {37,8}, {87,145}, {37,41}, 
    {12,115}, {21,116}, {67,138}, {67,137}, {71,117}, {71,116}, {32,186}, 
    {78,3}, {78,2}, {80,101}, {80,99}, {80,102}, {80,100}, {34,188}, {82,100}, 
    {82,101}, {82,102}, {85,49}, {86,50}, {87,146}, {12,116}, {75,23}, 
    {78,4}, {12,117}, {26,160}, {75,24}, {34,1}, {80,103}, {34,189}, {84,12}, 
    {84,13}, {36,20}, {87,17}, {12,118}, {18,123}, {18,124}, {55,14}, {21,117}, 
    {21,118}, {59,108}, {21,119}, {24,58}, {24,60}, {63,159}, {63,160}, 
    {63,158}, {24,55}, {24,56}, {24,59}, {24,57}, {67,139}, {26,161}, {26,162}, 
    {71,124}, {29,39}, {71,119}, {71,118}, {71,123}, {29,37}, {31,27}, 
    {71,120}, {71,125}, {71,122}, {71,121}, {29,38}, {31,25}, {75,26}, 
    {31,21}, {75,27}, {75,25}, {31,24}, {31,22}, {31,26}, {31,23}, {78,5}, 
    {78,10}, {78,6}, {78,9}, {32,187}, {76,148}, {78,8}, {78,7}, {34,4}, 
    {34,2}, {34,5}, {34,3}, {80,104}, {80,105}, {80,106}, {34,6}, {82,106}, 
    {35,0}, {82,104}, {35,1}, {82,103}, {82,105}, {34,190}, {84,17}, {84,14}, 
    {84,15}, {84,16}, {85,50}, {36,22}, {36,21}, {86,52}, {86,51}, {36,120}, 
    {36,163}, {37,9}, {37,32}, {88,24}, {88,110}, {88,111}, {12,119}, {59,109}, 
    {63,161}, {67,140}, {67,141}, {26,163}, {26,164}, {71,126}, {29,40}, 
    {32,188}, {80,107}, {34,7}, {82,109}, {35,2}, {82,108}, {82,110}, {82,107}, 
    {85,53}, {84,18}, {84,19}, {35,134}, {85,52}, {85,51}, {86,53}, {88,112}, 
    {12,120}, {12,121}, {15,103}, {51,8}, {55,15}, {21,120}, {21,123}, 
    {21,121}, {59,110}, {21,125}, {21,122}, {21,124}, {24,62}, {63,162}, 
    {63,163}, {71,128}, {24,61}, {24,97}, {24,98}, {67,143}, {67,142}, 
    {26,165}, {26,166}, {67,144}, {71,127}, {26,168}, {29,41}, {26,167}, 
    {31,28}, {71,129}, {29,43}, {29,42}, {71,130}, {71,133}, {71,131}, 
    {71,132}, {29,44}, {31,33}, {31,31}, {31,30}, {75,29}, {31,32}, {75,28}, 
    {78,11}, {75,30}, {31,29}, {75,31}, {78,14}, {78,12}, {78,13}, {78,18}, 
    {34,11}, {78,17}, {78,20}, {78,19}, {78,16}, {78,15}, {32,189}, {80,110}, 
    {80,109}, {34,10}, {34,8}, {34,9}, {80,108}, {82,113}, {35,3}, {82,112}, 
    {82,111}, {35,4}, {84,24}, {84,23}, {84,21}, {84,20}, {35,136}, {35,135}, 
    {84,25}, {86,54}, {36,121}, {85,55}, {36,23}, {85,54}, {84,22}, {86,55}, 
    {36,164}, {37,33}, {88,25}, {88,113}, {12,122}, {55,16}, {78,21}, {12,123}, 
    {67,146}, {67,145}, {75,32}, {78,23}, {78,22}, {34,12}, {82,114}, {82,115}, 
    {35,137}, {88,114}, {15,104}, {21,127}, {21,126}, {63,164}, {63,166}, 
    {24,100}, {63,165}, {24,99}, {24,101}, {63,167}, {67,148}, {67,150}, 
    {67,149}, {67,152}, {26,169}, {67,151}, {67,147}, {67,153}, {71,140}, 
    {71,143}, {71,145}, {71,135}, {71,139}, {71,136}, {29,46}, {29,49}, 
    {29,51}, {71,137}, {71,138}, {29,50}, {71,141}, {71,144}, {71,142}, 
    {29,52}, {29,48}, {71,134}, {29,45}, {29,47}, {31,35}, {75,38}, {75,36}, 
    {75,35}, {75,41}, {75,40}, {75,39}, {75,34}, {31,34}, {75,33}, {75,37}, 
    {31,36}, {78,26}, {78,30}, {78,27}, {78,29}, {78,28}, {78,25}, {78,31}, 
    {78,34}, {78,32}, {78,33}, {33,0}, {32,190}, {78,24}, {78,35}, {80,115}, 
    {80,118}, {80,120}, {80,119}, {80,113}, {80,117}, {80,112}, {80,114}, 
    {34,13}, {80,111}, {80,116}, {82,128}, {82,117}, {35,5}, {35,6}, {82,116}, 
    {82,121}, {82,127}, {82,119}, {82,126}, {82,123}, {82,122}, {82,125}, 
    {82,120}, {82,118}, {82,124}, {84,32}, {84,30}, {35,138}, {84,29}, 
    {84,35}, {84,33}, {35,139}, {84,28}, {84,26}, {84,27}, {35,141}, {84,31}, 
    {35,140}, {84,34}, {85,56}, {85,62}, {85,57}, {36,27}, {85,97}, {36,26}, 
    {85,61}, {85,60}, {36,25}, {85,59}, {36,24}, {85,58}, {86,61}, {86,97}, 
    {86,62}, {86,59}, {36,123}, {86,56}, {86,60}, {86,99}, {86,98}, {86,57}, 
    {86,58}, {36,122}, {87,18}, {36,167}, {87,19}, {36,165}, {36,166}, 
    {87,147}, {37,10}, {88,54}, {37,42}, {88,115}, {37,43}, {88,116}, {88,117}, 
    {88,131}, {88,130}, {37,58}, {88,141}, {15,105}, {63,169}, {63,168}, 
    {26,170}, {26,171}, {67,154}, {71,146}, {75,44}, {29,53}, {31,37}, 
    {75,42}, {75,45}, {31,38}, {78,36}, {75,43}, {80,121}, {34,14}, {82,129}, 
    {84,38}, {84,36}, {84,37}, {85,98}, {36,28}, {86,100}, {36,170}, {36,124}, 
    {36,168}, {36,169}, {87,148}, {37,34}, {15,106}, {80,122}, {51,9}, 
    {24,102}, {67,155}, {26,172}, {67,156}, {71,148}, {71,147}, {31,40}, 
    {29,54}, {29,55}, {71,151}, {71,150}, {71,149}, {75,46}, {75,49}, {75,48}, 
    {75,47}, {31,39}, {78,40}, {78,38}, {78,37}, {78,39}, {34,15}, {80,124}, 
    {80,125}, {34,16}, {80,123}, {82,131}, {82,130}, {82,133}, {35,7}, 
    {82,132}, {84,39}, {84,41}, {84,40}, {85,99}, {85,102}, {85,100}, {85,101}, 
    {86,101}, {36,125}, {87,20}, {87,21}, {87,22}, {37,11}, {37,35}, {88,118}, 
    {15,107}, {29,56}, {31,41}, {80,126}, {86,102}, {88,132}, {51,10}, 
    {37,59}, {15,108}, {75,50}, {80,127}, {82,134}, {85,103}, {86,103}, 
    {15,109}, {67,157}, {67,158}, {26,173}, {26,174}, {71,153}, {29,58}, 
    {29,57}, {71,152}, {78,41}, {80,128}, {80,130}, {80,129}, {34,19}, 
    {34,18}, {34,17}, {36,30}, {85,104}, {36,29}, {86,105}, {86,104}, {37,12}, 
    {87,149}, {18,125}, {63,170}, {67,161}, {67,159}, {67,160}, {71,162}, 
    {71,157}, {71,154}, {71,161}, {71,163}, {29,60}, {71,160}, {71,156}, 
    {71,155}, {71,159}, {71,158}, {29,59}, {75,61}, {75,56}, {75,54}, {75,97}, 
    {75,55}, {75,51}, {75,57}, {75,52}, {78,50}, {75,53}, {75,98}, {75,60}, 
    {31,42}, {75,59}, {75,58}, {75,62}, {78,42}, {78,45}, {78,44}, {78,52}, 
    {78,47}, {78,51}, {78,49}, {78,48}, {78,46}, {78,43}, {33,3}, {33,2}, 
    {33,4}, {33,1}, {78,53}, {80,136}, {80,139}, {80,137}, {80,141}, {80,142}, 
    {80,134}, {34,24}, {80,135}, {80,133}, {80,140}, {80,132}, {80,131}, 
    {34,23}, {34,21}, {34,20}, {80,138}, {82,143}, {82,149}, {35,10}, {82,144}, 
    {82,147}, {82,151}, {35,11}, {82,146}, {82,138}, {82,137}, {82,150}, 
    {82,141}, {82,139}, {82,148}, {82,140}, {35,9}, {35,8}, {82,135}, {82,136}, 
    {82,145}, {82,142}, {84,44}, {84,47}, {34,22}, {84,45}, {84,51}, {84,49}, 
    {84,43}, {84,54}, {84,42}, {35,143}, {84,50}, {84,46}, {35,142}, {84,53}, 
    {84,52}, {85,107}, {85,106}, {85,113}, {85,109}, {85,112}, {36,32}, 
    {85,110}, {85,111}, {85,105}, {85,108}, {36,31}, {36,127}, {86,116}, 
    {86,111}, {86,115}, {86,118}, {86,114}, {86,110}, {36,129}, {86,113}, 
    {86,117}, {36,128}, {86,108}, {84,48}, {86,112}, {86,109}, {86,106}, 
    {86,107}, {36,126}, {87,26}, {87,28}, {87,31}, {87,27}, {87,32}, {87,25}, 
    {87,23}, {36,171}, {87,29}, {36,173}, {36,172}, {87,24}, {87,30}, {87,154}, 
    {37,13}, {87,155}, {87,153}, {87,150}, {87,152}, {87,151}, {88,26}, 
    {88,28}, {88,27}, {88,57}, {88,56}, {88,55}, {88,58}, {37,51}, {37,52}, 
    {88,138}, {88,142}, {18,126}, {59,111}, {63,173}, {24,103}, {63,171}, 
    {63,172}, {67,162}, {67,163}, {26,177}, {26,175}, {67,164}, {26,176}, 
    {71,168}, {71,171}, {71,165}, {71,172}, {71,164}, {71,166}, {71,167}, 
    {71,170}, {29,97}, {71,175}, {71,174}, {29,61}, {71,169}, {71,173}, 
    {29,62}, {75,108}, {31,47}, {75,103}, {31,43}, {75,100}, {75,106}, 
    {75,109}, {31,48}, {75,105}, {75,102}, {75,110}, {75,101}, {75,107}, 
    {31,44}, {75,99}, {31,45}, {31,46}, {75,104}, {78,105}, {78,58}, {78,59}, 
    {78,62}, {78,60}, {78,54}, {78,57}, {78,101}, {78,61}, {33,5}, {78,103}, 
    {78,100}, {33,6}, {78,102}, {78,55}, {78,98}, {78,99}, {78,97}, {80,146}, 
    {80,148}, {80,151}, {80,145}, {34,25}, {80,153}, {80,144}, {80,154}, 
    {80,150}, {80,152}, {80,156}, {80,149}, {80,157}, {80,147}, {80,143}, 
    {34,26}, {80,155}, {34,27}, {35,13}, {78,56}, {82,177}, {82,168}, {35,15}, 
    {82,164}, {35,16}, {82,173}, {82,167}, {82,157}, {35,14}, {82,170}, 
    {82,165}, {82,166}, {82,152}, {82,159}, {82,174}, {82,171}, {82,163}, 
    {82,175}, {82,158}, {82,153}, {82,172}, {82,155}, {82,169}, {82,160}, 
    {82,176}, {82,156}, {35,12}, {82,154}, {82,161}, {82,162}, {84,61}, 
    {84,59}, {84,98}, {84,110}, {84,101}, {84,60}, {84,56}, {84,103}, {84,62}, 
    {84,99}, {84,58}, {84,106}, {84,55}, {84,97}, {84,102}, {84,104}, {84,107}, 
    {84,57}, {84,111}, {84,112}, {84,105}, {84,109}, {84,100}, {85,129}, 
    {85,132}, {36,33}, {85,131}, {85,136}, {85,134}, {36,34}, {85,125}, 
    {85,115}, {85,114}, {36,36}, {85,127}, {85,128}, {85,124}, {85,116}, 
    {85,121}, {84,108}, {85,117}, {36,35}, {85,123}, {85,122}, {85,118}, 
    {85,130}, {85,119}, {86,123}, {85,133}, {85,135}, {85,126}, {85,120}, 
    {86,124}, {86,126}, {86,120}, {36,130}, {86,133}, {86,131}, {36,131}, 
    {86,130}, {86,129}, {86,122}, {86,119}, {86,125}, {86,134}, {86,121}, 
    {86,127}, {87,41}, {87,46}, {87,36}, {87,39}, {36,174}, {87,43}, {87,50}, 
    {86,128}, {87,37}, {87,47}, {87,51}, {87,42}, {87,35}, {87,45}, {87,44}, 
    {87,49}, {87,48}, {86,132}, {87,40}, {87,34}, {87,38}, {37,14}, {37,15}, 
    {87,33}, {87,166}, {87,157}, {87,165}, {87,162}, {87,163}, {87,156}, 
    {87,159}, {87,167}, {87,161}, {87,160}, {87,158}, {87,164}, {88,29}, 
    {88,30}, {88,32}, {88,31}, {88,34}, {88,33}, {88,60}, {88,59}, {88,119}, 
    {88,120}, {88,133}, {37,56}, {37,60}, {88,143}, {37,61}, {18,127}, 
    {35,144}, {85,137}, {37,16}, {37,17}, {18,128}, {63,174}, {24,104}, 
    {71,176}, {75,112}, {75,113}, {75,111}, {78,107}, {78,106}, {33,7}, 
    {80,159}, {78,108}, {80,158}, {82,179}, {35,17}, {35,19}, {82,178}, 
    {35,18}, {84,113}, {84,115}, {84,114}, {85,138}, {36,37}, {36,175}, 
    {87,168}, {88,35}, {88,146}, {18,129}, {67,165}, {29,98}, {75,115}, 
    {75,114}, {78,109}, {35,20}, {35,145}, {86,135}, {88,139}, {18,130}, 
    {26,178}, {29,99}, {82,180}, {87,52}, {21,128}, {78,110}, {37,36}, 
    {21,129}, {29,100}, {33,8}, {87,53}, {21,130}, {71,177}, {31,50}, {75,116}, 
    {75,117}, {31,49}, {78,111}, {33,12}, {33,10}, {33,11}, {33,9}, {80,160}, 
    {34,28}, {84,117}, {84,116}, {84,119}, {84,118}, {35,146}, {85,139}, 
    {85,141}, {85,140}, {36,38}, {86,136}, {87,54}, {87,55}, {36,176}, 
    {88,36}, {88,61}, {37,53}, {59,112}, {75,118}, {78,112}, {82,181}, 
    {63,175}, {78,113}, {80,162}, {80,161}, {87,56}, {37,18}, {88,37}, 
    {88,62}, {24,105}, {71,178}, {71,179}, {75,119}, {24,106}, {34,29}, 
    {80,163}, {85,143}, {36,39}, {85,144}, {85,142}, {87,57}, {87,169}, 
    {24,107}, {78,116}, {78,115}, {78,114}, {80,164}, {80,168}, {80,167}, 
    {80,166}, {80,165}, {34,30}, {82,182}, {84,121}, {35,147}, {84,120}, 
    {85,145}, {86,139}, {86,138}, {36,132}, {86,137}, {87,60}, {87,59}, 
    {87,58}, {26,179}, {75,120}, {33,13}, {82,183}, {82,184}, {86,140}, 
    {87,61}, {87,170}, {88,38}, {88,121}, {88,148}, {26,180}, {33,14}, 
    {80,169}, {82,185}, {85,146}, {87,62}, {29,102}, {78,117}, {80,170}, 
    {82,186}, {82,187}, {82,188}, {84,126}, {84,122}, {36,40}, {84,125}, 
    {84,124}, {35,148}, {84,123}, {35,150}, {35,149}, {85,148}, {85,147}, 
    {36,41}, {36,42}, {36,134}, {86,141}, {36,133}, {87,99}, {87,100}, 
    {87,98}, {87,97}, {37,20}, {87,171}, {87,172}, {87,173}, {37,19}, {88,39}, 
    {88,42}, {88,41}, {88,40}, {88,147}, {31,51}, {34,37}, {84,127}, {85,149}, 
    {36,135}, {86,142}, {88,149}, {31,52}, {78,118}, {86,143}, {88,97}, 
    {40,10}, {60,188}, {0,10}, {0,23}, {0,25}, {0,27}, {0,31}, {0,32}, 
    {0,35}, {0,36}, {0,39}, {0,40}, {0,43}, {0,44}, {0,47}, {0,48}, {0,51}, 
    {0,52}, {0,55}, {0,56}, {0,59}, {0,60}, {0,134}, {0,135}, {0,138}, 
    {0,139}, {0,136}, {0,137}, {0,28}, {0,13}, {0,15}, {0,17}, {0,18}, 
    {0,19}, {0,20}, {0,61}, {0,62}, {0,97}, {0,98}, {0,99}, {0,100}, {0,140}, 
    {0,141}, {0,142}, {0,158}, {0,159}, {0,160}, {0,161}, {0,162}, {1,12}, 
    {1,13}, {1,14}, {0,9}, {0,109}, {1,3}, {1,8}, {0,110}, {0,29}, {0,30}, 
    {0,111}, {0,143}, {0,1}, {0,144}, {0,4}, {1,1}, {1,111}, {1,112}, {1,113}, 
    {1,114}, {1,115}, {1,116}, {1,117}, {1,118}, {1,119}, {1,120}, {0,7}, 
    {0,6}, {0,149}, {0,151}, {0,150}, {0,8}, {1,9}, {1,143}, {1,144}, {1,145}, 
    {1,146}, {1,147}, {1,148}, {1,149}, {1,150}, {1,151}, {1,152}, {1,153}, 
    {1,154}, {1,155}, {1,156}, {1,157}, {1,158}, {1,159}, {1,160}, {1,161}, 
    {1,162}, {1,163}, {1,164}, {1,165}, {1,166}, {1,167}, {1,168}, {1,2}, 
    {0,132}, {1,169}, {1,170}, {1,171}, {1,172}, {1,173}, {1,174}, {1,175}, 
    {1,176}, {1,177}, {1,178}, {1,179}, {1,180}, {1,181}, {1,182}, {1,183}, 
    {1,184}, {1,185}, {1,186}, {1,187}, {1,188}, {1,189}, {1,190}, {2,0}, 
    {2,1}, {2,2}, {2,3}, {0,33}, {0,21}, {0,34}, {0,14}
};

/* This returns ERROR if the code point doesn't exist. */
long int big5_to_unicode(int r, int c)
{
    assert(r >= 0 && r < 94);
    assert(c >= 0 && c < 191);
    return big5_forward[r][c];
}

/* This one returns 1 on success, 0 if the code point doesn't exist. */
int unicode_to_big5(long int unicode, int *r, int *c)
{
    int rr, cc;
    long int uu;
    int i, j, k;

    i = -1;
    j = lenof(big5_backward);
    while (j - i > 1) {
	k = (i + j) / 2;
	rr = big5_backward[k].r;
	cc = big5_backward[k].c;
	uu = big5_forward[rr][cc];
	if (unicode > uu)
	    i = k;
	else if (unicode < uu)
	    j = k;
	else {
	    *r = rr;
	    *c = cc;
	    return 1;
	}
    }
    return 0;
}

#ifdef TESTMODE

#include <stdio.h>

int main(void)
{
    int r, c, rr, cc, ret;
    long int u, uu;

    /*
     * N.B. Big5 <-> Unicode round-trip mapping is known not to
     * work, because of the various duplicate characters:
     * 
     *  - 0xA15A (0,26)  and 0xA1C4 (0,132) are both U+FF3F
     *  - 0xA1FE (0,190) and 0xA2AC (1,108) are both U+2571
     *  - 0xA240 (1,0)   and 0xA2AD (1,109) are both U+2572
     *  - 0xA2CC (1,140) and 0xA451 (3,17)  are both U+5341
     *  - 0xA2CE (1,142) and 0xA4CA (3,138) are both U+5345
     */

    for (r = 0; r < 94; r++) {
	for (c = 0; c < 191; c++) {
	    u = big5_to_unicode(r, c);
	    if (u != ERROR) {
		ret = unicode_to_big5(u, &rr, &cc);
		if (!ret)
		    printf("(%d,%d) -> U-%08lx but didn't convert back\n",
			   r, c, u);
		else if (rr != r || cc != c)
		    printf("(%d,%d) -> U-%08lx -> (%d,%d)\n",
			   r, c, u, rr, cc);
	    }
	}
    }

    for (u = 0; u < 0x10000L; u++) {
	ret = unicode_to_big5(u, &r, &c);
	if (ret) {
	    uu = big5_to_unicode(r, c);
	    if (uu == ERROR)
		printf("U-%08lx -> (%d,%d) but didn't convert back\n",
		       u, r, c);
	    else if (uu != u)
		printf("U-%08lx -> (%d,%d) -> U-%08lx\n", u, r, c, uu);
	}
    }

    return 0;
}

#endif
