# XRACER (C) 1999-2000 Richard W.M. Jones <rich@annexia.org> and other AUTHORS
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
# $Id: xracer-blenderexport.py,v 1.2 2000/01/30 12:11:31 rich Exp $

# This Python program grabs the contents of the Blender file you
# are currently working on, and writes it out to an export file.
# The export file contains information such as (u,v) texture
# coordinates and layers, something not normally available from
# Blender file formats.
#
# You *need* a Blender C-key for any of this to work. This situation
# is likely to remain until the release of Blender 2.0 under GPL in
# the summer.
#
# To run the script (from Blender, after you've drawn your scene):
#
#   SHIFT-F11.
#   Select OPEN NEW from the MenuBut.
#   Select the name of this file, and press F2.
#   Press ALT-PKEY to run the script.
#   The script will take a few seconds to run.
#
# If it worked, you'll have a file in the current directory called
# blender.export which you can then pass to the Perl tools to generate
# actual tracks and scenery objects.
#
# I'm really not a big fan of Python, but Blender uses it as its
# scripting language, so ... the aim is to do the minimum possible
# here, and get Perl to handle the main job.

import Blender

# Open output file.
filename = "blender.export"
file = open (filename, "w")

# Don't know how to do this more naturally (XXX).
meshtype = type (Blender.NMesh.GetRaw ())

# Iterate over the objects in turn.
objs = Blender.Object.Get ()
for obj in objs:

    # Get the name and layer that the object is in.
    name = obj.name
    layer = obj.Layer
    file.write ("# object name: %s layer: %d\n" % (name, layer))

    # Get the material.
    mat = Blender.Material.Get (name)

    # Get the mesh.
    try:
        mesh = obj.data

        if type(mesh) == meshtype:
            has_col = mesh.has_col
            has_uvco = mesh.has_uvco

            file.write ("%s %d %d %d " % (name, layer, has_col, has_uvco))
            if mat:
                file.write ("1\n")
            else:
                file.write ("0\n")

            num_faces = len (mesh.faces)
            file.write ("# faces\n")
            file.write ("%d\n" % num_faces)

            for face in mesh.faces:
                num_vertices = len (face.v)
                file.write ("# vertices\n")
                file.write ("%d\n" % (num_vertices))

                for vertex in face.v:
                    file.write ("%g %g %g " % (vertex.co[0],
                                               vertex.co[1],
                                               vertex.co[2]))
                    if has_uvco:
                        file.write ("%g %g " % (vertex.uvco[0],
                                                vertex.uvco[1]))
                    file.write ("%g %g %g\n" % (vertex.no[0],
                                                vertex.no[1],
                                                vertex.no[2]))
                if has_col:
                    file.write ("# colours\n")
                    num_col = len (face.col)
                    file.write ("%d\n" % (num_col))
                    for col in face.col:
                        file.write ("%g %g %g %g\n" % (col.r, col.g, col.b, col.a))

            file.write ("# matrices\n")
            file.write ("%g %g %g\n" % (obj.SizeX, obj.SizeY, obj.SizeZ))
            file.write ("%g %g %g\n" % (obj.RotX, obj.RotY, obj.RotZ))
            file.write ("%g %g %g\n" % (obj.LocX, obj.LocY, obj.LocZ))

            if mat:
                file.write ("# material\n")
                file.write ("%g %g %g\n" % (mat.R, mat.G, mat.B))
        else:
            file.write ("# not a mesh (type was: %s), ignored ...\n"
                        % (type (mesh)));
    except (SystemError):
        file.write ("# no object data, ignored ...\n")

file.close ()
