/* pathjumpui.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2012-2017 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "pathjumpui.hh"
#include <aguix/aguix.h>
#include <aguix/awindow.h>
#include <aguix/text.h>
#include <aguix/fieldlistview.h>
#include <aguix/button.h>
#include <aguix/solidbutton.h>
#include <aguix/util.h>
#include <aguix/choosebutton.h>
#include <aguix/cyclebutton.h>
#include <aguix/acontainerbb.h>
#include "worker_locale.h"
#include "nwc_path.hh"
#include <algorithm>
#include "worker.h"
#include "deeppathstore.hh"
#include "bookmarkdbproxy.hh"
#include "persdeeppathstore.hh"
#include "prefixdb.hh"
#include "wconfig.h"
#include "pers_kvp.hh"
#include <functional>

int PathJumpUI::s_pathjump_number = 1;

PathJumpUI::PathJumpUI( AGUIX &aguix, DeepPathStore &path_store, Worker &worker )
  : m_aguix( aguix ),
    m_worker( worker ),
    m_path_store( path_store ),
    m_current_depth( -1 ),
    m_show_hidden_entries( false ),
    m_entry_filter_mode( SHOW_ALL ),
    m_lv_dirty( false),
    m_current_time( 0 ),
    m_ignore_date_for_sorting( false ),
    m_oldest_time( 0 )
{
    std::list< std::pair< std::string, time_t > > paths = m_path_store.getPaths();
    int max_path_components = 0;
    std::list< std::pair< std::string, time_t > > pers_paths = worker.getPathsPers();

    m_bookmarks = Worker::getBookmarkDBInstance().getNamesOfEntries( "" );

    paths.sort();

    DeepPathStore temp_store_paths;

    for ( std::list< std::pair< std::string, time_t > >::const_iterator it1 = paths.begin();
          it1 != paths.end();
          it1++ ) {
        m_entries.push_back( pathjump_entry( PATH_SHOW, -1, it1->first, it1->second ) );

        temp_store_paths.storePath( it1->first, it1->second );

        int slashes = std::count( it1->first.begin(),
                                  it1->first.end(),
                                  '/' );

        if ( slashes + 1 > max_path_components ) {
            max_path_components = slashes + 1;
        }
    }

    DeepPathStore temp_store;

    time_t now = time( NULL );

    m_current_time = now;

    for ( std::list< std::string >::const_iterator it1 = m_bookmarks.begin();
          it1 != m_bookmarks.end();
          it1++ ) {
        //TODO it would be slightly faster to check if path is already
        //     stored in temp_store_paths but there is no
        //     corresponding function at the moment
        DeepPathNode::node_change_t changed = DeepPathNode::NODE_UNCHANGED;

        temp_store_paths.storePath( *it1, 0, &changed );

        if ( changed != DeepPathNode::NODE_UNCHANGED ) {
            temp_store.storePath( *it1, 0 );
        }
    }

    for ( auto it1 = pers_paths.begin();
          it1 != pers_paths.end();
          it1++ ) {
        //TODO it would be slightly faster to check if path is already
        //     stored in temp_store_paths but there is no
        //     corresponding function at the moment
        DeepPathNode::node_change_t changed = DeepPathNode::NODE_UNCHANGED;

        temp_store_paths.storePath( it1->first, it1->second, &changed );

        if ( changed != DeepPathNode::NODE_UNCHANGED ) {
            temp_store.storePath( it1->first, it1->second );
        }
    }

    std::list< std::pair< std::string, time_t > > temp_paths = temp_store.getPaths();

    for ( auto it1 = temp_paths.begin();
          it1 != temp_paths.end();
          it1++ ) {
        m_entries.push_back( pathjump_entry( BOOKMARK_HIDE, -1, it1->first, it1->second ) );

        int slashes = std::count( it1->first.begin(),
                                  it1->first.end(),
                                  '/' );

        if ( slashes + 1 > max_path_components ) {
            max_path_components = slashes + 1;
        }

        if ( it1->second != 0 ) {
            if ( m_oldest_time == 0 ||
                 m_oldest_time > it1->second ) {
                m_oldest_time = it1->second;
            }
        }
    }

    // without leading slash there could be this many components,
    // add one for the initial slash
    max_path_components++;

    m_win = std::unique_ptr<AWindow>( new AWindow( &m_aguix,
                                                   0, 0,
                                                   400, 400,
                                                   catalog.getLocaleCom( 60 ),
                                                   AWindow::AWINDOW_DIALOG ) );
    m_win->create();

    m_co1 = m_win->setContainer( new AContainer( m_win.get(), 1, 6 ), true );
    m_co1->setMaxSpace( 5 );

    AContainer *co1_2 = m_co1->add( new AContainerBB( m_win.get(), 1, 2 ), 0, 0 );
    co1_2->setMaxSpace( 5 );

    AContainer *co1_2_1 = co1_2->add( new AContainer( m_win.get(), 2, 1 ), 0, 0 );
    co1_2_1->setMaxSpace( 5 );
    co1_2_1->setBorderWidth( 0 );
    co1_2_1->add( new Text( &m_aguix, 0, 0, catalog.getLocale( 758 ) ),
                  0, 0, AContainer::CO_FIX );
    m_infixtext = co1_2_1->addWidget( new Text( &m_aguix, 0, 0, "" ),
                                      1, 0, AContainer::CO_INCW );


    m_hold_co = co1_2->add( new AContainer( m_win.get(), 4, 1 ), 0, 1 );
    m_hold_co->setMaxSpace( 5 );
    m_hold_co->setBorderWidth( 0 );
    m_hold_co->add( new Text( &m_aguix, 0, 0, catalog.getLocale( 1029 ) ),
                0, 0, AContainer::CO_FIX );
    m_hold_status_b = m_hold_co->addWidget( new Button( &m_aguix, 0, 0, catalog.getLocale( 1027 ), 0 ),
                                            1, 0, AContainer::CO_FIX );
    m_hold_co->add( new Text( &m_aguix, 0, 0, catalog.getLocale( 1030 ) ),
                2, 0, AContainer::CO_FIX );
    m_hold_filter_text = m_hold_co->addWidget( new Text( &m_aguix, 0, 0, "" ),
                                           3, 0, AContainer::CO_INCW );

    AContainer *co1_4 = m_co1->add( new AContainerBB( m_win.get(), 1, 2 ), 0, 1 );
    co1_4->setMaxSpace( 5 );
    AContainer *co1_4_1 = co1_4->add( new AContainer( m_win.get(), 2, 1 ), 0, 0 );
    co1_4_1->setMaxSpace( 5 );
    co1_4_1->setBorderWidth( 0 );
    co1_4_1->add( new Text( &m_aguix, 0, 0, catalog.getLocale( 980 ) ),
                  0, 0, AContainer::CO_FIX );
    m_filter_mode_cyb = co1_4_1->addWidget( new CycleButton( &m_aguix, 0, 0, 50, 0 ),
                                            1, 0, AContainer::CO_INCWNR );

    m_filter_mode_cyb->addOption( catalog.getLocale( 981 ) );
    m_filter_mode_cyb->addOption( catalog.getLocale( 982 ) );
    m_filter_mode_cyb->addOption( catalog.getLocale( 983 ) );

    m_filter_mode_cyb->resize( m_filter_mode_cyb->getMaxSize(),
                               m_filter_mode_cyb->getHeight() );
    co1_4_1->readLimits();

    AContainer *co1_op = co1_4->add( new AContainer( m_win.get(), 2, 1 ), 0, 1 );
    co1_op->setMaxSpace( 5 );
    co1_op->setBorderWidth( 0 );

    m_show_all_cb = (ChooseButton*)co1_op->add( new ChooseButton( &m_aguix, 0, 0,
                                                                  false,
                                                                  catalog.getLocale( 977 ),
                                                                  LABEL_RIGHT, 0 ),
                                                0, 0, AContainer::CO_FIXNR );

    m_ignore_date_cb = (ChooseButton*)co1_op->add( new ChooseButton( &m_aguix, 0, 0,
                                                                     m_ignore_date_for_sorting,
                                                                     catalog.getLocale( 1111 ),
                                                                     LABEL_RIGHT, 0 ),
                                                   1, 0, AContainer::CO_INCWNR );

    AContainer *co_age = m_co1->add( new AContainerBB( m_win.get(), 7, 1 ), 0, 2 );
    co_age->setMaxSpace( 5 );

    co_age->add( new Text( &m_aguix, 0, 0, catalog.getLocale( 1154 ) ),
                 0, 0, AContainer::CO_FIX );
    m_age_text = co_age->addWidget( new Text( &m_aguix, 0, 0, "----" ),
                                    1, 0, AContainer::CO_FIX );

    {
        int textw = 0;
        for ( int k = 1105; k <= 1110; k++ ) {
            int tw = m_aguix.getTextWidth( catalog.getLocale( k ) );
            if ( tw > textw ) {
                textw = tw;
            }
        }
        co_age->setMinWidth( textw, 1, 0 );
        co_age->setMaxWidth( textw, 1, 0 );
    }
    
    m_plus1h_b = co_age->addWidget( new Button( &m_aguix, 0, 0, catalog.getLocale( 1155 ), 0 ),
                                    2, 0, AContainer::CO_FIX );
    m_plus1d_b = co_age->addWidget( new Button( &m_aguix, 0, 0, catalog.getLocale( 1156 ), 0 ),
                                    3, 0, AContainer::CO_FIX );
    m_plus1w_b = co_age->addWidget( new Button( &m_aguix, 0, 0, catalog.getLocale( 1157 ), 0 ),
                                    4, 0, AContainer::CO_FIX );
    m_age_reset_b = co_age->addWidget( new Button( &m_aguix, 0, 0, catalog.getLocale( 1158 ), 0 ),
                                       5, 0, AContainer::CO_FIX );

    m_breadcrumb_co = m_co1->add( new AContainer( m_win.get(), max_path_components + 1, 1 ), 0, 3 );
    m_breadcrumb_co->setMaxSpace( 5 );

    for ( int i = 0; i < max_path_components; i++ ) {
        Button *b = (Button*)m_breadcrumb_co->add( new Button( &m_aguix,
                                                               0,
                                                               0,
                                                               "",
                                                               0 ), i, 0, AContainer::CO_INCW );
        m_breadcrumb_buttons.push_back( b );
        b->setAcceptFocus( false );
        b->hide();
    }

    AContainer *co1_5 = m_co1->add( new AContainer( m_win.get(), 2, 1 ), 0, 4 );
    co1_5->setMaxSpace( 5 );
    co1_5->setBorderWidth( 0 );

    m_lv = co1_5->addWidget( new FieldListView( &m_aguix, 
                                                0, 0, 
                                                400, 300, 0 ),
                             0, 0, AContainer::CO_MIN );

    m_lv->setNrOfFields( 3 );
    m_lv->setShowHeader( true );
    m_lv->setFieldText( 1, catalog.getLocale( 1112 ) );
    m_lv->setFieldText( 2, catalog.getLocale( 809 ) );
    m_lv->setHBarState( 2 );
    m_lv->setVBarState( 2 );
    m_lv->setAcceptFocus( true );
    m_lv->setDisplayFocus( true );
    m_lv->setFieldText( 0, catalog.getLocale( 978 ) );
    m_lv->setGlobalFieldSpace( 5 );

    m_lv->setOnDemandCallback( [this]( FieldListView *lv,
                                       int row,
                                       int field,
                                       struct FieldListView::on_demand_data &data )
                               {
                                   updateLVData( row, field, data );
                               } );

    m_class_counts_lv = co1_5->addWidget( new FieldListView( &m_aguix, 
                                                             0, 0, 
                                                             100, 100, 0 ),
                                          1, 0, AContainer::CO_INCH );
    m_class_counts_lv->setNrOfFields( 3 );
    m_class_counts_lv->setShowHeader( true );
    m_class_counts_lv->setFieldText( 0, catalog.getLocale( 978 ) );
    m_class_counts_lv->setFieldText( 1, catalog.getLocale( 1031 ) );
    m_class_counts_lv->setFieldText( 2, catalog.getLocale( 1032 ) );

    m_class_counts_lv->setHBarState( 2 );
    m_class_counts_lv->setVBarState( 2 );
    m_class_counts_lv->setAcceptFocus( true );
    m_class_counts_lv->setDisplayFocus( true );
    m_class_counts_lv->setGlobalFieldSpace( 5 );
    m_class_counts_lv->setConsiderHeaderWidthForDynamicWidth( true );

    m_class_counts_lv->maximizeX();

    co1_5->readLimits();

    AContainer *co1_3 = m_co1->add( new AContainer( m_win.get(), 5, 1 ), 0, 5 );
    co1_3->setMinSpace( 5 );
    co1_3->setMaxSpace( -1 );
    co1_3->setBorderWidth( 0 );
    m_okb = (Button*)co1_3->add( new Button( &m_aguix,
                                             0,
                                             0,
                                             catalog.getLocale( 761 ),
                                             0 ), 0, 0, AContainer::CO_FIX );
    m_panelize_b = co1_3->addWidget( new Button( &m_aguix,
                                                 0,
                                                 0,
                                                 catalog.getLocale( 1034 ),
                                                 0 ), 1, 0, AContainer::CO_FIX );
    m_checkb = (Button*)co1_3->add( new Button( &m_aguix,
                                                0,
                                                0,
                                                catalog.getLocale( 1115 ),
                                                0 ), 2, 0, AContainer::CO_FIX );
    m_cleanupb = (Button*)co1_3->add( new Button( &m_aguix,
                                                  0,
                                                  0,
                                                  catalog.getLocale( 984 ),
                                                  0 ), 3, 0, AContainer::CO_FIX );
    m_cancelb = (Button*)co1_3->add( new Button( &m_aguix,
                                                 0,
                                                 0,
                                                 catalog.getLocale( 633 ),
                                                 0 ), 4, 0, AContainer::CO_FIX );

    m_win->contMaximize( true );
    m_win->setDoTabCycling( true );

    m_entry_filter_age = 0;
}

PathJumpUI::~PathJumpUI()
{
}

int PathJumpUI::mainLoop()
{
    m_lv_dirty = true;

    apply_filter( m_filter, NULL );

    // do run2completion for the first call to have everything
    // initialized
    while ( ! m_apply_state.done ) {
        apply_filter_next();
    }

    highlight_best_hit( m_dirname );

    showData();

    maximizeWin();

    int tw = Worker::getPersKVPStore().getIntValue( "pathjumpwin-width",
                                                    m_win->getWidth(),
                                                    10000 );
    int th = Worker::getPersKVPStore().getIntValue( "pathjumpwin-height",
                                                    m_win->getHeight(),
                                                    10000 );
    m_win->resize( tw, th );

    m_win->show();

    m_lv->takeFocus();

    AGMessage *msg;
    int endmode = 0;

    for ( ; endmode == 0; ) {
        if ( m_apply_state.done ) {
            msg = m_aguix.WaitMessage( NULL );
        } else {
            msg = m_aguix.GetMessage( NULL );
        }
        if ( msg != NULL ) {
            switch ( msg->type ) {
              case AG_CLOSEWINDOW:
                  endmode = -1;
                  break;
              case AG_BUTTONCLICKED:
                  if ( msg->button.button == m_okb ) {
                      endmode = 1;
                  } else if ( msg->button.button == m_cleanupb ) {
                      cleanup();
                      endmode = -1;
                  } else if ( msg->button.button == m_checkb ) {
                      checkExistance();
                  } else if ( msg->button.button == m_panelize_b ) {
                      endmode = 2;
                  } else if ( msg->button.button == m_cancelb ) {
                      endmode = -1;
                  } else if ( msg->button.button == m_plus1h_b ) {
                      m_entry_filter_age += 1 * 60 * 60;

                      apply_filter( m_filter, [this]( int matches ) {
                              showData();
                              updateAgeText();
                          });
                  } else if ( msg->button.button == m_plus1d_b ) {
                      m_entry_filter_age += 1 * 24 * 60 * 60;

                      apply_filter( m_filter, [this]( int matches ) {
                              showData();
                              updateAgeText();
                          });
                  } else if ( msg->button.button == m_plus1w_b ) {
                      m_entry_filter_age += 7 * 24 * 60 * 60;

                      apply_filter( m_filter, [this]( int matches ) {
                              showData();
                              updateAgeText();
                          });
                  } else if ( msg->button.button == m_age_reset_b ) {
                      m_entry_filter_age = 0;

                      apply_filter( m_filter, [this]( int matches ) {
                              showData();
                              updateAgeText();
                          });
                  } else if ( msg->button.button == m_hold_status_b ) {
                      if ( m_hold_status.active ) {
                          resetHeldEntriesAndUpdateUI();
                      } else {
                          showHoldHelp();
                      }
                  } else {
                      for ( unsigned int i = 0; i < m_breadcrumb_buttons.size(); i++ ) {
                          if ( m_breadcrumb_buttons[i] == msg->button.button ) {
                              m_current_depth = i;
                              endmode = 1;
                          }
                      }
                  }
                  break;
              case AG_KEYPRESSED:
                  if ( msg->key.key == XK_BackSpace ) {
                      if ( KEYSTATEMASK( msg->key.keystate ) == ShiftMask ) {
                          m_filter = "";
                          apply_filter( m_filter, [this]( int matches ) {
                                  showData();
                              });
                      } else {
                          std::string cur_infix = m_filter;
                          
                          if ( cur_infix.length() > 0 ) {
                              cur_infix.resize( cur_infix.length() - 1 );
                              m_filter = cur_infix;
                              apply_filter( m_filter, [this]( int matches ) {
                                      showData();
                                  });
                          }
                      }
                  } else if ( msg->key.key == XK_Return ) {
                      endmode = 1;
                  } else if ( msg->key.key == XK_Escape ) {
                      endmode = -1;
                  } else if ( msg->key.key == XK_Left ) {
                      if ( ! m_current_components.empty() ) {
                          if ( m_current_depth < 0 ) {
                              m_current_depth = m_current_components.size();
                          } else if ( m_current_depth > 0 ) {
                              if ( m_current_depth > (int)m_current_components.size() &&
                                   m_current_components.size() >= 1 ) {
                                  m_current_depth = m_current_components.size();
                              }

                              m_current_depth--;
                          }

                          if ( m_entry_filter_mode != SHOW_ALL ) {
                              post_filtering();
                              showData();
                          } else {
                              update_breadcrumb();
                          }
                      }
                  } else if ( msg->key.key == XK_Right ) {
                      if ( ! m_current_components.empty() ) {
                          if ( KEYSTATEMASK( msg->key.keystate ) == ShiftMask ) {
                              m_current_depth = (int)( m_current_components.size() );
                          } else {
                              if ( m_current_depth < 0 ) {
                                  m_current_depth = 0;
                              } else if ( m_current_depth <= (int)( m_current_components.size() - 1 ) ) {
                                  m_current_depth++;
                              }
                          }

                          if ( m_entry_filter_mode != SHOW_ALL ) {
                              post_filtering();
                              showData();
                          } else {
                              update_breadcrumb();
                          }
                      }
                  } else if ( msg->key.key == XK_Up ) {
                      m_lv->takeFocus();
                  } else if ( msg->key.key == XK_Down ) {
                      m_lv->takeFocus();
                  } else if ( msg->key.key == XK_Next ) {
                  } else if ( msg->key.key == XK_Prior ) {
                  } else if ( msg->key.key == XK_Home ) {
                      // key is grabbed by listview so it has no effect at the moment
                      if ( ! m_current_components.empty() ) {
                          m_current_depth = 0;
                          update_breadcrumb();
                      }
                  } else if ( msg->key.key == XK_End ) {
                      // key is grabbed by listview so it has no effect at the moment
                      if ( ! m_current_components.empty() ) {
                          m_current_depth = m_current_components.size();
                          update_breadcrumb();
                      }
                  } else if ( IsModifierKey( msg->key.key ) ||
                              IsCursorKey( msg->key.key ) ||
                              IsPFKey( msg->key.key ) ||
                              IsFunctionKey( msg->key.key ) ||
                              IsMiscFunctionKey( msg->key.key ) ||
                              ( ( msg->key.key >= XK_BackSpace ) && ( msg->key.key <= XK_Escape ) ) ) {
                  } else if ( msg->key.key >= XK_0 && msg->key.key <= XK_9 &&
                              KEYSTATEMASK( msg->key.keystate ) == ControlMask ) {
                      if ( msg->key.key == XK_0 ) {
                          resetHeldEntriesAndUpdateUI();
                      } else {
                          holdCurrentEntriesAndUpdateUI( msg->key.key - XK_0 );
                      }
                  } else if ( strlen( msg->key.keybuf ) > 0 ) {
                      if ( msg->key.key == XK_a &&
                           KEYSTATEMASK( msg->key.keystate ) == ControlMask ) {
                          if ( ! m_hold_status.active ) {
                              m_show_hidden_entries = ! m_show_hidden_entries;
                              m_show_all_cb->setState( m_show_hidden_entries );
                              apply_filter( m_filter, [this]( int matches ) {
                                      showData();
                                  });
                          }
                      } else if ( msg->key.key == XK_t &&
                                  KEYSTATEMASK( msg->key.keystate ) == ControlMask ) {
                          m_ignore_date_for_sorting = ! m_ignore_date_for_sorting;
                          m_ignore_date_cb->setState( m_ignore_date_for_sorting );
                          apply_filter( m_filter, [this]( int matches ) {
                                  showData();
                              });
                      } else if ( msg->key.key == XK_f &&
                           KEYSTATEMASK( msg->key.keystate ) == ControlMask ) {
                          switch ( m_entry_filter_mode ) {
                              case SHOW_ONLY_SUBDIRS:
                                  m_entry_filter_mode = SHOW_ONLY_DIRECT_SUBDIRS;
                                  m_filter_mode_cyb->setOption( 2 );
                                  break;
                              case SHOW_ONLY_DIRECT_SUBDIRS:
                                  m_entry_filter_mode = SHOW_ALL;
                                  m_filter_mode_cyb->setOption( 0 );
                                  break;
                              default:
                                  m_entry_filter_mode = SHOW_ONLY_SUBDIRS;
                                  m_filter_mode_cyb->setOption( 1 );
                                  break;
                          }
                          apply_filter( m_filter, [this]( int matches ) {
                                  showData();
                              });
                      } else if ( msg->key.key == XK_h &&
                                  KEYSTATEMASK( msg->key.keystate ) == ControlMask ) {
                          m_entry_filter_age += 1 * 60 * 60;

                          apply_filter( m_filter, [this]( int matches ) {
                                  showData();
                                  updateAgeText();
                              });
                      } else if ( msg->key.key == XK_d &&
                                  KEYSTATEMASK( msg->key.keystate ) == ControlMask ) {
                          m_entry_filter_age += 1 * 24 * 60 * 60;

                          apply_filter( m_filter, [this]( int matches ) {
                                  showData();
                                  updateAgeText();
                              });
                      } else if ( msg->key.key == XK_w &&
                                  KEYSTATEMASK( msg->key.keystate ) == ControlMask ) {
                          m_entry_filter_age += 7 * 24 * 60 * 60;

                          apply_filter( m_filter, [this]( int matches ) {
                                  showData();
                                  updateAgeText();
                              });
                      } else {
                          std::string cur_infix = m_filter;
                          cur_infix += msg->key.keybuf;

                          apply_filter( cur_infix, [this]( int matches ) {
                                  showData();
                              });
                      }
                  }
                  break;
              case AG_FIELDLV_DOUBLECLICK:
                  if ( msg->fieldlv.lv == m_lv ) {
                      // double click in lv, actual element is unimportant here
                      endmode = 1;
                  } else if ( msg->fieldlv.lv == m_class_counts_lv ) {
                      holdCurrentEntriesAndUpdateUI( msg->fieldlv.row + 1 );
                  }
                  break;
                case AG_FIELDLV_ONESELECT:
                case AG_FIELDLV_MULTISELECT:
                    if ( msg->fieldlv.lv == m_lv ) {
                        handleMainLVRowChange( m_lv->getActiveRow() );
                    } else if ( msg->fieldlv.lv == m_class_counts_lv ) {
                        handleMatchClassRow( msg->fieldlv.row );
                    }
                    break;
                case AG_CHOOSECLICKED:
                    if ( msg->choose.button == m_show_all_cb ) {
                        m_show_hidden_entries = msg->choose.state;
                        apply_filter( m_filter, [this]( int matches ) {
                                showData();
                            });
                    } else if ( msg->choose.button == m_ignore_date_cb ) {
                        m_ignore_date_for_sorting = m_ignore_date_cb->getState();
                        apply_filter( m_filter, [this]( int matches ) {
                                showData();
                            });
                    }
                    break;
                case AG_CYCLEBUTTONCLICKED:
                    if ( msg->cyclebutton.cyclebutton == m_filter_mode_cyb ) {
                        switch ( msg->cyclebutton.option ) {
                            case 1:
                                m_entry_filter_mode = SHOW_ONLY_SUBDIRS;
                                break;
                            case 2:
                                m_entry_filter_mode = SHOW_ONLY_DIRECT_SUBDIRS;
                                break;
                            default:
                                m_entry_filter_mode = SHOW_ALL;
                                break;
                        }
                        apply_filter( m_filter, [this]( int matches ) {
                                showData();
                            });
                    }
                    break;
            }
            m_aguix.ReplyMessage( msg );
        }

        if ( ! m_apply_state.done ) {
            apply_filter_next();
        }
    }

    m_selected_path = "";

    if ( endmode == 2 ) {
    } else if ( endmode == 1 ) {
        std::string res = "";

        // finalize running filter
        while ( ! m_apply_state.done ) {
            apply_filter_next();
        }

        if ( m_current_depth <= 0 ) {
            res = "/";
        } else {
            for ( unsigned int i = 0; i < m_current_components.size(); i++ ) {
                if ( (int)i >= m_current_depth ) break;

                res += "/";
                res += m_current_components[i];
            }
        }

        m_selected_path = res;

        // check if this path is allowed to be tracked, than put its
        // access into the prefixdb
        if ( ! m_filter.empty() && ! wconfig->getPathJumpAllowDirs().empty() ) {
            const std::list< std::string > &l = wconfig->getPathJumpAllowDirs();

            for ( std::list< std::string >::const_iterator it1 = l.begin();
                  it1 != l.end();
                  it1++ ) {
                if ( AGUIXUtils::starts_with( m_selected_path, *it1 ) ) {
                    m_worker.getDirHistPrefixDB().pushAccess( m_filter, m_selected_path );
                    break;
                }
            }
        }

        m_worker.setPathJumpInProgress( m_filter );
    }

    Worker::getPersKVPStore().setIntValue( "pathjumpwin-width",
                                           m_win->getWidth() );
    Worker::getPersKVPStore().setIntValue( "pathjumpwin-height",
                                           m_win->getHeight() );

    m_win->hide();
    
    return endmode;
}

void PathJumpUI::showData()
{
    m_ondemand_info.matcher.setMatchString( m_filter );

    if ( m_lv_dirty ||
         m_entry_filter_mode == SHOW_ONLY_DIRECT_SUBDIRS ) {

        m_lv->setSize( 0 );
        m_entry_fullnames.resize( 0 );
    
        int size = 0, pos = 0;

        std::string prefix_filter;

        if ( m_entry_filter_mode == SHOW_ONLY_SUBDIRS ||
             m_entry_filter_mode == SHOW_ONLY_DIRECT_SUBDIRS ) {
            int i = 0;

            if ( m_current_depth > 0 ) {
                for ( std::vector< std::string >::const_iterator it1 = m_current_components.begin();
                      it1 != m_current_components.end();
                      it1++, i++ ) {
                    prefix_filter += "/";
                    prefix_filter += *it1;
                    if ( i + 1 >= m_current_depth ) break;
                }
            }
        }

        size_t l = prefix_filter.length();
        bool current_entry_found = false;

        m_ondemand_info.prefix_filter_length = l;

        for ( auto it1 = m_entries.begin();
              it1 != m_entries.end();
              it1++, pos++ ) {
            if ( ( it1->m_type == PATH_SHOW ||
                   it1->m_type == BOOKMARK_SHOW ) &&
                 it1->m_post_filtering &&
                 it1->m_always_ignore == false ) {
                int row = m_lv->addRow();

                m_entry_fullnames.push_back( std::make_pair( it1->m_path,
                                                             it1->m_last_access ) );

                if ( it1->m_blockcount >= 0 && ! m_filter.empty() ) {
                    std::string qual = AGUIXUtils::formatStringToString( "%3.0f%%",
                                                                         100.0 / (double)( it1->m_blockcount + 1 ) );
                    m_lv->setText( row, 0, qual );
                }

                m_lv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );

                if ( m_current_entry == it1->m_path ) {
                    m_lv->setActiveRow( row );
                    current_entry_found = true;
                }

                size++;
            }
        }

        if ( ! current_entry_found ) {
            for ( int row = 0; row < m_lv->getElements(); row++ ) {
                if ( AGUIXUtils::starts_with( m_current_entry,
                                              m_lv->getText( row, 2 ) ) ) {
                    m_lv->setActiveRow( row );
                    break;
                }
            }
        }

        calculateMatchClasses();

        m_class_counts_lv->setSize( 0 );

        for ( auto &e : m_match_classes ) {
            int row = m_class_counts_lv->addRow();

            std::string t1;

            if ( e.first >= 0 ) {
                t1 = AGUIXUtils::formatStringToString( "%3.0f%%",
                                                       100.0 / (double)( e.first + 1 ) );

                m_class_counts_lv->setText( row, 0, t1 );
            } else {
                m_class_counts_lv->setText( row, 0, "100%" );
            }

            if ( row == 0 ) {
                t1 = AGUIXUtils::formatStringToString( "%d",
                                                       e.second );
            } else {
                t1 = AGUIXUtils::formatStringToString( "+%d",
                                                       e.second );
            }

            m_class_counts_lv->setText( row, 1, t1 );

            if ( row < 9 ) {
                t1 = AGUIXUtils::formatStringToString( "Ctrl-%d",
                                                       row + 1 );

                m_class_counts_lv->setText( row, 2, t1 );
            }

            m_class_counts_lv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
        }

        m_class_counts_lv->redraw();

        m_lv->showActive();
        m_lv->redraw();
    }

    update_breadcrumb();

    m_lv_dirty = false;
}

std::string PathJumpUI::getSelectedPath()
{
    return m_selected_path;
}

void PathJumpUI::setCurrentDirname( const std::string &dirname )
{
    m_dirname = dirname;
}

void PathJumpUI::setCurrentBasename( const std::string &basename )
{
    m_basename = basename;
}

void PathJumpUI::maximizeWin()
{
    m_lv->maximizeX();
    m_lv->maximizeY();
    int my_w = m_lv->getWidth() + 10;
    int my_h = m_lv->getHeight() + 10;

    if ( my_w < 400 ) my_w = 400;
    if ( my_h < 300 ) my_h = 300;

    int rx, ry, rw, rh;

    m_aguix.getLargestDimensionOfCurrentScreen( &rx, &ry,
                                                &rw, &rh );

    int mw = rw * 80 / 100;
    int mh = rh * 80 / 100;
    m_co1->resize( mw, mh );
    m_co1->rearrange();

    if ( my_w < m_lv->getWidth() ) {
        m_co1->setMinWidth( my_w, 0, 4 );
    } else {
        m_co1->setMinWidth( m_lv->getWidth(), 0, 4 );
    }
    if ( my_h < m_lv->getHeight() ) {
        m_co1->setMinHeight( my_h, 0, 4 );
    } else {
        m_co1->setMinHeight( m_lv->getHeight(), 0, 4 );
    }
    m_win->contMaximize( true );
}

void PathJumpUI::highlight_best_hit( const std::string &filter )
{
    find_entry( filter, "" );

    post_filtering();

    showData();
}

void PathJumpUI::apply_filter( const std::string filter,
                               std::function< void( int ) > finalize )
{
    m_apply_state = apply_filter_state();

    m_filter = filter;

    std::string visible_filter = m_filter;
    visible_filter += "...";
    m_infixtext->setText( visible_filter.c_str() );

    m_apply_state.flmatch.setMatchString( filter );
    m_apply_state.flmatch.setMatchCaseSensitive( false );

    m_apply_state.flexible_string_filter = "*" + StringMatcherFNMatch::convertMatchStringToFlexibleMatch( filter ) + "*";

    m_apply_state.flprematcher.setMatchString( m_apply_state.flexible_string_filter );
    m_apply_state.flprematcher.setMatchCaseSensitive( false );

    m_apply_state.finalize = finalize;

    if ( filter.find( "*" ) != std::string::npos ) {
        std::string real_filter = "*";
        real_filter += filter;
        real_filter += "*";

        m_apply_state.match.setMatchString( real_filter );
        m_apply_state.match.setMatchCaseSensitive( false );

        m_apply_state.use_flexible = false;
    }

    m_apply_state.it1 = m_entries.begin();

    m_apply_state.filter = filter;
}

void PathJumpUI::apply_filter_next()
{
    if ( m_apply_state.done == true ) {
        return;
    }
    
    if ( m_apply_state.it1 != m_entries.end() ) {
        if ( m_apply_state.it1->m_always_ignore ) {
            m_apply_state.it1++;

            return;
        }

        if ( m_apply_state.use_flexible ) {
            // check with fnmatch first which is faster than the flexiblematcher but can't return the blockcount
            if ( m_apply_state.flprematcher.match( m_apply_state.it1->m_path ) ) {
                m_apply_state.it1->m_blockcount = m_apply_state.flmatch.countNonMatchingBlocks<false>( m_apply_state.it1->m_path, NULL );
            } else {
                m_apply_state.it1->m_blockcount = -1;
            }
        } else {
            m_apply_state.it1->m_blockcount = m_apply_state.match.match( m_apply_state.it1->m_path ) ? 0 : -1;
        }

        if ( m_apply_state.it1->m_blockcount == 0 ) {
            if ( m_apply_state.it1->m_type == PATH_SHOW ||
                 m_apply_state.it1->m_type == PATH_HIDE ) {
                m_apply_state.exact_path_matches++;
                m_apply_state.path_matches++;
            } else {
                m_apply_state.exact_bookmark_matches++;
                m_apply_state.bookmark_matches++;
            }
        } else if ( m_apply_state.it1->m_blockcount > 0 ) {
            if ( m_apply_state.it1->m_type == PATH_SHOW ||
                 m_apply_state.it1->m_type == PATH_HIDE ) {
                m_apply_state.path_matches++;
            } else {
                m_apply_state.bookmark_matches++;
            }
        }

        m_apply_state.it1++;

        return;
    }

    m_entries.sort( std::bind( pathjump_entry::compare,
                               std::placeholders::_1,
                               std::placeholders::_2,
                               m_ignore_date_for_sorting ) );

    for ( auto it1 = m_entries.begin();
          it1 != m_entries.end();
          it1++ ) {
        if ( it1->m_always_ignore ) continue;

        if ( it1->m_type == PATH_SHOW ||
             it1->m_type == PATH_HIDE ) {
            if ( it1->m_blockcount == 0 ||
                 ( it1->m_blockcount > 0 &&
                   ( ! m_apply_state.prefer_exact_matches || ( m_apply_state.prefer_exact_matches && m_apply_state.exact_path_matches == 0 ) ) ) ) {
                it1->m_type = PATH_SHOW;
                m_apply_state.matches++;
            } else {
                it1->m_type = PATH_HIDE;
            }
        } else {
            if ( ( m_apply_state.path_matches == 0 && ! m_apply_state.filter.empty() ) ||
                 m_show_hidden_entries == true ||
                 m_hold_status.active == true ) {
                if ( it1->m_blockcount == 0 ||
                     ( it1->m_blockcount > 0 &&
                       ( ! m_apply_state.prefer_exact_matches || ( m_apply_state.prefer_exact_matches && m_apply_state.exact_bookmark_matches == 0 ) ) ) ) {
                    it1->m_type = BOOKMARK_SHOW;
                    m_apply_state.matches++;
                } else {
                    it1->m_type = BOOKMARK_HIDE;
                }
            } else {
                    it1->m_type = BOOKMARK_HIDE;
            }
        }

        if ( m_entry_filter_age > 0 && m_current_time > m_entry_filter_age ) {
            if ( it1->m_last_access + m_entry_filter_age < m_current_time ) {
                if ( it1->m_type == BOOKMARK_SHOW ) {
                    it1->m_type = BOOKMARK_HIDE;
                } else {
                    it1->m_type = PATH_HIDE;
                }
            }
        }
    }

    // avoid calling find_entry when the filter hasn't been changed.
    // that way the current entry and depth does not change
    if ( m_apply_state.filter != m_previous_apply_filter ) {
        find_entry( m_current_entry, m_apply_state.filter );
        m_previous_apply_filter = m_apply_state.filter;
    }

    post_filtering();

    m_lv_dirty = true;

    m_apply_state.done = true;

    m_infixtext->setText( m_filter.c_str() );

    if ( m_apply_state.finalize ) {
        m_apply_state.finalize( m_apply_state.matches );
    }
}

int PathJumpUI::find_entry( const std::string &name, const std::string &filter )
{
    std::string entry_to_find;
    bool first_found = false;
    std::string best_hit;

    if ( ! filter.empty() ) {
        best_hit = entry_to_find = m_worker.getDirHistPrefixDB().getBestHit( filter );
    }

    if ( entry_to_find.empty() ) {
        entry_to_find = name;
    }

    for ( auto it1 = m_entries.begin();
          it1 != m_entries.end();
          it1++ ) {
        if ( it1->m_always_ignore ) continue;

        if ( it1->m_type == PATH_SHOW ||
             it1->m_type == BOOKMARK_SHOW ) {
            if ( AGUIXUtils::starts_with( it1->m_path,
                                          entry_to_find ) ||
                 first_found == false ) {
                m_current_entry = it1->m_path;

                if ( first_found ) {
                    break;
                } else {
                    first_found = true;
                }
            }
        }
    }

    m_current_depth = -1;

    if ( ! best_hit.empty() ) {
        std::string f = best_hit;
        f += "*";
        
        m_current_depth = find_best_matching_depth( m_current_entry, f );
    }

    if ( m_current_depth < 0 ) {
        if ( ! filter.empty() ) {
            m_current_depth = find_best_matching_depth( m_current_entry, filter );
        } else {
            // workaround to adjust the depth to the name argument
            m_current_depth = find_best_matching_depth( m_current_entry, name + "*" );
        }
    }

    if ( m_entry_filter_mode != SHOW_ALL ) {
        m_lv_dirty = true;
    }

    return 0;
}

int PathJumpUI::set_entry( const std::string &name )
{
    bool found = false;

    for ( auto it1 = m_entries.begin();
          it1 != m_entries.end();
          it1++ ) {
        if ( it1->m_always_ignore ) continue;

        if ( ( it1->m_type == PATH_SHOW ||
               it1->m_type == BOOKMARK_SHOW ) &&
             it1->m_path == name ) {
            m_current_entry = it1->m_path;
            found = true;
            break;
        }
    }

    if ( found ) {
        if ( ! m_filter.empty() &&
             m_entry_filter_mode == SHOW_ALL ) {
            m_current_depth = find_best_matching_depth( m_current_entry, m_filter );
        }
    } else {
        find_entry( name, m_filter );
    }

    if ( m_entry_filter_mode != SHOW_ALL ) {
        m_lv_dirty = true;
    }

    post_filtering();

    return 0;
}

int PathJumpUI::post_filtering()
{
    std::string prefix_filter;

    build_current_path_components( m_current_entry );

    if ( m_entry_filter_mode != SHOW_ONLY_SUBDIRS &&
         m_entry_filter_mode != SHOW_ONLY_DIRECT_SUBDIRS ) {

        for ( auto it1 = m_entries.begin();
              it1 != m_entries.end();
              it1++ ) {
            if ( it1->m_always_ignore ) continue;

            if ( it1->m_type == PATH_SHOW ||
                 it1->m_type == BOOKMARK_SHOW ) {

                if ( ! it1->m_post_filtering ) m_lv_dirty = true;

                it1->m_post_filtering = true;
            } else {

                if ( it1->m_post_filtering ) m_lv_dirty = true;

                it1->m_post_filtering = false;
            }
        }

        return 0;
    }

    if ( m_current_depth > 0 ) {
        int i = 0;
        for ( std::vector< std::string >::const_iterator it1 = m_current_components.begin();
              it1 != m_current_components.end();
              it1++, i++ ) {
            prefix_filter += "/";
            prefix_filter += *it1;
            if ( i + 1 >= m_current_depth ) break;
        }
    }

    size_t l = prefix_filter.length();
    std::map< std::string, bool > entry_visible;

    for ( auto it1 = m_entries.begin();
          it1 != m_entries.end();
          it1++ ) {
        if ( it1->m_always_ignore ) continue;

        bool new_state = false;

        if ( it1->m_type == PATH_SHOW ||
             it1->m_type == BOOKMARK_SHOW ) {
            new_state = true;
        }

        if ( ! prefix_filter.empty() &&
             ! NWC::Path::is_prefix_dir( prefix_filter, it1->m_path ) ) {
            new_state = false;
        }

        if ( m_entry_filter_mode == SHOW_ONLY_DIRECT_SUBDIRS &&
             new_state ) {

            size_t p2 = it1->m_path.find( '/', l + 1 );
            std::string s;
            if ( p2 != std::string::npos ) {
                s = std::string( it1->m_path, 0, p2 );
            } else {
                s = it1->m_path;
            }

            if ( entry_visible[s] == true ) {
                new_state = false;
            } else {
                entry_visible[s] = true;
            }
        }

        if ( it1->m_post_filtering != new_state ) m_lv_dirty = true;

        it1->m_post_filtering = new_state;
    }

    return 0;
}

int PathJumpUI::update_breadcrumb()
{
    unsigned int i = 0;
    unsigned int last_component = m_current_components.size();

    m_breadcrumb_buttons[i]->setText( 0, "/" );

    if ( m_current_depth == (int)i ||
         ( i == last_component && m_current_depth > (int)last_component ) ) {
        m_breadcrumb_buttons[i]->setFG( 0, 2 );
        m_breadcrumb_buttons[i]->setBG( 0, 3 );

        m_breadcrumb_co->setMinWidth( m_breadcrumb_buttons[i]->getMaximumWidth(),
                                      i, 0 );
    } else {
        m_breadcrumb_buttons[i]->setFG( 0, 1 );
        m_breadcrumb_buttons[i]->setBG( 0, 0 );

        m_breadcrumb_co->setMinWidth( 10,
                                      i, 0 );
    }

    m_breadcrumb_buttons[i]->show();

    m_breadcrumb_co->setMaxWidth( m_breadcrumb_buttons[i]->getMaximumWidth(),
                                  i, 0 );
    i++;

    for ( std::vector< std::string >::const_iterator it1 = m_current_components.begin();
          it1 != m_current_components.end();
          it1++ ) {
        if ( ! it1->empty() ) {
            m_breadcrumb_buttons[i]->setText( 0, it1->c_str() );

            if ( m_current_depth == (int)i ||
                 ( i == last_component && m_current_depth > (int)last_component ) ) {
                m_breadcrumb_buttons[i]->setFG( 0, 2 );
                m_breadcrumb_buttons[i]->setBG( 0, 3 );

                m_breadcrumb_co->setMinWidth( m_breadcrumb_buttons[i]->getMaximumWidth(),
                                              i, 0 );
            } else {
                m_breadcrumb_buttons[i]->setFG( 0, 1 );
                m_breadcrumb_buttons[i]->setBG( 0, 0 );

                m_breadcrumb_co->setMinWidth( 10,
                                              i, 0 );
            }

            m_breadcrumb_buttons[i]->show();

            m_breadcrumb_co->setMaxWidth( m_breadcrumb_buttons[i]->getMaximumWidth(),
                                          i, 0 );
            i++;
        }
    }

    for ( ; i < m_breadcrumb_buttons.size(); i++ ) {
        m_breadcrumb_buttons[i]->hide();
        m_breadcrumb_co->setMaxWidth( 0,
                                      i, 0 );
        m_breadcrumb_co->setMinWidth( 0,
                                      i, 0 );
    }

    m_win->updateCont();

    return 0;
}

int PathJumpUI::find_best_matching_depth( const std::string &filter )
{
    if ( m_lv->isValidRow( m_lv->getActiveRow() ) == true ) {
        return find_best_matching_depth( m_lv->getText( m_lv->getActiveRow(), 2 ),
                                         filter );
    }
    return -1;
}

int PathJumpUI::find_best_matching_depth( const std::string &path,
                                          const std::string &filter )
{
    std::vector< std::string > components;
    StringMatcherFNMatch matcher;
    int depth = -1;

    if ( filter.empty() ) return -1;

    AGUIXUtils::split_string( components, path, '/' );

    matcher.setMatchString( filter );
    matcher.setMatchCaseSensitive( false );

    if ( ! matcher.match( path ) ) {
        std::string flexible_filter;

        if ( filter[0] != '*' ) {
            flexible_filter = "*";
        }

        flexible_filter += StringMatcherFNMatch::convertMatchStringToFlexibleMatch( filter ) + "*";

        matcher.setMatchString( flexible_filter );
        matcher.setMatchCaseSensitive( false );
    }

    int i = 0;
    std::string prefix_path = "";

    for ( std::vector< std::string >::const_iterator it1 = components.begin();
          it1 != components.end();
          it1++ ) {
        if ( ! it1->empty() ) {
            prefix_path += "/" + *it1;

            if ( matcher.match( prefix_path ) ) {
                depth = i + 1;
                break;
            }

            i++;
        }
    }

    return depth;
}

int PathJumpUI::build_current_path_components()
{
    if ( m_lv->isValidRow( m_lv->getActiveRow() ) == true ) {
        return build_current_path_components( m_lv->getText( m_lv->getActiveRow(), 2 ) );
    }
    return 1;
}

int PathJumpUI::build_current_path_components( const std::string &path )
{
    std::vector< std::string > components;

    m_current_components.clear();

    AGUIXUtils::split_string( components, path, '/' );

    if ( components.size() >= m_breadcrumb_buttons.size() ) {
        // something wrong here, shouldn't happen
        return 1;
    }

    for ( std::vector< std::string >::const_iterator it1 = components.begin();
          it1 != components.end();
          it1++ ) {
        if ( ! it1->empty() ) {
            m_current_components.push_back( *it1 );
        }
    }

    return 0;
}

int PathJumpUI::cleanup()
{
    std::string msg, bt;

    std::list< std::pair< std::string, time_t > > temp_paths = m_worker.getPathsPers(),
        non_existing_entries;

    int non_existing = 0;
    size_t elements = temp_paths.size();
    size_t processed = 0;
    time_t last_update = 0;

    std::string orig_win_title = m_win->getTitle();

    m_win->setCursor( AGUIX::WAIT_CURSOR );
    m_aguix.Flush();

    for ( auto it1 = temp_paths.begin();
          it1 != temp_paths.end();
          it1++ ) {
        if ( ! NWC::FSEntry( it1->first ).entryExists() ) {
            non_existing_entries.push_back( *it1 );
            non_existing++;
        }

        processed++;

        time_t now = time( NULL );

        if ( now != last_update ) {
            last_update = now;

            std::string title = AGUIXUtils::formatStringToString( catalog.getLocale( 1113 ),
                                                                  ((double)processed) * 100.0 / elements );

            m_win->setTitle( title );
            m_aguix.Flush();
        }
    }

    m_win->setTitle( orig_win_title );
    m_win->unsetCursor();

    if ( non_existing > 0 ) {
        msg = AGUIXUtils::formatStringToString( catalog.getLocale( 985 ),
                                                non_existing, (int)elements );

        bt = catalog.getLocale( 11 );
        bt += "|";
        bt += catalog.getLocale( 8 );

        int res = m_win->request( catalog.getLocale( 123 ),
                                  msg.c_str(),
                                  bt.c_str() );

        if ( res == 0 ) {
            for ( auto it1 = non_existing_entries.begin();
                  it1 != non_existing_entries.end();
                  it1++ ) {
                m_worker.removePathPers( it1->first );
            }
        }
    } else {
        m_win->request( catalog.getLocale( 123 ),
                        catalog.getLocale( 986 ),
                        catalog.getLocale( 633 ) );
    }

    return 0;
}

int PathJumpUI::resetHeldEntries()
{
    for ( auto &e : m_entries ) {
        e.m_always_ignore = false;
    }

    return 0;
}

void PathJumpUI::resetHeldEntriesAndUpdateUI()
{
    resetHeldEntries();

    m_show_all_cb->setState( m_hold_status.base_show_hidden_state );
    m_show_all_cb->show();

    m_show_hidden_entries = m_hold_status.base_show_hidden_state;
    m_hold_status.active = false;
    m_hold_status.current_filter = "";

    m_hold_status_b->setText( 0, catalog.getLocale( 1027 ) );
    m_hold_filter_text->setText( "" );

    m_hold_co->readLimits();

    m_win->updateCont();

    apply_filter( m_filter, [this]( int matches ) {
            showData();
        });
}

int PathJumpUI::calculateMatchClasses()
{
    std::set< int > match_counts;
    std::list< int > sorted_counts;
    std::map< int, int > entries_per_match;

    for ( auto &e : m_entries ) {
        if ( ( e.m_type == PATH_SHOW ||
               e.m_type == BOOKMARK_SHOW ) &&
             e.m_post_filtering &&
             e.m_always_ignore == false ) {
            match_counts.insert( e.m_blockcount );
            entries_per_match[e.m_blockcount]++;
        }
    }

    for ( auto &c : match_counts ) {
        sorted_counts.push_back( c );
    }

    sorted_counts.sort();

    m_match_classes.clear();

    for ( auto &c : sorted_counts ) {
        m_match_classes.push_back( std::make_pair( c, entries_per_match[c] ) );
    }

    return 0;
}

int PathJumpUI::holdCurrentEntries( int top_x )
{
    std::list< std::pair< int, int > > sorted_counts;

    if ( top_x < 1 ) return 1;

    sorted_counts = m_match_classes;

    if ( top_x < (int)sorted_counts.size() ) {
        sorted_counts.resize( top_x );
    }

    for ( auto &e : m_entries ) {
        if ( ( e.m_type == PATH_SHOW ||
               e.m_type == BOOKMARK_SHOW ) &&
             e.m_post_filtering &&
             e.m_always_ignore == false ) {
            if ( std::find_if( sorted_counts.begin(),
                               sorted_counts.end(),
                               [&e]( std::pair< int, int > &mc ) {
                                   if ( mc.first == e.m_blockcount ) return true;
                                   return false;
                               } ) != sorted_counts.end() ) {
                // blockcount accepted, continue wiht next entry
                continue;
            }
        }

        e.m_always_ignore = true;
    }

    return 0;
}

int PathJumpUI::holdCurrentEntriesAndUpdateUI( int top_x )
{
    if ( ! m_hold_status.active ) {
        m_hold_status.base_show_hidden_state = m_show_hidden_entries;
        m_hold_status.active = true;

        m_show_all_cb->hide();

        m_hold_status_b->setText( 0, catalog.getLocale( 1028 ) );
        m_hold_status_b->resize( m_hold_status_b->getMaximumWidth(),
                                 m_hold_status_b->getHeight() );
    }

    if ( ! m_filter.empty() ) {
        if ( ! m_hold_status.current_filter.empty() ) {
            m_hold_status.current_filter += "+";
        }

        m_hold_status.current_filter += m_filter;
    }

    m_hold_filter_text->setText( m_hold_status.current_filter.c_str() );

    m_hold_co->readLimits();

    m_win->updateCont();

    int res = holdCurrentEntries( top_x );

    m_filter = "";

    apply_filter( m_filter, [this]( int matches ) {
            showData();
        });

    return res;
}

int PathJumpUI::handleMatchClassRow( int row )
{
    if ( row == 0 ) {
        m_lv->setActiveRow( 0 );

        handleMainLVRowChange( 0 );

        m_lv->centerActive();
        m_lv->redraw();

        return 0;
    }

    std::string s1 = m_class_counts_lv->getText( row, 0 );

    for ( int mainrow = 0; mainrow < m_lv->getElements(); mainrow++ ) {
        if ( m_lv->getText( mainrow, 0 ) == s1 ) {
            m_lv->setActiveRow( mainrow );

            handleMainLVRowChange( mainrow );

            m_lv->centerActive();
            m_lv->redraw();
            break;
        }
    }

    return 0;
}

int PathJumpUI::handleMainLVRowChange( int row )
{
    try {
        m_current_entry = m_entry_fullnames.at( row ).first;
        set_entry( m_current_entry );
        showData();
    } catch (...) {
    }

    return 0;
}

void PathJumpUI::showHoldHelp()
{
    RefCount<AFontWidth> lencalc( new AFontWidth( Worker::getAGUIX(), NULL ) );

    std::string message = catalog.getLocale( 1033 );

    TextStorageString ts( message, lencalc );

    std::string buttons = catalog.getLocale( 11 );

    Worker::getRequester()->request( catalog.getLocale( 124 ), ts, buttons.c_str() );
}

std::unique_ptr< NWC::Dir > PathJumpUI::getResultsAsDir()
{
    std::string name = AGUIXUtils::formatStringToString( "pathjump%d", s_pathjump_number++ );

    if ( s_pathjump_number < 0 ) {
        // avoid negative and zero number
        s_pathjump_number = 1;
    }

    std::unique_ptr< NWC::Dir > d( new NWC::VirtualDir( name ) );

    for ( auto &e : m_entry_fullnames ) {
        NWC::FSEntry fse( e.first );

        if ( fse.entryExists() ) {
            d->add( fse );
        }
    }

    return d;
}

void PathJumpUI::updateLVData( int row, int field,
                               struct FieldListView::on_demand_data &data )
{
    if ( row < 0 ||
         row >= (int)m_entry_fullnames.size() ) {
        return;
    }
    
    if ( field == 2 ) {
        std::string visibile_name;
        
        try {
            visibile_name = m_entry_fullnames.at( row ).first;
        } catch (...) {
        }

        if ( ! visibile_name.empty() ) {
            if ( m_entry_filter_mode == SHOW_ONLY_DIRECT_SUBDIRS ) {
                size_t p2 = visibile_name.find( '/', m_ondemand_info.prefix_filter_length + 1 );
                if ( p2 != std::string::npos ) {
                    std::string s( visibile_name, 0, p2 );

                    visibile_name = s;
                }
            }

            data.text = visibile_name;
            data.text_set = true;

            if ( ! m_filter.empty() ) {
                data.segments.clear();

                m_ondemand_info.matcher.countNonMatchingBlocks<true>( visibile_name, &data.segments );

                if ( data.segments.size() > 1 ) {
                    data.segments_set = true;
                }
            }

            data.done = true;
        }
    } else if ( field == 1 ) {
        time_t diff = 0;
        bool ignore = false;
        
        try {
            diff = m_entry_fullnames.at( row ).second;

            if ( diff == m_oldest_time ||
                 diff == 0 ) {
                ignore = true;
            } else {
                diff = m_current_time - diff;
            }
        } catch (...) {
        }

        std::string t;

        if ( ! ignore ) {
            if ( diff < 60 ) {
                t = AGUIXUtils::formatStringToString( catalog.getLocale( 1104 ), (int)( diff / 60 ) );
            } else if ( diff < 2 * 60 ) {
                t = AGUIXUtils::formatStringToString( catalog.getLocale( 1103 ), (int)( diff / 60 ) );
            } else if ( diff < 1 * 60 * 60 ) {
                t = AGUIXUtils::formatStringToString( catalog.getLocale( 1104 ), (int)( diff / 60 ) );
            } else if ( diff < 2 * 60 * 60 ) {
                t = AGUIXUtils::formatStringToString( catalog.getLocale( 1105 ), (int)( diff / 60 / 60 ) );
            } else if ( diff < 24 * 60 * 60 ) {
                t = AGUIXUtils::formatStringToString( catalog.getLocale( 1106 ), (int)( diff / 60 / 60 ) );
            } else if ( diff < 2 * 24 * 60 * 60 ) {
                t = AGUIXUtils::formatStringToString( catalog.getLocale( 1107 ), (int)( diff / 60 / 60 / 24 ) );
            } else if ( diff < 7 * 24 * 60 * 60 ) {
                t = AGUIXUtils::formatStringToString( catalog.getLocale( 1108 ), (int)( diff / 60 / 60 / 24 ) );
            } else if ( diff < 2 * 7 * 24 * 60 * 60 ) {
                t = AGUIXUtils::formatStringToString( catalog.getLocale( 1109 ), (int)( diff / 60 / 60 / 24 ) );
            } else {
                t = AGUIXUtils::formatStringToString( catalog.getLocale( 1110 ), (int)( diff / 60 / 60 / 24 / 7 ) );
            }
        }

        data.text = t;
        data.text_set = true;
    }
}

void PathJumpUI::checkExistance()
{
    std::list< std::string > non_existing_entries;
    size_t elements = m_entries.size();
    size_t processed = 0;
    time_t last_update = 0;
    std::string orig_win_title = m_win->getTitle();

    m_win->setCursor( AGUIX::WAIT_CURSOR );
    m_aguix.Flush();

    // cancel running filter before modifying the list
    m_apply_state = apply_filter_state();

    for ( auto it1 = m_entries.begin();
          it1 != m_entries.end(); ) {
        auto next_it = it1;
        next_it++;
        
        if ( ( it1->m_type == PATH_SHOW ||
               it1->m_type == BOOKMARK_SHOW ) &&
             it1->m_post_filtering &&
             it1->m_always_ignore == false ) {
            if ( ! NWC::FSEntry( it1->m_path ).entryExists() ) {
                non_existing_entries.push_back( it1->m_path );
                m_entries.erase( it1 );
            }
        }

        it1 = next_it;
        processed++;

        time_t now = time( NULL );

        if ( now != last_update ) {
            last_update = now;

            std::string title = AGUIXUtils::formatStringToString( catalog.getLocale( 1113 ),
                                                                  ((double)processed) * 100.0 / elements );

            m_win->setTitle( title );
            m_aguix.Flush();
        }
    }

    m_win->setTitle( orig_win_title );

    if ( ! non_existing_entries.empty() ) {
        std::string msg = AGUIXUtils::formatStringToString( catalog.getLocale( 1114 ),
                                                            (int)non_existing_entries.size() );

        std::string bt = catalog.getLocale( 11 );
        bt += "|";
        bt += catalog.getLocale( 225 );

        int res = m_win->request( catalog.getLocale( 123 ),
                                  msg.c_str(),
                                  bt.c_str() );

        if ( res == 0 ) {
            for ( auto it1 = non_existing_entries.begin();
                  it1 != non_existing_entries.end();
                  it1++ ) {
                m_worker.removePathPers( *it1 );
            }
        }
    }

    apply_filter( m_filter, [this]( int matches ) {
            showData();
        });

    m_win->unsetCursor();
}

void PathJumpUI::updateAgeText()
{
    if ( m_entry_filter_age == 0 ) {
        // nothing
        m_age_text->setText( "---" );
    } else if ( m_entry_filter_age <= 1 * 60 * 60 ) {
        // 1 hour
        std::string t1 = AGUIXUtils::formatStringToString( catalog.getLocale( 1105 ),
                                                           1 );
        m_age_text->setText( t1.c_str() );
    } else if ( m_entry_filter_age <= 23 * 60 * 60 ) {
        // x hours
        std::string t1 = AGUIXUtils::formatStringToString( catalog.getLocale( 1106 ),
                                                           (int)( ( m_entry_filter_age + 60 * 60 - 1 ) / ( 60 * 60 ) ) );
        m_age_text->setText( t1.c_str() );
    } else if ( m_entry_filter_age <= 1 * 24 * 60 * 60 ) {
        // 1 day
        std::string t1 = AGUIXUtils::formatStringToString( catalog.getLocale( 1107 ),
                                                           1 );
        m_age_text->setText( t1.c_str() );
    } else if ( m_entry_filter_age <= 6 * 24 * 60 * 60 ) {
        // x days
        std::string t1 = AGUIXUtils::formatStringToString( catalog.getLocale( 1108 ),
                                                           (int)( ( m_entry_filter_age + 24 * 60 * 60 - 1 ) / ( 24 * 60 * 60 ) ) );
        m_age_text->setText( t1.c_str() );
    } else if ( m_entry_filter_age <= 7 * 24 * 60 * 60 ) {
        // one week
        std::string t1 = AGUIXUtils::formatStringToString( catalog.getLocale( 1109 ),
                                                           1 );
        m_age_text->setText( t1.c_str() );
    } else {
        // x weeks
        std::string t1 = AGUIXUtils::formatStringToString( catalog.getLocale( 1110 ),
                                                           (int)( ( m_entry_filter_age + 7 * 24 * 60 * 60 - 1 ) / ( 7 * 24 * 60 * 60 ) ) );
        m_age_text->setText( t1.c_str() );
    }
}
