/*
 * crypt32 CRL functions tests
 *
 * Copyright 2005-2006 Juan Lang
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <stdio.h>
#include <stdarg.h>

#include <windef.h>
#include <winbase.h>
#include <winreg.h>
#include <winerror.h>
#include <wincrypt.h>

#include "wine/test.h"


static const BYTE bigCert[] = { 0x30, 0x7a, 0x02, 0x01, 0x01, 0x30, 0x02, 0x06,
 0x00, 0x30, 0x15, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13,
 0x0a, 0x4a, 0x75, 0x61, 0x6e, 0x20, 0x4c, 0x61, 0x6e, 0x67, 0x00, 0x30, 0x22,
 0x18, 0x0f, 0x31, 0x36, 0x30, 0x31, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30,
 0x30, 0x30, 0x30, 0x5a, 0x18, 0x0f, 0x31, 0x36, 0x30, 0x31, 0x30, 0x31, 0x30,
 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x30, 0x15, 0x31, 0x13, 0x30,
 0x11, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0a, 0x4a, 0x75, 0x61, 0x6e, 0x20,
 0x4c, 0x61, 0x6e, 0x67, 0x00, 0x30, 0x07, 0x30, 0x02, 0x06, 0x00, 0x03, 0x01,
 0x00, 0xa3, 0x16, 0x30, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01,
 0x01, 0xff, 0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x01 };
static const BYTE bigCert2[] = { 0x30, 0x7a, 0x02, 0x01, 0x01, 0x30, 0x02, 0x06,
 0x00, 0x30, 0x15, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13,
 0x0a, 0x41, 0x6c, 0x65, 0x78, 0x20, 0x4c, 0x61, 0x6e, 0x67, 0x00, 0x30, 0x22,
 0x18, 0x0f, 0x31, 0x36, 0x30, 0x31, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30,
 0x30, 0x30, 0x30, 0x5a, 0x18, 0x0f, 0x31, 0x36, 0x30, 0x31, 0x30, 0x31, 0x30,
 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x30, 0x15, 0x31, 0x13, 0x30,
 0x11, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0a, 0x41, 0x6c, 0x65, 0x78, 0x20,
 0x4c, 0x61, 0x6e, 0x67, 0x00, 0x30, 0x07, 0x30, 0x02, 0x06, 0x00, 0x03, 0x01,
 0x00, 0xa3, 0x16, 0x30, 0x14, 0x30, 0x12, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x01,
 0x01, 0xff, 0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff, 0x02, 0x01, 0x01 };
static const BYTE bigCertWithDifferentIssuer[] = { 0x30, 0x7a, 0x02, 0x01,
 0x01, 0x30, 0x02, 0x06, 0x00, 0x30, 0x15, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03,
 0x55, 0x04, 0x03, 0x13, 0x0a, 0x41, 0x6c, 0x65, 0x78, 0x20, 0x4c, 0x61, 0x6e,
 0x67, 0x00, 0x30, 0x22, 0x18, 0x0f, 0x31, 0x36, 0x30, 0x31, 0x30, 0x31, 0x30,
 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x18, 0x0f, 0x31, 0x36, 0x30,
 0x31, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x30,
 0x15, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0a, 0x4a,
 0x75, 0x61, 0x6e, 0x20, 0x4c, 0x61, 0x6e, 0x67, 0x00, 0x30, 0x07, 0x30, 0x02,
 0x06, 0x00, 0x03, 0x01, 0x00, 0xa3, 0x16, 0x30, 0x14, 0x30, 0x12, 0x06, 0x03,
 0x55, 0x1d, 0x13, 0x01, 0x01, 0xff, 0x04, 0x08, 0x30, 0x06, 0x01, 0x01, 0xff,
 0x02, 0x01, 0x01 };
static const BYTE CRL[] = { 0x30, 0x2c, 0x30, 0x02, 0x06, 0x00,
 0x30, 0x15, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0a,
 0x4a, 0x75, 0x61, 0x6e, 0x20, 0x4c, 0x61, 0x6e, 0x67, 0x00, 0x18, 0x0f, 0x31,
 0x36, 0x30, 0x31, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
 0x5a };
static const BYTE newerCRL[] = { 0x30, 0x2a, 0x30, 0x02, 0x06, 0x00, 0x30,
 0x15, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0a, 0x4a,
 0x75, 0x61, 0x6e, 0x20, 0x4c, 0x61, 0x6e, 0x67, 0x00, 0x17, 0x0d, 0x30, 0x36,
 0x30, 0x35, 0x31, 0x36, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a };
static const BYTE signedCRL[] = { 0x30, 0x45, 0x30, 0x2c, 0x30, 0x02, 0x06,
 0x00, 0x30, 0x15, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13,
 0x0a, 0x4a, 0x75, 0x61, 0x6e, 0x20, 0x4c, 0x61, 0x6e, 0x67, 0x00, 0x18, 0x0f,
 0x31, 0x36, 0x30, 0x31, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30,
 0x30, 0x5a, 0x30, 0x02, 0x06, 0x00, 0x03, 0x11, 0x00, 0x0f, 0x0e, 0x0d, 0x0c,
 0x0b, 0x0a, 0x09, 0x08, 0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00 };

static BOOL (WINAPI *pCertFindCertificateInCRL)(PCCERT_CONTEXT,PCCRL_CONTEXT,DWORD,void*,PCRL_ENTRY*);
static PCCRL_CONTEXT (WINAPI *pCertFindCRLInStore)(HCERTSTORE,DWORD,DWORD,DWORD,const void*,PCCRL_CONTEXT);
static BOOL (WINAPI *pCertIsValidCRLForCertificate)(PCCERT_CONTEXT, PCCRL_CONTEXT, DWORD, void*);

static void init_function_pointers(void)
{
    HMODULE hdll = GetModuleHandleA("crypt32.dll");
    pCertFindCertificateInCRL = (void*)GetProcAddress(hdll, "CertFindCertificateInCRL");
    pCertFindCRLInStore = (void*)GetProcAddress(hdll, "CertFindCRLInStore");
    pCertIsValidCRLForCertificate = (void*)GetProcAddress(hdll, "CertIsValidCRLForCertificate");
}

static void testCreateCRL(void)
{
    PCCRL_CONTEXT context;
    DWORD GLE;

    context = CertCreateCRLContext(0, NULL, 0);
    ok(!context && GetLastError() == E_INVALIDARG,
     "Expected E_INVALIDARG, got %08x\n", GetLastError());
    context = CertCreateCRLContext(X509_ASN_ENCODING, NULL, 0);
    GLE = GetLastError();
    ok(!context && (GLE == CRYPT_E_ASN1_EOD || GLE == OSS_MORE_INPUT),
     "Expected CRYPT_E_ASN1_EOD or OSS_MORE_INPUT, got %08x\n", GLE);
    context = CertCreateCRLContext(X509_ASN_ENCODING, bigCert, sizeof(bigCert));
    ok(!context, "Expected failure\n");
    context = CertCreateCRLContext(X509_ASN_ENCODING, signedCRL,
     sizeof(signedCRL) - 1);
    ok(!context, "Expected failure\n");
    context = CertCreateCRLContext(X509_ASN_ENCODING, signedCRL,
     sizeof(signedCRL));
    ok(context != NULL, "CertCreateCRLContext failed: %08x\n", GetLastError());
    if (context)
        CertFreeCRLContext(context);
    context = CertCreateCRLContext(X509_ASN_ENCODING, CRL, sizeof(CRL));
    ok(context != NULL, "CertCreateCRLContext failed: %08x\n", GetLastError());
    if (context)
        CertFreeCRLContext(context);
}

static void testDupCRL(void)
{
    PCCRL_CONTEXT context, dupContext;

    context = CertDuplicateCRLContext(NULL);
    ok(context == NULL, "expected NULL\n");
    context = CertCreateCRLContext(X509_ASN_ENCODING, signedCRL,
     sizeof(signedCRL));
    dupContext = CertDuplicateCRLContext(context);
    ok(dupContext != NULL, "expected a context\n");
    ok(dupContext == context, "expected identical context addresses\n");
    CertFreeCRLContext(dupContext);
    CertFreeCRLContext(context);
}

static void testAddCRL(void)
{
    HCERTSTORE store = CertOpenStore(CERT_STORE_PROV_MEMORY, 0, 0,
     CERT_STORE_CREATE_NEW_FLAG, NULL);
    PCCRL_CONTEXT context;
    BOOL ret;
    DWORD GLE;

    if (!store) return;

    /* Bad CRL encoding type */
    ret = CertAddEncodedCRLToStore(0, 0, NULL, 0, 0, NULL);
    ok(!ret && GetLastError() == E_INVALIDARG,
     "Expected E_INVALIDARG, got %08x\n", GetLastError());
    ret = CertAddEncodedCRLToStore(store, 0, NULL, 0, 0, NULL);
    ok(!ret && GetLastError() == E_INVALIDARG,
     "Expected E_INVALIDARG, got %08x\n", GetLastError());
    ret = CertAddEncodedCRLToStore(0, 0, signedCRL, sizeof(signedCRL), 0, NULL);
    ok(!ret && GetLastError() == E_INVALIDARG,
     "Expected E_INVALIDARG, got %08x\n", GetLastError());
    ret = CertAddEncodedCRLToStore(store, 0, signedCRL, sizeof(signedCRL), 0,
     NULL);
    ok(!ret && GetLastError() == E_INVALIDARG,
     "Expected E_INVALIDARG, got %08x\n", GetLastError());
    ret = CertAddEncodedCRLToStore(0, 0, signedCRL, sizeof(signedCRL),
     CERT_STORE_ADD_ALWAYS, NULL);
    ok(!ret && GetLastError() == E_INVALIDARG,
     "Expected E_INVALIDARG, got %08x\n", GetLastError());
    ret = CertAddEncodedCRLToStore(store, 0, signedCRL, sizeof(signedCRL),
     CERT_STORE_ADD_ALWAYS, NULL);
    ok(!ret && GetLastError() == E_INVALIDARG,
     "Expected E_INVALIDARG, got %08x\n", GetLastError());

    /* No CRL */
    ret = CertAddEncodedCRLToStore(0, X509_ASN_ENCODING, NULL, 0, 0, NULL);
    GLE = GetLastError();
    ok(!ret && (GLE == CRYPT_E_ASN1_EOD || GLE == OSS_MORE_INPUT),
     "Expected CRYPT_E_ASN1_EOD or OSS_MORE_INPUT, got %08x\n", GLE);
    ret = CertAddEncodedCRLToStore(store, X509_ASN_ENCODING, NULL, 0, 0, NULL);
    GLE = GetLastError();
    ok(!ret && (GLE == CRYPT_E_ASN1_EOD || GLE == OSS_MORE_INPUT),
     "Expected CRYPT_E_ASN1_EOD or OSS_MORE_INPUT, got %08x\n", GLE);

    /* Weird--bad add disposition leads to an access violation in Windows.
     * Both tests crash on some win9x boxes.
     */
    if (0)
    {
        ret = CertAddEncodedCRLToStore(0, X509_ASN_ENCODING, signedCRL,
         sizeof(signedCRL), 0, NULL);
        ok(!ret && (GetLastError() == STATUS_ACCESS_VIOLATION ||
                    GetLastError() == E_INVALIDARG /* Vista */),
         "Expected STATUS_ACCESS_VIOLATION or E_INVALIDARG, got %08x\n", GetLastError());
        ret = CertAddEncodedCRLToStore(store, X509_ASN_ENCODING, signedCRL,
         sizeof(signedCRL), 0, NULL);
        ok(!ret && (GetLastError() == STATUS_ACCESS_VIOLATION ||
                    GetLastError() == E_INVALIDARG /* Vista */),
         "Expected STATUS_ACCESS_VIOLATION or E_INVALIDARG, got %08x\n", GetLastError());
    }

    /* Weird--can add a CRL to the NULL store (does this have special meaning?)
     */
    context = NULL;
    ret = CertAddEncodedCRLToStore(0, X509_ASN_ENCODING, signedCRL,
     sizeof(signedCRL), CERT_STORE_ADD_ALWAYS, &context);
    ok(ret, "CertAddEncodedCRLToStore failed: %08x\n", GetLastError());
    if (context)
        CertFreeCRLContext(context);

    /* Normal cases: a "signed" CRL is okay.. */
    ret = CertAddEncodedCRLToStore(store, X509_ASN_ENCODING, signedCRL,
     sizeof(signedCRL), CERT_STORE_ADD_ALWAYS, NULL);
    ok(ret, "CertAddEncodedCRLToStore failed: %08x\n", GetLastError());
    /* and an unsigned one is too. */
    ret = CertAddEncodedCRLToStore(store, X509_ASN_ENCODING, CRL, sizeof(CRL),
     CERT_STORE_ADD_ALWAYS, NULL);
    ok(ret, "CertAddEncodedCRLToStore failed: %08x\n", GetLastError());

    ret = CertAddEncodedCRLToStore(store, X509_ASN_ENCODING, newerCRL,
     sizeof(newerCRL), CERT_STORE_ADD_NEW, NULL);
    ok(!ret && GetLastError() == CRYPT_E_EXISTS,
     "Expected CRYPT_E_EXISTS, got %08x\n", GetLastError());

    /* This should replace (one of) the existing CRL(s). */
    ret = CertAddEncodedCRLToStore(store, X509_ASN_ENCODING, newerCRL,
     sizeof(newerCRL), CERT_STORE_ADD_NEWER, NULL);
    ok(ret, "CertAddEncodedCRLToStore failed: %08x\n", GetLastError());

    CertCloseStore(store, 0);
}

static const BYTE v1CRLWithIssuerAndEntry[] = { 0x30, 0x44, 0x30, 0x02, 0x06,
 0x00, 0x30, 0x15, 0x31, 0x13, 0x30, 0x11, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13,
 0x0a, 0x4a, 0x75, 0x61, 0x6e, 0x20, 0x4c, 0x61, 0x6e, 0x67, 0x00, 0x18, 0x0f,
 0x31, 0x36, 0x30, 0x31, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30,
 0x30, 0x5a, 0x30, 0x16, 0x30, 0x14, 0x02, 0x01, 0x01, 0x18, 0x0f, 0x31, 0x36,
 0x30, 0x31, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a };
static const BYTE v2CRLWithIssuingDistPoint[] = {
0x30,0x70,0x02,0x01,0x01,0x30,0x02,0x06,0x00,0x30,0x15,0x31,0x13,0x30,0x11,
0x06,0x03,0x55,0x04,0x03,0x13,0x0a,0x4a,0x75,0x61,0x6e,0x20,0x4c,0x61,0x6e,
0x67,0x00,0x18,0x0f,0x31,0x36,0x30,0x31,0x30,0x31,0x30,0x31,0x30,0x30,0x30,
0x30,0x30,0x30,0x5a,0x30,0x16,0x30,0x14,0x02,0x01,0x01,0x18,0x0f,0x31,0x36,
0x30,0x31,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0xa0,0x27,
0x30,0x25,0x30,0x23,0x06,0x03,0x55,0x1d,0x1c,0x01,0x01,0xff,0x04,0x19,0x30,
0x17,0xa0,0x15,0xa0,0x13,0x86,0x11,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x77,
0x69,0x6e,0x65,0x68,0x71,0x2e,0x6f,0x72,0x67 };
static const BYTE verisignCRL[] = { 0x30, 0x82, 0x01, 0xb1, 0x30, 0x82, 0x01,
 0x1a, 0x02, 0x01, 0x01, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
 0x0d, 0x01, 0x01, 0x02, 0x05, 0x00, 0x30, 0x61, 0x31, 0x11, 0x30, 0x0f, 0x06,
 0x03, 0x55, 0x04, 0x07, 0x13, 0x08, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65,
 0x74, 0x31, 0x17, 0x30, 0x15, 0x06, 0x03, 0x55, 0x04, 0x0a, 0x13, 0x0e, 0x56,
 0x65, 0x72, 0x69, 0x53, 0x69, 0x67, 0x6e, 0x2c, 0x20, 0x49, 0x6e, 0x63, 0x2e,
 0x31, 0x33, 0x30, 0x31, 0x06, 0x03, 0x55, 0x04, 0x0b, 0x13, 0x2a, 0x56, 0x65,
 0x72, 0x69, 0x53, 0x69, 0x67, 0x6e, 0x20, 0x43, 0x6f, 0x6d, 0x6d, 0x65, 0x72,
 0x63, 0x69, 0x61, 0x6c, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65,
 0x20, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x73, 0x68, 0x65, 0x72, 0x73, 0x20, 0x43,
 0x41, 0x17, 0x0d, 0x30, 0x31, 0x30, 0x33, 0x32, 0x34, 0x30, 0x30, 0x30, 0x30,
 0x30, 0x30, 0x5a, 0x17, 0x0d, 0x30, 0x34, 0x30, 0x31, 0x30, 0x37, 0x32, 0x33,
 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x69, 0x30, 0x21, 0x02, 0x10, 0x1b, 0x51,
 0x90, 0xf7, 0x37, 0x24, 0x39, 0x9c, 0x92, 0x54, 0xcd, 0x42, 0x46, 0x37, 0x99,
 0x6a, 0x17, 0x0d, 0x30, 0x31, 0x30, 0x31, 0x33, 0x30, 0x30, 0x30, 0x30, 0x31,
 0x32, 0x34, 0x5a, 0x30, 0x21, 0x02, 0x10, 0x75, 0x0e, 0x40, 0xff, 0x97, 0xf0,
 0x47, 0xed, 0xf5, 0x56, 0xc7, 0x08, 0x4e, 0xb1, 0xab, 0xfd, 0x17, 0x0d, 0x30,
 0x31, 0x30, 0x31, 0x33, 0x31, 0x30, 0x30, 0x30, 0x30, 0x34, 0x39, 0x5a, 0x30,
 0x21, 0x02, 0x10, 0x77, 0xe6, 0x5a, 0x43, 0x59, 0x93, 0x5d, 0x5f, 0x7a, 0x75,
 0x80, 0x1a, 0xcd, 0xad, 0xc2, 0x22, 0x17, 0x0d, 0x30, 0x30, 0x30, 0x38, 0x33,
 0x31, 0x30, 0x30, 0x30, 0x30, 0x35, 0x36, 0x5a, 0xa0, 0x1a, 0x30, 0x18, 0x30,
 0x09, 0x06, 0x03, 0x55, 0x1d, 0x13, 0x04, 0x02, 0x30, 0x00, 0x30, 0x0b, 0x06,
 0x03, 0x55, 0x1d, 0x0f, 0x04, 0x04, 0x03, 0x02, 0x05, 0xa0, 0x30, 0x0d, 0x06,
 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x02, 0x05, 0x00, 0x03,
 0x81, 0x81, 0x00, 0x18, 0x2c, 0xe8, 0xfc, 0x16, 0x6d, 0x91, 0x4a, 0x3d, 0x88,
 0x54, 0x48, 0x5d, 0xb8, 0x11, 0xbf, 0x64, 0xbb, 0xf9, 0xda, 0x59, 0x19, 0xdd,
 0x0e, 0x65, 0xab, 0xc0, 0x0c, 0xfa, 0x67, 0x7e, 0x21, 0x1e, 0x83, 0x0e, 0xcf,
 0x9b, 0x89, 0x8a, 0xcf, 0x0c, 0x4b, 0xc1, 0x39, 0x9d, 0xe7, 0x6a, 0xac, 0x46,
 0x74, 0x6a, 0x91, 0x62, 0x22, 0x0d, 0xc4, 0x08, 0xbd, 0xf5, 0x0a, 0x90, 0x7f,
 0x06, 0x21, 0x3d, 0x7e, 0xa7, 0xaa, 0x5e, 0xcd, 0x22, 0x15, 0xe6, 0x0c, 0x75,
 0x8e, 0x6e, 0xad, 0xf1, 0x84, 0xe4, 0x22, 0xb4, 0x30, 0x6f, 0xfb, 0x64, 0x8f,
 0xd7, 0x80, 0x43, 0xf5, 0x19, 0x18, 0x66, 0x1d, 0x72, 0xa3, 0xe3, 0x94, 0x82,
 0x28, 0x52, 0xa0, 0x06, 0x4e, 0xb1, 0xc8, 0x92, 0x0c, 0x97, 0xbe, 0x15, 0x07,
 0xab, 0x7a, 0xc9, 0xea, 0x08, 0x67, 0x43, 0x4d, 0x51, 0x63, 0x3b, 0x9c, 0x9c,
 0xcd };
static const BYTE verisignCommercialSoftPubCA[] = {
0x30,0x82,0x02,0x40,0x30,0x82,0x01,0xa9,0x02,0x10,0x03,0xc7,0x8f,0x37,0xdb,0x92,
0x28,0xdf,0x3c,0xbb,0x1a,0xad,0x82,0xfa,0x67,0x10,0x30,0x0d,0x06,0x09,0x2a,0x86,
0x48,0x86,0xf7,0x0d,0x01,0x01,0x02,0x05,0x00,0x30,0x61,0x31,0x11,0x30,0x0f,0x06,
0x03,0x55,0x04,0x07,0x13,0x08,0x49,0x6e,0x74,0x65,0x72,0x6e,0x65,0x74,0x31,0x17,
0x30,0x15,0x06,0x03,0x55,0x04,0x0a,0x13,0x0e,0x56,0x65,0x72,0x69,0x53,0x69,0x67,
0x6e,0x2c,0x20,0x49,0x6e,0x63,0x2e,0x31,0x33,0x30,0x31,0x06,0x03,0x55,0x04,0x0b,
0x13,0x2a,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6e,0x20,0x43,0x6f,0x6d,0x6d,0x65,
0x72,0x63,0x69,0x61,0x6c,0x20,0x53,0x6f,0x66,0x74,0x77,0x61,0x72,0x65,0x20,0x50,
0x75,0x62,0x6c,0x69,0x73,0x68,0x65,0x72,0x73,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,
0x39,0x36,0x30,0x34,0x30,0x39,0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,
0x34,0x30,0x31,0x30,0x37,0x32,0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x61,0x31,0x11,
0x30,0x0f,0x06,0x03,0x55,0x04,0x07,0x13,0x08,0x49,0x6e,0x74,0x65,0x72,0x6e,0x65,
0x74,0x31,0x17,0x30,0x15,0x06,0x03,0x55,0x04,0x0a,0x13,0x0e,0x56,0x65,0x72,0x69,
0x53,0x69,0x67,0x6e,0x2c,0x20,0x49,0x6e,0x63,0x2e,0x31,0x33,0x30,0x31,0x06,0x03,
0x55,0x04,0x0b,0x13,0x2a,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6e,0x20,0x43,0x6f,
0x6d,0x6d,0x65,0x72,0x63,0x69,0x61,0x6c,0x20,0x53,0x6f,0x66,0x74,0x77,0x61,0x72,
0x65,0x20,0x50,0x75,0x62,0x6c,0x69,0x73,0x68,0x65,0x72,0x73,0x20,0x43,0x41,0x30,
0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,
0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xc3,0xd3,0x69,0x65,
0x52,0x01,0x94,0x54,0xab,0x28,0xc6,0x62,0x18,0xb3,0x54,0x55,0xc5,0x44,0x87,0x45,
0x4a,0x3b,0xc2,0x7e,0xd8,0xd3,0xd7,0xc8,0x80,0x86,0x8d,0xd8,0x0c,0xf1,0x16,0x9c,
0xcc,0x6b,0xa9,0x29,0xb2,0x8f,0x76,0x73,0x92,0xc8,0xc5,0x62,0xa6,0x3c,0xed,0x1e,
0x05,0x75,0xf0,0x13,0x00,0x6c,0x14,0x4d,0xd4,0x98,0x90,0x07,0xbe,0x69,0x73,0x81,
0xb8,0x62,0x4e,0x31,0x1e,0xd1,0xfc,0xc9,0x0c,0xeb,0x7d,0x90,0xbf,0xae,0xb4,0x47,
0x51,0xec,0x6f,0xce,0x64,0x35,0x02,0xd6,0x7d,0x67,0x05,0x77,0xe2,0x8f,0xd9,0x51,
0xd7,0xfb,0x97,0x19,0xbc,0x3e,0xd7,0x77,0x81,0xc6,0x43,0xdd,0xf2,0xdd,0xdf,0xca,
0xa3,0x83,0x8b,0xcb,0x41,0xc1,0x3d,0x22,0x48,0x48,0xa6,0x19,0x02,0x03,0x01,0x00,
0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x02,0x05,0x00,
0x03,0x81,0x81,0x00,0xb5,0xbc,0xb0,0x75,0x6a,0x89,0xa2,0x86,0xbd,0x64,0x78,0xc3,
0xa7,0x32,0x75,0x72,0x11,0xaa,0x26,0x02,0x17,0x60,0x30,0x4c,0xe3,0x48,0x34,0x19,
0xb9,0x52,0x4a,0x51,0x18,0x80,0xfe,0x53,0x2d,0x7b,0xd5,0x31,0x8c,0xc5,0x65,0x99,
0x41,0x41,0x2f,0xf2,0xae,0x63,0x7a,0xe8,0x73,0x99,0x15,0x90,0x1a,0x1f,0x7a,0x8b,
0x41,0xd0,0x8e,0x3a,0xd0,0xcd,0x38,0x34,0x44,0xd0,0x75,0xf8,0xea,0x71,0xc4,0x81,
0x19,0x38,0x17,0x35,0x4a,0xae,0xc5,0x3e,0x32,0xe6,0x21,0xb8,0x05,0xc0,0x93,0xe1,
0xc7,0x38,0x5c,0xd8,0xf7,0x93,0x38,0x64,0x90,0xed,0x54,0xce,0xca,0xd3,0xd3,0xd0,
0x5f,0xef,0x04,0x9b,0xde,0x02,0x82,0xdd,0x88,0x29,0xb1,0xc3,0x4f,0xa5,0xcd,0x71,
0x64,0x31,0x3c,0x3c
};
static const BYTE rootWithKeySignAndCRLSign[] = {
0x30,0x82,0x01,0xdf,0x30,0x82,0x01,0x4c,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,
0x5b,0xc7,0x0b,0x27,0x99,0xbb,0x2e,0x99,0x47,0x9d,0x45,0x4e,0x7c,0x1a,0xca,
0xe8,0x30,0x09,0x06,0x05,0x2b,0x0e,0x03,0x02,0x1d,0x05,0x00,0x30,0x10,0x31,
0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,
0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,
0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,
0x39,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,
0x43,0x65,0x72,0x74,0x31,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,
0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,
0x02,0x81,0x81,0x00,0xad,0x7e,0xca,0xf3,0xe5,0x99,0xc2,0x2a,0xca,0x50,0x82,
0x7c,0x2d,0xa4,0x81,0xcd,0x0d,0x0d,0x86,0xd7,0xd8,0xb2,0xde,0xc5,0xc3,0x34,
0x9e,0x07,0x78,0x08,0x11,0x12,0x2d,0x21,0x0a,0x09,0x07,0x14,0x03,0x7a,0xe7,
0x3b,0x58,0xf1,0xde,0x3e,0x01,0x25,0x93,0xab,0x8f,0xce,0x1f,0xc1,0x33,0x91,
0xfe,0x59,0xb9,0x3b,0x9e,0x95,0x12,0x89,0x8e,0xc3,0x4b,0x98,0x1b,0x99,0xc5,
0x07,0xe2,0xdf,0x15,0x4c,0x39,0x76,0x06,0xad,0xdb,0x16,0x06,0x49,0xba,0xcd,
0x0f,0x07,0xd6,0xea,0x27,0xa6,0xfe,0x3d,0x88,0xe5,0x97,0x45,0x72,0xb6,0x1c,
0xc0,0x1c,0xb1,0xa2,0x89,0xe8,0x37,0x9e,0xf6,0x2a,0xcf,0xd5,0x1f,0x2f,0x35,
0x5e,0x8f,0x3a,0x9c,0x61,0xb1,0xf1,0x6c,0xff,0x8c,0xb2,0x2f,0x02,0x03,0x01,
0x00,0x01,0xa3,0x42,0x30,0x40,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,
0xff,0x04,0x04,0x03,0x02,0x00,0x06,0x30,0x0f,0x06,0x03,0x55,0x1d,0x13,0x01,
0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x1d,0x06,0x03,0x55,0x1d,
0x0e,0x04,0x16,0x04,0x14,0x14,0x8c,0x16,0xbb,0xbe,0x70,0xa2,0x28,0x89,0xa0,
0x58,0xff,0x98,0xbd,0xa8,0x24,0x2b,0x8a,0xe9,0x9a,0x30,0x09,0x06,0x05,0x2b,
0x0e,0x03,0x02,0x1d,0x05,0x00,0x03,0x81,0x81,0x00,0x74,0xcb,0x21,0xfd,0x2d,
0x25,0xdc,0xa5,0xaa,0xa1,0x26,0xdc,0x8b,0x40,0x11,0x64,0xae,0x5c,0x71,0x3c,
0x28,0xbc,0xf9,0xb3,0xcb,0xa5,0x94,0xb2,0x8d,0x4c,0x23,0x2b,0x9b,0xde,0x2c,
0x4c,0x30,0x04,0xc6,0x88,0x10,0x2f,0x53,0xfd,0x6c,0x82,0xf1,0x13,0xfb,0xda,
0x27,0x75,0x25,0x48,0xe4,0x72,0x09,0x2a,0xee,0xb4,0x1e,0xc9,0x55,0xf5,0xf7,
0x82,0x91,0xd8,0x4b,0xe4,0x3a,0xfe,0x97,0x87,0xdf,0xfb,0x15,0x5a,0x12,0x3e,
0x12,0xe6,0xad,0x40,0x0b,0xcf,0xee,0x1a,0x44,0xe0,0x83,0xb2,0x67,0x94,0xd4,
0x2e,0x7c,0xf2,0x06,0x9d,0xb3,0x3b,0x7e,0x2f,0xda,0x25,0x66,0x7e,0xa7,0x1f,
0x45,0xd4,0xf5,0xe3,0xdf,0x2a,0xf1,0x18,0x28,0x20,0xb5,0xf8,0xf5,0x8d,0x7a,
0x2e,0x84,0xee };
static const BYTE eeCert[] = {
0x30,0x82,0x01,0x93,0x30,0x81,0xfd,0xa0,0x03,0x02,0x01,0x02,0x02,0x01,0x01,
0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,
0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,
0x72,0x74,0x31,0x30,0x1e,0x17,0x0d,0x30,0x37,0x30,0x35,0x30,0x31,0x30,0x30,
0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x30,0x37,0x31,0x30,0x30,0x31,0x30,0x30,
0x30,0x30,0x30,0x30,0x5a,0x30,0x10,0x31,0x0e,0x30,0x0c,0x06,0x03,0x55,0x04,
0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x32,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,
0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,
0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb8,0x52,0xda,0xc5,0x4b,0x3f,0xe5,0x33,
0x0e,0x67,0x5f,0x48,0x21,0xdc,0x7e,0xef,0x37,0x33,0xba,0xff,0xb4,0xc6,0xdc,
0xb6,0x17,0x8e,0x20,0x55,0x07,0x12,0xd2,0x7b,0x3c,0xce,0x30,0xc5,0xa7,0x48,
0x9f,0x6e,0xfe,0xb8,0xbe,0xdb,0x9f,0x9b,0x17,0x60,0x16,0xde,0xc6,0x8b,0x47,
0xd1,0x57,0x71,0x3c,0x93,0xfc,0xbd,0xec,0x44,0x32,0x3b,0xb9,0xcf,0x6b,0x05,
0x72,0xa7,0x87,0x8e,0x7e,0xd4,0x9a,0x87,0x1c,0x2f,0xb7,0x82,0x40,0xfc,0x6a,
0x80,0x83,0x68,0x28,0xce,0x84,0xf4,0x0b,0x2e,0x44,0xcb,0x53,0xac,0x85,0x85,
0xb5,0x46,0x36,0x98,0x3c,0x10,0x02,0xaa,0x02,0xbc,0x8b,0xa2,0x23,0xb2,0xd3,
0x51,0x9a,0x22,0x4a,0xe3,0xaa,0x4e,0x7c,0xda,0x38,0xcf,0x49,0x98,0x72,0xa3,
0x02,0x03,0x01,0x00,0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x22,0xf1,0x66,0x00,0x79,0xd2,
0xe6,0xb2,0xb2,0xf7,0x2f,0x98,0x92,0x7d,0x73,0xc3,0x6c,0x5c,0x77,0x20,0xe3,
0xbf,0x3e,0xe0,0xb3,0x5c,0x68,0xb4,0x9b,0x3a,0x41,0xae,0x94,0xa0,0x80,0x3a,
0xfe,0x5d,0x7a,0x56,0x87,0x85,0x44,0x45,0xcf,0xa6,0xd3,0x10,0xe7,0x73,0x41,
0xf2,0x7f,0x88,0x85,0x91,0x8e,0xe6,0xec,0xe2,0xce,0x08,0xbc,0xa5,0x76,0xe5,
0x4d,0x1d,0xb7,0x70,0x31,0xdd,0xc9,0x9a,0x15,0x32,0x11,0x5a,0x4e,0x62,0xc8,
0xd1,0xf8,0xec,0x46,0x39,0x5b,0xe7,0x67,0x1f,0x58,0xe8,0xa1,0xa0,0x5b,0xf7,
0x8a,0x6d,0x5f,0x91,0x18,0xd4,0x90,0x85,0xff,0x30,0xc7,0xca,0x9c,0xc6,0x92,
0xb0,0xca,0x16,0xc4,0xa4,0xc0,0xd6,0xe8,0xff,0x15,0x19,0xd1,0x30,0x61,0xf3,
0xef,0x9f };
static const BYTE rootSignedCRL[] = {
0x30,0x82,0x01,0x1d,0x30,0x81,0x87,0x02,0x01,0x01,0x30,0x0d,0x06,0x09,0x2a,
0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x30,0x10,0x31,0x0e,0x30,
0x0c,0x06,0x03,0x55,0x04,0x03,0x13,0x05,0x43,0x65,0x72,0x74,0x31,0x17,0x0d,
0x30,0x37,0x30,0x39,0x30,0x31,0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,
0x30,0x37,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x35,0x39,0x5a,0x30,0x14,
0x30,0x12,0x02,0x01,0x01,0x17,0x0d,0x30,0x37,0x30,0x39,0x30,0x31,0x30,0x30,
0x30,0x30,0x30,0x30,0x5a,0xa0,0x2d,0x30,0x2b,0x30,0x0a,0x06,0x03,0x55,0x1d,
0x14,0x04,0x03,0x02,0x01,0x01,0x30,0x1d,0x06,0x03,0x55,0x1d,0x23,0x04,0x16,
0x04,0x14,0x14,0x8c,0x16,0xbb,0xbe,0x70,0xa2,0x28,0x89,0xa0,0x58,0xff,0x98,
0xbd,0xa8,0x24,0x2b,0x8a,0xe9,0x9a,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,
0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x9b,0x2b,0x99,0x0d,
0x16,0x83,0x93,0x54,0x29,0x3a,0xa6,0x53,0x5d,0xf8,0xa6,0x73,0x9f,0x2a,0x45,
0x39,0x91,0xff,0x91,0x1c,0x27,0x06,0xe8,0xdb,0x72,0x3f,0x66,0x89,0x15,0x68,
0x55,0xd5,0x49,0x63,0xa6,0x00,0xe9,0x66,0x9c,0x97,0xf9,0xb3,0xb3,0x2b,0x1b,
0xc7,0x79,0x46,0xa8,0xd8,0x2b,0x78,0x27,0xa0,0x70,0x02,0x81,0xc6,0x40,0xb3,
0x76,0x32,0x65,0x4c,0xf8,0xff,0x1d,0x41,0x6e,0x16,0x09,0xa2,0x8a,0x7b,0x0c,
0xd0,0xa6,0x9b,0x61,0xa3,0x7c,0x02,0x91,0x79,0xdf,0x6a,0x5e,0x88,0x95,0x66,
0x33,0x17,0xcb,0x5a,0xd2,0xdc,0x89,0x05,0x62,0x97,0x60,0x73,0x7b,0x2c,0x1a,
0x90,0x20,0x73,0x24,0x9f,0x45,0x22,0x4b,0xc1,0x33,0xd1,0xda,0xd8,0x7e,0x1b,
0x3d,0x74,0xd6,0x3b };

static void testFindCRL(void)
{
    HCERTSTORE store = CertOpenStore(CERT_STORE_PROV_MEMORY, 0, 0,
     CERT_STORE_CREATE_NEW_FLAG, NULL);
    PCCRL_CONTEXT context;
    PCCERT_CONTEXT cert, endCert, rootCert;
    CRL_FIND_ISSUED_FOR_PARA issuedForPara = { NULL, NULL };
    DWORD count, revoked_count;
    BOOL ret;

    if (!store) return;
    if (!pCertFindCRLInStore || !pCertFindCertificateInCRL)
    {
        win_skip("CertFindCRLInStore or CertFindCertificateInCRL not available\n");
        return;
    }

    ret = CertAddEncodedCRLToStore(store, X509_ASN_ENCODING, signedCRL,
     sizeof(signedCRL), CERT_STORE_ADD_ALWAYS, NULL);
    ok(ret, "CertAddEncodedCRLToStore failed: %08x\n", GetLastError());

    /* Crashes
    context = pCertFindCRLInStore(NULL, 0, 0, 0, NULL, NULL);
     */

    /* Find any context */
    context = pCertFindCRLInStore(store, 0, 0, CRL_FIND_ANY, NULL, NULL);
    ok(context != NULL, "Expected a context\n");
    if (context)
        CertFreeCRLContext(context);
    /* Bogus flags are ignored */
    context = pCertFindCRLInStore(store, 0, 1234, CRL_FIND_ANY, NULL, NULL);
    ok(context != NULL, "Expected a context\n");
    if (context)
        CertFreeCRLContext(context);
    /* CRL encoding type is ignored too */
    context = pCertFindCRLInStore(store, 1234, 0, CRL_FIND_ANY, NULL, NULL);
    ok(context != NULL, "Expected a context\n");
    if (context)
        CertFreeCRLContext(context);

    /* This appears to match any cert */
    context = pCertFindCRLInStore(store, 0, 0, CRL_FIND_ISSUED_BY, NULL, NULL);
    ok(context != NULL, "Expected a context\n");
    if (context)
        CertFreeCRLContext(context);

    /* Try to match an issuer that isn't in the store */
    cert = CertCreateCertificateContext(X509_ASN_ENCODING, bigCert2,
     sizeof(bigCert2));
    ok(cert != NULL, "CertCreateCertificateContext failed: %08x\n",
     GetLastError());
    context = pCertFindCRLInStore(store, 0, 0, CRL_FIND_ISSUED_BY, cert, NULL);
    ok(context == NULL, "Expected no matching context\n");
    CertFreeCertificateContext(cert);

    /* Match an issuer that is in the store */
    cert = CertCreateCertificateContext(X509_ASN_ENCODING, bigCert,
     sizeof(bigCert));
    ok(cert != NULL, "CertCreateCertificateContext failed: %08x\n",
     GetLastError());
    context = pCertFindCRLInStore(store, 0, 0, CRL_FIND_ISSUED_BY, cert, NULL);
    ok(context != NULL, "Expected a context\n");
    if (context)
        CertFreeCRLContext(context);

    /* Try various find flags */
    context = pCertFindCRLInStore(store, 0, CRL_FIND_ISSUED_BY_SIGNATURE_FLAG,
     CRL_FIND_ISSUED_BY, cert, NULL);
    ok(!context || broken(context != NULL /* Win9x */), "unexpected context\n");
    /* The CRL doesn't have an AKI extension, so it matches any cert */
    context = pCertFindCRLInStore(store, 0, CRL_FIND_ISSUED_BY_AKI_FLAG,
     CRL_FIND_ISSUED_BY, cert, NULL);
    ok(context != NULL, "Expected a context\n");
    if (context)
        CertFreeCRLContext(context);

    if (0)
    {
        /* Crash or return NULL/STATUS_ACCESS_VIOLATION */
        pCertFindCRLInStore(store, 0, 0, CRL_FIND_ISSUED_FOR, NULL,
         NULL);
        pCertFindCRLInStore(store, 0, 0, CRL_FIND_ISSUED_FOR,
         &issuedForPara, NULL);
    }
    /* Test whether the cert matches the CRL in the store */
    issuedForPara.pSubjectCert = cert;
    issuedForPara.pIssuerCert = cert;
    context = pCertFindCRLInStore(store, 0, 0, CRL_FIND_ISSUED_FOR,
     &issuedForPara, NULL);
    ok(context != NULL || broken(!context /* Win9x, NT4 */),
     "Expected a context\n");
    if (context)
    {
        ok(context->cbCrlEncoded == sizeof(signedCRL),
         "unexpected CRL size %d\n", context->cbCrlEncoded);
        ok(!memcmp(context->pbCrlEncoded, signedCRL, context->cbCrlEncoded),
         "unexpected CRL data\n");
        CertFreeCRLContext(context);
    }

    ret = CertAddEncodedCRLToStore(store, X509_ASN_ENCODING,
     v1CRLWithIssuerAndEntry, sizeof(v1CRLWithIssuerAndEntry),
     CERT_STORE_ADD_ALWAYS, NULL);
    ok(ret, "CertAddEncodedCRLToStore failed: %08x\n", GetLastError());
    ret = CertAddEncodedCRLToStore(store, X509_ASN_ENCODING,
     v2CRLWithIssuingDistPoint, sizeof(v2CRLWithIssuingDistPoint),
     CERT_STORE_ADD_ALWAYS, NULL);
    ok(ret, "CertAddEncodedCRLToStore failed: %08x\n", GetLastError());
    ret = CertAddEncodedCRLToStore(store, X509_ASN_ENCODING,
     verisignCRL, sizeof(verisignCRL), CERT_STORE_ADD_ALWAYS, NULL);
    ok(ret, "CertAddEncodedCRLToStore failed: %08x\n", GetLastError());
    issuedForPara.pSubjectCert = cert;
    issuedForPara.pIssuerCert = cert;
    context = NULL;
    count = revoked_count = 0;
    do {
        context = pCertFindCRLInStore(store, 0, 0, CRL_FIND_ISSUED_FOR,
         &issuedForPara, context);
        if (context)
        {
            PCRL_ENTRY entry;

            count++;
            if (pCertFindCertificateInCRL(cert, context, 0, NULL, &entry) &&
             entry)
                revoked_count++;
        }
    } while (context);
    /* signedCRL, v1CRLWithIssuerAndEntry, and v2CRLWithIssuingDistPoint all
     * match cert's issuer, but verisignCRL does not, so the expected count
     * is 0.
     */
    ok(count == 3 || broken(count == 0 /* NT4, Win9x */),
     "expected 3 matching CRLs, got %d\n", count);
    /* Only v1CRLWithIssuerAndEntry and v2CRLWithIssuingDistPoint contain
     * entries, so the count of CRL entries that match cert is 2.
     */
    ok(revoked_count == 2 || broken(revoked_count == 0 /* NT4, Win9x */),
     "expected 2 matching CRL entries, got %d\n", revoked_count);

    CertFreeCertificateContext(cert);

    /* Try again with a cert that doesn't match any CRLs in the store */
    cert = CertCreateCertificateContext(X509_ASN_ENCODING,
     bigCertWithDifferentIssuer, sizeof(bigCertWithDifferentIssuer));
    ok(cert != NULL, "CertCreateCertificateContext failed: %08x\n",
     GetLastError());
    issuedForPara.pSubjectCert = cert;
    issuedForPara.pIssuerCert = cert;
    context = NULL;
    count = revoked_count = 0;
    do {
        context = pCertFindCRLInStore(store, 0, 0, CRL_FIND_ISSUED_FOR,
         &issuedForPara, context);
        if (context)
        {
            PCRL_ENTRY entry;

            count++;
            if (pCertFindCertificateInCRL(cert, context, 0, NULL, &entry) &&
             entry)
                revoked_count++;
        }
    } while (context);
    ok(count == 0, "expected 0 matching CRLs, got %d\n", count);
    ok(revoked_count == 0, "expected 0 matching CRL entries, got %d\n",
     revoked_count);
    CertFreeCertificateContext(cert);

    /* Test again with a real certificate and CRL.  The certificate wasn't
     * revoked, but its issuer does have a CRL.
     */
    cert = CertCreateCertificateContext(X509_ASN_ENCODING,
     verisignCommercialSoftPubCA, sizeof(verisignCommercialSoftPubCA));
    ok(cert != NULL, "CertCreateCertificateContext failed: %08x\n",
     GetLastError());
    issuedForPara.pIssuerCert = cert;
    issuedForPara.pSubjectCert = cert;
    context = NULL;
    count = revoked_count = 0;
    do {
        context = pCertFindCRLInStore(store, 0, 0, CRL_FIND_ISSUED_FOR,
         &issuedForPara, context);
        if (context)
        {
            PCRL_ENTRY entry;

            count++;
            if (pCertFindCertificateInCRL(cert, context, 0, NULL, &entry) &&
             entry)
                revoked_count++;
        }
    } while (context);
    ok(count == 1 || broken(count == 0 /* Win9x, NT4 */),
     "expected 1 matching CRLs, got %d\n", count);
    ok(revoked_count == 0, "expected 0 matching CRL entries, got %d\n",
     revoked_count);
    CertFreeCertificateContext(cert);

    CertCloseStore(store, 0);

    /* This test uses a synthesized chain (rootWithKeySignAndCRLSign ->
     * eeCert) whose end certificate is in the CRL.
     */
    store = CertOpenStore(CERT_STORE_PROV_MEMORY, 0, 0,
     CERT_STORE_CREATE_NEW_FLAG, NULL);
    /* Add a CRL for the end certificate */
    CertAddEncodedCRLToStore(store, X509_ASN_ENCODING,
     rootSignedCRL, sizeof(rootSignedCRL), CERT_STORE_ADD_ALWAYS, NULL);
    /* Add another CRL unrelated to the tested chain */
    CertAddEncodedCRLToStore(store, X509_ASN_ENCODING,
     verisignCRL, sizeof(verisignCRL), CERT_STORE_ADD_ALWAYS, NULL);
    endCert = CertCreateCertificateContext(X509_ASN_ENCODING,
     eeCert, sizeof(eeCert));
    rootCert = CertCreateCertificateContext(X509_ASN_ENCODING,
     rootWithKeySignAndCRLSign, sizeof(rootWithKeySignAndCRLSign));
    issuedForPara.pSubjectCert = endCert;
    issuedForPara.pIssuerCert = rootCert;
    context = NULL;
    count = revoked_count = 0;
    do {
        context = pCertFindCRLInStore(store, 0, 0, CRL_FIND_ISSUED_FOR,
         &issuedForPara, context);
        if (context)
        {
            PCRL_ENTRY entry;

            count++;
            if (pCertFindCertificateInCRL(endCert, context, 0, NULL, &entry) &&
             entry)
                revoked_count++;
        }
    } while (context);
    ok(count == 1 || broken(count == 0 /* Win9x, NT4 */),
     "expected 1 matching CRLs, got %d\n", count);
    ok(revoked_count == 1 || broken(revoked_count == 0 /* Win9x, NT4 */),
     "expected 1 matching CRL entries, got %d\n", revoked_count);

    /* Test CRL_FIND_ISSUED_BY flags */
    count = revoked_count = 0;
    do {
        context = pCertFindCRLInStore(store, 0, 0, CRL_FIND_ISSUED_BY,
         endCert, context);
        if (context)
        {
            PCRL_ENTRY entry;

            count++;
            if (pCertFindCertificateInCRL(endCert, context, 0, NULL, &entry) &&
             entry)
                revoked_count++;
        }
    } while (context);
    ok(count == 0, "expected 0 matching CRLs, got %d\n", count);
    ok(revoked_count == 0, "expected 0 matching CRL entries, got %d\n",
     revoked_count);
    count = revoked_count = 0;
    do {
        context = pCertFindCRLInStore(store, 0, 0, CRL_FIND_ISSUED_BY,
         rootCert, context);
        if (context)
        {
            PCRL_ENTRY entry;

            count++;
            if (pCertFindCertificateInCRL(endCert, context, 0, NULL, &entry) &&
             entry)
                revoked_count++;
        }
    } while (context);
    ok(count == 1, "expected 1 matching CRLs, got %d\n", count);
    ok(revoked_count == 1, "expected 1 matching CRL entries, got %d\n",
     revoked_count);
    count = revoked_count = 0;
    do {
        context = pCertFindCRLInStore(store, 0, CRL_FIND_ISSUED_BY_AKI_FLAG,
         CRL_FIND_ISSUED_BY, endCert, context);
        if (context)
        {
            PCRL_ENTRY entry;

            count++;
            if (pCertFindCertificateInCRL(endCert, context, 0, NULL, &entry) &&
             entry)
                revoked_count++;
        }
    } while (context);
    ok(count == 0, "expected 0 matching CRLs, got %d\n", count);
    ok(revoked_count == 0, "expected 0 matching CRL entries, got %d\n",
     revoked_count);
    count = revoked_count = 0;
    do {
        context = pCertFindCRLInStore(store, 0, CRL_FIND_ISSUED_BY_AKI_FLAG,
         CRL_FIND_ISSUED_BY, rootCert, context);
        if (context)
        {
            PCRL_ENTRY entry;

            count++;
            if (pCertFindCertificateInCRL(rootCert, context, 0, NULL, &entry) &&
             entry)
                revoked_count++;
        }
    } while (context);
    ok(count == 0 || broken(count == 1 /* Win9x */),
     "expected 0 matching CRLs, got %d\n", count);
    ok(revoked_count == 0, "expected 0 matching CRL entries, got %d\n",
     revoked_count);
    count = revoked_count = 0;
    do {
        context = pCertFindCRLInStore(store, 0,
         CRL_FIND_ISSUED_BY_SIGNATURE_FLAG, CRL_FIND_ISSUED_BY, endCert,
         context);
        if (context)
        {
            PCRL_ENTRY entry;

            count++;
            if (pCertFindCertificateInCRL(endCert, context, 0, NULL, &entry) &&
             entry)
                revoked_count++;
        }
    } while (context);
    ok(count == 0, "expected 0 matching CRLs, got %d\n", count);
    ok(revoked_count == 0, "expected 0 matching CRL entries, got %d\n",
     revoked_count);
    count = revoked_count = 0;
    do {
        context = pCertFindCRLInStore(store, 0,
         CRL_FIND_ISSUED_BY_SIGNATURE_FLAG, CRL_FIND_ISSUED_BY, rootCert,
         context);
        if (context)
        {
            PCRL_ENTRY entry;

            count++;
            if (pCertFindCertificateInCRL(endCert, context, 0, NULL, &entry) &&
             entry)
                revoked_count++;
        }
    } while (context);
    ok(count == 1, "expected 1 matching CRLs, got %d\n", count);
    ok(revoked_count == 1, "expected 1 matching CRL entries, got %d\n",
     revoked_count);
    CertFreeCertificateContext(rootCert);
    CertFreeCertificateContext(endCert);

    CertCloseStore(store, 0);
}

static void testGetCRLFromStore(void)
{
    HCERTSTORE store = CertOpenStore(CERT_STORE_PROV_MEMORY, 0, 0,
     CERT_STORE_CREATE_NEW_FLAG, NULL);
    PCCRL_CONTEXT context;
    PCCERT_CONTEXT cert;
    DWORD flags;
    BOOL ret;

    if (!store) return;

    /* Crash
    context = CertGetCRLFromStore(NULL, NULL, NULL, NULL);
    context = CertGetCRLFromStore(store, NULL, NULL, NULL);
     */

    /* Bogus flags */
    flags = 0xffffffff;
    context = CertGetCRLFromStore(store, NULL, NULL, &flags);
    ok(!context && GetLastError() == E_INVALIDARG,
     "Expected E_INVALIDARG, got %08x\n", GetLastError());

    /* Test an empty store */
    flags = 0;
    context = CertGetCRLFromStore(store, NULL, NULL, &flags);
    ok(context == NULL && GetLastError() == CRYPT_E_NOT_FOUND,
     "Expected CRYPT_E_NOT_FOUND, got %08x\n", GetLastError());

    ret = CertAddEncodedCRLToStore(store, X509_ASN_ENCODING, signedCRL,
     sizeof(signedCRL), CERT_STORE_ADD_ALWAYS, NULL);
    ok(ret, "CertAddEncodedCRLToStore failed: %08x\n", GetLastError());

    /* NULL matches any CRL */
    flags = 0;
    context = CertGetCRLFromStore(store, NULL, NULL, &flags);
    ok(context != NULL, "Expected a context\n");
    CertFreeCRLContext(context);

    /* This cert's issuer isn't in */
    cert = CertCreateCertificateContext(X509_ASN_ENCODING, bigCert2,
     sizeof(bigCert2));
    ok(cert != NULL, "CertCreateCertificateContext failed: %08x\n",
     GetLastError());
    context = CertGetCRLFromStore(store, cert, NULL, &flags);
    ok(context == NULL && GetLastError() == CRYPT_E_NOT_FOUND,
     "Expected CRYPT_E_NOT_FOUND, got %08x\n", GetLastError());
    CertFreeCertificateContext(cert);

    /* But this one is */
    cert = CertCreateCertificateContext(X509_ASN_ENCODING, bigCert,
     sizeof(bigCert));
    ok(cert != NULL, "CertCreateCertificateContext failed: %08x\n",
     GetLastError());
    context = CertGetCRLFromStore(store, cert, NULL, &flags);
    ok(context != NULL, "Expected a context\n");
    CertFreeCRLContext(context);
    CertFreeCertificateContext(cert);

    CertCloseStore(store, 0);
}

static void checkCRLHash(const BYTE *data, DWORD dataLen, ALG_ID algID,
 PCCRL_CONTEXT context, DWORD propID)
{
    BYTE hash[20] = { 0 }, hashProperty[20];
    BOOL ret;
    DWORD size;

    memset(hash, 0, sizeof(hash));
    memset(hashProperty, 0, sizeof(hashProperty));
    size = sizeof(hash);
    ret = CryptHashCertificate(0, algID, 0, data, dataLen, hash, &size);
    ok(ret, "CryptHashCertificate failed: %08x\n", GetLastError());
    ret = CertGetCRLContextProperty(context, propID, hashProperty, &size);
    ok(ret, "CertGetCRLContextProperty failed: %08x\n", GetLastError());
    ok(!memcmp(hash, hashProperty, size), "Unexpected hash for property %d\n",
     propID);
}

static void testCRLProperties(void)
{
    PCCRL_CONTEXT context = CertCreateCRLContext(X509_ASN_ENCODING,
     CRL, sizeof(CRL));

    ok(context != NULL, "CertCreateCRLContext failed: %08x\n", GetLastError());
    if (context)
    {
        DWORD propID, numProps, access, size;
        BOOL ret;
        BYTE hash[20] = { 0 }, hashProperty[20];
        CRYPT_DATA_BLOB blob;

        /* This crashes
        propID = CertEnumCRLContextProperties(NULL, 0);
         */

        propID = 0;
        numProps = 0;
        do {
            propID = CertEnumCRLContextProperties(context, propID);
            if (propID)
                numProps++;
        } while (propID != 0);
        ok(numProps == 0, "Expected 0 properties, got %d\n", numProps);

        /* Tests with a NULL cert context.  Prop ID 0 fails.. */
        ret = CertSetCRLContextProperty(NULL, 0, 0, NULL);
        ok(!ret && GetLastError() == E_INVALIDARG,
         "Expected E_INVALIDARG, got %08x\n", GetLastError());
        /* while this just crashes.
        ret = CertSetCRLContextProperty(NULL, CERT_KEY_PROV_HANDLE_PROP_ID, 0,
         NULL);
         */

        ret = CertSetCRLContextProperty(context, 0, 0, NULL);
        ok(!ret && GetLastError() == E_INVALIDARG,
         "Expected E_INVALIDARG, got %08x\n", GetLastError());
        /* Can't set the cert property directly, this crashes.
        ret = CertSetCRLContextProperty(context, CERT_CRL_PROP_ID, 0, CRL);
         */

        /* These all crash.
        ret = CertGetCRLContextProperty(context, CERT_ACCESS_STATE_PROP_ID, 0,
         NULL);
        ret = CertGetCRLContextProperty(context, CERT_HASH_PROP_ID, NULL, NULL);
        ret = CertGetCRLContextProperty(context, CERT_HASH_PROP_ID, 
         hashProperty, NULL);
         */
        /* A missing prop */
        size = 0;
        ret = CertGetCRLContextProperty(context, CERT_KEY_PROV_INFO_PROP_ID,
         NULL, &size);
        ok(!ret && GetLastError() == CRYPT_E_NOT_FOUND,
         "Expected CRYPT_E_NOT_FOUND, got %08x\n", GetLastError());
        /* And, an implicit property */
        ret = CertGetCRLContextProperty(context, CERT_ACCESS_STATE_PROP_ID,
         NULL, &size);
        ok(ret, "CertGetCRLContextProperty failed: %08x\n", GetLastError());
        ret = CertGetCRLContextProperty(context, CERT_ACCESS_STATE_PROP_ID,
         &access, &size);
        ok(ret, "CertGetCRLContextProperty failed: %08x\n", GetLastError());
        ok(!(access & CERT_ACCESS_STATE_WRITE_PERSIST_FLAG),
         "Didn't expect a persisted crl\n");
        /* Trying to set this "read only" property crashes.
        access |= CERT_ACCESS_STATE_WRITE_PERSIST_FLAG;
        ret = CertSetCRLContextProperty(context, CERT_ACCESS_STATE_PROP_ID, 0,
         &access);
         */

        /* Can I set the hash to an invalid hash? */
        blob.pbData = hash;
        blob.cbData = sizeof(hash);
        ret = CertSetCRLContextProperty(context, CERT_HASH_PROP_ID, 0, &blob);
        ok(ret, "CertSetCRLContextProperty failed: %08x\n",
         GetLastError());
        size = sizeof(hashProperty);
        ret = CertGetCRLContextProperty(context, CERT_HASH_PROP_ID,
         hashProperty, &size);
        ok(ret, "CertSetCRLContextProperty failed: %08x\n", GetLastError());
        ok(!memcmp(hashProperty, hash, sizeof(hash)), "Unexpected hash\n");
        /* Delete the (bogus) hash, and get the real one */
        ret = CertSetCRLContextProperty(context, CERT_HASH_PROP_ID, 0, NULL);
        ok(ret, "CertSetCRLContextProperty failed: %08x\n", GetLastError());
        checkCRLHash(CRL, sizeof(CRL), CALG_SHA1, context, CERT_HASH_PROP_ID);

        /* Now that the hash property is set, we should get one property when
         * enumerating.
         */
        propID = 0;
        numProps = 0;
        do {
            propID = CertEnumCRLContextProperties(context, propID);
            if (propID)
                numProps++;
        } while (propID != 0);
        ok(numProps == 1, "Expected 1 properties, got %d\n", numProps);

        /* Check a few other implicit properties */
        checkCRLHash(CRL, sizeof(CRL), CALG_MD5, context,
         CERT_MD5_HASH_PROP_ID);

        CertFreeCRLContext(context);
    }
}

static const BYTE bigCertWithCRLDistPoints[] = {
0x30,0x81,0xa5,0x02,0x01,0x01,0x30,0x02,0x06,0x00,0x30,0x15,0x31,0x13,0x30,
0x11,0x06,0x03,0x55,0x04,0x03,0x13,0x0a,0x4a,0x75,0x61,0x6e,0x20,0x4c,0x61,
0x6e,0x67,0x00,0x30,0x22,0x18,0x0f,0x31,0x36,0x30,0x31,0x30,0x31,0x30,0x31,
0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x18,0x0f,0x31,0x36,0x30,0x31,0x30,0x31,
0x30,0x31,0x30,0x30,0x30,0x30,0x30,0x30,0x5a,0x30,0x15,0x31,0x13,0x30,0x11,
0x06,0x03,0x55,0x04,0x03,0x13,0x0a,0x4a,0x75,0x61,0x6e,0x20,0x4c,0x61,0x6e,
0x67,0x00,0x30,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,
0x01,0x01,0x05,0x00,0x03,0x11,0x00,0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0xa3,0x26,0x30,0x24,0x30,0x22,0x06,
0x03,0x55,0x1d,0x1f,0x04,0x1b,0x30,0x19,0x30,0x17,0xa0,0x15,0xa0,0x13,0x86,
0x11,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x77,0x69,0x6e,0x65,0x68,0x71,0x2e,
0x6f,0x72,0x67 };

static void testIsValidCRLForCert(void)
{
    BOOL ret;
    PCCERT_CONTEXT cert1, cert2, cert3;
    PCCRL_CONTEXT crl;
    HCERTSTORE store;

    if(!pCertIsValidCRLForCertificate) return;

    crl = CertCreateCRLContext(X509_ASN_ENCODING, v1CRLWithIssuerAndEntry,
     sizeof(v1CRLWithIssuerAndEntry));
    ok(crl != NULL, "CertCreateCRLContext failed: %08x\n", GetLastError());
    cert1 = CertCreateCertificateContext(X509_ASN_ENCODING, bigCert,
     sizeof(bigCert));
    ok(cert1 != NULL, "CertCreateCertificateContext failed: %08x\n",
     GetLastError());

    /* Crash
    ret = CertIsValidCRLForCertificate(NULL, NULL, 0, NULL);
    ret = CertIsValidCRLForCertificate(cert1, NULL, 0, NULL);
     */

    /* Curiously, any CRL is valid for the NULL certificate */
    ret = pCertIsValidCRLForCertificate(NULL, crl, 0, NULL);
    ok(ret, "CertIsValidCRLForCertificate failed: %08x\n", GetLastError());

    /* Same issuer for both cert and CRL, this CRL is valid for that cert */
    ret = pCertIsValidCRLForCertificate(cert1, crl, 0, NULL);
    ok(ret, "CertIsValidCRLForCertificate failed: %08x\n", GetLastError());

    cert2 = CertCreateCertificateContext(X509_ASN_ENCODING,
     bigCertWithDifferentIssuer, sizeof(bigCertWithDifferentIssuer));
    ok(cert2 != NULL, "CertCreateCertificateContext failed: %08x\n",
     GetLastError());

    /* Yet more curious: different issuers for these, yet the CRL is valid for
     * that cert.  According to MSDN, the relevant bit to check is whether the
     * CRL has a CRL_ISSUING_DIST_POINT extension.
     */
    ret = pCertIsValidCRLForCertificate(cert2, crl, 0, NULL);
    ok(ret, "CertIsValidCRLForCertificate failed: %08x\n", GetLastError());

    CertFreeCRLContext(crl);

    /* With a CRL_ISSUING_DIST_POINT in the CRL, it returns FALSE, since the
     * cert doesn't have the same extension in it.
     */
    crl = CertCreateCRLContext(X509_ASN_ENCODING, v2CRLWithIssuingDistPoint,
     sizeof(v2CRLWithIssuingDistPoint));
    ok(crl != NULL, "CertCreateCRLContext failed: %08x\n", GetLastError());

    ret = pCertIsValidCRLForCertificate(cert1, crl, 0, NULL);
    ok(!ret && GetLastError() == CRYPT_E_NO_MATCH,
     "expected CRYPT_E_NO_MATCH, got %08x\n", GetLastError());
    ret = pCertIsValidCRLForCertificate(cert2, crl, 0, NULL);
    ok(!ret && GetLastError() == CRYPT_E_NO_MATCH,
     "expected CRYPT_E_NO_MATCH, got %08x\n", GetLastError());

    /* With a CRL_ISSUING_DIST_POINT in the CRL, it matches the cert containing
     * a CRL_DIST_POINTS_INFO extension.
     */
    cert3 = CertCreateCertificateContext(X509_ASN_ENCODING,
     bigCertWithCRLDistPoints, sizeof(bigCertWithCRLDistPoints));
    ok(cert3 != NULL, "CertCreateCertificateContext failed: %08x\n",
     GetLastError());
    ret = pCertIsValidCRLForCertificate(cert3, crl, 0, NULL);
    ok(ret, "CertIsValidCRLForCertificate failed: %08x\n", GetLastError());

    CertFreeCRLContext(crl);

    /* And again, with a real CRL, the CRL is valid for all three certs. */
    crl = CertCreateCRLContext(X509_ASN_ENCODING, verisignCRL,
     sizeof(verisignCRL));
    ok(crl != NULL, "CertCreateCRLContext failed: %08x\n", GetLastError());

    ret = pCertIsValidCRLForCertificate(cert1, crl, 0, NULL);
    ok(ret, "CertIsValidCRLForCertificate failed: %08x\n", GetLastError());
    ret = pCertIsValidCRLForCertificate(cert2, crl, 0, NULL);
    ok(ret, "CertIsValidCRLForCertificate failed: %08x\n", GetLastError());
    ret = pCertIsValidCRLForCertificate(cert3, crl, 0, NULL);
    ok(ret, "CertIsValidCRLForCertificate failed: %08x\n", GetLastError());

    CertFreeCRLContext(crl);

    /* One last test: a CRL in a different store than the cert is also valid
     * for the cert.
     */
    store = CertOpenStore(CERT_STORE_PROV_MEMORY, X509_ASN_ENCODING, 0,
     CERT_STORE_CREATE_NEW_FLAG, NULL);
    ok(store != NULL, "CertOpenStore failed: %08x\n", GetLastError());

    ret = CertAddEncodedCRLToStore(store, X509_ASN_ENCODING, verisignCRL,
     sizeof(verisignCRL), CERT_STORE_ADD_ALWAYS, &crl);
    ok(ret, "CertAddEncodedCRLToStore failed: %08x\n", GetLastError());

    ret = pCertIsValidCRLForCertificate(cert1, crl, 0, NULL);
    ok(ret, "CertIsValidCRLForCertificate failed: %08x\n", GetLastError());
    ret = pCertIsValidCRLForCertificate(cert2, crl, 0, NULL);
    ok(ret, "CertIsValidCRLForCertificate failed: %08x\n", GetLastError());
    ret = pCertIsValidCRLForCertificate(cert3, crl, 0, NULL);
    ok(ret, "CertIsValidCRLForCertificate failed: %08x\n", GetLastError());

    CertFreeCRLContext(crl);

    CertCloseStore(store, 0);

    CertFreeCertificateContext(cert3);
    CertFreeCertificateContext(cert2);
    CertFreeCertificateContext(cert1);
}

static const BYTE crlWithDifferentIssuer[] = {
 0x30,0x47,0x02,0x01,0x01,0x30,0x02,0x06,0x00,0x30,0x15,0x31,0x13,0x30,0x11,
 0x06,0x03,0x55,0x04,0x03,0x13,0x0a,0x41,0x6c,0x65,0x78,0x20,0x4c,0x61,0x6e,
 0x67,0x00,0x18,0x0f,0x31,0x36,0x30,0x31,0x30,0x31,0x30,0x31,0x30,0x30,0x30,
 0x30,0x30,0x30,0x5a,0x30,0x16,0x30,0x14,0x02,0x01,0x01,0x18,0x0f,0x31,0x36,
 0x30,0x31,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,0x30,0x5a };

static void testFindCertInCRL(void)
{
    BOOL ret;
    PCCERT_CONTEXT cert;
    PCCRL_CONTEXT crl;
    PCRL_ENTRY entry;

    if (!pCertFindCertificateInCRL)
    {
        win_skip("CertFindCertificateInCRL() is not available\n");
        return;
    }

    cert = CertCreateCertificateContext(X509_ASN_ENCODING, bigCert,
     sizeof(bigCert));
    ok(cert != NULL, "CertCreateCertificateContext failed: %08x\n",
     GetLastError());

    /* Crash
    ret = pCertFindCertificateInCRL(NULL, NULL, 0, NULL, NULL);
    ret = pCertFindCertificateInCRL(NULL, crl, 0, NULL, NULL);
    ret = pCertFindCertificateInCRL(cert, NULL, 0, NULL, NULL);
    ret = pCertFindCertificateInCRL(cert, crl, 0, NULL, NULL);
    ret = pCertFindCertificateInCRL(NULL, NULL, 0, NULL, &entry);
    ret = pCertFindCertificateInCRL(NULL, crl, 0, NULL, &entry);
    ret = pCertFindCertificateInCRL(cert, NULL, 0, NULL, &entry);
     */

    crl = CertCreateCRLContext(X509_ASN_ENCODING, verisignCRL,
     sizeof(verisignCRL));
    ret = pCertFindCertificateInCRL(cert, crl, 0, NULL, &entry);
    ok(ret, "CertFindCertificateInCRL failed: %08x\n", GetLastError());
    ok(entry == NULL, "Expected not to find an entry in CRL\n");
    CertFreeCRLContext(crl);

    crl = CertCreateCRLContext(X509_ASN_ENCODING, v1CRLWithIssuerAndEntry,
     sizeof(v1CRLWithIssuerAndEntry));
    ret = pCertFindCertificateInCRL(cert, crl, 0, NULL, &entry);
    ok(ret, "CertFindCertificateInCRL failed: %08x\n", GetLastError());
    ok(entry != NULL, "Expected to find an entry in CRL\n");
    CertFreeCRLContext(crl);

    /* Entry found even though CRL issuer doesn't match cert issuer */
    crl = CertCreateCRLContext(X509_ASN_ENCODING, crlWithDifferentIssuer,
     sizeof(crlWithDifferentIssuer));
    ret = pCertFindCertificateInCRL(cert, crl, 0, NULL, &entry);
    ok(ret, "CertFindCertificateInCRL failed: %08x\n", GetLastError());
    ok(entry != NULL, "Expected to find an entry in CRL\n");
    CertFreeCRLContext(crl);

    CertFreeCertificateContext(cert);
}

static void testVerifyCRLRevocation(void)
{
    BOOL ret;
    PCCERT_CONTEXT cert;
    PCCRL_CONTEXT crl;

    ret = CertVerifyCRLRevocation(0, NULL, 0, NULL);
    ok(ret, "CertVerifyCRLRevocation failed: %08x\n", GetLastError());
    ret = CertVerifyCRLRevocation(X509_ASN_ENCODING, NULL, 0, NULL);
    ok(ret, "CertVerifyCRLRevocation failed: %08x\n", GetLastError());

    cert = CertCreateCertificateContext(X509_ASN_ENCODING, bigCert,
     sizeof(bigCert));

    /* Check against no CRL */
    ret = CertVerifyCRLRevocation(0, cert->pCertInfo, 0, NULL);
    ok(ret, "CertVerifyCRLRevocation failed: %08x\n", GetLastError());
    ret = CertVerifyCRLRevocation(X509_ASN_ENCODING, cert->pCertInfo, 0, NULL);
    ok(ret, "CertVerifyCRLRevocation failed: %08x\n", GetLastError());

    /* Check against CRL with entry for the cert */
    crl = CertCreateCRLContext(X509_ASN_ENCODING, v1CRLWithIssuerAndEntry,
     sizeof(v1CRLWithIssuerAndEntry));
    ret = CertVerifyCRLRevocation(0, cert->pCertInfo, 1,
     (PCRL_INFO *)&crl->pCrlInfo);
    ok(!ret, "CertVerifyCRLRevocation should have been revoked\n");
    ret = CertVerifyCRLRevocation(X509_ASN_ENCODING, cert->pCertInfo, 1,
     (PCRL_INFO *)&crl->pCrlInfo);
    ok(!ret, "CertVerifyCRLRevocation should have been revoked\n");
    CertFreeCRLContext(crl);

    /* Check against CRL with different issuer and entry for the cert */
    crl = CertCreateCRLContext(X509_ASN_ENCODING, crlWithDifferentIssuer,
     sizeof(crlWithDifferentIssuer));
    ok(crl != NULL, "CertCreateCRLContext failed: %08x\n", GetLastError());
    ret = CertVerifyCRLRevocation(X509_ASN_ENCODING, cert->pCertInfo, 1,
     (PCRL_INFO *)&crl->pCrlInfo);
    ok(!ret, "CertVerifyCRLRevocation should have been revoked\n");
    CertFreeCRLContext(crl);

    /* Check against CRL without entry for the cert */
    crl = CertCreateCRLContext(X509_ASN_ENCODING, verisignCRL,
     sizeof(verisignCRL));
    ret = CertVerifyCRLRevocation(0, cert->pCertInfo, 1,
     (PCRL_INFO *)&crl->pCrlInfo);
    ok(ret, "CertVerifyCRLRevocation failed: %08x\n", GetLastError());
    ret = CertVerifyCRLRevocation(X509_ASN_ENCODING, cert->pCertInfo, 1,
     (PCRL_INFO *)&crl->pCrlInfo);
    ok(ret, "CertVerifyCRLRevocation failed: %08x\n", GetLastError());
    CertFreeCRLContext(crl);

    CertFreeCertificateContext(cert);
}

START_TEST(crl)
{
    init_function_pointers();

    testCreateCRL();
    testDupCRL();
    testAddCRL();
    testFindCRL();
    testGetCRLFromStore();

    testCRLProperties();

    testIsValidCRLForCert();
    testFindCertInCRL();
    testVerifyCRLRevocation();
}
