# Copyright 2019 Xilinx Inc.
# 
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
# 
#     http://www.apache.org/licenses/LICENSE-2.0
# 
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Exposes the Python wrapper for decent_q utils."""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

# pylint: disable=unused-import,wildcard-import, line-too-long
from tensorflow.core.framework import graph_pb2
from tensorflow.python.framework import errors
from tensorflow.python.pywrap_tensorflow import DecentQTransformWithStringInputs
from tensorflow.python.util import compat

def DecentQTransform(input_graph_def, inputs, outputs, transforms):
  """Python wrapper for the Graph Transform Tool.

  Gives access to all graph transforms available through the command line tool.
  See documentation at https://github.com/tensorflow/tensorflow/blob/master/tensorflow/tools/graph_transforms/README.md
  for full details of the options available.

  Args:
    input_graph_def: GraphDef object containing a model to be transformed.
    inputs: List of node names for the model inputs.
    outputs: List of node names for the model outputs.
    transforms: List of strings containing transform names and parameters.

  Returns:
    New GraphDef with transforms applied.
  """

  input_graph_def_string = input_graph_def.SerializeToString()
  inputs_string = compat.as_bytes(",".join(inputs))
  outputs_string = compat.as_bytes(",".join(outputs))
  transforms_string = compat.as_bytes(" ".join(transforms))
  with errors.raise_exception_on_not_ok_status() as status:
    output_graph_def_string = DecentQTransformWithStringInputs(
        input_graph_def_string, inputs_string, outputs_string,
        transforms_string, status)
  output_graph_def = graph_pb2.GraphDef()
  output_graph_def.ParseFromString(output_graph_def_string)
  return output_graph_def
