/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/
/*!
 *\file Render_template.hpp
 */

#ifndef RENDER_TEMPLATE_HPP
#define RENDER_TEMPLATE_HPP


/*!
 * \brief This class provides an interface for render classes like OpenGL or SVG.
 */
class Render_template
{ 

public:
     
    /*!
     * \brief Proceeds with the initialization of draw functions.
     */
    virtual void start_draw() = 0;

    /*!
     * \brief Proceeds with the initialization of container draw functions.
     */
    virtual void start_draw_containers() = 0;

    /*!
     * \brief Called when all container draws are finished.
     */
    virtual void end_draw_containers() = 0;
     
    /*!
     * \brief Proceeds with the initialization of state draw functions.
     */
    virtual void start_draw_states() = 0;

    /*!
     * \brief Called when all state draws are finished.
     */
    virtual void end_draw_states() = 0;

    /*!
     * \brief Proceeds with the initialization of arrow draw functions.
     */
    virtual void start_draw_arrows() = 0;

    /*!
     * \brief Called when all arrow draws are finished.
     */
    virtual void end_draw_arrows() = 0;

    /*!
     * \brief Proceeds with the initialization of counter draw functions.
     */
    virtual void start_draw_counter() = 0;
     
    /*!
     * \brief Called when all counter draws are finished.
     */
    virtual void end_draw_counter() = 0;

    /*!
     * \brief Called before ruler drawing.
     */
    virtual void start_ruler() = 0;
     
    /*!
     * \brief Called after ruler drawing.
     */
    virtual void end_ruler() = 0;
     
    /*!
     * \brief Called when all draws are finished.
     */
    virtual void end_draw() = 0;

    /*!
     * \brief Set the color for the further drawings.
     * \param r the red value. Within [0 ; 1].
     * \param g the green value. Within [0 ; 1].
     * \param b the blue value. Within [0 ; 1].   
     */
    virtual void set_color(float r, float g, float b) = 0;

    /*!
     * \brief Draw a text.
     * \param x the horizontal position of the left bottom corner of the text.
     * \param y the vertical position of the left bottom corner of the text.
     * \param z the deep position of the text.
     * \param s the text.
     */
    virtual void draw_text(const Element_pos x, const Element_pos y, const Element_pos z, const std::string s) = 0;

    /*!
     * \brief Draw a quad.
     * \param x the horizontal position of the left bottom corner of the quad.
     * \param y the vertical position of the left bottom corner of the quad.
     * \param z the deep position of the quad.
     * \param w the width of the quad.
     * \param h the height of the quad.
     */
    virtual void draw_quad(Element_pos x, Element_pos y, Element_pos z, Element_pos w, Element_pos h) = 0;

    /*!
     * \brief Draw a triangle.
     * \param x the horizontal position of the triangle center.
     * \param y the vertical position of the triangle center.
     * \param size the edge size.
     * \param r the rotation of triangle. (clockwise and in degree)
     */
    void draw_triangle(Element_pos x, Element_pos y,
                       Element_pos size, Element_pos r);

    /*!
     * \brief Draw a line.
     * \param x1 the horizontal position of the first point.
     * \param y1 the vertical position of the firt point.
     * \param x2 the horizontal position of the second point.
     * \param y2 the vertical position of the second point.
     * \param z the deep position of the triangle.
     */
    virtual void draw_line(Element_pos x1, Element_pos y1, Element_pos x2, Element_pos y2, Element_pos z) = 0;

    /*!
     * \brief Draw a circle.
     * \param x the horizontal position of the circle center.
     * \param y the vertical position of the circle center.
     * \param z the deep position of the circle.
     * \param r the circle radius.
     */
    virtual void draw_circle(Element_pos x, Element_pos y, Element_pos z, Element_pos r) = 0;

};



#endif
