-- test/test-tongue.codex.lua
--
-- Lua I18N library 'Tongue' -- Tests for the codex module
--
-- Copyright 2016 Daniel Silverstone <dsilvers@digital-scurf.org>
--
-- For Licence terms, see COPYING
--

-- Step one, start coverage

pcall(require, 'luacov')

local tongue = require 'tongue'

local testnames = {}

local real_assert = assert
local total_asserts = 0
local function assert(...)
   local retval = real_assert(...)
   total_asserts = total_asserts + 1
   return retval
end

local function add_test(suite, name, value)
   rawset(suite, name, value)
   testnames[#testnames+1] = name
end

local suite = setmetatable({}, {__newindex = add_test})

function suite.createCodexBasicCodex()
   local codex = assert(tongue.codex.create(), "Cannot create a codex")
   local pack = assert(codex:get_langpack("en"), "Cannot get a langpack")
   assert(pack:expand("Foo", {}), "Unable to perform an expansion")
end

function suite.createCodexVerifyParentage()
   local codex = assert(tongue.codex.create(), "Cannot create a codex")
   local pack1 = assert(codex:get_langpack("en"), "Cannot get a langpack")
   local pack2 = assert(codex:get_langpack("en", "gb"), "Cannot get a langpack")
   assert(pack2.parent == pack1, "Default pack generator not parenting properly")
end

function suite.createCodexExternalCreator()
   local onepack = tongue.langpack.create("en")
   local function createpack(codex, lang, sublang)
      return onepack
   end
   local codex = assert(tongue.codex.create(createpack), "Cannot create a codex")
   local pack1 = assert(codex:get_langpack("en"), "Cannot get a langpack")
   local pack2 = assert(codex:get_langpack("en", "gb"), "Cannot get a langpack")
   assert(pack2 == pack1, "Our pack creator not running")
end

function suite.createCodexPreAddLoader()
   local codex = assert(tongue.codex.create(createpack), "Cannot create a codex")
   local loaded = {}
   local function loader(codex, langpack, langentry)
      loaded[langentry] = true
      langpack:add_token("FOO", "BAR")
   end
   codex:add_loader(loader)
   assert(codex:expand("en", nil, "FOO") == "BAR", "Expansion failed")
   assert(loaded['en'], "Didn't load 'en'")
   assert(codex:expand("en", "GB", "FOO") == "BAR", "Expansion failed")
   assert(loaded['en_GB'], "Didn't load 'en_GB'")
end

function suite.createCodexPostAddLoader()
   local codex = assert(tongue.codex.create(createpack), "Cannot create a codex")
   local loaded = {}
   local function loader(codex, langpack, langentry)
      loaded[langentry] = true
      langpack:add_token("FOO", "BAR")
   end
   assert(codex:expand("en", nil, "FOO") ~= "BAR", "Expansion succeeded?")
   assert(codex:expand("en", "GB", "FOO") ~= "BAR", "Expansion succeeded?")
   codex:add_loader(loader)
   assert(codex:expand("en", nil, "FOO") == "BAR", "Expansion failed")
   assert(loaded['en'], "Didn't load 'en'")
   assert(codex:expand("en", "GB", "FOO") == "BAR", "Expansion failed")
   assert(loaded['en_GB'], "Didn't load 'en_GB'")
end

local count_ok = 0
for _, testname in ipairs(testnames) do
--   print("Run: " .. testname)
   local ok, err = xpcall(suite[testname], debug.traceback)
   if not ok then
      print(err)
      print()
   else
      count_ok = count_ok + 1
   end
end

print(tostring(count_ok) .. "/" .. tostring(#testnames) .. " [" .. tostring(total_asserts) .. "] OK")

os.exit(count_ok == #testnames and 0 or 1)
