/*
 * Copyright (C) 2008 Instituto Nokia de Tecnologia. All rights reserved.
 *
 * This file is part of QZion.
 *
 * QZion is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * QZion is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with QZion.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __QZIONABSTRACTCANVAS_H__
#define __QZIONABSTRACTCANVAS_H__

#include <QtCore>
#include <QtCore/QList>
#include <QtCore/QPoint>
#include <QtGui/QPainter>
#include <QMouseEvent>
#include <QWheelEvent>
#include <QGraphicsItem>

class QZionObject;
class QZionObjectPrivate;
class QZionAbstractCanvasPrivate;

/*!
    \class QZionAbstractCanvas
    \brief Container class.

    A QZionAbstractCanvas is a set of canvas items.
*/
class QZionAbstractCanvas
{
public:
    QZionAbstractCanvas();
    virtual ~QZionAbstractCanvas();

    /*!
      Returns a const pointer to the list holding all the items in the canvas.
    */
    const QList<QZionObject*> *items() const;

    /*!
      Returns the topmost object at a given position.

      \sa itemsAt()
    */
    QZionObject *itemAt(const QPoint &pos) const;

    /*!
      \overload

      It's equivalent to calling itemAt(QPoint(x, y)).

      \sa itemsAt()
    */
    QZionObject *itemAt(const int x, const int y) const;

    /*!
      Returns all the items at the given position, starting from the
      topmost one.
    */
    QList<QZionObject*> itemsAt(const QPoint &pos) const;
    /*!
      \overload

      It's equivalent to calling itemsAt(QPoint(x, y)).
    */
    QList<QZionObject*> itemsAt(const int x, const int y) const;

    /*!
      Virtual function to ensure an update is pending, called from children.
    */
    virtual void ensurePendingUpdate() = 0;

    /*!
      Virtual function to mark a rect as dirty in canvas.
      This dirty rect will be repainted in the next paint event.
    */
    virtual void invalidate(const QRect &r, bool translate = true) = 0;

    /*!
      Virtual function to mark a region as dirty in canvas.
      This dirty region will be repainted in the next paint event.
    */
    virtual void invalidate(const QRegion &r, bool translate = true) = 0;

    virtual int canvasWidth() const;
    virtual int canvasHeight() const;

    int zValue(QZionObject *obj) const;
    void setZValue(QZionObject *obj, int zValue);

    /*!
      Put the object \a obj in front of the reference object passed by \a ref.
      Both objects must be in the Canvas.
    */
    void stackAbove(QZionObject *obj, QZionObject *ref);

    /*!
      Put the object \a obj behind the reference object passed by \a ref.
      Both objects must be in the Canvas.
    */
    void stackBelow(QZionObject *obj, QZionObject *ref);

    /*!
      Bring an object to the front of the canvas.

    \sa setZValue(), lower(), stackAbove(), stackBelow()
    */
    void raise(QZionObject *obj);

    /*!
      Bring the object to the back of the canvas.

      \sa setZValue(), raise(), stackBelow(), stackAbove()
    */
    void lower(QZionObject *obj);

    /*!
      Add a QZionObject to this AbstractCanvas, removing it from a
      previous AbstractCanvas if needed.

      \sa removeObject()
    */
    void addObject(QZionObject *obj);

    /*!
      Remove a QZionObject from this AbstractCanvas.

      \sa addObject()
    */
    void removeObject(QZionObject *obj);

    virtual void updateChanges() = 0;

    virtual QSize size() const = 0;
    virtual void setSize(const QSize &size) = 0;

    /*!
      Translate a given point to local coordinates.

      Defaults to the identity functions.  QZionGroup should override it to
      translate the point according to its position.
    */
    virtual QPoint mapToLocal(const QPoint &pt) const;


    typedef void(QZionObject::*memberMouseAction)(QMouseEvent *);
    typedef void(QZionAbstractCanvas::*mouseAction)(QZionObject *,
                                                    QMouseEvent *);
    /*!
      \brief Handle Mouse Events: common code to all mouse events

      Function that contains common code to all mouseEvents. It receives the
      QMouseEvent to handle and what member functions must be called.

      \sa mouseMoveEvent(), mousePressEvent(), mouseReleaseEvent()
    */
    virtual void mouseEventHandler(QMouseEvent *e, memberMouseAction m_ma,
                                   mouseAction ma);


    /*!
      This function is called when an mouse Move event happens.

      You shouldn't need to call it manually, but you can to simulate events
      if needed.
    */
    virtual void mouseMoveEvent(QMouseEvent *e);

    /*!
      This function is called when an mouse Press event happens.

      You shouldn't need to call it manually, but you can to simulate events
      if needed.
    */
    virtual void mousePressEvent(QMouseEvent *e);

    /*!
      This function is called when an mouse Release event happens.

      You shouldn't need to call it manually, but you can to simulate events
      if needed.
    */
    virtual void mouseReleaseEvent(QMouseEvent *e);

    /*!
      This function is called when a wheel event happens.

      You shouldn't need to call it manually, but you can to simulate events
      if needed.
    */
    virtual void wheelEvent(QWheelEvent *e);

    /*!
      \internal
      Shortcut function for tracking children Move event.
    */
    virtual void mouseObjectMoveEvent(QZionObject *, QMouseEvent *) {}

    /*!
      \internal
      Shortcut function for tracking children Press event.
    */
    virtual void mouseObjectPressEvent(QZionObject *, QMouseEvent *) {}

    /*!
      \internal
      Shortcut function for tracking children Release event.
    */
    virtual void mouseObjectReleaseEvent(QZionObject *, QMouseEvent *) {}

    /*!
      \internal
      Shortcut function for tracking children Wheel event.
    */
    virtual void wheelObjectEvent(QZionObject *, QWheelEvent *) {}

protected:
    QZionAbstractCanvasPrivate *_QZionAbstractCanvas_data;

private:
    friend class QZionObjectPrivate;
    friend class QZionAbstractCanvasPrivate;
};

#endif
