#!/usr/bin/env python
# File created on 19 Mar 2011
from __future__ import division

__author__ = "Justin Kucyznski"
__copyright__ = "Copyright 2011, The QIIME Project"
__credits__ = ["Justin Kucyznski", "Jai Ram Rideout", "Greg Caporaso"]
__license__ = "GPL"
__version__ = "1.8.0"
__maintainer__ = "Justin Kucyznski"
__email__ = "justinak@gmail.com"

from os.path import exists

from cogent.util.unit_test import TestCase, main
from cogent.parse.tree import DndParser

from biom.parse import parse_biom_table
from biom.table import table_factory
from qiime.parse import parse_mapping_file
from qiime.util import (get_qiime_scripts_dir,
                        load_qiime_config,
                        get_tmp_filename,
                        get_qiime_temp_dir,
                        create_dir)
import qiime.simsam

import tempfile
import string
import random
import os
import shutil
import subprocess
import numpy


class SimsamTests(TestCase):

    def setUp(self):
        self.dirs_to_remove = []
        tmp_dir = get_qiime_temp_dir()
        self.test_out = get_tmp_filename(tmp_dir=tmp_dir,
                                         prefix='qiime_parallel_tests_',
                                         suffix='',
                                         result_constructor=str)
        self.dirs_to_remove.append(self.test_out)
        create_dir(self.test_out)
        
        self.map_f = map_lines.split('\n')
        self.otu_table = parse_biom_table(otu_table_lines.split('\n'))
        self.tutorial_map = tutorial_map.split('\n')
        self.tutorial_tree = DndParser(tutorial_tree)
        self.tutorial_otu_table = parse_biom_table(tutorial_otu_table.split('\n'))

    def tearDown(self):
        for d in self.dirs_to_remove:
            if os.path.exists(d):
                shutil.rmtree(d)

    def test_script(self):
        """ test the whole simsam script
        """
        qiime_config = load_qiime_config()
        tempdir = qiime_config['temp_dir'] or tempfile.gettempdir()
        maindir = os.path.join(tempdir,
         ''.join(random.choice(string.ascii_letters + string.digits) \
         for x in range(10)))


        os.makedirs(maindir)
        self.dirs_to_remove.append(maindir)
        otuf = os.path.join(maindir,'otuf')
        treef = os.path.join(maindir,'treef')

        otufh = open(otuf,'w')
        otufh.write(tutorial_otu_table)
        otufh.close()

        treefh = open(treef,'w')
        treefh.write(tutorial_tree)
        treefh.close()

        scripts_dir = get_qiime_scripts_dir()
        out_dir = os.path.join(maindir, 'simsam_out')
        cmd = os.path.join(scripts_dir,
                           'simsam.py -i %s -t %s -o %s -d .003 -n 3  ' %
                           (otuf, treef, out_dir))
        proc = subprocess.Popen(cmd,shell=True,
            stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        scriptout, scripterr = proc.communicate()

        if scriptout:
            raise RuntimeError('script returned stdout: ' + scriptout)
        if scripterr:
            raise RuntimeError('script returned stderr: ' + scripterr)

        num_replicates = 3 # ensure this matches cmd above

        res = open(os.path.join(out_dir, 'otuf_n%d_d0.003.biom' % num_replicates), 'U')
        orig_table = parse_biom_table(open(otuf,'U'))
        res_table = parse_biom_table(res)

        # 3 samples per input sample
        self.assertEqual(len(res_table.SampleIds),num_replicates*len(orig_table.SampleIds))

        # sample_ids have correct naming and order
        for i in range(len(orig_table.SampleIds)):
            for j in range(num_replicates):
                exp = orig_table.SampleIds[i] +'.'+str(j)
                self.assertEqual(res_table.SampleIds[i*num_replicates+j],exp)

        # same total sequences in each replicate sample
        num_orig_samples = len(orig_table.SampleIds)
        orig_sams = orig_table.iterSampleData()
        res_sams = res_table.iterSampleData()
        for i in range(num_orig_samples):
            orig_sam = orig_sams.next()
            for j in range(num_replicates):
                res_sam = res_sams.next()
                self.assertEqual(res_sam.sum(), orig_sam.sum())
        # would be nice to test that result otu table doesn't match input,
        # but not sure how probable that is, and don't want stochastic failures

    def test_script_nochange(self):
        """ simsam script with 0 distance should just replicate input samples
        """
        qiime_config = load_qiime_config()
        tempdir = qiime_config['temp_dir'] or tempfile.gettempdir()
        maindir = os.path.join(tempdir,
         ''.join(random.choice(string.ascii_letters + string.digits) \
         for x in range(10)))


        os.makedirs(maindir)
        self.dirs_to_remove.append(maindir)
        otuf = os.path.join(maindir,'otuf')
        treef = os.path.join(maindir,'treef')

        otufh = open(otuf,'w')
        otufh.write(tutorial_otu_table)
        otufh.close()

        treefh = open(treef,'w')
        treefh.write(tutorial_tree)
        treefh.close()

        scripts_dir = get_qiime_scripts_dir()
        out_dir = os.path.join(maindir, 'simsam_out')
        cmd = os.path.join(scripts_dir,
                           'simsam.py -i %s -t %s -o %s -d 0 -n 3  ' %
                           (otuf, treef, out_dir))

        proc = subprocess.Popen(cmd,shell=True, 
            stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        scriptout, scripterr = proc.communicate()

        if scriptout:
            raise RuntimeError('script returned stdout: ' + scriptout)
        if scripterr:
            raise RuntimeError('script returned stderr: ' + scripterr)

        num_replicates = 3 # ensure this matches cmd above
        res = open(os.path.join(out_dir, 'otuf_n%d_d0.0.biom' % num_replicates), 'U')
        orig_table = parse_biom_table(open(otuf,'U'))
        res_table = parse_biom_table(res)

        # 3 samples per input sample
        self.assertEqual(len(res_table.SampleIds),num_replicates*len(orig_table.SampleIds))

        # sample_ids have correct naming and order
        for i in range(len(orig_table.SampleIds)):
            for j in range(num_replicates):
                exp = orig_table.SampleIds[i] +'.'+str(j)
                self.assertEqual(res_table.SampleIds[i*num_replicates+j],exp)

        # same otu ids
        self.assertEqual(res_table.ObservationIds, orig_table.ObservationIds)

        # same otu table, just replicated thrice
        # note this requires the same sorting of otus, input is correct sorting
        num_orig_samples = len(orig_table.SampleIds)
        orig_sams = orig_table.iterSampleData()
        res_sams = res_table.iterSampleData()
        for i in range(num_orig_samples):
            orig_sam = orig_sams.next()
            for j in range(num_replicates):
                res_sam = res_sams.next()
                self.assertEqual(res_sam, orig_sam)

    def test_sim_otu_table(self):
        """ simulated otu table should be right order, number of seqs

        tree looks like:
                  /-A
                 |
        ---------|--B
                 |
                 |          /-C
                  \--------|
                            \-D
        """
        sample_ids = ['samB','samA']
        otu_ids = ['C','A']
        otu_mtx = numpy.array([ [3,9],
                                [5,0],
                                ])
        otu_metadata = [{'tax':'otu_C is cool'},{'tax':''}]
        tree = DndParser("(A:0.1,B:0.2,(C:0.3,D:0.4):0.5);")
        num_replicates = 3
        dissimilarity = 0.15
        rich_table = table_factory(otu_mtx,sample_ids,otu_ids,
            observation_metadata=otu_metadata)
        res_sam_names, res_otus, res_otu_mtx, res_otu_metadata = \
         qiime.simsam.sim_otu_table(sample_ids, otu_ids, rich_table.iterSamples(), otu_metadata,
         tree, num_replicates, dissimilarity)
        
        # dissim is too small to change otu C, it should always be there
        # with at least original # seqs, maybe more
        c_index = res_otus.index('C')
        c_row = res_otu_mtx[c_index]
        self.assertEqual(res_otu_metadata[c_index],{'tax':'otu_C is cool'})
        self.assertGreaterThan(c_row,[2,2,2,8,8,8])

        # order of samples should remain the same as input,
        # and eash replicate sample
        # should have same number or sequences as input
        for i in range(len(sample_ids)):
            for j in range(num_replicates):
                self.assertEqual(otu_mtx[:,i].sum(),
                 res_otu_mtx[:,num_replicates*i+j].sum())

    def test_sim_otu_table_new_otus(self):
        """Test large dissim to obtain OTUs that weren't in original table."""
        sample_ids = ['samB','samA']
        otu_ids = ['C','A']
        otu_mtx = numpy.array([ [3,9],
                                [5,0],
                                ])
        otu_metadata = [{'tax':'otu_C is cool'},{'tax':''}]
        tree = DndParser("(A:0.1,B:0.2,(C:0.3,D:0.4):0.5);")
        num_replicates = 3

        # Huge dissimilarity to ensure we get new OTUs.
        dissimilarity = 100000

        rich_table = table_factory(otu_mtx,sample_ids,otu_ids,
            observation_metadata=otu_metadata)

        otu_id_results = []
        otu_md_results = []
        for i in range(1000):
            res_sam_names, res_otus, res_otu_mtx, res_otu_metadata = \
                    qiime.simsam.sim_otu_table(sample_ids, otu_ids,
                            rich_table.iterSamples(), otu_metadata, tree,
                            num_replicates, dissimilarity)
            otu_id_results.extend(res_otus)
            otu_md_results.extend(res_otu_metadata)

        # We should see all OTUs show up at least once.
        self.assertContains(otu_id_results, 'A')
        self.assertContains(otu_id_results, 'B')
        self.assertContains(otu_id_results, 'C')
        self.assertContains(otu_id_results, 'D')

        # We should see at least one blank metadata entry since A and B are not
        # in the original table.
        self.assertContains(otu_md_results, None)

    def test_get_new_otu_id_small(self):
        """ small dissim should return old tip id"""
        tree = DndParser("(A:0.1,B:0.2,(C:0.3,D:0.4):0.5);")
        res = qiime.simsam.get_new_otu_id(old_otu_id='A', tree=tree, dissim=.05)
        self.assertEqual(res,'A')

    def test_get_new_otu_id_large(self):
        """  w/ large dissim, should at least sometimes return other tip"""
        tree = DndParser("(A:0.1,B:0.2,(C:0.3,D:0.4):0.5);")
        results = []
        for i in range(1000):
            results.append(qiime.simsam.get_new_otu_id(old_otu_id='D', 
            tree=tree, dissim=.6))
        self.assertContains(results,'C')
        self.assertContains(results,'D')
        self.assertNotContains(results,'A')
        self.assertNotContains(results,'B')

    def test_combine_sample_dicts(self):
        """ combining sample dicts should give correct otu table and sorting
        """
        d1 = {'otu2':0,'otu1':3}
        d2 = {'otu4':5}
        d3 = {}
        res_otu_mtx, res_otu_ids = qiime.simsam.combine_sample_dicts([d1,d2,d3])
        exp_otu_ids = ['otu1','otu2','otu4']
        exp_otu_mtx = numpy.array([ [3,0,0],
                                    [0,0,0],
                                    [0,5,0],
                                    ])
        self.assertEqual(res_otu_ids, exp_otu_ids)
        self.assertEqual(res_otu_mtx, exp_otu_mtx)

    def test_create_replicated_mapping_file(self):
        """Test creating replicate samples in a mapping file."""
        # 3 replicates, with two extra samples in the mapping file.
        obs = qiime.simsam.create_replicated_mapping_file(self.map_f, 3,
                self.otu_table.SampleIds)
        self.assertEqual(obs, exp_rep_map_lines)

        # Must specify at least one replicate.
        self.assertRaises(ValueError,
                qiime.simsam.create_replicated_mapping_file, self.map_f, 0,
                self.otu_table.SampleIds)

    def test_simsam_range_correct_number_of_output(self):
        """simsam_range yields correct number of output tables
        """
        actual = qiime.simsam.simsam_range(self.tutorial_otu_table,self.tutorial_tree,
                              [1],[0.1],self.tutorial_map)
        self.assertEqual(len(list(actual)),1)
        actual = qiime.simsam.simsam_range(self.tutorial_otu_table,self.tutorial_tree,
                              [1,2],[0.1],self.tutorial_map)
        self.assertEqual(len(list(actual)),2)
        actual = qiime.simsam.simsam_range(self.tutorial_otu_table,self.tutorial_tree,
                              [2],[0.1,0.001],self.tutorial_map)
        self.assertEqual(len(list(actual)),2)
        actual = qiime.simsam.simsam_range(self.tutorial_otu_table,self.tutorial_tree,
                              [1,2],[0.1,0.001],self.tutorial_map)
        self.assertEqual(len(list(actual)),4)
    
    def test_simsam_range_correct_size_of_output(self):
        """simsam_range yields tables with correct number of samples"""
        actual = qiime.simsam.simsam_range(self.tutorial_otu_table,self.tutorial_tree,
                              [1],[0.1],self.tutorial_map)
        actual = list(actual)
        self.assertEqual(len(actual[0][0].SampleIds),len(self.tutorial_otu_table.SampleIds))
        
        actual = qiime.simsam.simsam_range(self.tutorial_otu_table,self.tutorial_tree,
                              [2],[0.1],self.tutorial_map)
        actual = list(actual)
        self.assertEqual(len(actual[0][0].SampleIds),2*len(self.tutorial_otu_table.SampleIds))
        
        actual = qiime.simsam.simsam_range(self.tutorial_otu_table,self.tutorial_tree,
                              [4],[0.1],self.tutorial_map)
        actual = list(actual)
        self.assertEqual(len(actual[0][0].SampleIds),4*len(self.tutorial_otu_table.SampleIds))

    def test_simsam_range_functions_without_mapping_file(self):
        """simsam_range yields correct number of output tables
        """
        actual = qiime.simsam.simsam_range(self.tutorial_otu_table,self.tutorial_tree,[1],[0.1])
        self.assertEqual(len(list(actual)),1)
    
    def test_simsam_range_to_files(self):
        """simsam_range_to_files functions as expected """
        qiime.simsam.simsam_range_to_files(self.tutorial_otu_table,
                                        self.tutorial_tree,
                                        [2],
                                        [0.1],
                                        output_dir=self.test_out,
                                        mapping_f=self.tutorial_map,
                                        output_table_basename="hello",
                                        output_map_basename="world")
        self.assertTrue(exists('%s/hello_n2_d0.1.biom' % self.test_out))
        self.assertTrue(exists('%s/world_n2_d0.1.txt' % self.test_out))
        
        # confirm same sample ids in table and mapping file
        t = parse_biom_table(open('%s/hello_n2_d0.1.biom' % self.test_out))
        d, _, _ = \
         parse_mapping_file(open('%s/world_n2_d0.1.txt' % self.test_out))
        mapping_sample_ids = [e[0] for e in d]
        self.assertEqualItems(t.SampleIds,mapping_sample_ids)


map_lines = """#SampleID\tTreatment\tDescription
# Some epic comment
S1\tControl\tControl1
S2\tControl\tControl2
S3\tControl\tControl3
S4\tFast\tFast4
S5\tFast\tFast5"""

exp_rep_map_lines = """#SampleID\tTreatment\tDescription
# Some epic comment
S1.0\tControl\tControl1
S1.1\tControl\tControl1
S1.2\tControl\tControl1
S2.0\tControl\tControl2
S2.1\tControl\tControl2
S2.2\tControl\tControl2
S3.0\tControl\tControl3
S3.1\tControl\tControl3
S3.2\tControl\tControl3"""

otu_table_lines = """{"id": "None","format": "Biological Observation Matrix 1.0.0","format_url": "http://biom-format.org","type": "OTU table","generated_by": "BIOM-Format 1.1.2","date": "2013-03-27T13:59:38.949014","matrix_type": "sparse","matrix_element_type": "float","shape": [4, 3],     "data": [[0,0,1.0],[0,2,4.0],[1,1,5.0],[1,2,7.0],[2,0,1.0],[2,1,1.0],[2,2,9.0],[3,0,6.0],[3,1,10.0],[3,2,8.0]],"rows": [{"id": "OTU0", "metadata": null},{"id": "OTU1", "metadata": null},{"id": "OTU2", "metadata": null},{"id": "OTU3", "metadata": null}],"columns": [{"id": "S1", "metadata": null},{"id": "S2", "metadata": null},{"id": "S3", "metadata": null}]}"""

tutorial_tree = """(((((381:0.0213,(214:0.03728,253:0.00015)0.945:0.03224)0.763:0.00483,((269:0.02693,(231:0.00509,(105:0.01425,141:0.02641)0.846:0.01405)0.428:0.00519)0.622:0.00014,404:0.00524)0.795:0.00514)0.773:0.00508,(131:0.00518,(33:0.01631,((284:0.00828,(176:0.03098,388:0.01236)0.901:0.02175)0.885:0.01273,52:0.01046)0.743:0.00498)0.924:0.01603)0.779:0.00511)0.772:0.00014,153:0.00507)0.753:0.00602,(223:0.03237,(172:0.01733,81:0.03834)0.224:0.00414)0.845:0.01076,(136:0.00627,((((265:0.01557,200:0.00517)0.674:0.00014,((204:0.00015,(339:0.01613,(322:0.01633,268:0.01643)0.569:0.0107)0.885:0.00016)0.840:0.00527,((((((((280:0.02348,(395:0.00015,(48:0.03014,((30:0.02665,316:0.01921)0.813:0.01152,215:0.0242)0.850:0.01191)0.320:0.00016)0.912:0.02431)0.694:0.01482,(115:0.01526,364:0.08211)0.879:0.03637)0.677:0.03567,((((((162:0.06933,59:0.02113)0.991:0.08563,(308:0.02061,43:0.03488)0.894:0.04949)0.911:0.05006,(((344:0.00015,(146:0.00015,377:0.01634)0.924:0.0108)0.918:0.01069,((201:0.011,240:0.04792)1.000:0.00015,(61:0.00015,96:0.00523)0.781:0.00514)0.828:0.01056)0.809:0.00016,196:0.04505)0.213:0.00014)0.650:0.00529,(((161:0.01191,(390:0.04307,37:0.03893)0.933:0.03396)0.814:0.01401,68:0.04946)0.953:0.03303,((341:0.01127,393:0.02765)0.941:0.02238,(82:0.01112,(350:0.01141,(156:0.01636,356:0.00015)0.863:0.02214)0.946:0.02475)0.748:0.00565)0.761:0.00968)0.748:0.00836)0.927:0.0224,271:0.05902)0.753:0.00511,(((((((217:0.03796,379:0.00016)0.973:0.05805,(299:0.08963,(382:0.06426,((317:0.00016,((205:0.00532,264:0.03867)0.939:0.01605,(194:0.03374,(32:0.01052,(348:0.02212,157:0.02743)1.000:0.00014)0.868:0.02793)0.745:0.00531)0.336:0.01061)0.789:0.00604,334:0.02104)0.598:0.01527)0.687:0.00354)0.836:0.01564)0.811:0.01617,(292:0.06237,84:0.02159)0.934:0.04776)0.864:0.02103,301:0.06716)0.698:0.0046,272:0.00539)0.809:0.0115,88:0.05965)0.860:0.01208,(276:0.01065,279:0.03443)0.891:0.01124)0.090:0.00014)0.924:0.03938)0.953:0.05227,281:0.02828)0.691:0.00622,25:0.01213)0.727:0.00397,((261:0.01613,((147:0.01555,20:0.00016)0.967:0.02125,(107:0.01089,349:0.03426)0.757:0.00478)0.750:0.00518)0.799:0.0052,(259:0.01616,63:0.01053)0.764:0.00523)0.792:0.00511)1.000:0.00016,(72:0.05949,(1:0.01425,67:0.0377)0.751:0.00762)0.867:0.01609)0.807:0.00507,((49:0.01645,116:0.01633)0.736:0.00514,(398:0.00515,(((180:0.04458,99:0.0328)0.913:0.02521,(((410:0.05589,(((150:0.04425,(170:0.03163,((250:0.00693,331:0.00435)1.000:0.10845,357:0.01319)0.850:0.0225)0.879:0.02887)0.749:0.00795,(((((23:0.00919,248:0.08024)0.405:0.03691,(358:0.05635,369:0.07223)0.978:0.09469)0.888:0.05975,(234:0.07249,8:0.00016)0.712:0.01829)0.976:0.07916,(((275:0.094,(((114:0.0269,302:0.02202)0.985:0.06964,(213:0.06889,42:0.03436)0.415:0.01928)0.795:0.02064,((110:0.05188,342:0.01457)0.967:0.08524,((123:0.02756,343:0.0481)0.800:0.01738,((298:0.03283,(124:0.02507,6:0.03351)0.781:0.01076)0.939:0.03194,309:0.04124)0.820:0.01321)0.985:0.0961)0.928:0.06559)0.902:0.03886)0.684:0.03217,373:0.06838)0.909:0.03592,((290:0.02673,380:0.00015)1.000:0.16099,(((90:0.09952,192:0.10171)0.679:0.01316,(326:0.03972,45:0.09053)0.965:0.05309)0.115:0.00014,(375:0.00015,(221:0.00071,278:0.05255)1.000:0.08313)1.000:0.10921)0.623:0.0222)0.892:0.03509)0.465:0.00015)0.980:0.05443,(((306:0.08813,385:0.14214)0.269:0.00862,((256:0.01776,(273:0.07543,69:0.01333)0.591:0.02343)0.883:0.02549,((132:0.02365,219:0.01597)0.897:0.02388,(100:0.01243,50:0.0237)0.226:0.01766)0.961:0.04348)0.848:0.01577)0.998:0.08323,(241:0.23207,(130:0.24778,(53:0.12887,(129:0.07692,318:0.01288)0.900:0.04845)0.817:0.02143)0.888:0.05464)0.657:0.01537)0.822:0.01876)0.828:0.01549)0.773:0.01019,((98:0.12681,((148:0.0294,391:0.00571)0.989:0.07803,(389:0.10107,(252:0.00014,362:0.01104)0.964:0.06682)0.834:0.03217)0.762:0.0152)0.524:0.0181,(0:0.0483,(135:0.01151,(300:0.0175,(274:0.04561,((((166:0.02935,355:0.00015)0.833:0.00565,41:0.00014)0.807:0.00586,(226:0.01038,92:0.0044)0.792:0.00425)0.961:0.03236,((360:0.01752,7:0.0182)0.748:0.00495,(368:0.02316,288:0.01783)0.759:0.00622)0.707:0.00573)0.841:0.00015)0.949:0.02275)0.745:0.00559)0.855:0.02344)0.876:0.03532)0.885:0.02567)0.752:0.00645)0.782:0.00969,(((((((178:0.01576,(230:0.02704,64:0.02146)0.869:0.0108)0.809:0.01014,((122:0.00448,354:0.0225)0.855:0.01127,(333:0.01086,406:0.01648)0.748:0.00433)0.789:0.00624)0.171:0.00516,((416:0.04298,(400:0.01045,74:0.01051)0.923:0.00014)0.862:0.02166,(307:0.04097,(260:0.03574,335:0.0434)0.747:0.00875)0.916:0.02837)0.843:0.00987)0.804:0.00016,((237:0.09447,((370:0.01631,(319:0.04803,(60:0.01986,405:0.01742)0.560:0.01574)0.898:0.01971)0.918:0.01584,(384:0.02116,(245:0.01047,(177:0.0051,(183:0.03226,413:0.00014)0.826:0.00518)0.777:0.00501)0.923:0.0158)0.622:0.00016)0.685:0.00099)0.224:0.02406,((22:0.03142,5:0.06696)0.870:0.03448,47:0.0347)0.763:0.01052)0.847:0.01209)0.743:0.00534,((((62:0.00137,(121:0.00016,78:0.04376)1.000:0.10609)0.942:0.0378,(311:0.05626,407:0.06902)0.944:0.04614)0.703:0.00608,(((188:0.01993,202:0.02611)0.914:0.02118,(328:0.0273,337:0.00015)0.815:0.01019)0.852:0.01169,(330:0.03441,((386:0.13035,(392:0.00544,(321:0.02191,4:0.01061)0.763:0.0052)0.932:0.00014)0.671:0.01096,145:0.01556)0.829:0.01073)0.735:0.00529)0.840:0.01052)0.849:0.01531,(262:0.0683,((310:0.05551,((83:0.01296,(127:0.01909,212:0.01393)0.090:0.00499)0.876:0.01352,(104:0.00014,171:0.01061)0.895:0.01717)0.877:0.02683)0.940:0.03929,(119:0.0152,179:0.00197)0.889:0.02843)0.066:0.01551)0.839:0.01374)0.820:0.01069)0.869:0.01061,(((293:0.01741,168:0.04514)0.046:0.01491,345:0.03334)0.248:0.01629,(31:0.04727,97:0.04999)0.915:0.03556)0.811:0.01631)0.010:0.00016,(((94:0.0671,(108:0.00014,229:0.06991)0.630:0.01827)0.982:0.06031,(143:0.02201,((((((198:0.02745,(140:0.14724,75:0.02831)0.817:0.0209)0.851:0.01902,(((282:0.06783,54:0.00015)0.952:0.03641,((313:0.03746,80:0.00524)0.872:0.0215,2:0.07468)0.862:0.02589)0.916:0.03914,((367:0.0099,(((128:0.0425,((111:0.06727,11:0.00495)0.974:0.02953,283:0.02606)0.504:0.00357)0.862:0.02044,(289:0.04546,(399:0.00319,((((152:0.00014,19:0.06307)0.992:0.03752,154:0.00016)0.786:0.00014,134:0.06945)0.997:0.06109,51:0.00014)0.994:0.04556)0.353:0.00583)0.482:0.00828)0.933:0.03536,112:0.07957)0.734:0.00733)0.962:0.08492,403:0.10375)0.869:0.0525)0.894:0.03949)0.645:0.00925,((((287:0.00534,15:0.05518)0.920:0.03189,(((304:0.00508,409:0.00015)0.991:0.00014,(120:0.00015,(57:0.04309,56:0.0156)0.759:0.00015)0.902:0.01019)0.339:0.01644,173:0.094)0.787:0.01131)1.000:0.07731,(236:0.00625,((26:0.04569,(((351:0.005,(27:0.03624,(137:0.01569,(314:0.00015,408:0.03277)0.991:0.03257)0.806:0.00498)0.851:0.00588)0.928:0.01791,((133:0.04374,(227:0.00527,(412:0.00014,(175:0.00507,((95:0.01566,210:0.00014)0.438:0.01045,191:0.00016)0.781:0.00518)0.815:0.00508)0.859:0.01021)0.745:0.00667)0.735:0.01956,((((12:0.01588,415:0.01701)0.121:0.03139,(73:0.04886,(17:0.00016,(46:0.02083,378:0.01021)0.886:0.01027)0.785:0.019)0.719:0.02118)0.774:0.01959,329:0.01522)0.777:0.01121,(((286:0.00722,(394:0.01596,(372:0.00015,225:0.0446)0.884:0.0109)0.929:0.02558)0.584:0.00985,218:0.02283)0.888:0.01478,159:0.02121)0.739:0.00866)0.851:0.01129)0.728:0.00602)0.866:0.01998,93:0.04869)0.604:0.00297)0.648:0.01633,199:0.06704)0.788:0.01956)0.371:0.01052)0.827:0.01491,((244:0.0262,(126:0.00015,163:0.03192)0.984:0.04377)0.817:0.01306,((216:0.00014,(86:0.02257,(21:0.01127,34:0.01066)0.859:0.01088)0.622:0.017)0.998:0.19434,(233:0.00244,(182:0.01898,(239:0.02877,267:0.00015)0.839:0.01438)0.999:0.09419)0.975:0.15234)0.877:0.07457)0.893:0.0244)0.821:0.02013)0.998:0.10422,(195:0.10508,((249:0.0368,(336:0.04596,((263:0.02407,(277:0.01295,190:0.03788)0.823:0.01671)0.698:0.0068,197:0.01756)0.309:0.01631)0.860:0.01866)0.926:0.02656,(303:0.04293,(113:0.04423,347:0.04295)0.930:0.03972)0.885:0.02484)0.701:0.00015)0.902:0.03629)0.841:0.02905,(246:0.00014,(125:0.03009,184:0.0229)0.998:0.07478)0.999:0.10301)0.936:0.04978,((247:0.04204,((((((238:0.01393,(109:0.01081,39:0.02762)0.769:0.00519)0.758:0.00702,(257:0.01539,85:0.07408)0.746:0.00558)0.755:0.01039,(363:0.04294,155:0.00015)0.943:0.02426)0.894:0.01745,266:0.00586)0.948:0.03346,55:0.02705)0.739:0.00453,203:0.00015)0.855:0.02077)0.995:0.07638,327:0.00745)0.921:0.03692)0.553:0.01549)0.970:0.05544)0.858:0.02855,338:0.08163)0.892:0.03304)0.759:0.00673)0.945:0.02495,((((((((102:0.04317,36:0.02415)0.964:0.03758,65:0.00505)0.822:0.01078,366:0.00016)0.811:0.01537,(315:0.01071,((151:0.0,160:0.0):0.00016,340:0.00014)0.842:0.01037)0.951:0.02712)0.724:0.00057,(185:0.04527,(207:0.01304,76:0.00341)0.949:0.03474)0.845:0.0196)0.871:0.0106,(371:0.02805,(164:0.0104,242:0.02179)0.758:0.0052)0.771:0.00538)0.841:0.01097,174:0.13953)0.831:0.01033,(144:0.01866,(3:0.01578,312:0.00015)0.785:0.00532)0.780:0.00615)0.752:0.00572)0.667:0.00244)0.268:0.00339,((101:0.04199,77:0.00334)0.965:0.0345,((14:0.01106,294:0.00502)0.891:0.01811,(285:0.01062,397:0.01076)0.758:0.00896)0.163:0.01034)0.850:0.01331)0.563:0.00537)0.800:0.00519)0.930:0.00016)0.759:0.01023)1.000:0.00014)0.850:0.00015,(243:0.03373,220:0.01032)0.888:0.011)0.540:0.00014,(189:0.02629,(((139:0.0155,186:0.01757)0.420:0.01444,(((((((165:0.0059,58:0.03297)0.779:0.02132,((222:0.01678,(323:0.02243,44:0.04081)0.819:0.01102)0.063:0.00015,(106:0.03989,149:0.02047)0.775:0.01298)0.706:0.0074)0.957:0.03281,((((258:0.04247,87:0.0123)0.500:0.01067,235:0.00735)0.645:0.00296,208:0.00505)1.000:0.00015,((18:0.00454,(((10:0.04233,(414:0.00016,(142:0.01127,66:0.03479)0.756:0.00498)0.726:0.00685)0.486:0.01639,181:0.00014)0.784:0.00501,(167:0.01463,(320:0.00885,402:0.00881)0.791:0.00014)0.839:0.01499)0.773:0.00524)0.893:0.01079,(169:0.00517,(295:0.01586,297:0.03792)0.262:0.00016)0.778:0.00521)0.818:0.00528)0.764:0.01062)0.767:0.00486,70:0.00512)0.766:0.00495,(((332:0.00016,((325:0.01591,(383:0.00014,(361:0.01642,(138:0.04133,(158:0.0036,224:0.00657)0.840:0.01972)0.769:0.00881)0.777:0.00496)0.882:0.01036)0.752:0.00492,(24:0.03974,((((254:0.00541,(251:0.00015,(324:0.02187,((117:0.0052,(374:0.03165,270:0.02362)0.731:0.00708)0.791:0.00525,13:0.01621)0.757:0.00511)0.607:0.01283)0.889:0.0192)0.852:0.01583,305:0.01647)0.948:0.00015,211:0.00015)0.419:0.00016,(103:0.01686,209:0.05269)0.861:0.01595)0.937:0.01635)0.756:0.00523)0.878:0.01048)0.776:0.00238,(365:0.03251,((38:0.04434,79:0.00014)0.758:0.00016,(296:0.043,9:0.00518)0.693:0.0162)0.508:0.00805)0.766:0.00767)0.764:0.00313,(((359:0.02181,(16:0.04469,(232:0.01621,(118:0.03421,(29:0.01612,353:0.01494)0.293:0.01034)0.864:0.01326)0.747:0.01394)0.724:0.0072)0.911:0.01681,387:0.02755)0.761:0.00523,(346:0.01957,(376:0.04072,71:0.0547)0.829:0.0181)0.750:0.00673)0.823:0.01037)0.774:0.0054)0.789:0.005,(((228:0.00529,((401:0.02214,((187:0.00532,411:0.00526)0.801:0.00583,((89:0.027,193:0.00014)0.787:0.00524,91:0.01618)0.743:0.0045)0.548:0.00548)0.825:0.016,40:0.02807)0.778:0.00992)0.824:0.01011,255:0.05012)0.966:0.00014,(352:0.01585,396:0.00014)0.784:0.02134)0.880:0.0107)0.901:0.0194,(35:0.0209,(206:0.00836,291:0.06414)0.439:0.00793)0.753:0.00846)0.763:0.00968)0.942:0.02851,28:0.0208)0.742:0.01057)0.781:0.00811)0.802:0.02029)0.750:0.01578);"""

tutorial_otu_table = """{"rows": [{"id": "0", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "1", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "2", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Bacillales", "f__Staphylococcaceae"]}}, {"id": "3", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "4", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "5", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "6", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "7", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Lactobacillaceae"]}}, {"id": "8", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "9", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "10", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "11", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "12", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "13", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "14", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Streptococcaceae"]}}, {"id": "15", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "16", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "17", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "18", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "19", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "20", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "21", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "22", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "23", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Proteobacteria", "c__Deltaproteobacteria", "o__Desulfovibrionales", "f__Desulfovibrionaceae"]}}, {"id": "24", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "25", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "26", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "27", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "28", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__TM7", "c__TM7-3", "o__CW040", "f__F16"]}}, {"id": "29", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "30", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "31", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "32", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "33", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "34", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "35", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "36", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "37", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "38", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "39", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "40", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "41", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "42", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "43", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "44", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "45", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "46", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "47", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "48", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "49", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Deferribacteres", "c__Deferribacteres", "o__Deferribacterales", "f__Deferribacteraceae"]}}, {"id": "50", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "51", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "52", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "53", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "54", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "55", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "56", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "57", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "58", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "59", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "60", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "61", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "62", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "63", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "64", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "65", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "66", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "67", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "68", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "69", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "70", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "71", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "72", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "73", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "74", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "75", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "76", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "77", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "78", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "79", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "80", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "81", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "82", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "83", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "84", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "85", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "86", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "87", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "88", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "89", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "90", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "91", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "92", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "93", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "94", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "95", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Mollicutes", "o__RF39", "f__"]}}, {"id": "96", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "97", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "98", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "99", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "100", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "101", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "102", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "103", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "104", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "105", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "106", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "107", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "108", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "109", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Lactobacillaceae"]}}, {"id": "110", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "111", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "112", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "113", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "114", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "115", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "116", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Deferribacteres", "c__Deferribacteres", "o__Deferribacterales", "f__Deferribacteraceae"]}}, {"id": "117", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "118", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "119", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "120", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "121", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "122", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "123", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "124", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "125", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "126", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "127", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "128", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Mollicutes", "o__RF39", "f__"]}}, {"id": "129", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "130", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "131", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "132", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Peptococcaceae"]}}, {"id": "133", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "134", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "135", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "136", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "137", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "138", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "139", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "140", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "141", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "142", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "143", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "144", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "145", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "146", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "147", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "148", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "149", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "150", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "151", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "152", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "153", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "154", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales"]}}, {"id": "155", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "156", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "157", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "158", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "159", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "160", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "161", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "162", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Lactobacillaceae"]}}, {"id": "163", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "164", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "165", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "166", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "167", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "168", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "169", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "170", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "171", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "172", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "173", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "174", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "175", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "176", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "177", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Clostridiaceae"]}}, {"id": "178", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "179", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "180", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "181", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "182", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "183", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "184", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "185", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "186", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "187", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "188", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "189", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "190", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "191", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "192", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "193", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "194", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "195", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "196", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "197", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "198", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "199", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Peptococcaceae"]}}, {"id": "200", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "201", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "202", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "203", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "204", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "205", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "206", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "207", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "208", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "209", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "210", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "211", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "212", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "213", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "214", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "215", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "216", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "217", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "218", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "219", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "220", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "221", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "222", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "223", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "224", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "225", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "226", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "227", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "228", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "229", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "230", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "231", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "232", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "233", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "234", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "235", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "236", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "237", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "238", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Lactobacillaceae"]}}, {"id": "239", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "240", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "241", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "242", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "243", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "244", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "245", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "246", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "247", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "248", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "249", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "250", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "251", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "252", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "253", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "254", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "255", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "256", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "257", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "258", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "259", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "260", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "261", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "262", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "263", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "264", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "265", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "266", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "267", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "268", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Proteobacteria", "c__Deltaproteobacteria", "o__Desulfovibrionales", "f__Desulfovibrionaceae"]}}, {"id": "269", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "270", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "271", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "272", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "273", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "274", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "275", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "276", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "277", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "278", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "279", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "280", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "281", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "282", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "283", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "284", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "285", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "286", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Proteobacteria", "c__Deltaproteobacteria", "o__Desulfovibrionales", "f__Desulfovibrionaceae"]}}, {"id": "287", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "288", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "289", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "290", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "291", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "292", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "293", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "294", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "295", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "296", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "297", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Bacillales", "f__Staphylococcaceae"]}}, {"id": "298", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "299", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "300", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "301", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "302", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "303", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "304", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "305", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "306", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "307", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "308", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Prevotellaceae"]}}, {"id": "309", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "310", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "311", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "312", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "313", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "314", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "315", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "316", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "317", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Proteobacteria", "c__Epsilonproteobacteria", "o__Campylobacterales", "f__Helicobacteraceae"]}}, {"id": "318", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "319", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "320", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "321", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "322", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "323", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "324", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "325", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "326", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "327", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "328", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "329", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "330", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "331", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "332", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "333", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "334", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "335", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "336", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "337", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "338", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "339", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "340", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "341", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "342", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "343", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "344", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "345", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "346", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "347", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "348", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "349", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "350", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "351", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "352", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "353", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes"]}}, {"id": "354", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "355", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "356", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "357", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "358", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "359", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Clostridiales Family XIII. Incertae Sedis"]}}, {"id": "360", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "361", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "362", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "363", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "364", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "365", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "366", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "367", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "368", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "369", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "370", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "371", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "372", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "373", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "374", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "375", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "376", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "377", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "378", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "379", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "380", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "381", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "382", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "383", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "384", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "385", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "386", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "387", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "388", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "389", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "390", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "391", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "392", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "393", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "394", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "395", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "396", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "397", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "398", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "399", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "400", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "401", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "402", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "403", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "404", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "405", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Clostridiaceae"]}}, {"id": "406", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "407", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Turicibacterales", "f__Turicibacteraceae"]}}, {"id": "408", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "409", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "410", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "411", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "412", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "413", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "414", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "415", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "416", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}], "format": "Biological Observation Matrix v0.9", "data": [[0, 6, 2.0], [0, 7, 1.0], [1, 8, 1.0], [2, 8, 1.0], [3, 3, 1.0], [4, 0, 1.0], [5, 1, 1.0], [5, 2, 1.0], [5, 4, 1.0], [6, 3, 1.0], [6, 7, 1.0], [7, 4, 1.0], [8, 0, 1.0], [8, 2, 1.0], [8, 7, 1.0], [8, 8, 1.0], [9, 3, 1.0], [10, 0, 1.0], [10, 1, 1.0], [11, 0, 2.0], [11, 1, 1.0], [12, 2, 1.0], [12, 7, 1.0], [13, 7, 1.0], [14, 5, 1.0], [15, 2, 1.0], [16, 6, 1.0], [17, 6, 1.0], [18, 6, 1.0], [19, 0, 1.0], [19, 3, 1.0], [20, 5, 1.0], [21, 1, 1.0], [22, 5, 2.0], [23, 7, 1.0], [24, 3, 1.0], [24, 7, 1.0], [25, 7, 1.0], [26, 4, 1.0], [27, 6, 1.0], [28, 6, 2.0], [29, 6, 1.0], [30, 7, 1.0], [31, 3, 3.0], [32, 5, 1.0], [33, 2, 1.0], [34, 7, 1.0], [35, 5, 1.0], [36, 5, 2.0], [37, 0, 1.0], [37, 2, 1.0], [38, 1, 1.0], [38, 3, 1.0], [38, 4, 1.0], [38, 8, 3.0], [39, 5, 5.0], [40, 3, 1.0], [40, 5, 2.0], [41, 5, 1.0], [42, 5, 1.0], [43, 0, 1.0], [44, 5, 2.0], [45, 4, 1.0], [46, 5, 1.0], [47, 0, 16.0], [47, 4, 12.0], [48, 0, 6.0], [48, 2, 4.0], [48, 4, 2.0], [49, 5, 3.0], [49, 6, 5.0], [49, 7, 2.0], [49, 8, 6.0], [50, 3, 1.0], [51, 3, 1.0], [52, 4, 1.0], [53, 0, 1.0], [54, 4, 1.0], [55, 4, 1.0], [56, 5, 1.0], [57, 8, 1.0], [58, 0, 1.0], [59, 3, 1.0], [59, 6, 4.0], [59, 7, 10.0], [59, 8, 37.0], [60, 5, 1.0], [61, 5, 1.0], [62, 4, 1.0], [63, 0, 1.0], [63, 1, 3.0], [63, 2, 1.0], [63, 3, 2.0], [63, 4, 1.0], [63, 5, 9.0], [63, 6, 2.0], [63, 7, 4.0], [63, 8, 5.0], [64, 5, 1.0], [65, 8, 1.0], [66, 2, 1.0], [67, 1, 1.0], [68, 5, 1.0], [69, 5, 1.0], [70, 0, 2.0], [70, 1, 1.0], [70, 2, 10.0], [70, 3, 2.0], [70, 4, 24.0], [70, 7, 1.0], [70, 8, 1.0], [71, 3, 1.0], [72, 5, 1.0], [73, 4, 2.0], [73, 5, 2.0], [73, 6, 1.0], [73, 7, 4.0], [73, 8, 1.0], [74, 6, 2.0], [74, 7, 3.0], [74, 8, 2.0], [75, 2, 2.0], [75, 7, 1.0], [75, 8, 2.0], [76, 1, 1.0], [77, 1, 1.0], [78, 7, 2.0], [78, 8, 2.0], [79, 0, 1.0], [80, 7, 1.0], [81, 2, 1.0], [82, 2, 1.0], [83, 2, 1.0], [84, 3, 1.0], [84, 6, 1.0], [84, 7, 2.0], [84, 8, 19.0], [85, 2, 1.0], [86, 3, 1.0], [87, 3, 1.0], [88, 3, 1.0], [89, 6, 1.0], [89, 7, 3.0], [90, 7, 1.0], [91, 7, 1.0], [92, 1, 4.0], [92, 2, 4.0], [92, 4, 1.0], [92, 5, 2.0], [92, 7, 2.0], [92, 8, 1.0], [93, 7, 1.0], [94, 4, 1.0], [95, 2, 1.0], [96, 0, 1.0], [96, 1, 4.0], [96, 2, 2.0], [96, 3, 6.0], [97, 0, 2.0], [98, 8, 1.0], [99, 2, 1.0], [100, 0, 1.0], [100, 2, 2.0], [101, 6, 1.0], [102, 3, 1.0], [102, 6, 3.0], [102, 7, 1.0], [103, 3, 1.0], [104, 5, 1.0], [105, 5, 1.0], [106, 8, 1.0], [107, 1, 1.0], [108, 1, 1.0], [109, 0, 1.0], [109, 1, 1.0], [109, 3, 2.0], [109, 4, 4.0], [110, 6, 1.0], [111, 6, 5.0], [112, 5, 1.0], [113, 1, 2.0], [113, 2, 1.0], [114, 4, 1.0], [115, 0, 1.0], [116, 8, 1.0], [117, 2, 1.0], [117, 3, 1.0], [118, 1, 1.0], [119, 0, 1.0], [119, 3, 1.0], [119, 5, 1.0], [120, 0, 1.0], [121, 1, 1.0], [121, 8, 1.0], [122, 1, 1.0], [123, 0, 1.0], [123, 1, 3.0], [123, 2, 1.0], [124, 2, 1.0], [124, 7, 1.0], [125, 0, 1.0], [126, 7, 1.0], [127, 0, 1.0], [128, 5, 1.0], [129, 0, 2.0], [129, 1, 2.0], [129, 2, 2.0], [130, 4, 1.0], [131, 0, 1.0], [132, 0, 1.0], [132, 7, 1.0], [133, 2, 1.0], [134, 6, 1.0], [135, 6, 1.0], [136, 0, 1.0], [136, 2, 1.0], [137, 5, 1.0], [138, 0, 1.0], [139, 0, 1.0], [140, 5, 1.0], [141, 3, 1.0], [142, 5, 1.0], [143, 2, 2.0], [143, 7, 1.0], [144, 7, 2.0], [145, 3, 6.0], [145, 7, 1.0], [146, 5, 1.0], [147, 8, 1.0], [148, 0, 4.0], [148, 1, 9.0], [148, 3, 2.0], [148, 7, 2.0], [149, 7, 1.0], [150, 3, 1.0], [151, 8, 1.0], [152, 7, 1.0], [153, 3, 1.0], [153, 6, 1.0], [153, 7, 5.0], [153, 8, 2.0], [154, 8, 1.0], [155, 5, 1.0], [156, 6, 1.0], [157, 6, 1.0], [158, 0, 1.0], [158, 3, 1.0], [159, 1, 2.0], [160, 1, 1.0], [161, 7, 1.0], [161, 8, 1.0], [162, 0, 14.0], [162, 1, 1.0], [162, 2, 14.0], [162, 3, 1.0], [163, 2, 1.0], [164, 7, 1.0], [164, 8, 1.0], [165, 0, 1.0], [165, 2, 1.0], [166, 0, 1.0], [166, 2, 1.0], [166, 3, 1.0], [167, 0, 2.0], [167, 1, 3.0], [167, 2, 8.0], [167, 4, 1.0], [168, 7, 1.0], [169, 3, 1.0], [170, 0, 1.0], [170, 4, 1.0], [171, 7, 1.0], [172, 4, 1.0], [173, 4, 1.0], [174, 8, 1.0], [175, 6, 1.0], [175, 8, 1.0], [176, 6, 1.0], [177, 0, 1.0], [177, 4, 10.0], [178, 6, 2.0], [179, 5, 1.0], [180, 2, 1.0], [180, 5, 2.0], [180, 7, 1.0], [181, 4, 9.0], [181, 5, 1.0], [182, 3, 1.0], [183, 3, 1.0], [184, 3, 1.0], [185, 5, 2.0], [186, 1, 1.0], [187, 3, 1.0], [187, 5, 1.0], [187, 7, 1.0], [187, 8, 1.0], [188, 7, 1.0], [189, 2, 1.0], [190, 3, 1.0], [190, 7, 1.0], [191, 2, 1.0], [192, 4, 4.0], [192, 8, 2.0], [193, 2, 1.0], [194, 5, 1.0], [195, 2, 1.0], [196, 2, 1.0], [197, 7, 1.0], [198, 3, 1.0], [199, 5, 1.0], [200, 5, 1.0], [201, 2, 1.0], [202, 0, 2.0], [202, 1, 1.0], [202, 3, 5.0], [202, 5, 1.0], [203, 0, 29.0], [203, 1, 1.0], [203, 2, 10.0], [204, 7, 1.0], [205, 2, 1.0], [206, 8, 1.0], [207, 8, 1.0], [208, 2, 2.0], [209, 3, 1.0], [210, 6, 13.0], [211, 6, 1.0], [212, 3, 2.0], [213, 2, 1.0], [214, 5, 2.0], [214, 7, 1.0], [215, 2, 1.0], [216, 4, 2.0], [216, 7, 5.0], [217, 3, 1.0], [217, 4, 4.0], [218, 0, 1.0], [218, 5, 4.0], [219, 1, 1.0], [220, 0, 2.0], [220, 2, 2.0], [220, 3, 1.0], [221, 2, 1.0], [222, 3, 4.0], [223, 8, 1.0], [224, 0, 1.0], [224, 2, 1.0], [225, 0, 2.0], [225, 1, 2.0], [225, 3, 1.0], [226, 0, 1.0], [227, 3, 1.0], [228, 4, 1.0], [229, 0, 1.0], [229, 1, 4.0], [229, 3, 1.0], [230, 0, 1.0], [230, 3, 5.0], [230, 4, 17.0], [230, 5, 20.0], [231, 5, 1.0], [232, 5, 2.0], [233, 7, 1.0], [234, 8, 1.0], [235, 7, 1.0], [236, 7, 1.0], [237, 5, 1.0], [238, 0, 1.0], [238, 3, 1.0], [239, 7, 1.0], [240, 2, 1.0], [240, 7, 1.0], [241, 2, 1.0], [242, 0, 1.0], [243, 6, 1.0], [244, 7, 1.0], [245, 5, 1.0], [246, 0, 1.0], [247, 5, 7.0], [247, 7, 2.0], [247, 8, 2.0], [248, 6, 1.0], [249, 3, 1.0], [250, 5, 1.0], [250, 7, 1.0], [251, 5, 1.0], [252, 5, 1.0], [252, 6, 1.0], [253, 4, 1.0], [254, 0, 1.0], [254, 2, 1.0], [255, 7, 1.0], [256, 3, 1.0], [257, 3, 2.0], [258, 3, 1.0], [258, 5, 1.0], [259, 3, 2.0], [260, 7, 1.0], [261, 2, 1.0], [262, 3, 1.0], [263, 2, 1.0], [264, 0, 1.0], [265, 2, 1.0], [266, 8, 1.0], [267, 6, 1.0], [267, 8, 1.0], [268, 5, 1.0], [269, 1, 1.0], [269, 4, 1.0], [270, 8, 1.0], [271, 7, 1.0], [272, 3, 1.0], [273, 0, 1.0], [273, 7, 2.0], [274, 8, 1.0], [275, 0, 1.0], [275, 1, 5.0], [275, 2, 3.0], [275, 3, 2.0], [275, 8, 1.0], [276, 2, 2.0], [277, 5, 1.0], [278, 5, 6.0], [278, 7, 3.0], [279, 4, 1.0], [280, 3, 1.0], [281, 5, 1.0], [282, 1, 2.0], [282, 3, 2.0], [282, 7, 1.0], [283, 7, 1.0], [284, 4, 1.0], [285, 4, 1.0], [286, 6, 2.0], [286, 8, 1.0], [287, 6, 5.0], [288, 4, 1.0], [289, 1, 1.0], [289, 6, 1.0], [290, 6, 1.0], [291, 1, 1.0], [292, 8, 1.0], [293, 3, 1.0], [294, 1, 1.0], [294, 8, 1.0], [295, 7, 1.0], [296, 7, 1.0], [297, 8, 2.0], [298, 7, 1.0], [299, 5, 1.0], [300, 3, 1.0], [301, 5, 1.0], [302, 7, 1.0], [303, 1, 1.0], [304, 5, 1.0], [304, 7, 1.0], [305, 1, 1.0], [306, 5, 1.0], [307, 2, 1.0], [308, 7, 1.0], [309, 2, 2.0], [310, 5, 1.0], [311, 1, 1.0], [312, 0, 1.0], [312, 2, 2.0], [312, 5, 6.0], [313, 7, 1.0], [314, 8, 1.0], [315, 1, 2.0], [315, 3, 7.0], [315, 7, 2.0], [316, 7, 1.0], [317, 4, 5.0], [317, 5, 2.0], [318, 3, 1.0], [319, 0, 11.0], [319, 1, 12.0], [319, 2, 5.0], [319, 3, 13.0], [319, 4, 2.0], [320, 5, 1.0], [321, 6, 1.0], [321, 7, 1.0], [321, 8, 1.0], [322, 6, 1.0], [323, 6, 1.0], [324, 3, 1.0], [325, 6, 1.0], [326, 2, 1.0], [327, 1, 1.0], [327, 8, 1.0], [328, 5, 1.0], [329, 5, 1.0], [330, 4, 1.0], [331, 8, 1.0], [332, 0, 1.0], [333, 2, 2.0], [334, 1, 1.0], [335, 5, 1.0], [336, 1, 2.0], [336, 2, 1.0], [337, 1, 1.0], [338, 8, 1.0], [339, 2, 1.0], [340, 5, 1.0], [341, 0, 3.0], [342, 7, 1.0], [343, 7, 1.0], [344, 4, 2.0], [345, 8, 1.0], [346, 6, 1.0], [347, 0, 1.0], [347, 4, 2.0], [347, 6, 2.0], [347, 8, 3.0], [348, 5, 1.0], [349, 6, 2.0], [349, 7, 1.0], [350, 4, 3.0], [351, 1, 1.0], [352, 3, 1.0], [353, 3, 1.0], [354, 5, 1.0], [355, 2, 1.0], [356, 6, 1.0], [357, 3, 1.0], [358, 6, 1.0], [359, 5, 1.0], [359, 6, 1.0], [360, 7, 1.0], [361, 4, 2.0], [361, 6, 1.0], [362, 7, 1.0], [363, 7, 1.0], [364, 2, 1.0], [365, 6, 1.0], [366, 5, 1.0], [367, 1, 1.0], [368, 7, 1.0], [368, 8, 1.0], [369, 7, 1.0], [370, 3, 2.0], [371, 1, 2.0], [371, 2, 2.0], [371, 3, 4.0], [371, 5, 5.0], [371, 6, 1.0], [371, 7, 5.0], [372, 1, 1.0], [373, 7, 1.0], [374, 6, 1.0], [375, 2, 2.0], [376, 4, 1.0], [377, 5, 5.0], [377, 6, 9.0], [377, 7, 5.0], [377, 8, 3.0], [378, 0, 2.0], [378, 1, 18.0], [378, 3, 1.0], [378, 6, 21.0], [378, 7, 4.0], [378, 8, 4.0], [379, 3, 1.0], [380, 1, 2.0], [380, 5, 1.0], [381, 2, 1.0], [382, 0, 1.0], [382, 2, 1.0], [383, 2, 1.0], [384, 2, 1.0], [384, 7, 1.0], [385, 2, 1.0], [385, 8, 1.0], [386, 6, 1.0], [387, 1, 19.0], [387, 2, 2.0], [387, 4, 2.0], [387, 6, 3.0], [388, 7, 1.0], [389, 2, 1.0], [390, 2, 1.0], [391, 3, 1.0], [392, 2, 1.0], [393, 0, 1.0], [393, 1, 1.0], [393, 2, 1.0], [394, 1, 1.0], [394, 2, 2.0], [394, 4, 9.0], [394, 5, 1.0], [394, 6, 1.0], [394, 7, 1.0], [394, 8, 3.0], [395, 6, 2.0], [396, 1, 2.0], [396, 2, 3.0], [396, 3, 1.0], [396, 4, 4.0], [396, 6, 5.0], [396, 8, 4.0], [397, 6, 1.0], [398, 8, 1.0], [399, 1, 2.0], [399, 2, 1.0], [399, 3, 2.0], [399, 6, 1.0], [399, 7, 1.0], [399, 8, 1.0], [400, 1, 1.0], [401, 3, 1.0], [402, 1, 1.0], [403, 6, 1.0], [404, 6, 4.0], [405, 1, 1.0], [405, 6, 3.0], [406, 8, 1.0], [407, 3, 9.0], [407, 6, 3.0], [408, 4, 1.0], [409, 8, 1.0], [410, 0, 1.0], [411, 2, 1.0], [412, 6, 8.0], [412, 7, 10.0], [412, 8, 2.0], [413, 2, 1.0], [413, 3, 3.0], [414, 1, 1.0], [415, 3, 1.0], [416, 3, 1.0]], "columns": [{"id": "PC.354", "metadata": null}, {"id": "PC.355", "metadata": null}, {"id": "PC.356", "metadata": null}, {"id": "PC.481", "metadata": null}, {"id": "PC.593", "metadata": null}, {"id": "PC.607", "metadata": null}, {"id": "PC.634", "metadata": null}, {"id": "PC.635", "metadata": null}, {"id": "PC.636", "metadata": null}], "generated_by": "QIIME 1.4.0-dev, svn revision 2520", "matrix_type": "sparse", "shape": [417, 9], "format_url": "http://www.qiime.org/svn_documentation/documentation/biom_format.html", "date": "2011-12-20T17:28:05.299943", "type": "OTU table", "id": null, "matrix_element_type": "float"}"""

tutorial_map = """#SampleID	BarcodeSequence	LinkerPrimerSequence	Treatment	DOB	Description
#Example mapping file for the QIIME analysis package.  These 9 samples are from a study of the effects of exercise and diet on mouse cardiac physiology (Crawford, et al, PNAS, 2009).
PC.354	AGCACGAGCCTA	YATGCTGCCTCCCGTAGGAGT	Control	20061218	Control_mouse_I.D._354
PC.355	AACTCGTCGATG	YATGCTGCCTCCCGTAGGAGT	Control	20061218	Control_mouse_I.D._355
PC.356	ACAGACCACTCA	YATGCTGCCTCCCGTAGGAGT	Control	20061126	Control_mouse_I.D._356
PC.481	ACCAGCGACTAG	YATGCTGCCTCCCGTAGGAGT	Control	20070314	Control_mouse_I.D._481
PC.593	AGCAGCACTTGT	YATGCTGCCTCCCGTAGGAGT	Control	20071210	Control_mouse_I.D._593
PC.607	AACTGTGCGTAC	YATGCTGCCTCCCGTAGGAGT	Fast	20071112	Fasting_mouse_I.D._607
PC.634	ACAGAGTCGGCT	YATGCTGCCTCCCGTAGGAGT	Fast	20080116	Fasting_mouse_I.D._634
PC.635	ACCGCAGAGTCA	YATGCTGCCTCCCGTAGGAGT	Fast	20080116	Fasting_mouse_I.D._635
Photobomb	what?	this sample	shouldn't	be	in here!
PC.636	ACGGTGAGTGTC	YATGCTGCCTCCCGTAGGAGT	Fast	20080116	Fasting_mouse_I.D._636
"""


if __name__ == "__main__":
    main()
