%feature("docstring") OT::SobolSimulationAlgorithm
R"RAW(Sobol indices computation using iterative sampling.

The algorithm uses sampling of the distribution of the random vector :math:`\vect{X}` through the model :math:`f`
to iteratively estimate the Sobol indices.

At each iteration a Sobol' experiment is generated and evaluated by blocks through the model :math:`f`.
Then the distribution of the indices (first and total order) is computed on this current replication sample.
At the end of each iteration we update the global distribution of the indices.

Parameters
----------
X : :class:`~openturns.Distribution`
    The random vector to study.
f : :class:`~openturns.Function`
    The function to study.
estimator : :class:`~openturns.SobolIndicesAlgorithm`
    The estimator of the indices.

Notes
-----
The algorithm can operate on a multivariate model :math:`f`, in this case it operates on aggregated indices.

Several estimators are available (Saltelli, Jansen, ...).

Let us denote by :math:`n_X` the number of input variables. 
For any :math:`j = 1, \ldots, n_X`, let us denote by :math:`\Phi_j^F` (resp. :math:`\Phi_j^T`) the cumulative distribution 
function of the gaussian asymptotic distribution of the estimator 
of the Sobol' first (resp. total) order indice.
Let :math:`\alpha\in[0,1]` be the level of the confidence interval and :math:`\epsilon\in(0,1]` the 
length of this confidence interval. 
The algorithms stops when, on all components, first and total order indices haved been 
estimated with enough precision.

The precision is said to be sufficient if the length of the :math:`\alpha`-level confidence interval 
is smaller than :math:`\epsilon`:

.. math::

    (\Phi_j^F)^{-1}(1-\alpha/2) - (\Phi_j^F)^{-1}(\alpha/2) \leq \epsilon \textrm{ and } 
    (\Phi_j^T)^{-1}(1-\alpha/2) - (\Phi_j^T)^{-1}(\alpha/2) \leq \epsilon

for all :math:`j = 1, \ldots, n_X`.

The total number of samples evaluated by the algorithm is :math:`n_{iter} \times N \times (2 + n_X)` (no second order indices) where

- :math:`n_{iter}` is the actual number of outer iterations, as caped by :meth:`setMaximumOuterSampling`
- :math:`N` is the internal size of the Sobol' experiment, as set by :meth:`setExperimentSize`
- :math:`n_X` is the input dimension

That is why unlike other simulation algorithms the total of evaluations is not equal to
the product of iterations (obtained by `getOuterSampling` from the result) by the block size (:meth:`getBlockSize`).

This class makes use of the following :class:`~openturns.ResourceMap` entries:

- `SobolSimulationAlgorithm-DefaultIndexQuantileLevel`: default float value for the confidence level :math:`\alpha`
- `SobolSimulationAlgorithm-DefaultIndexQuantileEpsilon`: default float value for the confidence length :math:`\epsilon`
- `SobolSimulationAlgorithm-DefaultExperimentSize`: default integer value for :math:`N`

See also
--------
SobolSimulationResult, SobolIndicesExperiment, SobolIndicesAlgorithm

Examples
--------
>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> distribution = ot.JointDistribution([ot.Uniform(-1.0, 1.0)] * 3)
>>> model = ot.SymbolicFunction(['x1', 'x2', 'x3'], ['x1*x2+x3'])
>>> estimator = ot.SaltelliSensitivityAlgorithm()
>>> estimator.setUseAsymptoticDistribution(True)
>>> algo = ot.SobolSimulationAlgorithm(distribution, model, estimator)
>>> algo.setMaximumOuterSampling(25)  # maximum number of iterations
>>> algo.setExperimentSize(100)  # size of Sobol experiment at each iteration
>>> algo.setBlockSize(4) # number of points evaluated simultaneously
>>> algo.setIndexQuantileLevel(0.05)  # alpha
>>> algo.setIndexQuantileEpsilon(1e-2)  # epsilon
>>> algo.run()
>>> result = algo.getResult()
>>> fo = result.getFirstOrderIndicesEstimate()
>>> foDist = result.getFirstOrderIndicesDistribution()
)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::SobolSimulationAlgorithm::getResult
"Accessor to the result.

Returns
-------
result : :class:`~openturns.SobolSimulationResult`
    The simulation result."

// ---------------------------------------------------------------------

%feature("docstring") OT::SobolSimulationAlgorithm::setIndexQuantileLevel
"Accessor to the quantile level.

Parameters
----------
alpha : float
    The quantile level."

// ---------------------------------------------------------------------

%feature("docstring") OT::SobolSimulationAlgorithm::getIndexQuantileLevel
"Accessor to the quantile level.

Returns
-------
alpha : float
    The quantile level."

// ---------------------------------------------------------------------

%feature("docstring") OT::SobolSimulationAlgorithm::setIndexQuantileEpsilon
"Accessor to the quantile tolerance.

Parameters
----------
epsilon : float
    The quantile tolerance"

// ---------------------------------------------------------------------

%feature("docstring") OT::SobolSimulationAlgorithm::getIndexQuantileEpsilon
"Accessor to the criterion operator.

Returns
-------
epsilon : float
    The quantile tolerance"

// ---------------------------------------------------------------------

%feature("docstring") OT::SobolSimulationAlgorithm::drawFirstOrderIndexConvergence
"Draw the first order Sobol index convergence at a given level.

Parameters
----------
marginalIndex : int
    Index of the random vector component to consider
level : float, optional
    The expectation convergence is drawn at this given confidence length
    *level*. By default *level* is 0.95.

Returns
-------
graph : a :class:`~openturns.Graph`
    expectation convergence graph"

// ---------------------------------------------------------------------

%feature("docstring") OT::SobolSimulationAlgorithm::drawTotalOrderIndexConvergence
"Draw the total order Sobol index convergence at a given level.

Parameters
----------
marginalIndex : int
    Index of the random vector component to consider
level : float, optional
    The expectation convergence is drawn at this given confidence length
    *level*. By default *level* is 0.95.

Returns
-------
graph : a :class:`~openturns.Graph`
    expectation convergence graph"

// ---------------------------------------------------------------------

%feature("docstring") OT::SobolSimulationAlgorithm::setEstimator
"Sobol estimator accessor.

Parameters
----------
estimator : :class:`~openturns.SobolIndicesAlgorithm`
    The estimator of the indices."

// ---------------------------------------------------------------------

%feature("docstring") OT::SobolSimulationAlgorithm::getEstimator
"Sobol estimator accessor.

Returns
-------
estimator : :class:`~openturns.SobolIndicesAlgorithm`
    The estimator of the indices."

// ---------------------------------------------------------------------

%feature("docstring") OT::SobolSimulationAlgorithm::setExperimentSize
"Accessor to the experiment size.

Parameters
----------
size : int
    Internal size :math:`N` of the Sobol design of experiment.
    See :class:`~openturns.SobolIndicesExperiment`"

// ---------------------------------------------------------------------

%feature("docstring") OT::SobolSimulationAlgorithm::getExperimentSize
"Accessor to the experiment size.

Returns
-------
size : int
    Internal size :math:`N` of the Sobol design of experiment.
    See :class:`~openturns.SobolIndicesExperiment`"

// ---------------------------------------------------------------------

%feature("docstring") OT::SobolSimulationAlgorithm::getDistribution
"Accessor to the batch size.

Returns
-------
distribution : :class:`~openturns.Distribution`
    Distribution of the random variable."
