/*
 *  The ManaPlus Client
 *  Copyright (C) 2007  Joshua Langley <joshlangley@optusnet.com.au>
 *  Copyright (C) 2009-2010  The Mana Developers
 *  Copyright (C) 2011-2018  The ManaPlus Developers
 *
 *  This file is part of The ManaPlus Client.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef INPUT_INPUTACTIONMAP_H
#define INPUT_INPUTACTIONMAP_H

#include "sdlshared.h"

#include "actions/actions.h"
#include "actions/chat.h"
#include "actions/commands.h"
#include "actions/move.h"
#include "actions/pets.h"
#include "actions/statusbar.h"
#include "actions/tabs.h"
#include "actions/target.h"
#include "actions/windows.h"

#include "enums/input/inputcondition.h"
#include "enums/input/inputgroup.h"

#include "input/inputactiondata.h"
#include "input/inputmanager.h"

#include "localconsts.h"

#define defaultAction(name) \
        InputType::UNKNOWN, -1, \
        InputType::UNKNOWN, -1, \
        Input::GRP_DEFAULT, \
        name, \
        InputAction::NO_VALUE, 50

#define addKey(name) InputType::KEYBOARD, name
#define emptyKey InputType::UNKNOWN, -1
#define joystickButton(num) InputType::JOYSTICK, num

// keyData must be in same order as enum keyAction.
static const InputActionData inputActionData
    [CAST_SIZE(InputAction::TOTAL)] =
    {
    {"keyMoveUp",
        addKey(SDLK_UP),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveUp,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::ARROWKEYS,
        "",
        UseArgs_false,
        Protected_false},
    {"keyMoveDown",
        addKey(SDLK_DOWN),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveDown,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::ARROWKEYS,
        "",
        UseArgs_false,
        Protected_false},
    {"keyMoveLeft",
        addKey(SDLK_LEFT),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveLeft,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::ARROWKEYS,
        "",
        UseArgs_false,
        Protected_false},
    {"keyMoveRight",
        addKey(SDLK_RIGHT),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveRight,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::ARROWKEYS,
        "",
        UseArgs_false,
        Protected_false},
    {"keyAttack",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_REPEAT,
        &Actions::attack,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOFOLLOW |
            InputCondition::VALIDSPEED,
        "attack",
        UseArgs_false,
        Protected_false},
    {"keyTargetAttack",
        addKey(SDLK_LCTRL),
        joystickButton(0),
        Input::GRP_DEFAULT,
        &Actions::targetAttack,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOFOLLOW |
            InputCondition::VALIDSPEED |
            InputCondition::ALIVE,
        "targetattack",
        UseArgs_true,
        Protected_false},
    {"keyMoveToTarget",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveToTarget,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED,
        "movetotarget",
        UseArgs_false,
        Protected_false},
    {"keyChangeMoveToTarget",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeMoveToTarget,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED |
            InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToHome",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveToHome,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED,
        "movetohome",
        UseArgs_false,
        Protected_false},
    {"keySetHome",
        addKey(SDLK_KP5),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::setHome,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED,
        "sethome",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPoint",
        addKey(SDLK_RSHIFT),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::DEFAULT |
            InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyTalk",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::talk,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "talk",
        UseArgs_false,
        Protected_false},
    {"keyTarget",
        addKey(SDLK_LSHIFT),
        joystickButton(4),
        Input::GRP_DEFAULT,
        &Actions::stopAttack,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "stopattack",
        UseArgs_false,
        Protected_false},
    {"keyUnTarget",
        defaultAction(&Actions::untarget),
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "untarget",
        UseArgs_false,
        Protected_false},
    {"keyTargetMonster",
        addKey(SDLK_a),
        joystickButton(3),
        Input::GRP_DEFAULT | Input::GRP_REPEAT,
        &Actions::targetMonster,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET |
            InputCondition::VALIDSPEED,
        "targetmonster",
        UseArgs_false,
        Protected_false},
    {"keyTargetNPC",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::targetNPC,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "targetnpc",
        UseArgs_false,
        Protected_false},
    {"keyTargetPlayer",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::targetPlayer,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "targetplayer",
        UseArgs_false,
        Protected_false},
    {"keyPickup",
        addKey(SDLK_z),
        joystickButton(1),
        Input::GRP_DEFAULT,
        &Actions::pickup,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "pickup",
        UseArgs_false,
        Protected_false},
    {"keyChangePickupType",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changePickupType,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED |
            InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keyHideWindows",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::hideWindows,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "hidewindows",
        UseArgs_false,
        Protected_true},
    {"keyBeingSit",
        addKey(SDLK_s),
        joystickButton(2),
        Input::GRP_DEFAULT,
        &Actions::sit,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET |
            InputCondition::NOBLOCK,
        "sit",
        UseArgs_false,
        Protected_false},
    {"keyScreenshot",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::screenshot,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::NOTARGET |
            InputCondition::NOINPUT,
        "screenshot",
        UseArgs_false,
        Protected_true},
    {"keyTrade",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeTrade,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "changetrade",
        UseArgs_false,
        Protected_true},
    {"keyPathfind",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeMapMode,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET |
            InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_false},
    {"keyOK",
        addKey(SDLK_SPACE),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::ok,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::NOMODAL |
            InputCondition::NOAWAY |
            InputCondition::NONPCINPUT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyQuit",
        addKey(SDLK_ESCAPE),
#if defined(USE_SDL2) && defined(ANDROID)
        addKey(SDLK_AC_BACK),
#else  // defined(USE_SDL2) && defined(ANDROID)
        emptyKey,
#endif  // defined(USE_SDL2) && defined(ANDROID)
        Input::GRP_DEFAULT,
        &Actions::quit,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "quit",
        UseArgs_false,
        Protected_true},
    {"keyShortcutsKey",
        addKey(SDLK_MENU),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::DEFAULT |
            InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut16",
        defaultAction(&Actions::shortcut),
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut17",
        defaultAction(&Actions::shortcut),
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut18",
        defaultAction(&Actions::shortcut),
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut19",
        defaultAction(&Actions::shortcut),
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut20",
        defaultAction(&Actions::shortcut),
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyWindowHelp",
        addKey(SDLK_F1),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::helpWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT0,
        "help|openhelp",
        UseArgs_false,
        Protected_false},
    {"keyWindowStatus",
        addKey(SDLK_F2),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::statusWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "status|openstatus",
        UseArgs_false,
        Protected_false},
    {"keyWindowInventory",
        addKey(SDLK_F3),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::inventoryWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "inventory|openinventory",
        UseArgs_false,
        Protected_false},
    {"keyWindowEquipment",
        addKey(SDLK_F4),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::equipmentWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "equipment|openequipment",
        UseArgs_false,
        Protected_false},
    {"keyWindowSkill",
        addKey(SDLK_F5),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::skillDialogShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "skill|skills|openskills",
        UseArgs_false,
        Protected_false},
    {"keyWindowMinimap",
        addKey(SDLK_F6),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::minimapWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "minimap|openminimap",
        UseArgs_false,
        Protected_false},
    {"keyWindowChat",
        addKey(SDLK_F7),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::chatWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT0,
        "openchat",
        UseArgs_false,
        Protected_false},
    {"keyWindowShortcut",
        addKey(SDLK_F8),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::shortcutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "openshortcut|openshortcuts",
        UseArgs_false,
        Protected_false},
    {"keyWindowSetup",
        addKey(SDLK_F9),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::setupWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::NOTARGET |
            InputCondition::NOINPUT |
            InputCondition::ENABLED,
        "setup|opensetup",
        UseArgs_false,
        Protected_false},
    {"keyWindowDebug",
        addKey(SDLK_F10),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::debugWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::SHORTCUT0 |
            InputCondition::NOTARGET,
        "debug|opendebug",
        UseArgs_false,
        Protected_false},
    {"keyWindowSocial",
        addKey(SDLK_F11),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::socialWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "social|opensocial",
        UseArgs_false,
        Protected_false},
    {"keyWindowEmoteBar",
        addKey(SDLK_F12),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::emoteShortcutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "emotes|openemotes",
        UseArgs_false,
        Protected_false},
    {"keyWindowOutfit",
        addKey(SDLK_BACKQUOTE),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::outfitWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "openoutfit|openoutfits",
        UseArgs_false,
        Protected_false},
    {"keyWindowShop",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::shopWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "shop|openshop",
        UseArgs_false,
        Protected_false},
    {"keyWindowDrop",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::dropShortcutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "opendrop",
        UseArgs_false,
        Protected_false},
    {"keyWindowKills",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::killStatsWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "kills|openkills",
        UseArgs_false,
        Protected_false},
    {"keyWindowSpells",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::spellShortcutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "spells|openspells",
        UseArgs_false,
        Protected_false},
    {"keyWindowBotChecker",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "",
        UseArgs_false,
        Protected_false},
    {"keyWindowOnline",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::whoIsOnlineWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "online|openonline",
        UseArgs_false,
        Protected_false},
    {"keyWindowDidYouKnow",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::didYouKnowWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT0,
        "didyouknow|opendidyouknow",
        UseArgs_false,
        Protected_false},
    {"keyWindowQuests",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::questsWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "quests|openquests",
        UseArgs_false,
        Protected_false},
    {"keySocialPrevTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevSocialTab,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::NOINPUT |
            InputCondition::INGAME,
        "prevsocialtab",
        UseArgs_false,
        Protected_false},
    {"keySocialNextTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextSocialTab,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::NOINPUT |
            InputCondition::INGAME,
        "nextsocialtab",
        UseArgs_false,
        Protected_false},
    {"keySmilie",
        addKey(SDLK_LALT),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::DEFAULT |
            InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyWearOutfit",
        addKey(SDLK_RCTRL),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::DEFAULT |
            InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyCopyOutfit",
        addKey(SDLK_RALT),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::DEFAULT |
            InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyCopyEquipedOutfit",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::copyEquippedToOutfit,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED,
        "copyequippedtooutfit|itemstooutfit",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyChat",
        addKey(SDLK_RETURN),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_CHAT,
        &Actions::toggleChat,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::NOINPUT |
            InputCondition::NOAWAY |
            InputCondition::NOMODAL,
        "chat",
        UseArgs_false,
        Protected_false},
    {"keyChatScrollUp",
        addKey(SDLK_PAGEUP),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::scrollChatUp,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::DEFAULT |
            InputCondition::INGAME,
        "chatscrollup",
        UseArgs_false,
        Protected_false},
    {"keyChatScrollDown",
        addKey(SDLK_PAGEDOWN),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::scrollChatDown,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::DEFAULT |
            InputCondition::INGAME,
        "chatscrolldown",
        UseArgs_false,
        Protected_false},
    {"keyChatPrevTab",
        addKey(SDLK_KP7),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevChatTab,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::NOINPUT |
            InputCondition::INGAME,
        "prevtab",
        UseArgs_false,
        Protected_false},
    {"keyChatNextTab",
        addKey(SDLK_KP9),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextChatTab,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::NOINPUT |
            InputCondition::INGAME,
        "nexttab",
        UseArgs_false,
        Protected_false},
    {"keyChatCloseTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::closeChatTab,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::NOINPUT |
            InputCondition::INGAME,
        "close",
        UseArgs_false,
        Protected_false},
    {"keyChatPrevHistory",
        addKey(SDLK_KP7),
        emptyKey,
        Input::GRP_CHAT | Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::DEFAULT |
            InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyChatNextHistory",
        addKey(SDLK_KP9),
        emptyKey,
        Input::GRP_CHAT | Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::DEFAULT |
            InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyIgnoreInput1",
#ifdef USE_SDL2
        addKey(SDLK_LGUI),
#else  // USE_SDL2
        addKey(SDLK_LSUPER),
#endif  // USE_SDL2
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::ignoreInput,
        InputAction::NO_VALUE, 500,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyIgnoreInput2",
#ifdef USE_SDL2
        addKey(SDLK_RGUI),
#else  // USE_SDL2
        addKey(SDLK_RSUPER),
#endif  // USE_SDL2
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::ignoreInput,
        InputAction::NO_VALUE, 500,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyDirectUp",
        defaultAction(&Actions::directUp),
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "directup|turnup",
        UseArgs_false,
        Protected_false},
    {"keyDirectDown",
        defaultAction(&Actions::directDown),
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "directdown|turndown",
        UseArgs_false,
        Protected_false},
    {"keyDirectLeft",
        defaultAction(&Actions::directLeft),
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "directleft|turnleft",
        UseArgs_false,
        Protected_false},
    {"keyDirectRight",
        defaultAction(&Actions::directRight),
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "directright|turnright",
        UseArgs_false,
        Protected_false},
    {"keyCrazyMoves",
        addKey(SDLK_SLASH),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::crazyMoves,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED,
        "",
        UseArgs_false,
        Protected_true},
    {"keyChangeCrazyMoveType",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeCrazyMove,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED |
            InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keyQuickDrop",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::dropItem0,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED,
        "drop",
        UseArgs_false,
        Protected_true},
    {"keyQuickDropN",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::dropItem,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED,
        "dropn",
        UseArgs_false,
        Protected_true},
    {"keySwitchQuickDrop",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::switchQuickDrop,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMagicInma1",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::heal,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED,
        "heal",
        UseArgs_true,
        Protected_false},
    {"keyMagicItenplz",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::itenplz,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED,
        "itenplz",
        UseArgs_false,
        Protected_true},
    {"keyMagicAttack",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::magicAttack,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOFOLLOW |
            InputCondition::VALIDSPEED,
        "magicattack",
        UseArgs_false,
        Protected_false},
    {"keySwitchMagicAttack",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeMagicAttackType,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED |
            InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keySwitchPvpAttack",
        defaultAction(&Actions::changePvpMode),
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED |
            InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keyInvertDirection",
        defaultAction(&Actions::changeMoveType),
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED |
            InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_false},
    {"keyChangeAttackWeaponType",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeAttackWeaponType,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED |
            InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keyChangeAttackType",
        addKey(SDLK_END),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeAttackType,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED |
            InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keyChangeFollowMode",
        addKey(SDLK_KP1),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeFollowMode,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED |
            InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keyChangeImitationMode",
        addKey(SDLK_KP4),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeImitationMode,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED |
            InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keyDisableGameModifiers",
        addKey(SDLK_KP8),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeGameModifier,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED,
        "",
        UseArgs_false,
        Protected_true},
    {"keyChangeAudio",
        addKey(SDLK_KP3),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeAudio,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED,
        "audio",
        UseArgs_false,
        Protected_true},
    {"keyAway",
        addKey(SDLK_KP2),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::away,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED,
        "",
        UseArgs_false,
        Protected_false},
    {"keyRightClick",
        addKey(SDLK_TAB),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::mouseClick,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::NOINPUT |
            InputCondition::NOAWAY |
            InputCondition::NOMODAL |
            InputCondition::INGAME,
        "rightclick",
        UseArgs_false,
        Protected_false},
    {"keyCameraMode",
        addKey(SDLK_KP_PLUS),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::camera,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED |
            InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_false},
    {"keyMod",
        emptyKey,
        emptyKey,
        0,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIUp",
        addKey(SDLK_UP),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIDown",
        addKey(SDLK_DOWN),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUILeft",
        addKey(SDLK_LEFT),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIRight",
        addKey(SDLK_RIGHT),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUISelect",
        addKey(SDLK_SPACE),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUISelect2",
        addKey(SDLK_RETURN),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUICancel",
        addKey(SDLK_ESCAPE),
#if defined(USE_SDL2) && defined(ANDROID)
        addKey(SDLK_AC_BACK),
#else  // defined(USE_SDL2) && defined(ANDROID)
        emptyKey,
#endif  // defined(USE_SDL2) && defined(ANDROID)
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIHome",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIEnd",
        addKey(SDLK_END),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIDelete",
        addKey(SDLK_DELETE),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIBackSpace",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUITab",
        addKey(SDLK_TAB),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIPageUp",
        addKey(SDLK_PAGEUP),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIPageDown",
        addKey(SDLK_PAGEDOWN),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIInsert",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIMod",
        addKey(SDLK_LSHIFT),
        addKey(SDLK_RSHIFT),
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keySafeVideo",
        addKey(SDLK_KP_MINUS),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::safeVideoMode,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyStopSit",
        defaultAction(&Actions::stopSit),
        InputCondition::KEY_DOWN |
            InputCondition::DEFAULT |
            InputCondition::INGAME,
        "sit2",
        UseArgs_false,
        Protected_false},
    {"keyShowKeyboard",
        defaultAction(&Actions::showKeyboard),
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "showkeyboard",
        UseArgs_false,
        Protected_false},
    {"keyShowWindows",
        defaultAction(&Actions::showWindows),
        InputCondition::KEY_DOWN |
            InputCondition::DEFAULT |
            InputCondition::INGAME,
        "showwindows",
        UseArgs_false,
        Protected_false},
    {"keyChatMod",
        addKey(SDLK_LSHIFT),
        emptyKey,
        Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::DEFAULT |
            InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyMoveForward",
        defaultAction(&Actions::moveForward),
        InputCondition::KEY_DOWN | InputCondition::ARROWKEYS,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUICtrl",
        addKey(SDLK_LCTRL),
        addKey(SDLK_RCTRL),
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIb",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIc",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUId",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIe",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIf",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIh",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIk",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIu",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIv",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIw",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyShortcutsPrevTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevShortcutsTab,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::NOINPUT |
            InputCondition::INGAME,
        "shortcutprevtab",
        UseArgs_false,
        Protected_false},
    {"keyShortcutsNextTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextShortcutsTab,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::NOINPUT |
            InputCondition::INGAME,
        "shortcutnexttab",
        UseArgs_false,
        Protected_false},
    {"keyCommandsPrevTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevCommandsTab,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::NOINPUT |
            InputCondition::INGAME,
        "commandsprevtab",
        UseArgs_false,
        Protected_false},
    {"keyCommandsNextTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextCommandsTab,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::NOINPUT |
            InputCondition::INGAME,
        "commandsnexttab",
        UseArgs_false,
        Protected_false},
    {"keyOpenTrade",
        defaultAction(&Actions::openTrade),
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "trade",
        UseArgs_false,
        Protected_false},
    {"keyGUIF1",
        addKey(SDLK_F1),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF2",
        addKey(SDLK_F2),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF3",
        addKey(SDLK_F3),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF4",
        addKey(SDLK_F4),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF5",
        addKey(SDLK_F5),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF6",
        addKey(SDLK_F6),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF7",
        addKey(SDLK_F7),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF8",
        addKey(SDLK_F8),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF9",
        addKey(SDLK_F9),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF10",
        addKey(SDLK_F10),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF11",
        addKey(SDLK_F11),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF12",
        addKey(SDLK_F12),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyWindowAbout",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::aboutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT0,
        "about",
        UseArgs_false,
        Protected_false},
    {"keyWindowUpdater",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::updaterWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "updater",
        UseArgs_false,
        Protected_false},
    {"keyTargetClosestMonster",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_REPEAT,
        &Actions::targetClosestMonster,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET |
            InputCondition::VALIDSPEED,
        "targetclosestmonster",
        UseArgs_false,
        Protected_false},
    {"keyChatCloseAllTabs",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::closeAllChatTabs,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "closeall",
        UseArgs_false,
        Protected_true},
    {"keyChatIgnoreAllWhispers",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::ignoreAllWhispers,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "ignoreallwhispers",
        UseArgs_false,
        Protected_true},
    {"keyChatAnnounce",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::chatAnnounce,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "announce",
        UseArgs_true,
        Protected_true},
    {"keyIpcToggle",
        defaultAction(&Actions::ipcToggle),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "ipctoggle",
        UseArgs_false,
        Protected_true},
    {"keyWhere",
        defaultAction(&Actions::where),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "where",
        UseArgs_false,
        Protected_false},
    {"keyWho",
        defaultAction(&Actions::who),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "who",
        UseArgs_false,
        Protected_false},
    {"keyWhisper",
        defaultAction(&Actions::msg),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "w|whisper|msg",
        UseArgs_true,
        Protected_true},
    {"keyQuery",
        defaultAction(&Actions::query),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "q|query|j|join",
        UseArgs_true,
        Protected_false},
    {"keyClearChatTab",
        defaultAction(&Actions::clearChatTab),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "clear",
        UseArgs_false,
        Protected_false},
    {"keyIgnore",
        defaultAction(&Actions::chatIgnore),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "ignore",
        UseArgs_true,
        Protected_true},
    {"keyUnignore",
        defaultAction(&Actions::chatUnignore),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "unignore",
        UseArgs_true,
        Protected_true},
    {"keyFriend",
        defaultAction(&Actions::chatFriend),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "friend|befriend",
        UseArgs_true,
        Protected_true},
    {"keyDisregard",
        defaultAction(&Actions::chatDisregard),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "disregard",
        UseArgs_true,
        Protected_true},
    {"keyNeutral",
        defaultAction(&Actions::chatNeutral),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "neutral",
        UseArgs_true,
        Protected_true},
    {"keyBlacklist",
        defaultAction(&Actions::chatBlackList),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "blacklist",
        UseArgs_true,
        Protected_true},
    {"keyEnemy",
        defaultAction(&Actions::chatEnemy),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "enemy",
        UseArgs_true,
        Protected_true},
    {"keyErase",
        defaultAction(&Actions::chatErase),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "erase",
        UseArgs_true,
        Protected_true},
    {"keyCleanGraphics",
        defaultAction(&Actions::cleanGraphics),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "cleangraphics",
        UseArgs_false,
        Protected_false},
    {"keyCleanFonts",
        defaultAction(&Actions::cleanFonts),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "cleanfonts",
        UseArgs_false,
        Protected_false},
    {"keyCreateParty",
        defaultAction(&Actions::createParty),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "createparty",
        UseArgs_true,
        Protected_true},
    {"keyCreateGuild",
        defaultAction(&Actions::createGuild),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "createguild",
        UseArgs_true,
        Protected_true},
    {"keyParty",
        defaultAction(&Actions::party),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "party",
        UseArgs_true,
        Protected_true},
    {"keyMe",
        defaultAction(&Actions::me),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "me",
        UseArgs_true,
        Protected_true},
    {"keyToggle",
        defaultAction(&Actions::toggle),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "toggle",
        UseArgs_false,
        Protected_false},
    {"keyPresent",
        defaultAction(&Actions::present),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "present",
        UseArgs_false,
        Protected_false},
    {"keyPrintAll",
        defaultAction(&Actions::printAll),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "all",
        UseArgs_false,
        Protected_true},
    {"keyMove",
        defaultAction(&Actions::move),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "move",
        UseArgs_true,
        Protected_false},
    {"keySetTarget",
        defaultAction(&Actions::setTarget),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "target",
        UseArgs_true,
        Protected_false},
    {"keyAttackHuman",
        defaultAction(&Actions::attackHuman),
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOFOLLOW |
            InputCondition::VALIDSPEED,
        "atkhuman",
        UseArgs_false,
        Protected_false},
    {"keyOutfit",
        defaultAction(&Actions::commandOutfit),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "outfit",
        UseArgs_true,
        Protected_true},
    {"keyEmote",
        defaultAction(&Actions::commandEmote),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "emote",
        UseArgs_true,
        Protected_false},
    {"keyEmotePet",
        defaultAction(&Actions::commandEmotePet),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "emotepet",
        UseArgs_true,
        Protected_false},
    {"keyAwayMessage",
        defaultAction(&Actions::awayMessage),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "away",
        UseArgs_true,
        Protected_true},
    {"keyPseudoAway",
        defaultAction(&Actions::pseudoAway),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "pseudoaway",
        UseArgs_true,
        Protected_true},
    {"keyFollow",
        defaultAction(&Actions::follow),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "follow",
        UseArgs_true,
        Protected_false},
    {"keyNavigate",
        defaultAction(&Actions::navigate),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "navigate",
        UseArgs_true,
        Protected_false},
    {"keyImitation",
        defaultAction(&Actions::imitation),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "imitation",
        UseArgs_true,
        Protected_false},
    {"keySendMail",
        defaultAction(&Actions::sendMail),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "sendmail|mailsend|mailto",
        UseArgs_true,
        Protected_true},
    {"keyTradeCommand",
        defaultAction(&Actions::trade),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "trade",
        UseArgs_true,
        Protected_false},
    {"keyPriceLoad",
        defaultAction(&Actions::priceLoad),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "priceload",
        UseArgs_false,
        Protected_true},
    {"keyPriceSave",
        defaultAction(&Actions::priceSave),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "pricesave",
        UseArgs_false,
        Protected_true},
    {"keyCacheInfo",
        defaultAction(&Actions::cacheInfo),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "cacheinfo",
        UseArgs_false,
        Protected_true},
    {"keyDisconnect",
        defaultAction(&Actions::disconnect),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "disconnect",
        UseArgs_false,
        Protected_true},
    {"keyUndress",
        defaultAction(&Actions::undress),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "undress",
        UseArgs_true,
        Protected_false},
    {"keyDirs",
        defaultAction(&Actions::dirs),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "dirs",
        UseArgs_false,
        Protected_false},
    {"keyInfo",
        defaultAction(&Actions::info),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "info",
        UseArgs_false,
        Protected_false},
    {"keyWait",
        defaultAction(&Actions::wait),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "wait",
        UseArgs_true,
        Protected_true},
    {"keyUptime",
        defaultAction(&Actions::uptime),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "uptime",
        UseArgs_false,
        Protected_false},
    {"keyAddPriorityAttack",
        defaultAction(&Actions::addPriorityAttack),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "addpriorityattack",
        UseArgs_true,
        Protected_true},
    {"keyAddAttack",
        defaultAction(&Actions::addAttack),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "addattack",
        UseArgs_true,
        Protected_true},
    {"keyRemoveAttack",
        defaultAction(&Actions::removeAttack),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "removeattack|removeignoreattack",
        UseArgs_true,
        Protected_true},
    {"keyAddIgnoreAttack",
        defaultAction(&Actions::addIgnoreAttack),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "addignoreattack",
        UseArgs_true,
        Protected_true},
    {"keyDump",
        defaultAction(&Actions::dump),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "dump",
        UseArgs_false,
        Protected_true},
    {"keyServerIgnoreAll",
        defaultAction(&Actions::serverIgnoreAll),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "serverignoreall",
        UseArgs_false,
        Protected_true},
    {"keyServerUnIgnoreAll",
        defaultAction(&Actions::serverUnIgnoreAll),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "serverunignoreall",
        UseArgs_false,
        Protected_true},
    {"keySetDrop",
        defaultAction(&Actions::setDrop),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "setdrop",
        UseArgs_true,
        Protected_false},
    {"keyError",
        defaultAction(&Actions::error),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "error",
        UseArgs_false,
        Protected_true},
    {"keyDumpg",
        defaultAction(&Actions::dumpGraphics),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "dumpg",
        UseArgs_false,
        Protected_true},
    {"keyDumpe",
        defaultAction(&Actions::dumpEnvironment),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "dumpe",
        UseArgs_false,
        Protected_true},
    {"keyDumpt",
        defaultAction(&Actions::dumpTests),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "dumpt",
        UseArgs_false,
        Protected_true},
    {"keyDumpOGL",
        defaultAction(&Actions::dumpOGL),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "dumpogl",
        UseArgs_false,
        Protected_true},
    {"keyDumpGL",
        defaultAction(&Actions::dumpGL),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "dumpgl",
        UseArgs_false,
        Protected_true},
    {"keyDumpMods",
        defaultAction(&Actions::dumpMods),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "dumpMods",
        UseArgs_false,
        Protected_false},
    {"keyUrl",
        defaultAction(&Actions::url),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "url",
        UseArgs_true,
        Protected_true},
    {"keyOpenUrl",
        defaultAction(&Actions::openUrl),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "open",
        UseArgs_true,
        Protected_true},
    {"keyExecute",
        defaultAction(&Actions::execute),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "execute",
        UseArgs_true,
        Protected_true},
    {"keyTestSdlFont",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT,
#if defined USE_OPENGL && defined DEBUG_SDLFONT
        &Actions::testSdlFont,
#else  // defined USE_OPENGL && defined DEBUG_SDLFONT
        nullptr,
#endif  // defined USE_OPENGL && defined DEBUG_SDLFONT
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::INGAME,
#if defined USE_OPENGL && defined DEBUG_SDLFONT
        "testsdlfont",
#else  // defined USE_OPENGL && defined DEBUG_SDLFONT
        "",
#endif  // defined USE_OPENGL && defined DEBUG_SDLFONT
        UseArgs_false,
        Protected_true},
    {"keyEnableHightlight",
        defaultAction(&Actions::enableHighlight),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "enablehighlight",
        UseArgs_false,
        Protected_true},
    {"keyDisableHightlight",
        defaultAction(&Actions::disableHighlight),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "disablehighlight",
        UseArgs_false,
        Protected_true},
    {"keyDontRemoveName",
        defaultAction(&Actions::dontRemoveName),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "dontremovename",
        UseArgs_false,
        Protected_true},
    {"keyRemoveName",
        defaultAction(&Actions::removeName),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "removename",
        UseArgs_false,
        Protected_true},
    {"keyDisableAway",
        defaultAction(&Actions::disableAway),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "disableaway",
        UseArgs_false,
        Protected_true},
    {"keyEnableAway",
        defaultAction(&Actions::enableAway),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "enableaway",
        UseArgs_false,
        Protected_true},
    {"keyTestParticle",
        defaultAction(&Actions::testParticle),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "testparticle",
        UseArgs_true,
        Protected_false},
    {"keyCreateItems",
        defaultAction(&Actions::createItems),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "createitems",
        UseArgs_false,
        Protected_false},
    {"keyTalkRaw",
        defaultAction(&Actions::talkRaw),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "talkraw",
        UseArgs_true,
        Protected_true},
    {"keyTalkPet",
        defaultAction(&Actions::talkPet),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "talkpet",
        UseArgs_true,
        Protected_true},
    {"keyUploadConfig",
        defaultAction(&Actions::uploadConfig),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "uploadconfig",
        UseArgs_false,
        Protected_true},
    {"keyUploadServerConfig",
        defaultAction(&Actions::uploadServerConfig),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "uploadserverconfig",
        UseArgs_false,
        Protected_true},
    {"keyUploadLog",
        defaultAction(&Actions::uploadLog),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "uploadlog",
        UseArgs_false,
        Protected_true},
    {"keyGm",
        defaultAction(&Actions::gm),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "gm",
        UseArgs_true,
        Protected_true},
    {"keyHack",
        defaultAction(&Actions::hack),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "hack",
        UseArgs_true,
        Protected_true},
    {"keyDebugSpawn",
        defaultAction(&Actions::debugSpawn),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "debugspawn",
        UseArgs_true,
        Protected_true},
    {"keyPetSmilie",
        defaultAction(nullptr),
        InputCondition::KEY_DOWN |
            InputCondition::DEFAULT |
            InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyCatchPet",
        defaultAction(&Actions::catchPet),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "catchpet|catch|taming",
        UseArgs_true,
        Protected_true},
    {"keyChatServerIgnoreWhisper",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::serverIgnoreWhisper,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "serverignore",
        UseArgs_false,
        Protected_true},
    {"keyChatServerUnIgnoreWhisper",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::serverUnIgnoreWhisper,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "serverunignore",
        UseArgs_false,
        Protected_true},
    {"keyMercenaryFire",
        defaultAction(&Actions::mercenaryFire),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "mercenaryfire|firemercenary",
        UseArgs_false,
        Protected_true},
    {"keyTargetMercenary",
        defaultAction(&Actions::targetMercenary),
        InputCondition::KEY_DOWN |
            InputCondition::INGAME |
            InputCondition::NOTARGET,
        "targetmercenary",
        UseArgs_false,
        Protected_false},
    {"keyTargetPet",
        defaultAction(&Actions::targetPet),
        InputCondition::KEY_DOWN |
            InputCondition::INGAME |
            InputCondition::NOTARGET,
        "targetpet",
        UseArgs_false,
        Protected_false},
    {"keyPetSetName",
        defaultAction(&Actions::setPetName),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "setpetname|petsetname",
        UseArgs_false,
        Protected_true},
    {"keyHomunculusSetName",
        defaultAction(&Actions::setHomunculusName),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "sethomunname|homunsetname|sethomunculusname|homunculussetname",
        UseArgs_false,
        Protected_true},
    {"keyHomunculusFire",
        defaultAction(&Actions::fireHomunculus),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "firehomunculus|homunculusfire|firehomun|homunfire",
        UseArgs_false,
        Protected_true},
    {"keyWindowBank",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::bankWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT0,
        "bank|openbank",
        UseArgs_false,
        Protected_false},
    {"keyInventoryPrevTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevInvTab,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::NOINPUT |
            InputCondition::INGAME,
        "previnvtab",
        UseArgs_false,
        Protected_false},
    {"keyInventoryNextTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextInvTab,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::NOINPUT |
            InputCondition::INGAME,
        "nextinvtab",
        UseArgs_false,
        Protected_false},
    {"keyContextMenu",
        defaultAction(&Actions::contextMenu),
        InputCondition::KEY_DOWN |
            InputCondition::NOINPUT |
            InputCondition::NOAWAY |
            InputCondition::NOMODAL |
            InputCondition::INGAME,
        "contextmenu",
        UseArgs_false,
        Protected_false},
    {"keyBuy",
        defaultAction(&Actions::buy),
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "buy",
        UseArgs_false,
        Protected_false},
    {"keySell",
        defaultAction(&Actions::sell),
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "sell",
        UseArgs_false,
        Protected_false},
    {"keyWhisper2",
        defaultAction(&Actions::msg2),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "w2|whisper2|msg2",
        UseArgs_true,
        Protected_true},
    {"keyGuild",
        defaultAction(&Actions::guild),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "guild",
        UseArgs_true,
        Protected_true},
    {"keyNuke",
        defaultAction(&Actions::chatNuke),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "nuke",
        UseArgs_true,
        Protected_true},
    {"keyUse",
        defaultAction(&Actions::useItem),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "use|useitem",
        UseArgs_true,
        Protected_true},
    {"keyChatAdd",
        defaultAction(&Actions::chatAdd),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "addchat|chatadd",
        UseArgs_true,
        Protected_true},
    {"keyWhisperText",
        defaultAction(&Actions::msgText),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "wt|whispertext|msgtext",
        UseArgs_true,
        Protected_true},
    {"keyNavigateTo",
        defaultAction(&Actions::navigateTo),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "navigateto",
        UseArgs_true,
        Protected_false},
    {"keyCameraMove",
        defaultAction(&Actions::moveCamera),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "movecamera|cameramove",
        UseArgs_true,
        Protected_false},
    {"keyDropItem",
        defaultAction(&Actions::dropItemId),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "dropitem",
        UseArgs_true,
        Protected_true},
    {"keyDropItemAll",
        defaultAction(&Actions::dropItemIdAll),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "dropitemall",
        UseArgs_true,
        Protected_true},
    {"keyDropInv",
        defaultAction(&Actions::dropItemInv),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "dropinv",
        UseArgs_true,
        Protected_true},
    {"keyDropInvAll",
        defaultAction(&Actions::dropItemInvAll),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "dropinvall",
        UseArgs_true,
        Protected_true},
    {"keyUseInv",
        defaultAction(&Actions::useItemInv),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "useinv",
        UseArgs_true,
        Protected_true},
    {"keyInvToStorage",
        defaultAction(&Actions::invToStorage),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "invtostorage|invstorage",
        UseArgs_true,
        Protected_true},
    {"keyTradeAdd",
        defaultAction(&Actions::tradeAdd),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "tradeadd|addtrade",
        UseArgs_true,
        Protected_true},
    {"keyStorageToInv",
        defaultAction(&Actions::storageToInv),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "storagetoinv|storageinv",
        UseArgs_true,
        Protected_true},
    {"keyProtectItem",
        defaultAction(&Actions::protectItem),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "protectitem|itemprotect",
        UseArgs_true,
        Protected_true},
    {"keyUnProtectItem",
        defaultAction(&Actions::unprotectItem),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "unprotectitem|itemunprotect",
        UseArgs_true,
        Protected_true},
    {"keyKickParty",
        defaultAction(&Actions::kickParty),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "kickparty|partykick",
        UseArgs_true,
        Protected_true},
    {"keyAddText",
        defaultAction(&Actions::addText),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "addtext|textadd",
        UseArgs_true,
        Protected_true},
    {"keyKick",
        defaultAction(&Actions::kick),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "kick",
        UseArgs_true,
        Protected_true},
    {"keyLeaveParty",
        defaultAction(&Actions::leaveParty),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "leaveparty|partyleave",
        UseArgs_true,
        Protected_true},
    {"keyWarp",
        defaultAction(&Actions::warp),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "warp",
        UseArgs_true,
        Protected_true},
    {"keyClearChat",
        defaultAction(&Actions::clearChat),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "clearchat|chatclear",
        UseArgs_false,
        Protected_false},
    {"keyPetMoveUp",
        defaultAction(&Actions::petMoveUp),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "petmoveup|moveuppet",
        UseArgs_false,
        Protected_false},
    {"keyPetMoveDown",
        defaultAction(&Actions::petMoveDown),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "petmovedown|movedownpet",
        UseArgs_false,
        Protected_false},
    {"keyPetMoveLeft",
        defaultAction(&Actions::petMoveLeft),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "petmoveleft|moveleftpet",
        UseArgs_false,
        Protected_false},
    {"keyPetMoveRight",
        defaultAction(&Actions::petMoveRight),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "petmoveright|moverightpet",
        UseArgs_false,
        Protected_false},
    {"keyPetDirectUp",
        defaultAction(&Actions::petDirectUp),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "petdirectup|directuppet",
        UseArgs_false,
        Protected_false},
    {"keyPetDirectDown",
        defaultAction(&Actions::petDirectDown),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "petdirectdown|directdownpet",
        UseArgs_false,
        Protected_false},
    {"keyPetDirectLeft",
        defaultAction(&Actions::petDirectLeft),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "petdirectleft|directleftpet",
        UseArgs_false,
        Protected_false},
    {"keyPetDirectRight",
        defaultAction(&Actions::petDirectRight),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "petdirectright|directrightpet",
        UseArgs_false,
        Protected_false},
    {"keyPetAiStart",
        defaultAction(nullptr),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetAiStop",
        defaultAction(nullptr),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGeneralTab",
        defaultAction(&Actions::chatGeneralTab),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "generaltab|generalchattab|general",
        UseArgs_false,
        Protected_false},
    {"keyDebugTab",
        defaultAction(&Actions::chatDebugTab),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "debugtab|debugchattab",
        UseArgs_false,
        Protected_false},
    {"keyBattleTab",
        defaultAction(&Actions::chatBattleTab),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "battletab|battlechattab",
        UseArgs_false,
        Protected_false},
    {"keyTradeTab",
        defaultAction(&Actions::chatTradeTab),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "tradetab|tradechattab",
        UseArgs_false,
        Protected_false},
    {"keyLangTab",
        defaultAction(&Actions::chatLangTab),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "langtab|langchattab",
        UseArgs_false,
        Protected_false},
    {"keyGmTab",
        defaultAction(&Actions::chatGmTab),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "gmtab|gmchattab",
        UseArgs_false,
        Protected_false},
    {"keyPartyTab",
        defaultAction(&Actions::chatPartyTab),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "partytab|partychattab",
        UseArgs_false,
        Protected_false},
    {"keyGuildTab",
        defaultAction(&Actions::chatGuildTab),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "guildtab|guildchattab",
        UseArgs_false,
        Protected_false},
    {"keyDropClear",
        defaultAction(&Actions::clearDrop),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "cleardrops|cleardrop|cleardropwindow|dropclear|dropsclear",
        UseArgs_false,
        Protected_true},
    {"keyWindowCart",
        defaultAction(&Actions::cartWindowShow),
        InputCondition::KEY_DOWN | InputCondition::SHORTCUT0,
        "cart|opencart",
        UseArgs_false,
        Protected_false},
    {"keyMagicHealMostDamaged",
        defaultAction(&Actions::healmd),
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED,
        "healmd|saveass",
        UseArgs_false,
        Protected_false},
    {"keyPetMove",
        defaultAction(&Actions::petMove),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "movepet|petmove",
        UseArgs_true,
        Protected_false},
    {"keyHomunTalk",
        defaultAction(&Actions::homunTalk),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "homuntalk|merctalk",
        UseArgs_true,
        Protected_true},
    {"keyHomunEmote",
        defaultAction(&Actions::commandHomunEmote),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "homunemote|mercemote",
        UseArgs_true,
        Protected_false},
    {"keyHomunSmilie",
        defaultAction(nullptr),
        InputCondition::KEY_DOWN |
            InputCondition::DEFAULT |
            InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::KEY_DOWN | InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyKickGuild",
        defaultAction(&Actions::kickGuild),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "kickguild|guildkick",
        UseArgs_true,
        Protected_true},
    {"keyHat",
        defaultAction(&Actions::hat),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "hat|hats|showhat",
        UseArgs_false,
        Protected_false},
    {"keyCreatePublicRoom",
        defaultAction(&Actions::createPublicChatRoom),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "createroom|createpublicroom",
        UseArgs_true,
        Protected_true},
    {"keyJoinRoom",
        defaultAction(&Actions::joinChatRoom),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "joinroom",
        UseArgs_true,
        Protected_true},
    {"keyLeaveRoom",
        defaultAction(&Actions::leaveChatRoom),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "leaveroom",
        UseArgs_false,
        Protected_true},
    {"keyWindowStatusBar",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::quickWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "yellowbar|quickoptions",
        UseArgs_false,
        Protected_false},
    {"keyWindowMail",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::mailWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "mailwindow|mail",
        UseArgs_false,
        Protected_false},
    {"keyConfSet",
        defaultAction(&Actions::confSet),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "confset|setconf",
        UseArgs_true,
        Protected_true},
    {"keyServerConfSet",
        defaultAction(&Actions::serverConfSet),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "servconfset|setservconf",
        UseArgs_true,
        Protected_true},
    {"keyConfGet",
        defaultAction(&Actions::confGet),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "confget|getconf",
        UseArgs_true,
        Protected_true},
    {"keyServerConfGet",
        defaultAction(&Actions::serverConfGet),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "servconfget|getservconf",
        UseArgs_true,
        Protected_true},
    {"keyChangeTargetingType",
        defaultAction(&Actions::changeTargetingType),
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::VALIDSPEED | InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keyTestInfo",
        defaultAction(&Actions::testInfo),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "testinfo",
        UseArgs_false,
        Protected_true},
    {"keySlide",
        defaultAction(&Actions::slide),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "slide",
        UseArgs_true,
        Protected_true},
    {"keySelectSkillLevel",
        defaultAction(&Actions::selectSkillLevel),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "selectskilllevel",
        UseArgs_true,
        Protected_true},
    {"keySkill",
        defaultAction(&Actions::skill),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "skill|useskill",
        UseArgs_true,
        Protected_true},
    {"keyCraftShortcut1",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::craftKey,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN |
            InputCondition::NOMODAL |
            InputCondition::NOAWAY |
            InputCondition::NONPCINPUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyCraftShortcut2",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::craftKey,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN |
            InputCondition::NOMODAL |
            InputCondition::NOAWAY |
            InputCondition::NONPCINPUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyCraftShortcut3",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::craftKey,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN |
            InputCondition::NOMODAL |
            InputCondition::NOAWAY |
            InputCondition::NONPCINPUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyCraftShortcut4",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::craftKey,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN |
            InputCondition::NOMODAL |
            InputCondition::NOAWAY |
            InputCondition::NONPCINPUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyCraftShortcut5",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::craftKey,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN |
            InputCondition::NOMODAL |
            InputCondition::NOAWAY |
            InputCondition::NONPCINPUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyCraftShortcut6",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::craftKey,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN |
            InputCondition::NOMODAL |
            InputCondition::NOAWAY |
            InputCondition::NONPCINPUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyCraftShortcut7",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::craftKey,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN |
            InputCondition::NOMODAL |
            InputCondition::NOAWAY |
            InputCondition::NONPCINPUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyCraftShortcut8",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::craftKey,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN |
            InputCondition::NOMODAL |
            InputCondition::NOAWAY |
            InputCondition::NONPCINPUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyCraftShortcut9",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::craftKey,
        InputAction::NO_VALUE, 100,
        InputCondition::KEY_DOWN |
            InputCondition::NOMODAL |
            InputCondition::NOAWAY |
            InputCondition::NONPCINPUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyCraft",
        defaultAction(&Actions::craft),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "craft",
        UseArgs_true,
        Protected_true},
    {"keyChatClipboard",
        defaultAction(&Actions::chatClipboard),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "chatclipboard",
        UseArgs_true,
        Protected_true},
    {"keyNpcClipboard",
        defaultAction(&Actions::npcClipboard),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "npcclipboard",
        UseArgs_true,
        Protected_true},
    {"keyAddPickup",
        defaultAction(&Actions::addPickup),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "addpickup|pickupadd",
        UseArgs_true,
        Protected_true},
    {"keyRemovePickup",
        defaultAction(&Actions::removePickup),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "removepickup|pickupremove",
        UseArgs_true,
        Protected_true},
    {"keyIgnorePickup",
        defaultAction(&Actions::ignorePickup),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "ignorepickup|pickupignore",
        UseArgs_true,
        Protected_true},
    {"keyResetModifiers",
        defaultAction(&Actions::resetGameModifiers),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "resetyellowbar|resetmodifiers|resetgamemodifiers",
        UseArgs_false,
        Protected_true},
    {"keyBarToChat",
        defaultAction(&Actions::barToChat),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "bartochat",
        UseArgs_true,
        Protected_true},
    {"keySeenCmd",
        defaultAction(&Actions::seen),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "seen|lastseen",
        UseArgs_true,
        Protected_true},
    {"keyTargetSkillUnit",
        defaultAction(&Actions::targetSkillUnit),
        InputCondition::KEY_DOWN |
            InputCondition::INGAME |
            InputCondition::NOTARGET,
        "targetskillunit|targetskill",
        UseArgs_false,
        Protected_false},
    {"keyMonsterInfo",
        defaultAction(&Actions::monsterInfo),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "monsterinfo|mobinfo|mi",
        UseArgs_true,
        Protected_true},
    {"keyItemInfo",
        defaultAction(&Actions::itemInfo),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "iteminfo|ii",
        UseArgs_true,
        Protected_true},
    {"keyWhoDrops",
        defaultAction(&Actions::whoDrops),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "whodrops|whodrop",
        UseArgs_true,
        Protected_true},
    {"keyMobSearch",
        defaultAction(&Actions::mobSearch),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "mobsearch|monstersearch",
        UseArgs_true,
        Protected_true},
    {"keyMobSpawnSearch",
        defaultAction(&Actions::mobSpawnSearch),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "mobspawnsearch|monsterspawnsearch|whereis",
        UseArgs_true,
        Protected_true},
    {"keyPlayerGmCommands",
        defaultAction(&Actions::playerGmCommands),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "gmcommands|playergmcommands|playercommands",
        UseArgs_true,
        Protected_true},
    {"keyPlayerCharGmCommands",
        defaultAction(&Actions::playerCharGmCommands),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "gmcharcommands|playerchargmcommands|playercharcommands",
        UseArgs_true,
        Protected_true},
    {"keyCommandShowLevel",
        defaultAction(&Actions::commandShowLevel),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "showlevel|exp",
        UseArgs_true,
        Protected_true},
    {"keyCommandShowStats",
        defaultAction(&Actions::commandShowStats),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "showstats|stats",
        UseArgs_true,
        Protected_true},
    {"keyCommandShowStorageList",
        defaultAction(&Actions::commandShowStorage),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "showstorage|storagelist",
        UseArgs_true,
        Protected_true},
    {"keyCommandShowCartList",
        defaultAction(&Actions::commandShowCart),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "showcart|cartlist",
        UseArgs_true,
        Protected_true},
    {"keyCommandShowInventoryList",
        defaultAction(&Actions::commandShowInventory),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "showinventory|inventorylist|itemlist",
        UseArgs_true,
        Protected_true},
    {"keyLocatePlayer",
        defaultAction(&Actions::locatePlayer),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "locate|locateplayer|whereplayer",
        UseArgs_true,
        Protected_true},
    {"keyCommandShowAccountInfo",
        defaultAction(&Actions::commandShowAccountInfo),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "showaccountinfo|showaccinfo|accinfo",
        UseArgs_true,
        Protected_true},
    {"keyCommandSpawn",
        defaultAction(&Actions::commandSpawn),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "spawn|monster",
        UseArgs_true,
        Protected_true},
    {"keyCommandSpawnSlave",
        defaultAction(&Actions::commandSpawnSlave),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "spawnslave|monsterslave|summon",
        UseArgs_true,
        Protected_true},
    {"keyCommandSpawnClone",
        defaultAction(&Actions::commandSpawnClone),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "spawnclone|playerclone|clone",
        UseArgs_true,
        Protected_true},
    {"keyCommandSpawnSlaveClone",
        defaultAction(&Actions::commandSpawnSlaveClone),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "spawnslaveclone|playerslaveclone|slaveclone",
        UseArgs_true,
        Protected_true},
    {"keyCommandSpawnEvilClone",
        defaultAction(&Actions::commandSpawnEvilClone),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "spawnevilclone|playerevilclone|evilclone",
        UseArgs_true,
        Protected_true},
    {"keyCommandSavePosition",
        defaultAction(&Actions::commandSavePosition),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "savepos|saveposition",
        UseArgs_true,
        Protected_true},
    {"keyCommandLoadPosition",
        defaultAction(&Actions::commandLoadPosition),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "loadpos|loadposition",
        UseArgs_true,
        Protected_true},
    {"keyCommandRandomWarp",
        defaultAction(&Actions::commandRandomWarp),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "randomwarp|jump",
        UseArgs_true,
        Protected_true},
    {"keyCommandGotoNpc",
        defaultAction(&Actions::commandGotoNpc),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "gotonpc|warptonpc",
        UseArgs_true,
        Protected_true},
    {"keyCommandKiller",
        defaultAction(&Actions::commandKiller),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "setkiller|killer",
        UseArgs_true,
        Protected_true},
    {"keyCommandKillable",
        defaultAction(&Actions::commandKillable),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "setkillable|killable",
        UseArgs_true,
        Protected_true},
    {"keyCommandHeal",
        defaultAction(&Actions::commandHeal),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "gmheal",
        UseArgs_true,
        Protected_true},
    {"keyCommandAlive",
        defaultAction(&Actions::commandAlive),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "alive|revive",
        UseArgs_true,
        Protected_true},
    {"keyCommandDisguise",
        defaultAction(&Actions::commandDisguise),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "disguise",
        UseArgs_true,
        Protected_true},
    {"keyCommandImmortal",
        defaultAction(&Actions::commandImmortal),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "immortal",
        UseArgs_true,
        Protected_true},
    {"keyCommandHide",
        defaultAction(&Actions::commandHide),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "hide",
        UseArgs_true,
        Protected_true},
    {"keyCommandNuke",
        defaultAction(&Actions::commandNuke),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "gmnuke",
        UseArgs_true,
        Protected_true},
    {"keyCommandKill",
        defaultAction(&Actions::commandKill),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "kill",
        UseArgs_true,
        Protected_true},
    {"keyCommandJail",
        defaultAction(&Actions::commandJail),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "jail",
        UseArgs_true,
        Protected_true},
    {"keyCommandUnjail",
        defaultAction(&Actions::commandUnjail),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "unjail",
        UseArgs_true,
        Protected_true},
    {"keyCommandNpcMove",
        defaultAction(&Actions::commandNpcMove),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "npcmove|movenpc|warpnpc|npcwarp",
        UseArgs_true,
        Protected_true},
    {"keyCommandNpcHide",
        defaultAction(&Actions::commandNpcHide),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "npchide|hidenpc",
        UseArgs_true,
        Protected_true},
    {"keyCommandNpcShow",
        defaultAction(&Actions::commandNpcShow),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "npcshow|shownpc",
        UseArgs_true,
        Protected_true},
    {"keyCommandChangePartyLeader",
        defaultAction(&Actions::commandChangePartyLeader),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "gmpartyleader|changepartyleader",
        UseArgs_true,
        Protected_true},
    {"keyCommandPartyRecall",
        defaultAction(&Actions::commandPartyRecall),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "partyrecall|recallparty",
        UseArgs_true,
        Protected_true},
    {"keyCommandBreakGuild",
        defaultAction(&Actions::commandBreakGuild),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "breakguild|gmbreakguild",
        UseArgs_true,
        Protected_true},
    {"keyCommandGuildRecall",
        defaultAction(&Actions::commandGuildRecall),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "guildrecall|recallguild",
        UseArgs_true,
        Protected_true},
    {"keyMailTo",
        defaultAction(&Actions::mailTo),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "mailto",
        UseArgs_true,
        Protected_true},
    {"keyAdoptChild",
        defaultAction(&Actions::adoptChild),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "adoptchild|childadopt|askforchild",
        UseArgs_true,
        Protected_true},
    {"keyDumpMemmoryUsage",
        defaultAction(&Actions::dumpMemoryUsage),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "dumpmemoryusage|memoryusage",
        UseArgs_false,
        Protected_true},
    {"keySetEmoteType",
        defaultAction(&Actions::setEmoteType),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "setemotetype",
        UseArgs_true,
        Protected_true},
    {"keyCameraRestore",
        defaultAction(&Actions::restoreCamera),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "restorecamera|returncamera|camerarestore",
        UseArgs_false,
        Protected_false},
    {"keyShowSkillLevels",
        defaultAction(&Actions::showSkillLevels),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "showskilllevels",
        UseArgs_true,
        Protected_true},
    {"keyShowskilltype",
        defaultAction(&Actions::showSkillType),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "showskilltypes",
        UseArgs_true,
        Protected_true},
    {"keySelectSkillType",
        defaultAction(&Actions::selectSkillType),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "selectskilltype",
        UseArgs_true,
        Protected_true},
    {"keyShowSkillOffsetX",
        defaultAction(&Actions::showSkillOffsetX),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "showskilloffsetx",
        UseArgs_true,
        Protected_true},
    {"keyShowSkillOffsetY",
        defaultAction(&Actions::showSkillOffsetY),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "showskilloffsety",
        UseArgs_true,
        Protected_true},
    {"keySetSkillOffsetX",
        defaultAction(&Actions::setSkillOffsetX),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "setskilloffsetx",
        UseArgs_true,
        Protected_true},
    {"keySetSkillOffsetY",
        defaultAction(&Actions::setSkillOffsetY),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "setskilloffsety",
        UseArgs_true,
        Protected_true},
    {"keyLeaveGuild",
        defaultAction(&Actions::leaveGuild),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "leaveguild|guildleave",
        UseArgs_false,
        Protected_true},
    {"keyGuildNotice",
        defaultAction(&Actions::guildNotice),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "notice|guildnotice",
        UseArgs_true,
        Protected_true},
    {"keyPartyItemShare",
        defaultAction(&Actions::partyItemShare),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "itemshare|shareitem",
        UseArgs_true,
        Protected_true},
    {"keyPartyExpShare",
        defaultAction(&Actions::partyExpShare),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "expshare|shareexp",
        UseArgs_true,
        Protected_true},
    {"keyPartyAutoItemShare",
        defaultAction(&Actions::partyAutoItemShare),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "autoitemshare|shareautoitem|itemautoshare",
        UseArgs_true,
        Protected_true},
    {"keyCreateItem",
        defaultAction(&Actions::createItem),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "createitem",
        UseArgs_true,
        Protected_true},
    {"keyCopyOutfitToChat",
        defaultAction(&Actions::outfitToChat),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "outfittochat|copyoutfittochat",
        UseArgs_false,
        Protected_true},
    {"keyCommandTranslate",
        defaultAction(&Actions::translate),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "translate|trans|tran",
        UseArgs_false,
        Protected_true},
    {"keyCommandSendGuiKey",
        defaultAction(&Actions::sendGuiKey),
        InputCondition::KEY_DOWN | InputCondition::ENABLED,
        "guikey|sendguikey",
        UseArgs_true,
        Protected_true},
    {"keyMoveAttackUp",
        defaultAction(&Actions::moveAttackUp),
        InputCondition::KEY_DOWN | InputCondition::ENABLED,
        "attackmoveup|moveattackup",
        UseArgs_true,
        Protected_true},
    {"keyMoveAttackDown",
        defaultAction(&Actions::moveAttackDown),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "attackmovedown|moveattackdown",
        UseArgs_true,
        Protected_true},
    {"keyMovePriorityAttackUp",
        defaultAction(&Actions::movePriorityAttackUp),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "attackmovepriorityup|movepriorityattackup",
        UseArgs_true,
        Protected_true},
    {"keyMovePriorityAttackDown",
        defaultAction(&Actions::movePriorityAttackDown),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "attackmoveprioritydown|movepriorityattackdown",
        UseArgs_true,
        Protected_true},
    {"keyShowItems",
        defaultAction(&Actions::showItems),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "showitems|items|itemsshow",
        UseArgs_true,
        Protected_true},
    {"keyCommandSendMouseKey",
        defaultAction(&Actions::sendMouseKey),
        InputCondition::KEY_DOWN | InputCondition::ENABLED,
        "mousekey|sendmousekey",
        UseArgs_true,
        Protected_true},
    {"keyCommandSendChars",
        defaultAction(&Actions::sendChars),
        InputCondition::KEY_DOWN | InputCondition::ENABLED,
        "sendchar|sendchars|charsend",
        UseArgs_true,
        Protected_true},
    {"keyClearOutfit",
        defaultAction(&Actions::outfitClear),
        InputCondition::KEY_DOWN |
            InputCondition::DEFAULT |
            InputCondition::INGAME,
        "clearoutfit|outfitclear",
        UseArgs_false,
        Protected_true},
    {"keyClipboardCopy",
        defaultAction(&Actions::clipboardCopy),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "clipboardcopy|copyclipboard",
        UseArgs_true,
        Protected_true},
    {"keyGotoPc",
        defaultAction(&Actions::commandGotoPc),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "goto|gotopc",
        UseArgs_true,
        Protected_true},
    {"keyCommandRecallPc",
        defaultAction(&Actions::commandRecallPc),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "recall|recallpc",
        UseArgs_true,
        Protected_true},
    {"keyCommandIpCheck",
        defaultAction(&Actions::commandIpCheck),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "ipcheck|checkip",
        UseArgs_true,
        Protected_true},
    {"keyWindowServerInfo",
        defaultAction(&Actions::serverInfoWindowShow),
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "serverinfo|infoserver",
        UseArgs_false,
        Protected_false},
    {"keyMercenaryToMaster",
        defaultAction(&Actions::mercenaryToMaster),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "mercenarytomaster|mercmaster",
        UseArgs_false,
        Protected_false},
    {"keyHomunculusToMaster",
        defaultAction(&Actions::homunculusToMaster),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "homunculustomaster|homunmaster",
        UseArgs_false,
        Protected_false},
    {"keyHomunculusFeed",
        defaultAction(&Actions::homunculusFeed),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "homunculusfeed|homunfeed|feedhomun|feedhomunculus",
        UseArgs_false,
        Protected_false},
    {"keyPetFeed",
        defaultAction(&Actions::petFeed),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "petfeed|feedpet",
        UseArgs_false,
        Protected_false},
    {"keyPetDropLoot",
        defaultAction(&Actions::petDropLoot),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "petdroploot|petloot",
        UseArgs_false,
        Protected_true},
    {"keyPetReturnToEgg",
        defaultAction(&Actions::petReturnToEgg),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "pettoegg|petegg|petreturn|returnpet|petreturnegg",
        UseArgs_false,
        Protected_false},
    {"keyPetUnequip",
        defaultAction(&Actions::petUnequip),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "petunequip|unequippet",
        UseArgs_false,
        Protected_false},
    {"keyAddSkillShortcut",
        defaultAction(&Actions::addSkillShortcut),
        InputCondition::KEY_DOWN | InputCondition::INGAME,
        "addskillshortcut|skillshortcut",
        UseArgs_true,
        Protected_false},
    {"keyWindowClan",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::clanWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::KEY_DOWN |
            InputCondition::GAME |
            InputCondition::NOTARGET,
        "clanwindow|clan",
        UseArgs_false,
        Protected_false},
};

#undef defaultAction
#undef addKey
#undef joystickButton

#endif  // INPUT_INPUTACTIONMAP_H
