# Copyright 2013-2014 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Manage a KVM machine."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

__metaclass__ = type
__all__ = [
    "KVMFixture",
    ]


from datetime import timedelta
import itertools
import logging
import os.path
from pipes import quote
import textwrap
from uuid import uuid1

from fixtures import Fixture
from lxml import etree
from maastest.utils import (
    DEFAULT_STATE_DIR,
    extract_mac_ip_mapping,
    retries,
    run_command,
    )
import netaddr
from six import text_type
from testtools.content import text_content

# Maximum wait for the virtual machine to complete its cloud-init
# initialization.  It can take quite a while, because it performs an
# apt-get update.
CLOUDINIT_TIMEOUT = timedelta(minutes=10)

# The virtual machine will create this file on its filesystem once cloud-init
# is done, and the machine is ready for use.
CLOUDINIT_COMPLETION_MARKER = '/var/lib/cloud/instance/boot-finished'

# Locale to use on the virtual machine.  It needs to be one that is defined in
# the virtual machine, or postgres installation will break, as may other
# things.
#
# See bug 969462 and this discussion:
#   https://lists.ubuntu.com/archives/ubuntu-devel/2013-June/037195.html
VM_LOCALE = "en_US.UTF-8"

# Standard, non-interactive, "apt-get install" line.
APT_GET_INSTALL = [
    'sudo', 'DEBIAN_FRONTEND=noninteractive',
    'apt-get', 'install', '-y',
    ]

# Interface NATed to the host's network.
NATED_INTERFACE = 'eth0'


# Interface connected to the 'direct' network.
DIRECT_INTERFACE = 'eth1'


class KVMFixture(Fixture):
    """A fixture for a Kernel-based Virtual Machine.

    This fixture uses `uvtool` to get the images and create the virtual
    machine.
    """
    def __init__(self, series, architecture, proxy_url,
                 direct_interface=None, direct_network=None, name=None,
                 archives=None, ssh_key_dir=None, kvm_timeout=None,
                 simplestreams_filter=None):
        """Initialise a KVM-based Virtual Machine fixture.

        The VM's network is configured like this:
            - eth0 is a NATed interface which gets its IP from the
              KVM-provided dnsmasq server.
            - eth1 only exists if `direct_interface` is supplied.  In
              that case eth1 is bridged to the interface `direct_interface`
              on the host, and has a fixed IP (`direct_ip`).

        :param series: The Ubuntu series to run in VM (e.g. 'saucy').
        :type series: string
        :param architecture: The architecture of the VM (e.g. 'amd64').
        :type architecture: string
        :param proxy_url: The URL of an HTTP proxy.
        :type proxy_url: LocalProxyFixture
        :param name: An optional name for the VM.  If not provided, a
            random one will be generated.
        :type name: string
        :param direct_interface: An optional network interface name.  If
            provided, the VM will have a second interface (eth1) directly
            attached to the named physical interface of the host machine.
            If this is provided, the caller can provide a for
            'direct_network' to govern how the interface's network will be
            configured.
        :type direct_interface: string
        :param direct_network: An optional network definition.  This
            parameter is ignored if the parameter 'direct_interface' is
            not provided.  It is used to configure the network of the
            the eth1 interace of the VM instance.  If 'direct_interface' is
            provided and 'direct_network' is not provided, this will default
            to the network 192.168.2.0/24.
        :type direct_network: netaddr.IPNetwork
        :param archives: An optional list of repository names.  If provided,
            each repository will be added onto the VM.
        :type archives: list
        :param ssh_key_dir: Optional directory in which to store the SSH keys
            for connecting to this KVM instance.  Defaults to
            `/var/cache/maas-test`.
        :type ssh_key_dir: string.
        :param kvm_timeout: The number of seconds to wait for a KVM
            instance to start.
        :type kvm_timeout: int
        :param simplestreams_filter: Optional simplestreams filter use to
            retrieve the VM images.  This is a string using the standard
            simplestreams format: space-separated filter constraints (e.g.
            "label=alpha1 filter2=value").
        :type simplestreams_filter: string
        """
        super(KVMFixture, self).__init__()
        self.command_count = itertools.count(1)
        self.archives = archives
        self.series = series
        self.simplestreams_filter = simplestreams_filter
        self.architecture = architecture
        self.proxy_url = proxy_url
        self.direct_interface = direct_interface
        self.direct_network = direct_network
        self._ip_address = None
        if direct_interface is not None:
            if direct_network is None:
                # Generate a 'direct' private network.
                self.direct_network = netaddr.IPNetwork('192.168.2.0/24')
            self.direct_ip = "%s" % (
                netaddr.IPAddress(self.direct_network.first + 1))
        else:
            self.direct_ip = None
        if name is None:
            name = text_type(uuid1())
        self.name = name
        if ssh_key_dir is None:
            ssh_key_dir = DEFAULT_STATE_DIR
        self.ssh_private_key_file = os.path.join(
            ssh_key_dir, 'vm_ssh_id_rsa')
        self.kvm_timeout = kvm_timeout
        self.running = False

    def setUp(self):
        super(KVMFixture, self).setUp()
        self.import_image()
        self.generate_ssh_key()
        self.start()
        # From this point, if further setup fails, we'll need to clean up.
        self.addCleanup(self.destroy)

        self.wait_for_vm()
        self.wait_for_cloudinit()
        self.configure_network()
        self.configure_archives()
        self.install_base_packages()

        logging.info("Virtual machine %s is ready.", self.name)

    def direct_first_available_ip(self):
        """Returns the first available IP on the 'direct' network.

        The 'direct' network is the network directly connected to the host's
        machine."""
        return "%s" % netaddr.IPAddress(self.direct_network.first + 2)

    def identity(self):
        """Return the identity (user@ip-address) for this machine.

        This is the identity that can be used to ssh into this machine."""
        return "ubuntu@%s" % self.ip_address()

    def ip_address(self):
        """Return the IP address of the KVM instance.

        The IP address is discovered by calling uvt-kvm ip <name> for
        the KVM instance. Once the IP address has been fetched once it
        is cached here, so uvt-kvm ip need not be called again.
        """
        if self._ip_address is None:
            _, ip_address, _ = run_command(
                ["uvt-kvm", "ip", self.name], check_call=True)
            self._ip_address = ip_address.strip().decode("ascii")
        return self._ip_address

    def get_ssh_public_key_file(self):
        """Return path to public SSH key file, for VM sshd configuration."""
        return self.ssh_private_key_file + '.pub'

    def get_simplestreams_filter(self):
        """Return a list with the simplestreams filters for this VM."""
        filters = [
            "arch=%s" % self.architecture,
            "release=%s" % self.series,
        ]
        if self.simplestreams_filter is not None:
            filters.extend(self.simplestreams_filter.split())
        return filters

    def import_image(self):
        """Download the image required to create the virtual machine."""
        logging.info(
            "Downloading KVM image for %s..." % ' '.join(
                self.get_simplestreams_filter()))
        command = [
            "sudo",
            "http_proxy=%s" % self.proxy_url,
            "uvt-simplestreams-libvirt", "sync",
        ] + self.get_simplestreams_filter()
        run_command(command, check_call=True)
        logging.info(
            "Done downloading KVM image.")

    def generate_ssh_key(self):
        """Set up an ssh key pair for accessing the virtual machine.

        The key pair will be stored in the SSH key directory `ssh_key_dir`.
        If this did not exist yet, it will be created.
        """
        key_dir = os.path.dirname(self.ssh_private_key_file)
        if not os.path.exists(key_dir):
            os.makedirs(key_dir, mode=0o700)
        if not os.path.exists(self.ssh_private_key_file):
            run_command(
                [
                    'ssh-keygen',
                    '-t', 'rsa',
                    '-f', self.ssh_private_key_file,
                    '-N', '',
                ],
                check_call=True)

    def wait_for_vm(self):
        """Wait for the virtual machine to come up."""
        # Block until uvtool gives us the machine.  This really only blocks
        # until its SSH server comes up.
        logging.info("Waiting for the virtual machine to come up...")
        if self.kvm_timeout is not None:
            command = [
                'uvt-kvm', 'wait', '--timeout', unicode(self.kvm_timeout),
                self.name]
        else:
            command = ['uvt-kvm', 'wait', self.name]
        run_command(command, check_call=True)
        logging.info("Virtual machine is running.")

    def wait_for_cloudinit(self):
        """Wait for cloud-init to finish its work on the virtual machine."""
        logging.info("Waiting for cloud-init to finish its work...")
        for _ in retries(timeout=CLOUDINIT_TIMEOUT.total_seconds(), delay=5):
            return_code, _, stderr = self.run_command(
                ['test', '-f', CLOUDINIT_COMPLETION_MARKER])
            if return_code == 0:
                logging.info("Cloud-init run finished.")
                return
            # If the file does not exist, "test" returns 1.  If we get some
            # other nonzero return code, that means there was definitely a
            # real error.  In which case, let's not keep the user waiting for
            # the full timeout period.
            if return_code != 1:
                raise RuntimeError(
                    "Error contacting virtual machine %s.  "
                    "Error output was: '%s'"
                    % (self.name, stderr))

        raise RuntimeError(
            "Cloud-init initialization on virtual machine %s timed out.  "
            "Error output was: '%s'"
            % (self.name, stderr))

    def start(self):
        """Create and start the virtual machine.

        When this method returns, the VM is ready to be used."""
        logging.info(
            "Creating virtual machine %s, arch=%s...", self.name,
            self.architecture)
        template = make_kvm_template(self.direct_interface)
        command = [
            "sudo", "uvt-kvm", "create",
            "--ssh-public-key-file=%s" % self.get_ssh_public_key_file(),
            "--unsafe-caching",
            # On i386 hosts, qemu errors when the memory is > 2047 MB.
            "--memory", "2047",
            "--disk", "20",
            self.name
        ] + self.get_simplestreams_filter()
        command = command + ["--template", "-"]
        run_command(command, input=template, check_call=True)
        self.running = True
        logging.info(
            "Done creating virtual machine %s, arch=%s.", self.name,
            self.architecture)

    def configure_archives(self):
        """Add repositories to the VM and update the list of packages."""
        if self.archives is not None:
            for archive in self.archives:
                self.run_command(
                    ["sudo", "add-apt-repository", "-y", archive],
                    check_call=True)
        self.run_command(
            ["sudo", "apt-get", "update"], check_call=True)

    def configure_network(self):
        """Configure the virtual machine network.

        If this fixture was configured to create a bridged interface on the
        VM, this method configures the network: it configures and brings up
        the bridged network interface, eth1.
        """
        # Configure the bridged network if self.direct_interface is not
        # None.
        if self.direct_interface is not None:
            logging.info(
                "Configuring network interface on virtual machine %s...",
                self.name)
            network_config_snippet = make_network_interface_config(
                DIRECT_INTERFACE, self.direct_ip, self.direct_network,
                NATED_INTERFACE)
            self.run_command(
                ["sudo", "tee", "-a", "/etc/network/interfaces"],
                input=network_config_snippet, check_call=True)
            self.run_command(
                ["sudo", "ifup", DIRECT_INTERFACE], check_call=True)
            logging.info(
                "Done configuring network interface on virtual machine %s",
                self.name)

    def destroy(self):
        """Destroy the virtual machine."""
        logging.info("Destroying virtual machine %s...", self.name)
        run_command(
            ["sudo", "uvt-kvm", "destroy", self.name], check_call=True)
        self.running = False
        logging.info("Done destroying virtual machine %s.", self.name)

    def _get_base_ssh_options(self):
        """Return a list of ssh options for connecting to the VM.

        These are minimal options that are needed to connect at all, whether
        using `ssh` or `scp`.
        """
        return [
            "-i", self.ssh_private_key_file,
            "-o", "LogLevel=quiet",
            "-o", "UserKnownHostsFile=/dev/null",
            "-o", "StrictHostKeyChecking=no",
            ]

    def _make_ssh_command(self, command_line):
        """Compose an `ssh` command line to execute `command_line` in the VM.

        :param command_line: Shell command to be executed on the virtual
            machine, in the form of a sequence of arguments (starting with the
            executable).
        :return: An `ssh` command line to execute `command_line` on the virtual
            machine, in the form of a list.
        """
        remote_command_line = ' '.join(map(quote, command_line))
        return [
            "ssh",
            ] + self._get_base_ssh_options() + [
            self.identity(),
            "LC_ALL=%s" % VM_LOCALE,
            remote_command_line,
            ]

    def _make_apt_get_install_command(self, packages):
        """Install the given packages on the virtual machine."""
        return [
            'sudo',
            'http_proxy=%s' % self.proxy_url,
            'https_proxy=%s' % self.proxy_url,
            'DEBIAN_FRONTEND=noninteractive',
            'apt-get', 'install', '-y'
            ] + list(packages)

    def install_packages(self, packages):
        """Install the given packages on the virtual machine."""
        self.run_command(
            self._make_apt_get_install_command(packages),
            check_call=True)

    def install_base_packages(self):
        """Install the packages needed for this instance to work."""
        # Instal nmap so that 'get_ip_from_network_scan' can work.
        self.install_packages(['nmap'])

    def run_command(self, args, input=None, check_call=False):
        """Run the given command in the VM."""
        args = self._make_ssh_command(args)
        retcode, stdout, stderr = run_command(
            args, check_call=check_call, input=input)
        count = next(self.command_count)
        cmd_prefix = 'cmd #%04d' % count
        self.addDetail(cmd_prefix, text_content(' '.join(args)))
        self.addDetail(cmd_prefix + ' retcode', text_content(str(retcode)))
        input_detail = '' if input is None else input
        self.addDetail(cmd_prefix + ' input', text_content(input_detail))
        u_stdout = unicode(stdout, errors='replace')
        self.addDetail(cmd_prefix + ' stdout', text_content(u_stdout))
        u_stderr = unicode(stderr, errors='replace')
        self.addDetail(cmd_prefix + ' stderr', text_content(u_stderr))
        return retcode, stdout, stderr

    def upload_file(self, source, dest=None):
        """Copy the given file into the virtual machine's filesystem.

        :param source: File to copy.
        :param dest: Optional destination.  May be an absolute path, or one
            relative to the `ubuntu` user's home directory.  Defaults to the
            same base name as `source`, directly in the user's home directory.
        """
        if dest is None:
            dest = os.path.basename(source)
        run_command(
            ['scp'] + self._get_base_ssh_options() + [source, dest],
            check_call=True)

    def get_ip_from_network_scan(self, mac_address):
        """Return the IP address associated with a MAC address.

        The IP address is found by scanning the direct network using nmap.
        """
        network_repr = "%s" % self.direct_network
        nmap_scan_cmd = ['sudo', 'nmap', '-sP', network_repr, '-oX', '-']
        _, output, _ = self.run_command(
            nmap_scan_cmd, check_call=True)
        mapping = extract_mac_ip_mapping(output)
        ip = mapping.get(mac_address.upper())
        if ip is not None:
            return ip
        return None

KVM_TEMPLATE = """
  <domain type='kvm'>
    <os>
      <type>hvm</type>
      <boot dev='hd'/>
    </os>
    <devices>
      <interface type='network'>
        <source network='default'/>
        <model type='virtio'/>
      </interface>
      <serial type='pty'>
        <source path='/dev/pts/3'/>
        <target port='0'/>
      </serial>
      <graphics type='vnc' autoport='yes' listen='127.0.0.1'>
        <listen type='address' address='127.0.0.1'/>
      </graphics>
      <video/>
    </devices>
  </domain>
"""

KVM_DIRECT_INTERFACE_TEMPLATE = """
  <interface type='direct'>
      <source dev='%s' mode='vepa'/>
  </interface>
"""


def make_kvm_template(direct_interface=None):
    """Return a KVM template suitable for 'uvt-kvm create'.

    :param direct_interface: An optional interface name.  If provided, the
        returned template will include provision for a direct attachment of
        the virtual machine's NIC to the named physical interface.
        See http://libvirt.org/formatdomain.html#elementsNICSDirect for
        details.
    """
    parser = etree.XMLParser(remove_blank_text=True)
    xml_template = etree.fromstring(KVM_TEMPLATE, parser)
    if direct_interface is not None:
        xml_interface_template = etree.fromstring(
            KVM_DIRECT_INTERFACE_TEMPLATE % direct_interface, parser)
        interface_tags = xml_template.xpath("/domain/devices/interface")
        interface_tags[-1].addnext(xml_interface_template)
    return etree.tostring(xml_template)


NETWORK_CONFIG_TEMPLATE = textwrap.dedent("""
    auto %(direct_interface)s
    iface %(direct_interface)s inet static
    address %(ip_address)s
    network %(network)s
    netmask %(netmask)s
    broadcast %(broadcast)s
    post-up sysctl -w net.ipv4.ip_forward=1
    post-up iptables -t nat -A POSTROUTING -o %(interface)s -j MASQUERADE
    post-up iptables -A FORWARD -i %(interface)s -o %(direct_interface)s \
-m state --state RELATED,ESTABLISHED -j ACCEPT
    post-up iptables -A FORWARD -i %(direct_interface)s -o %(interface)s \
-j ACCEPT
    """)


def make_network_interface_config(direct_interface, ip_address,
                                  network, interface):
    """Return a network configuration snippet.

    The returned snippet defines a fixed-IP address network configuration
    for the given interface suitable for inclusion in
    /etc/network/interfaces.
    """
    return NETWORK_CONFIG_TEMPLATE % {
        'interface': interface,
        'ip_address': ip_address,
        'network': network.network,
        'netmask': network.netmask,
        'broadcast': network.broadcast,
        'direct_interface': direct_interface,
    }
