/**
 * \file pappsomspp/mass_range.h
 * \date 4/3/2015
 * \author Olivier Langella
 * \brief object to handle a mass range (an mz value + or - some delta)
 */

/*******************************************************************************
 * Copyright (c) 2015 Olivier Langella <Olivier.Langella@moulon.inra.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contributors:
 *     Olivier Langella <Olivier.Langella@moulon.inra.fr> - initial API and
 *implementation
 ******************************************************************************/

#pragma once

#include <map>

#include <QString>

#include "pappsomspp/export-import-config.h"
#include "pappsomspp/core/types.h"

namespace pappso
{

class PMSPP_LIB_DECL PrecisionBase
{
  protected:
  const pappso_double m_nominal;

  PrecisionBase(pappso_double nominal) : m_nominal(nominal)
  {
  }

  public:
  virtual Enums::PrecisionUnit unit() const = 0;
  virtual pappso_double getNominal() const final;
  virtual pappso_double delta(pappso_double value) const = 0;
  virtual QString toString() const                       = 0;
};


/** \def specific type for a dalton precision
 *
 */
class PMSPP_LIB_DECL DaltonPrecision : public PrecisionBase
{
  friend class PrecisionFactory;

  protected:
  DaltonPrecision(pappso_double x);

  public:
  virtual ~DaltonPrecision();

  virtual Enums::PrecisionUnit unit() const override;

  virtual pappso_double delta(pappso_double value) const override;

  virtual QString toString() const override;
};

/** \def specific type for a ppm precision
 *
 */
class PMSPP_LIB_DECL PpmPrecision : public PrecisionBase
{
  friend class PrecisionFactory;

  protected:
  PpmPrecision(pappso_double x);

  public:
  virtual ~PpmPrecision();

  virtual Enums::PrecisionUnit unit() const override;

  virtual pappso_double delta(pappso_double value) const override;

  virtual QString toString() const override;
};


/** \def specific type for a res precision
 *
 */
class PMSPP_LIB_DECL ResPrecision : public PrecisionBase
{
  friend class PrecisionFactory;

  protected:
  ResPrecision(pappso_double x);

  public:
  virtual ~ResPrecision();

  virtual Enums::PrecisionUnit unit() const override;

  virtual pappso_double delta(pappso_double value) const override;

  virtual QString toString() const override;
};


typedef const PrecisionBase *PrecisionPtr;


// was class Precision
class PMSPP_LIB_DECL PrecisionFactory
{
  using MapDaltonPrecision = std::map<pappso_double, DaltonPrecision *>;
  using MapPpmPrecision    = std::map<pappso_double, PpmPrecision *>;
  using MapResPrecision    = std::map<pappso_double, ResPrecision *>;

  private:
  static MapDaltonPrecision m_mapDalton;
  static MapPpmPrecision m_mapPpm;
  static MapResPrecision m_mapRes;

  public:
  /** @brief get a precision pointer from a string
   * @param str is of the form : "nominal_value unit", example : "0.02 dalton"
   */
  static PrecisionPtr fromString(const QString &str);


  /** @brief get a Dalton precision pointer
   * @param value the precision value in dalton
   */
  static PrecisionPtr getDaltonInstance(pappso_double value);


  /** @brief get a ppm precision pointer
   * @param value the precision value in ppm
   */
  static PrecisionPtr getPpmInstance(pappso_double value);

  /** @brief get a resolution precision pointer
   * @param value the precision value in resolution
   */
  static PrecisionPtr getResInstance(pappso_double value);


  /** @brief get the fraction of an existing precision pointer
   * @param origin the original precision pointer
   * @param fraction the fraction to apply on the original precision
   */
  static PrecisionPtr getPrecisionPtrFractionInstance(PrecisionPtr origin,
                                                      double fraction);

  /** @brief get a precision pointer instance
   * @param unit the precision unit
   * @param value the precision value
   */
  static PrecisionPtr getPrecisionPtrInstance(Enums::PrecisionUnit unit, double value);
};

} // namespace pappso
