/*
 * Copyright (c) 2007, Dennis M. Sosnoski All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 * 
 * Redistributions of source code must retain the above copyright notice, this list of conditions and the following
 * disclaimer. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 * following disclaimer in the documentation and/or other materials provided with the distribution. Neither the name of
 * JiBX nor the names of its contributors may be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.jibx.binding.generator;

import org.apache.bcel.generic.Type;
import org.jibx.binding.model.IClassItem;
import org.jibx.binding.util.StringArray;
import org.jibx.runtime.IUnmarshallingContext;

/**
 * Collection field customization information.
 * 
 * @author Dennis M. Sosnoski
 */
public class CollectionPropertyCustom extends MemberPropertyCustom
{
    /** Enumeration of allowed attribute names */
    public static final StringArray s_allowedAttributes =
        new StringArray(new String[] { "item-name", "item-type" }, MemberPropertyCustom.s_allowedAttributes);
    
    // values specific to member level
    private String m_itemType;
    
    private String m_itemName;
    
    /**
     * Constructor.
     * 
     * @param parent
     */
    protected CollectionPropertyCustom(NestingBase parent) {
        super(parent);
    }
    
    /**
     * Constructor with name known.
     * 
     * @param parent
     * @param name
     */
    protected CollectionPropertyCustom(NestingBase parent, String name) {
        super(parent, name);
    }
    
    /**
     * Make sure all attributes are defined.
     * 
     * @param uctx unmarshalling context
     */
    protected void preSet(IUnmarshallingContext uctx) {
        validateAttributes(uctx, s_allowedAttributes);
    }
    
    /**
     * Check if collection member.
     * 
     * @return <code>true</code>
     */
    public boolean isCollection() {
        return true;
    }
    
    /**
     * Get item type.
     * 
     * @return item type (<code>null</code> if none)
     */
    public String getItemType() {
        return m_itemType;
    }
    
    /**
     * Get item element name.
     * 
     * @return item name (<code>null</code> if none)
     */
    public String getItemName() {
        return m_itemName;
    }
    
    /**
     * Complete customization information based on access method information.
     * 
     * @param gmeth read access method (<code>null</code> if none)
     * @param smeth write access method (<code>null</code> if none)
     * @param type value type
     * @param req required member flag (<code>null</code> if unknown)
     * @param style representation style (<code>null</code> if unspecified)
     */
    /* package */void completeProperty(IClassItem gmeth, IClassItem smeth, String type, Boolean req, Integer style) {
        super.completeProperty(gmeth, smeth, type, req, style);
        if (type != null) {
            
            // set information from field
            String tname = getWorkingType();
            if (m_itemType == null) {
                if (tname.endsWith("[]")) {
                    m_itemType = tname.substring(0, tname.length() - 2);
                } else {
                    String sig;
                    if (gmeth == null) {
                        sig = smeth.getGenericsSignature();
                    } else {
                        sig = gmeth.getGenericsSignature();
                    }
                    int start = sig.indexOf('<');
                    int end = sig.lastIndexOf('>');
                    if (start > 0 && end > 0 && start < end) {
                        String tsig = sig.substring(start + 1, end);
                        if (tsig.indexOf('<') < 0) {
                            m_itemType = Type.getType(tsig).toString();
                        }
                    }
                }
            }
            if (m_itemName == null) {
                m_itemName = deriveItemName(getXmlName(), m_itemType, getParent().getNameStyle());
            }
        }
        if (m_itemType == null) {
            m_itemType = "java.lang.Object";
        }
        if (m_itemName == null) {
            m_itemName = "item";
        }
    }
    
    private static CollectionPropertyCustom factory(IUnmarshallingContext ictx) {
        return new CollectionPropertyCustom(getContainingClass(ictx));
    }
}