//
//  Lynkeos 
//  $Id: MyImageList.m,v 1.11 2005/01/27 23:06:02 j-etienne Exp $
//
//  Created by Jean-Etienne LAMIAUD on Wed Sep 24 2003.
//  Copyright (c) 2003-2005. Jean-Etienne LAMIAUD
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

#include "MyImageList.h"

#define K_LIST_KEY		@"images"
#define K_SINGLE_PRECISION_KEY  @"float"
#define K_RAW_STACK_KEY		@"stack"
#define K_BLACK_LEVEL_KEY	@"blackl"
#define K_WHITE_LEVEL_KEY	@"whitel"

@implementation MyImageList

//==============================================================================
// Coding
//==============================================================================

- (void)encodeWithCoder:(NSCoder *)encoder
{
   NSData *stackWrapper = nil;

   if ( _rawStack != nil )
      stackWrapper = [NSData dataWithBytesNoCopy:_rawStack 
                                             length:_stackSize
                                       freeWhenDone:NO];

   // List data
   [encoder encodeObject:_list forKey:K_LIST_KEY];
   // Stack data
   [encoder encodeBool:
#ifdef FLOAT_PIXELS
      YES
#else
      NO
#endif
      forKey:K_SINGLE_PRECISION_KEY];
   [encoder encodeObject:stackWrapper forKey:K_RAW_STACK_KEY];
   [encoder encodeDouble:_blackLevel forKey:K_BLACK_LEVEL_KEY];
   [encoder encodeDouble:_whiteLevel forKey:K_WHITE_LEVEL_KEY];
}

- (id)initWithCoder:(NSCoder *)decoder
{
   [self init];
   
   if ( [decoder containsValueForKey:K_LIST_KEY] )
   {
      NSData *stackWrapper;
      
      // List data
      _list = [[decoder decodeObjectForKey:K_LIST_KEY] retain];
      // Stack data
      stackWrapper = [decoder decodeObjectForKey:K_RAW_STACK_KEY];
      if ( stackWrapper != nil )
      {
         BOOL single = [decoder decodeBoolForKey:K_SINGLE_PRECISION_KEY];
         u_long i;
         REAL *buf;

         _stackSize = [stackWrapper length];

         if ( _stackSize != 0 )
         {
            // Convert, if the precision is different
#ifdef FLOAT_PIXELS
            if ( ! single )
            {
               double *source;

               _stackSize /= sizeof(double)/sizeof(float);
#else
            if ( single )
            {
              float *source;

               _stackSize *= sizeof(double)/sizeof(float);
#endif
               _rawStack = (RGB*)malloc( _stackSize );
               buf = (REAL*)_rawStack;
               source = (typeof(source))[stackWrapper bytes];
               for( i = 0; i < _stackSize/sizeof(REAL); i++ )
                  buf[i] = (REAL)source[i];
            }
            else
            {
               _rawStack = (RGB*)malloc( _stackSize );
               [stackWrapper getBytes:_rawStack];
            }
         }
      }
      if ( [decoder containsValueForKey:K_BLACK_LEVEL_KEY] &&
           [decoder containsValueForKey:K_WHITE_LEVEL_KEY])
      {
         _blackLevel = [decoder decodeDoubleForKey:K_BLACK_LEVEL_KEY];
         _whiteLevel = [decoder decodeDoubleForKey:K_WHITE_LEVEL_KEY];
      }
 
      return( self );
   }
   else
   {
      // File format is not compatible, abort loading
      [self release];
      return( nil );
   }
}

//==============================================================================
// Initializers, Constructors and destructors
//==============================================================================
- (id)init
{
   self = [super init];
   
   if (self)
   {
      // List
      _list = nil;
      // Stack data
      _rawStack = nil;
      _stackSize = 0;
      [self invalidateLevels];
   }
   
   return self;
}

- (id) initWithArray :(NSArray*)list
{
   [self init];
   _list = [list retain];
   
   return( self );
}

+ (id) imageListWithArray :(NSArray*)list
{
   return( [[[self alloc] initWithArray:list] autorelease] );
}

- (void) dealloc
{
   [_list release];
   
   if ( _rawStack != NULL )
      free( _rawStack );
   
   [super dealloc];
}

//==============================================================================
// Read accessors
//==============================================================================
   // List
- (NSMutableArray*) imageArray { return( _list ); }
   // Stack data
- (RGB*) stack { return( _rawStack ); }
- (u_long) stackSize { return(_stackSize); }
- (REAL) blackLevel { return(_blackLevel); }
- (REAL) whiteLevel { return(_whiteLevel); }
- (BOOL) validLevels { return( _whiteLevel > _blackLevel ); }


   // Enumerator access
- (MyImageListItem*) firstItem
{
   MyImageListEnumerator* list = [self imageEnumerator];
   MyImageListItem *item;
   
   while ( (item = [list nextObject]) != nil &&
           [item getSelectionState] != NSOnState )
      ;
   return( item );
}

- (MyImageListItem*) lastItem
{
   MyImageListEnumerator* list = [self imageEnumeratorStartAt:nil
						  directSense:NO];
   MyImageListItem *item;
   
   while ( (item = [list nextObject]) != nil &&
           [item getSelectionState] != NSOnState )
      ;
   return( item );
}

- (MyImageListEnumerator*)imageEnumerator
{
   return( [[[MyImageListEnumerator alloc] initWithImageList:_list] 
      autorelease] );
}

- (MyImageListEnumerator*) imageEnumeratorStartAt:(id)item 
                                      directSense:(BOOL)direct
{
   return( [[[MyImageListEnumerator alloc] initWithImageList:_list 
						     startAt:item directSense:direct]
      autorelease] );
}

//==============================================================================
// Write accessors
//==============================================================================
- (BOOL) setStack :(RGB*)stack size:(u_long)size
{
   // free the previous stack if any
   if ( _rawStack != NULL )
      free( _rawStack );
   _rawStack = stack;
   _stackSize = size;
   
   return( YES );
}
- (BOOL) setBlackLevel:(REAL)black whiteLevel:(REAL)white
{
   if ( _blackLevel == black && _whiteLevel == white )
      return( NO );

   _blackLevel = black;
   _whiteLevel = white;

   return( YES );
}

- (BOOL) invalidateLevels
{
   return( [self setBlackLevel:0.0 whiteLevel:-1.0] );
}


//==============================================================================
// Actions
//==============================================================================

- (BOOL) addItem :(MyImageListItem*)item
{
   if ( _list == nil )
      _list = [[NSMutableArray array] retain];
   
   [_list addObject: item];

   return( YES );
}

- (BOOL) deleteItem :(MyImageListItem*)item
{
   if ( [item isMemberOfClass: [MyMovieImage class]] )
   {
      MyMovie* movie = [(MyMovieImage*)item getParent];
      NSAssert( [_list containsObject:movie], 
                @"Cannot delete from a nonexistent movie!" );
      [movie deleteMovieImage:(MyMovieImage*)item];
   }
   else
   {
      NSAssert( [_list containsObject:item], 
                @"Cannot delete a nonexistent item!" );
      [_list removeObject:item];
   }
   
   return( YES );
}

- (BOOL) changeItemSelection :(MyImageListItem*)item value:(BOOL)v
{
   if ( [item getSelectionState] == (v ? NSOnState : NSOffState) )
      return( NO );
   
   // Set the desired value
   [item setSelected:v];
   
   return( YES );
}

@end
