/* Copyright (c) 2012, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#include "msm_sensor.h"
#define SENSOR_NAME "imx135"
#define PLATFORM_DRIVER_NAME "msm_camera_imx135"
#define imx135_obj imx135_##obj

DEFINE_MUTEX(imx135_mut);
static struct msm_sensor_ctrl_t imx135_s_ctrl;

static struct msm_camera_i2c_reg_conf imx135_start_settings[] = {
	{0x0100, 0x01},
};

static struct msm_camera_i2c_reg_conf imx135_stop_settings[] = {
	{0x0100, 0x00},
};

static struct msm_camera_i2c_reg_conf imx135_groupon_settings[] = {
	{0x104, 0x01},
};

static struct msm_camera_i2c_reg_conf imx135_groupoff_settings[] = {
	{0x104, 0x00},
};

static struct msm_camera_i2c_reg_conf imx135_recommend_settings[] = {
/* Global Settings */
	{0x0101, 0x00},
	{0x0105, 0x01},
	{0x0110, 0x00},
	{0x0220, 0x01},
	{0x3302, 0x11},
	{0x3833, 0x20},
	{0x3893, 0x00},
	{0x3906, 0x08},
	{0x3907, 0x01},
	{0x391B, 0x01},
	{0x3C09, 0x01},
	{0x600A, 0x00},
	{0x3008, 0xB0},
	{0x320A, 0x01},
	{0x320D, 0x10},
	{0x3216, 0x2E},
	{0x322C, 0x02},
	{0x3409, 0x0C},
	{0x340C, 0x2D},
	{0x3411, 0x39},
	{0x3414, 0x1E},
	{0x3427, 0x04},
	{0x3480, 0x1E},
	{0x3484, 0x1E},
	{0x3488, 0x1E},
	{0x348C, 0x1E},
	{0x3490, 0x1E},
	{0x3494, 0x1E},
	{0x3511, 0x8F},
	{0x364F, 0x2D},
	/*Defect Correction Recommended Setting */
	{0x380A, 0x00},
	{0x380B, 0x00},
	{0x4103, 0x00},
	/*Color Artifact Recommended Setting */
	{0x4243, 0x9A},
	{0x4330, 0x01},
	{0x4331, 0x90},
	{0x4332, 0x02},
	{0x4333, 0x58},
	{0x4334, 0x03},
	{0x4335, 0x20},
	{0x4336, 0x03},
	{0x4337, 0x84},
	{0x433C, 0x01},
	{0x4340, 0x02},
	{0x4341, 0x58},
	{0x4342, 0x03},
	{0x4343, 0x52},
	/*Moir reduction Parameter Setting	*/
	{0x4364, 0x0B},
	{0x4368, 0x00},
	{0x4369, 0x0F},
	{0x436A, 0x03},
	{0x436B, 0xA8},
	{0x436C, 0x00},
	{0x436D, 0x00},
	{0x436E, 0x00},
	{0x436F, 0x06},
	/*CNR parameter setting	*/
	{0x4281, 0x21},
	{0x4282, 0x18},
	{0x4283, 0x04},
	{0x4284, 0x08},
	{0x4287, 0x7F},
	{0x4288, 0x08},
	{0x428B, 0x7F},
	{0x428C, 0x08},
	{0x428F, 0x7F},
	{0x4297, 0x00},
	{0x4298, 0x7E},
	{0x4299, 0x7E},
	{0x429A, 0x7E},
	{0x42A4, 0xFB},
	{0x42A5, 0x7E},
	{0x42A6, 0xDF},
	{0x42A7, 0xB7},
	{0x42AF, 0x03},
	/*ARNR Parameter Setting*/
	{0x4207, 0x03},
	{0x4216, 0x08},
	{0x4217, 0x08},
	/*DLC Parameter Setting*/
	{0x4218, 0x00},
	{0x421B, 0x20},
	{0x421F, 0x04},
	{0x4222, 0x02},
	{0x4223, 0x22},
	{0x422E, 0x54},
	{0x422F, 0xFB},
	{0x4230, 0xFF},
	{0x4231, 0xFE},
	{0x4232, 0xFF},
	{0x4235, 0x58},
	{0x4236, 0xF7},
	{0x4237, 0xFD},
	{0x4239, 0x4E},
	{0x423A, 0xFC},
	{0x423B, 0xFD},
	/*HDR Setting*/
	{0x4300, 0x00},
	{0x4316, 0x12},
	{0x4317, 0x22},
	{0x4318, 0x00},
	{0x4319, 0x00},
	{0x431A, 0x00},
	{0x4324, 0x03},
	{0x4325, 0x20},
	{0x4326, 0x03},
	{0x4327, 0x84},
	{0x4328, 0x03},
	{0x4329, 0x20},
	{0x432A, 0x03},
	{0x432B, 0x20},
	{0x432C, 0x01},
	{0x432D, 0x01},
	{0x4338, 0x02},
	{0x4339, 0x00},
	{0x433A, 0x00},
	{0x433B, 0x02},
	{0x435A, 0x03},
	{0x435B, 0x84},
	{0x435E, 0x01},
	{0x435F, 0xFF},
	{0x4360, 0x01},
	{0x4361, 0xF4},
	{0x4362, 0x03},
	{0x4363, 0x84},
	{0x437B, 0x01},
	{0x4401, 0x03}, /*0x3F*/
	{0x4402, 0xFF},
	{0x4404, 0x13},
	{0x4405, 0x26},
	{0x4406, 0x07},
	{0x4408, 0x20},
	{0x4409, 0xE5},
	{0x440A, 0xFB},
	{0x440C, 0xF6},
	{0x440D, 0xEA},
	{0x440E, 0x20},
	{0x4410, 0x00},
	{0x4411, 0x00},
	{0x4412, 0x3F},
	{0x4413, 0xFF},
	{0x4414, 0x1F},
	{0x4415, 0xFF},
	{0x4416, 0x20},
	{0x4417, 0x00},
	{0x4418, 0x1F},
	{0x4419, 0xFF},
	{0x441A, 0x20},
	{0x441B, 0x00},
	{0x441D, 0x40},
	{0x441E, 0x1E},
	{0x441F, 0x38},
	{0x4420, 0x01},
	{0x4444, 0x00},
	{0x4445, 0x00},
	{0x4446, 0x1D},
	{0x4447, 0xF9},
	{0x4452, 0x00},
	{0x4453, 0xA0},
	{0x4454, 0x08},
	{0x4455, 0x00},
	{0x4456, 0x0F},
	{0x4457, 0xFF},
	{0x4458, 0x18},
	{0x4459, 0x18},
	{0x445A, 0x3F},
	{0x445B, 0x3A},
	{0x445C, 0x00},
	{0x445D, 0x28},
	{0x445E, 0x01},
	{0x445F, 0x90},
	{0x4460, 0x00},
	{0x4461, 0x60},
	{0x4462, 0x00},
	{0x4463, 0x00},
	{0x4464, 0x00},
	{0x4465, 0x00},
	{0x446C, 0x00},
	{0x446D, 0x00},
	{0x446E, 0x00},
	/*LSC Setting*/
	{0x452A, 0x02},
	/*White Balance Setting */
	{0x0712, 0x01},
	{0x0713, 0x00},
	{0x0714, 0x01},
	{0x0715, 0x00},
	{0x0716, 0x01},
	{0x0717, 0x00},
	{0x0718, 0x01},
	{0x0719, 0x00},
	/*Shading setting*/
	{0x4500, 0x1F},
};

static struct msm_camera_i2c_reg_conf imx135_prev_settings[] = {
	/* Clock Setting */
	{0x011E, 0x18},
	{0x011F, 0x00},
	{0x0301, 0x05},
	{0x0303, 0x01},
	{0x0305, 0x03},
	{0x0309, 0x05},
	{0x030B, 0x02},
	{0x030C, 0x00},
	{0x030D, 0x71},
	{0x030E, 0x01},
	{0x3A06, 0x12},
	/* Mode setting */
	{0x0108, 0x03},
	{0x0112, 0x0A},
	{0x0113, 0x0A},
	{0x0381, 0x01},
	{0x0383, 0x01},
	{0x0385, 0x01},
	{0x0387, 0x01},
	{0x0390, 0x01},
	{0x0391, 0x22},
	{0x0392, 0x00},
	{0x0401, 0x00},
	{0x0404, 0x00},
	{0x0405, 0x10},
	{0x4082, 0x01},
	{0x4083, 0x01},
	{0x7006, 0x04},
	/* OptionalFunction setting */
	{0x0700, 0x00},
	{0x3A63, 0x00},
	{0x4100, 0xF8},
	{0x4203, 0xFF},
	{0x4344, 0x00},
	{0x441C, 0x01},
	/* Size setting	*/
	{0x0340, 0x06},
	{0x0341, 0x2E},
	{0x0342, 0x11},
	{0x0343, 0xDC},
	{0x0344, 0x00},
	{0x0345, 0x00},
	{0x0346, 0x00},
	{0x0347, 0x00},
	{0x0348, 0x10},
	{0x0349, 0x6F},
	{0x034A, 0x0C},
	{0x034B, 0x2F},
	{0x034C, 0x08},
	{0x034D, 0x38},
	{0x034E, 0x06},
	{0x034F, 0x18},
	{0x0350, 0x00},
	{0x0351, 0x00},
	{0x0352, 0x00},
	{0x0353, 0x00},
	{0x0354, 0x08},
	{0x0355, 0x38},
	{0x0356, 0x06},
	{0x0357, 0x18},
	{0x301D, 0x30},
	{0x3310, 0x08},
	{0x3311, 0x38},
	{0x3312, 0x06},
	{0x3313, 0x18},
	{0x331C, 0x00},
	{0x331D, 0x52},
	{0x4084, 0x00},
	{0x4085, 0x00},
	{0x4086, 0x00},
	{0x4087, 0x00},
	{0x4400, 0x00},
	/* Global Timing Setting */
	{0x0830, 0x67},
	{0x0831, 0x27},
	{0x0832, 0x47},
	{0x0833, 0x27},
	{0x0834, 0x27},
	{0x0835, 0x1F},
	{0x0836, 0x87},
	{0x0837, 0x2F},
	{0x0839, 0x1F},
	{0x083A, 0x17},
	{0x083B, 0x02},
	/* Integration Time Setting */
	{0x0202, 0x06},
	{0x0203, 0x2A},
	/* Gain Setting	*/
	{0x0205, 0x00},
	{0x020E, 0x01},
	{0x020F, 0x00},
	{0x0210, 0x01},
	{0x0211, 0x00},
	{0x0212, 0x01},
	{0x0213, 0x00},
	{0x0214, 0x01},
	{0x0215, 0x00},
	/* HDR Setting */
	{0x0230, 0x00},
	{0x0231, 0x00},
	{0x0233, 0x00},
	{0x0234, 0x00},
	{0x0235, 0x40},
	{0x0238, 0x00},
	{0x0239, 0x04},
	{0x023B, 0x00},
	{0x023C, 0x01},
	{0x33B0, 0x04},
	{0x33B1, 0x00},
	{0x33B3, 0x00},
	{0x33B4, 0x00},
	{0x3800, 0x00},
};

static struct msm_camera_i2c_reg_conf imx135_LSCTable_settings[] = {
	{0x4800, 0x02},
	{0x4801, 0x68},
	{0x4802, 0x02},
	{0x4803, 0x4f},
	{0x4804, 0x02},
	{0x4805, 0x10},
	{0x4806, 0x01},
	{0x4807, 0xf3},
	{0x4808, 0x01},
	{0x4809, 0xc3},
	{0x480a, 0x01},
	{0x480b, 0xb3},
	{0x480c, 0x01},
	{0x480d, 0x9a},
	{0x480e, 0x01},
	{0x480f, 0x8e},
	{0x4810, 0x01},
	{0x4811, 0x84},
	{0x4812, 0x01},
	{0x4813, 0x7f},
	{0x4814, 0x01},
	{0x4815, 0x94},
	{0x4816, 0x01},
	{0x4817, 0x8a},
	{0x4818, 0x01},
	{0x4819, 0xb7},
	{0x481a, 0x01},
	{0x481b, 0xa6},
	{0x481c, 0x01},
	{0x481d, 0xf9},
	{0x481e, 0x01},
	{0x481f, 0xe2},
	{0x4820, 0x02},
	{0x4821, 0x62},
	{0x4822, 0x02},
	{0x4823, 0x37},
	{0x4824, 0x02},
	{0x4825, 0x26},
	{0x4826, 0x02},
	{0x4827, 0x0c},
	{0x4828, 0x01},
	{0x4829, 0xbb},
	{0x482a, 0x01},
	{0x482b, 0xaf},
	{0x482c, 0x01},
	{0x482d, 0x7c},
	{0x482e, 0x01},
	{0x482f, 0x70},
	{0x4830, 0x01},
	{0x4831, 0x50},
	{0x4832, 0x01},
	{0x4833, 0x4b},
	{0x4834, 0x01},
	{0x4835, 0x3e},
	{0x4836, 0x01},
	{0x4837, 0x39},
	{0x4838, 0x01},
	{0x4839, 0x4c},
	{0x483a, 0x01},
	{0x483b, 0x45},
	{0x483c, 0x01},
	{0x483d, 0x74},
	{0x483e, 0x01},
	{0x483f, 0x63},
	{0x4840, 0x01},
	{0x4841, 0xae},
	{0x4842, 0x01},
	{0x4843, 0x97},
	{0x4844, 0x02},
	{0x4845, 0x07},
	{0x4846, 0x01},
	{0x4847, 0xeb},
	{0x4848, 0x01},
	{0x4849, 0xf6},
	{0x484a, 0x01},
	{0x484b, 0xdf},
	{0x484c, 0x01},
	{0x484d, 0x93},
	{0x484e, 0x01},
	{0x484f, 0x8d},
	{0x4850, 0x01},
	{0x4851, 0x50},
	{0x4852, 0x01},
	{0x4853, 0x4b},
	{0x4854, 0x01},
	{0x4855, 0x22},
	{0x4856, 0x01},
	{0x4857, 0x20},
	{0x4858, 0x01},
	{0x4859, 0x10},
	{0x485a, 0x01},
	{0x485b, 0x0e},
	{0x485c, 0x01},
	{0x485d, 0x1d},
	{0x485e, 0x01},
	{0x485f, 0x18},
	{0x4860, 0x01},
	{0x4861, 0x48},
	{0x4862, 0x01},
	{0x4863, 0x3f},
	{0x4864, 0x01},
	{0x4865, 0x85},
	{0x4866, 0x01},
	{0x4867, 0x7b},
	{0x4868, 0x01},
	{0x4869, 0xd9},
	{0x486a, 0x01},
	{0x486b, 0xc5},
	{0x486c, 0x01},
	{0x486d, 0xdf},
	{0x486e, 0x01},
	{0x486f, 0xcf},
	{0x4870, 0x01},
	{0x4871, 0x87},
	{0x4872, 0x01},
	{0x4873, 0x7f},
	{0x4874, 0x01},
	{0x4875, 0x3d},
	{0x4876, 0x01},
	{0x4877, 0x3a},
	{0x4878, 0x01},
	{0x4879, 0x10},
	{0x487a, 0x01},
	{0x487b, 0x0e},
	{0x487c, 0x01},
	{0x487d, 0x00},
	{0x487e, 0x01},
	{0x487f, 0x00},
	{0x4880, 0x01},
	{0x4881, 0x0a},
	{0x4882, 0x01},
	{0x4883, 0x07},
	{0x4884, 0x01},
	{0x4885, 0x34},
	{0x4886, 0x01},
	{0x4887, 0x2f},
	{0x4888, 0x01},
	{0x4889, 0x77},
	{0x488a, 0x01},
	{0x488b, 0x69},
	{0x488c, 0x01},
	{0x488d, 0xcc},
	{0x488e, 0x01},
	{0x488f, 0xb9},
	{0x4890, 0x01},
	{0x4891, 0xea},
	{0x4892, 0x01},
	{0x4893, 0xdf},
	{0x4894, 0x01},
	{0x4895, 0x93},
	{0x4896, 0x01},
	{0x4897, 0x86},
	{0x4898, 0x01},
	{0x4899, 0x4c},
	{0x489a, 0x01},
	{0x489b, 0x48},
	{0x489c, 0x01},
	{0x489d, 0x20},
	{0x489e, 0x01},
	{0x489f, 0x1c},
	{0x48a0, 0x01},
	{0x48a1, 0x0d},
	{0x48a2, 0x01},
	{0x48a3, 0x09},
	{0x48a4, 0x01},
	{0x48a5, 0x1a},
	{0x48a6, 0x01},
	{0x48a7, 0x15},
	{0x48a8, 0x01},
	{0x48a9, 0x43},
	{0x48aa, 0x01},
	{0x48ab, 0x3d},
	{0x48ac, 0x01},
	{0x48ad, 0x84},
	{0x48ae, 0x01},
	{0x48af, 0x75},
	{0x48b0, 0x01},
	{0x48b1, 0xd3},
	{0x48b2, 0x01},
	{0x48b3, 0xbf},
	{0x48b4, 0x02},
	{0x48b5, 0x23},
	{0x48b6, 0x02},
	{0x48b7, 0x07},
	{0x48b8, 0x01},
	{0x48b9, 0xbc},
	{0x48ba, 0x01},
	{0x48bb, 0xac},
	{0x48bc, 0x01},
	{0x48bd, 0x7a},
	{0x48be, 0x01},
	{0x48bf, 0x6f},
	{0x48c0, 0x01},
	{0x48c1, 0x4c},
	{0x48c2, 0x01},
	{0x48c3, 0x47},
	{0x48c4, 0x01},
	{0x48c5, 0x3d},
	{0x48c6, 0x01},
	{0x48c7, 0x37},
	{0x48c8, 0x01},
	{0x48c9, 0x48},
	{0x48ca, 0x01},
	{0x48cb, 0x40},
	{0x48cc, 0x01},
	{0x48cd, 0x70},
	{0x48ce, 0x01},
	{0x48cf, 0x61},
	{0x48d0, 0x01},
	{0x48d1, 0xab},
	{0x48d2, 0x01},
	{0x48d3, 0x9a},
	{0x48d4, 0x02},
	{0x48d5, 0x03},
	{0x48d6, 0x01},
	{0x48d7, 0xe6},
	{0x48d8, 0x02},
	{0x48d9, 0x71},
	{0x48da, 0x02},
	{0x48db, 0x4a},
	{0x48dc, 0x02},
	{0x48dd, 0x07},
	{0x48de, 0x01},
	{0x48df, 0xef},
	{0x48e0, 0x01},
	{0x48e1, 0xbf},
	{0x48e2, 0x01},
	{0x48e3, 0xae},
	{0x48e4, 0x01},
	{0x48e5, 0x97},
	{0x48e6, 0x01},
	{0x48e7, 0x89},
	{0x48e8, 0x01},
	{0x48e9, 0x82},
	{0x48ea, 0x01},
	{0x48eb, 0x7a},
	{0x48ec, 0x01},
	{0x48ed, 0x91},
	{0x48ee, 0x01},
	{0x48ef, 0x83},
	{0x48f0, 0x01},
	{0x48f1, 0xb7},
	{0x48f2, 0x01},
	{0x48f3, 0xa9},
	{0x48f4, 0x01},
	{0x48f5, 0xf0},
	{0x48f6, 0x01},
	{0x48f7, 0xd9},
	{0x48f8, 0x02},
	{0x48f9, 0x55},
	{0x48fa, 0x02},
	{0x48fb, 0x32},
	{0x48fc, 0x02},
	{0x48fd, 0x4b},
	{0x48fe, 0x02},
	{0x48ff, 0x4c},
	{0x4900, 0x01},
	{0x4901, 0xec},
	{0x4902, 0x01},
	{0x4903, 0xf2},
	{0x4904, 0x01},
	{0x4905, 0xb1},
	{0x4906, 0x01},
	{0x4907, 0xb7},
	{0x4908, 0x01},
	{0x4909, 0x8a},
	{0x490a, 0x01},
	{0x490b, 0x8c},
	{0x490c, 0x01},
	{0x490d, 0x7d},
	{0x490e, 0x01},
	{0x490f, 0x7d},
	{0x4910, 0x01},
	{0x4911, 0x87},
	{0x4912, 0x01},
	{0x4913, 0x87},
	{0x4914, 0x01},
	{0x4915, 0xa8},
	{0x4916, 0x01},
	{0x4917, 0xa8},
	{0x4918, 0x01},
	{0x4919, 0xe2},
	{0x491a, 0x01},
	{0x491b, 0xda},
	{0x491c, 0x02},
	{0x491d, 0x38},
	{0x491e, 0x02},
	{0x491f, 0x30},
	{0x4920, 0x02},
	{0x4921, 0x0a},
	{0x4922, 0x02},
	{0x4923, 0x0e},
	{0x4924, 0x01},
	{0x4925, 0xae},
	{0x4926, 0x01},
	{0x4927, 0xaf},
	{0x4928, 0x01},
	{0x4929, 0x71},
	{0x492a, 0x01},
	{0x492b, 0x74},
	{0x492c, 0x01},
	{0x492d, 0x4b},
	{0x492e, 0x01},
	{0x492f, 0x4a},
	{0x4930, 0x01},
	{0x4931, 0x3b},
	{0x4932, 0x01},
	{0x4933, 0x3c},
	{0x4934, 0x01},
	{0x4935, 0x46},
	{0x4936, 0x01},
	{0x4937, 0x47},
	{0x4938, 0x01},
	{0x4939, 0x68},
	{0x493a, 0x01},
	{0x493b, 0x68},
	{0x493c, 0x01},
	{0x493d, 0x9e},
	{0x493e, 0x01},
	{0x493f, 0xa0},
	{0x4940, 0x01},
	{0x4941, 0xf4},
	{0x4942, 0x01},
	{0x4943, 0xec},
	{0x4944, 0x01},
	{0x4945, 0xdc},
	{0x4946, 0x01},
	{0x4947, 0xe7},
	{0x4948, 0x01},
	{0x4949, 0x8a},
	{0x494a, 0x01},
	{0x494b, 0x8e},
	{0x494c, 0x01},
	{0x494d, 0x4b},
	{0x494e, 0x01},
	{0x494f, 0x4b},
	{0x4950, 0x01},
	{0x4951, 0x1e},
	{0x4952, 0x01},
	{0x4953, 0x1f},
	{0x4954, 0x01},
	{0x4955, 0x0c},
	{0x4956, 0x01},
	{0x4957, 0x0b},
	{0x4958, 0x01},
	{0x4959, 0x18},
	{0x495a, 0x01},
	{0x495b, 0x17},
	{0x495c, 0x01},
	{0x495d, 0x42},
	{0x495e, 0x01},
	{0x495f, 0x3f},
	{0x4960, 0x01},
	{0x4961, 0x7b},
	{0x4962, 0x01},
	{0x4963, 0x79},
	{0x4964, 0x01},
	{0x4965, 0xcb},
	{0x4966, 0x01},
	{0x4967, 0xc1},
	{0x4968, 0x01},
	{0x4969, 0xd4},
	{0x496a, 0x01},
	{0x496b, 0xd4},
	{0x496c, 0x01},
	{0x496d, 0x80},
	{0x496e, 0x01},
	{0x496f, 0x81},
	{0x4970, 0x01},
	{0x4971, 0x3e},
	{0x4972, 0x01},
	{0x4973, 0x3a},
	{0x4974, 0x01},
	{0x4975, 0x10},
	{0x4976, 0x01},
	{0x4977, 0x0d},
	{0x4978, 0x01},
	{0x4979, 0x00},
	{0x497a, 0x01},
	{0x497b, 0x00},
	{0x497c, 0x01},
	{0x497d, 0x07},
	{0x497e, 0x01},
	{0x497f, 0x08},
	{0x4980, 0x01},
	{0x4981, 0x34},
	{0x4982, 0x01},
	{0x4983, 0x32},
	{0x4984, 0x01},
	{0x4985, 0x72},
	{0x4986, 0x01},
	{0x4987, 0x6d},
	{0x4988, 0x01},
	{0x4989, 0xc1},
	{0x498a, 0x01},
	{0x498b, 0xb6},
	{0x498c, 0x01},
	{0x498d, 0xe1},
	{0x498e, 0x01},
	{0x498f, 0xe5},
	{0x4990, 0x01},
	{0x4991, 0x8d},
	{0x4992, 0x01},
	{0x4993, 0x8f},
	{0x4994, 0x01},
	{0x4995, 0x4c},
	{0x4996, 0x01},
	{0x4997, 0x4d},
	{0x4998, 0x01},
	{0x4999, 0x1d},
	{0x499a, 0x01},
	{0x499b, 0x1d},
	{0x499c, 0x01},
	{0x499d, 0x0b},
	{0x499e, 0x01},
	{0x499f, 0x0b},
	{0x49a0, 0x01},
	{0x49a1, 0x18},
	{0x49a2, 0x01},
	{0x49a3, 0x16},
	{0x49a4, 0x01},
	{0x49a5, 0x40},
	{0x49a6, 0x01},
	{0x49a7, 0x3f},
	{0x49a8, 0x01},
	{0x49a9, 0x7c},
	{0x49aa, 0x01},
	{0x49ab, 0x77},
	{0x49ac, 0x01},
	{0x49ad, 0xcb},
	{0x49ae, 0x01},
	{0x49af, 0xc8},
	{0x49b0, 0x02},
	{0x49b1, 0x0a},
	{0x49b2, 0x02},
	{0x49b3, 0x0f},
	{0x49b4, 0x01},
	{0x49b5, 0xad},
	{0x49b6, 0x01},
	{0x49b7, 0xaf},
	{0x49b8, 0x01},
	{0x49b9, 0x74},
	{0x49ba, 0x01},
	{0x49bb, 0x73},
	{0x49bc, 0x01},
	{0x49bd, 0x49},
	{0x49be, 0x01},
	{0x49bf, 0x48},
	{0x49c0, 0x01},
	{0x49c1, 0x37},
	{0x49c2, 0x01},
	{0x49c3, 0x37},
	{0x49c4, 0x01},
	{0x49c5, 0x44},
	{0x49c6, 0x01},
	{0x49c7, 0x3f},
	{0x49c8, 0x01},
	{0x49c9, 0x68},
	{0x49ca, 0x01},
	{0x49cb, 0x65},
	{0x49cc, 0x01},
	{0x49cd, 0xa2},
	{0x49ce, 0x01},
	{0x49cf, 0x9d},
	{0x49d0, 0x01},
	{0x49d1, 0xf0},
	{0x49d2, 0x01},
	{0x49d3, 0xe6},
	{0x49d4, 0x02},
	{0x49d5, 0x4f},
	{0x49d6, 0x02},
	{0x49d7, 0x4c},
	{0x49d8, 0x01},
	{0x49d9, 0xf4},
	{0x49da, 0x01},
	{0x49db, 0xf8},
	{0x49dc, 0x01},
	{0x49dd, 0xb6},
	{0x49de, 0x01},
	{0x49df, 0xb4},
	{0x49e0, 0x01},
	{0x49e1, 0x90},
	{0x49e2, 0x01},
	{0x49e3, 0x91},
	{0x49e4, 0x01},
	{0x49e5, 0x7f},
	{0x49e6, 0x01},
	{0x49e7, 0x81},
	{0x49e8, 0x01},
	{0x49e9, 0x8a},
	{0x49ea, 0x01},
	{0x49eb, 0x88},
	{0x49ec, 0x01},
	{0x49ed, 0xa8},
	{0x49ee, 0x01},
	{0x49ef, 0xa9},
	{0x49f0, 0x01},
	{0x49f1, 0xde},
	{0x49f2, 0x01},
	{0x49f3, 0xd9},
	{0x49f4, 0x02},
	{0x49f5, 0x3b},
	{0x49f6, 0x02},
	{0x49f7, 0x30},
	{0x3A63, 0x01},
};

static struct msm_camera_i2c_reg_conf imx135_snap_settings[] = {
	/* Clock Setting	*/
	{0x011E, 0x18},
	{0x011F, 0x00},
	{0x0301, 0x05},
	{0x0303, 0x01},
	{0x0305, 0x03},
	{0x0309, 0x05},
	{0x030B, 0x01},
	{0x030C, 0x00},
	{0x030D, 0x60},/*0x64*/
	{0x030E, 0x01},
	{0x3A06, 0x11},
	/* Mode setting */
	{0x0108, 0x03},
	{0x0112, 0x0A},
	{0x0113, 0x0A},
	{0x0381, 0x01},
	{0x0383, 0x01},
	{0x0385, 0x01},
	{0x0387, 0x01},
	{0x0390, 0x00},
	{0x0391, 0x11},
	{0x0392, 0x00},
	{0x0401, 0x00},
	{0x0404, 0x00},
	{0x0405, 0x10},
	{0x4082, 0x01},
	{0x4083, 0x01},
	{0x7006, 0x04},
	/* OptionalFunction setting */
	{0x0700, 0x00},
	{0x3A63, 0x00},
	{0x4100, 0xF8},
	{0x4203, 0xFF},
	{0x4344, 0x00},
	{0x441C, 0x01},
	/* Size setting	*/
	{0x0340, 0x0C},
	{0x0341, 0x46},
	{0x0342, 0x11},
	{0x0343, 0xDC},
	{0x0344, 0x00},
	{0x0345, 0x00},
	{0x0346, 0x00},
	{0x0347, 0x00},
	{0x0348, 0x10},
	{0x0349, 0x6F},
	{0x034A, 0x0C},
	{0x034B, 0x2F},
	{0x034C, 0x10},
	{0x034D, 0x70},
	{0x034E, 0x0C},
	{0x034F, 0x30},
	{0x0350, 0x00},
	{0x0351, 0x00},
	{0x0352, 0x00},
	{0x0353, 0x00},
	{0x0354, 0x10},
	{0x0355, 0x70},
	{0x0356, 0x0C},
	{0x0357, 0x30},
	{0x301D, 0x30},
	{0x3310, 0x10},
	{0x3311, 0x70},
	{0x3312, 0x0C},
	{0x3313, 0x30},
	{0x331C, 0x01},
	{0x331D, 0x68},
	{0x4084, 0x00},
	{0x4085, 0x00},
	{0x4086, 0x00},
	{0x4087, 0x00},
	{0x4400, 0x00},
	/* Global Timing Setting */
	{0x0830, 0x7F},
	{0x0831, 0x37},
	{0x0832, 0x5F},
	{0x0833, 0x37},
	{0x0834, 0x37},
	{0x0835, 0x3F},
	{0x0836, 0xC7},
	{0x0837, 0x3F},
	{0x0839, 0x1F},
	{0x083A, 0x17},
	{0x083B, 0x02},
	/* Integration Time Setting */
	{0x0202, 0x0C},
	{0x0203, 0x42},
	/* Gain Setting	*/
	{0x0205, 0x00},
	{0x020E, 0x01},
	{0x020F, 0x00},
	{0x0210, 0x01},
	{0x0211, 0x00},
	{0x0212, 0x01},
	{0x0213, 0x00},
	{0x0214, 0x01},
	{0x0215, 0x00},
	/* HDR Setting */
	{0x0230, 0x00},
	{0x0231, 0x00},
	{0x0233, 0x00},
	{0x0234, 0x00},
	{0x0235, 0x40},
	{0x0238, 0x00},
	{0x0239, 0x04},
	{0x023B, 0x00},
	{0x023C, 0x01},
	{0x33B0, 0x04},
	{0x33B1, 0x00},
	{0x33B3, 0x00},
	{0x33B4, 0x00},
	{0x3800, 0x00},
};

static struct msm_camera_i2c_reg_conf imx135_hdr_settings[] = {
	/* Clock Setting */
	{0x011E, 0x18},
	{0x011F, 0x00},
	{0x0301, 0x05},
	{0x0303, 0x01},
	{0x0305, 0x03},
	{0x0309, 0x05},
	{0x030B, 0x02},
	{0x030C, 0x00},
	{0x030D, 0x71},
	{0x030E, 0x01},
	{0x3A06, 0x12},
	/* Mode setting	*/
	{0x0108, 0x03},
	{0x0112, 0x0E},
	{0x0113, 0x0A},
	{0x0381, 0x01},
	{0x0383, 0x01},
	{0x0385, 0x01},
	{0x0387, 0x01},
	{0x0390, 0x00},
	{0x0391, 0x11},
	{0x0392, 0x00},
	{0x0401, 0x00},
	{0x0404, 0x00},
	{0x0405, 0x10},
	{0x4082, 0x01},
	{0x4083, 0x01},
	{0x7006, 0x04},
	/* OptionnalFunction setting */
	{0x0700, 0x01},
	{0x3A63, 0x00},
	{0x4100, 0xF8},
	{0x4203, 0xFF},
	{0x4344, 0x00},
	{0x441C, 0x01},
	/* Size setting	*/
	{0x0340, 0x0C},
	{0x0341, 0x48},
	{0x0342, 0x11},
	{0x0343, 0xDC},
	{0x0344, 0x00},
	{0x0345, 0x08},
	{0x0346, 0x00},
	{0x0347, 0x00},
	{0x0348, 0x10},
	{0x0349, 0x67},
	{0x034A, 0x0C},
	{0x034B, 0x2F},
	{0x034C, 0x08},
	{0x034D, 0x30},
	{0x034E, 0x06},
	{0x034F, 0x18},
	{0x0350, 0x00},
	{0x0351, 0x00},
	{0x0352, 0x00},
	{0x0353, 0x00},
	{0x0354, 0x08},
	{0x0355, 0x30},
	{0x0356, 0x06},
	{0x0357, 0x18},
	{0x301D, 0x30},
	{0x3310, 0x08},
	{0x3311, 0x30},
	{0x3312, 0x06},
	{0x3313, 0x18},
	{0x331C, 0x00},
	{0x331D, 0x10},
	{0x4084, 0x00},
	{0x4085, 0x00},
	{0x4086, 0x00},
	{0x4087, 0x00},
	{0x4400, 0x00},
	/*Global Timing Setting	*/
	{0x0830, 0x67},
	{0x0831, 0x27},
	{0x0832, 0x47},
	{0x0833, 0x27},
	{0x0834, 0x27},
	{0x0835, 0x1F},
	{0x0836, 0x87},
	{0x0837, 0x2F},
	{0x0839, 0x1F},
	{0x083A, 0x17},
	{0x083B, 0x02},
	/*Integration Time Setting*/
	{0x0202, 0x0C},
	{0x0203, 0x44},
	/*Gain Setting */
	{0x0205, 0x00},
	{0x020E, 0x01},
	{0x020F, 0x00},
	{0x0210, 0x01},
	{0x0211, 0x00},
	{0x0212, 0x01},
	{0x0213, 0x00},
	{0x0214, 0x01},
	{0x0215, 0x00},
	/* HDR Setting */
	{0x0230, 0x00},
	{0x0231, 0x00},
	{0x0233, 0x00},
	{0x0234, 0x00},
	{0x0235, 0x40},
	{0x0238, 0x01}, /*Direct 1 / Auto 0*/
	{0x0239, 0x04},
	{0x023B, 0x03},
	{0x023C, 0x01},
	{0x33B0, 0x08},
	{0x33B1, 0x30},
	{0x33B3, 0x01},
	{0x33B4, 0x00},
	{0x3800, 0x00},
};

static struct v4l2_subdev_info imx135_subdev_info[] = {
	{
	.code		= V4L2_MBUS_FMT_SBGGR10_1X10,
	.colorspace	= V4L2_COLORSPACE_JPEG,
	.fmt		= 1,
	.order		= 0,
	},
	/* more can be supported, to be added later */
};

static struct msm_camera_i2c_conf_array imx135_init_conf[] = {
	{&imx135_recommend_settings[0],
	ARRAY_SIZE(imx135_recommend_settings), 0, MSM_CAMERA_I2C_BYTE_DATA},
	{&imx135_LSCTable_settings[0],
	ARRAY_SIZE(imx135_LSCTable_settings), 0, MSM_CAMERA_I2C_BYTE_DATA}
};

static struct msm_camera_i2c_conf_array imx135_confs[] = {
	{&imx135_snap_settings[0],
	ARRAY_SIZE(imx135_snap_settings), 0, MSM_CAMERA_I2C_BYTE_DATA},
	{&imx135_prev_settings[0],
	ARRAY_SIZE(imx135_prev_settings), 0, MSM_CAMERA_I2C_BYTE_DATA},
	{&imx135_hdr_settings[0],
	ARRAY_SIZE(imx135_hdr_settings), 0, MSM_CAMERA_I2C_BYTE_DATA},
};

static struct msm_sensor_output_info_t imx135_dimensions[] = {
	/* RES0 snapshot(FULL SIZE) */
	{
		.x_output = 4208,
		.y_output = 3120,
		.line_length_pclk = 4572,
		.frame_length_lines = 3290,
		.vt_pixel_clk = 360000000,
		.op_pixel_clk = 319680000,
		.binning_factor = 1,
	},
	/* RES1 4:3 preview(1/2HV QTR SIZE) */
	{
		.x_output = 2104,
		.y_output = 1560,
		.line_length_pclk = 4572,
		.frame_length_lines = 1582,
		.vt_pixel_clk = 360000000,
		.op_pixel_clk = 180000000,
		.binning_factor = 1,
	},
	/* RES2 4:3 HDR movie mode */
	{
		.x_output = 2096,
		.y_output = 1560,
		.line_length_pclk = 4572,
		.frame_length_lines = 3290,
		.vt_pixel_clk = 360000000,
		.op_pixel_clk = 180000000,
		.binning_factor = 1,
	},
};

static struct msm_sensor_output_reg_addr_t imx135_reg_addr = {
	.x_output = 0x34C,
	.y_output = 0x34E,
	.line_length_pclk = 0x342,
	.frame_length_lines = 0x340,
};

static struct msm_sensor_id_info_t imx135_id_info = {
	.sensor_id_reg_addr = 0x0000,
	.sensor_id = 0x1210,
};

static struct msm_sensor_exp_gain_info_t imx135_exp_gain_info = {
	.coarse_int_time_addr = 0x202,
	.global_gain_addr = 0x205,
	.vert_offset = 4,
};

static const struct i2c_device_id imx135_i2c_id[] = {
	{SENSOR_NAME, (kernel_ulong_t)&imx135_s_ctrl},
	{ }
};

static struct i2c_driver imx135_i2c_driver = {
	.id_table = imx135_i2c_id,
	.probe  = msm_sensor_i2c_probe,
	.driver = {
		.name = SENSOR_NAME,
	},
};

static struct msm_camera_i2c_client imx135_sensor_i2c_client = {
	.addr_type = MSM_CAMERA_I2C_WORD_ADDR,
};

static int __init msm_sensor_init_module(void)
{
	return i2c_add_driver(&imx135_i2c_driver);
}

static struct v4l2_subdev_core_ops imx135_subdev_core_ops = {
	.ioctl = msm_sensor_subdev_ioctl,
	.s_power = msm_sensor_power,
};

static struct v4l2_subdev_video_ops imx135_subdev_video_ops = {
	.enum_mbus_fmt = msm_sensor_v4l2_enum_fmt,
};

static struct v4l2_subdev_ops imx135_subdev_ops = {
	.core = &imx135_subdev_core_ops,
	.video  = &imx135_subdev_video_ops,
};

int32_t imx135_write_exp_gain(struct msm_sensor_ctrl_t *s_ctrl,
		uint16_t gain, uint32_t line)
{
	uint32_t fl_lines;
	uint8_t offset;
	fl_lines = s_ctrl->curr_frame_length_lines;
	fl_lines = (fl_lines * s_ctrl->fps_divider) / Q10;
	offset = s_ctrl->sensor_exp_gain_info->vert_offset;
	if (line > (fl_lines - offset))
		fl_lines = line + offset;

	s_ctrl->func_tbl->sensor_group_hold_on(s_ctrl);
	msm_camera_i2c_write(s_ctrl->sensor_i2c_client,
		s_ctrl->sensor_output_reg_addr->frame_length_lines, fl_lines,
		MSM_CAMERA_I2C_WORD_DATA);
	msm_camera_i2c_write(s_ctrl->sensor_i2c_client,
		s_ctrl->sensor_exp_gain_info->coarse_int_time_addr, line,
		MSM_CAMERA_I2C_WORD_DATA);
	msm_camera_i2c_write(s_ctrl->sensor_i2c_client,
		s_ctrl->sensor_exp_gain_info->global_gain_addr, gain,
		MSM_CAMERA_I2C_BYTE_DATA);
	s_ctrl->func_tbl->sensor_group_hold_off(s_ctrl);
	return 0;
}

static struct msm_sensor_fn_t imx135_func_tbl = {
	.sensor_start_stream = msm_sensor_start_stream,
	.sensor_stop_stream = msm_sensor_stop_stream,
	.sensor_group_hold_on = msm_sensor_group_hold_on,
	.sensor_group_hold_off = msm_sensor_group_hold_off,
	.sensor_set_fps = msm_sensor_set_fps,
	.sensor_write_exp_gain = imx135_write_exp_gain,
	.sensor_write_snapshot_exp_gain = imx135_write_exp_gain,
	.sensor_setting = msm_sensor_setting,
	.sensor_csi_setting = msm_sensor_setting1,
	.sensor_set_sensor_mode = msm_sensor_set_sensor_mode,
	.sensor_mode_init = msm_sensor_mode_init,
	.sensor_get_output_info = msm_sensor_get_output_info,
	.sensor_config = msm_sensor_config,
	.sensor_power_up = msm_sensor_power_up,
	.sensor_power_down = msm_sensor_power_down,
	.sensor_adjust_frame_lines = msm_sensor_adjust_frame_lines1,
	.sensor_get_csi_params = msm_sensor_get_csi_params,
};

static struct msm_sensor_reg_t imx135_regs = {
	.default_data_type = MSM_CAMERA_I2C_BYTE_DATA,
	.start_stream_conf = imx135_start_settings,
	.start_stream_conf_size = ARRAY_SIZE(imx135_start_settings),
	.stop_stream_conf = imx135_stop_settings,
	.stop_stream_conf_size = ARRAY_SIZE(imx135_stop_settings),
	.group_hold_on_conf = imx135_groupon_settings,
	.group_hold_on_conf_size = ARRAY_SIZE(imx135_groupon_settings),
	.group_hold_off_conf = imx135_groupoff_settings,
	.group_hold_off_conf_size =
		ARRAY_SIZE(imx135_groupoff_settings),
	.init_settings = &imx135_init_conf[0],
	.init_size = ARRAY_SIZE(imx135_init_conf),
	.mode_settings = &imx135_confs[0],
	.output_settings = &imx135_dimensions[0],
	.num_conf = ARRAY_SIZE(imx135_confs),
};

static struct msm_sensor_ctrl_t imx135_s_ctrl = {
	.msm_sensor_reg = &imx135_regs,
	.sensor_i2c_client = &imx135_sensor_i2c_client,
	.sensor_i2c_addr = 0x20,
	.sensor_output_reg_addr = &imx135_reg_addr,
	.sensor_id_info = &imx135_id_info,
	.sensor_exp_gain_info = &imx135_exp_gain_info,
	.cam_mode = MSM_SENSOR_MODE_INVALID,
	.msm_sensor_mutex = &imx135_mut,
	.sensor_i2c_driver = &imx135_i2c_driver,
	.sensor_v4l2_subdev_info = imx135_subdev_info,
	.sensor_v4l2_subdev_info_size = ARRAY_SIZE(imx135_subdev_info),
	.sensor_v4l2_subdev_ops = &imx135_subdev_ops,
	.func_tbl = &imx135_func_tbl,
	.clk_rate = MSM_SENSOR_MCLK_24HZ,
};

module_init(msm_sensor_init_module);
MODULE_DESCRIPTION("Sony 13MP Bayer sensor driver");
MODULE_LICENSE("GPL v2");
