package latexDraw.ui.components;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.geom.Line2D;

import javax.swing.ButtonGroup;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JPopupMenu;

import latexDraw.lang.LaTeXDrawLang;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.DrawPanel;
import latexDraw.ui.dialog.ExceptionFrameDialog;


/** 
 * This class defines the X-scale.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class XScale extends JComponent implements MouseListener,ItemListener
{
	private static final long serialVersionUID = 1L;

	/** The drawPanel which contains the pixels per centimetre */
	private DrawPanel drawPanel;
	
	/** The other scale : the Y-scale (Useful for a good display) */
	private YScale yScale;
	
	/** The size of the lines in axes */
	public static final int SIZE = 10;
	
	/** The popup menu which allows to set the type of measure (inch, cm, ...) */
	private JPopupMenu popupMenu;
	
	/** The label of the centimetre menu */
	public static final String LABEL_CM = LaTeXDrawLang.getOthersString("XScale.cm"); //$NON-NLS-1$
	
	/** The label of the inch menu */
	public static final String LABEL_INCH = LaTeXDrawLang.getOthersString("XScale.inch"); //$NON-NLS-1$
	
	/** The unit of length by default */
	public static final String DEFAULT_UNIT = LABEL_CM;
	
	/** The menu for centimetres */ 
	protected JCheckBoxMenuItem cmMenu;
	
	/** The menu for inches */
	protected JCheckBoxMenuItem inchMenu;
	
	
	
	/**
	 * The constructor.
	 * @param drawP The drawPanel which contains the pixels per centimetre.
	 * @param y The YScale bounds to the XScale.
	 * @throws IllegalArgumentException If drawP or y is null.
	 */
	public XScale(DrawPanel drawP, YScale y)
	{
		super();
		
		if(drawP==null || y==null)
			throw new IllegalArgumentException();
		
		setPreferredSize(new Dimension(500, SIZE));
		drawPanel = drawP;
		yScale = y;
		
		/* Creation of the popup menu */
		ButtonGroup buttonGroup = new ButtonGroup();
		popupMenu = new JPopupMenu();
		cmMenu = new JCheckBoxMenuItem(LABEL_CM);
		cmMenu.setActionCommand(LABEL_CM);
		cmMenu.addItemListener(this);
		inchMenu = new JCheckBoxMenuItem(LABEL_INCH);
		inchMenu.addItemListener(this);
		inchMenu.setActionCommand(LABEL_INCH);
		cmMenu.setSelected(true);
		popupMenu.add(cmMenu);
		popupMenu.add(inchMenu);
		buttonGroup.add(cmMenu);
		buttonGroup.add(inchMenu);
		addMouseListener(this);
		setDoubleBuffered(true);
	}
	

	
    
	/**
	 * Sets the Y-scale.
	 * @param y The new Y-scale.
	 * @throws IllegalArgumentException If y is null.
	 */
	public void setYScale(YScale y)
	{
		if(y==null)
			throw new IllegalArgumentException();
		
		yScale = y;
	}
	
	
	
	
	/**
	 * Allows to access to the popup menu.
	 * @return The popup menu.
	 */
	public JPopupMenu getPopupMenu()
	{
		return popupMenu;
	}
	
	
	
	
	/**
	 * Allows to know the X-coordinates of the start of the X-Scale.
	 * @return The X-coordinates of the start of the X-Scale.
	 */
	public float getStart()
	{
		float start;
		
		if(yScale==null)
    		start = 0;
    	else
    		if(yScale.isVisible())
    			  start = SIZE;
    		else  start = 0;
		
		return start;
	}
	
	
	
	
	@Override
    public void paintComponent(Graphics g)
    {   	
		try
		{
	    	Dimension d = getSize();
	    	double zoom = drawPanel.getZoom();
	    	double width = d.getWidth()*(1/zoom), i, j, start, cpt;
	    	double pixPerCm = drawPanel.getPixelsPerCm();
	    	Graphics2D g2 = (Graphics2D)g;
	    	
	    	g2.scale(zoom, zoom);
	    	g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
			g2.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
			g2.setStroke(new BasicStroke(0, BasicStroke.CAP_SQUARE, BasicStroke.JOIN_MITER));
			
			Color formerCol = g2.getColor();
	    	g2.setColor(Color.BLACK);
			
	    	if(yScale==null)
	    		start = 0.;
	    	else
	    		if(yScale.isVisible())
	    			  start = SIZE;
	    		else  start = 0.;
	  
    		if(inchMenu.isSelected())
    			pixPerCm*=PSTricksConstants.INCH_VAL_CM;
    		
    		if(pixPerCm/10.!=(int)(pixPerCm/10.))
			{
    			double tmp = pixPerCm/10.;
    			tmp = tmp-(int)tmp>0.5 ? ((int)tmp)+1 : (int)tmp;
    			pixPerCm = tmp*10;
			}

	    	if(pixPerCm>20/zoom)
	    	{    			    			
	    		double pixPerCm10 = pixPerCm/10.;
	    		
	    		for(i=start/zoom+pixPerCm10; i<width; i+=pixPerCm)
	    			for(j=i, cpt=1; cpt<10; j+=pixPerCm10, cpt++)
	    				g2.draw(new Line2D.Double(j,(SIZE/2.)/zoom,j,SIZE/zoom));
	    	}
	    	
	    	for(i=start/zoom; i<width;i+=pixPerCm)
	    		g2.draw(new Line2D.Double(i, 0., i, SIZE/zoom));
	    	
	    	g2.setColor(formerCol);
		}
		catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
    }


	
	
    /** 
     * Defines the actions that it must be realised by a right click of the mouse.
     * @param e Corresponds to the click of the mouse.
     */
	public void mouseClickedRightButton(MouseEvent e)
    {
		if(popupMenu!=null)
			popupMenu.show(e.getComponent(), e.getX(), e.getY());
    }
	
	


	 /** 
     * Defines the actions that it must be realised by a click of the mouse.
     * @param e The event
     */
	public void mouseClicked(MouseEvent e) 
	{
    	switch(e.getButton())
	    {
		    case MouseEvent.BUTTON1 :   	
		    			break;
			
		    case MouseEvent.BUTTON2 : 
		    			break;
	
		    case MouseEvent.BUTTON3 : 
						mouseClickedRightButton(e);    
						break;
				
		    default : break;
	    } // switch

    	e.consume();
	}

	

	public void mousePressed(MouseEvent e) 
	{
		/* No action to do. */
	}


	public void mouseReleased(MouseEvent e) 
	{
		/* No action to do. */
	}


	public void mouseEntered(MouseEvent e) 
	{
		/* No action to do. */
	}



	public void mouseExited(MouseEvent e) 
	{
		/* No action to do. */
	}



	public void itemStateChanged(ItemEvent e) 
	{
		Object o = e.getSource();
		
		if(o instanceof JCheckBoxMenuItem || o instanceof JComboBox)
		{
			boolean isInCm = true;

			if(o instanceof JCheckBoxMenuItem)
				drawPanel.getDraw().setUnitForPixelsPerCm(
						((JCheckBoxMenuItem)o).getActionCommand());
			else
			if(o instanceof JComboBox)
			{
				if(!(((String)((JComboBox)o).getSelectedItem())).equals(LABEL_CM))
					 isInCm = false;
				cmMenu.setSelected(isInCm);
				inchMenu.setSelected(!isInCm);
				
				drawPanel.getDraw().setUnitForPixelsPerCm(
										((String)((JComboBox)o).getSelectedItem()));
			}
			
			repaint();
			yScale.repaint();
		}
	}
	
	
	
	
	/**
	 * Sets the unit of length (and to change the scale of the XScale and to check the good menu).
	 * @param unitForPixelsPerCm The unitForPixelsPerCm to set.
	 * @throws IllegalArgumentException If unitForPixelsPerCm is not valid.
	 */
	public void setUnitForPixelsPerCm(String unitForPixelsPerCm)
	{
		if(unitForPixelsPerCm==null || (!unitForPixelsPerCm.equals(LABEL_CM) && !unitForPixelsPerCm.equals(LABEL_INCH)))
			throw new IllegalArgumentException();
		
		boolean inCm = unitForPixelsPerCm.equals(LABEL_CM);
		
		cmMenu.setSelected(inCm);
		inchMenu.setSelected(!inCm);
		repaint();
		yScale.repaint();
	}
}
