package latexDraw.parsers.pst;

import java.awt.Color;
import java.awt.geom.Point2D;
import java.awt.geom.Point2D.Double;
import java.util.Vector;

import latexDraw.figures.FramedBox;
import latexDraw.figures.Text;
import latexDraw.psTricks.PSTricksConstants;


/** 
 * This class defines a set of PSTricks parameters.
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 09/01/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.5<br>
 */
public class PSTParameters
{
	/**
	 * Defines the parameters of the pscustom parameters.
	 * @author Arnaud BLOUIN<br>
	 * @version 0.3<br>
	 */
	public class PsCustomParameters implements Cloneable
	{
		/** Defines if the command pscustom has a star. */
		public boolean hasStar;
		
		/** Defines if the current <code>PSTParameters</code> comes from a pscustom command. */
		public boolean fromPsCustom;
		
		/** The last position (cf. <code>\moveto</code> pstricks command). */
		public Point2D.Double lastPosition;
		
		
		
		public PsCustomParameters()
		{
			hasStar 		= false;
			fromPsCustom	= false;
			lastPosition	= new Point2D.Double();
		}
		
		
		
		@Override
		public Object clone() throws CloneNotSupportedException
		{
			PsCustomParameters pscp = (PsCustomParameters)super.clone();
			
			pscp.fromPsCustom	= fromPsCustom;
			pscp.hasStar		= hasStar;
			pscp.lastPosition	= (Double)lastPosition.clone();
			
			return pscp;
		}
	}
	
	
	
	/** 
	 * This class defines the position managed with the rput and uput commands of the figures.<br>
	 * This file is part of LaTeXDraw<br>
	 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
	 *<br>
	 *  LaTeXDraw is free software; you can redistribute it and/or modify
	 *  it under the terms of the GNU General Public License as published by
	 *  the Free Software Foundation; either version 2 of the License, or
	 *  (at your option) any later version.<br>
	 *<br>
	 *  LaTeXDraw is distributed without any warranty; without even the 
	 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
	 *  PURPOSE.  See the GNU General Public License for more details.<br>
	 * <br>
	 * 
	 * 03/16/06<br>
	 * @author Arnaud BLOUIN<br>
	 * @version 0.5<br>
	 */
	public static class PositionParameters implements Cloneable
	{
		public static final short REF_POINT_DEFAULT_CENTER= 0;
		
		public static final short REF_POINT_LEFT  		= 10;
		public static final short REF_POINT_RIGHT 		= 20;
		
		public static final short REF_POINT_TOP   		= 1;
		public static final short REF_POINT_BOTTOM 		= 2;
		public static final short REF_POINT_BASE 		= 3;
		
		/** True if the current {...} allows to sum all the rotations :<br>
		 * rput{30}(0,0){ rput{*40}{...} ... } -> false<br>
		 * rput{30}(0,0){ rput{40} {...} ... } -> true<br>
		 */
		public boolean isSumRotation;
		
		/** The x gap */
		public double dx;
		
		/** The y gap */
		public double dy;
		
		/** The rput angle in degrees */
		public double angleRput;
		
		/** The uput angle in degrees. */
		public double angleUput;
		
		/** The gap of the label */
		public double labelSep;
		
		/** The point of reference (for rput). */
		public int refPoint;
		
		/** The angle of reference (for uput in degrees). */
		public double refAngle;
		
		
		
		/**
		 * The constructor by default.
		 */
		public PositionParameters()
		{
			this(true, 0, 0, 0, 0, 0, REF_POINT_DEFAULT_CENTER, REF_POINT_RIGHT);
		}
		
		
		/**
		 * The main constructor.
		 * @param isSumRot True if allows to sum all the rotations
		 * @param x The x gap
		 * @param y The y gap
		 * @param rputA The rput angle
		 * @param uputA The uput angle
		 * @param labSep The gap of the label (for uput).
		 * @param refPt The point of reference (for rput).
		 * @param refA The angle of reference (for uput).
		 */
		public PositionParameters(boolean isSumRot, double x, double y, double rputA, 
							double uputA, double labSep, int refPt, double refA)
		{
			isSumRotation = isSumRot;
			dx = x;
			dy = y;
			angleRput = rputA;
			angleUput = uputA;
			labelSep  = labSep;
			
			if(refPt==REF_POINT_LEFT || refPt==REF_POINT_BOTTOM || refPt== REF_POINT_BOTTOM+REF_POINT_LEFT ||
				refPt==REF_POINT_BOTTOM+REF_POINT_RIGHT || refPt==REF_POINT_DEFAULT_CENTER || refPt==REF_POINT_RIGHT || 
				refPt==REF_POINT_TOP || refPt==REF_POINT_TOP+REF_POINT_LEFT || refPt==REF_POINT_TOP+REF_POINT_RIGHT || 
				refPt==REF_POINT_BASE || refPt==REF_POINT_BASE+REF_POINT_LEFT || refPt==REF_POINT_BASE+REF_POINT_RIGHT)
				refPoint = refPt;
			else
				refPoint = REF_POINT_DEFAULT_CENTER;
			
			if(refA==REF_POINT_LEFT || refA==REF_POINT_BOTTOM || refA== REF_POINT_BOTTOM+REF_POINT_LEFT ||
				refA==REF_POINT_BOTTOM+REF_POINT_RIGHT || refA==REF_POINT_DEFAULT_CENTER || refA==REF_POINT_RIGHT || 
				refA==REF_POINT_TOP || refA==REF_POINT_TOP+REF_POINT_LEFT || refA==REF_POINT_TOP+REF_POINT_RIGHT || 
				refA==REF_POINT_BASE || refA==REF_POINT_BASE+REF_POINT_LEFT || refA==REF_POINT_BASE+REF_POINT_RIGHT)
				refAngle = refA;
			else
				refAngle = REF_POINT_RIGHT;
		}
		
		
		
		@Override
		public String toString()
		{
			return "PositionParameters : <isSum="+ //$NON-NLS-1$
			isSumRotation +", dx="+dx+", dy="+dy+", angleRput="+angleRput+", labelSep="+//$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$//$NON-NLS-4$
			labelSep+", refPoint="+refPoint+">";//$NON-NLS-1$//$NON-NLS-2$
		}
		
		
		
		@Override
		public Object clone() throws CloneNotSupportedException
		{
			return super.clone();
		}
	}
	
	
	
	
	public static boolean isCentered = false;
	public static Point2D.Double pictureSWPt=null;
	public static Point2D.Double pictureNEPt=null;
	public static String tokenPosition = "";//$NON-NLS-1$
	
	private PositionParameters position;
	
	public String axesStyle;
	public String[] arrowStyle;
	public double arrowSizeD;
	public double arrowSizeN;
	public double arrowLgth;
	public double arrowInset;
	public double arrowTBarSD;
	public double arrowTBarSN;
	public double arrowBrLgth;
	public double arrowrBrLgth;
	public double arrowDotSD;
	public double arrowDotSN;
	public double arrowScale1;
	public double arrowScale2;
	public double arcSep;
	public double arcSepA;
	public double arcSepB;
	public boolean boxSep;
	public Color borderColor;
	public String borderPos;
	public double border;
	public double curvature1;
	public double curvature2;
	public double curvature3;
	public double dxIncrement;
	public double dyIncrement;
	public double dxLabelDist;
	public double dyLabelDist;
	public String dotStyle;
	public double dotScale1;
	public double dotScale2;
	public double dotAngle;
	public double dotStep;
	public double dashBlack;
	public double dashWhite;
	public boolean dbleLine;
	public double dbleSep;
	public Color dbleColor;
	public double degrees;
	public double frameSep;
	public double frameArc;
	public String fillStyle;
	public Color fillColor;
	public double gridWidth;
	public double gridLabel;
	public int gridDots;
	public double gradAngle;
	public Color gridColor;
	public double gradMidPoint;
	public Color gradBegin;
	public Color gradEnd;
	public int gradLines;
	public double gangle;
	public double hatchWidth;
	public double hatchSep;
	public Color hatchCol;
	public double hatchAngle;
	public boolean isCornerRel;
	public boolean isShadow;
	public double lineWidth;
	public Color lineColor;
	public Color labelsGridCol;
	public String labels;
	public double lineArc;
	public String lineStyle;
	public double ox;
	public double oy;
	public boolean onRadians;
	public Point2D.Double origin;
	public boolean specialCoor;
	public boolean showPoints;
	public boolean showOrigin;
	public double subGridWidth;
	public boolean swapAxes;
	public Color shadowCol;
	public Color subGridCol;
	public double shadowAngle;
	public double shadowSize;
	public int subGridDots;
	public int subGridDiv;
	public String ticks;
	public String ticksStyle;
	public double ticksSize;
	public boolean textItalic;
	public boolean textBold;
	public String textEnc;
	public String textShape;
	public String textSeries;
	public String textFamily;
	public Color textColor;
	public String textSize; 
	public String textParsed;
	public double unit;
	public double xUnit;
	public double yUnit;
	
	public Vector<FramedBox> psBox = new Vector<FramedBox>();
	public Text psBoxText;
	public boolean fromFramedBox;
	
	/** The parsed text which is transfered by psframe commands to the rput PSTParameters. */
	public String textForFrame;
	
	public PsCustomParameters psCustomP;
	
	
	/**
	 * The constructor by default.
	 */
	public PSTParameters()
	{
		psCustomP 		= new PsCustomParameters();
		textForFrame 	= "";//$NON-NLS-1$
		fromFramedBox 	= false;
		textParsed 		= "";//$NON-NLS-1$
		position 	= null;
		textBold 	= false;
		textItalic 	= false;
		boxSep		= PSTricksConstants.DEFAULT_BOX_SEP;
		frameSep	= PSTricksConstants.DEFAULT_FRAME_SEP;
		gradAngle	= PSTricksConstants.DEFAULT_GRADIENT_ANGLE;
		gradMidPoint = PSTricksConstants.DEFAULT_GRADIENT_MID_POINT;
		gradLines	= PSTricksConstants.DEFAULT_GRADIENT_LINES;
		gradBegin	= PSTricksConstants.DEFAULT_GRADIENT_START_COLOR;
		gradEnd		= PSTricksConstants.DEFAULT_GRADIENT_END_COLOR;
		gangle		= PSTricksConstants.DEFAULT_GANGLE;
		specialCoor = PSTricksConstants.DEFAULT_SPECIAL_COOR;
		onRadians	= PSTricksConstants.DEFAULT_ON_RADIANS;
		degrees		= PSTricksConstants.DEFAULT_DEGREES;
		unit		= PSTricksConstants.DEFAULT_UNIT;
		xUnit		= PSTricksConstants.DEFAULT_UNIT;
		yUnit		= PSTricksConstants.DEFAULT_UNIT;
		lineWidth 	= PSTricksConstants.DEFAULT_LINE_WIDTH;
		lineColor 	= PSTricksConstants.DEFAULT_LINE_COLOR;
		showPoints 	= PSTricksConstants.DEFAULT_SHOW_POINTS;
		lineArc 	= PSTricksConstants.DEFAULT_LINE_ARC;
		frameArc 	= PSTricksConstants.DEFAULT_FRAME_ARC;
		isCornerRel = PSTricksConstants.DEFAULT_CORNER_SIZE_RELATIVE;
		arcSep		= PSTricksConstants.DEFAULT_ARC_SEP;
		arcSepA 	= PSTricksConstants.DEFAULT_ARC_SEP_A;
		arcSepB 	= PSTricksConstants.DEFAULT_ARC_SEP_B;
		curvature1	= PSTricksConstants.DEFAULT_CURVATURE_NUM1;
		curvature2	= PSTricksConstants.DEFAULT_CRUVATURE_NUM2;
		curvature3	= PSTricksConstants.DEFAULT_CRUVATURE_NUM3;
		dotStyle	= PSTricksConstants.DEFAULT_DOT_STYLE;
		dotScale1	= PSTricksConstants.DEFAULT_DOT_SCALE1;
		dotScale2	= PSTricksConstants.DEFAULT_DOT_SCALE2;
		dotAngle	= PSTricksConstants.DEFAULT_DOT_ANGLE;
		gridWidth	= PSTricksConstants.DEFAULT_GRID_WIDTH;
		gridLabel	= PSTricksConstants.DEFAULT_GRID_LABEL/PSTricksConstants.CM_VAL_PT;
		subGridWidth= PSTricksConstants.DEFAULT_SUB_GRID_WIDTH;
		subGridCol	= PSTricksConstants.DEFAULT_SUB_GRID_COLOR;
		gridColor	= PSTricksConstants.DEFAULT_GRIDCOLOR;
		labelsGridCol	= PSTricksConstants.DEFAULT_LABELGRIDCOLOR;
		subGridDiv		= PSTricksConstants.DEFAULT_SUBGRIDDIV;
		gridDots		= PSTricksConstants.DEFAULT_GRIDDOTS;
		subGridDots		= PSTricksConstants.DEFAULT_SUBGRIDDOTS;
		swapAxes	= PSTricksConstants.DEFAULT_SWAP_AXES;
		origin 		= (Point2D.Double)PSTricksConstants.DEFAULT_ORIGIN.clone();
		lineStyle	= PSTricksConstants.DEFAULT_LINE_STYLE;
		dotStep		= PSTricksConstants.DEFAULT_DOT_STEP;
		dashBlack	= PSTricksConstants.DEFAULT_DASH_BLACK;
		dashWhite	= PSTricksConstants.DEFAULT_DASH_WHITE;
		borderColor	= PSTricksConstants.DEFAULT_BORDER_COLOR;
		border		= PSTricksConstants.DEFAULT_BORDER;
		dbleLine	= PSTricksConstants.DEFAULT_DOUBLE_LINE;
		dbleSep		= PSTricksConstants.DEFAULT_DOUBLE_SEP;
		dbleColor	= PSTricksConstants.DEFAULT_DOUBLE_COLOR;
		shadowCol	= PSTricksConstants.DEFAULT_SHADOW_COLOR;
		isShadow	= PSTricksConstants.DEFAULT_SHADOW;
		shadowAngle	= PSTricksConstants.DEFAULT_SHADOW_ANGLE;
		shadowSize	= PSTricksConstants.DEFAULT_SHADOW_SIZE;
		borderPos	= PSTricksConstants.DEFAULT_BORDERS_POS;
		fillStyle	= PSTricksConstants.DEFAULT_FILL_STYLE;
		fillColor	= PSTricksConstants.DEFAULT_FILL_COLOR;
		hatchWidth	= PSTricksConstants.DEFAULT_HATCH_WIDTH;
		hatchSep	= PSTricksConstants.DEFAULT_HATCH_SEP;
		hatchCol	= PSTricksConstants.DEFAULT_HATCHING_COLOR;
		hatchAngle	= PSTricksConstants.DEFAULT_HATCH_ANGLE;
		arrowStyle	= new String[2];
		arrowStyle[0] = PSTricksConstants.DEFAULT_ARROW_STYLE;
		arrowStyle[1] = PSTricksConstants.DEFAULT_ARROW_STYLE;
		arrowSizeD	= PSTricksConstants.DEFAULT_ARROW_SIZE_DIM;
		arrowSizeN	= PSTricksConstants.DEFAULT_ARROW_SIZE_NUM;
		arrowLgth	= PSTricksConstants.DEFAULT_ARROW_LENGTH;
		arrowInset	= PSTricksConstants.DEFAULT_ARROW_INSET;
		arrowTBarSD	= PSTricksConstants.DEFAULT_ARROW_TBARSIZE_DIM;
		arrowTBarSN	= PSTricksConstants.DEFAULT_ARROW_TBARSIZE_NUM;
		arrowBrLgth	= PSTricksConstants.DEFAULT_ARROW_BRACKET_LGTH;
		arrowrBrLgth= PSTricksConstants.DEFAULT_ARROW_RBRACKET_LGTH;
		arrowDotSD	= PSTricksConstants.DEFAULT_ARROW_DOTSIZE_DIM;
		arrowDotSN	= PSTricksConstants.DEFAULT_ARROW_DOTSIZE_NUM;
		arrowScale1	= PSTricksConstants.DEFAULT_ARROW_SCALE1;
		arrowScale2	= PSTricksConstants.DEFAULT_ARROW_SCALE2;
		textEnc		= null;
		textShape	= null;
		textSeries	= null;
		textFamily	= null;
		textColor	= Color.BLACK;
		textSize 	= PSTricksConstants.COMMAND_TEXT_NORMAL;
		psBoxText = new Text(null, false);
		psBox = new Vector<FramedBox>();
		showOrigin 	= PSTricksConstants.DEFAULT_SHOW_ORIGIN;
		axesStyle	= PSTricksConstants.DEFAULT_AXES_STYLE;
		ticks		= PSTricksConstants.DEFAULT_TICKS_DISPLAYED;
		ticksSize	= PSTricksConstants.DEFAULT_TICKS_SIZE;
		ticksStyle	= PSTricksConstants.DEFAULT_TICKS_STYLE;
		labels		= PSTricksConstants.DEFAULT_LABELS_DISPLAYED;
		ox			= PSTricksConstants.DEFAULT_OX;
		oy			= PSTricksConstants.DEFAULT_OY;
		dxIncrement	= PSTricksConstants.DEFAULT_DX;
		dyIncrement = PSTricksConstants.DEFAULT_DY;
		dxLabelDist	= PSTricksConstants.DEFAULT_DIST_X_LABEL;
		dyLabelDist	= PSTricksConstants.DEFAULT_DIST_Y_LABEL;
	}
	
	
	
	
	
	/**
	 * The constructor that copies a PSTParameters from another.
	 */
	public PSTParameters(PSTParameters toCopy)
	{
		try
		{
			psCustomP 	= (PsCustomParameters)toCopy.psCustomP.clone();
		} catch(CloneNotSupportedException e)
		{
			psCustomP = new PsCustomParameters();
		}
		textForFrame 	= "";//$NON-NLS-1$
		boxSep 			= toCopy.boxSep;
		frameSep 		= toCopy.frameSep;
		fromFramedBox 	= toCopy.fromFramedBox;
		textParsed  = "";//$NON-NLS-1$
		position    = null;
		gradAngle 	= toCopy.gradAngle;
		gradBegin	= toCopy.gradBegin;
		gradEnd		= toCopy.gradEnd;
		gradLines	= toCopy.gradLines;
		gradMidPoint = toCopy.gradMidPoint;
		textBold 	= toCopy.textBold;
		textItalic 	= toCopy.textItalic;
		gangle		= toCopy.gangle;
		specialCoor = toCopy.specialCoor;
		onRadians	= toCopy.onRadians;
		degrees		= toCopy.degrees;
		xUnit		= toCopy.xUnit;
		yUnit		= toCopy.yUnit;
		unit		= toCopy.unit;
		lineWidth 	= toCopy.lineWidth;
		lineColor 	= toCopy.lineColor;
		showPoints 	= toCopy.showPoints;
		lineArc 	= toCopy.lineArc;
		frameArc 	= toCopy.frameArc;
		isCornerRel = toCopy.isCornerRel;
		arcSep		= toCopy.arcSep;
		arcSepA 	= toCopy.arcSepA;
		arcSepB 	= toCopy.arcSepB;
		curvature1	= toCopy.curvature1;
		curvature2	= toCopy.curvature2;
		curvature3	= toCopy.curvature3;
		dotStyle	= toCopy.dotStyle;
		dotScale1	= toCopy.dotScale1;
		dotScale2	= toCopy.dotScale2;
		dotAngle	= toCopy.dotAngle;
		gridWidth	= toCopy.gridWidth;
		gridLabel	= toCopy.gridLabel;
		subGridWidth= toCopy.subGridWidth;
		subGridCol	= toCopy.subGridCol;
		gridColor	= toCopy.gridColor;
		labelsGridCol	= toCopy.labelsGridCol;
		subGridDiv		= toCopy.subGridDiv;
		gridDots		= toCopy.gridDots;
		subGridDots		= toCopy.subGridDots;
		swapAxes	= toCopy.swapAxes;
		origin 		= new Point2D.Double(toCopy.origin.x, toCopy.origin.y);
		lineStyle	= toCopy.lineStyle;
		dotStep		= toCopy.dotStep;
		dashBlack	= toCopy.dashBlack;
		dashWhite	= toCopy.dashWhite;
		borderColor	= toCopy.borderColor;
		border		= toCopy.border;
		dbleLine	= toCopy.dbleLine;
		dbleSep		= toCopy.dbleSep;
		dbleColor	= toCopy.dbleColor;
		shadowCol	= toCopy.shadowCol;
		isShadow	= toCopy.isShadow;
		shadowAngle	= toCopy.shadowAngle;
		shadowSize	= toCopy.shadowSize;
		borderPos	= toCopy.borderPos;
		fillStyle	= toCopy.fillStyle;
		fillColor	= toCopy.fillColor;
		hatchWidth	= toCopy.hatchWidth;
		hatchSep	= toCopy.hatchSep;
		hatchCol	= toCopy.hatchCol;
		hatchAngle	= toCopy.hatchAngle;
		arrowStyle	= new String[2];
		arrowStyle[0] = toCopy.arrowStyle[0];
		arrowStyle[1] = toCopy.arrowStyle[1];
		arrowSizeD	= toCopy.arrowSizeD;
		arrowSizeN	= toCopy.arrowSizeN;
		arrowLgth	= toCopy.arrowLgth;
		arrowInset	= toCopy.arrowInset;
		arrowTBarSD	= toCopy.arrowTBarSD;
		arrowTBarSN	= toCopy.arrowTBarSN;
		arrowBrLgth	= toCopy.arrowBrLgth;
		arrowrBrLgth= toCopy.arrowrBrLgth;
		arrowDotSD	= toCopy.arrowDotSD;
		arrowDotSN	= toCopy.arrowDotSN;
		arrowScale1	= toCopy.arrowScale1;
		arrowScale2	= toCopy.arrowScale2;
		textEnc		= toCopy.textEnc;
		textShape	= toCopy.textShape;
		textSeries	= toCopy.textSeries;
		textFamily	= toCopy.textFamily;
		textColor	= toCopy.textColor;
		textSize 	= toCopy.textSize;
		psBoxText = new Text(null, false);
		psBox = new Vector<FramedBox>();
		showOrigin 	= toCopy.showOrigin;
		axesStyle	= toCopy.axesStyle;
		ticks		= toCopy.ticks;
		ticksSize	= toCopy.ticksSize;
		ticksStyle	= toCopy.ticksStyle;
		labels		= toCopy.labels;
		ox			= toCopy.ox;
		oy			= toCopy.oy;
		dxIncrement	= toCopy.dxIncrement;
		dyIncrement = toCopy.dyIncrement;
		dxLabelDist	= toCopy.dxLabelDist;
		dyLabelDist	= toCopy.dyLabelDist;
	}





	/**
	 * @return the position.
	 */
	public PositionParameters getPosition()
	{
		return position;
	}





	/**
	 * Sets a new position.
	 */
	public void setPosition(double dx, double dy, boolean isSumRot, double rputA, double uputA, 
							int refPt, double labelSep, double refAngle)
	{
		position = new PositionParameters(isSumRot, dx, dy, rputA, uputA, labelSep, refPt, refAngle);
	}
	
	
	
	
	/**
	 * @return The whole rotation angle (in degrees).
	 */
	public static double getWholeRotationAngle(Vector<PSTParameters> params)
	{
		int i=0, size = params.size();
		double sum=0;
		
		if(size>0)
		{
			PositionParameters pos = params.elementAt(i).getPosition();
			if(pos!=null)
			{
				if(pos.isSumRotation)
					sum+=pos.angleRput;
				else sum = pos.angleRput;
			}
		}
		
		for(i=1; i<size; i++)
		{
			PositionParameters pos = params.elementAt(i).getPosition();
			if(pos!=null)
			{
				if(pos.isSumRotation)
					sum+=((360/params.elementAt(i-1).degrees)*pos.angleRput);
				else sum = ((360/params.elementAt(i-1).degrees)*pos.angleRput);
			}
		}
		return sum;
	}



	
	/**
	 * Reinitialises the general parameters.
	 * @since 1.9
	 */
	public static void reinitialize()
	{
		isCentered 	= false;
		pictureNEPt = null;
		pictureSWPt = null;
		tokenPosition = "";//$NON-NLS-1$
	}
}
