#include <gnutls/gnutls.h>
#include <gnutls/x509.h>
#include <VMState.h>
#include <KayaAPI.h>
#include <stdfuns.h>
#include "tls_glue.h"
#include "network_glue.h"

#if LIBGNUTLS_VERSION_MINOR >= 2
#define GSESSTYPE gnutls_session_t
#define GCERTTYPE gnutls_certificate_credentials_t
#define GTRANTYPE gnutls_transport_ptr_t
#define GDATTYPE gnutls_datum_t
#define GX509TYPE gnutls_x509_crt_t
#else
#define GSESSTYPE gnutls_session
#define GCERTTYPE gnutls_certificate_credentials
#define GTRANTYPE gnutls_transport_ptr
#define GDATTYPE gnutls_datum
#define GX509TYPE gnutls_x509_crt
#endif

void* do_gnutls_makecred() {
  GCERTTYPE cred;
  gnutls_certificate_allocate_credentials (&cred);
  gnutls_certificate_set_verify_flags(cred,GNUTLS_VERIFY_ALLOW_X509_V1_CA_CRT);
  return (void*)cred;
}

void do_addcert(void* vmptr, void* credptr, wchar_t* fn) {
  VMState* vm = (VMState*) vmptr;
  GCERTTYPE cred = (GCERTTYPE)credptr;
  int res = gnutls_certificate_set_x509_trust_file(cred,CSTRING(fn),GNUTLS_X509_FMT_PEM);
  if (res < 0) {
    vm->kaya_internalError(1);
  }
}

void do_verifycert(void* vmptr, void* sessptr, wchar_t* host) {
  VMState* vm = (VMState*) vmptr;
  GSESSTYPE sess = (GSESSTYPE) sessptr;
  unsigned int status;
  int ver = gnutls_certificate_verify_peers2(sess,&status);
  if (ver < 0) {
    vm->kaya_internalError(1);
  } else if (status & GNUTLS_CERT_SIGNER_NOT_FOUND) {
    vm->kaya_internalError(4);
    /*  } else if (status & GNUTLS_CERT_SIGNER_NOT_CA) {
    vm->kaya_internalError(4); */
  } else if (status & GNUTLS_CERT_INVALID) {
    vm->kaya_internalError(2);
  } else if (status & GNUTLS_CERT_REVOKED) {
    vm->kaya_internalError(3);
  } else {
    if (host != L"") {
      // verify the hostname
      unsigned int numcerts;
      const GDATTYPE* certs;
      GX509TYPE cert;
      if (gnutls_x509_crt_init(&cert) < 0) {
	vm->kaya_internalError(6);
	return;
      }
      certs = gnutls_certificate_get_peers(sess,&numcerts);
      if (numcerts == 0) {
	vm->kaya_internalError(6);
	return;
      }
      gnutls_x509_crt_import(cert,&certs[0],GNUTLS_X509_FMT_DER);
      if (!gnutls_x509_crt_check_hostname(cert,CSTRING(host))) {
	vm->kaya_internalError(5);
      }
      gnutls_x509_crt_deinit(cert);
    }
  }
}

void* do_gnutls_init(void* cred) {
  GSESSTYPE session;
  const int cert_type_priority[3] = { GNUTLS_CRT_X509, GNUTLS_CRT_OPENPGP, 0 };
  gnutls_init (&session, GNUTLS_CLIENT);
  gnutls_set_default_priority(session);
  gnutls_certificate_type_set_priority(session,cert_type_priority);
  gnutls_credentials_set (session, GNUTLS_CRD_CERTIFICATE, (GCERTTYPE) cred);
  return (void*)session;
}

int do_gnutls_transport(void* rawtls, void* rawconn) {
  GSESSTYPE tls = (GSESSTYPE) rawtls;
  ConnInfo* c = (ConnInfo*) rawconn;
  gnutls_transport_set_ptr(tls, (GTRANTYPE) c->cid);
  return gnutls_handshake(tls);
}

// http://www.gnu.org/software/gnutls/manual/gnutls.html
// gnutls_certificate_set_x509_trust_file
// gnutls_certificate_verify_peers2()
// /etc/ssl/certs/*.pem

// takes a void*, length(void*) as does g._r._recv
// so easy enough to make the binary versions
void do_gnutls_put(void* rawtls, wchar_t* rawdat) {
  GSESSTYPE tls = (GSESSTYPE) rawtls;
  char* dat = CSTRING(rawdat);
  gnutls_record_send(tls,dat,strlen(dat));
}

void do_gnutls_putbin(void* rawtls, void* rawdat, kint len) {
  GSESSTYPE tls = (GSESSTYPE) rawtls;
  gnutls_record_send(tls,(char*)rawdat,len);
}

void do_gnutls_putbyte(void* rawtls, kint byte) {
  GSESSTYPE tls = (GSESSTYPE) rawtls;
  char data = (char)byte;
  gnutls_record_send(tls,&data,1);
}

wchar_t* do_gnutls_get(void* rawtls, void* vmptr, kint rlen) {
  GSESSTYPE tls = (GSESSTYPE) rawtls;
  char buffer[rlen+1];
  int ret = gnutls_record_recv(tls, buffer, rlen);
  if (ret < 0) {
    VMState* vm = (VMState*)vmptr;
    vm->kaya_internalError(ret);
  }
  buffer[ret] = '\0';
  return KSTRING(buffer);
}

void* do_gnutls_getbin(void* rawtls, void* vmptr, KValue len) {
  GSESSTYPE tls = (GSESSTYPE) rawtls;
  char* buffer = (char*)GC_MALLOC_ATOMIC(sizeof(char)*4097);
  int ret = gnutls_record_recv(tls, buffer, 4096);
  if (ret < 0) {
    VMState* vm = (VMState*)vmptr;
    vm->kaya_internalError(ret);
  }
  KayaSetInt(len,ret);
  return (void*)buffer;
}

kint do_gnutls_getbyte(void* rawtls, void* vmptr) {
  GSESSTYPE tls = (GSESSTYPE) rawtls;
  char buffer[2];
  int ret = gnutls_record_recv(tls, buffer, 1);
  if (ret < 0) {
    VMState* vm = (VMState*)vmptr;
    vm->kaya_internalError(ret);
  }
  return (kint)buffer[0];
}


void do_gnutls_close(void* rawtls, void* rawcred) {
  GSESSTYPE tls = (GSESSTYPE) rawtls;
  GCERTTYPE cred = (GCERTTYPE) rawcred;
  gnutls_bye(tls, GNUTLS_SHUT_RDWR);
  gnutls_deinit(tls);
  gnutls_certificate_free_credentials(cred);
}
