# -*- encoding: utf-8 -*-

import json

import yaml

from juju.errors import JujuError
from juju.lib.testing import TestCase
from juju.lib.format import (
    PythonFormat, YAMLFormat, is_valid_charm_format, get_charm_formatter,
    get_charm_formatter_from_env)


class TestFormatLookup(TestCase):

    def test_is_valid_charm_format(self):
        """Verify currently valid charm formats"""
        self.assertFalse(is_valid_charm_format(0))
        self.assertTrue(is_valid_charm_format(1))
        self.assertTrue(is_valid_charm_format(2))
        self.assertFalse(is_valid_charm_format(3))

    def test_get_charm_formatter(self):
        self.assertInstance(get_charm_formatter(1), PythonFormat)
        self.assertInstance(get_charm_formatter(2), YAMLFormat)
        e = self.assertRaises(JujuError, get_charm_formatter, 0)
        self.assertEqual(
            str(e),
            "Expected charm format to be either 1 or 2, got 0")

    def test_get_charm_formatter_from_env(self):
        """Verifies _JUJU_CHARM_FORMAT can be mapped to valid formatters"""

        self.change_environment(_JUJU_CHARM_FORMAT="0")
        e = self.assertRaises(JujuError, get_charm_formatter_from_env)
        self.assertEqual(
            str(e),
            "Expected charm format to be either 1 or 2, got 0")

        self.change_environment(_JUJU_CHARM_FORMAT="1")
        self.assertInstance(get_charm_formatter_from_env(), PythonFormat)

        self.change_environment(_JUJU_CHARM_FORMAT="2")
        self.assertInstance(get_charm_formatter_from_env(), YAMLFormat)

        self.change_environment(_JUJU_CHARM_FORMAT="3")
        e = self.assertRaises(JujuError, get_charm_formatter_from_env)
        self.assertEqual(
            str(e),
            "Expected charm format to be either 1 or 2, got 3")


class TestPythonFormat(TestCase):

    def assert_format(self, data, expected):
        """Verify str output serialization; no roundtripping is supported."""
        formatted = PythonFormat().format(data)
        self.assertEqual(formatted, expected)

    def test_format(self):
        """Verifies Python str formatting of data"""
        self.assert_format(None, "None")
        self.assert_format("", "")
        self.assert_format("A string", "A string")
        self.assert_format(
            "High bytes: \xCA\xFE",
            "High bytes: \xca\xfe")
        self.assert_format(u"", "")
        self.assert_format(
            u"A unicode string (but really ascii)",
            "A unicode string (but really ascii)")
        e = self.assertRaises(UnicodeEncodeError, PythonFormat().format, u"中文")
        self.assertEqual(
                str(e),
                ("'ascii' codec can't encode characters in position 0-1: "
                "ordinal not in range(128)"))
        self.assert_format({}, "{}")
        self.assert_format(
            {u"public-address": u"ec2-1-2-3-4.compute-1.amazonaws.com",
             u"foo": u"bar",
             u"configured": True},
            ("{u'public-address': u'ec2-1-2-3-4.compute-1.amazonaws.com', "
             "u'foo': u'bar', u'configured': True}"))
        self.assert_format(False, "False")
        self.assert_format(True, "True")
        self.assert_format(0.0, "0.0")
        self.assert_format(3.14159, "3.14159")
        self.assert_format(6.02214178e23, "6.02214178e+23")
        self.assert_format(0, "0")
        self.assert_format(42, "42")

    def test_parse_keyvalue_pairs(self):
        """Verify reads in values as strings"""
        sample = self.makeFile("INPUT DATA")

        # test various styles of options being read
        options = ["alpha=beta",
                   "content=@%s" % sample]

        formatter = PythonFormat()
        data = formatter.parse_keyvalue_pairs(options)
        self.assertEquals(data["alpha"], "beta")
        self.assertEquals(data["content"], "INPUT DATA")

        # and check an error condition
        options = ["content=@missing"]
        error = self.assertRaises(
            JujuError, formatter.parse_keyvalue_pairs, options)
        self.assertEquals(
            str(error),
            "No such file or directory: missing (argument:content)")

        # and check when fed non-kvpairs the error makes sense
        options = ["foobar"]
        error = self.assertRaises(
            JujuError, formatter.parse_keyvalue_pairs, options)
        self.assertEquals(
            str(error), "Expected `option=value`. Found `foobar`")

    def assert_dump_load(self, data, expected):
        """Asserts expected formatting and roundtrip through dump/load"""
        formatter = PythonFormat()
        dumped = formatter.dump({"data": data})
        loaded = formatter.load(dumped)["data"]
        self.assertEqual(dumped, expected)
        self.assertEqual(dumped, json.dumps({"data": data}))
        self.assertEqual(data, loaded)
        if isinstance(data, str):
            # Verify promotion of str to unicode
            self.assertInstance(loaded, unicode)

    def test_dump_load(self):
        """Verify JSON roundtrip semantics"""
        self.assert_dump_load(None, '{"data": null}')
        self.assert_dump_load("", '{"data": ""}')
        self.assert_dump_load("A string", '{"data": "A string"}')
        e = self.assertRaises(
            UnicodeDecodeError,
            PythonFormat().dump, "High bytes: \xCA\xFE")
        self.assertEqual(
            str(e), 
            "'utf8' codec can't decode byte 0xca in position 12: "
            "invalid continuation byte")
        self.assert_dump_load(u"", '{"data": ""}')
        self.assert_dump_load(
            u"A unicode string (but really ascii)",
            '{"data": "A unicode string (but really ascii)"}')
        e = self.assertRaises(UnicodeEncodeError, PythonFormat().format, u"中文")
        self.assertEqual(
                str(e),
                ("'ascii' codec can't encode characters in position 0-1: "
                "ordinal not in range(128)"))
        self.assert_dump_load({}, '{"data": {}}')
        self.assert_dump_load(
            {u"public-address": u"ec2-1-2-3-4.compute-1.amazonaws.com",
             u"foo": u"bar",
             u"configured": True},
            ('{"data": {"public-address": '
             '"ec2-1-2-3-4.compute-1.amazonaws.com", "foo": "bar", '
             '"configured": true}}'))
        self.assert_dump_load(False, '{"data": false}')
        self.assert_dump_load(True, '{"data": true}')
        self.assert_dump_load(0.0, '{"data": 0.0}')
        self.assert_dump_load(3.14159, '{"data": 3.14159}')
        self.assert_dump_load(6.02214178e23, '{"data": 6.02214178e+23}')
        self.assert_dump_load(0, '{"data": 0}')
        self.assert_dump_load(42, '{"data": 42}')

    def test_should_delete(self):
        """Verify empty or whitespace only strings indicate deletion"""
        formatter = PythonFormat()
        self.assertFalse(formatter.should_delete("0"))
        self.assertFalse(formatter.should_delete("something"))
        self.assertTrue(formatter.should_delete(""))
        self.assertTrue(formatter.should_delete("   "))

        # Verify that format: 1 can only work with str values
        e = self.assertRaises(AttributeError, formatter.should_delete, 42)
        self.assertEqual(str(e), "'int' object has no attribute 'strip'")
        e = self.assertRaises(AttributeError, formatter.should_delete, None)
        self.assertEqual(str(e), "'NoneType' object has no attribute 'strip'")


class TestYAMLFormat(TestCase):

    def assert_format(self, data, expected):
        """Verify actual output serialization and roundtripping through YAML"""
        formatted = YAMLFormat().format(data)
        self.assertEqual(formatted, expected)
        self.assertEqual(data, yaml.safe_load(formatted))

    def test_format(self):
        """Verifies standard formatting of data into valid YAML"""
        self.assert_format(None, "")
        self.assert_format("", "''")
        self.assert_format("A string", "A string")
        # Note: YAML uses b64 encoding for byte strings tagged by !!binary
        self.assert_format(
            "High bytes: \xCA\xFE",
            "!!binary |\n    SGlnaCBieXRlczogyv4=")
        self.assert_format(u"", "''")
        self.assert_format(
            u"A unicode string (but really ascii)",
            "A unicode string (but really ascii)")
        # Any non-ascii Unicode will use UTF-8 encoding
        self.assert_format(u"中文", "\xe4\xb8\xad\xe6\x96\x87")
        self.assert_format({}, "{}")
        self.assert_format(
            {u"public-address": u"ec2-1-2-3-4.compute-1.amazonaws.com",
             u"foo": u"bar",
             u"configured": True},
            ("configured: true\n"
             "foo: bar\n"
             "public-address: ec2-1-2-3-4.compute-1.amazonaws.com"))
        self.assert_format(False, "false")
        self.assert_format(True, "true")
        self.assert_format(0.0, "0.0")
        self.assert_format(3.14159, "3.14159")
        self.assert_format(6.02214178e23, "6.02214178e+23")
        self.assert_format(0, "0")
        self.assert_format(42, "42")

    def assert_parse(self, data):
        """Verify input parses as expected, including from a data file"""
        formatter = YAMLFormat()
        formatted = formatter.format_raw(data)
        data_file = self.makeFile(formatted)
        kvs = ["formatted=%s" % formatted,
               "file=@%s" % data_file]
        parsed = formatter.parse_keyvalue_pairs(kvs)
        self.assertEqual(parsed["formatted"], data)
        self.assertEqual(parsed["file"], data)

    def test_parse_keyvalue_pairs(self):
        """Verify key value pairs parse for a wide range of YAML inputs."""
        formatter = YAMLFormat()
        self.assert_parse("")
        self.assert_parse("A string")
        self.assert_parse("High bytes: \xCA\xFE")

        # Raises an error if no such file
        e = self.assertRaises(
            JujuError,
            formatter.parse_keyvalue_pairs, ["content=@missing"])
        self.assertEquals(
            str(e),
            "No such file or directory: missing (argument:content)")

        # Raises an error if not of the form K=V or K=
        e = self.assertRaises(
            JujuError,
            formatter.parse_keyvalue_pairs, ["foobar"])
        self.assertEquals(
            str(e), "Expected `option=value`. Found `foobar`")

    def assert_dump_load(self, data, expected):
        """Asserts expected formatting and roundtrip through dump/load"""
        formatter = YAMLFormat()
        dumped = formatter.dump({"data": data})
        loaded = formatter.load(dumped)["data"]
        self.assertEqual(dumped, expected)
        self.assertEqual(data, loaded)
        if isinstance(data, str):
            # Verify that no promotion to unicode occurs for str values
            self.assertInstance(loaded, str)

    def test_dump_load(self):
        """Verify JSON roundtrip semantics"""
        self.assert_dump_load(None, "data: null")
        self.assert_dump_load("", "data: ''")
        self.assert_dump_load("A string", "data: A string")
        self.assert_dump_load("High bytes: \xCA\xFE",
                              "data: !!binary |\n    SGlnaCBieXRlczogyv4=")
        self.assert_dump_load(u"", "data: ''")
        self.assert_dump_load(
            u"A unicode string (but really ascii)",
            "data: A unicode string (but really ascii)")
        self.assert_dump_load(u"中文", "data: \xe4\xb8\xad\xe6\x96\x87")
        self.assert_dump_load({}, "data: {}")
        self.assert_dump_load(
            {u"public-address": u"ec2-1-2-3-4.compute-1.amazonaws.com",
             u"foo": u"bar",
             u"configured": True},
            ("data:\n"
             "    configured: true\n"
             "    foo: bar\n"
             "    public-address: ec2-1-2-3-4.compute-1.amazonaws.com"))
        self.assert_dump_load(False, "data: false")
        self.assert_dump_load(True, "data: true")
        self.assert_dump_load(0.0, "data: 0.0")
        self.assert_dump_load(3.14159, "data: 3.14159")
        self.assert_dump_load(6.02214178e23, "data: 6.02214178e+23")
        self.assert_dump_load(0, "data: 0")
        self.assert_dump_load(42, "data: 42")

    def test_should_delete(self):
        """Verify empty or whitespace only strings indicate deletion"""
        formatter = PythonFormat()
        self.assertFalse(formatter.should_delete("0"))
        self.assertFalse(formatter.should_delete("something"))
        self.assertTrue(formatter.should_delete(""))
        self.assertTrue(formatter.should_delete("   "))

        # Verify that format: 1 can only work with str values
        e = self.assertRaises(AttributeError, formatter.should_delete, 42)
        self.assertEqual(str(e), "'int' object has no attribute 'strip'")
        e = self.assertRaises(AttributeError, formatter.should_delete, None)
        self.assertEqual(str(e), "'NoneType' object has no attribute 'strip'")
